/*********************************************************/
/*  Master Class for 4 Operator FM Synth                 */
/*  by Perry R. Cook, 1995-96; recoded John ffitch 97/98 */
/*  This instrument contains 4 waves, 4 adsr, and        */
/*  various state vars.                                  */
/*                                                       */
/*  The basic Chowning/Stanford FM patent expired April  */
/*  1995, but there exist follow-on patents, mostly      */
/*  assigned to Yamaha.  If you are of the type who      */
/*  should worry about this (making money) worry away.   */
/*                                                       */
/*********************************************************/

#include "fm4op.h"

/***********************************************************/
/*  Two Zero Filter Class,                                 */
/*  by Perry R. Cook, 1995-96; recoded John ffitch 97/98   */
/*  See books on filters to understand more about how this */
/*  works.  Nothing out of the ordinary in this version.   */
/***********************************************************/

void make_TwoZero(TwoZero *p)
{
    p->zeroCoeffs[0] = 0.0f;
    p->zeroCoeffs[1] = 0.0f;
    p->gain = 1.0f;
    p->inputs[0] = 0.0f;
    p->inputs[1] = 0.0f;
    p->lastOutput = 0.0f;
}

void TwoZero_setZeroCoeffs(TwoZero* p, float *coeffs)
{
    p->zeroCoeffs[0] = coeffs[0];
    p->zeroCoeffs[1] = coeffs[1];
}

float TwoZero_tick(TwoZero *p, float sample)
  /*   Perform Filter Operation            */
{ /*  TwoZero is a two zero filter (duh!)  */
  /*  Look it up in your favorite DSP text */
    p->lastOutput = p->zeroCoeffs[0] * p->inputs[0] +
                    p->zeroCoeffs[1] * p->inputs[1];
    p->inputs[1] = p->inputs[0];
    p->inputs[0] = p->gain * sample;
    p->lastOutput += p->inputs[0];
    return p->lastOutput;
}



float Wave_tick(float *vTime, int len, float *data, float rate, float phase)
{                                /* Tick on vibrato table */
        long	temp;
        float	temp_time, alpha;
	float	lastOutput;

        *vTime += rate;            /*  Update current time    */
        while (*vTime >= len)      /*  Check for end of sound */
          *vTime -= len;           /*  loop back to beginning */
        while (*vTime < 0.0f)       /*  Check for end of sound */
          *vTime += len;           /*  loop back to beginning */

        temp_time = *vTime;

        if (phase != 0.0f) {
          temp_time += phase;      /*  Add phase offset       */
          while (temp_time >= len) /*  Check for end of sound */
	    temp_time -= len;      /*  loop back to beginning */
          while (temp_time < 0.0f)  /*  Check for end of sound */
	    temp_time += len;      /*  loop back to beginning */
        }
        temp = (long) temp_time;   /*  Integer part of time address    */
                                   /*  fractional part of time address */
        alpha = temp_time - (float)temp;
        lastOutput = data[temp]; /* Do linear interpolation */
                        /*  same as alpha*data[temp+1] + (1-alpha)data[temp] */
        lastOutput += (alpha * (data[temp+1] - lastOutput));
                                /* End of vibrato tick */
        return lastOutput;
}

/* ---------------------------------------------------------------------- */

static int 	FM_tabs_built = 0;
static float	FM4Op_gains[100];
static float	FM4Op_susLevels[16];
static float	FM4Op_attTimes[32];

void build_FM(void)
{                                /* The following tables are pre-built */
    float	temp = 1.0f;
    int		i;

    for (i=99; i>=0; i--) {
        FM4Op_gains[i] = temp;
        temp *= 0.933033f;
    }
    temp = 1.0f;
    for (i=15; i>=0; i--) {
        FM4Op_susLevels[i] = temp;
        temp *= 0.707101f;
    }
    temp = 8.498186f;
    for (i=0; i<32; i++) {
        FM4Op_attTimes[i] = temp;
        temp *= 0.707101f;
    }
    FM_tabs_built = 1;
}

void make_FM4Op(FM4OP *p)
{
    float	tempCoeffs[2] = {0.0f, -1.0f};
    FUNC	*ftp;

    if (!FM_tabs_built) build_FM(); /* Ensure tables exist */

    make_ADSR(&p->adsr[0]);
    make_ADSR(&p->adsr[1]);
    make_ADSR(&p->adsr[2]);
    make_ADSR(&p->adsr[3]);
    make_TwoZero(&p->twozero);
    if ((ftp = ftfind(p->vifn)) != NULL) p->vibWave = ftp;
    else perferror("No table for VibWaveato"); /* Expect sine wave */
    p->baseFreq = 440.0f;
    p->ratios[0] = 1.0f;
    p->ratios[1] = 1.0f;
    p->ratios[2] = 1.0f;
    p->ratios[3] = 1.0f;
    p->gains[0] = 1.0f;
    p->gains[1] = 1.0f;
    p->gains[2] = 1.0f;
    p->gains[3] = 1.0f;
    TwoZero_setZeroCoeffs(&p->twozero, tempCoeffs);
    p->twozero.gain = 0.0f;
}

void FM4Op_loadWaves(FM4OP *p)
{
    FUNC	*ftp;

    if ((ftp = ftfind(p->ifn0)) != NULL) p->waves[0] = ftp;
    else perferror("No table for FM4Op"); /* Expect sine wave */
    if ((ftp = ftfind(p->ifn1)) != NULL) p->waves[1] = ftp;
    else perferror("No table for  FM4Op"); /* Expect sine wave */
    if ((ftp = ftfind(p->ifn2)) != NULL) p->waves[2] = ftp;
    else perferror("No table for  FM4Op"); /* Expect sine wave */
    if ((ftp = ftfind(p->ifn3)) != NULL) p->waves[3] = ftp;
    else perferror("No table for  FM4Op"); /* Expect sine wave */
}

void FM4Op_setRatio(FM4OP *p, int whichOne, float ratio)
{
    p->ratios[whichOne] = ratio;
    if (ratio>0.0f)
        p->w_rate[whichOne] = p->baseFreq * ratio;
    else
        p->w_rate[whichOne] = ratio;
}

void FM4Op_keyOff(FM4OP *p)
{
    ADSR_keyOff(&p->adsr[0]);
    ADSR_keyOff(&p->adsr[1]);
    ADSR_keyOff(&p->adsr[2]);
    ADSR_keyOff(&p->adsr[3]);
}

#ifdef never
void FM4Op_controlChange(FM4OP *p, int number, float value)
{
#if defined(_debug_)
    printf("FM4Op : ControlChange: Number=%i Value=%f\n",number,value);
#endif
    else if (number == MIDI_after_touch)	{
        adsr[0]->setTarget(value * NORM_7);
        adsr[1]->setTarget(value * NORM_7);
        adsr[2]->setTarget(value * NORM_7);
        adsr[3]->setTarget(value * NORM_7);
    }
}
#endif

/*********************************************************/
/*  Algorithm 5 (TX81Z) Subclass of 4 Operator FM Synth  */
/*  by Perry R. Cook, 1995-96; recoded John ffitch 97/98 */
/*  This connection topology is 2 simple FM Pairs summed */
/*  together, like:                                      */
/*                                                       */
/*   Alg 5 is :      4->3--\                             */
/*                          + --> Out                    */
/*                   2->1--/                             */
/*                                                       */
/*  Controls: control1 = mod index 1                     */
/*            control2 = crossfade of two outputs        */
/*                                                       */
/*********************************************************/


float FM4Alg5_tick(FM4OP *p, float c1, float c2)
{
    float	temp,temp2;
    float	lastOutput;

    temp = p->gains[1] * ADSR_tick(&p->adsr[1]) *
         Wave_tick(&p->w_time[1], (int)p->waves[1]->flen, p->waves[1]->ftable,
                   p->w_rate[1], p->w_phase[1]);
    temp = temp * c1;
    p->w_phase[0] = p->waves[0]->flen * temp; /* addPhaseOffset */
    p->w_phase[3] = p->waves[0]->flen * p->twozero.lastOutput;
    temp =  p->gains[3] * ADSR_tick(&p->adsr[3]) *
         Wave_tick(&p->w_time[3], (int)p->waves[3]->flen, p->waves[3]->ftable,
                   p->w_rate[3], p->w_phase[3]);
    TwoZero_tick(&p->twozero, temp);
    p->w_phase[2] = p->waves[2]->flen * temp; /* addPhaseOffset */
    temp = (1.0f - ( c2 * 0.5f)) *  p->gains[0] *
      ADSR_tick(&p->adsr[0]) *
      Wave_tick(&p->w_time[0], (int)p->waves[0]->flen, p->waves[0]->ftable,
                             p->w_rate[0], p->w_phase[0]);
    temp +=  c2 * 0.5f *  p->gains[2] * ADSR_tick(&p->adsr[2]) *
      Wave_tick(&p->w_time[2], (int)p->waves[2]->flen, p->waves[2]->ftable,
                p->w_rate[2], p->w_phase[2]);

    temp2 = Wave_tick(&p->v_time, (int)p->vibWave->flen,
                      p->vibWave->ftable, p->v_rate, 0.0f) *
            *p->modDepth; /* Calculate amplitude mod */
    temp = temp * (1.0f + temp2); /*  and apply it to output */

    lastOutput = temp * 0.5f;
    return  lastOutput;
}

/***************************************************************/
/*  Tubular Bell (Orch. Chime) Subclass of Algorithm 5 (TX81Z) */
/*  Subclass of 4 Operator FM Synth by Perry R. Cook, 1995-96  */
/*  Recoded in C by John ffitch 1997-98                        */
/***************************************************************/

void tubebellset(FM4OP *p)
{
    float	amp = *p->amp * AMP_RSCALE; /* Normalised */

    make_FM4Op(p);
    FM4Op_loadWaves(p); /* 4 times  "rawwaves/sinewave.raw" */

    FM4Op_setRatio(p, 0, 1.0f   * 0.995f);
    FM4Op_setRatio(p, 1, 1.414f * 0.995f);
    FM4Op_setRatio(p, 2, 1.0f   * 1.005f);
    FM4Op_setRatio(p, 3, 1.414f         );
    p->gains[0] = amp * FM4Op_gains[94];
    p->gains[1] = amp * FM4Op_gains[76];
    p->gains[2] = amp * FM4Op_gains[99];
    p->gains[3] = amp * FM4Op_gains[71];
    ADSR_setAll(&p->adsr[0], 0.03f,0.00001f,0.0f,0.02f);
    ADSR_setAll(&p->adsr[1], 0.03f,0.00001f,0.0f,0.02f);
    ADSR_setAll(&p->adsr[2], 0.07f,0.00002f,0.0f,0.02f);
    ADSR_setAll(&p->adsr[3], 0.04f,0.00001f,0.0f,0.02f);
    p->twozero.gain = 0.5f;
    p->v_rate = 2.0f * p->vibWave->flen / esr; /* Vib rate */
                                /* Set Freq */
    p->baseFreq = *p->frequency;
    p->w_rate[0] = p->baseFreq *  p->ratios[0] * p->waves[0]->flen / esr;
    p->w_rate[1] = p->baseFreq *  p->ratios[1] * p->waves[1]->flen / esr;
    p->w_rate[2] = p->baseFreq *  p->ratios[2] * p->waves[2]->flen / esr;
    p->w_rate[3] = p->baseFreq *  p->ratios[3] * p->waves[3]->flen / esr;
    ADSR_keyOn(&p->adsr[0]);
    ADSR_keyOn(&p->adsr[1]);
    ADSR_keyOn(&p->adsr[2]);
    ADSR_keyOn(&p->adsr[3]);
}

void tubebell(FM4OP *p)
{
    float	amp = *p->amp * AMP_RSCALE; /* Normalised */
    float	*ar = p->ar;
    long	nsmps = ksmps;
    float	c1 = *p->control1;
    float	c2 = *p->control2;

                                /* Set Freq */
    p->baseFreq = *p->frequency;
    p->gains[0] = amp * FM4Op_gains[94];
    p->gains[1] = amp * FM4Op_gains[76];
    p->gains[2] = amp * FM4Op_gains[99];
    p->gains[3] = amp * FM4Op_gains[71];
    p->w_rate[0] = p->baseFreq *  p->ratios[0] * p->waves[0]->flen / esr;
    p->w_rate[1] = p->baseFreq *  p->ratios[1] * p->waves[1]->flen / esr;
    p->w_rate[2] = p->baseFreq *  p->ratios[2] * p->waves[2]->flen / esr;
    p->w_rate[3] = p->baseFreq *  p->ratios[3] * p->waves[3]->flen / esr;
    p->v_rate = *p->vibFreq * p->vibWave->flen / esr;

    do {
        float	lastOutput;
        lastOutput = FM4Alg5_tick(p, c1, c2);
        *ar++ = lastOutput*AMP_SCALE*1.8f;
    } while (--nsmps);
}

/*****************************************************************/
/*  Fender Rhodes Electric Piano Subclass of Algorithm 5 (TX81Z) */
/*  Subclass of 4 Operator FM Synth by Perry R. Cook, 1995-96    */
/*  Recoded in C by John ffitch 1997-98                          */
/*****************************************************************/

void rhodeset(FM4OP *p)
{
    float	amp = *p->amp * AMP_RSCALE; /* Normalised */

    make_FM4Op(p);
    FM4Op_loadWaves(p); /* 3 times  "sinewave.raw"; 1 x fwavblnk.raw */

    FM4Op_setRatio(p, 0, 1.0f);
    FM4Op_setRatio(p, 1, 0.5f);
    FM4Op_setRatio(p, 2, 1.0f);
    FM4Op_setRatio(p, 3, 15.0f);
    p->gains[0] = amp * FM4Op_gains[99];
    p->gains[1] = amp * FM4Op_gains[90];
    p->gains[2] = amp * FM4Op_gains[99];
    p->gains[3] = amp * FM4Op_gains[67];
    ADSR_setAll(&p->adsr[0], 0.05f,0.00003f,0.0f,0.02f);
    ADSR_setAll(&p->adsr[1], 0.05f,0.00003f,0.0f,0.02f);
    ADSR_setAll(&p->adsr[2], 0.05f,0.00005f,0.0f,0.02f);
    ADSR_setAll(&p->adsr[3], 0.05f,0.0002f,0.0f,0.02f);
    p->twozero.gain = 1.0f;
    p->v_rate = 2.0f * p->vibWave->flen / esr; /* Vib rate */
                                /* Set Freq */
    p->baseFreq = *p->frequency;
    p->w_rate[0] = p->baseFreq *  p->ratios[0] * p->waves[0]->flen / esr;
    p->w_rate[1] = p->baseFreq *  p->ratios[1] * p->waves[1]->flen / esr;
    p->w_rate[2] = p->baseFreq *  p->ratios[2] * p->waves[2]->flen / esr;
    p->w_rate[3] = p->baseFreq *  p->ratios[3] * p->waves[3]->flen / esr;
    ADSR_keyOn(&p->adsr[0]);
    ADSR_keyOn(&p->adsr[1]);
    ADSR_keyOn(&p->adsr[2]);
    ADSR_keyOn(&p->adsr[3]);
}

/***************************************************************/
/*  Wurlitzer Electric Piano Subclass of Algorithm 5 (TX81Z)   */
/*  Subclass of 4 Operator FM Synth by Perry R. Cook, 1995-96  */
/*  Recoded in C by John ffitch 1997-98                        */
/***************************************************************/


void wurleyset(FM4OP *p)
{
    float	amp = *p->amp * AMP_RSCALE; /* Normalised */

    make_FM4Op(p);
    FM4Op_loadWaves(p); /* 3 times  "sinewave.raw"; 1 x fwavblnk.raw */

    FM4Op_setRatio(p, 0, 1.0f);
    FM4Op_setRatio(p, 1, 4.05f);
    FM4Op_setRatio(p, 2, -510.0f);
    FM4Op_setRatio(p, 3, -510.0f);
    p->gains[0] = amp * FM4Op_gains[99];
    p->gains[1] = amp * FM4Op_gains[82];
    p->gains[2] = amp * FM4Op_gains[82];
    p->gains[3] = amp * FM4Op_gains[68];
    ADSR_setAll(&p->adsr[0], 0.05f,0.00003f,0.0f,0.02f);
    ADSR_setAll(&p->adsr[1], 0.05f,0.00003f,0.0f,0.02f);
    ADSR_setAll(&p->adsr[2], 0.05f,0.0002f,0.0f,0.02f);
    ADSR_setAll(&p->adsr[3], 0.05f,0.0003f,0.0f,0.02f);
    p->twozero.gain = 2.0f;
                                /* Set Freq */
    p->baseFreq = *p->frequency;
    p->w_rate[0] = p->baseFreq *  p->ratios[0] * p->waves[0]->flen / esr;
    p->w_rate[1] = p->baseFreq *  p->ratios[1] * p->waves[1]->flen / esr;
    p->w_rate[2] =                p->ratios[2] * p->waves[2]->flen / esr;
    p->w_rate[3] =                p->ratios[3] * p->waves[3]->flen / esr;
    ADSR_keyOn(&p->adsr[0]);
    ADSR_keyOn(&p->adsr[1]);
    ADSR_keyOn(&p->adsr[2]);
    ADSR_keyOn(&p->adsr[3]);
}

void wurley(FM4OP *p)
{
    float	amp = *p->amp * AMP_RSCALE; /* Normalised */
    float	*ar = p->ar;
    long	nsmps = ksmps;
    float	c1 = *p->control1;
    float	c2 = *p->control2;

                                /* Set Freq */
    p->baseFreq = *p->frequency;
    p->gains[0] = amp * FM4Op_gains[99];
    p->gains[1] = amp * FM4Op_gains[82];
    p->gains[2] = amp * FM4Op_gains[82];
    p->gains[3] = amp * FM4Op_gains[68];
    p->w_rate[0] = p->baseFreq *  p->ratios[0] * p->waves[0]->flen / esr;
    p->w_rate[1] = p->baseFreq *  p->ratios[1] * p->waves[1]->flen / esr;
    p->w_rate[2] =                p->ratios[2] * p->waves[2]->flen / esr;
    p->w_rate[3] =                p->ratios[3] * p->waves[3]->flen / esr;
    p->v_rate = *p->vibFreq * p->vibWave->flen / esr;

    do {
        float	lastOutput;
        lastOutput = FM4Alg5_tick(p, c1, c2);
        *ar++ = lastOutput*AMP_SCALE*1.9f;
    } while (--nsmps);
}

/*********************************************************/
/*  Algorithm 3 (TX81Z) Subclass of 4 Operator FM Synth  */
/*  by Perry R. Cook, 1995-96; recoded John ffitch 97/98 */
/*                                                       */
/*  Alg 3 is :          4--\                             */
/*                  3-->2-- + -->1-->Out                 */
/*                                                       */
/*  Controls: control1 = total mod index                 */
/*            control2 = crossfade of two modulators     */
/*                                                       */
/*********************************************************/

float FM4Alg3_tick(FM4OP *p, float c1, float c2)
{
    float	temp;
    float	lastOutput;

    temp = *p->modDepth * 0.2f * Wave_tick(&p->v_time, (int)p->vibWave->flen,
                                          p->vibWave->ftable, p->v_rate, 0.0f);
    p->w_rate[0] = p->baseFreq * (1.0f + temp) * p->ratios[0];
    p->w_rate[1] = p->baseFreq * (1.0f + temp) * p->ratios[1];
    p->w_rate[2] = p->baseFreq * (1.0f + temp) * p->ratios[2];
    p->w_rate[3] = p->baseFreq * (1.0f + temp) * p->ratios[3];

    temp = p->gains[2] * ADSR_tick(&p->adsr[2]) * 
      Wave_tick(&p->w_time[2], (int)p->waves[2]->flen, p->waves[2]->ftable,
                p->w_rate[2], p->w_phase[2]);
    p->w_phase[1] = p->waves[1]->flen * temp;
    p->w_phase[3] = p->waves[3]->flen * p->twozero.lastOutput;
    temp = (1.0f - (c2 * 0.5f)) * p->gains[3] * ADSR_tick(&p->adsr[3]) *
      Wave_tick(&p->w_time[3], (int)p->waves[3]->flen, p->waves[3]->ftable,
                p->w_rate[3], p->w_phase[3]);
    TwoZero_tick(&p->twozero, temp);

    temp += c2 * 0.5f * p->gains[1] * ADSR_tick(&p->adsr[1]) * 
      Wave_tick(&p->w_time[1], (int)p->waves[1]->flen, p->waves[1]->ftable,
                p->w_rate[1], p->w_phase[1]);
    temp = temp * c1;

     p->w_phase[0] = p->waves[0]->flen * temp;
    temp = p->gains[0] * ADSR_tick(&p->adsr[0]) *
      Wave_tick(&p->w_time[0], (int)p->waves[0]->flen, p->waves[0]->ftable,
                p->w_rate[0], p->w_phase[0]);

    lastOutput = temp * 0.5f;
    return lastOutput;
}

void heavymetset(FM4OP *p)
{
    make_FM4Op(p);
    FM4Op_loadWaves(p);         /* Mixed */
    FM4Op_setRatio(p, 0, 1.00f         );   
    FM4Op_setRatio(p, 1, 4.00f * 0.999f);
    FM4Op_setRatio(p, 2, 3.00f * 1.001f);
    FM4Op_setRatio(p, 3, 0.50f * 1.002f);
    ADSR_setAll(&p->adsr[0], 0.050f, 0.0100f, 1.0f, 0.001f); 
    ADSR_setAll(&p->adsr[1], 0.050f, 0.0010f, 1.0f, 0.0001f);
    ADSR_setAll(&p->adsr[2], 0.001f, 0.0020f, 1.0f, 0.0002f);
    ADSR_setAll(&p->adsr[3], 0.050f, 0.0010f, 0.2f, 0.0002f);
    p->twozero.gain = 2.0f;
/*     p->v_rate = 5.5 * p->vibWave->flen / esr;  Vib rate */
    ADSR_keyOn(&p->adsr[0]);
    ADSR_keyOn(&p->adsr[1]);
    ADSR_keyOn(&p->adsr[2]);
    ADSR_keyOn(&p->adsr[3]);
}

void heavymet(FM4OP *p)
{
    float	*ar = p->ar;
    long	nsmps = ksmps;
    float	amp = *p->amp * AMP_RSCALE; /* Normalised */
    float	c1 = *p->control1;
    float	c2 = *p->control2;

    p->baseFreq = *p->frequency;
    p->gains[0] = amp * FM4Op_gains[92];
    p->gains[1] = amp * FM4Op_gains[76];
    p->gains[2] = amp * FM4Op_gains[91];
    p->gains[3] = amp * FM4Op_gains[68];

    p->w_rate[0] = p->baseFreq *  p->ratios[0] * p->waves[0]->flen / esr;
    p->w_rate[1] = p->baseFreq *  p->ratios[1] * p->waves[1]->flen / esr;
    p->w_rate[2] = p->baseFreq *  p->ratios[2] * p->waves[2]->flen / esr;
    p->w_rate[3] = p->baseFreq *  p->ratios[3] * p->waves[3]->flen / esr;
    p->v_rate = *p->vibFreq * p->vibWave->flen / esr;
    do {
        float	lastOutput;
        lastOutput = FM4Alg3_tick(p, c1, c2);
        *ar++ = lastOutput*AMP_SCALE*2.0f;
    } while (--nsmps);
}

/**********************************************************/
/*  Algorithm 8 (TX81Z) Subclass of 4 Operator FM Synth   */
/*  by Perry R. Cook, 1995-96; recoded John ffitch 97-98  */
/*  This connection topology is simple Additive Synthesis */
/*                                                        */
/*                   1 --.                                */
/*                   2 -\|                                */
/*                       +-> Out                          */
/*                   3 -/|                                */
/*                   4 --                                 */
/*                                                        */
/*  Controls: control1 = op4 (fb) gain                    */
/*            control2 = op3 gain                         */
/*                                                        */
/**********************************************************/

float FM4Alg8_tick(FM4OP *p, float c1, float c2)
{
    float	temp;
    float	lastOutput;

    p->w_phase[3] = p->waves[3]->flen * p->twozero.lastOutput;

    temp = c1 * 2.0f * p->gains[3] * ADSR_tick(&p->adsr[3]) *
       Wave_tick(&p->w_time[3], (int)p->waves[3]->flen, p->waves[3]->ftable,
                 p->w_rate[3], p->w_phase[3]);
    TwoZero_tick(&p->twozero, temp);
    temp += c2 * 2.0f * p->gains[2] * ADSR_tick(&p->adsr[2]) * 
      Wave_tick(&p->w_time[2], (int)p->waves[2]->flen, p->waves[2]->ftable,
                p->w_rate[2], p->w_phase[2]);
    temp += p->gains[1] * ADSR_tick(&p->adsr[1]) * 
      Wave_tick(&p->w_time[1], (int)p->waves[1]->flen, p->waves[1]->ftable,
                   p->w_rate[1], p->w_phase[1]);
    temp += p->gains[0] * ADSR_tick(&p->adsr[0]) * 
      Wave_tick(&p->w_time[0], (int)p->waves[0]->flen, p->waves[0]->ftable,
                             p->w_rate[0], p->w_phase[0]);
    
    lastOutput = temp * 0.125f;
    return lastOutput;
}


/**************************************************************/
/*  Hammond(OID) Organ Subclass of Algorithm 8 (TX81Z)        */
/*  Subclass of 4 Operator FM Synth by Perry R. Cook, 1995-96 */ 
/*  Recoded in C by John ffitch 1997-98                       */
/**************************************************************/


void b3set(FM4OP *p)
{
    float	amp = *p->amp * AMP_RSCALE; /* Normalised */

    make_FM4Op(p);
    FM4Op_loadWaves(p);         /* sines */
    FM4Op_setRatio(p, 0, 0.999f);   
    FM4Op_setRatio(p, 1, 1.997f);
    FM4Op_setRatio(p, 2, 3.006f);
    FM4Op_setRatio(p, 3, 6.009f);

    p->gains[0] = amp * FM4Op_gains[95];
    p->gains[1] = amp * FM4Op_gains[95];
    p->gains[2] = amp * FM4Op_gains[99];
    p->gains[3] = amp * FM4Op_gains[95];
    ADSR_setAll(&p->adsr[0], 0.05f, 0.03f, 1.0f, 0.04f);
    ADSR_setAll(&p->adsr[1], 0.05f, 0.03f, 1.0f, 0.04f);
    ADSR_setAll(&p->adsr[2], 0.05f, 0.03f, 1.0f, 0.04f);
    ADSR_setAll(&p->adsr[3], 0.05f, 0.001f,0.4f, 0.06f);
    p->twozero.gain = 0.1f;
    ADSR_keyOn(&p->adsr[0]);
    ADSR_keyOn(&p->adsr[1]);
    ADSR_keyOn(&p->adsr[2]);
    ADSR_keyOn(&p->adsr[3]);
} 

void hammondB3(FM4OP *p)
{    
    float	amp = *p->amp * AMP_RSCALE; /* Normalised */
    float	*ar = p->ar;
    long	nsmps = ksmps;
    float	c1 = *p->control1;
    float	c2 = *p->control2;
    float	temp;

    p->baseFreq = *p->frequency;
    p->gains[0] = amp * FM4Op_gains[95];
    p->gains[1] = amp * FM4Op_gains[95];
    p->gains[2] = amp * FM4Op_gains[99];
    p->gains[3] = amp * FM4Op_gains[95];
    do {
        float	lastOutput;
        if (*p->modDepth > 0.0f) {
          p->v_rate = *p->vibFreq * p->vibWave->flen / esr;
          temp = 1.0f + (*p->modDepth * 0.1f *
                        Wave_tick(&p->v_time, (int)p->vibWave->flen,
                                  p->vibWave->ftable, p->v_rate, 0.0f));
          temp *= p->baseFreq;
          p->w_rate[0] = p->ratios[0] * temp * p->waves[0]->flen / esr;
          p->w_rate[1] = p->ratios[1] * temp * p->waves[1]->flen / esr;
          p->w_rate[2] = p->ratios[2] * temp * p->waves[2]->flen / esr;
          p->w_rate[3] = p->ratios[3] * temp * p->waves[3]->flen / esr;
        }
        lastOutput = FM4Alg8_tick(p, c1, c2);
        *ar++ = lastOutput*AMP_SCALE;
    } while (--nsmps);
}

/************************************************************/
/*  Algorithm 6 (TX81Z) Subclass of 4 Operator FM Synth     */
/*  by Perry R. Cook, 1995-96; recoded John ffitch 97-98    */ 
/*  This connection topology is three Carriers and a common */
/*  Modulator     /->1 -\                                   */
/*             4-|-->2 - +-> Out                            */
/*                \->3 -/                                   */
/*                                                          */
/*  Controls: control1 = vowel      	                    */
/*            control2 = spectral tilt                      */
/*                                                          */
/************************************************************/

float FM4Alg6_tick(FM4OPV *q)
{
    float	temp,temp2;
    FM4OP	*p = (FM4OP*)q;

    temp = p->gains[3] * ADSR_tick(&p->adsr[3]) *
      Wave_tick(&p->w_time[3], (int)p->waves[3]->flen, p->waves[3]->ftable,
                 p->w_rate[3], p->w_phase[3]);
                                /*  Calculate frequency mod  */
    temp2 = Wave_tick(&p->v_time, (int)p->vibWave->flen, p->vibWave->ftable,
                 p->v_rate, 0.0f) * *p->modDepth * 0.1f;

    temp2 = (1.0f + temp2) * p->baseFreq / esr;
    p->w_rate[0] = temp2 * p->ratios[0] * p->waves[0]->flen;
    p->w_rate[1] = temp2 * p->ratios[1] * p->waves[1]->flen;
    p->w_rate[2] = temp2 * p->ratios[2] * p->waves[2]->flen;
    p->w_rate[3] = temp2 * p->ratios[3] * p->waves[3]->flen;

    p->w_phase[0] = p->waves[0]->flen * temp * q->mods[0];
    p->w_phase[1] = p->waves[1]->flen * temp * q->mods[1];
    p->w_phase[2] = p->waves[2]->flen * temp * q->mods[2];
    p->w_phase[3] = p->waves[3]->flen * p->twozero.lastOutput;

    TwoZero_tick(&p->twozero, temp);

    temp =  p->gains[0] * q->tilt[0] * ADSR_tick(&p->adsr[0]) * 
      Wave_tick(&p->w_time[0], (int)p->waves[0]->flen, p->waves[0]->ftable,
                p->w_rate[0], p->w_phase[0]);
    temp += p->gains[1] * q->tilt[1] * ADSR_tick(&p->adsr[1]) * 
      Wave_tick(&p->w_time[1], (int)p->waves[1]->flen, p->waves[1]->ftable,
                p->w_rate[1], p->w_phase[1]);
    temp += p->gains[2] * q->tilt[2] * ADSR_tick(&p->adsr[2]) * 
      Wave_tick(&p->w_time[2], (int)p->waves[2]->flen, p->waves[2]->ftable,
                p->w_rate[2], p->w_phase[2]);
    
    return temp * 0.33f;
}

float phonGains[32][2] =
	{{1.0f, 0.0f},    /* eee */
         {1.0f, 0.0f},    /* ihh */
         {1.0f, 0.0f},    /* ehh */
         {1.0f, 0.0f},    /* aaa */
       
         {1.0f, 0.0f},    /* ahh */
         {1.0f, 0.0f},    /* aww */
         {1.0f, 0.0f},    /* ohh */
         {1.0f, 0.0f},    /* uhh */
      
         {1.0f, 0.0f},    /* uuu */
         {1.0f, 0.0f},    /* ooo */
         {1.0f, 0.0f},    /* rrr */
         {1.0f, 0.0f},    /* lll */

         {1.0f, 0.0f},    /* mmm */
         {1.0f, 0.0f},    /* nnn */
         {1.0f, 0.0f},    /* nng */
         {1.0f, 0.0f},    /* ngg */
      
         {0.0f, 1.0f},    /* fff */
         {0.0f, 1.0f},    /* sss */
         {0.0f, 1.0f},    /* thh */
         {0.0f, 1.0f},    /* shh */
      
         {0.0f, 1.0f},    /* xxx */
         {0.0f, 0.1f},    /* hee */
         {0.0f, 0.1f},    /* hoo */
         {0.0f, 0.1f},    /* hah */
      
         {1.0f, 0.1f},    /* bbb */
         {1.0f, 0.1f},    /* ddd */
         {1.0f, 0.1f},    /* jjj */
         {1.0f, 0.1f},    /* ggg */

         {1.0f, 1.0f},    /* vvv */
         {1.0f, 1.0f},    /* zzz */
         {1.0f, 1.0f},    /* thz */
         {1.0f, 1.0f}     /* zhh */
};         

float phonParams[32][4][3] =
    {{  { 273.0f, 0.996f,   0.0f},   /* eee (beet) */
        {2086.0f, 0.945f, -16.0f}, 
        {2754.0f, 0.979f, -12.0f}, 
        {3270.0f, 0.440f, -17.0f}},
     {  { 385.0f, 0.987f,  10.0f},   /* ihh (bit) */
	{2056.0f, 0.930f, -20.0f},
        {2587.0f, 0.890f, -20.0f}, 
        {3150.0f, 0.400f, -20.0f}},
     {  { 515.0f, 0.977f,  10.0f},   /* ehh (bet) */
        {1805.0f, 0.810f, -10.0f}, 
        {2526.0f, 0.875f, -10.0f}, 
        {3103.0f, 0.400f, -13.0f}},
     {  { 773.0f, 0.950f,  10.0f},   /* aaa (bat) */
        {1676.0f, 0.830f,  -6.0f},
        {2380.0f, 0.880f, -20.0f}, 
        {3027.0f, 0.600f, -20.0f}},
     
     {  { 770.0f, 0.950f,   0.0f},   /* ahh (father) */
        {1153.0f, 0.970f,  -9.0f},
        {2450.0f, 0.780f, -29.0f},
        {3140.0f, 0.800f, -39.0f}},
     {  { 637.0f, 0.910f,   0.0f},   /* aww (bought) */
        { 895.0f, 0.900f,  -3.0f},
        {2556.0f, 0.950f, -17.0f},
        {3070.0f, 0.910f, -20.0f}},
     {  { 637.0f, 0.910f,   0.0f},   /* ohh (bone)  NOTE::  same as aww (bought) */
        { 895.0f, 0.900f,  -3.0f},
        {2556.0f, 0.950f, -17.0f},
        {3070.0f, 0.910f, -20.0f}},
     {  { 561.0f, 0.965f,   0.0f},   /* uhh (but) */
        {1084.0f, 0.930f, -10.0f}, 
        {2541.0f, 0.930f, -15.0f}, 
        {3345.0f, 0.900f, -20.0f}},
    
     {  { 515.0f, 0.976f,   0.0f},   /* uuu (foot) */
        {1031.0f, 0.950f,  -3.0f},
        {2572.0f, 0.960f, -11.0f},
        {3345.0f, 0.960f, -20.0f}},
     {  { 349.0f, 0.986f, -10.0f},   /* ooo (boot) */
        { 918.0f, 0.940f, -20.0f},
        {2350.0f, 0.960f, -27.0f},
        {2731.0f, 0.950f, -33.0f}},
     {  { 394.0f, 0.959f, -10.0f},   /* rrr (bird) */
        {1297.0f, 0.780f, -16.0f},
        {1441.0f, 0.980f, -16.0f},
        {2754.0f, 0.950f, -40.0f}},
     {  { 462.0f, 0.990f,  +5.0f},   /* lll (lull) */
        {1200.0f, 0.640f, -10.0f},
        {2500.0f, 0.200f, -20.0f},
        {3000.0f, 0.100f, -30.0f}},
     
     {  { 265.0f, 0.987f, -10.0f},   /* mmm (mom) */
        {1176.0f, 0.940f, -22.0f},
        {2352.0f, 0.970f, -20.0f},
        {3277.0f, 0.940f, -31.0f}},
     {  { 204.0f, 0.980f, -10.0f},   /* nnn (nun) */
        {1570.0f, 0.940f, -15.0f},
        {2481.0f, 0.980f, -12.0f},
        {3133.0f, 0.800f, -30.0f}},
     {  { 204.0f, 0.980f, -10.0f},   /* nng (sang)    NOTE:: same as nnn */
        {1570.0f, 0.940f, -15.0f},
        {2481.0f, 0.980f, -12.0f},
        {3133.0f, 0.800f, -30.0f}},
     {  { 204.0f, 0.980f, -10.0f},   /* ngg (bong)    NOTE:: same as nnn */
        {1570.0f, 0.940f, -15.0f},
        {2481.0f, 0.980f, -12.0f},
        {3133.0f, 0.800f, -30.0f}},
     
     {  {1000.0f, 0.300f,   0.0f},   /* fff */
        {2800.0f, 0.860f, -10.0f},
        {7425.0f, 0.740f,   0.0f},
        {8140.0f, 0.860f,   0.0f}},
     {  {0.0f, 0.000f,   0.0f},      /* sss */
        {2000.0f, 0.700f, -15.0f},
        {5257.0f, 0.750f,  -3.0f}, 
        {7171.0f, 0.840f,   0.0f}},
     {  { 100.0f, 0.900f,   0.0f},   /* thh */
        {4000.0f, 0.500f, -20.0f},
        {5500.0f, 0.500f, -15.0f},
        {8000.0f, 0.400f, -20.0f}},
     {  {2693.0f, 0.940f,   0.0f},   /* shh */
        {4000.0f, 0.720f, -10.0f},
        {6123.0f, 0.870f, -10.0f},
        {7755.0f, 0.750f, -18.0f}},

     {  {1000.0f, 0.300f, -10.0f},   /* xxx           NOTE:: Not Really Done Yet */
        {2800.0f, 0.860f, -10.0f},
        {7425.0f, 0.740f,   0.0f},
        {8140.0f, 0.860f,   0.0f}},
     {  { 273.0f, 0.996f, -40.0f},   /* hee (beet)    (noisy eee) */
        {2086.0f, 0.945f, -16.0f}, 
        {2754.0f, 0.979f, -12.0f}, 
        {3270.0f, 0.440f, -17.0f}},
     {  { 349.0f, 0.986f, -40.0f},   /* hoo (boot)    (noisy ooo) */
        { 918.0f, 0.940f, -10.0f},
        {2350.0f, 0.960f, -17.0f},
        {2731.0f, 0.950f, -23.0f}},
     {  { 770.0f, 0.950f, -40.0f},   /* hah (father)  (noisy ahh) */
        {1153.0f, 0.970f,  -3.0f},
        {2450.0f, 0.780f, -20.0f},
        {3140.0f, 0.800f, -32.0f}},
     
     {  {2000.0f, 0.700f, -20.0f},   /* bbb           NOTE:: Not Really Done Yet */
        {5257.0f, 0.750f, -15.0f},
        {7171.0f, 0.840f,  -3.0f}, 
        {9000.0f, 0.900f,   0.0f}},
     {  { 100.0f, 0.900f,   0.0f},   /* ddd           NOTE:: Not Really Done Yet */
        {4000.0f, 0.500f, -20.0f},
        {5500.0f, 0.500f, -15.0f},
        {8000.0f, 0.400f, -20.0f}},
     {  {2693.0f, 0.940f,   0.0f},   /* jjj           NOTE:: Not Really Done Yet */
        {4000.0f, 0.720f, -10.0f},
        {6123.0f, 0.870f, -10.0f},
        {7755.0f, 0.750f, -18.0f}},
     {  {2693.0f, 0.940f,   0.0f},   /* ggg           NOTE:: Not Really Done Yet */
        {4000.0f, 0.720f, -10.0f},
        {6123.0f, 0.870f, -10.0f},
        {7755.0f, 0.750f, -18.0f}},
     
     {  {2000.0f, 0.700f, -20.0f},   /* vvv           NOTE:: Not Really Done Yet */
        {5257.0f, 0.750f, -15.0f},
        {7171.0f, 0.840f,  -3.0f}, 
        {9000.0f, 0.900f,   0.0f}},
     {  { 100.0f, 0.900f,   0.0f},   /* zzz           NOTE:: Not Really Done Yet */
        {4000.0f, 0.500f, -20.0f},
        {5500.0f, 0.500f, -15.0f},
        {8000.0f, 0.400f, -20.0f}},
     {  {2693.0f, 0.940f,   0.0f},   /* thz           NOTE:: Not Really Done Yet */
        {4000.0f, 0.720f, -10.0f},
        {6123.0f, 0.870f, -10.0f},
        {7755.0f, 0.750f, -18.0f}},
     {  {2693.0f, 0.940f,   0.0f},   /* zhh           NOTE:: Not Really Done Yet */
        {4000.0f, 0.720f, -10.0f},
        {6123.0f, 0.870f, -10.0f},
        {7755.0f, 0.750f, -18.0f}}
};

#define currentVowel (*q->control1)

void FMVoices_setFreq(FM4OPV *q, float frequency)
{    
    float	temp,temp2;
    int		tempi,tempi2;

    if (currentVowel < 16)	{
	tempi2 = (int)currentVowel;
	temp2 = 0.9f;
    }
    else if (currentVowel < 32)	{
	tempi2 =(int) currentVowel - 16;
	temp2 = 1.0f;
    }
    else if (currentVowel < 48)	{
	tempi2 = (int)currentVowel - 32;
	temp2 = 1.1f;
    }
    else if (currentVowel < 64)	{
	tempi2 = (int)currentVowel - 48;
	temp2 = 1.2f;
    }
    q->baseFreq = frequency;
    temp = (temp2 * phonParams[tempi2][0][0] / q->baseFreq) + 0.5f;
    tempi = (int) temp;
    FM4Op_setRatio((FM4OP*)q, 0, (float) tempi);
    temp = (temp2 * phonParams[tempi2][1][0] / q->baseFreq) + 0.5f;
    tempi = (int) temp;
    FM4Op_setRatio((FM4OP*)q, 1, (float) tempi);
    temp = (temp2 * phonParams[tempi2][2][0] / q->baseFreq) + 0.5f;
    tempi = (int) temp;
    FM4Op_setRatio((FM4OP*)q, 2, (float) tempi);    
    q->gains[0] = 1.0f;  /* pow(10.0f,phonParams[tempi2][0][2] * 0.05f); */
    q->gains[1] = 1.0f;  /* pow(10.0f,phonParams[tempi2][1][2] * 0.05f); */
    q->gains[2] = 1.0f;  /* pow(10.0f,phonParams[tempi2][2][2] * 0.05f); */
}

void FMVoiceset(FM4OPV *q)
{
    FM4OP	*p = (FM4OP *)q;
    float	amp = *q->amp * AMP_RSCALE;

    make_FM4Op(p);
    FM4Op_loadWaves(p);
    FM4Op_setRatio(p, 0, 2.00f);
    FM4Op_setRatio(p, 1, 4.00f);
    FM4Op_setRatio(p, 2, 12.0f);
    FM4Op_setRatio(p, 3, 1.00f);
    p->gains[3] = FM4Op_gains[80];
    ADSR_setAll(&p->adsr[0], 0.001f, 0.001f, FM4Op_susLevels[15], 0.001f);
    ADSR_setAll(&p->adsr[1], 0.001f, 0.001f, FM4Op_susLevels[15], 0.001f);
    ADSR_setAll(&p->adsr[2], 0.001f, 0.001f, FM4Op_susLevels[15], 0.001f);
    ADSR_setAll(&p->adsr[3], 0.05f,  0.05f,  FM4Op_susLevels[15], 0.0001f);
    p->twozero.gain = 0.0f;
/*     modDepth = 0.005; */
    q->tilt[0] = 1.0f;
    q->tilt[1] = 0.5f;
    q->tilt[2] = 0.2f;    
    q->mods[0] = 1.0f;
    q->mods[1] = 1.1f;
    q->mods[2] = 1.1f;
    p->baseFreq = 110.0f;
    FMVoices_setFreq(q, 110.0f);    
    q->tilt[0] = amp;
    q->tilt[1] = amp * amp;
    q->tilt[2] = amp * amp * amp;
    ADSR_keyOn(&p->adsr[0]);
    ADSR_keyOn(&p->adsr[1]);
    ADSR_keyOn(&p->adsr[2]);
    ADSR_keyOn(&p->adsr[3]);
    q->last_control = -1.0f;
}

void FMVoice(FM4OPV *q)
{
    FM4OP	*p = (FM4OP *)q;
    float	amp = *q->amp * AMP_RSCALE;
    float	*ar = q->ar;
    long	nsmps = ksmps;

    if (p->baseFreq != *q->frequency || *q->control1 != q->last_control) {
      q->last_control = *q->control1;
      p->baseFreq = *q->frequency;
      FMVoices_setFreq(q, p->baseFreq);
    }
    q->tilt[0] = amp;
    q->tilt[1] = amp * amp;
    q->tilt[2] = amp * amp * amp;
    p->gains[3] = FM4Op_gains[(int) (*p->control2 * 0.78125f)];

    do {
        float	lastOutput;
        lastOutput = FM4Alg6_tick(q);
        *ar++ = lastOutput*AMP_SCALE*0.8f;
    } while (--nsmps);
    
}

/* ********************************************************************** */

/*********************************************************/
/*  Algorithm 4 (TX81Z) Subclass of 4 Operator FM Synth  */
/*  by Perry R. Cook, 1995-96  Recoded John ffitch 97/98 */ 
/*                                                       */
/*  Alg 4 is :      4->3--\                              */
/*                     2-- + -->1-->Out                  */
/*                                                       */
/*  Controls: control1 = total mod index                 */
/*            control2 = crossfade of two                */
/*                          modulators                   */
/*                                                       */
/*********************************************************/

float FM4Alg4_tick(FM4OP *p, float c1, float c2)
{
    float	temp;
    float	lastOutput;

    temp = Wave_tick(&p->v_time, (int)p->vibWave->flen,
                     p->vibWave->ftable, p->v_rate, 0.0f) * 
      *p->modDepth * 0.2f;   
    temp = p-> baseFreq * (1.0f + temp)/ esr;
    p->w_rate[0] = p->ratios[0] * temp * p->waves[0]->flen;
    p->w_rate[1] = p->ratios[1] * temp * p->waves[1]->flen;
    p->w_rate[2] = p->ratios[2] * temp * p->waves[2]->flen;
    p->w_rate[3] = p->ratios[3] * temp * p->waves[3]->flen;
    
    p->w_phase[3] = p->waves[3]->flen * p->twozero.lastOutput;
    temp = p->gains[3] * ADSR_tick(&p->adsr[3]) * 
         Wave_tick(&p->w_time[3], (int)p->waves[3]->flen, p->waves[3]->ftable,
                   p->w_rate[3], p->w_phase[3]);
    TwoZero_tick(&p->twozero, temp);
    p->w_phase[2] = p->waves[2]->flen * temp;
    temp = (1.0f - (c2 * 0.5f)) * p->gains[2] * ADSR_tick(&p->adsr[2]) * 
      Wave_tick(&p->w_time[2], (int)p->waves[2]->flen, p->waves[2]->ftable,
                p->w_rate[2], p->w_phase[2]);
    temp += c2 * 0.5f * p->gains[1] * ADSR_tick(&p->adsr[1]) * 
         Wave_tick(&p->w_time[1], (int)p->waves[1]->flen, p->waves[1]->ftable,
                   p->w_rate[1], p->w_phase[1]);
    temp = temp * c1;
    p->w_phase[0] = p->waves[0]->flen * temp;
    temp = p->gains[0] * ADSR_tick(&p->adsr[0]) * 
         Wave_tick(&p->w_time[0], (int)p->waves[0]->flen, p->waves[0]->ftable,
                   p->w_rate[0], p->w_phase[0]);
    
    lastOutput = temp * 0.5f;
    return lastOutput;
}


void percfluteset(FM4OP *p)
{
    float	amp = *p->amp * AMP_RSCALE; /* Normalised */

    make_FM4Op(p);
    FM4Op_loadWaves(p);         /* sines */
    
    FM4Op_setRatio(p, 0, 1.50f     );   
    FM4Op_setRatio(p, 1, 3.00f * 0.995f);
    FM4Op_setRatio(p, 2, 2.99f * 1.005f);
    FM4Op_setRatio(p, 3, 6.00f * 0.997f);

    p->gains[0] = amp * FM4Op_gains[99];
    p->gains[1] = amp * FM4Op_gains[71];
    p->gains[2] = amp * FM4Op_gains[93];
    p->gains[3] = amp * FM4Op_gains[85];
    ADSR_setAll(&p->adsr[0], 0.001f, 0.001f,  FM4Op_susLevels[14], 0.001f);
    ADSR_setAll(&p->adsr[1], 0.05f,  0.0001f, FM4Op_susLevels[13], 0.0001f);
    ADSR_setAll(&p->adsr[2], 0.05f,  0.0020f, FM4Op_susLevels[11], 0.001f);
    ADSR_setAll(&p->adsr[3], 0.05f,  0.0010f, FM4Op_susLevels[13], 0.005f);
    p->twozero.gain = 0.0f;
/*     modDepth = 0.005f; */
    ADSR_keyOn(&p->adsr[0]);
    ADSR_keyOn(&p->adsr[1]);
    ADSR_keyOn(&p->adsr[2]);
    ADSR_keyOn(&p->adsr[3]);
}  


void percflute(FM4OP *p)
{
    float	*ar = p->ar;
    long	nsmps = ksmps;
    float	amp = *p->amp * AMP_RSCALE; /* Normalised */
    float	c1 = *p->control1;
    float	c2 = *p->control2;

    p->baseFreq = *p->frequency;
    p->gains[0] = amp * FM4Op_gains[99];
    p->gains[1] = amp * FM4Op_gains[71];
    p->gains[2] = amp * FM4Op_gains[93];
    p->gains[3] = amp * FM4Op_gains[85];
    do {
        float	lastOutput;
        lastOutput = FM4Alg4_tick(p, c1, c2);
        *ar++ = lastOutput*AMP_SCALE*2.0f;
    } while (--nsmps);
}
