/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTPFileModule.cpp

	Contains:	Implementation of module described in RTPFileModule.h. 
					
	$Log: RTPFileModule.cpp,v $
	Revision 1.2  1999/02/19 23:07:22  ds
	Created
	

*/

#include <string.h>

#include "RTPFileModule.h"
#include "RTPStream.h"

#include "RTSPProtocol.h"
#include "RTSPModule.h"

#include "OS.h"

StrPtrLen RTPFileModule::sSDPSuffix(".sdp", 4);
StrPtrLen RTPFileModule::sSDPHeader1("v=0\r\ns=", 7);
StrPtrLen RTPFileModule::sSDPHeader2;
StrPtrLen RTPFileModule::sSDPHeader3("c=IN IP4 ", 9);
StrPtrLen RTPFileModule::sSDPHeader4("\r\na=control:/\r\n", 15);

//StrPtrLen RTPFileModule::sSDPHeader("v=0\r\ns=test.mov\r\nc=IN IP4 17.221.41.108\r\na=control:/\r\n", 54);

RTPFileModule::RTPFileModule() : 	RTPModule(kRequestProcessingModule, "RTPFileModule")
{
	static StrPtrLen sEHeader("\r\ne=", 4);
	static StrPtrLen sUHeader("\r\nu=", 4);
	static StrPtrLen sHTTP("http://", 7);
	static StrPtrLen sAdmin("admin@", 6);
	
	//build the sdp that looks like: \r\ne=http://streaming.apple.com\r\ne=qts@apple.com\r\n.
	OSMutexLocker locker(RTSPServerInterface::GetRTSPPrefs()->GetMutex());
	StrPtrLen* theSDPURL = RTSPServerInterface::GetRTSPPrefs()->GetSDPURL();
	StrPtrLen* theAdminEmail = RTSPServerInterface::GetRTSPPrefs()->GetAdminEmail();
	StrPtrLen* theDefaultDNS = RTSPServerInterface::GetDefaultDNSName();
	
	UInt32 sdpURLLen = theSDPURL->Len;
	UInt32 adminEmailLen = theAdminEmail->Len;
	
	if (sdpURLLen == 0)
		sdpURLLen = theDefaultDNS->Len + sHTTP.Len + 1;
	if (adminEmailLen == 0)
		adminEmailLen = theDefaultDNS->Len + sAdmin.Len;	
	
	//calculate the length of the string & allocate memory
	sSDPHeader2.Len = (sEHeader.Len * 2) + sdpURLLen + adminEmailLen + 10;
	sSDPHeader2.Ptr = new char[sSDPHeader2.Len];

	//write it!
	StringFormatter sdpFormatter(sSDPHeader2);
	sdpFormatter.Put(sUHeader);

	//if there are preferences for default URL & admin email, use those. Otherwise, build the
	//proper string based on default dns name.
	if (theSDPURL->Len == 0)
	{
		sdpFormatter.Put(sHTTP);
		sdpFormatter.Put(*theDefaultDNS);
		sdpFormatter.PutChar('/');
	}
	else
		sdpFormatter.Put(*theSDPURL);
	
	sdpFormatter.Put(sEHeader);
	
	//now do the admin email.
	if (theAdminEmail->Len == 0)
	{
		sdpFormatter.Put(sAdmin);
		sdpFormatter.Put(*theDefaultDNS);
	}
	else
		sdpFormatter.Put(*theAdminEmail);
		
	sdpFormatter.PutEOL();
	sSDPHeader2.Len = (UInt32)sdpFormatter.GetCurrentOffset();
}

RTSPProtocol::RTSPStatusCode RTPFileModule::Describe(RTSPRequestInterface* inRequest, void** outCookie)
{
	//check to see if there is an SDP file on disk. If so, simply use that
	OSCharArrayDeleter thePath(inRequest->GetFullPath(qtssFilePathParam, &sSDPSuffix));

	//first locate the target movie
	UInt32 thePathLen = ::strlen(thePath.GetObject());
	thePath.GetObject()[thePathLen - sSDPSuffix.Len] = '\0';//truncate the .sdp
	
	FileSession* theFile = NULL;
	RTSPProtocol::RTSPStatusCode theError = this->CreateQTRTPFile(inRequest, thePath.GetObject(), &theFile);
	if (theError != RTSPProtocol::kSuccessOK)
		return theError;
	//set the cookie to be the file object. This way, we can keep this
	//file object around, and not have to reinitialize it later
	*outCookie = theFile;

	//replace the sacred character we have trodden on in order to truncate the path.
	thePath.GetObject()[thePathLen - sSDPSuffix.Len] = sSDPSuffix.Ptr[0];

	iovec theSDPVec[10];//1 for the RTSP header, 6 for the sdp header, 1 for the sdp body
	::memset(&theSDPVec[0], 0, sizeof(theSDPVec));
	
	//track the relevent portions of the sdp for parsing
	char* theSDPDataPtr = NULL;
	UInt32 theSDPDataLen = 0;
	
	//If there is an SDP file on disk, return it to the client
	OSFileSource theSDPFile(thePath.GetObject());
	if (theSDPFile.GetLength() > 0)
	{
		//sdpData is also one of these deleter objects, so it will get freed automatically
		theSDPDataPtr = new ('sdpf') char[theSDPFile.GetLength() + 2];
		
		UInt32 recvLen = 0;
		QTSS_ErrorCode err = theSDPFile.Read(theSDPDataPtr, theSDPFile.GetLength(), &recvLen);
		if ((err != QTSS_NoErr) || (recvLen != theSDPFile.GetLength()))
			AssertV(0, (int)err);//can we ever get here?

		//Now that we have the file data, send an appropriate describe
		//response to the client
		theSDPVec[1].iov_base = theSDPDataPtr;
		theSDPDataLen = theSDPVec[1].iov_len = theSDPFile.GetLength();
		RTPServerInterface::SendDescribeResponse(inRequest, &theSDPVec[0], 2, theSDPFile.GetLength());	
	}
	else
	{
		//if there is no sdp file on disk, let's generate one.
		UInt32 totalSDPLength = sSDPHeader1.Len;
		theSDPVec[1].iov_base = sSDPHeader1.Ptr;
		theSDPVec[1].iov_len = sSDPHeader1.Len;
		
		//filename goes here
		StrPtrLen* theFileName = inRequest->GetQTSSParameter(qtssFileNameParam);
		theSDPVec[2].iov_base = theFileName->Ptr;
		theSDPVec[2].iov_len = theFileName->Len;
		totalSDPLength += theFileName->Len;
		
		//url & admin email goes here
		theSDPVec[3].iov_base = sSDPHeader2.Ptr;
		theSDPVec[3].iov_len = sSDPHeader2.Len;
		totalSDPLength += sSDPHeader2.Len;

		//connection header
		theSDPVec[4].iov_base = sSDPHeader3.Ptr;
		theSDPVec[4].iov_len = sSDPHeader3.Len;
		totalSDPLength += sSDPHeader3.Len;
		
		//append IP addr
		StrPtrLen* theIPAddr = inRequest->GetSession()->GetSocket()->GetLocalAddrStr();
		theSDPVec[5].iov_base = theIPAddr->Ptr;
		theSDPVec[5].iov_len = theIPAddr->Len;
		totalSDPLength += theIPAddr->Len;

		//last static sdp line
		theSDPVec[6].iov_base = sSDPHeader4.Ptr;
		theSDPVec[6].iov_len = sSDPHeader4.Len;
		totalSDPLength += sSDPHeader4.Len;
		
		//now append content-determined sdp
		int theSDPBodyLen = 0;
		theSDPVec[7].iov_base = theSDPDataPtr = theFile->fFile.GetSDPFile(&theSDPBodyLen);
		theSDPDataLen = theSDPVec[7].iov_len = theSDPBodyLen;
		totalSDPLength += theSDPBodyLen;
		
		//in this particular case, we do not want the SDP parser to be the eventual
		//owner of this memory, because QTFile owns it.
		theFile->fSDPParser.DontDeleteSDP();
		
		Assert(theSDPBodyLen > 0);
		Assert(theSDPVec[2].iov_base != NULL);
		//ok, we have a filled out iovec. Let's send it!
		RTPServerInterface::SendDescribeResponse(inRequest, &theSDPVec[0], 8, totalSDPLength);
	}
	
	Assert(theSDPDataPtr != NULL);
	Assert(theSDPDataLen > 0);
	
	//now parse the sdp. We need to do this in order to extract codec information.
	//The SDP parser object will now take responsibility of the memory (one exception... see above)
	theFile->fSDPParser.Parse(theSDPDataPtr, theSDPDataLen);
		
	return RTSPProtocol::kSuccessOK;
}

RTSPProtocol::RTSPStatusCode RTPFileModule::CreateQTRTPFile(RTSPRequestInterface* inRTSPRequest, char* inPath, FileSession** outFile)
{	
	*outFile = new ('qtfl') FileSession();
	QTRTPFile::ErrorCode theErr = (*outFile)->fFile.Initialize(inPath);
	if (theErr != QTRTPFile::errNoError)
	{
		delete *outFile;
		*outFile = NULL;
		
		if (theErr == QTRTPFile::errFileNotFound)
			return inRTSPRequest->SendErrorResponse(RTSPProtocol::kClientNotFound,
													RTSPMessages::kNoSDPFileFound);
		if (theErr == QTRTPFile::errInvalidQuickTimeFile)
			return inRTSPRequest->SendErrorResponse(RTSPProtocol::kUnsupportedMediaType,
													RTSPMessages::kBadQTFile);
		if (theErr == QTRTPFile::errNoHintTracks)
			return inRTSPRequest->SendErrorResponse(RTSPProtocol::kUnsupportedMediaType,
													RTSPMessages::kFileIsNotHinted);
		if (theErr == QTRTPFile::errInternalError)
			return inRTSPRequest->SendErrorResponse(RTSPProtocol::kServerInternal,
													RTSPMessages::kBadQTFile);
		AssertV(0, theErr);
	}
	return RTSPProtocol::kSuccessOK;
}

RTSPProtocol::RTSPStatusCode RTPFileModule::NewSession(RTSPRequestInterface* /*inRequest*/,
										RTPSession* inSessionP, void* inCookie)
{
	//if this session belongs to us, we must have a cookie for it.
	if (inCookie == NULL)
		return NULL;
		
	//All we need to do at this point is link the file object created in the describe
	//with the newly created session. We specify 0 for inNumQualityLevels, because there is
	//no maximum number of quality levels for this module. 0 means infinity
	inSessionP->Bind(this, inCookie, 0);
	
	return RTSPProtocol::kSuccessOK;
}

RTSPProtocol::RTSPStatusCode RTPFileModule::ProcessRTSPRequest(RTSPRequestInterface* inRequest,
													RTPSession* inSession)
{
	//once we've bound a session, we'll start getting invoked here for RTSP requests.
	Assert(NULL != inRequest);

	switch (inRequest->GetMethod())
	{
		case RTSPProtocol::kSetupMethod:
			return this->DoSetup(inRequest, inSession);
		case RTSPProtocol::kPlayMethod:
			return this->DoPlay(inRequest, inSession);
		case RTSPProtocol::kTeardownMethod:
			inSession->SendTeardownResponse(inRequest);
			inSession->Signal(Task::kKillEvent);
			break;
		case RTSPProtocol::kPauseMethod:
			inSession->Pause();
			inSession->SendPauseResponse(inRequest);
		default:
			break;
	}			
	return RTSPProtocol::kSuccessOK;
}

QTSS_ErrorCode	RTPFileModule::GetParameter(QTSS_ParamKeywords inParam,
												RTPSession* inSession,
												RTPStream* inStream,
												void* outResult,
												UInt32* ioLen)
{
	Assert(inSession != NULL);
	FileSession* theFile = (FileSession*)inSession->GetMediaSrcRef();
	Assert(theFile != NULL);

	switch (inParam)
	{
		case qtssFirstSequenceNumber:
		{
			Assert(theFile != NULL);
			Assert(*ioLen == sizeof(SInt16));
			SInt16 theFirstSeq = theFile->fFile.GetNextTrackSequenceNumber(inStream->GetStreamID());
			::memcpy(outResult, &theFirstSeq, sizeof(SInt16));
			return QTSS_NoErr;
		}
		case qtssTimescale:
		{
			Assert(theFile != NULL);
			Assert(inStream != NULL);
			Assert(*ioLen == sizeof(SInt32));
			SInt32 theTimescale = theFile->fFile.GetTrackTimeScale(inStream->GetStreamID());
			::memcpy(outResult, &theTimescale, sizeof(SInt32));
			return QTSS_NoErr;
		}
		case qtssFirstTimestamp:
		{
			Assert(theFile != NULL);
			Assert(inStream != NULL);
			Assert(*ioLen == sizeof(SInt32));
			SInt32 theTimescale = theFile->fFile.GetSeekTimestamp(inStream->GetStreamID());
			::memcpy(outResult, &theTimescale, sizeof(SInt32));
			return QTSS_NoErr;
		}
		default:
			break;
	}			
	return QTSS_BadArgument;	
}

SInt64 RTPFileModule::SendPackets(RTPSession* inSession)
{
	Assert(inSession != NULL);
	FileSession* theFile = (FileSession*)inSession->GetMediaSrcRef();
	Assert(theFile != NULL);
	
	SInt64 curTime = OS::Milliseconds();
	
	while (true)
	{
		if (theFile->fNextPacket == NULL)
		{
			void* theCookie = NULL;
			Float64 theTransmitTime = theFile->fFile.GetNextPacket(&theFile->fNextPacket,
																	&theFile->fNextPacketLen, &theCookie);
#if RTPSESSION_LOGGING
	if (inSession->GetDebugFile() != NULL)
		if( theFile->fNextPacket == NULL )
			fprintf(inSession->GetDebugFile(), "END OF STREAM\n");
		else
			fprintf(inSession->GetDebugFile(), "Got packet with transmit time %.2f and RTP timestamp of %lu\n",theTransmitTime,*(UInt32 *)((char *)theFile->fNextPacket + 4));
#endif
#if RTP_FILE_MODULE_DEBUGGING
			float x = (float)theTransmitTime;
			printf("theTransmitTime = %f\n", x);
#endif
			theFile->fStream = (RTPStream*)theCookie;
			theFile->fPacketPlayTime = inSession->GetAdjustedPlayTime() + ((SInt64)(theTransmitTime * 1000));
		}
		
		//We are done playing all streams!
		if (theFile->fNextPacket == NULL)
			return -1;
		
		//If this packet isn't supposed to be sent now, tell the server when we next need to run
		SInt64 theRelativePacketTime = theFile->fPacketPlayTime - curTime;
		if (theRelativePacketTime > RTSPServerInterface::GetRTSPPrefs()->GetMaxAdvanceSendTimeInMilSecs())
			return theRelativePacketTime;
			
		//we have a packet that needs to be sent now
		Assert(theFile->fStream != NULL);
#if RTP_FILE_MODULE_DEBUGGING
		bool foundStream = false;
		for (OSQueueIter iter(inSession->GetStreamQueue()); !iter.IsDone(); iter.Next())
			if (iter.GetCurrent()->GetEnclosingObject() == theFile->fStream)
				foundStream = true;
		Assert(foundStream);
		printf("Sending a packet for cookie %ld. Length %ld. TrackID = %ld\n",(UInt32)theFile->fStream, theFile->fNextPacketLen, theFile->fStream->GetStreamID());
#endif

		//If the stream is video, we need to make sure that QTRTPFile knows what quality level we're at
		RTSPPrefs* thePrefs = RTSPServerInterface::GetRTSPPrefs();
		if ((!thePrefs->GetKeyFramesOnly()) && (!thePrefs->GetNoBFrames()) && 
				(theFile->fStream->GetCodecType() == RTPStream::kVideoCodecType))
			theFile->fFile.SetTrackQualityLevel(theFile->fStream->GetStreamID(), theFile->fStream->GetQualityLevel());
		
		theFile->fStream->SendRTP(theFile->fNextPacket, theFile->fNextPacketLen);
		theFile->fNextPacket = NULL;
	}
	Assert(0);
	return 0;
}

RTSPProtocol::RTSPStatusCode RTPFileModule::DoSetup(RTSPRequestInterface* inRequest, RTPSession* inSession)
{
	//unless there is a digit at the end of this path (representing trackID), don't
	//even bother with the request
	StrPtrLen* theDigit = inRequest->GetQTSSParameter(qtssFileDigitParam);
	if (theDigit->Len == 0)
		return inRequest->SendErrorResponse(RTSPProtocol::kClientBadRequest,
										RTSPMessages::kExpectedDigitFilename,
										inRequest->GetQTSSParameter(qtssURLParam));
		
	UInt32 theTrackID = ::strtol(theDigit->Ptr, NULL, 10);
	
	//setup this track in the file object 
	FileSession* theFile = (FileSession*)inSession->GetMediaSrcRef();
	QTRTPFile::ErrorCode theErr = theFile->fFile.AddTrack(theTrackID, RTSPServerInterface::GetRTSPPrefs()->GetAppendRandomOffsets());
	
	//if we get an error back, forward that error to the client
	if (theErr == QTRTPFile::errTrackIDNotFound)
		return inRequest->SendErrorResponse(RTSPProtocol::kClientNotFound, RTSPMessages::kTrackDoesntExist);
	else if (theErr != QTRTPFile::errNoError)
		return inRequest->SendErrorResponse(RTSPProtocol::kUnsupportedMediaType, RTSPMessages::kBadQTFile);

	//find the codec for this track ID (if applicable)
	StrPtrLen* theCodec = NULL;
	UInt32 theCodecType = RTPStream::kUnknownCodecType;
	
	for (UInt32 x = 0; x < theFile->fSDPParser.GetNumStreams(); x++)
	{
		SDPParser::StreamInfo* theStreamInfo = theFile->fSDPParser.GetStreamInfo(x);
		if (theStreamInfo->fTrackID == theTrackID)
		{
			theCodec = &theStreamInfo->fCodecName;
			theCodecType = theStreamInfo->fCodecType;
		}	
	}
	//Create a new RTP stream			
	RTPStream* newStream = NULL;
	RTSPProtocol::RTSPStatusCode error = inSession->AddStream(theTrackID, theCodec, theCodecType, inRequest, &newStream);
	if (error != RTSPProtocol::kSuccessOK)
		return error;
	
	//give the file some info it needs.
	theFile->fFile.SetTrackSSRC(theTrackID, newStream->GetSSRC());
	theFile->fFile.SetTrackCookie(theTrackID, newStream);
	
	//send the setup response
	newStream->AppendTransport(inRequest);
	inRequest->SendHeader();
	
	return RTSPProtocol::kSuccessOK;
}

RTSPProtocol::RTSPStatusCode RTPFileModule::DoPlay(RTSPRequestInterface* inRequest, RTPSession* inSession)
{
	//Have the file seek to the proper time.
	FileSession* theFile = (FileSession*)inSession->GetMediaSrcRef();
#if RTPSESSION_LOGGING
	if (inSession->GetDebugFile() != NULL)
		fprintf(inSession->GetDebugFile(), "Seeking to time %.2f\n",inRequest->GetStartTime());
#endif
	QTRTPFile::ErrorCode theErr = theFile->fFile.Seek(inRequest->GetStartTime());
	if (theErr != QTRTPFile::errNoError)
		return inRequest->SendErrorResponse(RTSPProtocol::kClientBadRequest, RTSPMessages::kSeekToNonexistentTime);
	
	//make sure to clear the next packet the server would have sent!
	theFile->fNextPacket = NULL;
	
	//Tell the server to start playing this movie. We do want it to send RTCP SRs, but
	//we DON'T want it to write the RTP header
	inSession->Play(inRequest, true, false);

	inSession->SendPlayResponse(inRequest);
	
	Float64 movieDuration = theFile->fFile.GetMovieDuration();
	inSession->GetSessionDictionary()->SetStandardValue(SessionDictionary::kMovieDuration, &movieDuration, sizeof(movieDuration));
	
	UInt64 movieSize = theFile->fFile.GetAddedTracksRTPBytes();
	inSession->GetSessionDictionary()->SetStandardValue(SessionDictionary::kMovieSizeInBytes, &movieSize, sizeof(movieSize));
	
	if (RTSPServerInterface::GetRTSPPrefs()->GetKeyFramesOnly())
		this->SetTrackQuality(inSession, &theFile->fFile, QTRTPFile::kKeyFramesOnly);
	if (RTSPServerInterface::GetRTSPPrefs()->GetNoBFrames())
		this->SetTrackQuality(inSession, &theFile->fFile, QTRTPFile::kNoBFrames);

	return RTSPProtocol::kSuccessOK;
}

void RTPFileModule::SetTrackQuality(RTPSession* inSession, QTRTPFile* inFile, UInt32 inQuality)
{
	for (OSQueueIter theIter(inSession->GetStreamQueue()); !theIter.IsDone(); theIter.Next())
	{
		RTPStream* theStream = (RTPStream*)theIter.GetCurrent()->GetEnclosingObject();
		if (theStream->GetCodecType() == RTPStream::kVideoCodecType)
			inFile->SetTrackQualityLevel(theStream->GetStreamID(), inQuality);
	}
}

void RTPFileModule::DestroySession(RTPSession* inSession)
{
	FileSession* theFile = (FileSession*)inSession->GetMediaSrcRef();
	Assert(theFile != NULL);
	delete theFile;
}

void RTPFileModule::DestroyCookie(void* inCookie)
{
	FileSession* theFile = (FileSession*)inCookie;
	Assert(theFile != NULL);
	delete theFile;	
}
