/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */

#ifndef __MW_
	#if __MacOSX__
		#include <mach/mach.h>
		#include <mach/cthreads.h>
	#endif
#endif

#include "mymutex.h"
#include "stdlib.h"

#include "atomic.h"

struct MyMutex
{
	int 				fCount;
	cthread_t 			fHolder;
	struct spin_lock	fMutexLock;
	port_t				fWaitPort;
	unsigned int		fNumWaiting;
};

typedef struct MyMutex MyMutex;

MyMutex* MMAllocateMutex();
void MMDisposeMutex(MyMutex* theMutex);
void MMGrab(MyMutex* theMutex);
int MMTryGrab(MyMutex* theMutex);
void MMRelease(MyMutex* theMutex);
cthread_t MMGetFirstWaitingThread(MyMutex* theMutex, int* listWasEmpty);
int MMAlreadyHaveControl(MyMutex* theMutex, cthread_t thread);
int MMTryAndGetControl(MyMutex* theMutex, cthread_t thread);
void MMReleaseControl(MyMutex* theMutex);
void MMStripOffWaitingThread(MyMutex* theMutex);
void MMAddToWaitList(MyMutex* theMutex, cthread_t thread);
void MMBlockThread(MyMutex* theMutex);
void MMUnblockThread(MyMutex* theMutex);

mymutex_t mymutex_alloc()
{
	return (mymutex_t)MMAllocateMutex();
}

void mymutex_free(mymutex_t theMutex_t)
{
	MMDisposeMutex((MyMutex*)theMutex_t);
}

void mymutex_lock(mymutex_t theMutex_t)
{
	MMGrab((MyMutex*)theMutex_t);
}

int mymutex_try_lock(mymutex_t theMutex_t)
{
	return MMTryGrab((MyMutex*)theMutex_t);
}

void mymutex_unlock(mymutex_t theMutex_t)
{
	MMRelease((MyMutex*)theMutex_t);
}

MyMutex* MMAllocateMutex()
{
	kern_return_t ret;
	MyMutex* newMutex = (MyMutex*)malloc(sizeof(MyMutex));
	if (newMutex == NULL)
		return NULL;
		
	newMutex->fCount = 0;
	newMutex->fHolder = 0;
	newMutex->fNumWaiting = 0;
	spin_lock_init(&newMutex->fMutexLock);
	ret = port_allocate(task_self(), &newMutex->fWaitPort);
	if (ret != KERN_SUCCESS)
	{
		free(newMutex);
		return NULL;
	}
	
	return newMutex;
}

void MMDisposeMutex(MyMutex* theMutex)
{
	port_deallocate(task_self(), theMutex->fWaitPort);
	free(theMutex);
}

void MMGrab(MyMutex* theMutex)
{
	cthread_t thread = cthread_self();
	
	if (theMutex->fHolder != thread) 
	{
		int waiting = atomic_add(&theMutex->fNumWaiting, 1);
		 
		if ((waiting > 1) || !spin_lock_try(&theMutex->fMutexLock))
		{
			do
			{
				// suspend ourselves until something happens
				MMBlockThread(theMutex);
			} while (!spin_lock_try(&theMutex->fMutexLock));
		}

		atomic_sub(&theMutex->fNumWaiting, 1);

		// we just got control, so reset fCount
		theMutex->fCount = 0; // gets incremented below...
		theMutex->fHolder = thread;
	}

	// we have control now, so increment the count
	++theMutex->fCount;
}

int MMTryGrab(MyMutex* theMutex)
{
	cthread_t thread = cthread_self();
	int haveControl;
	
	haveControl = (theMutex->fHolder == thread);
	if (!haveControl)
		haveControl = spin_lock_try(&theMutex->fMutexLock);

	if (haveControl)
	{
		theMutex->fHolder = thread;
		++theMutex->fCount;
	}
		
	return haveControl;
}

void MMRelease(MyMutex* theMutex)
{
	cthread_t thread = cthread_self();
	if (theMutex->fHolder != thread)
		return;
	
	if (!--theMutex->fCount) 
	{
		theMutex->fHolder = NULL;
		spin_lock_unlock(&theMutex->fMutexLock);	// let someone else deal with it
		if (theMutex->fNumWaiting > 0)
			MMUnblockThread(theMutex);
	}
}

void MMBlockThread(MyMutex* theMutex)
{
	kern_return_t ret;
	msg_header_t msg;

	msg.msg_size = sizeof(msg);
	msg.msg_local_port = theMutex->fWaitPort;
	ret = msg_receive(&msg, MSG_OPTION_NONE, 0);
}

void MMUnblockThread(MyMutex* theMutex)
{
	kern_return_t ret;
	msg_header_t msg;

	msg.msg_simple = TRUE;
	msg.msg_size = sizeof(msg);
    msg.msg_type = MSG_TYPE_NORMAL;
	msg.msg_local_port = PORT_NULL; 
	msg.msg_remote_port = theMutex->fWaitPort;
    msg.msg_id = 0;
	ret = msg_send(&msg, MSG_OPTION_NONE, 0);
}

