/***************************************************************************

 Linux SVGALib adaptation by Phillip Ezolt pe28+@andrew.cmu.edu
  
***************************************************************************/
#define __SVGALIB_C

#include <vga.h>
#include <vgagl.h>
#include <vgakeyboard.h>
#include <vgamouse.h>
#include <signal.h>
#include <linux/kd.h>
#include <sys/ioctl.h>
#include "xmame.h"
#include "devices.h"
#include <math.h>

static int startx, starty;
static int scaled_visual_width, scaled_visual_height;
static int video_mode       = -1;
static int tweaked_mode     = -1;
static int console_fd       = -1;
static int leds             =  0;
static int update_function  = -1;
static int text_mode        = TRUE;
static unsigned char *video_mem = NULL;
static unsigned char *scale_buffer = NULL;
static unsigned char *doublebuffer_buffer = NULL;
static vga_modeinfo video_modeinfo;
static struct sigaction sig2handler;
static struct sigaction oldsig2handler;
static void svgalib_update_linear(void);
static void svgalib_update_planar(void);
static void svgalib_update_gl(void);
static void svgalib_update_gl_scaled(void);
static void svgalib_update_linear_16bpp(void);
static void svgalib_update_gl_scaled_16bpp(void);

typedef void (*update_func)(void);

static update_func update_functions[8] = {
   svgalib_update_linear,
   svgalib_update_planar,
   svgalib_update_gl,
   svgalib_update_gl_scaled,
   svgalib_update_linear_16bpp,
   NULL,
   svgalib_update_gl,
   svgalib_update_gl_scaled_16bpp
};

/* tweaked modes */
#ifndef linux_powerpc
#include "twkuser.c"
#include "twkmodes.h"

struct tweaked_mode_struct
{
   int width;
   int height;
   Register *registers;
   Register *registers_scanline;
};

/* all available videomodes */
static struct tweaked_mode_struct tweaked_modes[] = {
{  320, 204, scr320x204, NULL },
{  288, 224, scr288x224scanlines },
{  256, 256, scr256x256, scr256x256scanlines },
{  240, 272, scr240x272, NULL },
{  224, 288, scr224x288, scr224x288scanlines },
{  200, 320, scr200x320, NULL },
{    0,   0, NULL,       NULL }
};

static void set_tweaked_mode(void)
{
	if (!text_mode && tweaked_mode >= 0)
	{
	   if (use_scanlines && tweaked_modes[tweaked_mode].registers_scanline)
	      outRegArray(tweaked_modes[tweaked_mode].registers_scanline, 25);
	   else
	      outRegArray(tweaked_modes[tweaked_mode].registers, 25);
	}
}
#endif /* ifndef linux_powerpc */

int sysdep_init(void)
{
   vga_init();
   return OSD_OK;
}

void sysdep_close(void)
{
}

int sysdep_set_video_mode (void)
{
   int i;
   
   if (!text_mode) return OSD_NOT_OK;
   
   vga_setmode(video_mode);
   gl_setcontextvga(video_mode);
   text_mode = FALSE;

   if (video_modeinfo.flags & IS_MODEX)
      update_function=1;
   else
   {
#ifndef linux_powerpc
      /* do we have a linear framebuffer ? */
      i = video_modeinfo.width * video_modeinfo.height *
         video_modeinfo.bytesperpixel;
      if (i <= 65536 || 
          (video_modeinfo.flags & IS_LINEAR) ||
          (use_linear && (video_modeinfo.flags && CAPABLE_LINEAR) &&
           vga_setlinearaddressing() >=  i))
      {
         video_mem  = vga_getgraphmem();
         video_mem += startx * video_modeinfo.bytesperpixel;
         video_mem += starty * video_modeinfo.width *
            video_modeinfo.bytesperpixel;
         set_tweaked_mode();
         if ((widthscale > 1 || heightscale > 2) &&
             doublebuffer_buffer == NULL)
         {
            doublebuffer_buffer = malloc(scaled_visual_width * 
               video_modeinfo.bytesperpixel);
            if (!doublebuffer_buffer)
            {
               fprintf(stderr_file, "Svgalib: Error: Couldn't allocate doublebuffer buffer\n");
               return OSD_NOT_OK;
            }
         }
         update_function=0;
         fprintf(stderr_file, "Svgalib: Info: Using a linear framebuffer to speed up\n");
      }
      else
#endif
      if(widthscale==1 && heightscale==1)
         update_function=2;
      else
      {
         /* with the debugger we can be called more then once ! */
         if (!scale_buffer)
         {
            scale_buffer = malloc(scaled_visual_width*scaled_visual_height*
               video_modeinfo.bytesperpixel);
            if (!scale_buffer)
            {
               fprintf(stderr_file, "Svgalib: Error: Couldn't allocate scale buffer\n");
               return OSD_NOT_OK;
            }
         }
         update_function=3;
      }
   }
      
   if (video_16bit)
      update_function+=4;
   else
      sysdep_mark_palette_dirty();
   
   osd_mark_dirty (0,0,bitmap->width-1,bitmap->height-1,1);
   
   return OSD_OK;
}

void sysdep_set_text_mode (void)
{
   if (text_mode) return;
   vga_setmode(TEXT);
   text_mode=TRUE;
}

void Sig2HandlerFunction(int n)
{
   keyboard_clearstate();
   if (vga_setmode(-1)<0x1410)   
   {
      (*(oldsig2handler.sa_handler))(n);
      sigaction(SIGUSR2, &sig2handler, NULL);
   }
#ifndef linux_powerpc   
   set_tweaked_mode();
#endif
   if (console_fd >= 0)
      ioctl(console_fd, KDSETLED, leds);
}

/* This name doesn't really cover this function, since it also sets up mouse
   and keyboard. This is done over here, since on most display targets the
   mouse and keyboard can't be setup before the display has. */
int sysdep_create_display(void)
{
   int i;
   int score, best_score = 0;
   vga_modeinfo *my_modeinfo;
   
   scaled_visual_width  = visual_width  * widthscale;
   scaled_visual_height = visual_height * heightscale;
   
   for (i=1; (my_modeinfo=vga_getmodeinfo(i)); i++)
   {
      if(video_16bit)
      {
         if(my_modeinfo->colors != 32768 &&
            my_modeinfo->colors != 65536)
            continue;
      }
      else
      {
         if(my_modeinfo->colors != 256)
            continue;
         if((my_modeinfo->flags & IS_MODEX) &&
            (!use_planar || widthscale != 1 || heightscale != 1))
         continue;
      }
      if (!vga_hasmode(i))
         continue;
      if (mode_disabled(my_modeinfo->width, my_modeinfo->height))
         continue;
      fprintf(stderr_file, "Svgalib: Info: Found videomode %dx%dx%d\n",
         my_modeinfo->width, my_modeinfo->height, my_modeinfo->colors);
      score = mode_match(my_modeinfo->width, my_modeinfo->height);
      if (score && score >= best_score)
      {
         best_score = score;
         video_mode = i;
         video_modeinfo = *my_modeinfo;
      }
   }
   
   if (use_tweak && !video_16bit && vga_hasmode(G320x200x256))
   {
      for(i=0; tweaked_modes[i].width; i++)
      {
         if (mode_disabled(tweaked_modes[i].width, tweaked_modes[i].height))
            continue;
         score = mode_match(tweaked_modes[i].width, tweaked_modes[i].height);
         if (score && score >= best_score)
         {
            best_score   = score;
            video_mode   = G320x200x256;
            tweaked_mode = i;
            memset(&video_modeinfo, 0, sizeof(video_modeinfo));
            video_modeinfo.bytesperpixel = 1;
            video_modeinfo.colors        = 256;
            video_modeinfo.maxpixels     = 65536;
            video_modeinfo.flags         = IS_LINEAR;
            video_modeinfo.width         = video_modeinfo.linewidth =
               tweaked_modes[i].width;
            video_modeinfo.height        = tweaked_modes[i].height;
         }
      }
   }
   
   if (best_score == 0)
   {
      fprintf(stderr_file, "Svgalib: Couldn't find a suitable mode for a resolution of %d x %d\n",
         scaled_visual_width, scaled_visual_height);
      return OSD_NOT_OK;
   }
   
   fprintf(stderr_file, "Svgalib: Info: Choose videomode %dx%dx%d\n",
      video_modeinfo.width, video_modeinfo.height, video_modeinfo.colors);
   
   startx = ((video_modeinfo.width  - scaled_visual_width ) / 2) & ~7;
   starty =  (video_modeinfo.height - scaled_visual_height) / 2;
   
   if (sysdep_set_video_mode()!=OSD_OK) return OSD_NOT_OK;
   
   fprintf(stderr_file, "Using a mode with a resolution of %d x %d, starting at %d x %d\n",
      video_modeinfo.width, video_modeinfo.height, startx, starty);

   if (vga_setmode(-1)<0x1410)
   {   
      memset(&sig2handler, 0, sizeof(struct sigaction));
      sig2handler.sa_handler=Sig2HandlerFunction;
      sigaction(SIGUSR2, &sig2handler, &oldsig2handler);
      fprintf(stderr_file, "Svgalib: Info: Using signals for console switches\n");
   }
   else
   {
      vga_runinbackground(VGA_COMEFROMBACK, sig2handler);
      fprintf(stderr_file, "Svgalib: Info: Using vga_runinbackground for console switches\n");
   }
   
   if(!video_16bit)
      for (i=0; i < 256; i++) gl_setpalettecolor(i, 0, 0, 0);

#ifdef USE_TIMER
   /* svgalib catches the timer alarm. so we have to arm it after all
      other svgalib initialising. */
   if (play_sound)
   {
      if(start_timer()==OSD_NOT_OK)
      {
         osd_close_display();
         return OSD_NOT_OK;
      }
   }
#endif

   /* init the keyboard */
   if ((console_fd = keyboard_init_return_fd()) < 0)
   {
      fprintf(stderr_file, "Svgalib: Error: Couldn't open keyboard\n");
      return OSD_NOT_OK;
   }
   ioctl(console_fd, KDSETLED, leds);
   local_key=keyboard_getstate();

   /* init the mouse */
   if(use_mouse)
   {
	int fd;
	vga_setmousesupport(TRUE);
	fd=mouse_init_return_fd("/dev/mouse",vga_getmousetype(),MOUSE_DEFAULTSAMPLERATE);
	if (fd<0)
	{
		perror("mouse_init");
		fprintf(stderr_file,"SVGALib: failed to open mouse device %d\n",fd);
		use_mouse=0;
	}
	else
	{
		/* fix ranges and initial position of mouse */
		mouse_setrange_6d(-500,500, -500,500, -500,500, -500,500,
                   -500,500, -500,500, MOUSE_6DIM);
		mouse_setposition_6d(0, 0, 0, 0, 0, 0, MOUSE_6DIM);
	}
   }
   
   return OSD_OK;
}

/* shut up the display */
void osd_close_display(void)
{
   if (vga_setmode(-1)<0x1410)
      sigaction(SIGUSR2, &oldsig2handler, NULL);
   if (console_fd >= 0)
   {
      ioctl(console_fd, KDSETLED, 8);
      keyboard_close();
   }
   sysdep_set_text_mode();
   if (scale_buffer)
      free(scale_buffer);
   if (doublebuffer_buffer)
      free(doublebuffer_buffer);
   osd_dirty_close();
   /* free the bitmap after cleaning the dirty stuff as it uses the bitmap */
   osd_free_bitmap(bitmap);
}

/*
 * Alloc global symbol totalcolors colors, not needed under
 * svgalib, under svgalib we always have access to all available colors
 */
void sysdep_alloc_palette(void)
{
}

void sysdep_alloc_palette_16bpp(unsigned short *pens)
{
   int r1,g1,b1;
   int r,g,b;
      
   for (r1 = 0; r1 < 32; r1++)
   {
      for (g1 = 0; g1 < 32; g1++)
      {
         for (b1 = 0; b1 < 32; b1++)
         {
            r = 255 * brightness * pow(r1 / 31.0, 1 / gamma_correction) / 100;
            g = 255 * brightness * pow(g1 / 31.0, 1 / gamma_correction) / 100;
            b = 255 * brightness * pow(b1 / 31.0, 1 / gamma_correction) / 100;
            if (video_modeinfo.colors == 32768)
               *pens++ = ((r>>3))     | ((g>>3)<<5) | ((b>>3)<<10);
            else
               *pens++ = ((r>>3)<<11) | ((g>>2)<<5) | ((b>>3));
         }
      }
   }
}

void sysdep_modify_pen(int pen,unsigned char red, unsigned char green, unsigned char blue)
{
   gl_setpalettecolor(pen,(red>>2),(green>>2),(blue>>2));
}

void sysdep_get_pen_16bpp(int pen, unsigned char* red, unsigned char *green,
   unsigned char *blue)
{
   if (video_modeinfo.colors == 32768)
   {
      *red   = (pen & 0x001F) << 3;
      *green = (pen & 0x03E0) >> 2;
      *blue  = (pen & 0xEC00) >> 7;
   }
   else
   {
      *red   = (pen & 0xF800) >> 8;
      *green = (pen & 0x07E0) >> 3;
      *blue  = (pen & 0x001F) << 3;
   }
}

/* Update the display. */
void sysdep_update_display(void)
{
   (*update_functions[update_function])();
}

#define SRC_PIXEL  unsigned char
#define DEST_PIXEL unsigned char

static void svgalib_update_linear(void)
{
#define DEST video_mem
#define DEST_WIDTH video_modeinfo.linewidth
#define DOUBLEBUFFER
#include "blit.h"
#undef DEST
#undef DEST_WIDTH
#undef DOUBLEBUFFER
}

static void svgalib_update_planar(void)
{
  /* use page flipping otherwise the screen tears in planar modes,
     unfortunatly this also means we can't use dirty in planar modes */
  static int page=0;
  if (page) page=0;
   else page=131072;
  vga_copytoplanar256(bitmap->line[visual.min_y] + visual.min_x,
     bitmap->line[1] - bitmap->line[0],
     (starty*video_modeinfo.width + startx + page)/4,
      video_modeinfo.width/4, visual_width, visual_height);
  vga_setdisplaystart(page);
}

static void svgalib_update_gl(void)
{
   int bitmap_linewidth = (bitmap->line[1] - bitmap->line[0]) /
      video_modeinfo.bytesperpixel;
#define PUT_IMAGE(X, Y, WIDTH, HEIGHT) \
      gl_putboxpart( \
         startx + X, starty + Y, \
         WIDTH, HEIGHT, \
         bitmap_linewidth, bitmap->height, \
         bitmap->line[0], X + visual.min_x, Y + visual.min_y);
         /* Note: we calculate the real bitmap->width, as used in
            osd_create_bitmap, this fixes the scewing bug in tempest
            & others */
#include "blit.h"
#undef PUT_IMAGE
}

static void svgalib_update_gl_scaled(void)
{
#define DEST scale_buffer
#define DEST_WIDTH scaled_visual_width
#define PUT_IMAGE(X, Y, WIDTH, HEIGHT) \
      gl_putboxpart( \
         startx + X, starty + Y, \
         WIDTH, HEIGHT, \
         scaled_visual_width, scaled_visual_height, \
         scale_buffer, X, Y );
#include "blit.h"
#undef DEST
#undef DEST_WIDTH
#undef PUT_IMAGE
}

#undef  SRC_PIXEL
#undef  DEST_PIXEL
#define SRC_PIXEL  unsigned short
#define DEST_PIXEL unsigned short

static void svgalib_update_linear_16bpp(void)
{
   int linewidth = video_modeinfo.linewidth/2;
#define DEST video_mem
#define DEST_WIDTH linewidth
#define DOUBLEBUFFER
#include "blit.h"
#undef DEST
#undef DEST_WIDTH
#undef DOUBLEBUFFER
}

static void svgalib_update_gl_scaled_16bpp(void)
{
#define DEST scale_buffer
#define DEST_WIDTH scaled_visual_width
#define PUT_IMAGE(X, Y, WIDTH, HEIGHT) \
      gl_putboxpart( \
         startx + X, starty + Y, \
         WIDTH, HEIGHT, \
         scaled_visual_width, scaled_visual_height, \
         scale_buffer, X, Y );
#include "blit.h"
#undef DEST
#undef DEST_WIDTH
#undef PUT_IMAGE
}

int sysdep_mapkey(char *arg) 
{
   return OSD_OK;
}

void sysdep_mouse_poll (void)
{
	int i, mouse_buttons;
	
	mouse_update();
	
	mouse_getposition_6d(&mouse_data[0].deltas[0],
           &mouse_data[0].deltas[1],
           &mouse_data[0].deltas[2],
           &mouse_data[0].deltas[3],
           &mouse_data[0].deltas[4],
           &mouse_data[0].deltas[5]);
	
	mouse_buttons = mouse_getbutton();

        for(i=0; i<JOY_BUTTONS; i++)
        {
           mouse_data[0].buttons[i] = mouse_buttons & (0x01 << i);
        }

	mouse_setposition_6d(0, 0, 0, 0, 0, 0, MOUSE_6DIM);
}

static const int led_flags[3] = {
  LED_NUM,
  LED_CAP,
  LED_SCR
};

void osd_led_w(int led,int on)
{
	if (led>=3) return;
	if (on)
		leds |=  led_flags[led];
	else
		leds &= ~led_flags[led];
	if (console_fd >= 0)
		ioctl(console_fd, KDSETLED, leds);
}
