;==============================================================================

; file: "_io.scm"

; Copyright (C) 1994-1998 by Marc Feeley, All Rights Reserved.

(##include "header.scm")

(##declare (not interrupts-enabled))

;------------------------------------------------------------------------------

; Ports and readtables are structures.

(define (##port? x)
  (and (##structure? x)
       (##eq? (##vector-ref x 0) ##port-tag)))

(define (##readtable? x)
  (and (##structure? x)
       (##eq? (##vector-ref x 0) ##readtable-tag)))

;------------------------------------------------------------------------------

; Generic port objects.

(define ##port-tag '#(port 0))

(##define-macro
  (make-port

    input?        ; #t if is an input port or an input-output port
    output?       ; #t if is an output port or an input-output port

    name          ; name associated with port (name can be any object)

    read-char     ; (lambda (port) ...)
    peek-char     ; (lambda (port) ...)
    char-ready?   ; (lambda (port) ...)
    flush-input   ; (lambda (port) ...)
    read-pos      ; (lambda (port) ...)

    write-char    ; (lambda (char port) ...)
    flush-output  ; (lambda (port) ...)
    write-pos     ; (lambda (port) ...)

    close         ; (lambda (port) ...)

    width         ; (lambda (port) ...)

    isatty        ; (lambda (port) ...)

    . others      ; other slots
  )
  `(##subtype-set!
     (##vector
       ##port-tag
       #t ; port is open
       ,input?
       ,output?
       ,name
       ,read-char
       ,peek-char
       ,char-ready?
       ,flush-input
       ,read-pos
       ,write-char
       ,flush-output
       ,write-pos
       ,close
       ,width
       ,isatty
       ,@others)
     (subtype-structure)))

(##define-macro (port-open? port)
  `(##vector-ref ,port 1))

(##define-macro (port-open?-set! port x)
  `(##vector-set! ,port 1 ,x))

(##define-macro (port-input? port)
  `(##vector-ref ,port 2))

(##define-macro (port-output? port)
  `(##vector-ref ,port 3))

(##define-macro (port-name port)
  `(##vector-ref ,port 4))

(##define-macro (port-name-set! port name)
  `(##vector-set! ,port 4 ,name))

(##define-macro (port-read-char port)
  `(let ((port ,port))
     ((##vector-ref port 5) port)))

(##define-macro (port-peek-char port)
  `(let ((port ,port))
     ((##vector-ref port 6) port)))

(##define-macro (port-char-ready? port)
  `(let ((port ,port))
     ((##vector-ref port 7) port)))

(##define-macro (port-flush-input port)
  `(let ((port ,port))
     ((##vector-ref port 8) port)))

(##define-macro (port-read-pos port)
  `(let ((port ,port))
     ((##vector-ref port 9) port)))

(##define-macro (port-write-char char port)
  `(let ((char ,char) (port ,port))
     ((##vector-ref port 10) char port)))

(##define-macro (port-flush-output port)
  `(let ((port ,port))
     ((##vector-ref port 11) port)))

(##define-macro (port-write-pos port)
  `(let ((port ,port))
     ((##vector-ref port 12) port)))

(##define-macro (port-close port)
  `(let ((port ,port))
     ((##vector-ref port 13) port)))

(##define-macro (port-width port)
  `(let ((port ,port))
     ((##vector-ref port 14) port)))

(##define-macro (port-isatty port)
  `(let ((port ,port))
     ((##vector-ref port 15) port)))

(##define-macro (port-input-char-count port)
  `(##vector-ref ,port 16))

(##define-macro (port-input-char-count-set! port pos)
  `(##vector-set! ,port 16 ,pos))

(##define-macro (port-input-line-count port)
  `(##vector-ref ,port 17))

(##define-macro (port-input-line-count-set! port pos)
  `(##vector-set! ,port 17 ,pos))

(##define-macro (port-input-line-start port)
  `(##vector-ref ,port 18))

(##define-macro (port-input-line-start-set! port pos)
  `(##vector-set! ,port 18 ,pos))

(##define-macro (port-output-char-count port)
  `(##vector-ref ,port 19))

(##define-macro (port-output-char-count-set! port pos)
  `(##vector-set! ,port 19 ,pos))

(##define-macro (port-output-line-count port)
  `(##vector-ref ,port 20))

(##define-macro (port-output-line-count-set! port pos)
  `(##vector-set! ,port 20 ,pos))

(##define-macro (port-output-line-start port)
  `(##vector-ref ,port 21))

(##define-macro (port-output-line-start-set! port pos)
  `(##vector-set! ,port 21 ,pos))

(define (##open-port? port)
  (port-open? port))

(define (##input-port? port)
  (port-input? port))

(define (##output-port? port)
  (port-output? port))

;------------------------------------------------------------------------------

; Position tracking for ports.

(##define-macro (exact-int.add1 n) ; Add 1 to a fixnum or bignum.
  `(let ((temp1 ,n))
     (if (##fixnum? temp1)
       (let ((temp2 (##fixnum.+ temp1 1)))
         (if (##fixnum.< temp2 0)
           (##+ temp1 1)
           temp2))
       (##+ temp1 1))))

(##define-macro (end-of-line? char)
  `(##char=? ,char #\newline))

(##define-macro (track-input-pos port char)
  `(if (##char? ,char)
     (let ((char-count (exact-int.add1 (port-input-char-count ,port))))
       (port-input-char-count-set! ,port char-count)
       (if (end-of-line? ,char)
         (begin
           (port-input-line-start-set! ,port char-count)
           (port-input-line-count-set! ,port 
             (exact-int.add1 (port-input-line-count ,port))))))))

(##define-macro (track-output-pos port char)
  `(let ((char-count (exact-int.add1 (port-output-char-count ,port))))
     (port-output-char-count-set! ,port char-count)
     (if (end-of-line? ,char)
       (begin
         (port-output-line-start-set! ,port char-count)
         (port-output-line-count-set! ,port 
           (exact-int.add1 (port-output-line-count ,port)))))))

;------------------------------------------------------------------------------

; "Stdio" ports.

(c-declare "#include \"os.h\"")

(##define-macro (io-getc stream)
  `((c-lambda ((pointer "___STREAM")) scheme-object "___io_getc") ,stream))

(##define-macro (io-peek stream)
  `((c-lambda ((pointer "___STREAM")) scheme-object "___io_peek") ,stream))

(##define-macro (io-ready stream)
  `((c-lambda ((pointer "___STREAM")) scheme-object "___io_ready") ,stream))

(##define-macro (io-putc stream char)
  `((c-lambda ((pointer "___STREAM") ucs4) scheme-object "___io_putc") ,stream ,char))

(##define-macro (io-flush stream)
  `((c-lambda ((pointer "___STREAM")) scheme-object "___io_flush") ,stream))

(##define-macro (io-close stream)
  `((c-lambda ((pointer "___STREAM")) scheme-object "___io_close") ,stream))

(##define-macro (io-free stream)
  `((c-lambda ((pointer "___STREAM")) scheme-object "___io_free") ,stream))

(##define-macro (io-width stream)
  `((c-lambda ((pointer "___STREAM")) scheme-object "___io_width") ,stream))

(##define-macro (io-isatty stream)
  `((c-lambda ((pointer "___STREAM")) scheme-object "___io_isatty") ,stream))

(##define-macro (io-stdin)
  `((c-lambda () (pointer "___STREAM") "___io_stdin")))

(##define-macro (io-stdout)
  `((c-lambda () (pointer "___STREAM") "___io_stdout")))

(##define-macro (io-stderr)
  `((c-lambda () (pointer "___STREAM") "___io_stderr")))

(##define-macro (io-interrupted-call? x)
  `((c-lambda (scheme-object) scheme-object "___result = ___BOOLEAN(___ARG1 == ___FIX((EINTR)));") ,x))

(##define-macro (io-open-input-file path encoding)
  `((c-lambda (char-string int int) (pointer "___STREAM") "___io_open_file")
    ,path
    -1
    ,encoding))

(##define-macro (io-open-input-pipe command encoding)
  `((c-lambda (char-string int int) (pointer "___STREAM") "___io_open_pipe")
    ,command
    -1
    ,encoding))

(##define-macro (io-open-output-file path encoding)
  `((c-lambda (char-string int int) (pointer "___STREAM") "___io_open_file")
    ,path
    1
    ,encoding))

(##define-macro (io-open-output-pipe command encoding)
  `((c-lambda (char-string int int) (pointer "___STREAM") "___io_open_pipe")
    ,command
    1
    ,encoding))

(##define-macro (io-open-input-output-file path encoding)
  `((c-lambda (char-string int int) (pointer "___STREAM") "___io_open_file")
    ,path
    0
    ,encoding))

(##define-macro (io-open-input-output-pipe command encoding)
  `((c-lambda (char-string int int) (pointer "___STREAM") "___io_open_pipe")
    ,command
    0
    ,encoding))

(define (##io-encoding char-encoding)
  (case char-encoding
    ((char)   1)
    ((latin1) 2)
    ((utf8)   3)
    ((byte)   4)
    ((ucs2)   5)
    ((ucs4)   6)
    (else     0)))

(define (##open-input-file-no-path-expand
          path
          #!optional (char-encoding #f))
  (let ((stream
         (io-open-input-file
          path
          (##io-encoding char-encoding))))
    (if stream
      (##make-io-port #t #f path stream stream)
      #f)))

(define (##open-input-file
          path
          #!optional (char-encoding #f))
  (let ((expanded-path (##path-expand path 'absolute)))
    (and expanded-path
         (##open-input-file-no-path-expand expanded-path char-encoding))))

(define (##open-input-pipe
          command
          #!optional (char-encoding #f))
  (let ((stream
         (io-open-input-pipe
          command
          (##io-encoding char-encoding))))
    (if stream
      (##make-io-port #t #f command stream stream)
      #f)))

(define (##open-output-file-no-path-expand
          path
          #!optional (char-encoding #f))
  (let ((stream
         (io-open-output-file
          path
          (##io-encoding char-encoding))))
    (if stream
      (##make-io-port #f #t path stream stream)
      #f)))

(define (##open-output-file
          path
          #!optional (char-encoding #f))
  (let ((expanded-path (##path-expand path 'absolute)))
    (and expanded-path
         (##open-output-file-no-path-expand expanded-path char-encoding))))

(define (##open-output-pipe
          command
          #!optional (char-encoding #f))
  (let ((stream
         (io-open-output-pipe
          command
          (##io-encoding char-encoding))))
    (if stream
      (##make-io-port #f #t command stream stream)
      #f)))

(define (##open-input-output-file-no-path-expand
          path
          #!optional (char-encoding #f))
  (let ((stream
         (io-open-input-output-file
          path
          (##io-encoding char-encoding))))
    (if stream
      (##make-io-port #t #t path stream stream)
      #f)))

(define (##open-input-output-file
          path
          #!optional (char-encoding #f))
  (let ((expanded-path (##path-expand path 'absolute)))
    (and expanded-path
         (##open-input-output-file-no-path-expand expanded-path char-encoding))))

(define (##open-input-output-pipe
          command
          #!optional (char-encoding #f))
  (let ((stream
         (io-open-input-output-pipe
          command
          (##io-encoding char-encoding))))
    (if stream
      (##make-io-port #t #t command stream stream)
      #f)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(##define-macro (io-exception-handler port x continue)
  `(cond ((io-interrupted-call? ,x)
          (let ()
            (##declare (interrupts-enabled))
            (,continue)))
         (else
          (##signal '##signal.io-error "IO error on" ,port)
          (,continue))))

(##define-macro (port-input-stream port)
  `(##vector-ref ,port 22))

(##define-macro (port-input-stream-set! port stream)
  `(##vector-set! ,port 22 ,stream))

(##define-macro (port-output-stream port)
  `(##vector-ref ,port 23))

(##define-macro (port-output-stream-set! port stream)
  `(##vector-set! ,port 23 ,stream))

(define (##make-io-port
          input?
          output?
          name
          input-stream
          output-stream)

  (define (stream-flush stream port)
    (let loop ()
      (let ((x (io-flush stream)))
        (if (##fixnum? x)
          (io-exception-handler port x (lambda () (loop)))
          #f))))

  (define (stream-close stream port)
    (let loop ()
      (let ((x (io-close stream)))
        (if (##fixnum? x)
          (io-exception-handler port x (lambda () (loop)))
          #f))))

  (let ((port
         (make-port

           input?
           output?

           name

           ; read-char
           (lambda (port)
             (let ((stream (port-input-stream port)))
               (let loop ()
                 (let ((c (io-getc stream)))
                   (if (##fixnum? c)
                     (io-exception-handler port c (lambda () (loop)))
                     (begin
                       (track-input-pos port c)
                       c))))))

           ; peek-char
           (lambda (port)
             (let ((stream (port-input-stream port)))
               (let loop ()
                 (let ((c (io-peek stream)))
                   (if (##fixnum? c)
                     (io-exception-handler port c (lambda () (loop)))
                     c)))))

           ; char-ready?
           (lambda (port)
             (let ((stream (port-input-stream port)))
               (io-ready stream)))

           ; flush-input
           (lambda (port)
             (stream-flush (port-input-stream port) port))

           ; read-pos
           (lambda (port)
             (##vector (port-input-char-count port)
                       (port-input-line-count port)
                       (port-input-line-start port)))

           ; write-char
           (lambda (char port)
             (let ((stream (port-output-stream port)))
               (let loop ()
                 (let ((x (io-putc stream char)))
                   (if (##fixnum? x)
                     (io-exception-handler port x (lambda () (loop)))
                     (begin
                       (track-output-pos port char)
                       #f))))))

           ; flush-output
           (lambda (port)
             (stream-flush (port-output-stream port) port))

           ; write-pos
           (lambda (port)
             (##vector (port-output-char-count port)
                       (port-output-line-count port)
                       (port-output-line-start port)))

           ; close
           (lambda (port)
             (let ((input-stream (port-input-stream port))
                   (output-stream (port-output-stream port)))
               (stream-close input-stream port)
               (io-free input-stream)
               (if (##not (##eq? input-stream output-stream))
                 (begin
                   (stream-close output-stream port)
                   (io-free output-stream)))
               (port-input-stream-set! port #f)
               (port-output-stream-set! port #f)
               (port-open?-set! port #f)
               (##void)))

           ; width
           (lambda (port)
             (let ((stream (port-output-stream port)))
               (io-width stream)))

           ; isatty
           (lambda (port)
             (let ((stream (port-input-stream port)))
               (io-isatty stream)))

           0 ; input-char-count
           0 ; input-line-count
           0 ; input-line-start
           0 ; output-char-count
           0 ; output-line-count
           0 ; output-line-start

           input-stream
           output-stream
           )))
    (##make-will port (lambda () (##close-port port)))
    port))

(define ##stdin
  (let ((stream (io-stdin)))
    (##make-io-port #t #f '(stdin) stream stream)))

(define ##stdout
  (let ((stream (io-stdout)))
    (##make-io-port #f #t '(stdout) stream stream)))

(define ##stderr
  (let ((stream (io-stderr)))
    (##make-io-port #f #t '(stderr) stream stream)))

;------------------------------------------------------------------------------

; String ports.

(##define-macro (port-string port)
  `(##vector-ref ,port 22))

(##define-macro (port-string-set! port str)
  `(##vector-set! ,port 22 ,str))

(define (##make-string-port input? output? str-or-len)

  ; assumes string is at most (max-fixnum32) characters long

  (define (string-extend str)
    (let* ((n (##string-length str))
           (m (##fixnum.+ (##fixnum.* n 2) 1))
           (result (##make-string m #\space)))
      (let loop ((i (##fixnum.- n 1)))
        (if (##fixnum.< i 0)
          result
          (begin
            (##string-set! result i (##string-ref str i))
            (let ()
              (##declare (interrupts-enabled))
              (loop (##fixnum.- i 1))))))))

  (make-port

    input?
    output?

    '(string)

    ; read-char
    (lambda (port)
      (let ((n (port-output-char-count port))
            (i (port-input-char-count port)))
        (if (##fixnum.< i n)
          (let ((c (##string-ref (port-string port) i)))
            (track-input-pos port c)
            c)
          #!eof)))

    ; peek-char
    (lambda (port)
      (let ((n (port-output-char-count port))
            (i (port-input-char-count port)))
        (if (##fixnum.< i n)
          (##string-ref (port-string port) i)
          #!eof)))

    ; char-ready?
    (lambda (port)
      (let ((n (port-output-char-count port))
            (i (port-input-char-count port)))
        (##fixnum.< i n)))

    ; flush-input
    (lambda (port)
      #f)

    ; read-pos
    (lambda (port)
      (##vector (port-input-char-count port)
                (port-input-line-count port)
                (port-input-line-start port)))

    ; write-char
    (lambda (char port)
      (let* ((n (port-output-char-count port))
             (str (port-string port))
             (len (##string-length str)))
        (let ((s (if (##fixnum.< n len)
                   str
                   (let ((extended-str (string-extend str)))
                     (port-string-set! port extended-str)
                     extended-str))))
          (##string-set! s n char)
          (track-output-pos port char)
          #f)))

    ; flush-output
    (lambda (port)
      #f)

    ; write-pos
    (lambda (port)
      (##vector (port-output-char-count port)
                (port-output-line-count port)
                (port-output-line-start port)))

    ; close
    (lambda (port)
      (let* ((n (port-output-char-count port))
             (i (port-input-char-count port))
             (str (port-string port)))
        (port-open?-set! port #f)
        (if (##fixnum.= i 0)
          (begin
            (##string-shrink! str n)
            str)
          (##substring str i n))))

    ; width
    (lambda (port)
      79)

    ; isatty
    (lambda (port)
      #f)

    0 ; input-char-count
    0 ; input-line-count
    0 ; input-line-start
    (if (##string? str-or-len) ; output-char-count
      (##string-length str-or-len)
      0)
    0 ; output-line-count
    0 ; output-line-start

    (if (##string? str-or-len) ; string
      str-or-len
      (##make-string str-or-len #\space))
    ))

(define (##open-input-string str)
  (##make-string-port #t #f str))

(define (##open-output-string)
  (##make-string-port #f #t 127))

;------------------------------------------------------------------------------

; Generic port operations.

(define (##port-name port)
  (port-name port))

(define (##read-char port)
  (port-read-char port))

(define (##peek-char port)
  (port-peek-char port))

(define (##char-ready? port)
  (port-char-ready? port))

(define (##write-char c port)
  (port-write-char c port))

(define (##write-substring s i j port)
  (let loop ((i i))
    (if (##fixnum.< i j)
      (begin
        (port-write-char (##string-ref s i) port)
        (let ()
          (##declare (interrupts-enabled))
          (loop (##fixnum.+ i 1)))))))

(define (##write-string s port)
  (##write-substring s 0 (##string-length s) port))

(define (##close-port port)
  (port-close port))

(define (##newline port)
  (##write-char #\newline port))

(define (##port-width port)
  (port-width port))

(define (##port-isatty port)
  (port-isatty port))

(define (##flush-output port)
  (port-flush-output port))

(##declare (interrupts-enabled))

;------------------------------------------------------------------------------

; A readenv structure maintains the "read environment" throughout the
; reading of a Scheme datum.  It includes the port from which to read,
; the readtable, the error procedure, the wrap and unwrap procedures,
; and the position where the currently being read datum started.

(define (##make-readenv port readtable error-proc wrapper unwrapper)
  (##vector port readtable error-proc wrapper unwrapper 0))

(define (##readenv-port re)             (##vector-ref re 0))
(define (##readenv-readtable re)        (##vector-ref re 1))
(define (##readenv-error-proc re)       (##vector-ref re 2))
(define (##readenv-wrap re x)           ((##vector-ref re 3) re x))
(define (##readenv-unwrap re x)         ((##vector-ref re 4) re x))
(define (##readenv-filepos re)          (##vector-ref re 5))
(define (##readenv-filepos-set! re pos) (##vector-set! re 5 pos))

(define (##readenv-current-filepos re)
  (let* ((port
          (##readenv-port re))
         (line
          (port-input-line-count port))
         (char-count
          (port-input-char-count port))
         (col
          (##fixnum.- char-count
                      (port-input-line-start port))))
    (##make-filepos line col char-count)))

(define (##readenv-previous-filepos re offset)
  (let* ((port
          (##readenv-port re))
         (line
          (port-input-line-count port))
         (char-count
          (##fixnum.- (port-input-char-count port) offset))
         (col
          (##fixnum.- char-count
                      (port-input-line-start port))))
    (##make-filepos line col char-count)))

(##define-macro (##peek-next-char-or-eof re) ; possibly returns end-of-file
  `(port-peek-char (##readenv-port ,re)))

(##define-macro (##read-next-char-or-eof re) ; possibly returns end-of-file
  `(port-read-char (##readenv-port ,re)))

(define (##make-filepos line col char-count)
  (if (and (##fixnum.< line (max-lines))
           (##not (##fixnum.< (max-fixnum32-div-max-lines) col)))
    (##fixnum.+ line (##fixnum.* col (max-lines)))
    (##fixnum.- 0 char-count)))

(define (##filepos-line filepos)
  (if (##fixnum.< filepos 0)
    0
    (##fixnum.modulo filepos (max-lines))))

(define (##filepos-col filepos)
  (if (##fixnum.< filepos 0)
    (##fixnum.- 0 filepos)
    (##fixnum.quotient filepos (max-lines))))

(define ##main-readtable #f)

(define (current-readtable)
  (##current-readtable))

(define (set-case-conversion! conversion? #!optional (r (absent-obj)))
  (force-vars (conversion? r)
    (let ((rt
           (if (##eq? r (absent-obj))
             (##current-readtable)
             r)))
      (check-readtable rt (set-case-conversion! conversion? r)
        (begin
          (##readtable-case-conversion?-set! rt conversion?)
          (##void))))))

(define (set-keywords-allowed! allowed? #!optional (r (absent-obj)))
  (force-vars (allowed? r)
    (let ((rt
           (if (##eq? r (absent-obj))
             (##current-readtable)
             r)))
      (check-readtable rt (set-keywords-allowed! allowed? r)
        (begin
          (##readtable-keywords-allowed?-set! rt allowed?)
          (##void))))))

(define (##read port rt)

  (define (noop re x) x) ; do not wrap datum

  (let ((re (##make-readenv port rt ##read-error noop noop)))
    (##read-datum-or-eof re)))

(define (##read-error re msg . args)
  (##apply
   ##signal
   (##cons '##signal.read-error
           (##cons (##readenv-port re)
                   (##cons (##readenv-filepos re)
                           (##cons msg
                                   args))))))

;------------------------------------------------------------------------------

; A writeenv structure maintains the "write environment" throughout
; the writing of a Scheme datum.  It includes the port on which to
; write, the readtable, and the display and force flags.

(define (##make-writeenv port readtable force? display?)
  (##vector port readtable force? display?))

(define (##writeenv-port we)      (##vector-ref we 0))
(define (##writeenv-readtable we) (##vector-ref we 1))
(define (##writeenv-force? we)    (##vector-ref we 2))
(define (##writeenv-display? we)  (##vector-ref we 3))

(define (##wr-unlimited obj we)
  (##wr-limited obj we (max-fixnum32)))

(define (##wr-limited obj we limit)
  (##fixnum.- limit (##wr obj we limit)))

(define (##default-wr obj we limit)
  (if (##fixnum.< 0 limit)

    (cond ((##symbol? obj)
           (##wr-symbol obj we limit))
          ((##keyword? obj)
           (##wr-keyword obj we limit))
          ((##pair? obj)
           (##wr-pair obj #t we limit))
          ((##complex? obj)
           (##wr-number obj we limit))
          ((##char? obj)
           (##wr-char obj we limit))
          ((##string? obj)
           (##wr-string obj we limit))
          ((##vector? obj)
           (##wr-vector "#" (##vector->list obj) we limit))
          ((##eq? obj #t)
           (##wr-str "#t" we limit))
          ((##eq? obj #f)
           (##wr-str "#f" we limit))
          ((##eq? obj '())
           (##wr-str "()" we limit))
          ((##unbound? obj)
           (##wr-str "#<unbound>" we limit))
          ((##eq? obj (absent-obj))
           (##wr-str "#<absent>" we limit))
          ((##eq? obj (##void))
           (##wr-str "#<void>" we limit))
          ((##port? obj)
           (##wr-port obj we limit))
          ((##readtable? obj)
           (##wr-readtable obj we limit))
          ((##structure? obj)
           (##wr-structure obj we limit))
          ((##pointer? obj)
           (##wr-pointer obj we limit))
          ((##procedure? obj)
           (##wr-procedure obj we limit))
          ((##will? obj)
           (##wr-will obj we limit))
          ((##promise? obj)
           (##wr-promise obj we limit))
          ((##u8vector? obj)
           (##wr-vector "#u8"
                        (##u8vector->list obj)
                        we
                        limit))
          ((##u16vector? obj)
           (##wr-vector "#u16"
                        (##u16vector->list obj)
                        we
                        limit))
          ((##u32vector? obj)
           (##wr-vector "#u32"
                        (##u32vector->list obj)
                        we
                        limit))
          ((##f32vector? obj)
           (##wr-vector "#f32"
                        (##f32vector->list obj)
                        we
                        limit))
          ((##f64vector? obj)
           (##wr-vector "#f64"
                        (##f64vector->list obj)
                        we
                        limit))
          (else
           (##wr-other obj we limit)))

    0))

(define ##wr #f)
(set! ##wr ##default-wr)

(define ##wr-str
  (let ()
    (##declare (not inline)) ; it isn't worth inlining this function
    (lambda (s we limit)
      (##wr-substr s 0 (##string-length s) we limit))))

(define (##wr-substr s i j we limit)
  (let ((len (##fixnum.- j i))
        (port (##writeenv-port we)))
    (if (##fixnum.< limit len)
      (begin
        (##write-substring s i (##fixnum.+ i limit) port)
        0)
      (begin
        (##write-substring s i j port)
        (##fixnum.- limit len)))))

(define (##wr-ch c we limit)
  (if (##fixnum.< 0 limit)
    (begin
      (##write-char c (##writeenv-port we))
      (##fixnum.- limit 1))
    0))

(define (##wr-adr type obj we limit)
  (##wr-str ">" we
            (##wr-str (##number->string (##type-cast obj (type-fixnum)) 16) we
                      (##wr-str " #x" we
                                (##wr-str type we
                                          (##wr-str "#<" we limit))))))

(define (##wr-id-in type id name we limit)
  (##wr-str ">" we
            (##wr name we
                  (##wr-str " in " we
                            (##wr-str id we
                                      (##wr-str " " we
                                                (##wr-str type we
                                                          (##wr-str "#<" we limit))))))))

(define (##wr-named type name we limit)
  (##wr-str ">" we
            (##wr name we
                  (##wr-str " " we
                            (##wr-str type we
                                      (##wr-str "#<" we limit))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Write methods for each object type

(define (##wr-number obj we limit)
  (##wr-str (##number->string obj 10) we limit))

(define (##wr-char obj we limit)
  (if (##writeenv-display? we)
    (##wr-ch obj we limit)
    (let ((x (##assq-cdr obj
                         (##readtable-named-char-table
                          (##writeenv-readtable we)))))
      (if x
        (##wr-str (##car x) we
                  (##wr-str "#\\" we limit))
        (let ((n (##fixnum.<-char obj)))
          (if (##fixnum.< n 32)
            (##wr-str (##number->string n 16) we
                      (##wr-str "#\\#x" we limit))
            (##wr-ch obj we
                     (##wr-str "#\\" we limit))))))))

(define (##wr-pair obj allow-read-macros-at-tail? we limit)

  (define (wr-tail lst limit)
    (if (##fixnum.< 0 limit)
      (let ((lst (if (##writeenv-force? we)
                   (force-vars (lst) lst)
                   lst)))
        (cond ((##pair? lst)
               (let* ((head (##car lst))
                      (rest (##cdr lst))
                      (x (and allow-read-macros-at-tail?
                              (##check-for-read-macro head rest we limit))))
                 (if x
                   (##wr-str ")" we
                             (##wr (##car rest) we
                                   (##wr-str x we
                                             (##wr-str " . " we limit))))
                   (wr-tail rest
                            (##wr head we
                                  (##wr-str " " we limit))))))
              ((##null? lst)
               (##wr-str ")" we limit))
              (else
               (##wr-str ")" we
                         (##wr lst we
                               (##wr-str " . " we limit))))))
      0))

  (let* ((head (##car obj))
         (rest (##cdr obj))
         (x (and allow-read-macros-at-tail?
                 (##check-for-read-macro head rest we limit))))
    (if x
      (##wr (##car rest) we
            (##wr-str x we limit))
      (wr-tail rest
               (##wr head we
                     (##wr-str "(" we limit))))))

(define (##check-for-read-macro head rest we limit)
  (if (and (##pair? rest) (##null? (##cdr rest)))
    (case head
      ((quote)
       "'")
      ((quasiquote)
       "`")
      ((unquote)
       ; We have to check that the next character written after the comma
       ; won't be a @ because the reader would subsequently interpret this
       ; as a ",@" readmacro.  The algorithm is slow but correct and
       ; modular.
       (if (##fixnum.< 1 limit) ; speed up ",,,,,,xxx" case
         (let ((we2
                (##make-writeenv
                 (##make-string-port #f #t 1)
                 (##writeenv-readtable we)
                 (##writeenv-force? we)
                 (##writeenv-display? we))))
           (##wr (##car rest) we2 1)
           (if (##char=? (##string-ref (port-string (##writeenv-port we2)) 0)
                         #\@)
             ", " ; force a space after the comma
             ","))
         ","))
      ((unquote-splicing)
       ",@")
      (else
       #f))
    #f))

(define (##wr-procedure obj we limit)
  (let ((name (##object->global-var->identifier obj)))
    (if name
      (##wr-named "procedure" name we limit)
      (cond ((##closure? obj)
             (##wr-adr "procedure" obj we limit))
            ((##subprocedure? obj)
             (let ((parent (##object->global-var->identifier
                             (##subprocedure-parent obj))))
               (if parent
                 (##wr-id-in "subprocedure"
                             (##number->string (##subprocedure-id obj) 10)
                             parent we limit)
                 (##wr-adr "procedure" obj we limit))))
            (else
             (##wr-adr "procedure" obj we limit))))))

(define (##wr-vector prefix elems we limit)
  (let ((limit (##wr-str prefix we limit)))
    (if (##pair? elems)
      (##wr-pair elems #f we limit)
      (##wr-str "()" we limit))))

(define (##wr-symbol obj we limit)
  (let ((s (##symbol->string obj)))
    (if (or (##writeenv-display? we)
            (##not (##escape-symbol? obj we)))
      (##wr-str s we limit)
      (##wr-str "|" we
                (##wr-escaped-string s #\| we
                                     (##wr-str "|" we limit))))))

(define (##escape-symbol? sym we)
  (let* ((s (##symbol->string sym))
         (n (##string-length s)))
    (or (##fixnum.= n 0)
        (and (##fixnum.= n 1)
             (##char=? (##string-ref s 0) #\.))
        (and (##char=? (##string-ref s 0) #\#)
             (or (##fixnum.= n 1)
                 (##not (##char=? (##string-ref s 1) #\#))))
        (##string->number s 10)
        (##readtable-parse-keyword (##writeenv-readtable we) s)
        (let loop ((i (##fixnum.- n 1)))
          (if (##fixnum.< i 0)
            #f
            (let ((c (##string-ref s i)))
              (or (##char=? c #\|)
                  (##readtable-char-delimiter? (##writeenv-readtable we) c)
                  (##not (##char=? c (##readtable-convert-case
                                      (##writeenv-readtable we)
                                      c)))
                  (loop (##fixnum.- i 1)))))))))

(define (##wr-keyword obj we limit)
  (let ((str (##keyword->string obj)))
    (if (##eq? (##readtable-keywords-allowed? (##writeenv-readtable we))
               'prefix)
      (##wr-str str we
                (##wr-str ":" we limit))
      (##wr-str ":" we
                (##wr-str str we limit)))))

(define (##wr-port obj we limit)
  (##wr-named (if (##input-port? obj)
                (if (##output-port? obj) "input-output-port" "input-port")
                "output-port")
              (##port-name obj)
              we
              limit))

(define (##wr-readtable obj we limit)
  (##wr-adr "readtable" obj we limit))

(define (##wr-structure obj we limit)
  (let* ((tag (##vector-ref obj 0))
         (n (##vector-ref tag 1)))
    (let loop ((i 1)
               (limit (##wr (##vector-ref tag 0) we
                            (##wr-str "#s(" we limit))))
      (if (##fixnum.< 0 limit)
        (if (##fixnum.< n i)
          (##wr-str ")" we limit)
          (let ((field (##vector-ref tag (##fixnum.+ i 1)))
                (value (##vector-ref obj i)))
            (let ((value (if (##writeenv-force? we)
                           (force-vars (value) value)
                           value)))
              (loop (##fixnum.+ i 1)
                    (##wr-str ")" we
                              (##wr value we
                                    (##wr-str " " we
                                              (##wr field we
                                                    (##wr-str " (" we limit)))))))))
        0))))

(define (##wr-pointer obj we limit)
  (##wr-adr "pointer" obj we limit))

(define (##wr-promise obj we limit)
  (if (##writeenv-force? we)
    (force-vars (obj)
      (##wr obj we limit))
    (##wr-adr "promise" obj we limit)))

(define ##escape-ctrl-chars? #f)
(set! ##escape-ctrl-chars? #t)

(define (##wr-string obj we limit)
  (if (##writeenv-display? we)
    (##wr-str obj we limit)
    (##wr-str "\"" we
              (##wr-escaped-string obj #\" we
                                   (##wr-str "\"" we limit)))))

(define (##wr-escaped-string s special-escape we limit)
  (let loop ((i 0) (j 0) (limit limit))
    (if (##fixnum.< j (##string-length s))
      (let* ((c (##string-ref s j))
             (x (cond ((##char=? c special-escape)
                       c)
                      ((##assq-cdr c
                                   (##readtable-escaped-char-table
                                    (##writeenv-readtable we)))
                       => ##car)
                      (else
                       #f)))
             (n (##fixnum.<-char c))
             (j+1 (##fixnum.+ j 1)))
        (if (if (##fixnum.< n 32) ##escape-ctrl-chars? x)
          (let ((new-limit
                 (##wr-substr s i j we limit)))
            (if x
              (loop j+1
                    j+1
                    (##wr-ch x we
                             (##wr-str "\\" we
                                       new-limit)))
              (let ((oct (##number->string n 8)))
                (loop j+1
                      j+1
                      (##wr-str oct we
                                (##wr-str (if (##fixnum.< n 8) "\\00" "\\0")
                                          we
                                          new-limit))))))
          (loop i j+1 limit)))
      (##wr-substr s i j we limit))))

(define (##wr-will obj we limit)
  (##wr-adr "will" obj we limit))

(define (##wr-other obj we limit)
  (let ((x (##assq-cdr obj
                       (##readtable-sharp-bang-table
                        (##writeenv-readtable we)))))
    (if x
      (##wr-str (##car x) we
                (##wr-str "#!" we limit))
      (##wr-str "#<unknown>" we limit))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##write obj port rt force?)
  (let ((we (##make-writeenv port rt force? #f)))
    (##wr-unlimited obj we)))

(define (##display obj port rt force?)
  (let ((we (##make-writeenv port rt force? #t)))
    (##wr-unlimited obj we)))

(define (##pretty obj port rt force? col width)
  (let ((we (##make-writeenv port rt force? #f)))

    (define (force-if-appropriate x)
      (if (##writeenv-force? we)
        (force-vars (x) x)
        x))

    (define (write-str str)
      (##write-string str (##writeenv-port we)))

    (define (spaces n)
      (if (##fixnum.< 0 n)
        (let ((m (if (##fixnum.< 40 n) 40 n)))
          (##write-substring "                                        "
                             0
                             m
                             (##writeenv-port we))
          (spaces (##fixnum.- n m)))))

    (define (indent to from)
      (if (##fixnum.< to from)
        (begin
          (##newline (##writeenv-port we))
          (spaces to))
        (spaces (##fixnum.- to from))))

    (define (obj->string obj width)
      (let* ((port
              (##open-output-string))
             (we
              (##make-writeenv
               port
               (##writeenv-readtable we)
               (##writeenv-force? we)
               (##writeenv-display? we))))
        (##wr-limited obj we (##fixnum.+ width 1))
        (let* ((str (##close-port port))
               (len (##string-length str)))
          (if (##fixnum.< width len) #f str))))

    (define (p obj col width extra pp-pair)
      (let ((obj (force-if-appropriate obj)))
        (if (or (##pair? obj)
                (##vector? obj)
                (##u8vector? obj)
                (##u16vector? obj)
                (##u32vector? obj)
                (##f32vector? obj)
                (##f64vector? obj)
                (and (##structure? obj)
                     (##not (##port? obj))
                     (##not (##readtable? obj))))
          (let ((str
                 (obj->string obj (##fixnum.- (##fixnum.- width col) extra))))
            (if str
              (begin
                (write-str str)
                (##fixnum.+ col (##string-length str)))
              (cond ((##pair? obj)
                     (pp-pair obj col width extra))
                    ((##vector? obj)
                     (write-str "#")
                     (let ((col* (##fixnum.+ col 1))
                           (elems (##vector->list obj)))
                       (pp-list elems col* width extra pp-expr)))
                    ((##u8vector? obj)
                     (write-str "#u8")
                     (let ((col* (##fixnum.+ col 3))
                           (elems (##u8vector->list obj)))
                       (pp-list elems col* width extra pp-expr)))
                    ((##u16vector? obj)
                     (write-str "#u16")
                     (let ((col* (##fixnum.+ col 4))
                           (elems (##u16vector->list obj)))
                       (pp-list elems col* width extra pp-expr)))
                    ((##u32vector? obj)
                     (write-str "#u32")
                     (let ((col* (##fixnum.+ col 4))
                           (elems (##u32vector->list obj)))
                       (pp-list elems col* width extra pp-expr)))
                    ((##f32vector? obj)
                     (write-str "#f32")
                     (let ((col* (##fixnum.+ col 4))
                           (elems (##f32vector->list obj)))
                       (pp-list elems col* width extra pp-expr)))
                    ((##f64vector? obj)
                     (write-str "#f64")
                     (let ((col* (##fixnum.+ col 4))
                           (elems (##f64vector->list obj)))
                       (pp-list elems col* width extra pp-expr)))
                    (else
                     (write-str "#s")
                     (let ((col* (##fixnum.+ col 2))
                           (elems
                             (let* ((tag (##vector-ref obj 0))
                                    (n (##vector-ref tag 1)))
                               (let loop ((i n) (l '()))
                                 (if (##fixnum.< 0 i)
                                   (let ((field
                                          (##vector-ref tag (##fixnum.+ i 1)))
                                         (value
                                          (##vector-ref obj i)))
                                     (loop (##fixnum.- i 1)
                                           (##cons (##list field value) l)))
                                   (##cons (##vector-ref tag 0) l))))))
                       (pp-pair elems col* width extra))))))
          (##fixnum.+ col (##wr-unlimited obj we)))))

    (define (pp-expr expr col width extra)
      (let* ((head (force-if-appropriate (##car expr)))
             (style (pp-style head)))
        (if style
          (style expr col width extra)
          (if (##symbol? head)
            (if (##fixnum.< (##string-length (##symbol->string head)) 8)
              (pp-call expr col width extra pp-expr)
              (pp-general expr col width extra #f #f #f pp-expr))
            (pp-list expr col width extra pp-expr)))))

    ; (head item1
    ;       item2
    ;       item3)
    (define (pp-call expr col width extra pp-item)
      (write-str "(")
      (let* ((head (##car expr))
             (rest (##cdr expr))
             (col* (##fixnum.+ (##fixnum.+ col 1)
                               (##wr-unlimited head we))))
        (pp-down rest col* (##fixnum.+ col* 1) width extra pp-item)))

    ; (item1
    ;  item2
    ;  item3)
    (define (pp-list l col width extra pp-item)
      (write-str "(")
      (let ((col* (##fixnum.+ col 1)))
        (pp-down l col* col* width extra pp-item)))

    (define (pp-down l col1 col2 width extra pp-item)
      (let loop ((l l) (col* col1))
        (if (##pair? l)
          (let ((rest (##cdr l)))
            (let* ((rest (force-if-appropriate rest))
                   (extra* (if (##null? rest) (##fixnum.+ extra 1) 0)))
              (indent col2 col*)
              (loop rest (p (##car l) col2 width extra* pp-item))))
          (if (##null? l)
            (begin
              (write-str ")")
              (##fixnum.+ col* 1))
            (begin
              (indent col2 col*)
              (write-str ".")
              (indent col2 col*)
              (let* ((extra* (##fixnum.+ extra 1))
                     (col** (p l col2 width extra* pp-item)))
                (write-str ")")
                (##fixnum.+ col** 1)))))))

    (define (pp-expr-list l col width extra)
      (pp-list l col width extra pp-expr))

    (define (pp-abbrev expr col width extra prefix)
      (let ((rest (force-if-appropriate (##cdr expr))))
        (if (and (##pair? rest) (##null? (##cdr rest)))
          (let ((col* (##fixnum.+ col (##string-length prefix))))
            (write-str prefix)
            (p (##car rest) col* width extra pp-expr))
          (pp-call expr col width extra pp-expr))))

    (define (pp-general expr col width extra named? pp-1 pp-2 pp-3)

      (define (tail1 rest col1 col2 col3)
        (if (and pp-1 (##pair? rest))
          (begin
            (indent col3 col2)
            (let* ((val1 (##car rest))
                   (rest (force-if-appropriate (##cdr rest)))
                   (extra* (if (##null? rest) (##fixnum.+ extra 1) 0))
                   (col* (p val1 col3 width extra* pp-1)))
              (tail2 rest col1 col* col3)))
          (tail2 rest col1 col2 col3)))

      (define (tail2 rest col1 col2 col3)
        (if (and pp-2 (##pair? rest))
          (begin
            (indent col3 col2)
            (let* ((val1 (##car rest))
                   (rest (force-if-appropriate (##cdr rest)))
                   (extra* (if (##null? rest) (##fixnum.+ extra 1) 0))
                   (col* (p val1 col3 width extra* pp-2)))
              (tail3 rest col1 col*)))
          (tail3 rest col1 col2)))

      (define (tail3 rest col1 col2)
        (pp-down rest col2 col1 width extra pp-3))

      (write-str "(")
      (let* ((head (##car expr))
             (rest (force-if-appropriate (##cdr expr)))
             (col* (##fixnum.+ (##fixnum.+ col 1)
                               (##wr-unlimited head we))))
        (if (and named? (##pair? rest))
          (begin
            (write-str " ")
            (let* ((name (##car rest))
                   (rest (force-if-appropriate (##cdr rest)))
                   (col** (##fixnum.+ (##fixnum.+ col* 1)
                                      (##wr-unlimited name we))))
              (tail1 rest (##fixnum.+ col 2) col** (##fixnum.+ col** 1))))
          (tail1 rest (##fixnum.+ col 2) col* (##fixnum.+ col* 1)))))

    (define (pp-quote expr col width extra)
      (pp-abbrev expr col width extra "'"))

    (define (pp-quasiquote expr col width extra)
      (pp-abbrev expr col width extra "`"))

    (define (pp-unquote expr col width extra)
      (pp-abbrev expr col width extra ","))

    (define (pp-unquote-splicing expr col width extra)
      (pp-abbrev expr col width extra ",@"))

    (define (pp-lambda expr col width extra)
      (pp-general expr col width extra #f pp-expr-list #f pp-expr))

    (define (pp-if expr col width extra)
      (pp-general expr col width extra #f pp-expr #f pp-expr))

    (define (pp-set! expr col width extra)
      (pp-general expr col width extra #f pp-expr #f pp-expr))

    (define (pp-cond expr col width extra)
      (pp-call expr col width extra pp-expr-list))

    (define (pp-case expr col width extra)
      (pp-general expr col width extra #f pp-expr #f pp-expr-list))

    (define (pp-and expr col width extra)
      (pp-call expr col width extra pp-expr))

    (define (pp-or expr col width extra)
      (pp-call expr col width extra pp-expr))

    (define (pp-let expr col width extra)
      (let* ((rest (force-if-appropriate (##cdr expr)))
             (named? (and (##pair? rest) (##symbol? (##car rest)))))
        (pp-general expr col width extra named? pp-expr-list #f pp-expr)))

    (define (pp-let* expr col width extra)
      (pp-general expr col width extra #f pp-expr-list #f pp-expr))

    (define (pp-letrec expr col width extra)
      (pp-general expr col width extra #f pp-expr-list #f pp-expr))

    (define (pp-begin expr col width extra)
      (pp-general expr col width extra #f #f #f pp-expr))

    (define (pp-do expr col width extra)
      (pp-general expr col width extra #f pp-expr-list pp-expr-list pp-expr))

    (define (pp-define expr col width extra)
      (pp-general expr col width extra #f pp-expr-list #f pp-expr))

    (define (pp-style x)
      (case x
        ((quote) pp-quote)
        ((quasiquote) pp-quasiquote)
        ((unquote) pp-unquote)
        ((unquote-splicing) pp-unquote-splicing)
        ((lambda) pp-lambda)
        ((if) pp-if)
        ((set!) pp-set!)
        ((cond) pp-cond)
        ((case) pp-case)
        ((and) pp-and)
        ((or) pp-or)
        ((let) pp-let)
        ((let*) pp-let*)
        ((letrec) pp-letrec)
        ((begin) pp-begin)
        ((do) pp-do)
        ((define) pp-define)
        (else #f)))

    (p obj col width 0 pp-expr)))

(define (##pretty-print obj port rt)
  (##pretty obj
            port
            rt
            (if-forces #t #f)
            0
            (##port-width port))
  (##newline port))

(define (##object->string obj rt width force?)
  (let* ((port (##open-output-string))
         (we (##make-writeenv port rt force? #f)))
    (##wr-limited obj we (##fixnum.+ width 1))
    (let* ((str (##close-port port))
           (len (##string-length str)))
      (if (##fixnum.< width len)
        (begin
          (if (##fixnum.< 0 width)
            (begin
              (##string-set! str (##fixnum.- width 1) #\.)
              (if (##fixnum.< 1 width)
                (begin
                  (##string-set! str (##fixnum.- width 2) #\.)
                  (if (##fixnum.< 2 width)
                    (##string-set! str (##fixnum.- width 3) #\.))))))
          (##string-shrink! str width)
          str)
        str))))

;------------------------------------------------------------------------------

(define (##current-input-port)
  (##dynamic-ref '##current-input-port ##stdin))

(define (##current-output-port)
  (##dynamic-ref '##current-output-port ##stdout))

(define (##current-readtable)
  (##dynamic-ref '##current-readtable ##main-readtable))

;------------------------------------------------------------------------------

; IEEE Scheme procedures:

(define (call-with-input-file
         path
         proc
         #!optional (char-encoding (absent-obj)))
  (force-vars (path proc char-encoding)
    (check-string path (call-with-input-file path proc char-encoding)
      (check-procedure proc (call-with-input-file path proc char-encoding)
        (let ((port
               (##open-input-file
                path
                (if (##eq? char-encoding (absent-obj)) #f char-encoding))))
          (if port
            (let ((result (proc port)))
              (##close-port port)
              result)
            (trap-open-file
             (call-with-input-file path proc char-encoding))))))))

(define (call-with-output-file
         path
         proc
         #!optional (char-encoding (absent-obj)))
  (force-vars (path proc char-encoding)
    (check-string path (call-with-output-file path proc char-encoding)
      (check-procedure proc (call-with-output-file path proc char-encoding)
        (let ((port
               (##open-output-file
                path
                (if (##eq? char-encoding (absent-obj)) #f char-encoding))))
          (if port
            (let ((result (proc port)))
              (##close-port port)
              result)
            (trap-open-file
             (call-with-output-file path proc char-encoding))))))))

(define (input-port? x)
  (force-vars (x)
    (and (##port? x) (##input-port? x))))

(define (output-port? x)
  (force-vars (x)
    (and (##port? x) (##output-port? x))))

(define (current-input-port)
  (##current-input-port))

(define (current-output-port)
  (##current-output-port))

(define (open-input-file
         path
         #!optional (char-encoding (absent-obj)))
  (force-vars (path char-encoding)
    (check-string path (open-input-file path char-encoding)
      (let ((port
             (##open-input-file
              path
              (if (##eq? char-encoding (absent-obj)) #f char-encoding))))
        (if port
          port
          (trap-open-file (open-input-file path char-encoding)))))))

(define (open-output-file
         path
         #!optional (char-encoding (absent-obj)))
  (force-vars (path char-encoding)
    (check-string path (open-output-file path char-encoding)
      (let ((port
             (##open-output-file
              path
              (if (##eq? char-encoding (absent-obj)) #f char-encoding))))
        (if port
          port
          (trap-open-file (open-output-file path char-encoding)))))))

(define (close-input-port p)
  (force-vars (p)
    (check-input-port p (close-input-port p)
      (##close-port p))))

(define (close-output-port p)
  (force-vars (p)
    (check-output-port p (close-output-port p)
      (##close-port p))))

(define-system (##eof-object? x)
  (##eq? x #!eof))

(define (eof-object? x)
  (force-vars (x)
    (##eof-object? x)))

(define (read #!optional (p (absent-obj)) (r (absent-obj)))
  (force-vars (p r)
    (let* ((port
            (if (##eq? p (absent-obj))
              (##current-input-port)
              p))
           (rt
            (if (##eq? r (absent-obj))
              (##current-readtable)
              r)))
      (check-input-port port (read p r)
        (check-open-port port (read p r)
          (check-readtable rt (read p r)
            (##read port rt)))))))

(define (read-char #!optional (p (absent-obj)))
  (force-vars (p)
    (let ((port
           (if (##eq? p (absent-obj))
             (##current-input-port)
             p)))
      (check-input-port port (read-char p)
        (check-open-port port (read-char p)
          (##read-char port))))))

(define (peek-char #!optional (p (absent-obj)))
  (force-vars (p)
    (let ((port
           (if (##eq? p (absent-obj))
             (##current-input-port)
             p)))
      (check-input-port port (peek-char p)
        (check-open-port port (peek-char p)
          (##peek-char port))))))
  
(define (write obj #!optional (p (absent-obj)) (r (absent-obj)))
  (force-vars (obj p r)
    (let* ((port
            (if (##eq? p (absent-obj))
              (##current-output-port)
              p))
           (rt
            (if (##eq? r (absent-obj))
              (##current-readtable)
              r)))
      (check-output-port port (write obj p r)
        (check-open-port port (write obj p r)
          (check-readtable rt (write obj p r)
            (begin
              (##write obj port rt (if-forces #t #f))
              (##void))))))))

(define (display obj #!optional (p (absent-obj)) (r (absent-obj)))
  (force-vars (obj p r)
    (let* ((port
            (if (##eq? p (absent-obj))
              (##current-output-port)
              p))
           (rt
            (if (##eq? r (absent-obj))
              (##current-readtable)
              r)))
      (check-output-port port (display obj p r)
        (check-open-port port (display obj p r)
          (check-readtable rt (display obj p r)
            (begin
              (##display obj port rt (if-forces #t #f))
              (##void))))))))

(define (newline #!optional (p (absent-obj)))
  (force-vars (p)
    (let ((port
           (if (##eq? p (absent-obj))
             (##current-output-port)
             p)))
      (check-output-port port (newline p)
        (check-open-port port (newline p)
          (begin
            (##newline port)
            (##void)))))))

(define (write-char c #!optional (p (absent-obj)))
  (force-vars (c p)
    (let ((port
           (if (##eq? p (absent-obj))
             (##current-output-port)
             p)))
      (check-char c (write-char c p)
        (check-output-port port (write-char c p)
          (check-open-port port (write-char c p)
            (begin
              (##write-char c port)
              (##void))))))))

;------------------------------------------------------------------------------

; R4RS Scheme procedures:

(define (with-input-from-file
         path
         thunk
         #!optional (char-encoding (absent-obj)))
  (force-vars (path thunk char-encoding)
    (check-string path (with-input-from-file path thunk char-encoding)
      (check-procedure thunk (with-input-from-file path thunk char-encoding)
        (let ((port
               (##open-input-file
                path
                (if (##eq? char-encoding (absent-obj)) #f char-encoding))))
          (if port
            (let ((result
                    (##dynamic-let
                      (##list (##cons '##current-input-port port))
                      thunk)))
              (##close-port port)
              result)
            (trap-open-file (with-input-from-file path thunk char-encoding))))))))

(define (with-output-to-file
         path
         thunk
         #!optional (char-encoding (absent-obj)))
  (force-vars (path thunk char-encoding)
    (check-string path (with-output-to-file path thunk char-encoding)
      (check-procedure thunk (with-output-to-file path thunk char-encoding)
        (let ((port
               (##open-output-file
                path
                (if (##eq? char-encoding (absent-obj)) #f char-encoding))))
          (if port
            (let ((result
                    (##dynamic-let
                      (##list (##cons '##current-output-port port))
                      thunk)))
              (##close-port port)
              result)
            (trap-open-file (with-output-to-file path thunk char-encoding))))))))

(define (char-ready? #!optional (p (absent-obj)))
  (force-vars (p)
    (let ((port
           (if (##eq? p (absent-obj))
             (##current-input-port)
             p)))
      (check-input-port port (char-ready? p)
        (check-open-port port (char-ready? p)
          (##char-ready? port))))))

(define (transcript-on path)
  (check-string path (transcript-on path)
    (##void)))

(define (transcript-off)
  (##void))

;------------------------------------------------------------------------------

; Non-standard procedures:

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (open-input-pipe
         command
         #!optional (char-encoding (absent-obj)))
  (force-vars (command char-encoding)
    (check-string command (open-input-pipe command char-encoding)
      (let ((port
             (##open-input-pipe
              command
              (if (##eq? char-encoding (absent-obj)) #f char-encoding))))
        (if port
          port
          (trap-open-pipe (open-input-pipe command char-encoding)))))))

(define (open-output-pipe
         command
         #!optional (char-encoding (absent-obj)))
  (force-vars (command char-encoding)
    (check-string command (open-output-pipe command char-encoding)
      (let ((port
             (##open-output-pipe
              command
              (if (##eq? char-encoding (absent-obj)) #f char-encoding))))
        (if port
          port
          (trap-open-pipe (open-output-pipe command char-encoding)))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (open-input-string s)
  (force-vars (s)
    (check-string s (open-input-string s)
      (##open-input-string s))))

(define (open-output-string)
  (##open-output-string))

(define (with-input-from-string s thunk)
  (force-vars (s thunk)
    (check-string s (with-input-from-string s thunk)
      (check-procedure thunk (with-input-from-string s thunk)
        (let ((port (##open-input-string s)))
          (##dynamic-let
            (##list (##cons '##current-input-port port))
            thunk))))))

(define (with-output-to-string thunk)
  (force-vars (thunk)
    (check-procedure thunk (with-output-to-string thunk)
      (let ((port (##open-output-string)))
        (##dynamic-let
          (##list (##cons '##current-output-port port))
          thunk)
        (##close-port port)))))

(define (call-with-input-string s proc)
  (force-vars (s proc)
    (check-string s (call-with-input-string s proc)
      (check-procedure proc (call-with-input-string s proc)
        (let ((port (##open-input-string s)))
          (proc port))))))

(define (call-with-output-string proc)
  (force-vars (proc)
    (check-procedure proc (call-with-output-string proc)
      (let ((port (##open-output-string)))
        (proc port)
        (##close-port port)))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (with-input-from-port port thunk)
  (force-vars (port thunk)
    (check-input-port port (with-input-from-port port thunk)
      (check-open-port port (with-input-from-port port thunk)
        (check-procedure thunk (with-input-from-port port thunk)
          (##dynamic-let
            (##list (##cons '##current-input-port port))
            thunk))))))

(define (with-output-to-port port thunk)
  (force-vars (port thunk)
    (check-output-port port (with-output-to-port port thunk)
      (check-open-port port (with-output-to-port port thunk)
        (check-procedure thunk (with-output-to-port port thunk)
          (##dynamic-let
            (##list (##cons '##current-output-port port))
            thunk))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (file-exists? path)
  (force-vars (path)
    (check-string path (file-exists? path)
      (let ((port (##open-input-file path #f)))
        (if port
          (begin
            (##close-port port)
            #t)
          #f)))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (flush-output #!optional (p (absent-obj)))
  (force-vars (p)
    (let ((port
           (if (##eq? p (absent-obj))
             (##current-output-port)
             p)))
      (check-output-port port (flush-output p)
        (check-open-port port (flush-output p)
          (begin
            (##flush-output port)
            (##void)))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (pretty-print obj #!optional (p (absent-obj)) (r (absent-obj)))
  (force-vars (obj p r)
    (let* ((port
            (if (##eq? p (absent-obj))
              (##current-output-port)
              p))
           (rt
            (if (##eq? r (absent-obj))
              (##current-readtable)
              r)))
      (check-output-port port (pretty-print obj p r)
        (check-open-port port (pretty-print obj p r)
          (check-readtable rt (pretty-print obj p r)
            (begin
              (##pretty-print obj port rt)
              (##void))))))))

(define (pp obj #!optional (p (absent-obj)) (r (absent-obj)))
  (force-vars (obj p r)
    (let* ((port
            (if (##eq? p (absent-obj))
              (##current-output-port)
              p))
           (rt
            (if (##eq? r (absent-obj))
              (##current-readtable)
              r)))
      (check-output-port port (pp obj p r)
        (check-open-port port (pp obj p r)
          (check-readtable rt (pp obj p r)
            (begin
              (if (##procedure? obj)
                (##pretty-print (##decompile obj) port rt)
                (##pretty-print obj port rt))
              (##void))))))))

;------------------------------------------------------------------------------

; The reader.

(##declare (inlining-limit 300))

(##define-macro (* . args)                `(##fixnum.* ,@args))
(##define-macro (+ . args)                `(##fixnum.+ ,@args))
(##define-macro (- . args)                `(##fixnum.- ,@args))
(##define-macro (< . args)                `(##fixnum.< ,@args))
(##define-macro (assq . args)             `(##assq ,@args))
(##define-macro (cdr . args)              `(##cdr ,@args))
(##define-macro (char-downcase . args)    `(##char-downcase ,@args))
(##define-macro (char-upcase . args)      `(##char-upcase ,@args))
(##define-macro (char<? . args)           `(##char<? ,@args))
(##define-macro (char=? . args)           `(##char=? ,@args))
(##define-macro (char? . args)            `(##char? ,@args))
(##define-macro (cons . args)             `(##cons ,@args))
(##define-macro (eq? . args)              `(##eq? ,@args))
(##define-macro (exact? . args)           `(##exact? ,@args))
(##define-macro (for-each . args)         `(##for-each ,@args))
(##define-macro (integer? . args)         `(##integer? ,@args))
(##define-macro (list . args)             `(##list ,@args))
(##define-macro (make-string . args)      `(##make-string ,@args))
(##define-macro (make-vector . args)      `(##make-vector ,@args))
(##define-macro (modulo . args)           `(##modulo ,@args))
(##define-macro (not . args)              `(##not ,@args))
(##define-macro (null? . args)            `(##null? ,@args))
(##define-macro (pair? . args)            `(##pair? ,@args))
(##define-macro (quotient . args)         `(##quotient ,@args))
(##define-macro (real? . args)            `(##real? ,@args))
(##define-macro (reverse . args)          `(##reverse ,@args))
(##define-macro (set-cdr! . args)         `(##set-cdr! ,@args))
(##define-macro (string->number . args)   `(##string->number ,@args))
(##define-macro (string->symbol . args)   `(##string->symbol ,@args))
(##define-macro (string-length . args)    `(##string-length ,@args))
(##define-macro (string-ref . args)       `(##string-ref ,@args))
(##define-macro (string-set! . args)      `(##string-set! ,@args))
(##define-macro (string-ci=? . args)      `(##string-ci=? ,@args))
(##define-macro (symbol->string . args)   `(##symbol->string ,@args))
(##define-macro (vector . args)           `(##vector ,@args))
(##define-macro (vector-ref . args)       `(##vector-ref ,@args))
(##define-macro (vector-set! . args)      `(##vector-set! ,@args))
(##define-macro (vector? . args)          `(##vector? ,@args))

(##define-macro (make-u8vect n)           `(##make-u8vector ,n 0))
(##define-macro (u8vect-set! . args)      `(##u8vector-set! ,@args))
(##define-macro (make-u16vect n)          `(##make-u16vector ,n 0))
(##define-macro (u16vect-set! . args)     `(##u16vector-set! ,@args))
(##define-macro (make-u32vect n)          `(##make-u32vector ,n 0))
(##define-macro (u32vect-set! . args)     `(##u32vector-set! ,@args))
(##define-macro (make-f32vect n)          `(##make-f32vector ,n (inexact-0)))
(##define-macro (f32vect-set! . args)     `(##f32vector-set! ,@args))
(##define-macro (make-f64vect n)          `(##make-f64vector ,n (inexact-0)))
(##define-macro (f64vect-set! . args)     `(##f64vector-set! ,@args))

(##define-macro (unicode->character . args) `(##fixnum.->char ,@args))
(##define-macro (character->unicode . args) `(##fixnum.<-char ,@args))
(##define-macro (in-unicode-range? n) `(##not (##< ##max-unicode ,n)))

(##define-macro (string->keyword-object . args) `(##string->keyword ,@args))

(##define-macro (in-integer-range? n lo hi)
  `(and (##not (##< ,n ,lo)) (##not (##< ,hi ,n))))

(##define-macro (false-obj) #f)

; Tables for reader.

(define ##standard-escaped-char-table
  '((#\\ . #\\)
    (#\a . #\bel)
    (#\b . #\backspace)
    (#\t . #\tab)
    (#\n . #\newline)
    (#\v . #\vt)
    (#\f . #\page)
    (#\r . #\return)))

(define ##standard-named-char-table
  '(("newline"   . #\newline) ; here to take precedence over linefeed
    ("space"     . #\space)
    ("nul"       . #\nul)
    ("bel"       . #\bel)
    ("backspace" . #\backspace)
    ("tab"       . #\tab)
    ("linefeed"  . #\linefeed)
    ("vt"        . #\vt)
    ("page"      . #\page)
    ("return"    . #\return)
    ("rubout"    . #\rubout)))

(define ##standard-sharp-bang-table
  '(("optional" . #!optional)
    ("rest"     . #!rest)
    ("key"      . #!key)
    ("eof"      . #!eof)
    (""         . #!)))

;==============================================================================

; For compatibility between the interpreter and compiler, this section
; must be the same as the corresponding section in the file
; "gsc/_source.scm" (except that ## and ** are exchanged).

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; A chartable structure is a vector-like data structure which is
; indexed using a character.

(define (##make-chartable default)
  (vector (make-vector 128 default) default '()))

(define (##chartable-ref ct c)
  (let ((i (character->unicode c)))
    (if (< i 128)
      (vector-ref (vector-ref ct 0) i)
      (let ((x (assq i (vector-ref ct 2))))
        (if x
          (cdr x)
          (vector-ref ct 1))))))

(define (##chartable-set! ct c val)
  (let ((i (character->unicode c)))
    (if (< i 128)
      (vector-set! (vector-ref ct 0) i val)
      (let ((x (assq i (vector-ref ct 2))))
        (if x
          (set-cdr! x val)
          (vector-set! ct 2 (cons (cons i val) (vector-ref ct 2))))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; A readtable structure contains parsing information for the reader.
; It indicates what action must be taken when a given character is
; encountered.

(define ##readtable-tag '#(readtable 0))

(define (##make-readtable
         case-conversion?
         keywords-allowed?
         escaped-char-table
         named-char-table
         sharp-bang-table
         char-delimiter?-table
         char-handler-table)
  (##subtype-set!
    (vector
     ##readtable-tag
     case-conversion?
     keywords-allowed?
     escaped-char-table
     named-char-table
     sharp-bang-table
     char-delimiter?-table
     char-handler-table)
    (subtype-structure)))

(define (##readtable-case-conversion? rt)
  (vector-ref rt 1))

(define (##readtable-case-conversion?-set! rt x)
  (vector-set! rt 1 x))

(define (##readtable-keywords-allowed? rt)
  (vector-ref rt 2))

(define (##readtable-keywords-allowed?-set! rt x)
  (vector-set! rt 2 x))

(define (##readtable-escaped-char-table rt)
  (vector-ref rt 3))

(define (##readtable-escaped-char-table-set! rt x)
  (vector-set! rt 3 x))

(define (##readtable-named-char-table rt)
  (vector-ref rt 4))

(define (##readtable-named-char-table-set! rt x)
  (vector-set! rt 4 x))

(define (##readtable-sharp-bang-table rt)
  (vector-ref rt 5))

(define (##readtable-sharp-bang-table-set! rt x)
  (vector-set! rt 5 x))

(define (##readtable-char-delimiter?-table rt)
  (vector-ref rt 6))

(define (##readtable-char-delimiter?-table-set! rt x)
  (vector-set! rt 6 x))

(define (##readtable-char-handler-table rt)
  (vector-ref rt 7))

(define (##readtable-char-handler-table-set! rt x)
  (vector-set! rt 7 x))

(define (##readtable-char-delimiter? rt c)
  (##chartable-ref (##readtable-char-delimiter?-table rt) c))

(define (##readtable-char-delimiter?-set! rt c delimiter?)
  (##chartable-set! (##readtable-char-delimiter?-table rt) c delimiter?))

(define (##readtable-char-handler rt c)
  (##chartable-ref (##readtable-char-handler-table rt) c))

(define (##readtable-char-handler-set! rt c handler)
  (##chartable-set! (##readtable-char-handler-table rt) c handler))

(define (##readtable-char-class-set! rt c delimiter? handler)
  (begin
    (##readtable-char-delimiter?-set! rt c delimiter?)
    (##readtable-char-handler-set! rt c handler)))

(define (##readtable-convert-case rt c)
  (let ((case-conversion? (##readtable-case-conversion? rt)))
    (if case-conversion?
      (if (eq? case-conversion? 'upcase)
        (char-upcase c)
        (char-downcase c))
      c)))

(define (##readtable-string-convert-case! rt s)
  (let ((case-conversion? (##readtable-case-conversion? rt)))
    (if case-conversion?
      (if (eq? case-conversion? 'upcase)
        (let loop ((i (- (string-length s) 1)))
          (if (not (< i 0))
            (begin
              (string-set! s i (char-upcase (string-ref s i)))
              (loop (- i 1)))))
        (let loop ((i (- (string-length s) 1)))
          (if (not (< i 0))
            (begin
              (string-set! s i (char-downcase (string-ref s i)))
              (loop (- i 1)))))))))

(define (##readtable-parse-keyword rt s)
  (let ((keywords-allowed? (##readtable-keywords-allowed? rt)))
    (and keywords-allowed?
         (let ((len (string-length s)))
           (and (< 1 len)
                (if (eq? keywords-allowed? 'prefix)
                    (and (char=? (string-ref s 0) #\:)
                         (string->keyword-object
                          (substring s 1 len)))
                    (and (char=? (string-ref s (- len 1)) #\:)
                         (string->keyword-object
                          (substring s 0 (- len 1))))))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Error handling.

(define (##read-error-datum-or-eof-expected re)
  ((##readenv-error-proc re) re "Datum or EOF expected"))

(define (##read-error-datum-expected re)
  ((##readenv-error-proc re) re "Datum expected"))

(define (##read-error-improperly-placed-dot re)
  ((##readenv-error-proc re) re "Improperly placed dot"))

(define (##read-error-incomplete-form-eof-reached re)
  ((##readenv-error-proc re) re "Incomplete form, EOF reached"))

(define (##read-error-incomplete re)
  ((##readenv-error-proc re) re "Incomplete form"))

(define (##read-error-char-name re str)
  ((##readenv-error-proc re) re "Invalid '#\\' name:" str))

(define (##read-error-illegal-char re c)
  ((##readenv-error-proc re) re "Illegal character:" c))

(define (##read-error-u8 re)
  ((##readenv-error-proc re) re "8 bit exact integer expected"))

(define (##read-error-u16 re)
  ((##readenv-error-proc re) re "16 bit exact integer expected"))

(define (##read-error-u32 re)
  ((##readenv-error-proc re) re "32 bit exact integer expected"))

(define (##read-error-f32/f64 re)
  ((##readenv-error-proc re) re "Inexact real expected"))

(define (##read-error-hex re)
  ((##readenv-error-proc re) re "Invalid hexadecimal escape"))

(define (##read-error-escaped-char re c)
  ((##readenv-error-proc re) re "Invalid escaped character:" c))

(define (##read-error-vector re)
  ((##readenv-error-proc re) re "'(' expected"))

(define (##read-error-sharp-token re str)
  ((##readenv-error-proc re) re "Invalid token:" str))

(define (##read-error-sharp-bang-name re str)
  ((##readenv-error-proc re) re "Invalid '#!' name:" str))

(define (##read-error-char-range re)
  ((##readenv-error-proc re) re "Character out of range"))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Procedures to read single characters.

(define (##peek-next-char re) ; never returns end-of-file
  (let ((next (##peek-next-char-or-eof re)))
    (if (char? next)
      next
      (##read-error-incomplete-form-eof-reached re))))

(define (##read-next-char re) ; never returns end-of-file
  (let ((c (##read-next-char-or-eof re)))
    (if (char? c)
      c
      (##read-error-incomplete-form-eof-reached re))))

(define (##read-next-char-expecting re c) ; only accepts c as the next char
  (let ((x (##read-next-char-or-eof re)))
    (if (char? x)
      (if (not (char=? x c))
        (##read-error-incomplete re))
      (##read-error-incomplete-form-eof-reached re))
    x))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Procedures to read datums.

; (##read-datum-or-eof re) attempts to read a datum in the read
; environment "re", skipping all whitespace and comments in the
; process.  The "pos" field of the read environment indicates the
; position where the enclosing datum starts (e.g. list or vector).  If
; a datum is read it is returned (wrapped if the read environment asks
; for it); if the end-of-file is reached the end-of-file object is
; returned (never wrapped); otherwise an error is signaled.  The read
; environment's "pos" field is only modified if a datum was read, in
; which case it is the position where the datum starts.

(define (##read-datum-or-eof re)
  (let ((obj (##read-datum-or-none re)))
    (if (eq? obj (##none-marker))
      (let ((c (##peek-next-char-or-eof re)))
        (if (char? c)
          (begin
            (##readenv-filepos-set! re (##readenv-current-filepos re))
            (##read-next-char-or-eof re) ; to make sure reader makes progress
            (##read-error-datum-or-eof-expected re))
          (begin
            (##read-next-char-or-eof re) ; to make sure reader makes progress
            c))) ; end-of-file was reached so return end-of-file object
      obj)))

; (##read-datum re) attempts to read a datum in the read environment
; "re", skipping all whitespace and comments in the process.  The
; "pos" field of the read environment indicates the position where the
; enclosing datum starts (e.g. list or vector).  If a datum is read it
; is returned (wrapped if the read environment asks for it); if the
; end-of-file is reached or no datum can be read an error is signaled.
; The read environment's "pos" field is only modified if a datum was
; read, in which case it is the position where the datum starts.

(define (##read-datum re)
  (let ((obj (##read-datum-or-none re)))
    (if (eq? obj (##none-marker))
      (begin
        (##readenv-filepos-set! re (##readenv-current-filepos re))
        (##read-next-char-or-eof re) ; to make sure reader makes progress
        (##read-error-datum-expected re))
      obj)))

; (##read-datum-or-none re) attempts to read a datum in the read
; environment "re", skipping all whitespace and comments in the
; process.  The "pos" field of the read environment indicates the
; position where the enclosing datum starts (e.g. list or vector).  If
; a datum is read it is returned (wrapped if the read environment asks
; for it); if the end-of-file is reached or no datum can be read the
; "none-marker" is returned.  The read environment's "pos" field is
; only modified if a datum was read, in which case it is the position
; where the datum starts.

(define (##read-datum-or-none re)
  (let ((obj (##read-datum-or-none-or-dot re)))
    (if (eq? obj (##dot-marker))
      (begin
        (##readenv-filepos-set! re (##readenv-previous-filepos re 1))
        (##read-error-improperly-placed-dot re))
      obj)))

; (##read-datum-or-none-or-dot re) attempts to read a datum in the
; read environment "re", skipping all whitespace and comments in the
; process.  The "pos" field of the read environment indicates the
; position where the enclosing datum starts (e.g. list or vector).  If
; a datum is read it is returned (wrapped if the read environment asks
; for it); if a lone dot is read the "dot-marker" is returned; if the
; end-of-file is reached or no datum can be read the "none-marker" is
; returned.  The read environment's "pos" field is only modified if a
; datum was read, in which case it is the position where the datum
; starts.

(define (##read-datum-or-none-or-dot re)
  (let ((next (##peek-next-char-or-eof re)))
    (if (char? next)
      ((##readtable-char-handler (##readenv-readtable re) next) re next)
      (##none-marker))))

; Special objects returned by ##read-datum-or-none-or-dot.

(define (##none-marker) '#(none)) ; indicates no following datum
(define (##dot-marker) '#(dot))   ; indicates an isolated dot

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Procedure to read a list of datums (possibly an improper list).

(define (##build-list re allow-improper? start-pos close)
  (let ((obj (##read-datum-or-none re)))
    (if (eq? obj (##none-marker))
      (begin
        (##read-next-char-expecting re close)
        '())
      (let ((lst (cons obj '())))
        (##readenv-filepos-set! re start-pos) ; restore pos
        (let loop ((end lst))
          (let ((obj
                 (if allow-improper?
                   (##read-datum-or-none-or-dot re)
                   (##read-datum-or-none re))))
            (cond ((eq? obj (##none-marker))
                   (##read-next-char-expecting re close)
                   lst)
                  ((eq? obj (##dot-marker))
                   (let ((obj (##read-datum re)))
                     (set-cdr! end obj)
                     (##readenv-filepos-set! re start-pos) ; restore pos
                     (let ((x (##read-datum-or-none re))) ; skip whitespace!
                       (if (eq? x (##none-marker))
                         (begin
                           (##read-next-char-expecting re close)
                           lst)
                         (begin
                           (##readenv-filepos-set! re start-pos) ; restore pos
                           (##read-error-incomplete re))))))
                  (else
                   (##readenv-filepos-set! re start-pos) ; restore pos
                   (let ((tail (cons obj '())))
                     (set-cdr! end tail)
                     (loop tail))))))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Procedure to read a vector or byte vector.

(define (##build-vector re kind start-pos close)

  (define (exact-integer-check n lo hi)
    (and (integer? n)
         (exact? n)
         (in-integer-range? n lo hi)))

  (define (inexact-real-check n)
    (and (real? n)
         (not (exact? n))))

  (let loop ((i 0))
    (let* ((x (##read-datum-or-none re))
           (x-pos (##readenv-filepos re)))
      (##readenv-filepos-set! re start-pos) ; restore pos
      (if (eq? x (##none-marker))
        (begin
          (##read-next-char-expecting re close)
          (case kind
            ((vector)    (make-vector i #f))
            ((u8vector)  (make-u8vect i))
            ((u16vector) (make-u16vect i))
            ((u32vector) (make-u32vect i))
            ((f32vector) (make-f32vect i))
            ((f64vector) (make-f64vect i))))
        (let ((vect (loop (+ i 1))))
          (case kind
            ((vector)
             (vector-set! vect i x))
            ((u8vector)
             (let ((ux (##readenv-unwrap re x)))
               (if (not (exact-integer-check ux 0 255))
                 (begin
                   (##readenv-filepos-set! re x-pos) ; restore pos of element
                   (##read-error-u8 re)))
               (u8vect-set! vect i ux)))
            ((u16vector)
             (let ((ux (##readenv-unwrap re x)))
               (if (not (exact-integer-check ux 0 65535))
                 (begin
                   (##readenv-filepos-set! re x-pos) ; restore pos of element
                   (##read-error-u16 re)))
               (u16vect-set! vect i ux)))
            ((u32vector)
             (let ((ux (##readenv-unwrap re x)))
               (if (not (exact-integer-check ux 0 4294967295))
                 (begin
                   (##readenv-filepos-set! re x-pos) ; restore pos of element
                   (##read-error-u32 re)))
               (u32vect-set! vect i ux)))
            ((f32vector)
             (let ((ux (##readenv-unwrap re x)))
               (if (not (inexact-real-check ux))
                 (begin
                   (##readenv-filepos-set! re x-pos) ; restore pos of element
                   (##read-error-f32/f64 re)))
               (f32vect-set! vect i ux)))
            ((f64vector)
             (let ((ux (##readenv-unwrap re x)))
               (if (not (inexact-real-check ux))
                 (begin
                   (##readenv-filepos-set! re x-pos) ; restore pos of element
                   (##read-error-f32/f64 re)))
               (f64vect-set! vect i ux))))
          vect)))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Procedures to read delimited tokens.

(define (##build-delimited-string re c i)
  (let loop ((i i))
    (let ((next (##peek-next-char-or-eof re)))
      (if (or (not (char? next))
              (##readtable-char-delimiter? (##readenv-readtable re) next))
        (make-string i c)
        (begin
          (##read-next-char-or-eof re) ; skip "next"
          (let ((s (loop (+ i 1))))
            (string-set! s i next)
            s))))))

(define (##build-delimited-number/keyword/symbol re c)
  (let ((s (##build-delimited-string re c 1)))
    (or (string->number s 10)
        (begin
          (##readtable-string-convert-case! (##readenv-readtable re) s)
          (or (##readtable-parse-keyword (##readenv-readtable re) s)
              (string->symbol s))))))

(define (##build-delimited-symbol re c i)
  (let ((s (##build-delimited-string re c i)))
    (##readtable-string-convert-case! (##readenv-readtable re) s)
    (string->symbol s)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##build-escaped-string-up-to re close)

  (define (char-octal? c)
    (and (not (char<? c #\0)) (not (char<? #\7 c))))

  (define (char-hexadecimal? c)
    (or (and (not (char<? c #\0)) (not (char<? #\9 c)))
        (and (not (char<? c #\a)) (not (char<? #\f c)))
        (and (not (char<? c #\A)) (not (char<? #\F c)))))

  (define (unicode n)
    (if (in-unicode-range? n)
      (unicode->character n)
      (##read-error-char-range re)))

  (define (read-escape-octal c)
    (let ((str (let loop ((i 1))
                 (let ((next (##peek-next-char-or-eof re)))
                   (if (and (< i 3)
                            (char? next)
                            (char-octal? next))
                     (begin
                       (##read-next-char-or-eof re) ; skip "next"
                       (let ((s (loop (+ i 1))))
                         (string-set! s i next)
                         s))
                     (make-string i #\space))))))
      (string-set! str 0 c)
      (unicode (string->number str 8))))

  (define (read-escape-hexadecimal)
    (let ((next (##peek-next-char-or-eof re)))
      (if (and (char? next)
               (char-hexadecimal? next))
        (begin
          (##read-next-char-or-eof re) ; skip "next"
          (let ((str (let loop ((i 1))
                       (let ((next2 (##peek-next-char-or-eof re)))
                         (if (and (char? next2)
                                  (char-hexadecimal? next2))
                           (begin
                             (##read-next-char-or-eof re) ; skip "next2"
                             (let ((s (loop (+ i 1))))
                               (string-set! s i next2)
                               s))
                           (make-string i #\space))))))
            (string-set! str 0 next)
            (unicode (string->number str 16))))
        (##read-error-hex re))))

  (define (read-escape)
    (let ((next (##read-next-char re)))
      (cond ((char-octal? next)
             (read-escape-octal next))
            ((char=? next #\x)
             (read-escape-hexadecimal))
            ((char=? next close)
             close)
            (else
             (let ((x (assq next
                            (##readtable-escaped-char-table
                             (##readenv-readtable re)))))
               (if x
                 (cdr x)
                 (##read-error-escaped-char re next)))))))

  (define max-chunk-length 512)

  (define (read-chunk)
    (let loop ((i 0))
      (if (< i max-chunk-length)
        (let ((c (##read-next-char re)))
          (cond ((char=? c close)
                 (make-string i #\space))
                ((char=? c #\\)
                 (let* ((c (read-escape))
                        (s (loop (+ i 1))))
                   (string-set! s i c)
                   s))
                (else
                 (let ((s (loop (+ i 1))))
                   (string-set! s i c)
                   s))))
        (make-string i #\space))))

  (let ((chunk1 (read-chunk)))
    (if (< (string-length chunk1) max-chunk-length)
      chunk1
      (let loop ((chunks (list chunk1)))
        (let* ((new-chunk (read-chunk))
               (new-chunks (cons new-chunk chunks)))
          (if (< (string-length new-chunk) max-chunk-length)
            (##append-strings (reverse new-chunks))
            (loop new-chunks)))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Procedures to handle comments.

(define (##skip-extended-comment re open1 open2 close1 close2)
  (let loop ((level 0) (c (##read-next-char re)))
    (cond ((char=? c open1)
           (let ((c (##read-next-char re)))
             (if (char=? c open2)
               (loop (+ level 1) (##read-next-char re))
               (loop level c))))
          ((char=? c close1)
           (let ((c (##read-next-char re)))
             (if (char=? c close2)
               (if (< 0 level)
                 (loop (- level 1) (##read-next-char re))
                 #f) ; comment has ended
               (loop level c))))
          (else
           (loop level (##read-next-char re))))))

(define (##skip-single-line-comment re)
  (let loop ()
    (let ((next (##peek-next-char-or-eof re)))
      (if (char? next)
        (begin
          (##read-next-char-or-eof re) ; skip "next"
          (if (not (char=? next #\newline))
            (loop)))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Procedure to read datums starting with '#'.

(define (##read-sharp re c)
  (let ((start-pos (##readenv-current-filepos re)))
    (##read-next-char-or-eof re) ; skip #\#
    (let ((next (##peek-next-char re)))
      (cond ((char=? next #\()
             (##read-next-char-or-eof re) ; skip #\(
             (##readenv-filepos-set! re start-pos) ; set pos to start of datum
             (let ((vect (##build-vector re 'vector start-pos #\))))
               (##readenv-wrap re vect)))
            ((char=? next #\\)
             (##read-next-char-or-eof re) ; skip #\\
             (##readenv-filepos-set! re start-pos) ; set pos to start of datum
             (let ((c (##read-next-char re)))
               (if (##readtable-char-delimiter?
                    (##readenv-readtable re)
                    (##peek-next-char-or-eof re))
                 (##readenv-wrap re c)
                 (let ((name (##build-delimited-string re c 1)))
                   (let ((x (##read-assoc-string-ci=?
                             name
                             (##readtable-named-char-table
                              (##readenv-readtable re)))))
                     (if x
                       (##readenv-wrap re (cdr x))
                       (let ((n (string->number name 10)))
                         (if (and n
                                  (integer? n)
                                  (exact? n))
                           (if (in-unicode-range? n)
                             (##readenv-wrap re (unicode->character n))
                             (##read-error-char-range re))
                           (##read-error-char-name re name)))))))))
            ((char=? next #\|)
             (let ((old-pos (##readenv-filepos re)))
               (##readenv-filepos-set! re start-pos) ; in case error in comment
               (##read-next-char-or-eof re) ; skip #\|
               (##skip-extended-comment re #\# #\| #\| #\#)
               (##readenv-filepos-set! re old-pos) ; restore pos
               (##read-datum-or-none-or-dot re))) ; read what follows comment
            ((char=? next #\!)
             (##read-next-char-or-eof re) ; skip #\!
             (##readenv-filepos-set! re start-pos) ; set pos to start of datum
             (let ((name (##build-delimited-string re #\space 0)))
               (let ((x (##read-assoc-string-ci=?
                         name
                         (##readtable-sharp-bang-table
                          (##readenv-readtable re)))))
                 (if x
                   (##readenv-wrap re (cdr x))
                   (##read-error-sharp-bang-name re name)))))
            ((char=? next #\#)
             (##read-next-char-or-eof re) ; skip #\#
             (##readenv-filepos-set! re start-pos) ; set pos to start of datum
             (let ((sym (##build-delimited-symbol re #\# 2)))
               (##readenv-wrap re sym)))
            (else
             (##readenv-filepos-set! re start-pos) ; set pos to start of datum
             (let* ((s
                     (##build-delimited-string re c 1))
                    (obj
                     (or (string->number s 10)
                         (let ()

                           (define (build-vect re kind)
                             (let ((c (##read-next-char re)))
                               (if (char=? c #\()
                                 (##build-vector re kind start-pos #\))
                                 (##read-error-vector re))))

                           (cond ((string-ci=? s "#f")
                                  (false-obj))
                                 ((string-ci=? s "#t")
                                  #t)
                                 ((string-ci=? s "#u8")
                                  (build-vect re 'u8vector))
                                 ((string-ci=? s "#u16")
                                  (build-vect re 'u16vector))
                                 ((string-ci=? s "#u32")
                                  (build-vect re 'u32vector))
                                 ((string-ci=? s "#f32")
                                  (build-vect re 'f32vector))
                                 ((string-ci=? s "#f64")
                                  (build-vect re 'f64vector))
                                 (else
                                  (##read-error-sharp-token re s)))))))
               (##readenv-wrap re obj)))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (##read-whitespace re c)
  (##read-next-char-or-eof re) ; skip whitespace character
  (##read-datum-or-none-or-dot re)) ; read what follows whitespace

(define (##read-single-line-comment re c)
  (##skip-single-line-comment re) ; skip comment
  (##read-datum-or-none-or-dot re)) ; read what follows comment

(define (##read-escaped-string re c)
  (let ((start-pos (##readenv-current-filepos re)))
    (##read-next-char-or-eof re) ; skip #\"
    (##readenv-filepos-set! re start-pos) ; set pos to start of datum
    (let ((str (##build-escaped-string-up-to re c)))
      (##readenv-wrap re str))))

(define (##read-escaped-symbol re c)
  (let ((start-pos (##readenv-current-filepos re)))
    (##read-next-char-or-eof re) ; skip #\|
    (##readenv-filepos-set! re start-pos) ; set pos to start of datum
    (let ((sym (string->symbol (##build-escaped-string-up-to re c))))
      (##readenv-wrap re sym))))

(define (##read-quotation re c)
  (let ((start-pos (##readenv-current-filepos re)))
    (##read-next-char-or-eof re) ; skip #\'
    (##readenv-filepos-set! re start-pos) ; set pos to start of datum
    (let ((obj (##read-datum re)))
      (##readenv-filepos-set! re start-pos) ; set pos to start of datum
      (##readenv-wrap
       re
       (list (##readenv-wrap re 'quote) obj)))))

(define (##read-quasiquotation re c)
  (let ((start-pos (##readenv-current-filepos re)))
    (##read-next-char-or-eof re) ; skip #\`
    (##readenv-filepos-set! re start-pos) ; set pos to start of datum
    (let ((obj (##read-datum re)))
      (##readenv-filepos-set! re start-pos) ; set pos to start of datum
      (##readenv-wrap
       re
       (list (##readenv-wrap re 'quasiquote) obj)))))

(define (##read-unquotation re c)
  (let ((start-pos (##readenv-current-filepos re)))
    (##read-next-char-or-eof re) ; skip #\,
    (##readenv-filepos-set! re start-pos) ; set pos to start of datum
    (let ((next (##peek-next-char re)))
      (if (char=? next #\@)
        (begin
          (##read-next-char-or-eof re) ; skip #\@
          (let ((obj (##read-datum re)))
            (##readenv-filepos-set! re start-pos) ; set pos to start of datum
            (##readenv-wrap
             re
             (list (##readenv-wrap re 'unquote-splicing) obj))))
        (let ((obj (##read-datum re)))
          (##readenv-filepos-set! re start-pos) ; set pos to start of datum
          (##readenv-wrap
           re
           (list (##readenv-wrap re 'unquote) obj)))))))

(define (##read-list re c)
  (let ((start-pos (##readenv-current-filepos re)))
    (##read-next-char-or-eof re) ; skip #\( or #\[ or #\{
    (##readenv-filepos-set! re start-pos) ; set pos to start of datum
    (let ((close
           (cond ((char=? c #\[) #\])
                 ((char=? c #\{) #\})
                 (else           #\)))))
      (let ((lst (##build-list re #t start-pos close)))
        (##readenv-wrap re lst)))))

(define (##read-none re c)
  (##none-marker))

(define (##read-illegal re c)
  (let ((start-pos (##readenv-current-filepos re)))
    (##read-next-char-or-eof re) ; skip illegal character
    (##readenv-filepos-set! re start-pos) ; set pos to illegal char
    (##read-error-illegal-char re c)))

(define (##read-dot re c)
  (let ((start-pos (##readenv-current-filepos re)))
    (##read-next-char-or-eof re) ; skip #\.
    (let ((next (##peek-next-char-or-eof re)))
      (if (or (not (char? next))
              (##readtable-char-delimiter? (##readenv-readtable re) next))
        (##dot-marker)
        (begin
          (##readenv-filepos-set! re start-pos) ; set pos to start of datum
          (let ((obj (##build-delimited-number/keyword/symbol re c)))
            (##readenv-wrap re obj)))))))

(define (##read-number/keyword/symbol re c)
  (let ((start-pos (##readenv-current-filepos re)))
    (##read-next-char-or-eof re) ; skip "c"
    (##readenv-filepos-set! re start-pos) ; set pos to start of datum
    (let ((obj (##build-delimited-number/keyword/symbol re c)))
      (##readenv-wrap re obj))))

(define (##read-assoc-string-ci=? x lst)
  (let loop ((lst lst))
    (if (pair? lst)
      (let ((couple (car lst)))
        (let ((y (car couple)))
          (if (string-ci=? x y)
            couple
            (loop (cdr lst)))))
      #f)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Setup the standard readtable.

(define (##make-standard-readtable)
  (let ((rt
         (##make-readtable
          #f ; preserve case in symbols, character names, etc
          #t ; keywords ending with ":" are allowed
          ##standard-escaped-char-table
          ##standard-named-char-table
          ##standard-sharp-bang-table
          (##make-chartable #f) ; all chars are non-delimiters
          (##make-chartable ##read-number/keyword/symbol))))

    (if (##comply-to-standard-scheme?) ; force compliance to standard Scheme?
      (begin
        (##readtable-case-conversion?-set! rt #t)
        (##readtable-keywords-allowed?-set! rt #f)))

    ; setup control characters

    (let loop ((i 31))
      (if (not (< i 0))
        (begin
          (##readtable-char-class-set!
           rt
           (unicode->character i)
           #t
           ##read-illegal)
          (loop (- i 1)))))

    ; setup whitespace characters

    (##readtable-char-class-set! rt #\space    #t ##read-whitespace)
    (##readtable-char-class-set! rt #\linefeed #t ##read-whitespace)
    (##readtable-char-class-set! rt #\return   #t ##read-whitespace)
    (##readtable-char-class-set! rt #\tab      #t ##read-whitespace)
    (##readtable-char-class-set! rt #\page     #t ##read-whitespace)

    ; setup handlers for non-whitespace delimiters

    (##readtable-char-class-set! rt #\; #t ##read-single-line-comment)

    (##readtable-char-class-set! rt #\" #t ##read-escaped-string)
    (##readtable-char-class-set! rt #\| #t ##read-escaped-symbol)

    (##readtable-char-class-set! rt #\' #t ##read-quotation)
    (##readtable-char-class-set! rt #\` #t ##read-quasiquotation)
    (##readtable-char-class-set! rt #\, #t ##read-unquotation)

    (##readtable-char-class-set! rt #\( #t ##read-list)
    (##readtable-char-class-set! rt #\) #t ##read-none)

    (##readtable-char-class-set! rt #\[ #t ##read-list)
    (##readtable-char-class-set! rt #\] #t ##read-none)

    (##readtable-char-class-set! rt #\{ #t ##read-illegal)
    (##readtable-char-class-set! rt #\} #t ##read-illegal)

    ; setup handlers for "#" and "." (these are NOT delimiters)

    (##readtable-char-class-set! rt #\# #f ##read-sharp)
    (##readtable-char-class-set! rt #\. #f ##read-dot)

    rt))

(if (not ##main-readtable)
  (set! ##main-readtable
    (##make-standard-readtable)))

;==============================================================================
