/* $Id: protocol.c,v 1.3 1999/07/31 12:47:11 stano Exp $

   Protocol between selector and indicator, using properties

   (C) 1999 Stanislav Meduna <stano@eunet.sk>
*/

#include <xkbselx.h>

#include <config.h>

#include <X11/Xatom.h>
#include <string.h>
#include <libintl.h>
#include <sys/time.h>
#include <unistd.h>

#define _(s) dgettext(PACKAGE, s)

static Atom cur_sel_atom = (Atom) 0;
static Atom indicator_atom = (Atom) 0;
static Atom magic_atom = (Atom) 0;

static int bad_window = 0;
static int (*old_handler)(Display *, XErrorEvent *) = 0;
static int (*old_after)() = 0;

/* === Forward declarations === */

static int fill_atoms(Display *disp);

static int start_synch_and_catch(Display *disp);
static int stop_synch_and_catch(Display *disp);
static int bad_window_catcher(Display *disp, XErrorEvent *ev);

static int get_win_for_notify(Display *disp, Window *win);


/* === Public interface === */

/* Publish the current selection and inform the indicator, if any */
int publish_current_sel(Display *disp, const sel_info_t *sel)
{
	int r;
	char prop[MAX_SHORTCUT_LEN+MAX_DATA_PATH_LEN+2];

	Window root = RootWindow(disp, DefaultScreen(disp));
	Window ind_w;

	/* Ensure we know the atoms */
	r = fill_atoms(disp);

	if (r < 0)
		return r;

	/* Construct property value */
	sprintf(prop, "%s %s", sel->map_name, sel->shortcut);

	/* Set it */
	XChangeProperty(
		disp, root,
		cur_sel_atom, XA_STRING,
		8,
		PropModeReplace,
		prop,
		strlen(prop));

	r = get_win_for_notify(disp, &ind_w);
	if (r < 0)
		return r;

	if (ind_w != None)
	{
		struct timeval tv;

		XButtonEvent ev;

		gettimeofday(&tv, NULL);

		/* Construct a keypress/release event and send it */
		ev.type      = ButtonPress;
		ev.display   = disp;
		ev.window    = ind_w;
		ev.root      = root;
		ev.subwindow = None;
		ev.time      = tv.tv_sec * 1000 + tv.tv_usec / 1000;
		ev.x         = 0;
		ev.y         = 0;
		ev.x_root    = 0;
		ev.y_root    = 0;
		ev.state     = 0;
		ev.button    = Button1;
		ev.same_screen = True;

		XSendEvent(disp, ind_w, True, ButtonPressMask , (XEvent *) &ev);
		ev.type = ButtonRelease;
		ev.state     = Button1Mask;
		ev.time += 100;
		XSendEvent(disp, ind_w, True, ButtonReleaseMask , (XEvent *) &ev);
	}


	return 0;
}

/* Get the current selection */
int get_current_sel(Display *disp, sel_info_t *sel, int *found)
{
	int r;
	char prop[MAX_SHORTCUT_LEN+MAX_DATA_PATH_LEN+2];

	Window root = RootWindow(disp, DefaultScreen(disp));

	Atom act_type = (Atom) 0;
	int  act_fmt = 0;
	unsigned long nitems = 0;
	unsigned long after = 0;
	unsigned char *sel_prop = 0;

	/* Ensure we know the atoms */
	r = fill_atoms(disp);

	if (r < 0)
		return r;

	*sel->shortcut = 0;
	*sel->map_name = 0;
	*found = 0;

	/* Query the window for the current mapping property */
	r = XGetWindowProperty(
		disp, root,
		cur_sel_atom,
		0L, sizeof(prop),
		False,
		XA_STRING,
		&act_type,
		&act_fmt,
		&nitems, &after,
		&sel_prop);

	if (r == Success)
	{
		if (nitems > 0)
		{
			unsigned char *p;

			/* Find first space */
			p = strchr(sel_prop, ' ');

			if (p == NULL || p - sel_prop + 1 > sizeof(sel->map_name))
			{
				XFree(sel_prop);
				return -1;
			}

			/* Map name */
			*p = 0;
			strcpy(sel->map_name, sel_prop);

			p++;

			if (strlen(p) + 1 > sizeof(sel->shortcut))
			{
				XFree(sel_prop);
				return -1;
			}

			strcpy(sel->shortcut, p);
			*found = 1;

			XFree(sel_prop);
		}
	}

	return 0;
}

/* Tell the system that we want/don't want to be informed about changes */
int notify_me(Display *disp, Window win, int yes)
{
	int r;

	Window root = RootWindow(disp, DefaultScreen(disp));
	Window ind_w;

	/* Ensure we know the atoms */
	r = fill_atoms(disp);

	if (r < 0)
		return r;

	/* Check if there is already an indicator */
	r = get_win_for_notify(disp, &ind_w);
	if (r < 0)
		return r;

	if (yes)
	{
		long prop = win;
		unsigned char *magic = PACKAGE " " VERSION;

		if (ind_w != None)
		{
			fprintf(stderr, _("There already is an indicator - won't indicate anything\n"));
			return -1;
		}


		/* Set it */
		XChangeProperty(
			disp, root,
			indicator_atom, XA_WINDOW,
			32,
			PropModeReplace,
			(unsigned char *) &prop,
			1);

		/* And the magic too */
		XChangeProperty(
			disp, win,
			magic_atom, XA_STRING,
			8,
			PropModeReplace,
			magic,
			strlen(magic));

	}
	else
	{
		if (ind_w != win)
		{
			fprintf(stderr, _("The window is not an indicator one - won't delete the property\n"));
			return -1;
		}

		XDeleteProperty(disp, root, indicator_atom);
	}


	return 0;
}

/* === Private functions === */

static int fill_atoms(Display *disp)
{
	static char *names[] = { "XKBSEL_CURRENT_KEYBOARD", "XKBSEL_INDICATOR", "XKBSEL_MAGIC" };
	Atom atoms[sizeof(names)/sizeof(names[0])];
	Status r;

	/* Save a server roundtrip if already known */
	if (cur_sel_atom != (Atom) 0 && indicator_atom != (Atom) 0)
		return 0;

	r = XInternAtoms(disp,
		names, sizeof(names)/sizeof(names[0]),
		False, atoms);

	if (! r)
		return -1;

	cur_sel_atom = atoms[0];
	indicator_atom = atoms[1];
	magic_atom = atoms[2];

	return 0;
}

static int bad_window_catcher(Display *disp, XErrorEvent *ev)
{
	if (ev->error_code == BadWindow)
	{
		bad_window = 1;
		return 0;
	}

	return (*old_handler)(disp, ev);
}

static int start_synch_and_catch(Display *disp)
{
	old_after = XSynchronize(disp, True);
	bad_window = 0;
	old_handler = XSetErrorHandler(bad_window_catcher);

	return 0;
}

static int stop_synch_and_catch(Display *disp)
{
	XSetErrorHandler(old_handler);
	old_handler = 0;
	XSetAfterFunction(disp, old_after);

	return 0;
}

static int get_win_for_notify(Display *disp, Window *win)
{
	int r;
	char prop[MAX_SHORTCUT_LEN+MAX_DATA_PATH_LEN+2];

	Window root = RootWindow(disp, DefaultScreen(disp));

	Atom act_type = (Atom) 0;
	int  act_fmt = 0;
	unsigned long nitems = 0;
	unsigned long after = 0;
	unsigned char *ind_prop = 0;

	/* Ensure we know the atoms */
	r = fill_atoms(disp);

	if (r < 0)
		return r;

	*win = None;

	/* Query the window for the indicator property */
	r = XGetWindowProperty(
		disp, root,
		indicator_atom,
		0L, sizeof(prop),
		False,
		XA_WINDOW,
		&act_type,
		&act_fmt,
		&nitems, &after,
		&ind_prop);

	if (r == Success)
	{
		if (nitems > 0)
		{
			Window ind_w = *(long *) ind_prop;

			XFree(ind_prop);

			start_synch_and_catch(disp);

			/* Check the magic */
			r = XGetWindowProperty(
				disp, ind_w,
				magic_atom,
				0L, sizeof(prop),
				False,
				XA_STRING,
				&act_type,
				&act_fmt,
				&nitems, &after,
				&ind_prop);

			if (! bad_window && r == Success && nitems > 0)
				*win = ind_w;

			stop_synch_and_catch(disp);
		}
	}

	return 0;
}

