/*
 * Copyright (c) 1994  Software Research Associates, Inc.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Software Research Associates not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  Software Research
 * Associates makes no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied
 * warranty.
 *
 * Author:  Makoto Ishisone, Software Research Associates, Inc., Japan
 */
#include <stdio.h>
#include <stdlib.h>

#include "im.h"
#include "imreq.h"

static int checkRequest
( IMConnection *conn, int *majorp, int *minorp, int *arglenp ) ;
static void ServerAuthPhase
( IMConnection *conn ) ;
static void CommunicationPhase
( IMConnection *conn ) ;
static int initialDispatcher
( IMConnection *conn ) ;
static int mainDispatcher
( IMConnection *conn ) ;

/*- checkRequest: read and examine request header -*/
static int checkRequest
( IMConnection *conn, int *majorp, int *minorp, int *arglenp )
{
  IMBuffer *ibp = IM_INBUF( conn ) ;
  int len ;

  /* check if there is enough data for the request header */
  if( IMBUFLEN( ibp ) < 4 )
    return 0 ;

#ifdef XIM_BC
  if( conn->has_length_bug ){
    len = ( int )IMGetC16( conn, 2 ) ;
  } else {
    len = ( int )IMGetC16( conn, 2 ) * 4 ;
  }
#else /* XIM_BC */
  len = (int)IMGetC16(conn, 2) * 4;
#endif /* XIM_BC */
  
  if( IMBUFLEN( ibp ) >= len + 4 ){
    /*
     * All the request data is in the buffer.
     * Retrieve the request header, and discard it.
     */
    *majorp = IMGetC8( conn, 0 ) ;
    *minorp = IMGetC8( conn, 1 ) ;
    *arglenp = len ;
    IMBufDiscard( ibp, 4 ) ;
    return 1 ;
  } else {
    return 0 ;
  }
}

#ifdef XIM_BC
/*- countConnectRequestLen: count length of XIM_CONNECT request -*/
int countConnectRequestLen( IMConnection *conn )
{
  IMBuffer *ibp = IM_INBUF( conn ) ;
  int data_length = IMBUFLEN( ibp ) ;
  int num_proto_names ;
  int offset ;
  int i ;

  /*
   * Encoding of XIM_CONNECT request is:
   *
   *	4		request header
   *	1  CARD8	byte order
   *	1		unused
   *  2  CARD16	major protocol version
   *  2  CARD16	minor protocol version
   *	2  CARD16	number of auth protocol names
   *  N  LISTofSTRING	auth protocol names
   *
   * where STRING is:
   *  2  CARD16	length (N)
   *  N  LISTofChar	string
   *  p		pad, p = Pad(2+N)
   */
  
  /* skip request header */
  data_length -= 4 ;
  offset = 4 ;

  if( data_length < 8 )
    return 0 ;	/* incomplete data */

  /* get the number of auth protocol names */
  num_proto_names = ( int )IMGetC16( conn, offset + 6 ) ;

  /* skip fixed part */
  data_length -= 8 ;
  offset += 8 ;

  /* count variable (i.e. list of string) part */
  for( i = 0 ; i < num_proto_names ; i++ ){
    int str_len, pad_len ;

    if( data_length < 2 )
      return 0 ;	/* incomplete data */
    str_len = ( int )IMGetC16( conn, offset ) ;
    pad_len = ( 4 - ( ( 2 + str_len ) % 4 ) ) % 4 ;
    data_length -= 2 + str_len + pad_len ;
    offset += 2 + str_len + pad_len ;
    if( data_length < 0 )
      return 0 ;	/* incomplete data */
  }
  return offset - 4;		/* 4 for the header */
}
#endif /* XIM_BC */

/*
 * Phase shifters
 */

/*- ServerAuthPhase: move to server authentication phase -*/
static void ServerAuthPhase( IMConnection *conn )
{
  /* not yet implemented */
  IMSendSimpleRequest( conn, XIM_AUTH_NG, 0 ) ;
  IMSchedule( conn, SCHED_CLOSE ) ;
}

/*- CommunicationPhase: move to comunication (i.e. main) phase -*/
static void CommunicationPhase( IMConnection *conn )
{
  IMPutHeader( conn, XIM_CONNECT_REPLY, 0, 4 ) ;
  IMPutC16( conn, XIM_MAJOR_PROTOCOL_VERSION ) ;
  IMPutC16( conn, XIM_MINOR_PROTOCOL_VERSION ) ;
  IMSchedule( conn, SCHED_WRITE ) ;
  
  conn->dispatcher = mainDispatcher ;
}

/*
 * Request dispatchers
 */
/*
 * IM Server Ѵ饤Ȥǽ³׵˺
 * ٥ȥǥѥåؿǤ롣ƤѴ¦ѿν
 * Ǥġ餯ϡǽΥ٥ȤʤȤɤΤ褦׵(
 Ȥ٥ȥץ㡼᥽åɤȤ)Τʬʤ
 * 
 */
static int initialDispatcher( IMConnection *conn )
{
  IMBuffer *ibp = IM_INBUF( conn ) ;
  int major, minor, arglen ;
  int major_protocol, minor_protocol ;
  int num_auth ;
#ifdef XIM_BC
  int counted_len ;
  int field_len ;
#endif /* XIM_BC */

  /*
   * ѴФ˺ǽƤå XIM_CONNECT Ǥʤ
   * ʤʤλˤϡХȥ¾ξήƤ롣ǽ顢
   * ѴФϥХȥΤʤΤǡORDER_UNKNOWN ηˤʤ
   * Ƥ롣
   */
  if( conn->byte_order == ORDER_UNKNOWN ){
    char *p = IMBUFDATA( ibp ) ;
    
    if( IMBUFLEN( ibp ) < 5 )
      return 0 ;

    switch( p[4] ){
    case 'B' :	/* ӥåǥ(Ĥޤȥ礫) */
      conn->byte_order = ORDER_BIG ;
      break ;
    case 'l' :	/* ȥ륨ǥ(ƥ) */
      conn->byte_order = ORDER_LITTLE ;
      break ;
    default :	/* ȼ¤ϡ֤ȤäΤäȵƤ뤱ɡ */
		/* ߥɥ륨ǥʤïȤʤ*/
      goto send_ng ;
    }
  }
  
#ifdef XIM_BC
  /*
   * Xlib implementation of early X11R6 has a bug in the
   * length field of request header.  The field should
   * represent the length of request data as the number
   * of 4byte units, but the buggy Xlib puts number of
   * bytes instead.
   */
  
  /* length of data by examining contents */
  counted_len = countConnectRequestLen( conn ) ;/* in bytes */
  if( counted_len == 0 )
    return 0 ;		/* incomplete */
  /* length of data by reading length field */
  field_len = IMGetC16( conn, 2 ) ;		/* num of 4byte element */
  
  /*
   * If the request packet comforms to the specification,
   *    field_len * 4 == counted_len
   * but in case of buggy Xlib implementation,
   *    field_len == counted_len
   */
  if( counted_len == field_len * 4 ){
    conn->has_length_bug = 0 ;
  } else if( counted_len == field_len ){
    /* buggy Xlib implementation */
    conn->has_length_bug = 1 ;
  } else {
    /* totally broken */
    goto send_ng ;
  }
#endif
  
  /*
   * See if the entire data of the request is ready.
   */
  if( !checkRequest( conn, &major, &minor, &arglen ) )
    return 0 ;
  
  /*
   * ǽ XIM_CONNECTʳΥåäѲ롣
   */
  if( major != XIM_CONNECT || minor != 0 || arglen < 8 ){
    goto send_ng ;
  }
  
  major_protocol = ( int )IMGetC16( conn, 2 ) ;
  minor_protocol = ( int )IMGetC16( conn, 4 ) ;
  conn->major_protocol_version = major_protocol ;
  conn->minor_protocol_version = minor_protocol ;
  
  num_auth = ( int )IMGetC16( conn, 6 ) ;
  
  if( major_protocol > XIM_MAJOR_PROTOCOL_VERSION ||
     ( major_protocol == XIM_MAJOR_PROTOCOL_VERSION &&
       minor_protocol > XIM_MINOR_PROTOCOL_VERSION ) ){
    goto send_ng ;
  }
  
  if( num_auth > 0 ){
    ServerAuthPhase( conn ) ;
  } else {
    CommunicationPhase( conn ) ;
  }
  
  IMBufDiscard( ibp, arglen ) ;
  
  return 1 ;
  
  /* ǽ³˼Ԥν*/
 send_ng :
  IMSendSimpleRequest( conn, XIM_AUTH_NG, 0 ) ;
  IMSchedule( conn, SCHED_SHUTDOWN ) ;
  return 0 ;
}

/*- mainDispatcher: main dispatcher -*/
static int mainDispatcher( IMConnection *conn )
{
  IMBuffer *ibp = IM_INBUF( conn ) ;
  int major, minor, arglen ;
  IMRequest *req ;

  /*
   * When X transport is used, NUL bytes might appear between
   * requests.  So discard them first.
   */
  IMBufDiscardNUL( ibp ) ;

  /*
   * Check if the entire request data is on the input buffer.
   */
  if( !checkRequest( conn, &major, &minor, &arglen ) )
    return 0 ;

  /*
   * Check major opcode.
   */
  req = IMMainDispatchTable[ major ] ;
  if (req == NULL) {
    IMSendBadProtocol( conn, "Invalid major opcode" ) ;
    goto ret ;
  }

  /*
   * Then, check minor opcode.
   */
  while( req != NULL ){
    if( req->minor == minor )
      break ;
    req = req->next ;
  }
  if( req == NULL ){
    IMSendBadProtocol( conn, "Invalid minor opcode" ) ;
    goto ret ;
  }

#ifdef DEBUG
  printf( "Major: %d, Minor: %d, Length:%d\n", major, minor, arglen ) ;
#endif
  /*
   * Opcode is valid. Call the request processing routine.
   */
  ( *req->proc )( conn, major, minor, arglen ) ;

 ret:
  /*
   * Discard the argument portion.
   */
  IMBufDiscard( ibp, arglen ) ;

  return 1 ;
}

/*
 * Public functions
 */

void IMSetInitialDispatcher( IMConnection *conn )
{
  conn->dispatcher = initialDispatcher ;
  return ;
}

void IMDispatch( IMConnection *conn, int cond )
{
  conn->dispatching = True ;
  switch( cond ){
  case TRANSPORT_OK :
    /*
     * Call dispatcher while data is ready.
     */
    while( ( *conn->dispatcher )( conn ) )
      /* empty body */;
    
    /*
     * Do compaction to the input buffer.
     */
    IMBufCompact(IM_INBUF(conn));
    break;
    
  case TRANSPORT_ERROR :
    IMSchedule( conn, SCHED_SHUTDOWN ) ;
    break;
    
  case TRANSPORT_EOF :
#if defined(DEBUG)
    fprintf( stderr, "Transport End of File? \n" ) ;
#endif
    IMSchedule( conn, SCHED_CLOSE ) ;
    break;
  }
  
  /*
   * If there's something to be done (i.e. scheduled),
   * do it.
   */
  if( !IMQueueEmpty( conn->protocol_widget ) ){
    IMProcessQueue( conn->protocol_widget ) ;
  }
  conn->dispatching = False ;
  return ;
}

void IMSchedule( IMConnection *conn, int type )
{
  if( conn->schedule & type )
    return ;	/* already scheduled */

  if( conn->schedule == 0 ){
    IMPushQueue( conn ) ;
  }
  conn->schedule |= type ;
  return ;
}

void IMProcessQueue( Widget w )
{
  IMConnection *conn ;
  IMConnection *tmp = NULL ;

  while( ( conn = IMPopQueue( w ) ) != NULL ){
    int schedule ;

    schedule       = conn->schedule ;
    conn->schedule = 0 ;

#if defined(DEBUG)
    fprintf( stderr, "Current schedule is %d\n", schedule ) ;
#endif

    if( schedule & SCHED_SHUTDOWN ){
#if defined(DEBUG)
      fprintf( stderr, "Warning: Schedule is shutdown.\n" ) ;
#endif
      IMCloseConnection( conn ) ;
    } else {
      if( schedule & SCHED_WRITE ){
	int ret = IMFlush( conn ) ;
#if defined(DEBUG)
	fprintf( stderr, "IMFlush? ???\n" ) ;
#endif
	switch( ret ){
	case TRANSPORT_ERROR :
	  IMCloseConnection( conn ) ;
	  continue ;
	case TRANSPORT_PARTIAL :
	  /* to be queued again */
	  conn->schedule = schedule ;
	  conn->queue_next = tmp ;
	  tmp = conn ;
	  continue ;
	}
      } else {
#if defined(DEBUG)
	fprintf( stderr, "Warning: Schedule is unknown....\n" ) ;
#endif
      }
      if( schedule & SCHED_CLOSE ){
#if defined(DEBUG)
      fprintf( stderr, "Warning: Schedule is \"close\".\n" ) ;
#endif
	IMCloseConnection( conn ) ;
      }
    }
  }

  /* reschedule */
  while( tmp != NULL ){
    IMConnection *next = tmp->queue_next ;
    
    IMPushQueue( tmp ) ;
    tmp = next ;
  }
  return ;
}
