/* sfm -- Simple File Manager
   Copyright (C) 1998 Pixel (Pascal Rigaux)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */
#include "sfm_server.h"
#include "sfm_view.h"
#include "filetype.h"
#include <stdlib.h>
#include <strings.h>
#include <unistd.h>
#include <signal.h>
#include <stddef.h>
#include <sys/un.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <gtk/gtkmain.h>


#define max_nbchilds 10
static int childs[max_nbchilds];
GList *selection = NULL;

static int compare(char *a, char *b)
{
  if (!a) return -1;
  if (!b) return 1;
  return strcmp(a + sizeof(TypePaste), b + sizeof(TypePaste));
}

static void free_it(char *p) { free(p); }
static void free_selection()
{
  g_list_foreach(selection, (GFunc) free_it, NULL);
  g_list_free(selection);
}

static void quit(int status)
{
  int i;

  /*fprintf(stderr, "SERVER: bye\n"); */
  saveProgForFileType();
  free_selection();
  freeProgForFileType();
  for (i = 0; i < max_nbchilds; i++) 
    if (childs[i]) {
      if (kill(childs[i], SIGKILL)) die("kill (server)");
    }
  unlink(socket_name);
  myexit(status);
}

static void received_sigint(int signal)
{
  quit(0);
}

static void received_sigusr2(int signal)
{
  int i;
  /*printf("SERVER: Sending back\n"); */
  /* send back the signal to all the childs */
  for (i = 0; i < max_nbchilds; i++) 
    if (childs[i]) {
      if (kill(childs[i], SIGUSR2)) die("kill (server)");
    }
}


static void server_received_sigchild(int signal)
{
  int pid, status, j, trouve = -1;

  if ((pid = waitpid(-1, &status, WNOHANG)) > 0 && !isfileTypeSigChld(pid)) {
    /*printf("SERVER: Child ended %d\n", pid);*/
    for (j = 0; j < max_nbchilds; j++) {
      if (childs[j] == pid) childs[j] = 0; else if (childs[j]) trouve = j;
    }

    if (status) quit(status);

    if (trouve >= 0) { if (kill(childs[trouve], SIGHUP)) die("kill (server)"); }
    else { quit(0); }
  }
}


void create_child(int number, int *argc, char ***argv) 
{
  if ((childs[number] = fork()) == 0) {

    SfmView *view;

    mystart();
    view = sfm_view_new(number, argc, argv);

    gtk_main();
    sfm_view_free(view);
    myexit(0);
  }
}

static void send_it(char *p, int fd)
{
  /*printf("Sending %s (%d)\n", p + 4, * (TypePaste*)p);*/
  send_msg(fd, p + sizeof(TypePaste));
  send_msg(fd, "\n");
  send_int(fd, * (TypePaste *) p);
}

extern void start_server(int argc, char **argv)
{
  struct sockaddr_un addr;
  int fd, child_fd, i, j, size;
  char *p, *q;

  /* is there already a server ? */
  if (connect_server(&fd)) {
    /* then ask a new child */
    send_msg(fd, "new\n");
    send_int(fd, -1);
    for (i = 0; i < argc; i++) { send_msg(fd, argv[i]); send_msg(fd, "\n"); }
    send_msg(fd, "\n");
    close(fd);
    myexit(0);
  }
  addr.sun_family=AF_UNIX;
  strcpy(addr.sun_path, socket_name);
  size = offsetof(struct sockaddr_un, sun_path) + strlen (socket_name) + 1;

  unlink(socket_name);
  if (bind(fd, (struct sockaddr *) &addr, size) == -1) die("bind");
  if (listen(fd, SOMAXCONN) == -1) die("listen");
  
  for (i = 0; i < max_nbchilds; i++) childs[i] = 0;
  create_child(1, &argc, &argv);
  selection = NULL;

  install_signal_handler(SIGINT, received_sigint);
  install_signal_handler(SIGUSR2, received_sigusr2);
  install_signal_handler(SIGCHLD, server_received_sigchild);

  while ((child_fd = accept(fd, NULL, NULL)) > 0) {
    init_line_by_line_read();
    p = line_by_line_read(child_fd);
    
    if (streq(p, "set")) {
      TypePaste type_paste;

      type_paste = atoi(line_by_line_read(child_fd));
      if (atoi(line_by_line_read(child_fd)) == TYPE_OVERRIDE) { free_selection(); selection = NULL; }

      while (strlen(p = line_by_line_read(child_fd)) > 0) {
	q = char_malloc(strlen(p) + sizeof(TypePaste) + 1);
	* (TypePaste *) q = type_paste;
	strcpy(q + sizeof(TypePaste), p);
	selection = g_list_insert_sorted(selection, q, (GCompareFunc) compare);
      }

    } else if (streq(p, "get")) {
      GList *l, *l2;
      if (selection) {
	send_it(selection->data, child_fd);
	for (l = selection; (l2 = g_list_next(l)); l = l2) {
	  if (compare(l2->data, l->data)) send_it(l2->data, child_fd);
	}
      }
      send_msg(child_fd, "\n");

    } else if (streq(p, "get_prog")) {

      initProgForFileType();
      p = strdup(line_by_line_read(child_fd));
      if ((q = file2prog(p)) == NULL) q = "";
      send_msg(child_fd, q); send_msg(child_fd, "\n");
      free(p);

    } else if (streq(p, "get_progs")) {
      char **r;
      int nb;

      initProgForFileType();
      p = strdup(line_by_line_read(child_fd));
      r = file2progs(p, &nb);
      send_int(child_fd, nb);
      for (i = 0; i < nb; i++) { send_msg(child_fd, r[i]); send_msg(child_fd, "\n"); }
      free(r);
      free(p);

    } else if (streq(p, "add_prog")) {

      initProgForFileType();
      p = strdup(line_by_line_read(child_fd));
      q = strdup(line_by_line_read(child_fd));
      addFile2prog(p, q);
      free(p);
      free(q);

    } else if (streq(p, "new")) {

      i = atoi(line_by_line_read(child_fd));
      
      if (i == -1) { /* the server must find a new number for a child */
	int argc_;
	char **argv_ = pchar_malloc(tmpSize);
	for (i = 0; i < tmpSize && strlen(p = line_by_line_read(child_fd)) > 0; i++) argv_[i] = strdup(p);
	argc_ = i;

	for (j = 0; j < max_nbchilds; j++)
	  if (!childs[j]) { 
	    create_child(j, &argc_, &argv_); 
	    break; 
	  }
	pchar_free(argv_, i);
      }
      else if (childs[i]) { if (kill(childs[i], SIGHUP)) die("kill (server)"); }
      else create_child(i, &argc, &argv);

    }
    end_line_by_line_read();
    close(child_fd);
  }
}


