;; edges.jl -- Identify all window edges
;; $Id: edges.jl,v 1.8 1999/11/25 23:34:37 john Exp $

;; Copyright (C) 1999 John Harper <john@dcs.warwick.ac.uk>

;; This file is part of sawmill.

;; sawmill is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; sawmill is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with sawmill; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

(provide 'edges)

;; returns (X-EDGES . Y-EDGES), X-EDGES is a list of (X Y1 Y2 OPEN-P), and
;; Y-EDGES is a list of (Y X1 X2 OPEN-P). OPEN-P is t if the edge is
;; left or top edge of a window. For the root window, the meaning of
;; OPEN-P is reversed

;; the returned lists may contain duplicates, and are unsorted

;; keywords:
;;	:with-ignored-windows t
;;	:windows-to-ignore LIST
;;	:windows LIST
;;	:include-root t

(defun get-visible-window-edges (&rest args)
  (let
      ((with-ignored-windows (car (cdr (memq ':with-ignored-windows args))))
       (windows-to-ignore (car (cdr (memq ':windows-to-ignore args))))
       (windows (cdr (memq ':windows args)))
       x-edges y-edges)
    (mapc (lambda (w)
	    (when (and (window-visible-p w)
		       (or with-ignored-windows
			   (not (window-get w 'ignored)))
		       (not (memq w windows-to-ignore))
		       (or (null windows) (memq w (car windows))))
	      (let
		  ((dims (window-frame-dimensions w))
		   (coords (window-position w)))
		(setq x-edges (cons (list (car coords) (cdr coords)
					  (+ (cdr coords) (cdr dims)) t)
				    (cons (list (+ (car coords) (car dims))
						(cdr coords)
						(+ (cdr coords) (cdr dims))
						nil)
					  x-edges)))
		(setq y-edges (cons (list (cdr coords) (car coords)
					  (+ (car coords) (car dims)) t)
				    (cons (list (+ (cdr coords) (cdr dims))
						(car coords)
						(+ (car coords) (car dims))
						nil)
					  y-edges))))))
	  (managed-windows))
    (when (car (cdr (memq ':include-root args)))
      (setq x-edges (cons (list 0 0 (screen-height) nil)
			  (cons (list (screen-width) 0 (screen-height) t)
				x-edges)))
      (setq y-edges (cons (list 0 0 (screen-width) nil)
			  (cons (list (screen-height) 0 (screen-width) t)
				y-edges))))
    (cons x-edges y-edges)))

(defun grid-from-edges (x-edges y-edges)
  (cons (uniquify-list (mapcar car x-edges))
	(uniquify-list (mapcar car y-edges))))

(defmacro edges-abs (x)
  `(max x (- x)))

;; returns (EDGE-1 EDGE-2) where they're within EPSILON of each other
(defun edges-within-epsilon (list-1 list-2 epsilon)
  (catch 'out
    (mapc (lambda (edge-1)
	    (mapc (lambda (edge-2)
		    (when (and (< (nth 1 edge-1) (nth 2 edge-2))
			       (> (nth 2 edge-1) (nth 1 edge-2))
			       (<= (- (max (car edge-1) (car edge-2))
				      (min (car edge-1) (car edge-2)))
				   epsilon))
		      (throw 'out (cons edge-1 edge-2))))
		  list-2))
	  list-1)
    nil))

;; returns the new (X . Y) of WINDOW
(defun snap-window-position-to-edges (window coords &optional epsilon edges)
  (let*
      ((dims (window-frame-dimensions window))
       (w-x-edges (list (list (car coords) (cdr coords)
			      (+ (cdr coords) (cdr dims)))
			(list (+ (car coords) (car dims))
			      (cdr coords) (+ (cdr coords) (cdr dims)))))
       (w-y-edges (list (list (cdr coords) (car coords)
			      (+ (car coords) (car dims)))
			(list (+ (cdr coords) (cdr dims))
			      (car coords) (+ (car coords) (car dims)))))
       tem)
    (setq coords (cons (car coords) (cdr coords)))
    (unless edges
      (setq edges (get-visible-window-edges)))
    (unless epsilon
      (setq epsilon 8))
    (when (setq tem (edges-within-epsilon w-x-edges (car edges) epsilon))
      (setq tem (+ (car coords) (- (car (cdr tem)) (car (car tem)))))
      (when (and (> tem (- (car dims))) (< tem (screen-width)))
	(rplaca coords tem)))
    (when (setq tem (edges-within-epsilon w-y-edges (cdr edges) epsilon))
      (setq tem (+ (cdr coords) (- (car (cdr tem)) (car (car tem)))))
      (when (and (> tem (- (cdr dims))) (< tem (screen-height)))
	(rplacd coords tem)))
    coords))
