/* network.c - general socket and network code callbacks  */

#include <stdio.h>
#include <stdlib.h>

#include <string.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <sys/stat.h>
#include <fcntl.h>

#include <unistd.h>

#include <gnome.h>

#include "napster.h"
#include "network.h"
#include "gnapster.h"
#include "ui.h"

extern CONN_INFO conn_info;
extern USER_INFO user_info;
extern DOWNLOAD_TAB download_tab;
extern CONSOLE_TAB console_tab;
extern MOTD_TAB motd_tab;

void network_handle_accept_header(gpointer data, int source, GdkInputCondition condition) {
   char buf[4096];
   int n;
   NAP_DOWNLOAD *download;
   
   download = (NAP_DOWNLOAD *)data;
   
   n = recv(source, buf, 4, 0);
   if (n <= 0) {
      printf("\nClosing connection\n");
      gdk_input_remove(download->sock_input);
      close(download->sock);
      free(download);
   } else if (n > 0) {
      struct stat st;
      char *temp_str;
      char *user, *file, *size;
      
      bzero(buf, sizeof(buf));
      n = recv(download->sock, buf, 4096, 0);
      buf[n] = '\0';
      user = strdup(buf);
      file = strchr(user, '\"');
      if (!file) {
	 printf("Bad header...\n");
	 gdk_input_remove(download->sock_input);
	 close(download->sock);
	 free(user);
	 free(download);
	 return;
      }
      file[0] = '\0';
      file++;
      size = strchr(file, '\"');
      if (!size) {
	 printf("Bad header\n");
	 gdk_input_remove(download->sock_input);
	 close(download->sock);
	 free(user);
	 free(download);
	 return;
      }
      size[0] = '\0';
      size++;
      download->user = strdup(user);
      download->file = strdup(file);
      download->total = atoi(size);
      
      download->trunc_file = strrchr(file, '\\');
      if (!download->trunc_file) {
	 printf("Bad header\n");
	 gdk_input_remove(download->sock_input);
	 close(download->sock);
	 free(user);
	 free(download);
	 return;
      }
      download->trunc_file = strdup(download->trunc_file);
      download->trunc_file++;
      
      free(user);
      
      download->fdesc = open_file(download, download->trunc_file);
      if (!download->fdesc) {
	 gnome_error_dialog("Uh oh.  The path you've chosen does not exist or you do not have write permissions\nPlease open up gnapster's preferences and pick a directory you know you have access to");
	 gdk_input_remove(download->sock_input);
	 close(download->sock);
	 free(download);
	 return;
      }
      
      //fstat(download->fdesc->_fileno, &st);
      stat(download->fpath, &st);
      temp_str = g_strdup_printf("%i", (int)st.st_size);
      send(download->sock, temp_str, strlen(temp_str), 0);
      free(temp_str);
      gdk_input_remove(download->sock_input);
      download->size = 0;
      download->connection = 0;
      temp_str = g_strdup_printf("Receiving %s from %s...\n", 
				 download->trunc_file, download->user);
      gnapster_text_insert(console_tab.text, console_tab.vbox, temp_str);
      free(temp_str);
      gnapster_clist_append(download_tab.clist, download_tab.vbox,
			    download, download->trunc_file,
			    napster_conn_as_str(download->connection),
			    download->user, "0.00M", "0%");
      gnapster_update_status(conn_info.status, 0);
      
      download->sock_input = gdk_input_add(download->sock, GDK_INPUT_READ,
					   network_handle_download, download);
   }
}

void network_handle_connection(gpointer data, int source, GdkInputCondition condition) {
   NAP_DOWNLOAD *download;
   struct sockaddr_in from;
   int len;
   
   download = malloc(sizeof(NAP_DOWNLOAD));
   download->sock = 
     accept(conn_info.bind_sock, (struct sockaddr *)&from, &len);
   if (download->sock < 0) {
      printf("accept error\n");
      return;
   }
   printf("Got connection from %s\n", inet_ntoa(from.sin_addr));
   send(download->sock, "\n", 1, 0);
   
   download->sock_input = gdk_input_add(download->sock, GDK_INPUT_READ,
					network_handle_accept_header, download);
}

void network_handle_download(gpointer data, int source, GdkInputCondition condition) {
   char buf[2048], *temp_str;
   int n;
   NAP_DOWNLOAD *download;
   
   download = (NAP_DOWNLOAD *)data;
   
   bzero(buf, sizeof(buf));
   n = recv(source, buf, 2048, 0);
   if (n == 0) {
      gdk_input_remove(download->sock_input);
      fclose(download->fdesc);
      close(download->sock);
      if (download->size == download->total) {
	 temp_str = g_strdup_printf("Download complete for %s\n",
				    download->trunc_file);
	 gnapster_text_insert(console_tab.text, console_tab.vbox, temp_str);
	 free(temp_str);
      } else {
	 temp_str = g_strdup_printf("Download cancelled by %s for %s (%i bytes lost)\n",
				    download->user, download->trunc_file, download->total - download->size);
	 gnapster_text_insert(console_tab.text, console_tab.vbox, temp_str);
	 free(temp_str);
      }
      gtk_clist_remove(GTK_CLIST(download_tab.clist),
		       gtk_clist_find_row_from_data(GTK_CLIST(download_tab.clist), download));
      free(download);
      return;
   } else if (n > 0) {
      fwrite(buf, n, sizeof(char), download->fdesc);
      download->size += n;
      temp_str = g_strdup_printf("%.02fM", ((float)download->size / 1024) / 1024);
      gtk_clist_set_text(GTK_CLIST(download_tab.clist),
			 gtk_clist_find_row_from_data(GTK_CLIST(download_tab.clist),
						      download),
			 3, temp_str);
      free(temp_str);
      temp_str = g_strdup_printf("%i%%", (int)((float)download->size / (float)download->total * 100));
      gtk_clist_set_text(GTK_CLIST(download_tab.clist),
			 gtk_clist_find_row_from_data(GTK_CLIST(download_tab.clist),
						      download),
			 4, temp_str);
      free(temp_str);
   }
}

void network_conn_user_header_cb(gpointer data, int source, GdkInputCondition condition) {
   NAP_DOWNLOAD *download;
   char c;
   int n;
   
   download = (NAP_DOWNLOAD *)data;

   n = recv(source, &c, 1, 0);
   if (n <= 0) {
      char *temp_str;
      download->header[download->header_val] = '\0';
      temp_str = g_strdup_printf("Connection closed by remote user %s\n",
				 download->user);
      gnapster_text_insert(console_tab.text, console_tab.vbox, temp_str);
      free(temp_str);
      close(source);
      gdk_input_remove(download->sock_input);
      free(download);
      return;
   }
   
   if (c == '\377') {
      char buf[20 + 1];
      int i = 1, x = 0;
      char *temp_str;
      download->header[download->header_val] = '\0';
      while(download->header[i] && i < 20) {
	 buf[x++] = download->header[i];
	 i++;
      }
      download->size = 0;
      download->total = atoi(buf);
      
      download->fdesc = open_file(download, download->trunc_file);      
/*      temp_str = g_strdup_printf("%s/%s", user_info.download_dir, download->trunc_file);
      if (file_exists(temp_str)) {
	 printf("File exists, attempting to write to %s~\n",
		download->trunc_file);
	 free(temp_str);
	 temp_str = g_strdup_printf("%s/%s~", 
				    user_info.download_dir, download->trunc_file);
	 if (file_exists(temp_str)) {
	    printf("Backup file exists, aborting download.\n");
	    gdk_input_remove(download->sock_input);
	    close(download->sock);
	    free(temp_str);
	    free(download);
	    return;
	 }
      }
      download->fdesc = fopen(temp_str, "w");
      free(temp_str);*/
      if (!download->fdesc) {
	 gnome_error_dialog("Uh oh.  The path you've chosen does not exist or you do not have write permissions\nPlease open up napster's preferences and pick a directory you know you have access to");
	 gdk_input_remove(download->sock_input);
	 close(source);
	 free(download);
	 return;
      }
      temp_str = g_strdup_printf("Receiving file %s from %s...\n",
				 download->trunc_file, download->user);
      gnapster_text_insert(console_tab.text, console_tab.vbox, temp_str);
      free(temp_str);
      fwrite("\377", 1, sizeof(char), download->fdesc);
      download->size = 1;
      gnapster_clist_append(download_tab.clist, download_tab.vbox,
			    download, download->trunc_file,
			    napster_conn_as_str(download->connection),
			    download->user, "0.00M", "0%");
      gdk_input_remove(download->sock_input);
      download->sock_input = gdk_input_add(source, GDK_INPUT_READ,
					   network_handle_download, download);
   } else {
      if (download->header_val < 10) download->header[download->header_val++] = c;
   }
}

void network_conn_user_cb(gpointer data, int source, GdkInputCondition condition) {
   NAP_DOWNLOAD *download;
   
   download = (NAP_DOWNLOAD *)data;

   gdk_input_remove(download->sock_input);

   if (connect(source, (struct sockaddr *)&download->saddr, 
	       sizeof(download->saddr)) < 0) {
      gnapster_text_insert(console_tab.text, console_tab.vbox,
			   "* Error connecting to user...\n");
      close(source);
      free(download);
      return;
   }

   fcntl(source, F_SETFL, 0);   
   
   napster_send_get(source, user_info.user, download->file, 
		    user_info.connection);

   download->sock_input = gdk_input_add(source, GDK_INPUT_READ,
					network_conn_user_header_cb, download);
}

void network_conn_cb(gpointer data, int source, GdkInputCondition condition) {
   int n, port_int;
   char buf[1024];
   char *ip, *port;
   
//   perror("connect");
   
   /* FIX THIS LATER */
   if (errno != 0 && errno != EINPROGRESS) {
      perror("connect");
      gdk_input_remove(conn_info.sock_input);
      close(source);
      gnapster_update_status("Unknown fatal error connecting to server (try again?)", 0);
      conn_info.connected = 0;
      conn_info.connecting = 0;
      return;
   }

   fcntl(source, F_SETFL, 0);
   
   n = recv(source, buf, 1024, 0);
   if (n <= 0) {
      perror("recv");
      gdk_input_remove(conn_info.sock_input);
      close(source);
      gnapster_update_status("Unknown fatal error connecting to server (try again?)", 0);
      conn_info.connected = 0;
      conn_info.connecting = 0;
      return;
   }
   ip = strdup(buf);
   port = strchr(ip, ':');
   port[0] = '\0';
   port++;
   
   port_int = atoi(port);
   
   if (conn_info.ip) free(conn_info.ip);
   conn_info.ip = strdup(ip);
   conn_info.port = port_int;
   
   printf("ip = %s, port = %i\n", conn_info.ip, conn_info.port);
   
   close(source);
   gdk_input_remove(conn_info.sock_input);

   if (!strcmp(conn_info.ip, "127.0.0.1")) {
      gnapster_update_status("Napster query server reported all servers were busy", 0);
      conn_info.connected = 0;
      conn_info.connecting = 0;
      return;
   }
   
   gnapster_update_status("Best host found, connecting...", -1);
   napster_connect(conn_info.ip, conn_info.port);
}

void network_read_cb(gpointer data, int source, GdkInputCondition condition) {
   int n;
   char buf[355], buf_holder[355];
   int len, cmd, old_n, i;
   
   /* Ok if you get stuff from the socket, they've responded one way or another */
   n = 0; len = 4;
   bzero(buf, sizeof(buf));
   while(n < len) {
      bzero(buf_holder, sizeof(buf_holder));
      old_n = n;
      n += recv(source, buf_holder, (len - n), 0);
      if (n == old_n) {
	 n = 0;
	 break;
      }
      for(i=0; i<(n-old_n); i++) {
	 buf[old_n + i] = buf_holder[i];
      }
   }
   if (n <= 0) {
      perror("recv");
      gnapster_update_status("Disconnected", 0);
      gtk_text_set_point(GTK_TEXT(motd_tab.text), 0);
      gtk_text_forward_delete(GTK_TEXT(motd_tab.text),
			      gtk_text_get_length(GTK_TEXT(motd_tab.text)));
      conn_info.connected = 0;
      conn_info.connecting = 0;
      gdk_input_remove(conn_info.sock_input);
      close(conn_info.sock);
      return;
   }

   len = (buf[0] < 0) ? 256 + buf[0] : buf[0];
   cmd = buf[2];
   
   n = 0;
   bzero(buf, sizeof(buf));
   while(n < len) {
      bzero(buf_holder, sizeof(buf_holder));
      n += recv(source, buf_holder, (len - n), 0);
      strcat(buf, buf_holder);
   }

   if (cmd == '\0') {
      printf("\\0 -> %s\n", buf);
   }
   conn_info.connecting = 0;
   conn_info.connected = 1;
   
   napster_handle_data(cmd, buf);
}

void network_conn_real_cb(gpointer data, int source, GdkInputCondition condition) {
   char temp_str[80];
   
   fcntl(source, F_SETFL, 0);

   sprintf(temp_str, "%s %s %i \"%s\" %i", user_info.user, user_info.pass,
	   6699, "nap v0.8", user_info.connection);
   printf("logging in (new_user = %i, %s)\n", user_info.new_user, temp_str);
   if (user_info.new_user)
     napster_send(source, NAPSTER_NEW_USER, user_info.user);
   napster_send(source, NAPSTER_LOGIN, temp_str);
   
   sprintf(temp_str, "Connected (%s:%i)...awaiting login reply...", 
	   conn_info.ip, conn_info.port);
   gnapster_update_status(temp_str, -1);
   printf("network_conn_real_cb\n");
   gdk_input_remove(conn_info.sock_input);
   conn_info.sock_input = gdk_input_add(source, GDK_INPUT_READ,
					network_read_cb, NULL);
}
