/*
** 1999-09-11 -	A fun Rename command, with support for both simplistic replacement in filenames
**		as well as powerful regular expression replacement thingie. The former is handy
**		for changing the extension in all selected names from ".jpg" to ".jpeg", for
**		example. The latter RE mode can be used to do rather complex mappings and trans-
**		forms on the filenames. For example, consider having a set of 100 image files,
**		named "frame0.gif", "frame1.gif", ..., "frame99.gif". You want to rename these
**		to include a movie index (11), and also (of course) change the extension to ".png".
**		No problem. Just enter "^frame([0-9]+)\.gif$" in the 'From' text entry field
**		on the Reg Exp page, and "frame-11-$1.png" in the 'To' entry, and bam! The point
**		here is that you can use up to 9 parenthesized (?) subexpressions, and then access
**		the text that matched each with $n (n is a digit, 1 to 9) in the 'To' string.
**		If you've programmed any Perl, you might be familiar with the concept.
*/

#include "gentoo.h"

#include <ctype.h>
#include <regex.h>

#include "dialog.h"
#include "dirpane.h"
#include "errors.h"
#include "overwrite.h"
#include "strutil.h"

#include "cmd_renamere.h"

/* ----------------------------------------------------------------------------------------- */

typedef struct {
	Dialog		*dlg;

	GtkWidget	*nbook;

	/* Simple replace mode. */
	GtkWidget	*r_vbox;
	GtkWidget	*r_from;
	GtkWidget	*r_to;
	GtkWidget	*r_cnocase;
	GtkWidget	*r_cglobal;

	/* Full RE matching mode. */
	GtkWidget	*re_vbox;
	GtkWidget	*re_from;
	GtkWidget	*re_to;
	GtkWidget	*re_cnocase;

	MainInfo	*min;
	DirPane		*src;
	gint		page;
	GSList		*selection;
} RenREInfo;

/* ----------------------------------------------------------------------------------------- */

static gint rename_simple(RenREInfo *ri)
{
	const gchar	*from, *to, *ptr, *base;
	gint		err = 0;
	gsize		flen;
	GString		*nn;
	GSList		*iter;
	gboolean	nocase, global;

	if((from = gtk_entry_get_text(GTK_ENTRY(ri->r_from))) == NULL)
		return -1;
	flen = strlen(from);

	to = gtk_entry_get_text(GTK_ENTRY(ri->r_to));

	nocase = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ri->r_cnocase));
	global = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ri->r_cglobal));

	ovw_overwrite_begin(ri->min, "\"%s\" already exists - proeceed with rename?", 0U);
	nn = g_string_new("");
	for(iter = ri->selection; !err && iter != NULL; iter = g_slist_next(iter))
	{
		g_string_truncate(nn, 0);

		base = DP_SEL_NAME(iter);
		while((ptr = nocase ? str_strcasechr(base, *from) : strchr(base, *from)) != NULL)
		{
			for(; base < ptr; base++)
				g_string_append_c(nn, *base);
			if((nocase ? str_strcasencmp(ptr, from, flen) : strncmp(ptr, from, flen)) == 0)
			{
				g_string_append(nn, to);
				base += flen;
				if(!global)
					break;
			}
			else
				g_string_append_c(nn, *base++);
		}
		g_string_append(nn, base);
		if((strcmp(DP_SEL_NAME(iter), nn->str) != 0) && (strchr(nn->str, G_DIR_SEPARATOR) == NULL))
		{
			OvwRes	ores;

			ores = ovw_overwrite_file(ri->min, nn->str, NULL);
			if(ores == OVW_SKIP)
				continue;
			else if(ores == OVW_CANCEL)
				break;
			if(rename(DP_SEL_NAME(iter), nn->str))
			{
				err = errno;
				dp_rescan(ri->src);
				err_set(ri->min, err, "RenameRE", DP_SEL_NAME(iter));
			}
			else
				dp_unselect(ri->src, DP_SEL_INDEX(ri->src, iter));
		}
	}
	g_string_free(nn, TRUE);
	ovw_overwrite_end(ri->min);

	return err;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-09-11 -	Go through <def>, copying characters to <str>. If the sequence $n is found,
**		where n is a digit 1..9, replace that with the n:th piece of <src>, as described
**		by <match>. To get a single dollar in the output, write $$ in <def>.
**		Returns TRUE if the entire <def> was successfully interpolated, FALSE on error.
*/
static gboolean interpolate(GString *str, const gchar *def, const gchar *src, const regmatch_t *match)
{
	const gchar	*ptr;

	g_string_truncate(str, 0);
	for(ptr = def; *ptr;)
	{
		if(*ptr == '$')
		{
			ptr++;
			if(isdigit((unsigned char) *ptr))
			{
				gint	slot, i;

				slot = *ptr++ - '0';
				if((i = match[slot].rm_so) != -1)
				{
					for(; i < match[slot].rm_eo; i++)
						g_string_append_c(str, src[i]);
				}
				else
					return FALSE;
			}
			else if(*ptr == '$')
				g_string_append_c(str, '$'), ptr++;
			else
				return FALSE;
		}
		else
			g_string_append_c(str, *ptr++);
	}
	return TRUE;
}

static gint rename_regexp(RenREInfo *ri)
{
	const gchar	*fromdef, *to;
	gint		reerr, reflags, err = 0;
	regex_t		fromre;

	if((fromdef = gtk_entry_get_text(GTK_ENTRY(ri->re_from))) == NULL)
		return -1;
	if((to = gtk_entry_get_text(GTK_ENTRY(ri->re_to))) == NULL)
		return -1;

	reflags = REG_EXTENDED | gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ri->re_cnocase));
	if((reerr = regcomp(&fromre, fromdef, reflags)) == 0)
	{
		GSList		*iter;
		GString		*nn;
		regmatch_t	match[10];

		ovw_overwrite_begin(ri->min, "\"%s\" already exists - proeceed with rename?", 0U);
		nn = g_string_new("");
		for(iter = ri->selection; iter != NULL; iter = g_slist_next(iter))
		{
			if(regexec(&fromre, DP_SEL_NAME(iter), sizeof match / sizeof match[0], match, 0) == 0)
			{
				if(interpolate(nn, to, DP_SEL_NAME(iter), match))
				{
					if((strcmp(DP_SEL_NAME(iter), nn->str) != 0) && (strchr(nn->str, G_DIR_SEPARATOR) == NULL))
					{
						OvwRes	ores;

						ores = ovw_overwrite_file(ri->min, nn->str, NULL);
						if(ores == OVW_SKIP)
							continue;
						else if(ores == OVW_CANCEL)
							break;
						if(rename(DP_SEL_NAME(iter), nn->str))
						{
							err = errno;
							dp_rescan(ri->src);
							err_set(ri->min, err, "RenameRE", DP_SEL_NAME(iter));
						}
						else
							dp_unselect(ri->src, DP_SEL_INDEX(ri->src, iter));
					}
				}
			}
		}
		g_string_free(nn, TRUE);
		ovw_overwrite_end(ri->min);
	}
	else
	{
		gchar	buf[128] = "Regular expression error:\n";

		regerror(reerr, &fromre, buf + strlen(buf), sizeof buf - strlen(buf) - 1);
		dlg_dialog_async_new_error(buf);
	}

	return err;
}

/* ----------------------------------------------------------------------------------------- */

static void evt_nbook_switchpage(GtkWidget *wid, GtkNotebookPage *page, gint page_num, gpointer user)
{
	RenREInfo	*ri = user;

	ri->page = page_num;
	gtk_widget_grab_focus(ri->page ? ri->re_from : ri->r_from);
}

int cmd_renamere(MainInfo *min, DirPane *src, DirPane *dst, CmdArg *ca)
{
	GtkWidget		*label, *table;
	static RenREInfo	ri = { NULL };

	ri.min	= min;
	ri.src	= src;

	if((ri.selection = dp_get_selection(ri.src)) == NULL)
		return 1;

	if(ri.dlg == NULL)
	{
		ri.nbook = gtk_notebook_new();
		gtk_signal_connect(GTK_OBJECT(ri.nbook), "switch-page", GTK_SIGNAL_FUNC(evt_nbook_switchpage), &ri);

		ri.r_vbox = gtk_vbox_new(FALSE, 0);
		label = gtk_label_new("Look for substring in all filenames, and replace\nit with another string.");
		gtk_box_pack_start(GTK_BOX(ri.r_vbox), label, FALSE, FALSE, 0);
		gtk_widget_show(label);
		table = gtk_table_new(2, 2, FALSE);
		label = gtk_label_new("Replace");
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,  0,0,0,0);
		gtk_widget_show(label);
		ri.r_from = gtk_entry_new();
		gtk_table_attach(GTK_TABLE(table), ri.r_from, 1, 2, 0, 1, GTK_EXPAND|GTK_FILL,0,0,0);
		gtk_widget_show(ri.r_from);
		label = gtk_label_new("With");
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,  0,0,0,0);
		gtk_widget_show(label);
		ri.r_to = gtk_entry_new();
		gtk_table_attach(GTK_TABLE(table), ri.r_to, 1, 2, 1, 2, GTK_EXPAND|GTK_FILL,0,0,0);
		gtk_widget_show(ri.r_to);
		gtk_box_pack_start(GTK_BOX(ri.r_vbox), table, FALSE, FALSE, 0);
		gtk_widget_show(table);
		table = gtk_table_new(1, 2, FALSE);
		ri.r_cnocase = gtk_check_button_new_with_label("Ignore Case?");
		gtk_table_attach(GTK_TABLE(table), ri.r_cnocase, 0, 1, 0, 1, GTK_EXPAND|GTK_FILL,0,0,0);
		gtk_widget_show(ri.r_cnocase);
		ri.r_cglobal = gtk_check_button_new_with_label("Replace All?");
		gtk_table_attach(GTK_TABLE(table), ri.r_cglobal, 1, 2, 0, 1, GTK_EXPAND|GTK_FILL,0,0,0);
		gtk_widget_show(ri.r_cglobal);
		gtk_box_pack_start(GTK_BOX(ri.r_vbox), table, FALSE, FALSE, 0);
		gtk_widget_show(table);

		gtk_notebook_append_page(GTK_NOTEBOOK(ri.nbook), ri.r_vbox, gtk_label_new("Simple"));
		gtk_widget_show(ri.r_vbox);


		ri.re_vbox = gtk_vbox_new(FALSE, 0);
		label = gtk_label_new(	"Execute the 'From' RE on each filename, storing\n"
					"parenthesised subexpression matches. Then replace\n"
					"any occurance of $n in 'To', where n is the index\n"
					"(counting from 1) of a subexpression, with the text\n"
					"that matched, and use the result as a new filename.");

		gtk_box_pack_start(GTK_BOX(ri.re_vbox), label, FALSE, FALSE, 0);
		gtk_widget_show(label);
		table = gtk_table_new(2, 2, FALSE);
		label = gtk_label_new("From");
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,  0,0,0,0);
		gtk_widget_show(label);
		ri.re_from = gtk_entry_new();
		gtk_table_attach(GTK_TABLE(table), ri.re_from, 1, 2, 0, 1,  GTK_EXPAND|GTK_FILL,0,0,0);
		gtk_widget_show(ri.re_from);
		label = gtk_label_new("To");
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,  0,0,0,0);
		gtk_widget_show(label);
		ri.re_to = gtk_entry_new();
		gtk_table_attach(GTK_TABLE(table), ri.re_to, 1, 2, 1, 2,  GTK_EXPAND|GTK_FILL,0,0,0);
		gtk_widget_show(ri.re_to);
		gtk_box_pack_start(GTK_BOX(ri.re_vbox), table, FALSE, FALSE, 0);
		gtk_widget_show(table);
		ri.re_cnocase = gtk_check_button_new_with_label("Ignore Case?");
		gtk_box_pack_start(GTK_BOX(ri.re_vbox), ri.re_cnocase, FALSE, FALSE, 0);
		gtk_widget_show(ri.re_cnocase);

		gtk_notebook_append_page(GTK_NOTEBOOK(ri.nbook), ri.re_vbox, gtk_label_new("Reg Exp"));
		gtk_widget_show(ri.re_vbox);

		ri.dlg = dlg_dialog_sync_new(ri.nbook, "RenameRE", "OK|Cancel");

		ri.page = 0;
	}
	gtk_notebook_set_page(GTK_NOTEBOOK(ri.nbook), ri.page);
	gtk_widget_grab_focus(ri.page ? ri.re_from : ri.r_from);

	if(dlg_dialog_sync_wait(ri.dlg) == DLG_POSITIVE)
	{
		gint	err;

		err_clear(ri.min);
		if(ri.page == 0)
			err = rename_simple(&ri);
		else
			err = rename_regexp(&ri);
		if(!err)
			dp_rescan(ri.src);
		else
			err_show(ri.min);
	}

	dp_free_selection(ri.selection);
	ri.selection = NULL;

	return 1;
}
