/* 
 *      XSitecopy, for managing remote web sites with a GNOME interface.
 *      Copyright (C) 1999, Lee Mallabone <lee0@callnetuk.com
 *                                                                        
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *     
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *     
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *  
 */
#include <gnome.h>

/* Gnome message/dialog functions. This contains mostly wrappers for the
 * sitecopy API from the gnome API. 
 */

#include <stdio.h>
#include <stdlib.h>

#include "common.h"
#include "sites.h"
#include "rcfile.h"
#include "gcommon.h"
#include "misc.h"
#include "config.h"
#include "frontend.h"

#define RCNAME "/." PACKAGE "rc"
#define COPYNAME "/." PACKAGE "/"

GtkWidget *connection_label = NULL;

extern GtkWidget *app_bar;
extern struct site_t *selected_site;
extern GtkWidget *error_log_list, *error_log_window;

/* For the 'main' upload window */
extern GtkWidget *upload_window, *main_progressbar, *job_progressbar, *error_button;
extern GtkWidget *status_label, *op_label, *file_label, *dir_label;
extern GtkWidget *begin_button;

extern struct site_t *all_sites;

/* This value is what makes the entire "job" progress bar possible. */
float uploaded_bytes = 0.0;

extern GtkWidget *sitecopy;

/*************************************************/

int fe_gtk_question (char *question, GnomeReplyCallback yes_action) {
   GtkWidget *question_window;
   question_window = gnome_app_question (GNOME_APP (sitecopy), 
													  (const gchar *) question,
													  yes_action, NULL);
   gtk_widget_show (question_window);
   return 1;
}

void fe_transfer_progress (size_t progress, size_t total) {
   
   float percentage, div1, div2;
   size_t upload_size;

   /* Calculate the amount to display the current file's progress */
   div1 = (float) progress;
   div2 = (float) total;

   percentage = div1 / div2;
   gtk_progress_bar_update (GTK_PROGRESS_BAR (main_progressbar), percentage);
   
   /* Calculate the amount to display for total job progress */
   upload_size = selected_site->totalnew + selected_site->totalchanged;
   gtk_progress_bar_update (GTK_PROGRESS_BAR (job_progressbar),
			    (uploaded_bytes + (float) progress) / (float) upload_size);

   /* Keep the interface sensitive */
   while (gtk_events_pending() > 0)
     gtk_main_iteration();
}

void fe_status (char *message) {
   gnome_app_message (GNOME_APP (sitecopy), (const gchar *) message);
}
   
/* 
 * TODO: I need to add support for yes-no confirmation stuff.
*/

void fe_connection (fe_conn_status status) {
  switch(status) {
  case (0):
	 gtk_label_set (GTK_LABEL (connection_label), "Looking up hostname...");
	 break;
  case (1):
	 gtk_label_set (GTK_LABEL (connection_label), "Attempting to connect...");
	 break;
  case (2):
	 gtk_label_set (GTK_LABEL (connection_label), "Connected.");
	 break;
  }
  while (gtk_events_pending() > 0)
	 gtk_main_iteration();
}


bool fe_can_update (const struct site_file_t *file) {
   g_print ("Confirmation given to upload file %s.\n", file->rel_local);
   return true;
}

void fe_updating (const struct site_file_t *file) {
   char *file_status;

   file_status = malloc(strlen(selected_site->server) + 25);
   sprintf (file_status, "Commiting updates to %s...", selected_site->server);
   gtk_label_set (GTK_LABEL (status_label), file_status);
   if (file->dir) {
      if (file->diff == file_new) {
	 gtk_label_set (GTK_LABEL (op_label), "Creating directory...");
	 gtk_label_set (GTK_LABEL (file_label), file->full_local);
	 /*	    gtk_label_set (GTK_LABEL (dir_label), file->directory); */
	 gtk_label_set (GTK_LABEL (dir_label), "");
      } else {
	 /* can we move dirs yet? */
	 gtk_label_set (GTK_LABEL (op_label), "Deleting directory...");
	 gtk_label_set (GTK_LABEL (dir_label), "");
      }
   } else {
      switch (file->diff) {
       case file_changed:
	 gtk_label_set (GTK_LABEL (op_label), "Uploading...");
	 gtk_label_set (GTK_LABEL (file_label), file->full_local);
	 gtk_label_set (GTK_LABEL (dir_label), file->full_remote);
	 break;
       case file_new:
	 gtk_label_set (GTK_LABEL (op_label), "Uploading...");
	 gtk_label_set (GTK_LABEL (file_label), file->full_local);
	 gtk_label_set (GTK_LABEL (dir_label), file->full_remote);
	 break;
       case file_deleted:
	 gtk_label_set (GTK_LABEL (op_label), "Deleting...");
	 gtk_label_set (GTK_LABEL (file_label), file->full_local);
	 gtk_label_set (GTK_LABEL (dir_label), file->full_remote);
	 break;
       case file_moved:
	 gtk_label_set (GTK_LABEL (op_label), "Moving...");
	 gtk_label_set (GTK_LABEL (file_label), file->old->full_local);
	 gtk_label_set (GTK_LABEL (dir_label), file->directory);
	 break;
       case file_unchanged:
	 gtk_label_set (GTK_LABEL (op_label), "ARG! The file hasn't changed, we shouldn't be doing anything!");
      }
   }
   while (gtk_events_pending() > 0)
     gtk_main_iteration();
}

/* Once a file has been updated, any errors with it are recorded in the 
 * error list, and the progress bar is reset. 
 */
void fe_updated (const struct site_file_t *file, const bool success,
		 const char *error) {
   gchar *error_item[2];
   
   if (!success) {
      error_item[0] = malloc (strlen(file->full_local) + 1);
      error_item[1] = malloc (strlen(error) + 1);
      strcpy (error_item[0], file->full_local);
      strcpy (error_item[1], error);
      gtk_clist_append (GTK_CLIST (error_log_list), error_item);
      DEBUG (DEBUG_GNOME, "Error \"%s\" with file: %s\n", 
				 error, file->full_local);
   }
   gtk_progress_bar_update (GTK_PROGRESS_BAR (main_progressbar), 0.0);
   uploaded_bytes += (float) file->localsize;
   while (gtk_events_pending() > 0)
     gtk_main_iteration();
}

int verifysite_gnome(struct site_t *a_site) {
  int ret = rc_verifysite( a_site );
  if (!ret)
	 return 0;

  switch( ret ) {
  case 0:
	 return;
  case SITE_NOSERVER:
	 gnome_error_dialog("Server not specified.");
	 break;
  case SITE_NOUSERNAME:
	 gnome_error_dialog("Username not specified.");
	 break;
  case SITE_NOPASSWORD:
	 gnome_error_dialog("Password not specified.");
	 break;
  case SITE_NOREMOTEDIR:
	 gnome_error_dialog("Remote directory not specified.");
	 break;
  case SITE_NOLOCALDIR:
	 gnome_error_dialog("Local directory not specified.");
	 break;
  case SITE_ACCESSLOCALDIR:
	 gnome_error_dialog("Could not read local directory.");
	 break;
  case SITE_INVALIDPORT:
	 gnome_error_dialog("Invalid port.");
	 break;
  case SITE_NOMAINTAIN:
	 gnome_error_dialog("The chosen protocol cannot maintain symbolic links");
	 break;
  case SITE_NOREMOTEREL:
	 gnome_error_dialog("The chosen protocol cannot use a relative remote directory.");
	 break;
  case SITE_NOPERMS:
	 gnome_error_dialog("The protocol you are attempting to use does\nnot currently support maintaining permissions.");
	 break;
  case SITE_NOLOCALREL:
	 gnome_error_dialog("Could not use a 'relative' local directory");
  default:
	 gnome_error_dialog("There was an undetermined problem verifying the correctness of your site definition. Please report this to the maintainer.");
	 break; 
  }
  return ret;
}

/* Coming soon... */
void fe_synching (const struct site_file_t *file ) {
}
void fe_synched (const struct site_file_t *file, const bool success,
		 const char *error) {
}
