/*  Screem:  toolbar.c,
 *  Handles creation of / interaction with the toolbars
 *
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gnome.h>

#include "editMenu.h"
#include "editor.h"
#include "html.h"
#include "site.h"
#include "siteUI.h"
#include "toolbar.h"
#include "todo.h"

extern Site *current_site;
extern Page *current_page;
extern GtkWidget *app;

extern void spell_check( void );

typedef enum _Align {
	LEFT,
	CENTRE,
	JUSTIFY,
	RIGHT
} Align;

static void toolbar_markup( GtkWidget *widget, gchar *open );
static void indent( GtkWidget *widget, gint alt );
static void tag_help( GtkWidget *widget, gpointer data );
static void external_view( GtkWidget *widget, gpointer data );
static void align( GtkWidget *widget, Align a );

GtkWidget *create_toolbar1()
{
	GtkWidget *toolbar;

	GtkWidget *new_button;
	GtkWidget *open_button;
	GtkWidget *save_button;

	new_button = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_NEW );
        open_button = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_OPEN );
        save_button = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_SAVE );

	toolbar = gtk_toolbar_new( GTK_ORIENTATION_HORIZONTAL,
				   GTK_TOOLBAR_ICONS );

	/* new button */
        gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
                                _("New Site"), "", 
                                 new_button, 
				 GTK_SIGNAL_FUNC( screem_site_create_site )
				 ,0 );
        /* open button */
        gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
                                 _("Open Site"), "", 
                                 open_button, 
				 GTK_SIGNAL_FUNC( screem_site_open ), 0 );
        /* save button */
        gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
                                 _("Save Site"), "", 
                                 save_button, 
				 GTK_SIGNAL_FUNC( screem_site_save_proxy ), 
				 0 );
        return toolbar;
}

GtkWidget *create_toolbar2()
{
        GtkWidget *toolbar;

	GtkWidget *cut_button;
	GtkWidget *copy_button;
	GtkWidget *paste_button;

	cut_button = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_CUT );
	copy_button = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_COPY );
	paste_button = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_PASTE );

        toolbar = gtk_toolbar_new( GTK_ORIENTATION_HORIZONTAL,
                                    GTK_TOOLBAR_ICONS );
        /* cut */
        gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
                                 _("Cut selected text to clipboard"), "", 
				 cut_button, GTK_SIGNAL_FUNC( cut ), 0 );
        /* copy */
        gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
                                 _("Copy selected text to clipboard"), "", 
				 copy_button, GTK_SIGNAL_FUNC( copy ), 0 );
        /* paste */
        gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
                                 _("Paste text from clipboard"), "", 
				 paste_button, GTK_SIGNAL_FUNC( paste ), 0 );
        
        return toolbar;
}

GtkWidget *create_toolbar3()
{
        GtkWidget *toolbar;

	GtkWidget *print_button = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_PRINT );
	GtkWidget *search_button =  gnome_stock_new_with_icon(GNOME_STOCK_PIXMAP_SEARCH );
	GtkWidget *view_button =  gnome_stock_new_with_icon(GNOME_STOCK_PIXMAP_EXEC );
	GtkWidget *todo_button =  gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_INDEX );
	GtkWidget *spell_button = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_SPELLCHECK );

        toolbar = gtk_toolbar_new( GTK_ORIENTATION_HORIZONTAL,
                                    GTK_TOOLBAR_ICONS );
        /* print */
        gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
                                 _("Print current page"), "", 
				 print_button, 0, 0 );
        /* search */
        gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
                                 _("Search for text in current page"), "", 
				 search_button, GTK_SIGNAL_FUNC( find ), 0 );
        /* view in browser */
        gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
                                 _("View in external browser"), "",
				 view_button, 
				 GTK_SIGNAL_FUNC( external_view ), 0 );
        /* todo */
        gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
                                 _("View Site ToDo list"), "",
				 todo_button, 
				 GTK_SIGNAL_FUNC( todo_view ), 0 );
        /* spellcheck */
        gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
                                 _("Spellcheck page"), "",
                                 spell_button, 
				 GTK_SIGNAL_FUNC( spell_check ), 0 );

        return toolbar;
}

GtkWidget *create_toolbar4()
{
        GtkWidget *toolbar;

        GtkWidget *tag_combo;
        GList *tag_strings = NULL;
        GtkWidget *help_button;
        gint num;

        toolbar = gtk_toolbar_new( GTK_ORIENTATION_HORIZONTAL,
                                   GTK_TOOLBAR_ICONS );

        help_button = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_HELP );
        tag_strings = NULL;

	for( num = 0; html_tags[ num ]; num ++ ) {
		tag_strings = g_list_append( tag_strings, html_tags[ num ] );
	}
  
	tag_combo = gtk_combo_new();
        gtk_combo_set_case_sensitive( GTK_COMBO( tag_combo ), FALSE );
        gtk_combo_set_value_in_list( GTK_COMBO( tag_combo ), TRUE, FALSE );
        gtk_combo_set_popdown_strings( GTK_COMBO( tag_combo ), tag_strings );
        g_list_free( tag_strings );
        gtk_entry_set_editable( GTK_ENTRY( GTK_COMBO( tag_combo )->entry ),
                                FALSE );
        gtk_widget_set_usize( tag_combo, 128, - 1 );
        gtk_widget_show( tag_combo );

        /* tag help selection */
        gtk_toolbar_append_element( GTK_TOOLBAR( toolbar ),
                                    GTK_TOOLBAR_CHILD_WIDGET, tag_combo, "",
				    _("tag to get help on"), "", NULL,
                                    0, 0 );
        /* help about tag */
        gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
                                 _("help about tag"), "", help_button,
                                 GTK_SIGNAL_FUNC( tag_help ), tag_combo );

        return toolbar;
}

GtkWidget *create_toolbar5()
{
        GtkWidget *toolbar;

        GtkWidget *bold_button;
        GtkWidget *italic_button;
        GtkWidget *under_button;

        toolbar = gtk_toolbar_new( GTK_ORIENTATION_HORIZONTAL,
                                    GTK_TOOLBAR_ICONS );

        bold_button = gnome_stock_new_with_icon(GNOME_STOCK_PIXMAP_TEXT_BOLD );
        italic_button = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_TEXT_ITALIC );
        under_button = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_TEXT_UNDERLINE );


        /* bold */
        gtk_toolbar_append_element( GTK_TOOLBAR( toolbar ),
                                    GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
                                    NULL, "",
                                    _("Bold text"), "", bold_button,
                                    GTK_SIGNAL_FUNC( toolbar_markup ), "<b>" );
        /* italic */
        gtk_toolbar_append_element( GTK_TOOLBAR( toolbar ),
                                    GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
                                    NULL, "",
                                    _("Italic text"), "", italic_button,
				    GTK_SIGNAL_FUNC( toolbar_markup ), "<i>" );
        /* underlined */
        gtk_toolbar_append_element( GTK_TOOLBAR( toolbar ),
                                    GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
                                    NULL, "",
                                    _("Underlined text"), "", under_button,
				    GTK_SIGNAL_FUNC( toolbar_markup ), "<u>" );
        return toolbar;
}

GtkWidget *create_toolbar6()
{
        GtkWidget *toolbar;

        GtkWidget *bullet_button;
        GtkWidget *numbered_button;

        toolbar = gtk_toolbar_new( GTK_ORIENTATION_HORIZONTAL,
                                   GTK_TOOLBAR_ICONS );
        bullet_button = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_TEXT_BULLETED_LIST );
        numbered_button = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_TEXT_NUMBERED_LIST );
        
        /* bulletted list */
        gtk_toolbar_append_element(GTK_TOOLBAR( toolbar ),
				   GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
				   NULL, "",
				   _("Bulleted list"), "", bullet_button,
				   GTK_SIGNAL_FUNC( toolbar_markup ), "<ul>" );
        /* number list */
        gtk_toolbar_append_element(GTK_TOOLBAR( toolbar ),
				   GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
				   NULL, "",
				   _("Number list"), "", numbered_button,
				   GTK_SIGNAL_FUNC( toolbar_markup ), "<ol>" );
        return toolbar;
}

GtkWidget *create_toolbar7()
{
        GtkWidget *toolbar;

        GtkWidget *indent_dec_button;
        GtkWidget *indent_inc_button;

	GtkWidget *left_align_button;
	GtkWidget *centre_align_button;
	GtkWidget *right_align_button;
	GtkWidget *justify_align_button;
        
        toolbar = gtk_toolbar_new( GTK_ORIENTATION_HORIZONTAL,
                                   GTK_TOOLBAR_ICONS );

        indent_dec_button = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_TEXT_UNINDENT );
        indent_inc_button = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_TEXT_INDENT );

        /* indent - */
        gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
                                 _("decrease indentation"),"", 
				 indent_dec_button,
                                 GTK_SIGNAL_FUNC( indent ), 
				 GINT_TO_POINTER( -1 ) );
        /* indent + */
        gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
                                 _("increase indentation"),"", 
				 indent_inc_button,
                                 GTK_SIGNAL_FUNC( indent ), 
				 GINT_TO_POINTER( +1 ) );

        left_align_button = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_ALIGN_LEFT );
        centre_align_button = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_ALIGN_CENTER );
        right_align_button = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_ALIGN_RIGHT );
        justify_align_button = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_ALIGN_JUSTIFY );

	/* align left */
        gtk_toolbar_append_element( GTK_TOOLBAR( toolbar ),
                                    GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
                                    NULL, "",
                                    _( "align left" ), "", left_align_button,
                                    GTK_SIGNAL_FUNC( align ), 
				    GINT_TO_POINTER( LEFT ) );
        /* align centre */
        gtk_toolbar_append_element( GTK_TOOLBAR( toolbar ),
                                    GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
                                    NULL, "",
                                    _( "align centre" ), "", 
				    centre_align_button,
				    GTK_SIGNAL_FUNC( align ),
				    GINT_TO_POINTER( CENTRE ) );
        /* align right */
        gtk_toolbar_append_element( GTK_TOOLBAR( toolbar ),
                                    GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
                                    NULL, "",
                                    _( "align right" ), "", 
				    right_align_button,
				    GTK_SIGNAL_FUNC( align ), 
				    GINT_TO_POINTER( RIGHT ) );
        /* align justify */
        gtk_toolbar_append_element( GTK_TOOLBAR( toolbar ),
				    GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
				    NULL, "",
				    _( "justify" ), "", 
				    justify_align_button,
				    GTK_SIGNAL_FUNC( align ), 
				    GINT_TO_POINTER( JUSTIFY ) );

        return toolbar;
}

/* deals with B I U UL OL */
static void toolbar_markup( GtkWidget *widget, gchar *open )
{
	gchar *close;
	gboolean has_selection;
	gboolean pressed;
	static gboolean toggle = FALSE;

	if( toggle ) {
		toggle = FALSE;
		return;
	}

	pressed = !gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	close = g_strdup_printf( "%c/%s", open[ 0 ], &open[ 1 ] );
	has_selection = screem_editor_has_selection();

	if( has_selection ) {
		toggle = TRUE;
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
					      pressed );
	} else if( pressed ) {
		open = NULL;
	} else {
		g_free( close );
		close = NULL;
	}

	screem_editor_insert_markup( open, close );

	g_free( close );
}

static void indent( GtkWidget *widget, gint alt )
{
    	static gint indent_level = 0;

	indent_level += alt;

	if( indent_level < 0 ) {
		indent_level = 0;
		return;
	}

	if( alt < 0 )
		screem_editor_insert_markup( NULL, "</blockquote>" );
	else
		screem_editor_insert_markup( "<blockquote>", NULL );
}

static void external_view( GtkWidget *widget, gpointer data )
{
	Page *page;
	const gchar *pathname;

	if( current_site )
		page = screem_site_get_current_page( current_site );
	else
		page = current_page;

	g_return_if_fail( page != NULL );

	pathname = screem_page_get_pathname( page );

	g_return_if_fail( pathname != NULL );

	gnome_url_show( pathname );
}

static void tag_help( GtkWidget *widget, gpointer data )
{
	GtkCombo *combo = GTK_COMBO( data );
        gchar *tag;
        gchar *path;

        tag = gtk_entry_get_text( GTK_ENTRY( combo->entry ) );
	
	path = g_strdup_printf( "%s%chelp%cindex.html#%s", 
                                gnome_libdir_file( "screem" ), G_DIR_SEPARATOR,
                                G_DIR_SEPARATOR, tag );
	if( path ) {
                gnome_help_goto( 0, path );
                g_free( path );
        }
}

static void align( GtkWidget *widget, Align a )
{
	static gboolean toggle = FALSE;
	static GtkWidget *current = NULL;
	static Align pa = LEFT;
	gchar *tag = NULL;

	if( toggle ) {
		toggle = FALSE;
		return;
	}

	/* is the click on the same alignement we are already set on? */
	if( current == widget )
		return;
  
	if( current ) {
		toggle = TRUE;
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( current ),
					      FALSE );
		if( pa == CENTRE ) {
			tag = "</center>";
		} else if( pa != LEFT ) {
			tag = "</div>";
		}
		screem_editor_insert_markup( NULL, tag );
	}

	/* now we set the new alignment */
	current = widget;
	pa = a;
	switch( a ) {
	case CENTRE:
		tag = "<center>";
		break;
	case RIGHT:
		tag = "<div align=\"right\">";
		break;
	case JUSTIFY:
		tag = "<div align=\"justify\">";
		break;
	case LEFT:
		return;
		break;
	}

	screem_editor_insert_markup( tag, NULL );
}
