/*  Screem:  preferences.c,
 *  This file deals with the programs settings, loading/saving them and
 *  editing them. 
 *
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */
#include <config.h>
#include <gnome.h>

#include <glade/glade.h>

#include <gnome-xml/debugXML.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>

#include "editor.h"
#include "helpers.h"
#include "html.h"
#include "preferences.h"
#include "site.h"
#include "siteUI.h"
#include "support.h"
#include "tag_tree.h"
#ifdef USE_GTKEXTEXT
#include "gtkextext.h"
#endif

extern GList *helpers;
extern GnomeClient *client;
extern GtkWidget *app;
extern Preferences *cfg;

static 	GladeXML *xml;

static void display_tab( void );
static void tag_colour_tab( void );
static void helpers_tab( void );
static void perl_tab( void );
static void java_tab( void );

void preferences_changed( void );
void preferences_apply( GtkWidget *widget, gint p, gpointer data );
void preferences_done( GtkWidget *widget, gpointer data );

void font_changed( GtkWidget *widget );

void tag_colour_sel( GtkEntry *widget );
void tag_use_change( GtkWidget *widget );
void tag_col_change( GnomeColorPicker *cp, guint r, guint g, guint b,
		     guint a );

void helper_clicked( GtkCList *list, gint row, gint column,
		     GdkEventButton *event );
void helper_add( GtkWidget *button, gpointer data );
void helper_remove( GtkWidget *button, gpointer data );

#define HELPER_NAME "Helpers/pathname"
#define HELPER_TYPE "Helpers/mimetype"
#define HELPER_DEFAULT "invalid"

#define DEFAULT_FONT "-*-helvetica-bold-r-normal--12-*-*-*-*-*-*-*"

static const gchar* colgroups[] = {
	"DEFAULT_COLOURS/", "GLOBAL_STRUCTURE_COLOURS/",
	"BODY_STRUCTURE_COLOURS/", "TEXT_DIRECTION_COLOURS/",
	"TEXT_STRUCTURE_COLOURS/", "TEXT_PARAGRAPHS_COLOURS/",
	"TEXT_CHANGES_COLOURS/", "LIST_COLOURS/",
	"TABLES_COLOURS/", "LINKS_COLOURS/",
	"OBJECTS_COLOURS/", "STYLE_SHEETS_COLOURS/",
	"FONTS_COLOURS/", "FRAMES_COLOURS/",
	"FORMS_COLOURS/", "SCRIPTS_COLOURS/",
	"JSP_COLOURS/", NULL
};

void load_preferences()
{
     	Hi_colours *hc;
	gchar *p = "";
	gchar *m = "";
	gchar *path;
	gint n;
	Helper *helper;

	gint colgroup;
	gchar *prefix;

        gnome_config_push_prefix( "/screem/" );
        cfg = (Preferences *)g_malloc( sizeof( Preferences ) );

	cfg->box = NULL;

	cfg->tag_width = gnome_config_get_int( "Config/tag_width=300" );
	cfg->tag_height = gnome_config_get_int( "Config/tag_height=125" );
	cfg->file_width = gnome_config_get_int( "Config/file_width=300" );
        cfg->file_height = gnome_config_get_int( "Config/file_height=300" );
	cfg->editor_width = gnome_config_get_int( "Config/editor_width=500" );
	cfg->editor_height =
		gnome_config_get_int( "Config/editor_height=425" );
	cfg->preview_width = 
		gnome_config_get_int( "Config/preview_width=500" );
	cfg->preview_height = 
		gnome_config_get_int( "Config/preview_height=425" );
	cfg->tag_highlighting = 
		gnome_config_get_int( "Config/tag_highlighting=0" );
	cfg->inline_tagging = 
		gnome_config_get_int( "Config/inline_tagging=0" );
	cfg->undo_levels = gnome_config_get_int( "Config/undo_levels" );
	cfg->font_name = 
		gnome_config_get_string( "Config/font_name="DEFAULT_FONT );
	if( cfg->font_name ) {
		cfg->font = gdk_font_load( cfg->font_name );
	} else {
		cfg->font = NULL;
	}

	cfg->back[ 0 ] =
                gnome_config_get_int( "Config/back_red=65535" );
        cfg->back[ 1 ] =
                gnome_config_get_int( "Config/back_green=65535" );
        cfg->back[ 2 ] =
                gnome_config_get_int( "Config/back_blue=65535" );
        cfg->fore[ 0 ] =
                gnome_config_get_int( "Config/fore_red=0" );
        cfg->fore[ 1 ] =
                gnome_config_get_int( "Config/fore_green=0" );
        cfg->fore[ 2 ] =
                gnome_config_get_int( "Config/fore_blue=0" );


	cfg->default_filename =
		gnome_config_get_string("Config/default_filename=index.html" );

	cfg->java_compiler =
		gnome_config_get_string( "Config/java_compiler=javac" );

	/* load recent sites */
	cfg->recent_sites = NULL;
	if( gnome_config_has_section( "Recent_sites" ) ) {
		n = 0;
		/* read the paths in and place them in the list */
		while( strcmp( p, "invalid" ) ) {
			path = g_strdup_printf( "Recent_sites/path%i=%s",
						n, HELPER_DEFAULT );
			p = gnome_config_get_string( path );
			g_free( path );
			if( strcmp( p, "invalid" ) )
				cfg->recent_sites = 
					g_list_append( cfg->recent_sites, p );
			else {
				g_free( p );
				p = "invalid";
			}
			n ++;
		}
	}

	/* load recent pages */
	cfg->recent_pages = NULL;
	if( gnome_config_has_section( "Recent_pages" ) ) {
		n = 0;
		p = "";
		/* read the paths in and place them in the list */
		while( strcmp( p, "invalid" ) ) {
			path = g_strdup_printf( "Recent_pages/path%i=%s",
						n, HELPER_DEFAULT );
			p = gnome_config_get_string( path );
			g_free( path );
			if( strcmp( p, "invalid" ) )
				cfg->recent_pages = 
					g_list_append( cfg->recent_pages, p );
			else {
				g_free( p );
				p = "invalid";
			}
			n ++;
		}
	}

	/* load global tag tree */
	cfg->tag_tree = screem_tag_tree_load();

	/* load helpers */
	cfg->helpers = NULL;
	if( gnome_config_has_section( "Helpers" ) ) {
		n = 0;
		p = "";
		/* read the helpers in and place them in the list */
		while( strcmp( p, "invalid" ) && strcmp( m, "invalid" ) ) {
			path = g_strdup_printf( "%s%i=%s", HELPER_NAME, n,
						HELPER_DEFAULT );
			p = gnome_config_get_string( path );
			g_free( path );
			path = g_strdup_printf( "%s%i=%s", HELPER_TYPE, n,
						HELPER_DEFAULT );
			m = gnome_config_get_string( path );
			g_free( path );

			/* do we create a new Helper and add it? */
			if( strcmp( p, "invalid" ) && strcmp( m, "invalid" ) ){
				helper = screem_helper_new();
				helper->path = p;
				helper->mime_type = m;
				screem_helper_add( helper );
			}
			n ++;
		}
	}

	/* load html colours */
	cfg->colours = NULL;
	for( colgroup = 0; colgroups[ colgroup ]; colgroup ++ ) {
		hc = g_new( Hi_colours, 1 );
		prefix = g_strconcat( gnome_client_get_config_prefix( client ),
				      colgroups[ colgroup ], NULL );
		gnome_config_push_prefix( prefix );
		hc->useFore = gnome_config_get_int( "useFore=0" );
		hc->useBack = gnome_config_get_int( "useBack=0" );
		hc->fore[ 0 ] = gnome_config_get_int( "fr=0" );
		hc->fore[ 1 ] = gnome_config_get_int( "fg=0" );
		hc->fore[ 2 ] = gnome_config_get_int( "fb=0" );
		hc->fore[ 3 ] = gnome_config_get_int( "fa=0" );
		hc->back[ 0 ] = gnome_config_get_int( "br=0" );
		hc->back[ 1 ] = gnome_config_get_int( "bg=0" );
		hc->back[ 2 ] = gnome_config_get_int( "bb=0" );
		hc->back[ 3 ] = gnome_config_get_int( "ba=0" );
		cfg->colours = g_list_append( cfg->colours, hc );
		g_free( prefix );
		gnome_config_pop_prefix();
	}

	/* load perl colours */
	cfg->perl_colours = NULL;

	/* load java colours */
	cfg->java_colours = NULL;

	if( ! gnome_config_has_section( "DEFAULT_COLOURS" ) ) {
		/* no default colours, lets set some defaults */
		hc = cfg->colours->data;
		hc->useFore = gnome_config_get_int( "useFore=1" );
		hc->fore[ 0 ] = gnome_config_get_int( "fr=0" );
		hc->fore[ 1 ] = gnome_config_get_int( "fg=0" );
		hc->fore[ 2 ] = gnome_config_get_int( "fb=65535" );
		hc->fore[ 3 ] = gnome_config_get_int( "fa=0" );
	}

	gnome_config_pop_prefix();
}

void save_preferences()
{
	GList *list;
	Hi_colours *hc;
	Helper *helper;
	gchar *path;
	gchar *pathname;
	gint n;
	gint colgroup;
	gchar *prefix;

	GtkWidget *widget;

	gnome_config_push_prefix( gnome_client_get_config_prefix( client ) );

	widget = gtk_object_get_data( GTK_OBJECT( app ), "tag_window" );
   	gnome_config_set_int( "Config/tag_width", widget->allocation.width );
	gnome_config_set_int( "Config/tag_height", widget->allocation.height );

	widget = gtk_object_get_data( GTK_OBJECT( app ), "sitebook" );
	gnome_config_set_int( "Config/file_width", widget->allocation.width );
        gnome_config_set_int( "Config/file_height",widget->allocation.height );

	gnome_config_set_int("Config/editor_width", screem_editor_get_width());
	gnome_config_set_int( "Config/editor_height",
			      screem_editor_get_height() );

	widget = gtk_object_get_data( GTK_OBJECT( app ), "preview" );
	gnome_config_set_int("Config/preview_width", widget->allocation.width);
	gnome_config_set_int( "Config/preview_height",
			      widget->allocation.height );

	gnome_config_set_int("Config/tag_highlighting", cfg->tag_highlighting);
	gnome_config_set_int( "Config/inline_tagging", cfg->inline_tagging );
	gnome_config_set_int( "Config/undo_levels", cfg->undo_levels  );
	gnome_config_set_string( "Config/font_name", cfg->font_name );
	gnome_config_set_int( "Config/back_red", cfg->back[ 0 ] );
	gnome_config_set_int( "Config/back_green", cfg->back[ 1 ] );
	gnome_config_set_int( "Config/back_blue", cfg->back[ 2 ] );
	gnome_config_set_int( "Config/fore_red", cfg->fore[ 0 ] );
	gnome_config_set_int( "Config/fore_green", cfg->fore[ 1 ] );
	gnome_config_set_int( "Config/fore_blue",  cfg->fore[ 2 ] );
	gnome_config_set_string( "Config/default_filename", 
				 cfg->default_filename );
	gnome_config_set_string( "Config/java_compiler", cfg->java_compiler );

	/* save recent sites */
	gnome_config_clean_section( "Recent_sites" );
	gnome_config_sync ();
	for( n = 0, list = cfg->recent_sites; list; list = list->next ) {
		pathname = (gchar*)list->data;
		path = g_strdup_printf( "Recent_sites/path%i", n );
		gnome_config_set_string( path, pathname );
		g_free( path );
		n ++;
	}

	/* save recent pages */
	gnome_config_clean_section( "Recent_pages" );
	gnome_config_sync ();
	for( n = 0, list = cfg->recent_pages; list; list = list->next ) {
		pathname = (gchar*)list->data;
		path = g_strdup_printf( "Recent_pages/path%i", n );
		gnome_config_set_string( path, pathname );
		g_free( path );
		n ++;
	}

	/* save helpers */
	gnome_config_clean_section( "Helpers" );
	gnome_config_sync ();
	for( n = 0, list = cfg->helpers; list; list = list->next ) {
		helper = (Helper*)list->data;
		if( helper ) {
			path = g_strdup_printf( "%s%i", HELPER_NAME, n );
			gnome_config_set_string( path, helper->path );
			g_free( path );
			path = g_strdup_printf( "%s%i", HELPER_TYPE, n );
			gnome_config_set_string( path, helper->mime_type );
			g_free( path );
			n ++;
		}
	}

	/* save colours */
	list = cfg->colours;
	for( colgroup = 0; colgroups[ colgroup ]; colgroup ++ ) {
		hc = (Hi_colours*)list->data;
		prefix = g_strconcat( gnome_client_get_config_prefix( client ),
				      colgroups[ colgroup ], NULL );
		gnome_config_push_prefix( prefix );
		gnome_config_set_int( "useFore", hc->useFore );
		gnome_config_set_int( "useBack", hc->useBack );
		gnome_config_set_int( "fr", hc->fore[ 0 ] );
		gnome_config_set_int( "fg", hc->fore[ 1 ] );
		gnome_config_set_int( "fb", hc->fore[ 2 ] );
		gnome_config_set_int( "fa", hc->fore[ 3 ] );
		gnome_config_set_int( "br", hc->back[ 0 ] );
		gnome_config_set_int( "bg", hc->back[ 1 ] );
		gnome_config_set_int( "bb", hc->back[ 2 ] );
		gnome_config_set_int( "ba", hc->back[ 3 ] );
		g_free( prefix );
		gnome_config_pop_prefix();
		list = list->next;
	}

	gnome_config_sync();
	gnome_config_pop_prefix();
}

void edit_preferences()
{
        if( cfg->box ) {
                gdk_window_raise( cfg->box->window );
                gdk_window_show( cfg->box->window );
                return;
        }

	xml = glade_xml_new( GLADE_PATH"/preferences.glade", "propertybox" );
	glade_xml_signal_autoconnect( xml );
	cfg->box = glade_xml_get_widget( xml, "propertybox" );

	display_tab();
	tag_colour_tab();
	helpers_tab();
	java_tab();
}


static void display_tab()
{
	GtkWidget *label;
	GtkWidget *entry;
	GtkWidget *toggle = NULL;
	GtkStyle *style;
	GdkColor *c;
	GtkWidget *picker;
	GtkWidget *fontpicker1 = glade_xml_get_widget( xml, "fontpicker1" );
    
	/* set cfg values */
	entry = glade_xml_get_widget( xml, "default_filename" );
	gtk_entry_set_text( GTK_ENTRY( entry ), cfg->default_filename );

	if( cfg->font_name ) {
		entry = glade_xml_get_widget( xml, "font_name_entry" );
		gtk_entry_set_text( GTK_ENTRY( entry ), cfg->font_name );
		gnome_font_picker_set_font_name(GNOME_FONT_PICKER(fontpicker1),
						cfg->font_name );
	}

	toggle = glade_xml_get_widget( xml, "syntax_highlighting" );
	gtk_toggle_button_set_state( GTK_TOGGLE_BUTTON( toggle ),
				     cfg->tag_highlighting );

	toggle = glade_xml_get_widget( xml, "inline_tagging" );
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON( toggle ),
				    cfg->inline_tagging );
	
	style = screem_editor_get_style();

#ifdef USE_GTKEXTEXT
	c = &style->bg[ 0 ];
#else
	c = &style->base[ 0 ];
#endif

	picker = glade_xml_get_widget( xml, "editor_background" );
	gnome_color_picker_set_i16( GNOME_COLOR_PICKER( picker ),
				    c->red, c->green, c->blue, 0 );
#ifdef USE_GTKEXTEXT
	c = &style->fg[ 0 ];
#else
	c = &style->text[ 0 ];
#endif
	picker = glade_xml_get_widget( xml, "editor_text" );
	gnome_color_picker_set_i16( GNOME_COLOR_PICKER( picker ),
				    c->red, c->green, c->blue, 0 );
}

void font_changed( GtkWidget *widget )
{
	gchar *font;
	GtkWidget *entry = glade_xml_get_widget( xml, "font_name_entry" );

	font = gnome_font_picker_get_font_name( GNOME_FONT_PICKER( widget ) );
	gtk_entry_set_text( GTK_ENTRY( entry ), font );
}

static void tag_colour_tab()
{
	GList *list;
	GList *tags = NULL;
	Hi_colours *hc;

	GtkWidget *combo;
	GtkWidget *toggle;
	GtkWidget *picker;

	gint tag_count;

	cfg->coloursBak = g_list_copy( cfg->colours );
        for( list = cfg->coloursBak; list; list = list->next ) {
                if( list->data ) {
                        hc = (Hi_colours*)g_malloc( sizeof( Hi_colours ) );
                        memcpy( hc, list->data, sizeof( Hi_colours ) );
                        list->data = hc;
                }
        }
   
	tags = g_list_append( tags, unknown_tags );
        for( tag_count = 0; TAG_GROUPS[ tag_count ]; tag_count ++ )
                tags = g_list_append( tags, TAG_GROUPS[ tag_count ][ 0 ] );
	cfg->current = cfg->coloursBak->data;

	combo = glade_xml_get_widget( xml, "tag_group_selection" );
        gtk_combo_set_popdown_strings( GTK_COMBO( combo ), tags );
        g_list_free( tags );

	
        /* set initial fore colour for unknown tags and set active if needed */
        toggle = glade_xml_get_widget( xml, "fore_check" );
	picker = glade_xml_get_widget( xml, "fore_picker" );
	gtk_signal_connect_object( GTK_OBJECT( toggle ), "toggled",
                                   GTK_SIGNAL_FUNC( change_state ), 
                                   (gpointer)picker );
	if( cfg->current->useFore )
                gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ),
					      TRUE );
        gnome_color_picker_set_i16( GNOME_COLOR_PICKER( picker ),
                                    cfg->current->fore[ 0 ],
                                    cfg->current->fore[ 1 ],
                                    cfg->current->fore[ 2 ],
                                    cfg->current->fore[ 3 ] );
     

        /* set initial back colour for unknown tags and set active if needed */
	toggle = glade_xml_get_widget( xml, "back_check" );
	picker = glade_xml_get_widget( xml, "back_picker" );
	gtk_signal_connect_object( GTK_OBJECT( toggle ), "toggled",
                                   GTK_SIGNAL_FUNC( change_state ), 
                                   (gpointer)picker );
        if( cfg->current->useBack )
                gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ),
					      TRUE );
        gnome_color_picker_set_i16( GNOME_COLOR_PICKER( picker ),
                                    cfg->current->back[ 0 ],
                                    cfg->current->back[ 1 ],
                                    cfg->current->back[ 2 ],
                                    cfg->current->back[ 3 ] );
      
}

void tag_colour_sel( GtkEntry *widget )
{
	gchar *group;
        GList *list;
        gint num;
	GtkWidget *fore;
	GtkWidget *fore_check;
	GtkWidget *back;
	GtkWidget *back_check;

        group = gtk_entry_get_text( widget );
        list = cfg->coloursBak;

        /* which group was selected? */
        for( num = 0; TAG_GROUPS[ num ]; num ++ ) {
                list = list->next;
                if( ! strcmp( group, TAG_GROUPS[ num ][ 0 ] ) )
                        break;
        }

        if( ! TAG_GROUPS[ num ] )
                list = cfg->coloursBak;
        
        cfg->current = list->data;

	fore = glade_xml_get_widget( xml, "fore_picker" );
  	fore_check = glade_xml_get_widget( xml, "fore_check" );
	back = glade_xml_get_widget( xml, "back_picker" );
  	back_check = glade_xml_get_widget( xml, "back_check" );

        /* now set the colour selectors/checkboxes */
        gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( fore_check ),
                                     cfg->current->useFore );
        gnome_color_picker_set_i16( GNOME_COLOR_PICKER( fore ),
                                    cfg->current->fore[ 0 ],
                                    cfg->current->fore[ 1 ],
                                    cfg->current->fore[ 2 ],
                                    cfg->current->fore[ 3 ] );
        gtk_widget_set_sensitive( fore, cfg->current->useFore );
        gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( back_check ),
                                     cfg->current->useBack );
        gnome_color_picker_set_i16( GNOME_COLOR_PICKER( back ),
                                    cfg->current->back[ 0 ],
                                    cfg->current->back[ 1 ],
                                    cfg->current->back[ 2 ],
                                    cfg->current->back[ 3 ] );
        gtk_widget_set_sensitive( back, cfg->current->useBack );
}

void tag_use_change( GtkWidget *widget  )
{
	GtkWidget *check;
	gboolean state;

	state = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

	check = glade_xml_get_widget( xml, "fore_check" );

        if( widget == check )
                cfg->current->useFore = state;
        else 
                cfg->current->useBack = state;

        preferences_changed();
}

void tag_col_change( GnomeColorPicker *cp, guint r, guint g, guint b,
		     guint a )
{
	GtkWidget *fore;
	Hi_colours *hc = cfg->current;

	fore = glade_xml_get_widget( xml, "fore_picker" );

        if( (GtkWidget*)cp == fore ) {
                hc->fore[ 0 ] = r;
                hc->fore[ 1 ] = g;
                hc->fore[ 2 ] = b;
                hc->fore[ 3 ] = a;
        } else {
                hc->back[ 0 ] = r;
                hc->back[ 1 ] = g;
                hc->back[ 2 ] = b;
                hc->back[ 3 ] = a;
        }

        preferences_changed();
}

static void helpers_tab()
{
	GtkWidget *clist1;
	GList *list;
	Helper *helper;
	gchar *entry[ 2 ];

	clist1 = glade_xml_get_widget( xml, "helpers_list" );

	/* add current helpers to the clist */
	for( list = cfg->helpers; list; list = list->next ) {
		helper =  (Helper*)list->data;
		if( helper ) {
			entry[ 0 ] = helper->path;
			entry[ 1 ] = helper->mime_type;
			gtk_clist_append( GTK_CLIST( clist1 ), entry );
		}
	}
}

void helper_clicked( GtkCList *list, gint row, gint column,
			    GdkEventButton *event )
{
	GtkWidget *path = glade_xml_get_widget( xml, "helper_path" );
	GtkWidget *mime_type = glade_xml_get_widget( xml, "helper_mime_type" );
	gchar *entry[ 2 ];

	gtk_object_set_data( GTK_OBJECT( list ), "row", (gpointer)row );

	gtk_clist_get_text( GTK_CLIST( list ), row, 0, &entry[ 0 ] );
	gtk_clist_get_text( GTK_CLIST( list ), row, 1, &entry[ 1 ] );

	gtk_entry_set_text( GTK_ENTRY( path ), entry[ 0 ] );
	gtk_entry_set_text( GTK_ENTRY( mime_type ), entry[ 1 ] );
}

void helper_add( GtkWidget *button, gpointer data )
{
	GtkWidget *list = glade_xml_get_widget( xml, "helpers_list" );
	GtkWidget *path = glade_xml_get_widget( xml, "helper_path" );
	GtkWidget *mime_type = glade_xml_get_widget( xml, "helper_mime_type" );
	gchar *entry[ 2 ];
	Helper *helper;

	entry[ 0 ] = gtk_entry_get_text( GTK_ENTRY( path ) );
	entry[ 1 ] = gtk_entry_get_text( GTK_ENTRY( mime_type ) );
	gtk_clist_append( GTK_CLIST( list ), entry );

	helper = screem_helper_new();
	helper->path = g_strdup( entry[ 0 ] );
	helper->mime_type = g_strdup( entry[ 1 ] );
	screem_helper_add( helper );

	preferences_changed();
}

void helper_remove( GtkWidget *button, gpointer data )
{
	GtkWidget *list = glade_xml_get_widget( xml, "helpers_list" );
	GtkWidget *path = glade_xml_get_widget( xml, "helper_path" );
	gint row = (gint)gtk_object_get_data( GTK_OBJECT( list ), "row" );
	gchar *p = gtk_entry_get_text( GTK_ENTRY( path ) );

	gtk_clist_remove( GTK_CLIST( list ), row );

	screem_helper_remove( p );

        preferences_changed();
}

static void perl_tab()
{

}

static void java_tab()
{
	GtkWidget *compiler_entry = glade_xml_get_widget( xml, "compiler" );

	gtk_entry_set_text( GTK_ENTRY( compiler_entry ), cfg->java_compiler );
}


void preferences_changed()
{
	if( !GTK_NOTEBOOK( GNOME_PROPERTY_BOX(cfg->box)->notebook )->cur_page )
                return;

	gnome_property_box_changed( GNOME_PROPERTY_BOX( cfg->box ) );
}

void preferences_apply( GtkWidget *propbox, gint p, gpointer data )
{
	GtkWidget *widget;

	GtkWidget *colourpicker;
	GtkStyle  *style;
	GdkColor  *c;
	guint16 junk;

	Hi_colours *hc;

	GList *list;

	/* we apply once we receive the signal to apply all, rather than
	   go through and do each tab at a time */
	if( p != -1 )
		return;

	/* default filename */
	widget = glade_xml_get_widget( xml, "default_filename" );
	cfg->default_filename = 
		g_strdup( gtk_entry_get_text( GTK_ENTRY( widget ) ) );

	/* font */
	widget = glade_xml_get_widget( xml, "font_name_entry" );
	g_free( cfg->font_name );
	cfg->font_name = g_strdup( gtk_entry_get_text( GTK_ENTRY( widget ) ) );
      	cfg->font = gdk_font_load( cfg->font_name );
	
	/* tag highlighting */
	widget = glade_xml_get_widget( xml, "syntax_highlighting");
	cfg->tag_highlighting = 
		gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

	/* inline tagging */
	widget = glade_xml_get_widget( xml, "inline_tagging" );
	cfg->inline_tagging = 
		gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

  	/* background colour */
	colourpicker = glade_xml_get_widget( xml, "editor_background" );
     	style = screem_editor_get_style();
	style->font = cfg->font;

#ifdef USE_GTKEXTEXT
	c = &style->bg[ 0 ];
#else
	c = &style->base[ 0 ];
#endif
	gnome_color_picker_get_i16( GNOME_COLOR_PICKER( colourpicker ),
				    &c->red, &c->green, &c->blue, &junk );
	cfg->back[ 0 ] = c->red;
	cfg->back[ 1 ] = c->green;
	cfg->back[ 2 ] = c->blue;

	/* text colour */
	colourpicker = glade_xml_get_widget( xml, "editor_text" );
#ifdef USE_GTKEXTEXT
	c = &style->fg[ 0 ];
#else
	c = &style->text[ 0 ];
#endif
	gnome_color_picker_get_i16( GNOME_COLOR_PICKER( colourpicker ),
				    &c->red, &c->green, &c->blue, &junk );
	cfg->fore[ 0 ] = c->red;
	cfg->fore[ 1 ] = c->green;
	cfg->fore[ 2 ] = c->blue;

	widget = gtk_object_get_data( GTK_OBJECT( app ), "editor" );
	gtk_widget_set_style( widget, style );

#ifdef USE_GTKEXTEXT
	gtk_extext_style_insert(GTK_EXTEXT( widget ),"Default",
				style->font,
				&style->fg[ 0 ],
				&style->bg[ 0 ] );
#endif
	/* set the new highlighting colours */
	list = cfg->colours;
        cfg->colours = cfg->coloursBak;
        for( ; list; list = list->next ) {
                if( list->data )
                        g_free( list->data );
        }
	cfg->coloursBak = g_list_copy( cfg->colours );
        for( list = cfg->coloursBak; list; list = list->next ) {
                if( list->data ) {
                        hc = (Hi_colours*)g_malloc( sizeof( Hi_colours ) );
                        memcpy( hc, list->data, sizeof( Hi_colours ) );
                        list->data = hc;
                }
        }

	/* save them */
	save_preferences();

	/* if we are editing a page then we need to reinsert it all in
	   the possibly new font / highlighting colours */
	screem_editor_changed();
}

void preferences_done( GtkWidget *widget, gpointer data )
{
	GList *list;

	/* free the backup highlighting colour data */
        for( list = cfg->coloursBak; list; list = list->next ) {
                if( list->data )
                        g_free( list->data );
        }
        if( cfg->coloursBak )
                g_list_free( cfg->coloursBak );

	gtk_widget_destroy( widget );
	cfg->box = NULL;
}
