/*  Screem:  html.c,
 *  This file provides function for manipulation of the html
 * 
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <ctype.h>
#include <gnome.h>
#include <string.h>

#include "html.h"
#include "htmlfuncs.h"

gint in_attribute( gchar *text, gint pos )
{
	gint ipos = pos;

	gchar a = '"';
   	gchar c;
	gchar d;

       	/* go back until start of text, or we hit a '<' or '>' */
      	while( (--pos) >= 0 ) {
		c = text[ pos ];
		if( c == a )
			break;
		else if( c == '<' || c == '>' ) {
			return FALSE;
		}
	}
    	if( pos <= 0 )
		return FALSE;

	if( text[ pos - 1 ] != '=' )
		return FALSE;
     
	/* we have an opening, but do we have a closing? */
	d = '\0';
    	while( ipos < strlen( text ) ) {
		c = text[ ipos ];
		if( c == a ) {
			if( d != '=' )
				return pos + 1;
			else
				return FALSE;
		}
		else if( c == '<' || c == '>' )
			return FALSE;
		ipos ++;
		d = c;
	}

	return FALSE;
}

gint in_entity( gchar *text, gint pos )
{
	gint ipos = pos;

	gchar a = '&';
	gchar b = ';';
	gchar c;

       	/* go back until start of text, or we hit a '<' or '>' */
      	while( (--pos) >= 0 ) {
		c = text[ pos ];
		if( c == a )
			break;
		else if( c == b ||  c == '<' || c == '>' || isspace( c ) )
			return FALSE;
	}
	if( pos < 0 )
		return FALSE;

	/* we have an opening, but do we have a closing? */
	while( ipos < strlen( text ) ) {
		c = text[ ipos ];
		if( c == b )
			return pos + 1;
		else if( c == a || c == '<' || c == '>' || isspace( c ) )
			return FALSE;
		ipos ++;
	}

	return FALSE;
}

/* returns FALSE, or the start of the tag + 1 */
gint in_tag( gchar *text, gint pos )
{
	gint ipos = pos;

	gchar a = '<';
	gchar b = '>';

       	/* go back until start of text, or we hit a '<' or '>' */
      	while( (--pos) >= 0 ) {
		if( text[ pos ] == a )
			break;
		else if( text[ pos ] == b )
			return FALSE;
	}
	if( pos < 0 )
		return FALSE;

	/* we have an opening, but do we have a closing? */
	while( ipos < strlen( text ) ) {
		if( text[ ipos ] == a )
			return FALSE;
		if( text[ ipos ] == b )
			return pos + 1;
		ipos ++;
	}
	return FALSE;
}

gchar* next_tag( gchar *text, gint *pos )
{
        gint i, j;
        gchar *tag;

        i = 0;
        while( i < strlen( text ) ) {
                if( text[ i ] == '<' )
                        break;
                i ++;
        }

        /* we came to the end ? */
        if( i == strlen( text ) )
            return NULL;

        /* we've got a possible tag */
        j = i;
        while( j < strlen( text ) ) {
                if( text[ j ] == '>' )
                        break;
                j ++;
        }

        /* we came to the end ? */
        if( j == strlen( text ) )
            return NULL;

        tag = g_strndup( &text[ i ], j - i + 1 );

        *pos += j;

        return tag;
}

gchar *previous_tag( gchar *text, gint *pos )
{
        gint i, j;
        gchar *tag;

        i = *pos;
        while( i > 0 ) {
                if( text[ i ] == '>' )
                        break;
                i --;
        }

        /* we came to the end ? */
        if( i == 0 )
            return NULL;

        /* we've got a possible tag */
        j = i;
        while( j > 0 ) {
                if( text[ j ] == '<' )
                        break;
                j --;
        }

        /* we came to the end ? */
        if( j < 0 )
            return NULL;

        tag = g_strndup( &text[ j ], i - j + 1 );

	*pos = j;

        return tag;
}

/* returns the position in the htmlTags array, or -1 */
gint binary_search_tags( gchar *t )
{
	gint start;
	gint end;
	gint mid;
	gint res;

	start = 0;
	end = html_tags_size;

	while( end >= start && start < html_tags_size ) {
		mid = ( start + end ) / 2;
		res = strcasecmp( t, html_tags[ mid ] );
		if( res < 0 )
			end = mid - 1;
		else if( res > 0 )
			start = mid + 1;
		else {
			/* found */
			return mid;
		}
	}

	return -1;
}

gchar *tag_name( gchar *tag )
{
	gchar *name;
	gchar *temp;

	name = temp = tag + 1;

  	while( ! isspace( *temp ) && *temp != '>' )
		temp ++;
       	*temp = 0;

	name = g_strndup( name, temp - name );

	return name;
}

gchar* current_context( gchar *text, gint pos, gint *start, gint *end )
{
	gint i = -1;
	gint j = -1;
	gint num;
	gchar *tag;
	gchar *tag2 = NULL;
	GList *tags = NULL;
	gboolean retval = FALSE;

	g_return_val_if_fail( text != NULL, NULL );

	/* find last opening tag that allows close tags and isn't already
	   closed */

	while( ( ! retval ) && ( tag = previous_tag( text, &pos ) ) ) {
		/* tag found is it an open tag? */
		num = -2;
		tag2 = NULL;
		i = pos;
		if( text[ pos + 1 ] == '/' ) 
			/* add to list of already closed tags */
			tags = g_list_prepend( tags, tag_name( tag ) );
		else {
			tag2 = tag_name( tag );
			num = binary_search_tags( tag2 );
		}

		if( num > -1 )
			retval = html_tags_close[ num ];
		else
			retval = ( num != -2 );
		if( ! retval )
			g_free( tag2 );
		else {
			j = i + strlen( tag );
			/* is this the tag we are looking for? */
			if( tags && ! strcmp( (char*)tags->data + 1, tag2 ) ) {
				/* no */
				retval = FALSE;
				g_free( tag2 );
				tags = g_list_remove( tags, tags->data );
			} else if( num != 2 && 
				   html_tags_close[ num ] == OPTIONAL &&
				   tags != NULL ) {
				/* found unclosed tag which is optional to
				   close or not, and list isn't empty so
				   we ignore it */
				retval = FALSE;
				g_free( tag2 );
			}
		}
		g_free( tag );
	}

	if( start )
		*start = i;
	if( end )
		*end = j;

	if( ! retval )
		tag2 = NULL;

	return tag2;
}
