/*  Screem: fileops.c,
 *  This file provides file/directory copying/moving/deletion
 * 
 *  Copyright (C) 1999 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <dirent.h>
#include <errno.h>
#include <gnome.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "site.h"
#include "page.h"
#include "fileops.h"

#define COPY_BUFFER_SIZE 4096

gboolean copy_file( const gchar *source, const gchar *dest )
{
	struct stat s;
	char buffer[ COPY_BUFFER_SIZE ];
	FILE *r;
	FILE *w;
	gint size;
	static GtkWidget *prompt;
	static gint button;
	gchar *message;

	/* does dest already exist? */
	if( ! stat( dest, &s ) ) {
		message = g_strconcat( dest, _(": already exists\noverwrite?"),
				       NULL );
		prompt = gnome_message_box_new( message,
						GNOME_MESSAGE_BOX_WARNING,
						GNOME_STOCK_BUTTON_YES,
						GNOME_STOCK_BUTTON_NO,
						NULL );
		g_free( message );
		button = gnome_dialog_run_and_close( GNOME_DIALOG( prompt ) );
			
		if( button == 1 )
			return FALSE;

		/* if button == 0 then it was yes so we just continue */
	}

  	if( ! ( r = fopen( source, "r" ) ) )
		return FALSE;
	
	if( ! ( w = fopen( dest, "w" ) ) ) {
		fclose( r );
		return FALSE;
	}

	while( ( size = fread( buffer, 1, 4096, r ) ) )	{
		if( fwrite( buffer, 1, size, w ) != size ) {
			fclose( r );
			fclose( w );
			return FALSE;
		}
	}

	fclose( r );
	fclose( w );
	return TRUE;
}
/*****************************************************************************/
gboolean copy_dir( const gchar *source, const gchar *dest, gboolean move )
{
	struct stat s;
	DIR *dir;
        struct dirent *entry = NULL;
	gchar *orig;
	gchar *new;

	if( stat( dest, &s ) < 0 ) {
		if( errno != ENOENT )
			return FALSE;
		if( mkdir( dest, 0x1e4 ) < 0 )
			return FALSE;
	}

	if( ! g_file_test( dest, G_FILE_TEST_ISDIR ) )
		return FALSE;

	if( ! ( dir = opendir( source ) ) )
		return FALSE;

	while( ( entry = readdir( dir ) ) ) {
		if( strcmp( "..", entry->d_name ) &&
		    strcmp( ".", entry->d_name ) ) {
			orig = g_strdup_printf("%s%c%s", source,
					       G_DIR_SEPARATOR, entry->d_name);
			new = g_strdup_printf( "%s%c%s", dest, G_DIR_SEPARATOR,
					       entry->d_name );
			if( ! g_file_test( orig, G_FILE_TEST_ISDIR ) ) {
				if( move )
					move_file( orig, new );
				else
					copy_file( orig ,new );
			} else
				copy_dir( orig, new, move );
		
			g_free( orig );
			g_free( new );
		}
	}

	closedir( dir );

	if( move )
		rmdir( source );

	return TRUE;
}
/*****************************************************************************/
gboolean move_file( gchar *source, gchar *dest )
{
	struct stat s;
	static GtkWidget *prompt;
	static gint button;
	gchar *message;

	/* does dest already exist? */
	if( ! stat( dest, &s ) ) {
		message = g_strconcat( dest, _(": already exists\noverwrite?"),
				       NULL );
		prompt = gnome_message_box_new( message,
						GNOME_MESSAGE_BOX_WARNING,
						GNOME_STOCK_BUTTON_YES,
						GNOME_STOCK_BUTTON_NO,
						NULL );
		g_free( message );

		button = gnome_dialog_run_and_close( GNOME_DIALOG( prompt ) );

		if( button == 1 )
			return FALSE;

		/* if button == 0 then it was yes so we just continue */
	}

	if( rename( source, dest ) < 0 ) {
		if( copy_file( source, dest ) ) {
			return ( unlink( source ) < 0 );
		} else
			return FALSE;
	}

	return TRUE;
}
/*****************************************************************************/
gboolean delete_file( const gchar *file )
{
	if( unlink( file ) < 0 )
		return FALSE;

	return TRUE;
}
/*****************************************************************************/
gboolean delete_dir( const gchar *path )
{
	DIR *dir;
        struct dirent *entry = NULL;
	gchar *new;

	dir = opendir( path );
	if( ! dir )
		return FALSE;

	while( ( entry = readdir( dir ) ) ) {
		if( strcmp( "..", entry->d_name ) &&
		    strcmp( ".", entry->d_name ) ) {
			new = g_strdup_printf("%s%c%s", path,
					      G_DIR_SEPARATOR, entry->d_name);
			if( ! g_file_test( new, G_FILE_TEST_ISDIR ) ) {
				delete_file( new );
			} else
				delete_dir( new );
			g_free( new );
		}
	}

	closedir( dir );
	rmdir( path );

	return TRUE;
}
/*****************************************************************************/
/* recusivley makes directories (if needed), starting at current path */
gboolean mkdir_recursive( const gchar *path )
{
	gchar **dir = NULL;
	gchar *p;
	gint pos;

	dir = g_strsplit( path, G_DIR_SEPARATOR_S, 255 );

    	for( pos = 0; dir[ pos ]; pos ++ ) {
		if( strlen( dir[ pos ] ) ) {
			if( chdir( dir[ pos ] ) < 0 ) {
				if( errno != ENOENT ) {
					perror( "chdir" );
					g_strfreev( dir );
					return FALSE;
				} else if( mkdir( dir[ pos ], 0x1e4 ) < 0 ) {
					perror( "mkdir" );
					g_strfreev( dir );
					return FALSE;
				}
				pos --;  /* decrease pos so that we try to
					    enter the directory we just made
					    on the next loop */
			}
		}
	}

	g_strfreev( dir );
	return TRUE;
}
/*****************************************************************************/
/* converts the given path into one relative to the current directory */
gchar *relative_path( gchar *text, const gchar *root )
{
      	gchar *tmp;
	gchar *temp;
	gchar **current;
	gchar **dropped;
	gboolean set = FALSE;
	gint count = 0;
	gint prev = 0;

	const gchar *pathname;
	gchar cwd[ 16384 ];
    
	getcwd( cwd, 16384 );

	tmp = g_strdup( cwd );

	if( root )
		pathname = root;
	else
		pathname = tmp;


	if( ! strncmp( text, pathname, strlen( pathname ) ) ) {
		current = g_strsplit( tmp, G_DIR_SEPARATOR_S, 255 );
		dropped = g_strsplit( text, G_DIR_SEPARATOR_S, 255 );

		while( dropped[ count ] && current[ count ] && ! set ){
			if( ! strcmp( dropped[ count ], current[ count ] ) )
				count ++;
			else    /* didn't match at position */
				set = TRUE;
		}

		for( prev = count; current[ prev ]; prev ++ )
			current[ prev ] = g_strdup( ".." );
		
		if( current[ count ] )
			temp = g_strjoinv( "/", &current[ count ] );
		else
			temp = g_strdup( "." );
		
		text = g_strdup_printf( "%s/%s", temp,
					g_strjoinv("/", &dropped[ count ] ) );
		g_free( temp );
		g_free( tmp );
		g_strfreev( dropped );
		g_strfreev( current );
	} else
		text = g_strdup( text );


	return text;
}
/***************************************************************************/
/* converts the given relative path to a full pathname,
   we should be in the path that it is relative from before calling the
   function */
gchar *relative_to_full( gchar *relPath )
{
	gchar **rel;
	gint count;
	gchar cwd[ 16384 ];
	gchar *path = NULL;

	rel = g_strsplit( relPath, G_DIR_SEPARATOR_S, 255 );

	for( count = 0; rel[ count ]; count ++ ) {
		if( g_file_test( rel[ count ], G_FILE_TEST_ISDIR ) )
			chdir( rel[ count ] );
		else
			path = rel[ count ];
	}

	getcwd( cwd, 16384 );
	path = g_strjoin( G_DIR_SEPARATOR_S, cwd, path, NULL );

	g_strfreev( rel );
	return path;
}
