/*  Screem:  site.c,
 *  contains the real site struct definition, along with access functions.
 *  also contains functions for loading, creating, saving sites and the
 *  site project file
 * 
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <glib.h>
#include <string.h>
#include <unistd.h>
#include <gnome-xml/debugXML.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>
#include <libgnome/libgnome.h>
#include <sys/stat.h>

#include "fileops.h"
#include "page.h"
#include "site.h"
#include "todo.h"
#include "xml.h"

/* don't like including these here */
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include "editMenu.h"

typedef struct _Site {
	gchar *name;
	gchar *pathname;
	gchar *remote_url;
	gchar *remote_method;
	gchar *remote_path;
	gchar *remote_user;
	gchar *remote_pass;
	gchar *http_url;
	gchar *cvs_root;
	gchar *template_path;

	GList *pages;
	GList *tasks;
	Page  *current;   /* the current page */

	gboolean use_cvs;
	gboolean use_template;

	gboolean is_import;  /* flag to signify if we are importing a
				non screem site */

	gint open_pages;     /* the number of open pages */
} RealSite;

static void screem_page_update_all_links( Site *site, Page *page, gchar *src,
					  gchar *dest );
static void screem_page_update_links( Site *site, Page *page, gchar *src,
				      gchar *dest );


static void screem_site_set_pages( Site *site, GList *list );
static void screem_site_set_tasks( Site *site, GList *list );


/**
 * screem_site_new:
 *
 * Creates and initialises a new site struct
 *
 * return values: a Site
 */
Site* screem_site_new()
{
	RealSite *site;

	site = (RealSite *)g_malloc( sizeof( RealSite ) );

	site->name = NULL;
	site->pathname = NULL;
	site->remote_url = NULL;
	site->remote_method = NULL;
	site->remote_path = NULL;
	site->remote_user = NULL;
	site->remote_pass = NULL;
	site->cvs_root = NULL;
	site->template_path = NULL;
	site->pages = NULL;
	site->tasks = NULL;
	site->current = NULL;
	site->http_url = NULL;
	site->is_import = FALSE;
	site->open_pages = -1;
	
	return (Site*)site;
}

/**
 * screem_site_destroy:
 * @site: the site to be destroyed
 *
 * Destroys a site struct
 *
 * return values: none
 */

void screem_site_destroy( Site *site )
{
	RealSite *s;
	g_return_if_fail( site );

	s = (RealSite*)site;

	screem_site_purge( site );

	g_free( s->name );
	g_free( s->pathname );
	g_free( s->remote_url );
	g_free( s->remote_method );
	g_free( s->remote_path );
	g_free( s->remote_user );
	g_free( s->remote_pass );
	g_free( s->cvs_root );
	g_free( s->template_path );
	g_list_free( s->pages );
	g_list_free( s->tasks );
	g_free( s->http_url );
	g_free( s );
}

/**
 * screem_site_purge:
 * @site: the site to be purged
 *
 * removes all the pages and tasks from a site freeing up any memory 
 * they are using
 *
 * return values: none
 */
void screem_site_purge( Site *site )
{
	RealSite *s;
	GList *list;
	Page  *p;
	Todo_item *item;

	g_return_if_fail( site );

	s = (RealSite*) site;

	for( list = s->pages; list; list = list->next ) {
		p = (Page*)list->data;
		screem_page_destroy( p );
	}
	g_list_free( s->pages );
	s->pages = NULL;
	screem_site_set_current_page( site, NULL );

	for( list = s->tasks; list; list = list->next ) {
		item = (Todo_item*)list->data;
		screem_todo_item_destroy( item );
	}
}

/**
 * screem_site_set_name:
 * @site: the site
 * @name: the name to give the site
 *
 * Sets / changes the name of the site
 *
 * return values: none
 */
void screem_site_set_name( Site *site, const gchar *name )
{
	RealSite *s;
	g_return_if_fail( site );
	g_return_if_fail( name );

	s = (RealSite*)site;

	/* does the site already have a name set? */
	if( s->name )
		g_free( s->name );
	
	s->name = g_strdup( name );
}

const gchar* screem_site_get_name( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->name;
}

/**
 * screem_site_set_pathname:
 * @site: the site
 * @pathname: the pathname that contains the site
 *
 * Sets / changes the pathname of the site
 * It does NOT copy the site to the new pathname if changing, its
 * also does NOT alter the page pathnames to use the new site directory
 *
 * return values: none
 */
void screem_site_set_pathname( Site *site, const gchar *pathname )
{
	RealSite *s;

	g_return_if_fail( site );
	g_return_if_fail( pathname );

	s = (RealSite*)site;

	/* does the site already have a pathname set? */
	if( s->pathname )
		g_free( s->pathname );

	/* the pathname must end with a G_DIR_SEPARATOR */
	if( pathname[ strlen( pathname ) - 1 ] != G_DIR_SEPARATOR ) {
		s->pathname = g_strconcat( pathname, G_DIR_SEPARATOR_S, 
					      NULL );
	} else {
		s->pathname = g_strdup( pathname );
	}
}


const gchar* screem_site_get_pathname( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->pathname;
}

/**
 * screem_site_set_remote_url:
 * @site: the site
 * @remote_url: the url of the remote site
 *
 * Sets / changes the address of the remote site for uploading to
 *
 * return values: none
 */
void screem_site_set_remote_url( Site *site, const gchar *remote_url )
{
	RealSite *s;

	g_return_if_fail( site );
	g_return_if_fail( remote_url );

	s = (RealSite*)site;

	/* does the site already have a remote url set? */
	if( s->remote_url )
		g_free( s->remote_url );

	s->remote_url = g_strdup( remote_url );
}

const gchar* screem_site_get_remote_url( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->remote_url;
}

/**
 * screem_site_set_remote_method:
 * @site: the site
 * @remote_method: the upload method for the site
 *
 * Sets / changes the method used to copy the site to the remote url
 *
 * return values: none
 */
void screem_site_set_remote_method( Site *site, const gchar *remote_method )
{
	RealSite *s;

	g_return_if_fail( site );
	g_return_if_fail( remote_method );

	s = (RealSite*)site;

	/* does the site already have a remote method set? */
	if( s->remote_method )
		g_free( s->remote_method );

	s->remote_method = g_strdup( remote_method );
}

const gchar* screem_site_get_remote_method( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->remote_method;
}

/**
 * screem_site_set_remote_path:
 * @site: the site
 * @remote_path: the path at the remote url in which the site will be placed
 *
 * Sets / changes the remote location of the site on remote url
 *
 * return values: none
 */
void screem_site_set_remote_path( Site *site, const gchar *remote_path )
{
	RealSite *s;

	g_return_if_fail( site );
	g_return_if_fail( remote_path );

	s = (RealSite*)site;

	/* does the site already have a remote path set? */
	if( s->remote_path )
		g_free( s->remote_path );

	s->remote_path = g_strdup( remote_path );
}

const gchar* screem_site_get_remote_path( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->remote_path;
}

/**
 * screem_site_set_remote_user
 * @site: the site
 * @remote_user: the username to login to the remote url as
 *
 * Sets / changes the username with which to login to the remote url as
 *
 * return values: none
 */
void screem_site_set_remote_user( Site *site, const gchar *remote_user )
{
	RealSite *s;

	g_return_if_fail( site );
	g_return_if_fail( remote_user );

	s = (RealSite*)site;
	
	/* does the site already have a remote usernmae set? */
	if( s->remote_user )
		g_free( s->remote_user );

	s->remote_user = g_strdup( remote_user );
}

const gchar* screem_site_get_remote_user( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->remote_user;
}

/**
 * screem_site_set_remote_pass
 * @site: the site
 * @remote_pass: the password to login to the remote url
 *
 * Sets / changes the password with which to login to the remote url
 *
 * return values: none
 */
void screem_site_set_remote_pass( Site *site, const gchar *remote_pass )
{
	RealSite *s;

	g_return_if_fail( site );
	g_return_if_fail( remote_pass );

	s = (RealSite*)site;

	/* does the site already have a remote password set? */
	if( s->remote_pass )
		g_free( s->remote_pass );

	s->remote_pass = g_strdup( remote_pass );
}

const gchar* screem_site_get_remote_pass( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->remote_pass;
}

/**
 * screem_site_set_cvs_root:
 * @site: the site
 * @cvs_root: the location of the CVS repository for the site to use
 *
 * Sets / changes the location of the CVS repository that will be used by
 * the site
 *
 * return values: none
 */
void screem_site_set_cvs_root( Site *site, const gchar *cvs_root )
{
	RealSite *s;
	g_return_if_fail( site );
	g_return_if_fail( cvs_root );

	s = (RealSite*)site;

	/* does the site already have a cvs root set? */
	if( s->cvs_root )
		g_free( s->cvs_root );

	s->cvs_root = g_strdup( cvs_root );
	s->use_cvs = TRUE;
}

const gchar* screem_site_get_cvs_root( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->cvs_root;
}

/**
 * screem_site_set_use_cvs:
 * @site: the site
 * @status: if we should use cvs or not
 *
 * return values: none
 */
void screem_site_set_use_cvs( Site *site, gboolean status )
{
	RealSite *s;
	g_return_if_fail( site );

	s = (RealSite*)site;

	s->use_cvs = status;
	if( ! status && s->cvs_root ) {
		g_free( s->cvs_root );
		s->cvs_root = NULL;
	}
}

/**
 * screem_site_set_template_file:
 * @site: the site
 * @template_file: the location of the site's standard page template
 *
 * Sets / changes the location of the site's standard page template, which
 * will be used when adding new pages to the site.
 *
 * return values: none
 */
void screem_site_set_template_path( Site *site, const gchar *template_path )
{
	RealSite *s;
	g_return_if_fail( site );
	g_return_if_fail( template_path );

	s = (RealSite*)site;

	/* does the site already have a cvs root set? */
	if( s->template_path )
		g_free( s->template_path );

	s->template_path = g_strdup( template_path );
}

const gchar* screem_site_get_template_path( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->template_path;
}

/**
 * screem_site_set_use_template:
 * @site: the site
 * @status: if we should use a template or not
 *
 * return values: none
 */
void screem_site_set_use_template( Site *site, gboolean status )
{
	RealSite *s;
	g_return_if_fail( site );

	s = (RealSite*)site;

	s->use_template = status;
	if( ! status && s->template_path ) {
		g_free( s->template_path );
		s->template_path = NULL;
	}
}

/**
 * screem_site_set_http_url:
 * @site: the site
 * @http_url: the url that the pages can be viewed at
 *
 * return values: none
 */
void screem_site_set_http_url( Site *site, const gchar *http_url )
{
	RealSite *s;
	g_return_if_fail( site );
	g_return_if_fail( http_url );

	s = (RealSite*)site;

	/* does the site already have a remote usernmae set? */
	if( s->http_url )
		g_free( s->http_url );

	s->http_url = g_strdup( http_url );
}

const gchar* screem_site_get_http_url( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->http_url;
}

/**
 * screem_site_set_current_page:
 * @site: the site
 * @page: the page
 *
 * return values: none
 */
void screem_site_set_current_page( Site *site, Page *page )
{
	g_return_if_fail( site != NULL );

	((RealSite*)site)->current = page;
}

/**
 * screem_site_get_current_page:
 * @site: the site
 *
 * return values: the current page
 */
Page *screem_site_get_current_page( Site *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return ((RealSite*)site)->current;
}

/**
 * screem_site_set_is_import:
 * @site: the site
 * @val: the status to set
 *
 * return values: none
 */
void screem_site_set_is_import( Site *site, gboolean val )
{
	g_return_if_fail( site != NULL );

	((RealSite*)site)->is_import = val;
}
/**
 * screem_site_get_is_import:
 * @site: the site
 *
 * return values: boolean
 */
gboolean screem_site_get_is_import( Site *site )
{
	g_return_val_if_fail( site != NULL, FALSE );

	return ((RealSite*)site)->is_import;
}

void screem_site_set_open_pages( Site *site, gint num )
{
	g_return_if_fail( site != NULL );

	if( num < 0 )
		num = 0;

	((RealSite*)site)->open_pages = num;
}

gint screem_site_get_open_pages( Site *site )
{
	g_return_val_if_fail( site != NULL, -1 );

	return ((RealSite*)site)->open_pages;
}

/**
 * screem_site_save:
 * @site: the site
 *
 * Saves all the open pages in the site along with the site's project file
 *
 * return values: none
 */
void screem_site_save( Site *site )
{
	GList *pages;
	Page *p;

	g_return_if_fail( site );

	/* first the project file */
	screem_site_write_project_file( site );

	/* now the pages */
	for( pages=screem_site_get_pages( site ); pages; pages=pages->next ) {
		p = (Page*)pages->data;
		if( p )
			screem_page_save( p );
	}
}

/**
 * screem_site_create:
 * @site: the site
 *
 * Creates the site at site->pathname, return TRUE if successful
 * FALSE otherwise.
 * This will NOT load the site (ie fill in site->pages etc)
 * to do this call screem_site_load() after a successfull screem_site_create()
 *
 * return values: an boolean
 */
gboolean screem_site_create( Site *site )
{
	DIR *dir;
	gchar *temp;
	gchar cwd[ 16384 ];

	const gchar *pathname;

	g_return_val_if_fail( site != NULL, FALSE );

	pathname = screem_site_get_pathname( site );

	g_return_val_if_fail( pathname != NULL, FALSE );

	/* does the pathname exist as a directory? */
	dir = opendir( pathname );

	/* if it doesn't exist then... */
	if( ( ! dir ) && ( errno == ENOENT ) ) {

		/* get current directory */
		getcwd( cwd, 16384 );

		/* is the pathname from the root directory? */
		if( pathname[ 0 ] == G_DIR_SEPARATOR )
			chdir( G_DIR_SEPARATOR_S );

		if( ! mkdir_recursive( pathname ) )
			return FALSE;
	       
		/* the directory will now exist */
	} else if( ! dir ) {
		return FALSE;
	} else {
		/* successful, we can close it now */
		closedir( dir );
	}

	/* we must ensure that the pathname ends with a G_DIR_SEPARATOR */
	if( pathname[ strlen( pathname ) - 1] != G_DIR_SEPARATOR ) {
		temp = g_strdup_printf( "%s%c", pathname, 
					G_DIR_SEPARATOR );
		screem_site_set_pathname( site, temp );
		g_free( temp );
	}

	/* write the initial project file */
	screem_site_write_project_file( site );

	chdir( screem_site_get_pathname( site ) );

	return TRUE;
}

/**
 * screem_site_load:
 * @site: the site
 *
 * loads the site and project file
 *
 * return values: an boolean
 */
gboolean screem_site_load( Site *site )
{
	gchar *project_file;
	xmlDocPtr doc;

	const gchar *pathname;

	g_return_val_if_fail( site != NULL, FALSE );

	pathname = screem_site_get_pathname( site );

	g_return_val_if_fail( pathname != NULL, FALSE );

	if( ! g_file_test( pathname, G_FILE_TEST_ISDIR ) ) {
		return FALSE;
	}

	project_file = g_strconcat( pathname, "project.screem", NULL );
	doc = xmlParseFile( project_file );

	if( ! doc ) {
		/* we failed to load the project file, we must be importing */
		screem_site_set_is_import( site, TRUE );
	} else {
		screem_site_parse_project_file( site, doc );
		xmlFreeDoc( doc );
	}

	g_free( project_file );
	chdir( pathname );

	return TRUE;
}

/**
 * screem_site_write_project_file:
 * @site: the site
 *
 * creates and writes a site's project file
 *
 * return values: an boolean
 */
gboolean screem_site_write_project_file( Site *site )
{
	gchar *project_file;
	xmlDocPtr doc;
	xmlNsPtr ns;
	xmlNodePtr tree;
	xmlNodePtr subtree;
	FILE *kludge;
	Todo_item *item;
	GList *tasks;

	const gchar *pathname;
	const gchar *name;
	const gchar *remote_url;
	const gchar *remote_method;
	const gchar *remote_user;
	const gchar *remote_pass;
	const gchar *remote_path;
	const gchar *http_url;
	const gchar *cvs_root;
	const gchar *template_path;

	g_return_val_if_fail( site != NULL, FALSE );

	pathname = screem_site_get_pathname( site );

	g_return_val_if_fail( pathname != NULL, FALSE );

	name = screem_site_get_name( site );
	remote_url = screem_site_get_remote_url( site );
	remote_method = screem_site_get_remote_method( site );
	remote_user = screem_site_get_remote_user( site );
	remote_pass = screem_site_get_remote_pass( site );
	remote_path = screem_site_get_remote_path( site );
	http_url = screem_site_get_http_url( site );
	cvs_root = screem_site_get_cvs_root( site );
	template_path = screem_site_get_template_path( site );

	doc = xmlNewDoc( XML_DEFAULT_VERSION );
	doc->root = xmlNewDocNode( doc, NULL, "SCREEM_PROJECT", NULL );
	ns = xmlNewNs( doc->root, "http://www.screem.org/", "screem" );
	xmlNewChild( doc->root, ns, "title", name );
	xmlNewChild( doc->root, ns, "local", pathname ); 

	tree = xmlNewChild( doc->root, ns, "remote", remote_url );
	if( remote_url )
		xmlSetProp( tree, "method", remote_method );
	else
		xmlSetProp( tree, "method", "Local" );
	xmlSetProp( tree, "username", remote_user );
	xmlSetProp( tree, "password", remote_pass );
	xmlSetProp( tree, "path", remote_path );

	xmlNewChild( doc->root, ns, "http", http_url );

	tree = xmlNewChild( doc->root, ns, "cvs", cvs_root );
	if( cvs_root )
		xmlSetProp( tree, "use", "true" );
	else
		xmlSetProp( tree, "use", "false" );

	xmlNewChild( doc->root, ns, "template", template_path );

	tree = xmlNewChild( doc->root, ns, "tasks", NULL );
        for( tasks=screem_site_get_tasks( site ); tasks; tasks=tasks->next ) {
                item = (Todo_item*)tasks->data;
		subtree = xmlNewChild( tree, ns, "task", NULL );
		xmlSetProp( subtree, "name", item->task );
		xmlSetProp( subtree, "assigned", item->assigned );
		xmlSetProp( subtree, "priority", item->priority );
		xmlSetProp( subtree, "linkedTo", item->linked_to );
		xmlSetProp( subtree, "description", item->description);
	}

	project_file = g_strconcat( pathname, "project.screem", NULL );

	kludge = fopen( project_file, "w" );
	if( kludge ) {
		xmlDocDump( kludge, doc );
		fclose( kludge );
		chmod( project_file, S_IRUSR | S_IWUSR );
	}

	xmlFreeDoc( doc );
	g_free( project_file );

	return (gboolean)kludge;  /* will be NULL (FALSE) if we failed */
}

/**
 * screem_site_parse_project_file:
 * @site: the site
 * @doc: the xml tree of the project file
 *
 * parses the given sites project file
 *
 * return values: an boolean
 */
gboolean screem_site_parse_project_file( Site *site, xmlDocPtr doc )
{
	gchar *string;
	xmlNodePtr node;
	Todo_item *item;

	g_return_val_if_fail( site, FALSE );
	g_return_val_if_fail( doc, FALSE );

	/* get the site name */
	string = xml_get_value( doc->root, "title" );
	if( string )
		screem_site_set_name( site, string );

	/* get the site pathname */
	string = xml_get_value( doc->root, "local" );
	if( string )
		screem_site_set_pathname( site, string );

	/* get the remote url */
	node = xml_search_child( doc->root, "remote" );
	string = xml_get_value( doc->root, "remote" );
	if( string )
		screem_site_set_remote_url( site, string );
	if( node ) {
		string = xml_get_value( node, "method" );
		if( string )
			screem_site_set_remote_method( site, string );
      		string = xml_get_value( node, "path" );
		if( string )
			screem_site_set_remote_path( site, string );
		string = xml_get_value( node, "username" );
		if( string )
			screem_site_set_remote_user( site, string );
		string = xml_get_value( node, "password" );
		if( string )
			screem_site_set_remote_pass( site, string );
	}

	if( ! screem_site_get_remote_method( site ) )
		screem_site_set_remote_method( site,_( "Local" ) );

	/* get http url */
   	string = xml_get_value( doc->root, "http" );
	if( string )
		screem_site_set_http_url( site, string );

	/* get the cvs repository path */
	node = xml_search_child( doc->root, "cvs" );
	string = xml_get_value( node, "use" );
	if( strcmp( "use", string ) ) {
		string = xml_get_value( doc->root, "cvs" );
		if( string )
			screem_site_set_cvs_root( site, string );
	}

	/* get the template path */
	string = xml_get_value( doc->root, "template" );
	if( string )
		screem_site_set_template_path( site, string );

	/* get the tasks */
	node = xml_search_child( doc->root, "tasks" );
	if( node ) {
		node = node->childs;
                while( node ) {
			item = screem_todo_item_new();;
                        item->task = g_strdup( xml_get_value( node, "name"));
                        item->assigned = g_strdup(xml_get_value(node,
								"assigned"));
                        item->priority = g_strdup(xml_get_value(node,
                                                                  "priority"));
                        item->linked_to = g_strdup(xml_get_value(node,
                                                                  "linkedTo"));
                        item->description = g_strdup(xml_get_value(node, "description" ) );
                        screem_site_add_task( site, item );
                        node = node->next;
		}
	}

	return TRUE;
}


/**
 * screem_site_locate_page:
 * @site: the site
 * @path: the pages pathname
 *
 * search the site for a page with the given pathname and return the page
 * if found
 *
 * return values: a Page
 */
Page* screem_site_locate_page( Site *site, const gchar *path )
{
	GList *list;
	Page *page = NULL;
	const gchar *pathname;

	g_return_val_if_fail( site, NULL );
	g_return_val_if_fail( path, NULL );

	for( list = screem_site_get_pages(site ); list && ( ! page );
	     list = list->next ) {
		page = (Page*)list->data;
		pathname = screem_page_get_pathname( page );
		if( ! pathname || (pathname && strcmp(path, pathname)) )
			page = NULL;
	}

	return page;
}

/**
 * screem_site_add_page:
 * @site: the site
 * @path: the pages pathname
 *
 * Adds a new page to the site
 *
 * return values: none
 */
void screem_site_add_page( Site *site, const gchar *path )
{
	Page *page;
	GList *list;

	g_return_if_fail( site != NULL );
	g_return_if_fail( path != NULL );

	list = screem_site_get_pages( site );

	page = screem_page_new();
	screem_page_set_pathname( page, path );

	list = g_list_append( list, page );
	screem_site_set_pages( site, list );
}

static void screem_site_set_pages( Site *site, GList *list )
{
	g_return_if_fail( site != NULL );

	((RealSite*)site)->pages = list;
}

/**
 * screem_site_remove_page:
 * @site: the site
 * @path: the pages pathname
 *
 * Removes the page with the given pathname
 *
 * return values: a Page
 */
Page* screem_site_remove_page( Site *site, const gchar *path )
{
	Page *page;
	GList *list;
	gint len;
	
	g_return_val_if_fail( site != NULL, NULL );
	g_return_val_if_fail( path != NULL, NULL );

	list = screem_site_get_pages( site );

	len = g_list_length( list );

	page = screem_site_locate_page( site, path );

	if( page ) {
		list = g_list_remove( list, page );
		screem_site_set_pages( site, list );
	}
	return page;
}

/**
 * screem_site_get_pages:
 * @site: the site
 *
 * return values: a list of all the pages in the site
 */
GList* screem_site_get_pages( Site *site )
{
	return ((RealSite*)site)->pages;
}

/**
 * screem_site_locate_task:
 * @site: the site
 * @task: the name of the task to locate
 *
 * finds the task with the given name.
 *
 * return values: a Todo_item pointer
 */
Todo_item* screem_site_locate_task( Site *site, const gchar *task )
{
	GList *list;
	Todo_item *item = NULL;

	g_return_val_if_fail( site, NULL );
	g_return_val_if_fail( task, NULL );

	for( list = screem_site_get_tasks( site ); ( list ) && ( ! item ) ; 
	     list = list->next ) {
		item = (Todo_item*)list->data;
		if( strcmp( task, item->task ) )
			item = NULL;
       	}

	return item;
}

/**
 * screem_site_add_task:
 * @site: the site
 * @item: the item to add
 *
 * adds the given todo item to the sites list of tasks
 *
 * return values: none
 */
void screem_site_add_task( Site *site, Todo_item *item )
{
	GList *list;

	g_return_if_fail( site != NULL );
	g_return_if_fail( item != NULL );

	list = screem_site_get_tasks( site );

	list = g_list_append( list, item );
}

static void screem_site_set_tasks( Site *site, GList *list )
{
	g_return_if_fail( site != NULL );

	((RealSite*)site)->tasks = list;
}


/**
 * screem_site_remove_task:
 * @site: the site
 * @task: the name of the task to remove
 *
 * finds the task with the given name and removes it
 *
 * return values: a Todo_item pointer
 */
Todo_item* screem_site_remove_task( Site *site, const gchar *task )
{
	Todo_item *item = NULL;
	GList *list;

	g_return_val_if_fail( site, NULL );
	g_return_val_if_fail( task, NULL );

	list = screem_site_get_tasks( site );

	item = screem_site_locate_task( site, task );
	
	if( item ) {
		list = g_list_remove( list, item );
		screem_site_set_tasks( site, list );
	}

	return item;
}

/**
 * screem_site_get_tasks:
 * @site: the site
 *
 * return values: a list of all the tasks in the site
 */
GList* screem_site_get_tasks( Site *site )
{
	return ((RealSite*)site)->tasks;
}

/**
 * screem_site_file_change:
 * @site: the site
 * @src:  the original file
 * @dest: the files new location
 *
 * scans all pages in the site that have links to src and changes them
 * to link to dest
 *
 * return values: none
 */
void screem_site_file_change( Site *site, gchar *src, gchar *dest )
{
	GList *list;

	gchar cwd[ 16384 ];

	Page *page;
	const gchar *page_path;

	getcwd( cwd, 16384 );

	/* go through all pages in the site and update any links */
	for( list = screem_site_get_pages( site ); list; list = list->next ) {
		page = list->data;

		screem_page_load( page );

		page_path = screem_page_get_pathname( page );

		if( ! strcmp( src, page_path ) ) {
			/* it appears we moved this page */
			screem_page_update_all_links( site, page, src, dest );
		} else {
			/* 
			 * it wasn't this page we moved so some links
			 * in it (page) may need updating 
			 */
			screem_page_update_links( site, page, src, dest );
		}
	}

	chdir( cwd );
}

/* for when we move the given page */
static void screem_page_update_all_links( Site *site, Page *page, gchar *src,
					  gchar *dest )
{
	gchar *text;
	gchar *link;
	gchar *temp;

	gchar *path;
	gchar *temp2;
	gchar *temp3;
	gchar *temp4;
	gchar *new_link;

	struct stat s;

	gchar *page_path;
	gchar *new_path;
	gboolean is_full;
	const gchar *spath;

	g_return_if_fail( page != NULL );
	g_return_if_fail( src != NULL );
	g_return_if_fail( dest != NULL );

	text = page->data;

	/* regexp for matching a tag with an ="*" attribute,
	   which may be a filename */
	link = g_strdup_printf( "<[^<]*=\\\"*\\\"[^<>]*>" );

	page_path = g_dirname( src );
	new_path = g_dirname( dest );
	spath = screem_site_get_pathname( site );

 	temp = text;
	while( temp && ( text = find_text( temp, link, NULL ) ) ) {
		/* possible link */
		temp = text;
		while( *temp++ && *temp != '>' );
		
		path = g_strndup( text, temp - text + 1 );
		temp2 = path;
		while( ( temp3 = find_text( temp2, "=\\\"*\\\"", NULL ) ) ) {
			/* is temp3 a file in the site? */
			temp4 = temp2;  /* store the position we found it at */
			temp2 = temp3;
			temp2 += 2; /* bypass the first " */
			while( *(++temp2) != '"' );
			temp3 = g_strndup( temp3 + 2, temp2 - temp3 - 2 );

			/* is it actually a file? */
			chdir( page_path );
			if( ! stat( temp3, &s ) ) {
				/* convert temp3 into a new relative path
				   if its not absolute */
				is_full = ( temp3[ 0 ] == G_DIR_SEPARATOR );
				temp = NULL;
				if( ! is_full ) {
					temp = relative_to_full( temp3 );
					/* we have a full path */
					chdir( new_path );
					new_link = relative_path( temp,spath );
				} else {
					new_link = g_strdup( temp );
				}
				g_free( temp );
				/* FIXME: replace the old link with
				   new_link */
			}

			g_free( temp3 );
		}

		temp = text + strlen( path );

     		g_free( path );
	}

	g_free( page_path );
	g_free( new_path );
}

/* for when the page has a link to the moved file */
static void screem_page_update_links( Site *site, Page *page, gchar *src,
				      gchar *dest )
{
	gchar *link;
	gchar *file;
	gchar *file_reg;

	gchar *temp;
	gchar *text;
	gchar *path;
	gchar *pos;
	gchar *start;
	gchar *chunk;
	gboolean is_full;

	gchar *page_path;
	const gint src_len = strlen( src );
	const gboolean is_dir = g_file_test( src, G_FILE_TEST_ISDIR );
	const gchar *spath;

	gchar *new_path;

	file = src + src_len - ( 1 + is_dir );
	while( *file != G_DIR_SEPARATOR )
		file --;
	file = g_strdup( ++file );

	file_reg = g_strdup_printf( "[^>\"]*%s[^\">]*", file );
	link = g_strdup_printf( "<[^<]*=\\\"%s\\\"[^<>]*>", file_reg );

	g_free( file );
	
	page_path = g_dirname( screem_page_get_pathname( page ) );

	spath = screem_site_get_pathname( site );

	temp = page->data;
	while( temp && ( text = find_text( temp, link, NULL ) ) ) {
		/* 
		 * we have a possible link at location
		 * pointed to by text, now locate the
		 * exact point of path in the text
		 */
		path = find_text( text, file_reg, NULL );
		/*
		 * path will now point to the start of
		 * the pathname in the text, that may be
		 * the same path as src 
		 */
		pos = text;
		text = path;
		while( *text++ != '"' );
		path = g_strndup( path, text - path - 1 );
		/* 
		 * path now == the full path of the link 
		 * id if it is a full or relative link
		 */
		chdir( page_path );

		if( ! ( is_full = ( path[ 0 ] == G_DIR_SEPARATOR ) ) )
			path = relative_to_full( path );
		
		if( ! strncmp( path, src, src_len ) ) {
			/*
			 * The link is to src or possibly to a sub dir of src
			 * if it is a directory 
			 */
			new_path = g_strdup_printf( "%s%s", dest,
						    path + src_len );
			start = g_strndup( page->data, pos - page->data );
			/* convert new_path to a relative path */
			chdir( page_path );
			if( ! is_full )
				temp = relative_path( new_path, spath );
			else
				temp = g_strdup( new_path );
			g_free( new_path );
			/* replace text at point text with temp */
			pos = g_strdup( pos );
			chunk = find_text( pos, file_reg, temp );
		  	g_free( page->data );
			page->data = g_strconcat( start, chunk, NULL );
			text = page->data + strlen( start ) + strlen( temp );
			g_free( chunk );
			g_free( start );
			g_free( temp );
			/* pos isn't freed here as find_text does it */
		}

		g_free( path );
		temp = text;
	}

	g_free( link );
	g_free( file_reg );
	g_free( page_path );
}
