/*  Screem:  imageWizard.c,
 * A new improved image wizard
 *
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include <gmodule.h>
#include <gnome.h>

#include <glade/glade.h>

#include "site.h"
#include "page.h"
#include "editor.h"
#include "fileops.h"

#include "image.xpm"

extern GtkWidget *app;
extern Site *current_site;
extern Page *current_page;

static GladeXML *xml;

static GtkWidget* create_dialog ();
void imageWizard();
void image_wizard_preview( GtkEntry *widget );
void image_wizard_closed( GtkWidget *widget, GdkEvent *event, gpointer data );
void image_wizard_clicked( GtkWidget *widget, gint button, gpointer data );

/*********************************************************************/
G_MODULE_EXPORT const gchar*
g_module_check_init( GModule *module )
{
        g_print("imageWizard: check-init\n");
        return NULL;
}
/*********************************************************************/
G_MODULE_EXPORT void
g_module_unload( GModule *module )
{
        g_print( "imageWizard: unloaded\n" );
}
/*********************************************************************/
G_MODULE_EXPORT void 
init() 
{
        GtkWidget *imageButton;
	GtkWidget *toolbar;

        GnomeUIInfo menuinfo[] = { 
                {
                        GNOME_APP_UI_ITEM, N_( "Image..." ),
                        N_( "Insert an Image" ),
                        imageWizard, NULL, NULL,
                        GNOME_APP_PIXMAP_STOCK,
                        GNOME_STOCK_MENU_BLANK,
                        0,
                        GDK_CONTROL_MASK, NULL
                },
                GNOMEUIINFO_END
        };

   	toolbar = gtk_object_get_data( GTK_OBJECT( app ), "wizardbar" );
        /* place menu item after image under insert */
        gnome_app_insert_menus( GNOME_APP( app ),
                                _("_Insert/"), menuinfo);

        /* place a button on the wizards toolbar */
        imageButton = gnome_pixmap_new_from_xpm_d( IMAGE_ICON );
        gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
                                 _("Image Wizard"), "", imageButton ,
                                 imageWizard, 0 );

        g_print( "imageWizard: initialised\n" );
}
/*********************************************************************/
void imageWizard()
{
	static GtkWidget *dialog = NULL;
	Page *page;

	if( current_site )
		page = screem_site_get_current_page( current_site );
	else
		page = current_page;

	g_return_if_fail( page != NULL );
        
        if( dialog ) {
                gdk_window_raise( dialog->window );
                gdk_window_show( dialog->window );
                return;
        }

	xml = glade_xml_new( GLADE_PATH"/imageWizard.glade",
			     "imagewizard_dialog" );
	glade_xml_signal_autoconnect( xml );

	dialog = glade_xml_get_widget( xml, "imagewizard_dialog" );

	gtk_object_set_data( GTK_OBJECT( dialog ), "dialog", &dialog );
}
/*********************************************************************/
void image_wizard_preview( GtkEntry *widget )
{
	GtkPixmap *image;
	gchar *filename;
	GtkWidget *frame;
	GtkWidget *width;
	GtkWidget *height;

	gchar *w;
	gchar *h;

	GdkImlibImage *i;
    	GdkPixmap *pixmap;
	GdkBitmap *mask;

	GtkWidget *junk;

	GtkWidget *iw;
	GtkWidget *ih;
	GtkWidget *fs;

	struct stat s;
	gchar *size;

	frame = glade_xml_get_widget( xml, "frame1" );
	image = gtk_object_get_data( GTK_OBJECT( frame ), "image" );
	width = glade_xml_get_widget( xml, "width_entry" );
	height = glade_xml_get_widget( xml, "height_entry" );

	iw = glade_xml_get_widget( xml, "initial_width" );
	ih = glade_xml_get_widget( xml, "initial_height" );
	fs = glade_xml_get_widget( xml, "file_size" );

	filename = gtk_entry_get_text( widget );

	if( ! filename )
		return;

	/* does the filename exist?  if it doesn't then we don't want
	   to try loading it as imlib spews loads warning message and
	   attempts to load the image with imagemagick etc */
       if( stat( filename, &s ) < 0 )
	       return;
       /* if its a directory we aren't interested */
       if( g_file_test( filename, G_FILE_TEST_ISDIR ) )
	       return;

	i = gdk_imlib_load_image( filename );

	if( ! i )
		return;

	/* set the width/height fields */
	w = g_strdup_printf( "%i", i->rgb_width );
	h = g_strdup_printf( "%i", i->rgb_height );
	size = g_strdup_printf( "%i bytes", (gint)s.st_size );
	gtk_entry_set_text( GTK_ENTRY( width ), w );
	gtk_entry_set_text( GTK_ENTRY( height ), h );

	gtk_label_set( GTK_LABEL( iw ), w );
	gtk_label_set( GTK_LABEL( ih ), h );
	gtk_label_set( GTK_LABEL( fs ), size );

	g_free( w );
	g_free( h );
	g_free( size );

	gdk_imlib_render( i, 160, 120 );
	pixmap = gdk_imlib_move_image( i );
	mask = gdk_imlib_move_mask( i );

	if( ! image ) {
		junk = glade_xml_get_widget( xml, "junk" );
		gtk_widget_destroy( junk );
		image = GTK_PIXMAP( gtk_pixmap_new( pixmap, mask ) );
		gtk_widget_show( GTK_WIDGET( image ) );
                gtk_container_add (GTK_CONTAINER (frame), GTK_WIDGET(image));
                gtk_object_set_data (GTK_OBJECT (frame), "image", image);
	}

	gtk_pixmap_set( GTK_PIXMAP( image ), pixmap, mask );
	gtk_widget_show( GTK_WIDGET( image ) );
	
}
/*********************************************************************/
void image_wizard_closed( GtkWidget *widget, GdkEvent *event, gpointer data )
{
	GtkWidget **d;

	d = gtk_object_get_data( GTK_OBJECT( widget ), "dialog" );
	if( d )
		*d = NULL;
}
/*********************************************************************/
void image_wizard_clicked( GtkWidget *widget, gint button, gpointer data )
{
	GtkWidget **d;
	GtkWidget *fileEntry;
	GtkWidget *altEntry;
	GtkWidget *widthEntry;
	GtkWidget *heightEntry;
	GtkWidget *alignEntry;
	GtkWidget *copy;
	GtkWidget *thumb;
	gchar *filename;
	gchar *alt;
	gdouble width;
	gdouble height;
	gchar *align;
	GtkWidget *border;
	gint b;
	GtkWidget *hspace;
	gint hs;
	GtkWidget *vspace;
	gint vs;
       	gboolean copyNeeded;
	gboolean makeThumb;

	gchar *dest;
	gchar *dfile;
	gchar *tfile = NULL;
	Page *page;

	GdkImlibImage *i, *i2;

	gchar *imgTag;
	gchar *aTag;

	gint pos;

	gchar *dir;

	const gchar *pathname;

	if( current_site )
		page = screem_site_get_current_page( current_site );
	else
		page = current_page;

	g_return_if_fail( page != NULL );

	pathname = screem_page_get_pathname( page );

	d = gtk_object_get_data( GTK_OBJECT( widget ), "dialog" );

	fileEntry = glade_xml_get_widget( xml, "image_path_entry" );
	altEntry = glade_xml_get_widget( xml, "alt_entry" );
	widthEntry = glade_xml_get_widget( xml, "w_num_entry" );
	heightEntry = glade_xml_get_widget( xml, "h_num_entry" );
	alignEntry = glade_xml_get_widget( xml, "align_entry" );
	copy = glade_xml_get_widget( xml, "copy" );
	thumb = glade_xml_get_widget( xml, "thumb" );
	border = glade_xml_get_widget( xml, "border" );
	hspace = glade_xml_get_widget( xml, "hspace" );
	vspace = glade_xml_get_widget( xml, "vspace" );

	if( button < 2 ) { /* ok or applied */
		/* get the filename */
		filename = gtk_entry_get_text( GTK_ENTRY( fileEntry ) );

		/* get the alt text */
		alt = gtk_entry_get_text( GTK_ENTRY( altEntry ) );
		
		/* get the width value */
		width = gnome_number_entry_get_number( GNOME_NUMBER_ENTRY( widthEntry ) );
		/* get the height value */
		height = gnome_number_entry_get_number( GNOME_NUMBER_ENTRY( heightEntry ) );

		/* get the alignment */
		align = gtk_entry_get_text( GTK_ENTRY( alignEntry ) );

		/* get the border size */
		b = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON(border));
		
		/* get hspace size */
		hs = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(hspace));

		/* get vspace size */
		vs = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(vspace));

		/* do we need to copy the file to the pages directory? */
		copyNeeded = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( copy ) );
		/* do we need to make a thumbnail? */
		makeThumb = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( thumb ) );

		/* copy the file if we need to */
		if( copyNeeded ) {
			dest = g_strndup( pathname, 
					  g_filename_index( pathname ) );
			dfile = g_strdup_printf( "%s%s", dest, filename + g_filename_index( filename ) );
			g_free( dest );
			copy_file( filename, dfile );
			filename = dfile;
		}

		/* make the thumbnail if we need to */
		if( makeThumb ) {
			dfile = g_strndup( filename, g_extension_pointer( filename ) - filename - 1);
			tfile = g_strdup_printf("%s-thumb.%s", dfile,
						g_extension_pointer(filename));
			g_free( dfile );
		
			i = gdk_imlib_load_image( filename );
			i2 = gdk_imlib_clone_scaled_image( i, (gint)width,
							   (gint)height );
			gdk_imlib_destroy_image( i );
			gdk_imlib_save_image( i2, tfile, NULL );
			gdk_imlib_destroy_image( i2 );

			dfile = tfile;
			tfile = filename;
			filename = dfile;
		}
		dir = g_dirname( pathname );
		chdir( dir );
		g_free( dir );
		/* now construct the tag we are going to insert */
		imgTag = g_strdup_printf( "<img src=\"%s\" alt=\"%s\" width=%i height=%i align=\"%s\" border=%i hspace=%i vspace=%i>", relative_path(filename, NULL), alt, (gint)width, (gint)height, align, b, hs, vs );

		aTag = imgTag;

		/* if its a thumbnail we need to wrap this up in a href tag */
		if( makeThumb ) {
			dfile = filename;
			filename = tfile;
			tfile = filename;
		
			aTag = g_strdup_printf( "<a href=\"%s\">%s</a>", 
						relative_path(filename, NULL),
						imgTag );
			g_free( imgTag );
		}

		/* now we insert it at the current cursor position */
		pos = screem_editor_get_pos();
		screem_editor_insert( pos, aTag );

		g_free( aTag );
	}

	if( button == 2 || button == 0 ) { /* ok or close clicked */
		gtk_widget_destroy( GTK_WIDGET( widget ) );
		if( d )
			*d = NULL;
		else
			g_error( "Error in image wizard clicked callback" );
	}
}
/*********************************************************************/
