/*

howcome  27/9/95:

This is the style sheet module (i.e. the "styler") of Arena. It will
parse CSS style sheets into structures that can be queried by the
formatter of e.g an HTML browser.

Typically, the application will create a new style sheet initialized
with the application fallback values. For this, use StyleGetInit

Then, if the user has a personal style sheet -- or if incoming
documents have style sheets -- they can be parsed and added to an
exsiting style sheet by StyleChew

As the HTML/SGML parser encounters tags, FormatElementStart should be
called to have style properties evaluated for the specific element in
this specific context. The set of properties are pushed onto a stack
which corresponds to the document structure.

The formatter can query the style properties of the current element
StyleGet.

FormatElementEnd pops a set of style properties from the stack and
shoul be called when the parser reports the end of an element.


History:

howcome 15/2/94: written from scratch

Dave Raggett:

How about the following notation for style:

        * -> font.name = times

        H1 -> font.name = garamond
        H[2-6] -> font.name = helvetica

        P.CLASS = abstract ->
            {
                font.name = helvetica
                font.size = 10 pt
                indent.left = 4 em
                indent.right = 4 em
            }

The left hand side of the "->" operator matches tag names and uses
the dot notation for attribute values. You can use regular expressions
such as [digit-digit] [letter-letter] * and ?. The browser is responsible
for sorting the rules in order of most to least specificity so it doesn't
matter that the first rule matches anything.

The right hand side simply sets property values. The HTML formatter knows
what properties are relevant to which elements. Some properties are
specific to one kind of element, while others are generic. Properties
set by a more specific rule (matching the current element) override values
set by least specific rules or values inherited from nested or peer elements.


howcome 1/4/95: 

  Dave's proposal for syntax has been implemented with some shortcuts.
  The "->" construct was replaced by ":" whish should have the same associations. 

  The "P.CLASS = abstract" construct is more general than required for
  the class attribute. As a shortcut, "P [abstract]" has been implemented.

howcome 5/4/95: 

  pattern searches implemented /H1 P/: color.text = #FFF

  Also, commands can be put on the same line with a ";" separating.

  # this is a comment
  / so is this

howcome 5/95: changes to the syntax:

  class is now always denoted p.foo
  elemets and hints can be grouped:

  H1, H2: font.family = helvetica; font.color = #F00

howcome 8/95:

  style sheet module updated to support the august 10 version of

    http://www.w3.org/hypertext/WWW/Style/css/draft.html

*/


#include <stdio.h>
#include <stdarg.h>
#include <signal.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>

#include "defs.h"
#include "bridge.h"
#include "colour.h"
#ifdef ARENA_DEBUG	/* QingLong.14-12-96 */
#  include "debug.h"
#endif
#include "display.h"
#include "dither.h"
#include "html.h"
#include "image.h"
#include "main.h"
#include "parsehtml.h"
#include "scrollbar.h" /* used by StyleZoom() *//* Stephen McCamant, 13-09-97*/
#include "status.h"
#include "style_i.h"
#include "style.h"
#include "util.h"
#include "x11.h"

double lens_factor = 1.0;

#if 0
static HTArray*    style_stack = NULL;
static StyleFlat*  current_flat;
static StyleSheet* current_sheet;
#endif

/* 
  str2list is a convenience function to make a list out of two strings
*/

HTList* str2list(char *s, char *t)
{
 HTList* l = HTList_new();

 if (t) HTList_addObject(l, strdup(t));
 if (s) HTList_addObject(l, strdup(s));
 return l;			      
}


/*
  pt2px converts points to pixels

  display_pt2px =  (25.4 / 72.0) * 
    (((double) DisplayWidth(display,screen)) / ((double) DisplayWidthMM(display, screen)));
*/

int pt2px(double pt)
{
 return (int) ((display_pt2px * pt) + 0.5);
}


/* 
 * The Flag functions are used to remember various settings.
 * E.g., if there is a big initial, we want that property to inherit
 * to child elements, but we do not want it to be used more than once.
 * Depending on HTML/SGML parser, this problem may have more elegant solution
 */


Bool StyleGetFlag(Doc* theDoc, int flag)
{
 switch (flag)
   {
    case S_INITIAL_FLAG:
        return theDoc->current_sheet->initial_flag;
        break;

    case S_LEADING_FLAG:
        return theDoc->current_sheet->leading_flag;
        break;

    case S_INDENT_FLAG:
        return theDoc->current_sheet->indent_flag;
        break;

    case S_MARGIN_TOP_FLAG:
        return theDoc->current_sheet->margin_top_flag;
        break;
   }

 return False;
}


void StyleSetFlag(Doc* theDoc, int flag, Bool value)
{
 switch (flag)
   {
    case S_INITIAL_FLAG:
        theDoc->current_sheet->initial_flag = value;
        break;

    case S_LEADING_FLAG:
        theDoc->current_sheet->leading_flag = value;
        break;

    case S_INDENT_FLAG:
        theDoc->current_sheet->indent_flag = value;
        break;

    case S_MARGIN_TOP_FLAG:
        theDoc->current_sheet->margin_top_flag = value;
        break;	
   }
}



/*
 * methods
 */


StyleRule* NewStyleRule(void)
{
 return (StyleRule*)Arena_CAlloc(1, sizeof(StyleRule), False);
}


StyleSelector* NewStyleSelector(void)
{
 return (StyleSelector*)Arena_CAlloc(1, sizeof(StyleSelector), False);
}


StyleSimpleSelector* NewStyleSimpleSelector(void)
{
 return (StyleSimpleSelector*)Arena_CAlloc(1, sizeof(StyleSimpleSelector),
					    False);
}


StyleSheet* NewStyleSheet(void)
{
 return (StyleSheet*)Arena_CAlloc(1, sizeof(StyleSheet), False);
}


Bool DelStyleSheet(StyleSheet* thestylesheet)
{
 if (thestylesheet != NULL)
   {
    free(thestylesheet);
    return True;
   }
  else
   return False;
}


/*
 * NewStyleFlat allocates memory AND sets initial values. CHANGE!
 */
StyleFlat* NewStyleFlat(void)
{
 StyleFlat* sf;

 sf = (StyleFlat*)Arena_CAlloc(1, sizeof(StyleFlat), False);

 sf->value[ArenaSI_MARGIN_LEFT].value   = 20;
 sf->value[ArenaSI_MARGIN_LEFT].type    = ST_PX;

 sf->value[ArenaSI_MARGIN_RIGHT].value  = 20;
 sf->value[ArenaSI_MARGIN_RIGHT].type   = ST_PX;

 sf->value[ArenaSI_MARGIN_TOP].value    = 5;
 sf->value[ArenaSI_MARGIN_TOP].type     = ST_PX;

 sf->value[ArenaSI_MARGIN_BOTTOM].value = 5;
 sf->value[ArenaSI_MARGIN_BOTTOM].type  = ST_PX;

 sf->value[ArenaSI_FONT].value          = -2;

 return sf;
}


StyleValue* NewStyleValue(void)
{
 return (StyleValue*)Arena_CAlloc(1, sizeof(StyleValue), False);
}



void FreeStyleSelector(StyleSelector* selector)
{
 if (selector)
   {
    StyleSelector* prev_selector = selector;

    while ((selector = selector->ancestor))
      {
       Free(prev_selector->ss.class);
       Free(prev_selector);
       prev_selector = selector;
      }
    
    if (prev_selector == NULL)
      {
       Free(prev_selector->ss.class);
       Free(prev_selector);
      }
   }
}


void FreeStyleRule(StyleRule* r)
{
 if (r)
   {
    FreeStyleSelector(r->selector);
    Free(r->warning);
   }
}


void FreeStyleSheet(StyleSheet* s)
{
 if (s)
   {
    HTList* l;
    StyleRule* r;
    StyleInternalProperty i;

    for (i = ArenaSI_UNKNOWN; i < ArenaSI_NPROPERTY; i++)
      {
       if ((l = s->iproperty[i]))
	 while ((r = (StyleRule*)HTList_nextObject(l)))
	   FreeStyleRule(r);
      }

    free(s);
   }
}


/*
void StyleFree(StyleSheet *sheet)
{
    int i;

    for (i = ArenaSI_UNKNOWN; i < ArenaSI_NPROPERTY; i++) {
        HTList_destroy(sheet->element[i]);
    }
    Free(sheet);
}
*/


StyleSelector* CopySelector(StyleSelector* s)
{
 StyleSelector* sc = NULL;

 if (s)
   {
    sc = NewStyleSelector();
    memcpy(sc, s, sizeof(StyleSelector));
    sc->ancestor = CopySelector (s->ancestor);
   }

 return sc;
}


StyleSheet* StyleCopy(StyleSheet* s)
{
 StyleSheet* sc = NULL;
 StyleInternalProperty i;
 HTList *l, *lc;
 StyleRule *r, *rc;


 if (s)	/* QingLong.09-12-96 */
   {
    sc = NewStyleSheet();
    for (i = ArenaSI_UNKNOWN; i < ArenaSI_NPROPERTY; i++)
      {
       l = s->iproperty[i];
       lc = sc->iproperty[i];
       while ((r = (StyleRule*)HTList_nextObject(l)))
         {
          rc = NewStyleRule();
          rc->iproperty = r->iproperty;
          rc->value_p = r->value_p;
          rc->weight = r->weight;
          if (r->warning) rc->warning = strdup(r->warning);

          rc->selector = CopySelector(r->selector);
          if (lc == NULL) lc = sc->iproperty[i] = HTList_new();
          HTList_addObject(lc, (void *)rc);
         }
      }
   }

 return sc;
}


/* find, and grade selector matches */

/* 
   SimpleSelectorMatch matches two "selector utits", i.e. an element
   with accompanying class (and, in level 2, attributes in general.

   r = rule, s = stack

   The gradins schems is simple: 

   0 = failure, no match
   1 = element matches, no classes specified in rule
   2 = element matches, classes specified and matches

*/
int SimpleSelectorMatch(StyleSimpleSelector* r, StyleSimpleSelector* s)
{
 if ((r) && (s))
   if (r->element == s->element) 
     {
      if (r->class)
	{
	 if (s->class)
	   return ((strcasecmp(r->class, s->class) == 0) ? 2 : 0);
	  else
	   return 0;
	}
       else
	return 1;
     }

 return 0;
}		


int SelectorMatch(StyleSelector* s, HTArray* the_style_stack)
{
 StyleSimpleSelector* stack_el;
 int pos = the_style_stack->size - 1;

 return (SimpleSelectorMatch(&s->ss, (StyleSimpleSelector*)the_style_stack->data[the_style_stack->size - 1]));

 while (pos && (stack_el = (StyleSimpleSelector*)the_style_stack->data[pos--]))
   {
    if (SimpleSelectorMatch(&s->ss, stack_el)) s = s->ancestor;
    if (s == NULL) return 0;
   }

 return 1;
}


/*
 * StyleDeductValue is used to convert all units into pixels.
 * For percentages, we must keep the previous values (if inherited)
 * except for margins...
 */
void StyleDeductValue(StyleValue *rule_value, StyleValue *prev_value,
		      StyleValue *value)
{
  value->type = rule_value->type;
  value->value = rule_value->value;
  if (rule_value->iproperty == ArenaSI_COLOUR)
    {
        if(rule_value->type == ST_RGB)
            value->value = rgb2ix( (int)(rule_value->value >> 24), (int)((rule_value->value >> 16) & 0xFF), 
                                   (int)((rule_value->value >> 8) & 0xFF), (int)(rule_value->value & 0xFF), False);
        else
            value->value = rgb2ix( (int)(rule_value->value >> 24), (int)((rule_value->value >> 16) & 0xFF), 
                                   (int)((rule_value->value >> 8) & 0xFF), (int)(rule_value->value & 0xFF), False);
    }
    else
    {
        if(rule_value->type == ST_PERCENT)
        {
            if((rule_value->iproperty != ArenaSI_MARGIN_LEFT) &&
               (rule_value->iproperty != ArenaSI_MARGIN_RIGHT) &&
               (rule_value->iproperty != ArenaSI_MARGIN_TOP) &&
               (rule_value->iproperty != ArenaSI_MARGIN_BOTTOM))
            {
                value->value = (int) (((double)prev_value->value) * ((double)rule_value->value) / 100.0);
                value->type = ST_PX;
            }
        }
        else
            if(rule_value->type == ST_PT)
            {
                if(rule_value->iproperty != ArenaSI_FONT_SIZE)
                {
                    value->value = (long)(pt2px((double)rule_value->value));
                    value->type = ST_PX;
                }
            }
    }
}


/* 
 * StyleEval flattens the whole list of properties.
 * One could argue that properties should only be evaluated when needed.
 * That would complicate things a bit, and may not be worth it...
 */
StyleFlat* StyleEval(StyleSheet* sheet, HTArray* the_style_stack)
{
 StyleSimpleSelector* stack_el;
 StyleFlat *last_flat = NULL, *flat = NewStyleFlat();
 StyleRule* rule;
 StyleInternalProperty i;
 HTList* l;
 Bool recompute_font = False;


#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
 if (REGISTER_TRACE) Arena_TracePrint("StyleEval", ")\n");
#endif

 if (the_style_stack->size <= 0) return NULL;
 if (the_style_stack->size > 1)
   {
    StyleSimpleSelector* last_stack_el = (StyleSimpleSelector*)the_style_stack->data[the_style_stack->size - 2];
    last_flat = last_stack_el->flat;
   }


 stack_el = (StyleSimpleSelector*) the_style_stack->data[the_style_stack->size - 1];


 /* copy the inherited properties from ancestor */

 for (i = ArenaSI_UNKNOWN; i < ArenaSI_NPROPERTY; i++)
   {
    if (last_flat)
      {
       switch (i)
	 {
	    /* these properies are not inherited */
            case ArenaSI_BACKGROUND:
            case ArenaSI_BACKGROUND_COLOUR:
            case ArenaSI_BACKGROUND_IMAGE:
            case ArenaSI_TEXT_DECORATION:

            case ArenaSI_DISPLAY:
            case ArenaSI_WIDTH:
            case ArenaSI_HEIGHT:
            case ArenaSI_FLOAT:
            case ArenaSI_CLEAR:
            case ArenaSI_PACK:

            case ArenaSI_PADDING_LEFT:
            case ArenaSI_PADDING_RIGHT:
            case ArenaSI_PADDING_TOP:
            case ArenaSI_PADDING_BOTTOM:

            case ArenaSI_MARGIN_LEFT:
            case ArenaSI_MARGIN_RIGHT:
            case ArenaSI_MARGIN_TOP:
            case ArenaSI_MARGIN_BOTTOM:

            case ArenaSI_BORDER_STYLE_LEFT:
            case ArenaSI_BORDER_STYLE_RIGHT:
            case ArenaSI_BORDER_STYLE_TOP:
            case ArenaSI_BORDER_STYLE_BOTTOM:

            case ArenaSI_BORDER_WIDTH_LEFT:
            case ArenaSI_BORDER_WIDTH_RIGHT:
            case ArenaSI_BORDER_WIDTH_TOP:
            case ArenaSI_BORDER_WIDTH_BOTTOM:

 	    case ArenaSI_BORDER_COLOUR_LEFT:
            case ArenaSI_BORDER_COLOUR_RIGHT:
            case ArenaSI_BORDER_COLOUR_TOP:
            case ArenaSI_BORDER_COLOUR_BOTTOM:

                break;

            default:
                flat->value[i] = last_flat->value[i];
                flat->weight[i] = 0;
                flat->origin[i] = 0;
                flat->specificity[i] = 0;
	 }
      }

    recompute_font += (flat->value[ArenaSI_FONT].value == -2);

    /* go through list of rules associated with this property */
    switch (i)
      {
       case ArenaSI_FONT:
	 if (recompute_font)
	   flat->value[ArenaSI_FONT].value =  GetFont((char *) flat->value[ArenaSI_FONT_FAMILY].value,
                                                      (int)flat->value[ArenaSI_FONT_SIZE].value,
                                                      (int)flat->value[ArenaSI_FONT_WEIGHT].value,
                                                      (int)flat->value[ArenaSI_FONT_STYLE_SLANT].value, False);
	 break;

       default:
	 l = sheet->iproperty[i];
	 while ((rule = (StyleRule*)HTList_nextObject(l)))
	   {
	    int current_selector_match = 1, tmp_match;

	    tmp_match = SelectorMatch(rule->selector, the_style_stack);

	    if (tmp_match >= current_selector_match)
	      {
	       current_selector_match = tmp_match;
	       if ((rule->weight >= flat->weight[i]) && 
		   (rule->origin >= flat->origin[i])
#if 0
		   &&
		   (rule->specificity >= flat->specificity[i])
#endif
		   )
		 {
		 /* ok, the rule applies  */
		  if ((i == ArenaSI_FONT_SIZE) ||
		      (i == ArenaSI_FONT_FAMILY) ||
		      (i == ArenaSI_FONT_WEIGHT) ||
		      (i == ArenaSI_FONT_STYLE_SLANT))
		    if (flat->value[i].value != rule->value_p->value)
		      recompute_font++;

		  flat->weight[i] = rule->weight;
		  flat->origin[i] = rule->origin;
		  flat->specificity[i] = rule->specificity;

		  /* Here the magic will appear ! */
#if 1
		  if (last_flat)
		    StyleDeductValue(rule->value_p, &(last_flat->value[i]), &(flat->value[i]));
		   else
		    StyleDeductValue(rule->value_p, NULL, &(flat->value[i]));
#else

		  /* StyleDeductValue(rule->value_p, &(flat->value[i])); */

		  flat->value[i].type = rule->value_p->type;
		  if (i != ArenaSI_COLOUR)
		    flat->value[i].value = rule->value_p->value;
		   else
		    flat->value[i].value = rgb2ix( (int)(rule->value_p->value >> 24), (int)((rule->value_p->value >> 16) & 0xFF), 
						   (int)((rule->value_p->value >> 8) & 0xFF), (int)(rule->value_p->value & 0xFF), False);
#endif
		 }
	      }
	   }
      }
   }
 return flat;
}


/* 
 * StyleGet is used by the UA to pull values out of a flattened stylesheet.
 */
long StyleGet(Doc* theDoc, StyleInternalProperty iproperty)
{
#ifdef ARENA_DEBUG
 char Iam[] = "StyleGet";
#endif


 if (theDoc->current_flat)
   {
#if 1	/* QingLong.24-10-97 */
    switch (iproperty)
      {
       case ArenaSI_BACKGROUND:
	 if ((theDoc->current_flat->value[iproperty].value) &&
	     (theDoc->current_flat->origin[ArenaSI_BACKGROUND_COLOUR] >
	      theDoc->current_flat->origin[iproperty]))
#if 1	/* QingLong.24-10-97 */
	   return 0;
# else
	   {
	    BG_Style* theBGstyle = (BG_Style*)(theDoc->current_flat->value[iproperty].value);

	    theBGstyle->flag &= ~S_BACKGROUND_IMAGE;
	    theBGstyle->image = NULL; /* Cut! Memory leakage possible? */
	   }
#endif
	 return (theDoc->current_flat->value[iproperty].value);
	 break;

       case ArenaSI_BACKGROUND_IMAGE:
	 if ((theDoc->current_flat->value[iproperty].value) &&
	     (theDoc->current_flat->origin[ArenaSI_BACKGROUND_COLOUR] >
	      theDoc->current_flat->origin[iproperty]))
	   return 0;

	 return (theDoc->current_flat->value[iproperty].value);
	 break;

       case ArenaSI_MARGIN_LEFT:
       case ArenaSI_MARGIN_RIGHT:
       case ArenaSI_MARGIN_TOP:
       case ArenaSI_MARGIN_BOTTOM:
	 if (theDoc->current_flat->value[iproperty].type == ST_PERCENT)
	   {
	    Frame* parent_frame = (Frame*)Pop();

	    if (parent_frame)
	      {
	       Push(parent_frame);
	       if ((iproperty != ArenaSI_MARGIN_LEFT) &&
		   (iproperty != ArenaSI_MARGIN_RIGHT))
		 return ((int)(((float)parent_frame->height)*((float)theDoc->current_flat->value[iproperty].value)/100.0));
	        else
		 return ((int)(((float)(parent_frame->width - parent_frame->leftmargin - parent_frame->rightmargin))*((float)theDoc->current_flat->value[iproperty].value)/100.0));
	      }
	     else
	      return 0;  /* error, no parent frame!!! */
	   }

	 return (theDoc->current_flat->value[iproperty].value);
	 break;

       default:
	 return (theDoc->current_flat->value[iproperty].value);
	 break;
      }
    /* End ``switch (iproperty)'' */
# else
     if ((iproperty != ArenaSI_MARGIN_LEFT) &&
	 (iproperty != ArenaSI_MARGIN_RIGHT) &&
	 (iproperty != ArenaSI_MARGIN_TOP) &&
	 (iproperty != ArenaSI_MARGIN_BOTTOM))
       return (theDoc->current_flat->value[iproperty].value);
      else
       {
	if (theDoc->current_flat->value[iproperty].type == ST_PERCENT)
	  {
#if 1	/* QingLong.01-02-97 */
	   Frame *parent_frame;
# else
	   Frame *parent_frame, *curr_frame;
#endif

#if 0
	   curr_frame = (Frame *)Pop();
	   if (curr_frame)
	     {
#endif
	      parent_frame = (Frame *)Pop();
	      if (parent_frame)
		{
		 Push(parent_frame);
#if 0
		 Push(curr_frame);  /* restore the stack */
#endif
		 if ((iproperty != ArenaSI_MARGIN_LEFT) &&
		     (iproperty != ArenaSI_MARGIN_RIGHT))
		   return ((int)(((float)parent_frame->height)*((float)theDoc->current_flat->value[iproperty].value)/100.0));
		  else
		   return ((int)(((float)(parent_frame->width - parent_frame->leftmargin - parent_frame->rightmargin))*((float)theDoc->current_flat->value[iproperty].value)/100.0));
		}
#if 0
	       else
		 {
		  Push(curr_frame);
		  return 0; /* error, no parent frame */
		 }
	     }
#endif
	    else
	     return 0;  /* error, no frames!!! */
	  }
	 else
	  return (theDoc->current_flat->value[iproperty].value);
       }
#endif
   }
  else
   {
#ifdef ARENA_DEBUG
    if (STYLE_TRACE)
      Arena_TracePrint(Iam, " WARNING! No current flat!\n");
#endif
    return -1;
   }
}


/*
 * Store and manage style rules.
 */


/*
 * StyleAddRule adds a rule to the pool of style rules.
 */
void StyleAddRule(StyleSheet* sheet, StyleSelector* selector, HTList* values,
		  Byte origin, Byte weight)
{
 HTList* l = values;
 StyleValue* sv_p;

 while ((sv_p = (StyleValue*)HTList_nextObject(l)))
   {
    StyleInternalProperty iproperty = sv_p->iproperty;
    StyleRule* rule = NewStyleRule();

    rule->selector = selector;
    rule->iproperty = iproperty;
    rule->value_p = sv_p;
    rule->weight = weight;
    rule->origin = origin;
    rule->specificity = 0;

    if (sheet->iproperty[iproperty] == NULL)
      sheet->iproperty[iproperty] = HTList_new();

    HTList_addObject(sheet->iproperty[iproperty], (void *)rule);
   }
}


void StyleAddSimpleRule(StyleSheet* sheet, int element,
			StyleInternalProperty iproperty, Byte type,
			int origin, long value)
{
 if (sheet)
   {
    StyleSelector* s = NewStyleSelector();
    StyleValue* sv_p = NewStyleValue();
    StyleRule*  rule = NewStyleRule();

    s->ss.element = element;

    sv_p->type = type;
    sv_p->value = value;
    sv_p->iproperty = iproperty;

    rule->selector = s;
    rule->iproperty = iproperty;
    rule->value_p = sv_p;
    rule->weight = SW_NORMAL;
    rule->origin = origin;
    rule->specificity = 0;

    if (sheet->iproperty[iproperty] == NULL)
      sheet->iproperty[iproperty] = HTList_new();

    HTList_addObject(sheet->iproperty[iproperty], (void *)rule);
   }
}


/*
 * Parse section
 */

StyleExternalProperty str2eproperty(char *s)
{
 int l = Arena_StrLen(s);

 if (strncasecmp(s, "font", 4) == 0)
   {
    if (l == 4) return ArenaSE_FONT;

    if ((l ==  9) && (strcasecmp(s + 5, "size") == 0))   return ArenaSE_FONT_SIZE;
    if ((l == 11) && (strcasecmp(s + 5, "family") == 0)) return ArenaSE_FONT_FAMILY;
    if ((l == 11) && (strcasecmp(s + 5, "weight") == 0)) return ArenaSE_FONT_WEIGHT;
    if ((l == 10) && (strcasecmp(s + 5, "style") == 0))  return ArenaSE_FONT_STYLE;
   }
 else
   if ((strcasecmp(s, "color") == 0) || (strcasecmp(s, "colour") == 0))
     {
      return ArenaSE_COLOUR;
     }
   else
     if (strcasecmp(s, "background") == 0)
       {
        return ArenaSE_BACKGROUND;
       }
     else
       if (strncasecmp(s, "bg", 2) == 0)
	 {
	  if ((l == 18) && (strcasecmp(s + 3, "blend-direction") == 0))
            return ArenaSE_BG_BLEND_DIRECTION;
	  if ((l == 8) && (strcasecmp(s + 3, "style") == 0))
            return ArenaSE_BG_STYLE;
	  if ((l == 10) && (strcasecmp(s + 3, "postion") == 0))
            return ArenaSE_BG_POSITION;
	 }
       else
	 if (strncasecmp(s, "text", 4) == 0)
	   {
	    if ((l == 9) && (strcasecmp(s + 5, "decoration") == 0))
	      return ArenaSE_TEXT_DECORATION;
	    if ((l == 14) && (strcasecmp(s + 5, "transform") == 0))
	      return ArenaSE_TEXT_TRANSFORM;
	    if ((l == 10) && (strcasecmp(s + 5, "align") == 0))
	      return ArenaSE_TEXT_ALIGN;
	    if ((l == 11) && (strcasecmp(s + 5, "indent") == 0))
	      return ArenaSE_TEXT_INDENT;
	   }
	 else
	   if (strncasecmp(s, "padding", 7) == 0)
	     {
	      return ArenaSE_PADDING;
	     }
	   else
	     if(strncasecmp(s, "bg", 2) == 0)
	       {
		if ((l == 8 ) && (strcasecmp(s + 3, "style") == 0))
		  return ArenaSE_BG_STYLE;
		if ((l == 11 ) && (strcasecmp(s + 3, "position") == 0))
		  return ArenaSE_BG_POSITION;
		if ((l == 18 ) && (strcasecmp(s + 3, "blend-direction") == 0))
		  return ArenaSE_BG_BLEND_DIRECTION;
	       }
	     else
	       if (strncasecmp(s, "letter", 6) == 0)
		 {
		  if ((l == 14) && (strcasecmp(s + 7, "spacing") == 0))
		    return ArenaSE_LETTER_SPACING;
		 }
	       else
		 if (strncasecmp(s,"word", 4) == 0)
		   {
		    if ((l == 12) && (strcasecmp(s + 5, "spacing") == 0))
		      return ArenaSE_WORD_SPACING;
		   }
		 else
		   if (strncasecmp(s, "margin", 6) == 0)
		     {
		      if (l == 6)
			return ArenaSE_MARGIN;
		      if ((l == 10) && (strcasecmp(s + 7, "top") == 0))
			return ArenaSE_MARGIN_TOP;
		      if ((l == 12) && (strcasecmp(s + 7, "right") == 0))
			return ArenaSE_MARGIN_RIGHT;
		      if ((l == 13) && (strcasecmp(s + 7, "bottom") == 0))
			return ArenaSE_MARGIN_BOTTOM;
		      if ((l == 11) && (strcasecmp(s + 7, "left") == 0))
			return ArenaSE_MARGIN_LEFT;
		     }
		   else
		     if (strcasecmp(s, "line-height") == 0)
		       {
			return ArenaSE_LINE_HEIGHT;
		       }
		     else
		       if (strcasecmp(s, "vertical-align") == 0)
			 {
			  return ArenaSE_VERTICAL_ALIGN;
			 }
		       else
			 if (strcasecmp(s, "display") == 0)
			   {
			    return ArenaSE_DISPLAY;
			   }
			 else
			   if (strcasecmp(s, "width") == 0)
			     {
			      return ArenaSE_WIDTH;
			     }
			   else
			     if (strcasecmp(s, "height") == 0)
			       {
				return ArenaSE_HEIGHT;
			       }
			     else
			       if (strcasecmp(s, "float") == 0)
				 {
				  return ArenaSE_FLOAT;
				 }
			       else
				 if (strcasecmp(s,"clear") == 0)
				   {
				    return ArenaSE_CLEAR;
				   }
				 else
				   if (strcasecmp(s, "pack") == 0)
				     {
				      return ArenaSE_PACK;
				     }

#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
 if (STYLE_TRACE)
   Arena_TracePrint("StyleExternalProperty",
		    " Unknown style property `%s'\n", s);
#endif

 return ArenaSE_UNKNOWN;
}


int element_enum(char *s, int *token_class_p)
{
    int c, len;

    if (!s)
        return UNKNOWN;

    len = Arena_StrLen(s);
    c = TOLOWER(*s);

    if (isalpha(c))
    {
        if (c == 'a')
        {
            if (len == 1 && strncasecmp(s, "a", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_ANCHOR;
            }

            if (len == 3 && strncasecmp(s, "alt", len) == 0)
            {
                *token_class_p = EN_BLOCK;
                return TAG_ALT;
            }

            if (len == 5 && strncasecmp(s, "added", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_ADDED;
            }

            if (len == 7 && strncasecmp(s, "address", len) == 0)
            {
                *token_class_p = EN_BLOCK;
                return TAG_ADDRESS;
            }

            if (len == 8 && strncasecmp(s, "abstract", len) == 0)
            {
                *token_class_p = EN_BLOCK;
                return TAG_ABSTRACT;
            }
        }
        else if (c == 'b')
        {
            if (len == 1)
            {
                *token_class_p = EN_TEXT;
                return TAG_BOLD;
            }

            if (len == 2 && strncasecmp(s, "br", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_BR;
            }

            if (len == 4 && strncasecmp(s, "body", len) == 0)
            {
                *token_class_p = EN_MAIN;
                return TAG_HTML;
            }

            if (len == 10 && strncasecmp(s, "blockquote", len) == 0)
            {
                *token_class_p = EN_BLOCK;
                return TAG_QUOTE;
            }

            if (len == 4 && strncasecmp(s, "base", len) == 0)
            {
                *token_class_p = EN_SETUP;
                return TAG_BASE;
            }
        }
        else if (c == 'c')
        {
            if (len == 4)
            {
                if (strncasecmp(s, "code", len) == 0)
                {
                    *token_class_p = EN_TEXT;
                    return TAG_CODE;
                }

                if (strncasecmp(s, "cite", len) == 0)
                {
                    *token_class_p = EN_TEXT;
                    return TAG_CITE;
                }
            }
            else if (len == 7 && (strncasecmp(s, "caption", len) == 0))/* howcome 3/2/95: = -> == after hint from P.M.Hounslow@reading.ac.uk */
            {
                *token_class_p = EN_BLOCK;
                return TAG_CAPTION;
            }
        }
        else if (c == 'd')
        {
            if (len == 3 && strncasecmp(s, "dfn", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_DFN;
            }

            /* howcome 11/8/95: added upport for DIR */

            if (len == 3 && strncasecmp(s, "dir", len) == 0)
            {
                *token_class_p = EN_LIST;
                return TAG_UL;
            }


            if (len != 2)
                return 0;

            if (strncasecmp(s, "dl", len) == 0)
            {
                *token_class_p = EN_LIST;
                return TAG_DL;
            }

            if (strncasecmp(s, "dt", len) == 0)
            {
                *token_class_p = EL_DEFLIST;
                return TAG_DT;
            }

            if (strncasecmp(s, "dd", len) == 0)
            {
                *token_class_p = EL_DEFLIST;
                return TAG_DD;
            }
        }
        else if (c == 'e')
        {
            if (len == 2 && strncasecmp(s, "em", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_EM;
            }
        }
        else if (c == 'f')
        {
            if (len == 3 && strncasecmp(s, "fig", len) == 0)
            {
                *token_class_p = EN_BLOCK;
                return TAG_FIG;
            }
        }
        else if (c == 'h')
        {
            if (len == 4) {
                if (strncasecmp(s, "head", len) == 0)
                {
                    *token_class_p = EN_SETUP;
                    return TAG_HEAD;
                } 

                /* added by howcome 27/8/95 */

                else if (strncasecmp(s, "html", len) == 0)
                {
                    *token_class_p = EN_SETUP;
                    return TAG_HTML;
                }
            }

            if (len != 2)
                return 0;

            *token_class_p = EN_HEADER;
            c = TOLOWER(s[1]);

            switch (c)
            {
                case '1':
                    return TAG_H1;
                case '2':
                    return TAG_H2;
                case '3':
                    return TAG_H3;
                case '4':
                    return TAG_H4;
                case '5':
                    return TAG_H5;
                case '6':
                    return TAG_H6;
                case 'r':
                    *token_class_p = EN_BLOCK;
                    return TAG_HR;
            }
        }
        else if (c == 'i')
        {
            if (len == 1)
            {
                *token_class_p = EN_TEXT;
                return TAG_ITALIC;
            }

            if (len == 3 && strncasecmp(s, "img", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_IMG;
            }

            if (len == 5 && strncasecmp(s, "input", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_INPUT;
            }

            if (len == 7 && strncasecmp(s, "isindex", len) == 0)
            {
                *token_class_p = EN_SETUP;
                return TAG_ISINDEX;
            }
        }
        else if (c == 'k')
        {
            if (len == 3 && strncasecmp(s, "kbd", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_KBD;
            }
        }
        else if (c == 'l')
        {
            if (len == 2 && strncasecmp(s, "li", len) == 0)
            {
                *token_class_p = EN_LIST;
                return TAG_LI;
            }

            if (len == 4 && strncasecmp(s, "link", len) == 0)
            {
                *token_class_p = EN_SETUP;
                return TAG_LINK;
            }

        }
        else if (c == 'm')
        {
            if (len == 4 && strncasecmp(s, "math", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_MATH;
            }

            if (len == 6 && strncasecmp(s, "margin", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_MARGIN;
            }

            /* howcome 11/8/95: added MENU to be compatible with HTML2 */
            if (len == 4 && strncasecmp(s, "menu", len) == 0)
            {
                *token_class_p = EN_LIST;
                return TAG_UL;
            }


        }
        else if (c == 'n')
        {
            if (len == 4 && strncasecmp(s, "note", len) == 0)
            {
                *token_class_p = EN_BLOCK;
                return TAG_NOTE;
            }
        }
        else if (c == 'o')
        {
            if (len == 2 && strncasecmp(s, "ol", len) == 0)
            {
                *token_class_p = EN_LIST;
                return TAG_OL;
            }

            if (len == 6 && strncasecmp(s, "option", len) == 0)
            {
                *token_class_p = EN_TEXT;  /* kludge for error recovery */
                return TAG_OPTION;
            }
        }
        else if (c == 'p')
        {
            if (len == 1)
            {
                *token_class_p = EN_BLOCK;
                return TAG_P;
            }

            if (len == 3 && strncasecmp(s, "pre", len) == 0)
            {
                *token_class_p = EN_BLOCK;
                return TAG_PRE;
            }
        }
        else if (c == 'q')
        {
            if (len == 1)
            {
                *token_class_p = EN_TEXT;
                return TAG_Q;
            }

            if (len == 5 && strncasecmp(s, "quote", len) == 0)
            {
                *token_class_p = EN_BLOCK;
                return TAG_QUOTE;
            }
        }
        else if (c == 'r')
        {
            if (len == 7 && strncasecmp(s, "removed", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_REMOVED;
            }
        }
        else if (c == 's')
        {
            if (len == 1)
            {
                *token_class_p = EN_TEXT;
                return TAG_STRIKE;
            }

            if (len == 3 && strncasecmp(s, "sup", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_SUP;
            }

            if (len == 3 && strncasecmp(s, "sub", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_SUB;
            }

            if (len == 4 && strncasecmp(s, "samp", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_SAMP;
            }

            if (len == 5 && strncasecmp(s, "small", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_SMALL;
            }

            if (len == 6 && strncasecmp(s, "strong", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_STRONG;
            }

            if (len == 6 && strncasecmp(s, "select", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_SELECT;
            }

            if (len == 6 && strncasecmp(s, "strike", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_STRIKE;
            }

            /* howcome 26/2/95 */

            if (len == 5 && strncasecmp(s, "style", len) == 0)
            {
                *token_class_p = EN_SETUP;
                return TAG_STYLE;
            }

        }
        else if (c == 't')
        {
            if (len == 5 && strncasecmp(s, "title", len) == 0)
            {
                *token_class_p = EN_SETUP;
                return TAG_TITLE;
            }

            if (len == 2 && strncasecmp(s, "tt", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_TT;
            }

            if (len == 2 && strncasecmp(s, "tr", len) == 0)
            {
                *token_class_p = EN_TABLE;
                return TAG_TR;
            }

            if (len == 2 && strncasecmp(s, "th", len) == 0)
            {
                *token_class_p = EN_TABLE;
                return TAG_TH;
            }

            if (len == 2 && strncasecmp(s, "td", len) == 0)
            {
                *token_class_p = EN_TABLE;
                return TAG_TD;
            }

            if (len == 5 && strncasecmp(s, "table", len) == 0)
            {
                *token_class_p = EN_BLOCK;
                return TAG_TABLE;
            }

            if (len == 8 && strncasecmp(s, "textarea", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_TEXTAREA;
            }
        }
        else if (c == 'u')
        {
            if (len == 1)
            {
                *token_class_p = EN_TEXT;
                return TAG_UNDERLINE;
            }

            if (len == 2 && strncasecmp(s, "ul", len) == 0)
            {
                *token_class_p = EN_LIST;
                return TAG_UL;
            }
        }
        else if (c == 'v')
        {
            if (len == 3 && strncasecmp(s, "var", len) == 0)
            {
                *token_class_p = EN_TEXT;
                return TAG_VAR;
            }
        }
        else if (c == 'x')
        {
            if (len == 3 && strncasecmp(s, "xmp", len) == 0)
            {
                *token_class_p = EN_BLOCK;
                return TAG_PRE;
            }
        }
    }

    *token_class_p = EN_UNKNOWN;
    return UNKNOWN; /* unknown tag */
}


Bool StyleParseColour(const char *s, StyleType* type_p, long *value_p)
{
 if (ParseColour(s, value_p))
   {
    (*type_p) = ST_RGB;
    return True;
   }
  else
   return False;
}


Bool StyleParseLength(char *s, Bool allow_percent,
		      StyleType* type_p, long *value_p)
{
 if (strstr(s, "px"))
   {
    *type_p = ST_PX;
    *value_p = (long)atoi(s);
    return True;
   }
  else
   if (strstr(s, "pt"))
     {
      *type_p = ST_PT;
      *value_p = (long)(atoi(s));
      return True;
     }
    else
     if (strstr(s, "em"))
       {
        *type_p = ST_EM;
        *value_p = (long)atoi(s);
        return True;
       }
      else
       if (allow_percent && (strstr(s,"%")))
	 {
	  *type_p = ST_PERCENT;
	  *value_p = (long)atoi(s);
	  return True;
	 }

 return False;
}


Bool StyleParseNumber(char *s, StyleType *type_p, long *value_p)
{
 double d = atof(s);

 *type_p = ST_FLOAT;
 *value_p = (long)(d * 1000.0);
 return True;
}



StyleSelector* ParseSelector(char* str, HTList* selectors)
{
 char *p;
 char *elem_marker = NULL, *subelem_marker = NULL;
 StyleSelector *selector = NULL, *prev_selector = NULL;
 int token_class;
 int i;
 int element;
 char *class;

 p = str_tok(str, " \t", &elem_marker);
 if(!p) return NULL;

 do
   {
    if (*p == '.')
      {
       subelem_marker = p + 1;
       class = str_tok(NULL," \t",&subelem_marker);
       for (i = 1; i <= TAG_FORM; i++)
	 {
	  selector = NewStyleSelector();
	  selector->ss.element = i;
	  if (class && *class) selector->ss.class = strdup(class);
	  selector->ancestor = prev_selector;
	  prev_selector = selector;
	  HTList_addObject(selectors, (void *)selector);
	 }

       return NULL;
      }
     else
      {
       p = str_tok(p, ".", &subelem_marker);
       element = element_enum(p, &token_class);

       if (element == UNKNOWN) return NULL;

       class = str_tok(NULL, " \t", &subelem_marker);
       selector = NewStyleSelector();
       selector->ss.element = element;
       if (class && *class) selector->ss.class = strdup(class);
       selector->ancestor = prev_selector;
       prev_selector = selector;
      }
   }
 while ( (p = str_tok(NULL, "() \t", &elem_marker)) );

 return selector;
}


BG_Style* StyleGetBackground(Doc* theDoc, void *value_p, char *str)
{
#ifdef ARENA_DEBUG	/* QingLong.27-03-97 */
 char Iam[] = "StyleGetBackground";
#endif


#ifdef ARENA_DEBUG
 if (STYLE_TRACE)
   if (value_p == NULL)
     Arena_TracePrint(Iam, " ERROR? The value_p is NULL.\n");
#endif

 if (strcmp(str, ":") == 0)
   {
    return (BG_Style*)value_p;
   }
  else
   {
    Bool GotStyleSuccessfully = False, CreatedNewBG_Style, is_numeric;
    char *tmp, *str_copy, *str_copy_p;
    Image* image;
    BG_Style* bg_style;


    if (value_p)
      {
       bg_style = (BG_Style*)value_p;
       CreatedNewBG_Style = False;
      }
     else
      {
       bg_style = (BG_Style*)Arena_CAlloc(1, sizeof(BG_Style), False);
       bg_style->flag = S_BACKGROUND_X_REPEAT | S_BACKGROUND_Y_REPEAT;
       bg_style->x_pos = 50;
       bg_style->y_pos = 50;
       CreatedNewBG_Style = True;
#ifdef ARENA_DEBUG
       if (STYLE_TRACE && VERBOSE_TRACE)
	 Arena_TracePrint(Iam,
			  "Created new BG_Style "POINTER_FORMAT".\n",
			  bg_style);
#endif
      }


    str_copy_p = unquote(str_copy = strdup(str));

    /* try looking up the name */
    if (strcasecmp(str_copy_p, "fixed") == 0)
      {
       bg_style->flag |= S_BACKGROUND_FIXED;
       GotStyleSuccessfully = True;
      }
     else
      {
       if (strcasecmp(str_copy_p, "repeat-x") == 0)
	 {
	  bg_style->flag &= ~S_BACKGROUND_Y_REPEAT;
	  GotStyleSuccessfully = True;
	 }
        else
	 {
	  if (strcasecmp(str_copy_p, "repeat-y") == 0)
	    {
	     bg_style->flag &= ~S_BACKGROUND_X_REPEAT;
	     GotStyleSuccessfully = True;
	    }
	   else
	    {
	     if (strcasecmp(str_copy_p, "no-repeat") == 0)
	       {
		bg_style->flag &= ~(S_BACKGROUND_X_REPEAT |
				    S_BACKGROUND_Y_REPEAT);
		GotStyleSuccessfully = True;
	       }
	      else
	       {
		if (strncmp(str_copy_p, "url(", 4) == 0)
		  {
		   if ((image = GetImage(str_copy_p + 4,
					 Arena_StrLen(str_copy_p) - 5,
					 theDoc->pending_reload))) 
		     {
		      bg_style->image = image;
		      bg_style->flag |= S_BACKGROUND_IMAGE;
		      GotStyleSuccessfully = True;
		     }
		  }
		 else
		  {
		   is_numeric = True;
		   tmp = str_copy_p;
		   while (*tmp && is_numeric)
		     {
		      is_numeric = (isdigit(*tmp) || (*tmp == '%'));
		      tmp++;
		     }

		   if (is_numeric)
		     {
		      if (bg_style->flag & S_BACKGROUND_ORIGIN)
			{
			 bg_style->y_pos = atoi(str_copy_p);
			}
		       else
			{
			 bg_style->flag |= S_BACKGROUND_ORIGIN;
			 bg_style->x_pos = atoi(str_copy_p);
			 bg_style->y_pos = bg_style->x_pos;
			}

		      GotStyleSuccessfully = True;
		     }
		    else
		     {
		      StyleType type;
		      long colour;

		      if (StyleParseColour(str_copy_p, &type, &colour))
			{
			 bg_style->r = (colour >> 16) & 0xFF;
			 bg_style->g = (colour >>  8) & 0xFF;
			 bg_style->b = (colour      ) & 0xFF;
			 bg_style->flag |= S_BACKGROUND_COLOUR;
			 GotStyleSuccessfully = True;
#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
			 if (STYLE_TRACE && VERBOSE_TRACE)
			   Arena_TracePrint(Iam,
					    " %x %x %x -> 0x%x\n",
					    bg_style->r,
					    bg_style->g,
					    bg_style->b,
					    colour);
#endif
			}
		       else
			{
			 if ((image = GetImage(str_copy_p,
					       Arena_StrLen(str_copy_p),
					       theDoc->pending_reload)))
			   {
			    bg_style->image = image;
			    bg_style->flag |= S_BACKGROUND_IMAGE;
			    GotStyleSuccessfully = True;
			   }
			}
		     }
		  }
	       }
	    }
	 }
      }

    Free(str_copy);
 
    if (!GotStyleSuccessfully)
      {
#ifdef ARENA_DEBUG
       if (STYLE_TRACE && VERBOSE_TRACE)
	 Arena_TracePrint(Iam, " Failed to parse bg style assignment.\n");
#endif
       if (CreatedNewBG_Style)
	 {
#ifdef ARENA_DEBUG
	  if (STYLE_TRACE && VERBOSE_TRACE)
	    Arena_TracePrint(Iam,
			     " Freeing BG_Style "POINTER_FORMAT".\n",
			     bg_style);
#endif
       Free(bg_style);
	 }
      }

    return (bg_style);
   }
}


void StyleAddValue(HTList* values,
		   StyleType type, long value, StyleInternalProperty iproperty)
{
 if (values)
   {
    StyleValue *sv_p = NewStyleValue();

    sv_p->type = type;
    sv_p->value = value;
    sv_p->iproperty = iproperty;
    HTList_addObject(values, (void *)sv_p);
   }
}



/* ?? */

HTList* ParseAssignment(Doc* theDoc, char *s, int *weight_p)
{
 StyleExternalProperty eproperty;
 char *str;
 char *elem_str, *value_str;
 HTList *values = NULL;
 StyleType type;
 long value;
 long bg_value_save;   /* A value got in a previous ArenaSE_BACKGROUND */
#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
 char Iam[] = "ParseAssignment";
#endif


 if (s == NULL) return NULL;

 bg_value_save = 0;

 str = str_tok(s," \t:", &elem_str);
 if (!str) return False;

 eproperty = str2eproperty(str);

 if (eproperty == ArenaSE_UNKNOWN) return NULL;

#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
 if (STYLE_TRACE & VERBOSE_TRACE)
   Arena_TracePrint(Iam, " eproperty = %d.\n", eproperty);
#endif

 /* get everything up to '!' */

 str = str_tok(NULL, "!", &elem_str);
 if (str == NULL) return NULL;

 str = str_tok(str, ",\t ", &value_str);

 values = HTList_new();
 while (str)
   {
    switch (eproperty)
      {
        case ArenaSE_FONT_SIZE:
            if (strcasecmp(str,"xx-large") == 0) {
                StyleAddValue(values, ST_PX, (long) 21, ArenaSI_FONT_SIZE);
            } else if (strcasecmp(str,"x-large") == 0) {
                StyleAddValue(values, ST_PX, (long) 17, ArenaSI_FONT_SIZE);
            } else if (strcasecmp(str,"large") == 0) {
                StyleAddValue(values, ST_PX, (long)14, ArenaSI_FONT_SIZE);
            } else if (strcasecmp(str,"medium") == 0) {
                StyleAddValue(values, ST_PX, (long)12, ArenaSI_FONT_SIZE);
            } else if (strcasecmp(str,"small") == 0) {
                StyleAddValue(values, ST_PX, (long)10, ArenaSI_FONT_SIZE);
            } else if (strcasecmp(str,"x-small") == 0) {
                StyleAddValue(values, ST_PX, (long)9, ArenaSI_FONT_SIZE);
            } else if (strcasecmp(str,"xx-small") == 0) {
                StyleAddValue(values, ST_PX, (long)8, ArenaSI_FONT_SIZE);
            } else {
                if (StyleParseLength(str, True, &type, &value))
                    StyleAddValue(values, type, value, ArenaSI_FONT_SIZE);
            }
            break;

        case ArenaSE_FONT_FAMILY:
            StyleAddValue(values, ST_STR, (long)strdup(str), ArenaSI_FONT_FAMILY);
            break;

        case ArenaSE_FONT_WEIGHT:

            if (strcasecmp(str,"extra-light") == 0) {
                StyleAddValue(values, ST_FLAG, SV_FONT_WEIGHT_EXTRA_LIGHT, ArenaSI_FONT_WEIGHT);
            }
            else if (strcasecmp(str,"light") == 0) {
                StyleAddValue(values, ST_FLAG, SV_FONT_WEIGHT_LIGHT, ArenaSI_FONT_WEIGHT);
            }
            else if (strcasecmp(str,"medium") == 0) {
                StyleAddValue(values, ST_FLAG, SV_FONT_WEIGHT_MEDIUM, ArenaSI_FONT_WEIGHT);
            }
            else if (strcasecmp(str,"demi-bold") == 0) {
                StyleAddValue(values, ST_FLAG, SV_FONT_WEIGHT_DEMI_BOLD, ArenaSI_FONT_WEIGHT);
            }
            else if (strcasecmp(str,"bold") == 0) {
                StyleAddValue(values, ST_FLAG, SV_FONT_WEIGHT_BOLD, ArenaSI_FONT_WEIGHT);
            }
            else if (strcasecmp(str,"extra-bold") == 0) {
                StyleAddValue(values, ST_FLAG, SV_FONT_WEIGHT_EXTRA_BOLD, ArenaSI_FONT_WEIGHT);
            } else
                if (StyleParseNumber(str, &type, &value))
                    StyleAddValue(values, type, value, ArenaSI_FONT_WEIGHT);
            break;

        case ArenaSE_FONT_STYLE:

            if (strncasecmp(str,"italic", 6) == 0) { /* italics */
                StyleAddValue(values, ST_FLAG, SV_FONT_STYLE_ITALIC, ArenaSI_FONT_STYLE_SLANT);
            } else if (strcasecmp(str,"roman") == 0) {
                StyleAddValue(values, ST_FLAG, SV_FONT_STYLE_ROMAN, ArenaSI_FONT_STYLE_SLANT);
            } else if (strcasecmp(str,"small-caps") == 0) {
                StyleAddValue(values, ST_FLAG, SV_FONT_STYLE_SMALL_CAPS, ArenaSI_FONT_STYLE_SMALL_CAPS);
            }
            break;

        case ArenaSE_LINE_HEIGHT:
            if (StyleParseLength(str, True, &type, &value))
                StyleAddValue(values, type, value, ArenaSI_LINE_HEIGHT);
            break;

        case ArenaSE_FONT:
	  break;

        case ArenaSE_COLOUR:
	  /* try looking up the name */
	  if (StyleParseColour(str, &type, &value))
	    StyleAddValue(values, type, value, ArenaSI_COLOUR);
	  /* else  We should provide a "suitable" default value here
	   * - rhw  2-April-1997
	   */
	  break;	    

        case ArenaSE_BACKGROUND:
	  {
	   long tmp_value = (long)StyleGetBackground(theDoc,
						     (void *)bg_value_save,
						     str);

	   /* Check if got new style successfully and use it, else skip it. */
	   if ((bg_value_save == 0) && (tmp_value)) bg_value_save = tmp_value;
	  }
	  break;

        case ArenaSE_BG_BLEND_DIRECTION:
	  break;

        case ArenaSE_BG_STYLE:
            break;

        case ArenaSE_BG_POSITION:
            break;

        case ArenaSE_WORD_SPACING:
        case ArenaSE_LETTER_SPACING:
            if (!strcasecmp("normal", str)) {
                StyleAddValue(values, ST_FLAG, S_NORMAL, ArenaSI_LETTER_SPACING);
            } else 
                if (StyleParseLength(str, False, &type, &value))
                    StyleAddValue(values, type, value, ArenaSI_LETTER_SPACING);
            break;

        case ArenaSE_TEXT_DECORATION:
            if (! strcasecmp(str,"underline")) {
                StyleAddValue(values, ST_FLAG, SV_TEXT_DECORATION_UNDERLINE, ArenaSI_TEXT_DECORATION);
            } else if (! strcasecmp(str,"overline")) {
                StyleAddValue(values, ST_FLAG, SV_TEXT_DECORATION_OVERLINE, ArenaSI_TEXT_DECORATION);
            } else if (! strcasecmp(str,"line-through")) {
                StyleAddValue(values, ST_FLAG, SV_TEXT_DECORATION_LINE_THROUGH, ArenaSI_TEXT_DECORATION);
            } else if (! strcasecmp(str,"blink")) {
                StyleAddValue(values, ST_FLAG, SV_TEXT_DECORATION_BLINK, ArenaSI_TEXT_DECORATION);
            }
            break;

        case ArenaSE_VERTICAL_ALIGN:
            break;

        case ArenaSE_TEXT_TRANSFORM:
            break;

        case ArenaSE_TEXT_ALIGN:
            if (strcasecmp(str,"left") == 0) {	
                StyleAddValue(values, ST_FLAG, ALIGN_LEFT, ArenaSI_TEXT_ALIGN);
            } else if (strcasecmp(str,"center") == 0) {	
                StyleAddValue(values, ST_FLAG, ALIGN_CENTER, ArenaSI_TEXT_ALIGN);
            } else if (strcasecmp(str,"right") == 0) {	
                StyleAddValue(values, ST_FLAG, ALIGN_RIGHT, ArenaSI_TEXT_ALIGN);
            } else if (strcasecmp(str,"justify") == 0) {
                StyleAddValue(values, ST_FLAG, ALIGN_JUSTIFY, ArenaSI_TEXT_ALIGN);
            }
            break;

        case ArenaSE_TEXT_INDENT:
	  if (StyleParseLength(str, True, &type, &value))
	    StyleAddValue(values, type, value, ArenaSI_TEXT_INDENT);
	  break;

        case ArenaSE_PADDING:
	  if (StyleParseLength(str, True, &type, &value))
	    {
	     StyleAddValue(values, type, value, ArenaSI_PADDING_LEFT);
	     StyleAddValue(values, type, value, ArenaSI_PADDING_RIGHT);
	     StyleAddValue(values, type, value, ArenaSI_PADDING_TOP);
	     StyleAddValue(values, type, value, ArenaSI_PADDING_BOTTOM);
            }
	  break;

        case ArenaSE_MARGIN_LEFT:
        case ArenaSE_MARGIN_RIGHT:
        case ArenaSE_MARGIN_TOP:
        case ArenaSE_MARGIN_BOTTOM:
	  {
	   StyleInternalProperty Iproperty;

	   switch (eproperty)
	     {
	      case ArenaSE_MARGIN_LEFT:
		Iproperty = ArenaSI_MARGIN_LEFT;
		break;

	      case ArenaSE_MARGIN_RIGHT:
		Iproperty = ArenaSI_MARGIN_RIGHT;
		break;

	      case ArenaSE_MARGIN_TOP:
		Iproperty = ArenaSI_MARGIN_TOP;
		break;

	      case ArenaSE_MARGIN_BOTTOM:
		Iproperty = ArenaSI_MARGIN_BOTTOM;
		break;

	      default:
		Iproperty = ArenaSI_UNKNOWN;
		break;
	     }

	   if (strcasecmp(str, "auto") == 0)
	      {
               StyleAddValue(values, ST_FLAG, SV_MARGIN_AUTO, Iproperty);
	      }
	    else
	     if (StyleParseLength(str, True, &type, &value))
	       StyleAddValue(values, type, value, Iproperty);
	  }
	  break;

        case ArenaSE_MARGIN:
            break;

        case ArenaSE_WIDTH:
            break;

        case ArenaSE_HEIGHT:
            break;

        case ArenaSE_FLOAT:
            break;

        case ArenaSE_CLEAR:
            break;

        case ArenaSE_PACK:
            break;

            /* border style */

        case ArenaSE_LIST_STYLE:
            break;

        case ArenaSE_MAGNIFICATION:
            break;

        case ArenaSE_WHITE_SPACE:
            break;

        default:
#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
            if (STYLE_TRACE)
	      Arena_TracePrint(Iam,
			      "Unknown property enum `%d'\n",
			      eproperty);
#endif
            break;
        }
    /* End ``switch (eproperty)'' */

    str = str_tok(NULL, ",\t ", &value_str);
   }
 /* End ``while (str)'' */

 /* If bg_value_save turns out to be null,
  * then we have no success in backround attributes parsing at all.
  */
 if ((eproperty == ArenaSE_BACKGROUND) && (bg_value_save))
   {
    BG_Style* theBGstyle = (BG_Style*)bg_value_save;

    StyleAddValue(values, ST_BGSTYLE, bg_value_save, ArenaSI_BACKGROUND);
    if (theBGstyle->flag & S_BACKGROUND_COLOUR)
      {
       Byte red = theBGstyle->r, green = theBGstyle->g, blue = theBGstyle->b;

       StyleAddValue(values, ST_RGB, (long)((red << 16) | (green << 8) | blue),
		     ArenaSI_BACKGROUND_COLOUR);
      }
   }

 *weight_p = SW_NORMAL;

 str = str_tok(NULL, "! \t;", &elem_str);
 if (str)
   {
    if (strcasecmp(str, "important") == 0)
      {
       *weight_p = SW_IMPORTANT;
      }
     else
      if (strcasecmp(str, "legal") == 0)
	{
	 *weight_p = SW_LEGAL;
        }
   }

 return values;
}


/*
 * StyleChew is the main entry point to the CSS parser. 
 *
 * *sheet is a pointer to a StyleSheet structure
 *        into which the assignments will be added.
 *
 * *str is a pointer to a string containing CSS syntax.
 *      The pointer will not be freed,a dn the content will remain unchanged.
 *
 * base_weight identifies the source of the style sheet, author or reader.
 */
void StyleChew(Doc* theDoc, StyleSheet* sheet, char *str, Byte origin)
{
 if ((str) && (*str) && (sheet))
   {
    char *p, *s;
    char* sheet_marker = NULL;
#ifdef ARENA_DEBUG
    char Iam[] = "StyleChew";
#endif


#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
    if (STYLE_TRACE & VERBOSE_TRACE) Arena_TracePrint(Iam, ")\n");
#endif

    /* h1, p: font-family = helvetica times, font-size = 12pt; */

    s = strdup(str);  /* we will modify the string so we make a copy */

    {
     int c;

     for (p = s; (c = (unsigned char)*p); p++)
       if (c != '\t' && isspace(c)) *p = ' ';
    }

    if ((p = str_tok(s, "}", &sheet_marker)))
      {
       char *address_marker = NULL, *rule_marker = NULL;
       int weight;
       long value;
       void *bg_save;
       HTList *l, *selectors, *values;
       StyleSelector *selector;
#if 0
       int bg_weight = 0;
#endif
#ifdef ARENA_DEBUG
       StyleSelector* sc = NULL;
#endif


       do
	 {
	  p = str_tok(p, "{", &rule_marker);

#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
	  if (STYLE_TRACE) Arena_TracePrint(Iam, "\n\t(address) %s -> ", p);
#endif

	  selectors = HTList_new();

	  {
	   StyleSelector* sel;

	   p = str_tok(p, ",", &address_marker);
	   do
	     {
	      if ((sel = ParseSelector(p, selectors)))
		{
#ifdef ARENA_DEBUG
		 if (STYLE_TRACE)
		   {
		    for ( ; (sc); sc = sc->ancestor)
		      Arena_DebugPrint(" %d-", sc->ss.element);
		   }
#endif
		 HTList_addObject(selectors, (void *)sel);
		}
	     }
	   while ((p = str_tok(NULL, ",", &address_marker)));
	  }

	  /* now we know what elements we are addressing */

	  p = str_tok(NULL, ";}", &rule_marker);
	  bg_save = NULL;
	  do
	    {
#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
	     if (STYLE_TRACE) if (p) Arena_DebugPrint(" (assignment) %s ", p);
#endif

	     value = (long)bg_save;  /* this hack is only for keeping bg_style
					during the parsing of one element
					--Spif 23-Nov-95 */

	     if ((values = ParseAssignment(theDoc, p, &weight)) != NULL)
	       {
#if 0
		if (property == S_BACKGROUND)
		  {
		   bg_save = (void *)value;
		   bg_weight = weight;
		  }
		 else
#endif
		  {
		   l = selectors;
		   while ((selector = (StyleSelector*)HTList_nextObject(l))) 
		     {
#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
		      if (STYLE_TRACE && selector)
			{
			 Arena_DebugPrint("\n\tnext selector ");
			 for (sc = selector; sc; sc = sc->ancestor)
			   {
			    Arena_DebugPrint(" %d-", sc->ss.element);
			   }

			 Arena_DebugPrint("\n");
			}
#endif
		      StyleAddRule(sheet, selector, values, origin, weight);
		     }
		  }
	       }
	    }
	  while ((p = str_tok(NULL, ";}", &rule_marker)));

#if 0
	  if (bg_save)
	    {
	     l = selectors;
	     while ((selector = (StyleSelector*)HTList_nextObject(l))) 
	       {
#  ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
		if (STYLE_TRACE) 
		  {
		   Arena_DebugPrint("\n\tnext selector ");
		   do
		     {
		      Arena_DebugPrint(" %d-", sc->ss.element);
		     }
		   while ((sc = sc->ancestor));

		   Arena_DebugPrint("\n");
		  }
#  endif
		StyleAddRule(sheet, selector,
			     S_BACKGROUND, (long)bg_save, bg_weight);
	       }
	    }
#endif

#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
	  if (STYLE_TRACE) Arena_DebugPrint("\n");
#endif

#if 0
	  l = selectors;
	  while ((selector = (StyleSelector*)HTList_nextObject(l)))
	    {
#  ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
	     if (STYLE_TRACE && VERBOSE_TRACE)
	       Arena_TracePrint(Iam,
				" freeing selector "POINTER_FORMAT".\n",
				selector);
#  endif
	     FreeStyleSelector(selector);
	    }
	  HTList_delete(selectors);
#endif

	 }
       while ((p = str_tok(NULL, "}", &sheet_marker))); 

       Free(s);
#ifdef ARENA_DEBUG	/* QingLong.14-12-96 */
       StyleSane(sheet);
#endif
      }
   }
}


/* StyleGetInit returns an initialized style sheet,
 * typically the application's default.
 */
StyleSheet* StyleGetInit()
{
 StyleSheet* sheet = NewStyleSheet();


 /* set toplevel fallback values */

 StyleAddSimpleRule(sheet, TAG_HTML,
		    ArenaSI_FONT_FAMILY,           ST_STR,     S_FALLBACK,
		    (long)"helvetica");
 StyleAddSimpleRule(sheet, TAG_HTML,
		    ArenaSI_FONT_SIZE,             ST_PX,      S_FALLBACK,
		    (long)12);
 StyleAddSimpleRule(sheet, TAG_HTML,
		    ArenaSI_FONT_WEIGHT,           ST_FLAG,    S_FALLBACK,
		    SV_FONT_WEIGHT_MEDIUM);
 StyleAddSimpleRule(sheet, TAG_HTML,
		    ArenaSI_FONT_STYLE_SLANT,      ST_FLAG,    S_FALLBACK,
		    SV_FONT_STYLE_NORMAL);
 StyleAddSimpleRule(sheet, TAG_HTML,
		    ArenaSI_FONT_STYLE_SMALL_CAPS, ST_FLAG,    S_FALLBACK,
		    SV_FONT_STYLE_NORMAL);

 StyleAddSimpleRule(sheet, TAG_HTML,
		    ArenaSI_COLOUR,                 ST_RGB,     S_FALLBACK,
		    (long)(100 << 0));  /* i.e. a dark blue */

 {
  BG_Style* arena_bg_style;


  arena_bg_style = (BG_Style*)Arena_CAlloc(1, sizeof(BG_Style), False);

  if (UsePaper)
    {
     arena_bg_style->flag  = (S_BACKGROUND_X_REPEAT | S_BACKGROUND_Y_REPEAT |
			      S_BACKGROUND_ORIGIN |
			      S_BACKGROUND_IMAGE);
     arena_bg_style->x_pos = 0;
     arena_bg_style->y_pos = 0;
     arena_bg_style->image = arena_bg_image;
    }
   else
    {
     arena_bg_style->flag  = S_BACKGROUND_COLOUR;			     
     arena_bg_style->r = 220;
     arena_bg_style->g = 209;
     arena_bg_style->b = 186;
    }

    StyleAddSimpleRule(sheet, TAG_HTML,
		       ArenaSI_BACKGROUND,         ST_BGSTYLE, S_FALLBACK,
		       (long)arena_bg_style);
   }

 StyleAddSimpleRule(sheet, TAG_HTML,
		    ArenaSI_BACKGROUND_COLOUR,      ST_RGB,     S_FALLBACK,
		    (long)(windowColour));

 StyleAddSimpleRule(sheet, TAG_HTML,
		    ArenaSI_MARGIN_LEFT,    ST_PX,   S_FALLBACK, 6);
 StyleAddSimpleRule(sheet, TAG_HTML,
		    ArenaSI_MARGIN_RIGHT,   ST_PX,   S_FALLBACK, 6);
 StyleAddSimpleRule(sheet, TAG_HTML,
		    ArenaSI_MARGIN_TOP,     ST_PX,   S_FALLBACK, 0);
 StyleAddSimpleRule(sheet, TAG_HTML,
		    ArenaSI_MARGIN_BOTTOM,  ST_PX,   S_FALLBACK, 0);

 StyleAddSimpleRule(sheet, TAG_HTML,
		    ArenaSI_PADDING_LEFT,   ST_PX,   S_FALLBACK, 0);
 StyleAddSimpleRule(sheet, TAG_HTML,
		    ArenaSI_PADDING_RIGHT,  ST_PX,   S_FALLBACK, 0);
 StyleAddSimpleRule(sheet, TAG_HTML,
		    ArenaSI_PADDING_TOP,    ST_PX,   S_FALLBACK, 6);
 StyleAddSimpleRule(sheet, TAG_HTML,
		    ArenaSI_PADDING_BOTTOM, ST_PX,   S_FALLBACK, 6);

 StyleAddSimpleRule(sheet, TAG_HTML,
		    ArenaSI_TEXT_ALIGN,   ST_FLAG,   S_FALLBACK, ALIGN_LEFT);

 /* set per-tag fallback values -- exception to the normal fallback values */

 StyleAddSimpleRule(sheet, TAG_H1, ArenaSI_FONT_FAMILY,    ST_STR,    S_FALLBACK, (long)"charter"); 
 StyleAddSimpleRule(sheet, TAG_H1, ArenaSI_FONT_SIZE,      ST_PX,	    S_FALLBACK, 24);
 StyleAddSimpleRule(sheet, TAG_H1, ArenaSI_FONT_WEIGHT,    ST_FLAG,   S_FALLBACK, SV_FONT_WEIGHT_BOLD);
 StyleAddSimpleRule(sheet, TAG_H1, ArenaSI_MARGIN_LEFT,    ST_PX,     S_FALLBACK, 5);

 StyleAddSimpleRule(sheet, TAG_H2, ArenaSI_FONT_FAMILY,    ST_STR,    S_FALLBACK, (long)"charter"); 
 StyleAddSimpleRule(sheet, TAG_H2, ArenaSI_FONT_SIZE,      ST_PX,     S_FALLBACK, 18);
 StyleAddSimpleRule(sheet, TAG_H2, ArenaSI_FONT_WEIGHT,    ST_FLAG,   S_FALLBACK, SV_FONT_WEIGHT_BOLD);
 StyleAddSimpleRule(sheet, TAG_H2, ArenaSI_MARGIN_LEFT,    ST_PX,     S_FALLBACK, 5);

 StyleAddSimpleRule(sheet, TAG_H3, ArenaSI_FONT_FAMILY,    ST_STR,    S_FALLBACK, (long) "times"); 
 StyleAddSimpleRule(sheet, TAG_H3, ArenaSI_FONT_SIZE,      ST_PX,	    S_FALLBACK, 14);
 StyleAddSimpleRule(sheet, TAG_H2, ArenaSI_FONT_WEIGHT,    ST_FLAG,   S_FALLBACK,	SV_FONT_WEIGHT_BOLD);
 StyleAddSimpleRule(sheet, TAG_H3, ArenaSI_MARGIN_LEFT,    ST_PX,     S_FALLBACK, 5);

 StyleAddSimpleRule(sheet, TAG_H4, ArenaSI_FONT_FAMILY,    ST_STR,    S_FALLBACK, (long)"charter");
 StyleAddSimpleRule(sheet, TAG_H4, ArenaSI_FONT_SIZE,      ST_FLAG,   S_FALLBACK, 12);
 StyleAddSimpleRule(sheet, TAG_H4, ArenaSI_FONT_WEIGHT,    ST_FLAG,   S_FALLBACK, SV_FONT_WEIGHT_BOLD);
 StyleAddSimpleRule(sheet, TAG_H4, ArenaSI_MARGIN_LEFT,    ST_PX,	    S_FALLBACK, 5);

 StyleAddSimpleRule(sheet, TAG_H5, ArenaSI_FONT_FAMILY,    ST_STR,    S_FALLBACK, (long) "charter");
 StyleAddSimpleRule(sheet, TAG_H5, ArenaSI_FONT_SIZE,      ST_PX,     S_FALLBACK, 12);
 StyleAddSimpleRule(sheet, TAG_H5, ArenaSI_FONT_WEIGHT,    ST_FLAG,   S_FALLBACK, SV_FONT_WEIGHT_BOLD);
 StyleAddSimpleRule(sheet, TAG_H5, ArenaSI_MARGIN_LEFT,    ST_PX,     S_FALLBACK, 5);

 StyleAddSimpleRule(sheet, TAG_H6, ArenaSI_FONT_FAMILY,    ST_STR,    S_FALLBACK, (long)"charter");
 StyleAddSimpleRule(sheet, TAG_H6, ArenaSI_FONT_SIZE,      ST_PX,     S_FALLBACK, 12);
 StyleAddSimpleRule(sheet, TAG_H6, ArenaSI_FONT_WEIGHT,    ST_FLAG,   S_FALLBACK, SV_FONT_WEIGHT_BOLD);
 StyleAddSimpleRule(sheet, TAG_H6, ArenaSI_MARGIN_LEFT,    ST_FLAG,   S_FALLBACK, 5);

 /*    StyleAddSimpleRule(sheet, TAG_LI, S_MARGIN_LEFT, 	15, 			S_FALLBACK);*/
 
 StyleAddSimpleRule(sheet, TAG_DL, ArenaSI_MARGIN_LEFT,    ST_PX,     S_FALLBACK, 15);
 StyleAddSimpleRule(sheet, TAG_UL, ArenaSI_MARGIN_LEFT,    ST_PX,     S_FALLBACK, 15);
 StyleAddSimpleRule(sheet, TAG_OL, ArenaSI_MARGIN_LEFT,    ST_PX,	    S_FALLBACK, 15);

 StyleAddSimpleRule(sheet, TAG_DT, ArenaSI_FONT_WEIGHT,    ST_FLAG,   S_FALLBACK, SV_FONT_WEIGHT_BOLD);
 StyleAddSimpleRule(sheet, TAG_DD, ArenaSI_FONT_WEIGHT,    ST_FLAG,   S_FALLBACK, SV_FONT_WEIGHT_MEDIUM);

 StyleAddSimpleRule(sheet, TAG_ADDRESS, ArenaSI_TEXT_ALIGN, ST_FLAG,  S_FALLBACK, ALIGN_RIGHT);

 StyleAddSimpleRule(sheet, TAG_MATH, ArenaSI_FONT_SIZE,   ST_PX,      S_FALLBACK, 14);
 StyleAddSimpleRule(sheet, TAG_SMALL, ArenaSI_FONT_SIZE,  ST_PX,      S_FALLBACK, 8);
 StyleAddSimpleRule(sheet, TAG_SUB, 	ArenaSI_FONT_SIZE,   ST_PX,      S_FALLBACK, 8);
 StyleAddSimpleRule(sheet, TAG_SUP, 	ArenaSI_FONT_SIZE,   ST_PX,      S_FALLBACK, 8);

 StyleAddSimpleRule(sheet, TAG_STRONG, ArenaSI_FONT_WEIGHT,      ST_FLAG,  S_FALLBACK, SV_FONT_WEIGHT_BOLD);
 StyleAddSimpleRule(sheet, TAG_BOLD, ArenaSI_FONT_WEIGHT,        ST_FLAG,  S_FALLBACK, SV_FONT_WEIGHT_BOLD);

 StyleAddSimpleRule(sheet, TAG_EM, ArenaSI_FONT_STYLE_SLANT,     ST_FLAG,    S_FALLBACK, SV_FONT_STYLE_ITALIC);
 StyleAddSimpleRule(sheet, TAG_ITALIC, ArenaSI_FONT_STYLE_SLANT, ST_FLAG,    S_FALLBACK, SV_FONT_STYLE_ITALIC);

 StyleAddSimpleRule(sheet, TAG_PRE, ArenaSI_FONT_FAMILY,  ST_STR,    S_FALLBACK, (long)"courier");
 StyleAddSimpleRule(sheet, TAG_TT, ArenaSI_FONT_FAMILY,   ST_STR,    S_FALLBACK, (long)"courier");
 StyleAddSimpleRule(sheet, TAG_CODE, ArenaSI_FONT_FAMILY,	ST_STR,    S_FALLBACK, (long)"courier");
    /* added UV 1997-04-07 */
 StyleAddSimpleRule(sheet, TAG_HTML_SOURCE, ArenaSI_FONT_FAMILY, ST_STR, S_FALLBACK, (long)"courier");


 /* set default style for TAG_ABSTRACT */

 StyleAddSimpleRule(sheet, TAG_ABSTRACT, ArenaSI_FONT_STYLE_SLANT, ST_FLAG,  S_FALLBACK, SV_FONT_STYLE_ITALIC);
 StyleAddSimpleRule(sheet, TAG_ABSTRACT, ArenaSI_FONT_WEIGHT, ST_FLAG, S_FALLBACK, SV_FONT_WEIGHT_BOLD);

 StyleAddSimpleRule(sheet, TAG_ABSTRACT, ArenaSI_MARGIN_RIGHT, ST_PX,  S_FALLBACK, 40);
 StyleAddSimpleRule(sheet, TAG_ABSTRACT, ArenaSI_MARGIN_LEFT, ST_PX,   S_FALLBACK, 40);

 /* set default style for TAG_QUOTE */

 StyleAddSimpleRule(sheet, TAG_QUOTE, ArenaSI_FONT_STYLE_SLANT, ST_FLAG,     S_FALLBACK, SV_FONT_STYLE_ITALIC);
 StyleAddSimpleRule(sheet, TAG_QUOTE, ArenaSI_MARGIN_RIGHT, ST_PX,     S_FALLBACK, 30);
 StyleAddSimpleRule(sheet, TAG_QUOTE, ArenaSI_MARGIN_LEFT, ST_PX,      S_FALLBACK, 50);

 /* set style for tables */

 StyleAddSimpleRule(sheet, TAG_TH, 	ArenaSI_FONT_WEIGHT, ST_FLAG,     S_FALLBACK, SV_FONT_WEIGHT_BOLD);
/*
  StyleAddSimpleRule(sheet, TAG_TH, 	ArenaSI_TEXT_ALIGN,  ST_FLAG,     S_FALLBACK, ALIGN_CENTER);
  StyleAddSimpleRule(sheet, TAG_TD, 	ArenaSI_TEXT_ALIGN,  ST_FLAG,     S_FALLBACK, ALIGN_CENTER);
  */
 /* set style for HR */

 StyleAddSimpleRule(sheet, TAG_HR, 	ArenaSI_COLOUR, ST_RGB,            S_FALLBACK, (float) (255 << 24));

 return sheet;
}


void StyleParse(Doc* theDoc)
{
#ifdef ARENA_DEBUG	/* QingLong.24-10-97 */
 char Iam[] = "StyleParse";
#endif


#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
 if (STYLE_TRACE) Arena_TracePrint(Iam, ")\n");
#endif

 if (!theDoc->head_style && !theDoc->link_style)
   {
    theDoc->style = NULL; /* i.e. style->default */
    return;
   }

 if (theDoc->user_style)
   {
   /* this document contains the user's style sheet if it exists */

    Announce(_("Applying personal style sheet..."));

    /*	rgbClear();*/

    if (context->style)
      {
       FreeStyleSheet(context->style);
       context->style = StyleGetInit();
      }

    StyleChew(theDoc, context->style, theDoc->head_style, S_READER);
    StyleChew(theDoc, context->style, theDoc->link_style, S_READER);
    theDoc->style = context->style;

#ifdef ARENA_DEBUG	/* QingLong.14-12-96 */
    if (STYLE_TRACE)
      {
       Arena_TracePrint(Iam, " context->style follows:\n");
       StyleSane(context->style);
      }
#endif

    Announce(_("Applying personal style sheet... done."));
    return;
   }

 /* this is a normal incoming document with style sheet attached */

 Announce(_("applying document's style sheet..."));

 rgbClear();
 theDoc->style = StyleCopy(context->style);
#if 1	/* QingLong.24-10-97 */
 StyleChew(theDoc, theDoc->style, theDoc->head_style, S_AUTHOR);
 StyleChew(theDoc, theDoc->style, theDoc->link_style, S_AUTHOR);
# else
 StyleChew(theDoc, theDoc->style, theDoc->head_style, S_READER);
 StyleChew(theDoc, theDoc->style, theDoc->link_style, S_READER);
#endif

#ifdef ARENA_DEBUG	/* QingLong.14-12-96 */
 if (STYLE_TRACE)
   {
    Arena_TracePrint(Iam,
		     " The doc "POINTER_FORMAT" ->style follows\n",
		     theDoc);
    StyleSane(context->style);
   }
#endif
 Announce(_("Applying document's style sheet... done."));
}


/*
 * StyleZoom is a bit too Arena-specific.
 */
void StyleZoomChange(Doc* theDoc, double f)
{
#ifdef ARENA_DEBUG
 char Iam[] = "StyleZoomChange";
#endif


 if (theDoc)
   {
    Announce(_("Loading fonts..."));
    ShowBusy();

    lens_factor *= f;
    DisplaySizeChanged(theDoc, True);
    /* 13-09-97, Stephen McCamant:
     * Because zooming in or out changes the size of the document,
     * as well as moving to its beginning,
     * it needs to redraw the scrollbars to reflect the new position.
     * QinLong.18-11-97: The DisplayScrollBar() is called by DisplayDoc().
     */
    DisplayDoc(theDoc, False);
    Announce("%s", theDoc->url);
    HideBusy();
   }
#ifdef ARENA_DEBUG
  else
   {
    if (DISPLAY_TRACE)
      Arena_TracePrint(Iam, " ERROR! The doc to zoom is NULL.\n");
   }
#endif
}


char *StyleLoad(char *href, int hreflen, Bool reload)
{
 Doc *doc = NULL;
#ifdef ARENA_DEBUG	/* QingLong.10-02-97 */
 char Iam[] = "StyleLoad";
 char* thehref = (char*)Arena_CAlloc(hreflen + 1, sizeof(char), False);

 strncpy(thehref, href, hreflen+1);
#endif

 /* check for null name */

 doc = GetStyleSheet(href, hreflen, reload);

 if (doc)
   {
#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
    if (STYLE_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam, " \"%s\" succedded?\n", thehref);
    Free(thehref);
#endif
    return(doc->content_buffer);
   }
  else
   {
#ifdef ARENA_DEBUG	/* QingLong.10-02-97 */
    if (STYLE_TRACE)
      Arena_TracePrint(Iam, " \"%s\" FAILED!\n", thehref);
    Free(thehref);
#endif
    return NULL;
   }
}


Bool StyleClearDoc(Doc* theDoc)
{
 if (theDoc)
   {
    HTArray_clear(theDoc->style_stack);
    theDoc->style_stack  = NULL;
    theDoc->current_flat = NULL;
    return True;
   }
  else
   {
    return False;
   }
}



/* formatting functions, here starts what should become format.c */

#if 0
StyleStackElement* NewStackElement(void)
{
 return (StyleStackElement*)Arena_CAlloc(1, sizeof(StyleStackElement), False);
}
#endif


/* 
 * FormatElementStart is called when a new element (implied or not) is started.
 * It will call StyleEval to flatten a set of properties
 * (called a stack element) to be returned by StyleGet.
 * The stack element is then pushed onto the stack.
 */
void FormatElementStart(Doc* theDoc, int element, char *class, int class_len)
{
#ifdef ARENA_DEBUG
 char Iam[] = "FormatElementStart";
#endif

 if (theDoc)
   {
    Frame *new_frame, *old_frame;
    StyleSimpleSelector* stack_el = NewStyleSimpleSelector();
    StyleSheet* sheet = (theDoc->style ? theDoc->style : context->style);


    if ((element < 0) || (element >= TAG_LAST))
      {
#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
       if (REGISTER_TRACE && VERBOSE_TRACE)
	 Arena_TracePrint(Iam, " element = %d\n", element);
#endif
       element = TAG_P; /* ugly */
      }

    theDoc->current_sheet = sheet;

#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
    if (REGISTER_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam, " %d\n", element);
#endif

    if (theDoc->style_stack == NULL) theDoc->style_stack = HTArray_new(20);

    stack_el->element = element;
    if (class) stack_el->class = strndup(class, class_len);

    HTArray_addObject(theDoc->style_stack, (void *)stack_el); 

    stack_el->flat = StyleEval(sheet, theDoc->style_stack);

#if 0
    {
     void **data = NULL;
     StyleSimpleSelector* ss = (StyleSimpleSelector*)HTArray_firstObject(theDoc->style_stack, data);

#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
     if (REGISTER_TRACE && VERBOSE_TRACE)
       Arena_TracePrint(Iam, " %d\n", element);
#endif
     while (ss)
       {
#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
	if (REGISTER_TRACE && VERBOSE_TRACE)
	  Arena_TracePrint(Iam, " %d\n", ss->element);
#endif
	ss = HTArray_nextObject(theDoc->style_stack, data);
       }
    }
#endif

    theDoc->current_flat = stack_el->flat;
    StyleSetFlag(theDoc, S_MARGIN_TOP_FLAG, (int)StyleGet(theDoc,
						       ArenaSI_MARGIN_TOP));
    if (element == TAG_P)
      {
       StyleSetFlag(theDoc, S_INDENT_FLAG,
		    (int)StyleGet(theDoc, ArenaSI_TEXT_INDENT));
       StyleSetFlag(theDoc, S_LEADING_FLAG, True);
      }

    if (element == TAG_HTML || element == TAG_HTML_SOURCE || 
	(/*element != TAG_TABLE && */
	 element != TAG_ABSTRACT &&
	 element != TAG_BLOCKQUOTE && element != TAG_CAPTION &&
	 element != TAG_NOTE && element != TAG_PRE &&
	 element != TAG_QUOTE))  
      return;

    if (prepass || damn_table) return;
    old_frame = (Frame *)Pop();
    Push(old_frame);

    if (paintStartLine > 0)
      Push((Frame*)1);
     else
      {
       new_frame = (Frame*)Arena_CAlloc(1, sizeof(Frame), False);
       PixOffset += (StyleGet(theDoc, ArenaSI_PADDING_TOP) +
		     StyleGet(theDoc, ArenaSI_MARGIN_TOP)); /* must be S_PADDING_TOP */
       new_frame->offset      = PixOffset;
       new_frame->leftmargin  = StyleGet(theDoc, ArenaSI_PADDING_LEFT);
       new_frame->rightmargin = StyleGet(theDoc, ArenaSI_PADDING_RIGHT);
       new_frame->indent      = (old_frame->indent +
				 StyleGet(theDoc, ArenaSI_MARGIN_LEFT));
       new_frame->width       = (old_frame->width  -
				 StyleGet(theDoc, ArenaSI_MARGIN_LEFT) -
				 StyleGet(theDoc, ArenaSI_MARGIN_RIGHT));
       new_frame->style = 0;
       new_frame->border = 0;
#ifdef STYLE_COLOUR_BORDER
       new_frame->cb_ix = 0;
#else
       new_frame->cb_ix = 0;
#endif
       new_frame->flow = old_frame->flow; /* StyleGet(S_ALIGN) */
       new_frame->next = new_frame->child = NULL;
       new_frame->box_list = NULL;
       PrintBeginFrame(new_frame);
       Push(new_frame);
      }
   }
#ifdef ARENA_DEBUG
  else
   {
    if (STYLE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The doc is NULL.\n");
   }
#endif
}


/*
 * FormatElementEnd pops a set of flattened style properties from the stack.
 * Note that calls to FormatElementStart and FormatElementEnd must
 * always match each other. I.e. your SGML/HTML parser must add any
 * implied/omitted tags.
 */
void FormatElementEnd(Doc* theDoc)
{
#ifdef ARENA_DEBUG
 char Iam[] = "FormatElementEnd";
#endif

 if (theDoc)
   {
    int element = 0;
    Frame* old_frame;
    StyleSimpleSelector* stack_el;


#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
    if (REGISTER_TRACE && VERBOSE_TRACE) Arena_TracePrint(Iam, ")\n");
#endif

#if 0
    {
     void** data = NULL;
     StyleSimpleSelector* ss = (StyleSimpleSelector*)HTArray_firstObject(theDoc->style_stack, data);
     while (ss)
       {
#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
	if (REGISTER_TRACE && VERBOSE_TRACE)
	  Arena_TracePrint(Iam, " %d\n", ss->element);
#endif
	ss = HTArray_nextObject(theDoc->style_stack, data);
       }
    }
#endif

    if (theDoc->style_stack->size > 0)
      {
       stack_el = (StyleSimpleSelector*)theDoc->style_stack->data[theDoc->style_stack->size - 1];
       element = stack_el->element;

       Free(stack_el->flat);
       Free(stack_el->class);
       Free(stack_el);
       theDoc->style_stack->data[theDoc->style_stack->size - 1] = NULL;
       theDoc->style_stack->size--;
      }

    if (theDoc->style_stack->size > 0)
      {
       stack_el = (StyleSimpleSelector*)theDoc->style_stack->data[theDoc->style_stack->size - 1];
       theDoc->current_flat = stack_el->flat;
      }
     else
      {
#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
       if (REGISTER_TRACE && VERBOSE_TRACE)
	 Arena_TracePrint(Iam, " stack empty!!\n");
#endif
       theDoc->current_flat = NULL;
      }

    if (element == TAG_HTML || element == TAG_HTML_SOURCE || 
	(/*element != TAG_TABLE && */
	 element != TAG_ABSTRACT &&
	 element != TAG_BLOCKQUOTE && element != TAG_CAPTION &&
	 element != TAG_NOTE && element != TAG_PRE &&
	 element != TAG_QUOTE))
      return;

    if (prepass || damn_table)  return;
    old_frame = (Frame*)Pop();

    if (old_frame)
      {
       Frame* parent_frame;

       if (old_frame != (Frame*)1)
	 {
	  parent_frame = (Frame*)Pop();
	  if (parent_frame) Push(parent_frame);

#if 0	/* QingLong.17-10-97. appears to be useless. (??????) */
	  PixOffset += StyleGet(theDoc, ArenaSI_PADDING_TOP);
	  old_frame->offset -= StyleGet(theDoc, ArenaSI_PADDING_TOP);
#endif
	  old_frame->height = PixOffset - old_frame->offset;
	  SetBeginFrameHeight(old_frame);

	  PrintFrameSize(old_frame);
	  PrintEndFrame(((parent_frame == (Frame*)1) ? NULL : parent_frame),
			old_frame);
	  FreeFrames(old_frame); 
	 }
      }
   }
#ifdef ARENA_DEBUG
  else
   {
    if (STYLE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The doc is NULL.\n");
   }
#endif
}
