/*
   dither.c 
            - Dave Raggett, 16th December 1994
            - revised 9th Feb 1994 to speed decoding & add gif89 support
            - separated from gif.c by Phill Hallam-Baker

   The colours are preallocated by image.c with 4:8:4 R:G:B colours and
   16 grey scales. These are rendered using an ordered dither based on
   16x16 dither matrices for monochrome, greyscale and colour images.

   This approach ensures that we won't run out of colours regardless of
   how many images are on screen or how many independent copies of www
   are running at any time. The scheme is compatible with HP's MPOWER
   tools. The code will be extended to take advantage of 24 bit direct
   colour or secondary true colour hardware colourmaps as soon as practical.
*/


#include <X11/Intrinsic.h>
#include <stdio.h>
#include <stdlib.h>

#include "arena.h"
#include "colour_types.h"
#include "colour.h"
#include "defs.h"
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
#  include "debug.h"
# else
#  include "error.h"
#endif
#include "image.h"
#include "main.h"
#include "util.h"


int Magic256[256] =    /* for halftoning */
{
  0, 223,  48, 207,  14, 237,  62, 221,   3, 226,  51, 210,  13, 236,  61, 220,
175,  80, 128,  96, 189,  94, 141, 110, 178,  83, 130,  99, 188,  93, 140, 109,
191,  32, 239,  16, 205,  46, 253,  30, 194,  35, 242,  19, 204,  45, 252,  29,
112, 143,  64, 159, 126, 157,  78, 173, 115, 146,  67, 162, 125, 156,  77, 172,
 11, 234,  59, 218,   5, 228,  53, 212,   8, 231,  56, 215,   6, 229,  54, 213,
186,  91, 138, 107, 180,  85, 132, 101, 183,  88, 135, 104, 181,  86, 133, 102,
202,  43, 250,  27, 196,  37, 244,  21, 199,  40, 247,  24, 197,  38, 245,  22,
123, 154,  75, 170, 117, 148,  69, 164, 120, 151,  72, 167, 118, 149,  70, 165,
 12, 235,  60, 219,   2, 225,  50, 209,  15, 238,  63, 222,   1, 224,  49, 208,
187,  92, 139, 108, 177,  82, 129,  98, 190,  95, 142, 111, 176,  81, 128,  97,
203,  44, 251,  28, 193,  34, 241,  18, 206,  47, 254,  31, 192,  33, 240,  17,
124, 155,  76, 171, 114, 145,  66, 161, 127, 158,  79, 174, 113, 144,  65, 160,
  7, 230,  55, 214,   9, 232,  57, 216,   4, 227,  52, 211,  10, 233,  58, 217,
182,  87, 134, 103, 184,  89, 136, 105, 179,  84, 131, 100, 185,  90, 137, 106,
198,  39, 246,  23, 200,  41, 248,  25, 195,  36, 243,  20, 201,  42, 249,  26,
119, 150,  71, 166, 121, 152,  73, 168, 116, 147,  68, 163, 122, 153,  74, 169
};


int Magic16[256] =    /* for 16 levels of gray */
{
  0, 13,  3, 12,  1, 14,  4, 13,  0, 13,  3, 12,  1, 14,  4, 13,
 10,  5,  8,  6, 11,  6,  8,  6, 10,  5,  8,  6, 11,  5,  8,  6,
 11,  2, 14,  1, 12,  3, 15,  2, 11,  2, 14,  1, 12,  3, 15,  2,
  7,  8,  4,  9,  7,  9,  5, 10,  7,  9,  4, 10,  7,  9,  5, 10,
  1, 14,  3, 13,  0, 13,  3, 12,  0, 14,  3, 13,  0, 13,  3, 13,
 11,  5,  8,  6, 11,  5,  8,  6, 11,  5,  8,  6, 11,  5,  8,  6,
 12,  3, 15,  2, 12,  2, 14,  1, 12,  2, 15,  1, 12,  2, 14,  1,
  7,  9,  4, 10,  7,  9,  4, 10,  7,  9,  4, 10,  7,  9,  4, 10,
  1, 14,  4, 13,  0, 13,  3, 12,  1, 14,  4, 13,  0, 13,  3, 12,
 11,  5,  8,  6, 10,  5,  8,  6, 11,  6,  8,  7, 10,  5,  8,  6,
 12,  3, 15,  2, 11,  2, 14,  1, 12,  3, 15,  2, 11,  2, 14,  1,
  7,  9,  4, 10,  7,  9,  4,  9,  7,  9,  5, 10,  7,  8,  4,  9,
  0, 14,  3, 13,  1, 14,  3, 13,  0, 13,  3, 12,  1, 14,  3, 13,
 11,  5,  8,  6, 11,  5,  8,  6, 11,  5,  8,  6, 11,  5,  8,  6,
 12,  2, 14,  1, 12,  2, 15,  1, 11,  2, 14,  1, 12,  2, 15,  2,
  7,  9,  4, 10,  7,  9,  4, 10,  7,  9,  4, 10,  7,  9,  4, 10

};


int Magic32[256] =    /* for 8 levels of green */
{
  0, 27,  6, 25,  2, 29,  8, 27,  0, 27,  6, 26,  2, 29,  7, 27,
 21, 10, 16, 12, 23, 11, 17, 13, 22, 10, 16, 12, 23, 11, 17, 13,
 23,  4, 29,  2, 25,  6, 31,  4, 24,  4, 29,  2, 25,  5, 31,  4,
 14, 17,  8, 19, 15, 19,  9, 21, 14, 18,  8, 20, 15, 19,  9, 21,
  1, 28,  7, 27,  1, 28,  6, 26,  1, 28,  7, 26,  1, 28,  7, 26,
 23, 11, 17, 13, 22, 10, 16, 12, 22, 11, 16, 13, 22, 10, 16, 12,
 25,  5, 30,  3, 24,  4, 30,  3, 24,  5, 30,  3, 24,  5, 30,  3,
 15, 19,  9, 21, 14, 18,  8, 20, 15, 18,  9, 20, 14, 18,  8, 20,
  1, 29,  7, 27,  0, 27,  6, 25,  2, 29,  8, 27,  0, 27,  6, 25,
 23, 11, 17, 13, 22, 10, 16, 12, 23, 12, 17, 13, 21, 10, 16, 12,
 25,  5, 31,  3, 23,  4, 29,  2, 25,  6, 31,  4, 23,  4, 29,  2,
 15, 19,  9, 21, 14, 18,  8, 20, 15, 19, 10, 21, 14, 18,  8, 19,
  1, 28,  7, 26,  1, 28,  7, 26,  0, 28,  6, 26,  1, 28,  7, 26,
 22, 11, 16, 12, 22, 11, 17, 13, 22, 10, 16, 12, 23, 11, 17, 13,
 24,  5, 30,  3, 24,  5, 30,  3, 24,  4, 30,  2, 24,  5, 30,  3,
 14, 18,  9, 20, 15, 19,  9, 20, 14, 18,  8, 20, 15, 19,  9, 21
};


int Magic64[256] =    /* for 4 levels of red and blue */
{
  0, 55, 12, 51,  3, 59, 15, 55,  1, 56, 13, 52,  3, 58, 15, 54,
 43, 20, 32, 24, 47, 23, 35, 27, 44, 20, 32, 24, 47, 23, 35, 27,
 47,  8, 59,  4, 51, 11, 63,  7, 48,  9, 60,  5, 50, 11, 62,  7,
 28, 35, 16, 39, 31, 39, 19, 43, 28, 36, 16, 40, 31, 39, 19, 43,
  3, 58, 15, 54,  1, 56, 13, 52,  2, 57, 14, 53,  1, 57, 13, 53,
 46, 22, 34, 26, 45, 21, 33, 25, 45, 22, 33, 26, 45, 21, 33, 25,
 50, 11, 62,  7, 48,  9, 60,  5, 49, 10, 61,  6, 49,  9, 61,  5,
 30, 38, 18, 42, 29, 37, 17, 41, 30, 37, 18, 41, 29, 37, 17, 41,
  3, 58, 15, 54,  0, 56, 12, 52,  4, 59, 16, 55,  0, 55, 12, 51,
 46, 23, 34, 27, 44, 20, 32, 24, 47, 23, 35, 27, 44, 20, 32, 24,
 50, 11, 62,  7, 48,  8, 60,  4, 51, 12, 63,  8, 47,  8, 59,  4,
 31, 38, 19, 42, 28, 36, 16, 40, 31, 39, 19, 43, 28, 36, 16, 40,
  2, 57, 14, 53,  2, 57, 14, 53,  1, 56, 13, 52,  2, 58, 14, 54,
 45, 21, 33, 25, 46, 22, 34, 26, 44, 21, 32, 25, 46, 22, 34, 26,
 49, 10, 61,  6, 49, 10, 61,  6, 48,  9, 60,  5, 50, 10, 62,  6,
 29, 37, 17, 41, 30, 38, 18, 42, 29, 36, 17, 40, 30, 38, 18, 42
};

static const unsigned short ditherColourMask[5] = { 0x0, 0x1, 0x3, 0x7, 0xF};


/*
 * Universal ditherer.
 */
Bool ArenaDither(unsigned short   red,
		 unsigned short green,
		 unsigned short  blue,
		 unsigned int x, unsigned int y,
		 unsigned int depth,
		 ArenaImageingCapability imageing,
		 unsigned char* theColourL)
{
 Bool DitherSuccess = False;
#ifdef ARENA_DEBUG
 char Iam[] = "Arena_dither";
#endif


 if (theColourL)
   {
    int c, r, g, b;
    unsigned short grey = (3 * red + 6 * green + blue)/10;


    switch (depth)
      {
       case 1:
       case 2:
       case 4:
	 c = ((y % 16) << 4) + (x % 16);

	 switch (imageing)
	   {
	    case MONO:
	      theColourL[0] = greymap[((grey < Magic256[c]) ? 0 : 15)];
	      DitherSuccess = True;
	      break;

	    case GREY4:
	      g = grey & 0xF0;
	      if (grey - g > Magic16[c]) g += 16;

	      theColourL[0] = greymap[(min(g, 0xF0)) >> 4];
	      DitherSuccess = True;
	      break;

	    case COLOUR232:
	      if (red == green  && green == blue)
		{
		 g = grey & 0xF0;
		 if (grey - g > Magic16[c]) g += 16;

		 theColourL[0] = greymap[(min(g, 0xF0)) >> 4];
		}
	       else
		{
		 r = red   & 0xC0;
		 g = green & 0xE0;
		 b = blue  & 0xC0;

		 if (red   - r > Magic64[c]) r += 0x40;
		 if (green - g > Magic32[c]) g += 0x20;
		 if (blue  - b > Magic64[c]) b += 0x40;

		 r = min(r, 255) & 0xC0;
		 g = min(g, 255) & 0xE0;
		 b = min(b, 255) & 0xC0;

		 theColourL[0] = stdcmap[(r >> 6) | (g >> 3) | (b >> 1)];
		}

	      DitherSuccess = True;
	      break;

	    case COLOUR444:
#ifdef ARENA_DEBUG
	      if (IMAGE_TRACE)
		Arena_TracePrint(Iam,
				 " ERROR! Cannot dither"
				 " true colour 12 ---> depth %d.\n",
				 depth);
#endif
	      break;

	    case COLOUR888:
#ifdef ARENA_DEBUG
	      if (IMAGE_TRACE)
		Arena_TracePrint(Iam,
				 " ERROR! Cannot dither"
				 " true colour 24 ---> depth %d.\n",
				 depth);
#endif
	      break;

	    case TrueCOLOUR:
#ifdef ARENA_DEBUG
	      if (IMAGE_TRACE)
		Arena_TracePrint(Iam,
				 " ERROR! Cannot dither"
				 " true colour 48 ---> depth %d.\n",
				 depth);
#endif
	      break;
	   }
	 /* End ``switch (imageing)'' */

	 if (DitherSuccess) theColourL[0] &= ditherColourMask[depth];
	 break;


       case 8:
	 c = ((y % 16) << 4) + (x % 16);

	 switch (imageing)
	   {
	    case MONO:
	      theColourL[0] = greymap[((grey < Magic256[c]) ? 0 : 15)];
	      DitherSuccess = True;
	      break;

	    case GREY4:
	      g = grey & 0xF0;
	      if (grey - g > Magic16[c]) g += 16;

	      theColourL[0] = greymap[(min(g, 0xF0))>> 4];
	      DitherSuccess = True;
	      break;

	    case COLOUR232:
	    case COLOUR444:
	      if (red == green  && green == blue)
		{
		 g = grey & 0xF0;
		 if (grey - g > Magic16[c]) g += 16;

		 theColourL[0] = greymap[(min(g, 0xF0)) >> 4];
		}
	       else
		{
		 r = red   & 0xC0;
		 g = green & 0xE0;
		 b = blue  & 0xC0;

		 if (red   - r > Magic64[c]) r += 64;
		 if (green - g > Magic32[c]) g += 32;
		 if (blue  - b > Magic64[c]) b += 64;

		 r = min(r, 255) & 0xC0;
		 g = min(g, 255) & 0xE0;
		 b = min(b, 255) & 0xC0;

		 theColourL[0] = stdcmap[(r >> 6) | (g >> 3) | (b >> 1)];
		}

	      DitherSuccess = True;
	      break;

	    case COLOUR888:
	      if (red == green  && green == blue)
		{
		 g = grey & 0xF0;
		 if (grey - g > Magic16[c]) g += 16;

		 theColourL[0] = greymap[(min(g, 0xF0)) >> 4];
		}
	       else
		{
		 theColourL[0] = stdcmap[((red   >> 16) |
					  (green >>  8) |
					  (blue       ))];
		}

	      DitherSuccess = True;
	      break;

	    case TrueCOLOUR:
#ifdef ARENA_DEBUG
	      if (IMAGE_TRACE)
		Arena_TracePrint(Iam,
				 " ERROR! Cannot dither"
				 " true colour 48 ---> depth %d.\n",
				 depth);
#endif
	      break;
	   }
	 /* End ``switch (imageing)'' */
	 break;


       case 16:
	 {
	  unsigned long colour;

	  GetColour(red, green, blue, &colour);
	  theColourL[0] = (colour & 0x00FF);   /* LSB first! */
	  theColourL[1] = (colour & 0xFF00) >> 8;
	  DitherSuccess = True;
	 }
       break;


       case 24:
	 theColourL[0] = blue;   /* LSB first! */
	 theColourL[1] = green;
	 theColourL[2] = red;
         theColourL[3] = '\0';
	 DitherSuccess = True;
	 break;


       case 48:
	 theColourL[0] = (blue  & 0x00FF);   /* LSB first! */
	 theColourL[1] = (blue  & 0xFF00) >> 8;
	 theColourL[2] = (green & 0x00FF);
	 theColourL[3] = (green & 0xFF00) >> 8;
	 theColourL[4] = (red   & 0x00FF);
	 theColourL[5] = (red   & 0xFF00) >> 8;
	 DitherSuccess = True;
	 break;


       default:
#ifdef ARENA_DEBUG
	 if (IMAGE_TRACE)
	   Arena_TracePrint(Iam,
			    " ERROR!"
			    " Cannot dither \"%d\" ---> depth %d.\n",
			    (int)imageing, depth);
#endif
	 break;
      }
    /* End ``switch (depth)''*/
   }

 return DitherSuccess;
}
