
/* ====================================================================
 * Copyright (c) 1996 Vidya Media Ventures, Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of this source code or a derived source code must
 *    retain the above copyright notice, this list of conditions and the
 *    following disclaimer. 
 *
 * 2. Redistributions of this module or a derived module in binary form
 *    must reproduce the above copyright notice, this list of conditions
 *    and the following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY VIDYA MEDIA VENTURES, INC. ``AS IS'' AND 
 * ANY EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL VIDYA MEDIA VENTURES, INC.
 * OR ITS EMPLOYEES BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 * This software is a contribution to and makes use of the Apache HTTP
 * server which is written, maintained and copywritten by the Apache Group.
 * See http://www.apache.org/ for more information.
 *
 * This software makes use of libpq which an interface to the PostgreSQL
 * database.  PostgreSQL is copyright (c) 1994 by the Regents of the
 * University of California.  As of this writing, more information on
 * PostgreSQL can be found at http://www.postgresSQL.org/
 *
 */

/*
 * 
 * PostgreSQL authentication
 *
 * Version 0.6 (July 1998)
 *
 * Needs libpq-fe.h and libpq.a
 *
 * Outline:
 *
 * One database, and one (or two) tables.  One table holds the username and
 * the encryped password.  The other table holds the username and the names
 * of the group to which the user belongs.  It is possible to have username,
 * groupname and password in the same table.
 *
 * 
 * Directives:
 *
 * Auth_PGhost   	Hostname of the machine running
 *		   	the postmaster. The effective uid
 *		   	of the server should be allowed
 *		   	access. 
 *
 * Auth_PGport	        Port that the postmaster is listening to.
 *
 * Auth_PGoptions	Options to set on the database connection(optional)
 *
 * Auth_PGdatabase	Name of the database in which the following
 *			table(s) are
 *			
 * Auth_PGpwd_table	Contains at least the fields with the
 *			username and the (encrypted) password
 *
 * Auth_PGgrp_table	Contains at least the fields with the
 *			username and the groupname. A user which
 *			is in multiple groups has therefore
 *			multiple entries
 *
 * Auth_PGuid_field	Name of the field containing the username
 * Auth_PGpwd_field     Fieldname for the passwords
 * Auth_PGgid_field	Fieldname for the groupname
 *
 * Auth_PG_nopasswd	<on|off>   [defaults to off]
 *			skip password comparison if passwd field is
 *			empty.
 *
 * Auth_PG_authoratative <on|off>  [defaults to on]
 *			When turned off, errors in authentication will be
 *			allowed to fall through to possibly be covered by
 *			other authorization modules or parent directory
 *			authorization schemes.  Exercise caution when deciding
 *			to turn this off!
 *
 * Auth_PG_authorative <on|off>  [defaults to on]
 *			Same as Auth_PG_Authoratative.  This is here for
 *			backwards compatability.
 *
 * Auth_PG_encrypted	<on|off>   [defaults to on]
 *			Controls weather this module expects passwords in the database
 *			to be encrypted or not.  When turned off, you can use 
 *			unencrypted passwords in your database.  Exercise caution
 *			when deciding to turn this off!
 *
 * Auth_PGpwd_whereclause  
 *                      Add this sql fragement to the end of the
 *			SQL select statement on the password table.
 *                      It would be going onto the end of a where 
 *			clause so it should start with a conjunctive
 *                      like 'and' or 'or'.
 *
 * Auth_PGgrp_whereclause  
 *                      Add this sql fragement to the end of the
 *			SQL select statement on the group table.
 *                      It would be going onto the end of a where 
 *			clause so it should start with a conjunctive
 *                      like 'and' or 'or'.
 *
 *
 * Adam Sussman (asussman@vidya.com) Feb, 1996
 *
 * see http://www.postgreSQL.org/
 *
 *
 * Version 0.0  (Feb 1996) First release (adaptation from mod_auth_msql.c v0.5)
 *			   mod_auth_msql.c was written by Dirk.vanGulik@jrc.it; 
 *                 	   [http://ewse.ceo.org; http://me-www.jrc.it/~dirkx]
 *	   0.1  (Mar 1996) Correct PGgid_field command
 *	   0.2  (Mar 1996) Added use and Auth_PGgrp_whereclause
 *	   0.3  (May 1996) Some sundry patches to fix grp_whereclause and some compilation
 *			   issues.  Thanks to Randy Terbrush.
 *	   0.4  (Jun 1996) Got rid of pg_set_string_slot, use the stock function in http_config.c
 *			   Made Auth_PGoptions take an actual string argument.
 *			   Fixed some logical sillyness in check_auth
 *			   Made command error strings more reasonable.
 *			   Try to weed out possibility of function names conflicting 
 *			     with other 3rd party modules.
 *			   Added Auth_PG_authorative directive
 *			   Added Auth_PG_enrypted directive
 *	   0.5  (Dec 1996) Some cosmetic changes to make apache 1.2 happy.
 *			   Added correctly spelled Auth_PG_Authoratative command.
 *	   0.6	(13 July 1998) Renamed to mod_auth_pgsql 
 *				Some changes just to make it compile under apache 1.3
 *				tested on apache 1.3 postgreSQL 6.4
 *				Giuseppe Tanzilli g.tanzilli@eurolink.it
 *	   0.7  (11 November 1998)	
 *				Some little changes just to make it compile in 
 *				Apache 1.3.3 & PostgreSQL 6.4
 *				the Where clause commands now need double quote
 *	   0.7.1 (02 March 1999)
 *			   (Matthias Eckermann, eckerman@lrz.uni-muenchen.de)
 *			   added 'AUTH_PGSQL_VERSION' & 'pg_auth_init_handler'
 */

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_protocol.h"
#include <libpq-fe.h>
#undef palloc

#define	AUTH_PGSQL_VERSION	"0.7.1"

/* Apache 1.2 changed some prototypes.  This is for backwards compatability */

#if (MODULE_MAGIC_NUMBER >= 19960725)
#define IS_APACHE_12         1
#define _const const
#else
#define _const
#endif

/* Apache 1.3 changed some function names  */
#if (MODULE_MAGIC_NUMBER >= 19980520)
#define IS_APACHE_13         1
#define _const const
#define PQsetdb(M_PGHOST,M_PGPORT,M_PGOPT,M_PGTTY,M_DBNAME)   PQsetdbLogin(M_PGHOST, M_PGPORT, M_PGOPT, M_PGTTY, M_DBNAME, NULL, NULL)
#endif

/* Apache 1.3.1 */ 
#if (MODULE_MAGIC_NUMBER >= 19980713)
#include"ap_compat.h"
#else
#include"compat.h"
#endif



#ifndef MAX_STRING_LEN
#define MAX_STRING_LEN 8000
#endif

typedef struct  {

    char *auth_pg_host;
    char *auth_pg_database;
    char *auth_pg_port;
    char *auth_pg_options;

    char *auth_pg_pwd_table;
    char *auth_pg_grp_table;

    char *auth_pg_pwd_field;
    char *auth_pg_uname_field;
    char *auth_pg_grp_field;

    int auth_pg_nopasswd;
    int auth_pg_authorative;
    int auth_pg_encrypted;

    char *auth_pg_pwd_whereclause;
    char *auth_pg_grp_whereclause;

} pg_auth_config_rec;

void *create_pg_auth_dir_config (pool *p, char *d)
{
   pg_auth_config_rec		*new_rec;

   new_rec =  pcalloc (p, sizeof(pg_auth_config_rec));
   if (new_rec == NULL) return NULL;

   /* sane defaults */
   new_rec->auth_pg_nopasswd = 0;
   new_rec->auth_pg_authorative = 1;
   new_rec->auth_pg_encrypted = 1;

   return new_rec;
}

_const
char *pg_set_passwd_flag (cmd_parms *cmd, pg_auth_config_rec *sec, int arg) {
    sec->auth_pg_nopasswd=arg;
    return NULL;
}

_const
char* pg_set_encrypted_flag (cmd_parms *cmd, pg_auth_config_rec *sec, int arg) {
    sec->auth_pg_encrypted=arg;
    return NULL;
}

_const
char* pg_set_authorative_flag (cmd_parms *cmd, pg_auth_config_rec *sec, int arg) {
    sec->auth_pg_authorative=arg;
    return NULL;
}

command_rec pg_auth_cmds[] = {
{ "Auth_PGhost", set_string_slot,
    (void*)XtOffsetOf(pg_auth_config_rec, auth_pg_host),
    OR_AUTHCFG, TAKE1, "the host name of the postgreSQL server." },

{ "Auth_PGdatabase", set_string_slot,
    (void*)XtOffsetOf(pg_auth_config_rec, auth_pg_database),
    OR_AUTHCFG, TAKE1, "the name of the database that contains authorization information. " },

{ "Auth_PGport", set_string_slot,
    (void*)XtOffsetOf(pg_auth_config_rec, auth_pg_port),
    OR_AUTHCFG, TAKE1, "the port the postmaster is running on. " },

{ "Auth_PGoptions", set_string_slot,
    (void*)XtOffsetOf(pg_auth_config_rec, auth_pg_options),
    OR_AUTHCFG, RAW_ARGS, "an options string to be sent to the postgres backed process. " },

{ "Auth_PGpwd_table", set_string_slot,
    (void*)XtOffsetOf(pg_auth_config_rec, auth_pg_pwd_table),
    OR_AUTHCFG, TAKE1, "the name of the table containing username/password tuples." },

{ "Auth_PGgrp_table", set_string_slot,
    (void*)XtOffsetOf(pg_auth_config_rec, auth_pg_grp_table),
    OR_AUTHCFG, TAKE1, "the name of the table containing username/group tuples." },

{ "Auth_PGpwd_field", set_string_slot,
    (void*)XtOffsetOf(pg_auth_config_rec, auth_pg_pwd_field),
    OR_AUTHCFG, TAKE1, "the name of the password field." },

{ "Auth_PGuid_field", set_string_slot,
    (void*)XtOffsetOf(pg_auth_config_rec, auth_pg_uname_field),
    OR_AUTHCFG, TAKE1, "the name of the user-id field." },

{ "Auth_PGgid_field", set_string_slot,
    (void*)XtOffsetOf(pg_auth_config_rec, auth_pg_grp_field),
    OR_AUTHCFG, TAKE1, 
	"the name of the group-name field." },

{ "Auth_PG_nopasswd", pg_set_passwd_flag, NULL, OR_AUTHCFG, FLAG, 
	"'on' or 'off'" },

{ "Auth_PG_encrypted", pg_set_encrypted_flag, NULL, OR_AUTHCFG, FLAG, 
	"'on' or 'off'" },

{ "Auth_PG_authorative", pg_set_authorative_flag, NULL, OR_AUTHCFG, FLAG, 
	"'on' or 'off'" },

{ "Auth_PG_authoratative", pg_set_authorative_flag, NULL, OR_AUTHCFG, FLAG, 
	"'on' or 'off'" },

{ "Auth_PGgrp_whereclause", set_string_slot,
    (void*)XtOffsetOf(pg_auth_config_rec, auth_pg_grp_whereclause),
    OR_AUTHCFG, TAKE1,
	"an SQL fragement that can be attached to the end of a where clause." },

{ "Auth_PGpwd_whereclause", set_string_slot,
    (void*)XtOffsetOf(pg_auth_config_rec, auth_pg_pwd_whereclause),
    OR_AUTHCFG, TAKE1,
	"an SQL fragement that can be attached to the end of a where clause." },

{ NULL }
};

module pgsql_auth_module;

char pg_errstr[MAX_STRING_LEN];
		 /* global errno to be able to handle config/sql 
		 * failures separately
		 */

/* Do a query and return the (0,0) value.  The query is assumed to be
 * a select.
 */
char *do_pg_query(request_rec *r, char *query, pg_auth_config_rec *sec) {

    PGconn		*pg_conn;
	PGresult	*pg_result;

	char		*val;
 	char 		*result=NULL;

	pg_errstr[0]='\0';


	pg_conn = PQsetdb(sec->auth_pg_host, sec->auth_pg_port, sec->auth_pg_options,
			 NULL, sec->auth_pg_database);

	if (PQstatus(pg_conn) != CONNECTION_OK) {
		sprintf(pg_errstr, "PGSQL 1: %s", PQerrorMessage(pg_conn));
		return NULL;
	}

	pg_result = PQexec(pg_conn, query);

	if (pg_result == NULL)
	{
		sprintf(pg_errstr, "PGSQL 2: %s -- Query: %s ", PQerrorMessage(pg_conn), query);
		PQfinish(pg_conn);
		return NULL;
	}

	if (PQresultStatus(pg_result) == PGRES_EMPTY_QUERY) {
		PQclear(pg_result);
		PQfinish(pg_conn);
		return NULL;
	}

	if (PQresultStatus(pg_result) != PGRES_TUPLES_OK) {
		sprintf(pg_errstr, "PGSQL 3: %s -- Query: %s", PQerrorMessage(pg_conn), query);
		PQclear(pg_result);
		PQfinish(pg_conn);
		return NULL;
	}

	if (PQntuples(pg_result) == 1) {
		val = PQgetvalue(pg_result, 0, 0);
		if (val == NULL) {
			sprintf(pg_errstr, "PGSQL 4: %s", PQerrorMessage(pg_conn));
			PQclear(pg_result);
			PQfinish(pg_conn);
			return NULL;
		}

		if (!(result = (char*)palloc(r->pool, strlen(val)+1))) {
			sprintf (pg_errstr, "Could not get memory for Postgres query.");
			PQclear(pg_result);
			PQfinish(pg_conn);
			return NULL;
		}

		strcpy(result, val);
	}

	/* ignore errors here ! */
	PQclear(pg_result);
	PQfinish(pg_conn);
	return result;
}
	
char *get_pg_pw(request_rec *r, char *user, pg_auth_config_rec *sec) {
  	char 		query[MAX_STRING_LEN];
	char            *pass, *p;
	if (
	    (!sec->auth_pg_pwd_table) ||
	    (!sec->auth_pg_pwd_field) ||
	    (!sec->auth_pg_uname_field)
	   ) {
		sprintf(pg_errstr,
			"PG: Missing parameters for password lookup: %s%s%s",
			(sec->auth_pg_pwd_table ? "" : "Password table "),
			(sec->auth_pg_pwd_field ? "" : "Password field name "),
			(sec->auth_pg_uname_field ? "" : "UserID field name ")
			);
		return NULL;
		};

    	sprintf(query,"select %s from %s where %s='%s' ",
		sec->auth_pg_pwd_field,
		sec->auth_pg_pwd_table,
		sec->auth_pg_uname_field,
		user);

	if (sec->auth_pg_pwd_whereclause) {
		sprintf(query, "%s %s ", query, sec->auth_pg_pwd_whereclause);
	}
	pass = do_pg_query(r,query,sec);
	if (pass)
	        for (p = pass; *p; p++) if (*p == ' ') *p = 0;
	return pass;
}	   

char *get_pg_grp(request_rec *r, char *group,char *user, pg_auth_config_rec *sec) {
  	char 		query[MAX_STRING_LEN];

	if (
	    (!sec->auth_pg_grp_table) ||
	    (!sec->auth_pg_grp_field) ||
	    (!sec->auth_pg_uname_field)
	   ) {
		sprintf(pg_errstr,
			"PG: Missing parameters for password lookup: %s%s%s",
			(sec->auth_pg_grp_table ? "" : "Group table "),
			(sec->auth_pg_grp_field ? "" : "GroupID field name "),
			(sec->auth_pg_uname_field ? "" : "UserID field name ")
			);
		return NULL;
		};

    	sprintf(query,"select %s from %s where %s='%s' and %s='%s'",
		sec->auth_pg_grp_field,
		sec->auth_pg_grp_table,
		sec->auth_pg_uname_field,user,
		sec->auth_pg_grp_field,group
		);

	if (sec->auth_pg_grp_whereclause)
		sprintf(query, "%s %s ", query, sec->auth_pg_grp_whereclause);

	return do_pg_query(r,query,sec);
}	   


int pg_authenticate_basic_user (request_rec *r)
{
    pg_auth_config_rec *sec =
      (pg_auth_config_rec *)get_module_config (r->per_dir_config,
						&pgsql_auth_module);
    conn_rec *c = r->connection;
    char *sent_pw, *real_pw;
    int res;

    if ((res = get_basic_auth_pw (r, (const char **)&sent_pw)))
        return res;

    /* if *password* checking is configured in any way, i.e. then
     * handle it, if not decline and leave it to the next in line..  
     * We do not check on dbase, group, userid or host name, as it is
     * perfectly possible to only do group control and leave
     * user control to the next guy in line.
     */
    if (
    	(!sec->auth_pg_pwd_table) && 
    	(!sec->auth_pg_pwd_field) 
	 ) return DECLINED;

    pg_errstr[0]='\0';

    if(!(real_pw = get_pg_pw(r, c->user, sec ))) {
	if ( pg_errstr[0] ) {
		res = SERVER_ERROR;
		} else {
		if (sec->auth_pg_authorative) {
			/* force error and access denied */
	        	sprintf(pg_errstr,"mod_auth_pgsql: Password for user %s not found (PG-Authorative)", c->user);
			note_basic_auth_failure (r);
			res = AUTH_REQUIRED;
		} else {
			/* allow fall through to another module */
			return DECLINED;
		}
	}
	log_reason (pg_errstr, r->filename, r);
	return res;
    }    

    /* allow no password, if the flag is set and the password
     * is empty. But be sure to log this.
     */

    if ((sec->auth_pg_nopasswd) && (!strlen(real_pw))) {
        sprintf(pg_errstr,"PG: user %s: Empty password accepted",c->user);
	log_reason (pg_errstr, r->uri, r);
	return OK;
	};

    /* if the flag is off however, keep that kind of stuff at
     * an arms length.
     */
    if ((!strlen(real_pw)) || (!strlen(sent_pw))) {
        sprintf(pg_errstr,"PG: user %s: Empty Password(s) Rejected",c->user);
	log_reason (pg_errstr, r->uri, r);
	note_basic_auth_failure (r);
	return AUTH_REQUIRED;
	};

    if (sec->auth_pg_encrypted)
	sent_pw = (char*)crypt(sent_pw, real_pw);

    if(strcmp(real_pw,sent_pw)) {
        sprintf(pg_errstr,"PG user %s: password mismatch",c->user);
	log_reason (pg_errstr, r->uri, r);
	note_basic_auth_failure (r);
	return AUTH_REQUIRED;
    }
    return OK;
}
    
/* Checking ID */
    
int pg_check_auth(request_rec *r) {
    pg_auth_config_rec *sec =
      (pg_auth_config_rec *)get_module_config (r->per_dir_config,
						&pgsql_auth_module);
    char *user = r->connection->user;
    int m = r->method_number;
    int group_result = DECLINED;

    array_header *reqs_arr = (array_header *) requires (r);
    require_line *reqs = reqs_arr ? (require_line *)reqs_arr->elts : NULL;

    register int x,res;
    const char *t;
    char *w;

    pg_errstr[0]='\0';

    /* if we cannot do it; leave it to some other guy 
     */
    if ((!sec->auth_pg_grp_table)&&(!sec->auth_pg_grp_field)) 
	return DECLINED;

    if (!reqs_arr) {
	if (sec->auth_pg_authorative)
	{
		/* force error and access denied */
	       	sprintf(pg_errstr,"mod_auth_pgsql: user %s denied, no access rules specified (PG-Authorative)", user);
		log_reason (pg_errstr, r->uri, r);
		note_basic_auth_failure (r);
		res = AUTH_REQUIRED;
	} else {
		return DECLINED;
	}
    }
    
    for(x=0; x < reqs_arr->nelts; x++) {
      
	if (! (reqs[x].method_mask & (1 << m))) continue;
	
        t = reqs[x].requirement;
        w = getword(r->pool, &t, ' ');

        if(!strcmp(w,"valid-user"))
            return OK;
	
        if(!strcmp(w,"user")) {
            while(t[0]) {
                w = getword_conf (r->pool, &t);
                if(!strcmp(user,w))
                    return OK;
            }
	    if (sec->auth_pg_authorative) {
		/* force error and access denied */
	       	sprintf(pg_errstr,"mod_auth_pgsql: user %s denied, no access rules specified (PG-Authorative)", user);
		log_reason (pg_errstr, r->uri, r);
		note_basic_auth_failure (r);
		return AUTH_REQUIRED;
	    }

        } else if (!strcmp(w,"group")) {
	   /* look up the membership for each of the groups in the table */
	   pg_errstr[0] = '\0';

           while(t[0]) {
                if (get_pg_grp(r,getword(r->pool, &t, ' '),user,sec)) {
			group_result = OK;
			};
       		};

	   if (pg_errstr[0]) {
		log_reason(pg_errstr, r->filename, r);
		return SERVER_ERROR;
	   } 

	   if (group_result == OK)
		return OK;

	   if (sec->auth_pg_authorative) {
           	sprintf(pg_errstr,"user %s not in right groups (PG-Authorative)", user);
		log_reason (pg_errstr, r->uri, r);
           	note_basic_auth_failure(r);
		return AUTH_REQUIRED;
		};
           }
        }
    
    return DECLINED;
}

void pg_auth_init_handler(server_rec *s, pool *p)
{
#if MODULE_MAGIC_NUMBER >= 19980527
    ap_add_version_component("AuthPostgreSQL/" AUTH_PGSQL_VERSION);
#endif
}

module pgsql_auth_module = {
   STANDARD_MODULE_STUFF,
   pg_auth_init_handler,	/* initializer */
   create_pg_auth_dir_config,	/* dir config creater */
   NULL,			/* dir merger --- default is to override */
   NULL,			/* server config */
   NULL,			/* merge server config */
   pg_auth_cmds,		/* command table */
   NULL,			/* handlers */
   NULL,			/* filename translation */
   pg_authenticate_basic_user,	/* check_user_id */
   pg_check_auth,		/* check auth */
   NULL,			/* check access */
   NULL,			/* type_checker */
   NULL,			/* pre-run fixups */
   NULL				/* logger */
};
