/*
 *
 *  (c) COPYRIGHT MIT and INRIA, 1996.
 *  Please first read the full copyright statement in file COPYRIGHT.
 *
 */

/*
 *
 * Author: S Bonhomme
 *
 */

/* Included headerfiles */
#define THOT_EXPORT extern
#include "amaya.h"
#include "message.h"
#include "trans.h"
#include "tree.h"
#include "fetchHTMLname.h"

#include "HTMLimage_f.h"
#include "XMLparser_f.h"
#ifdef MATHML
#include "MathML.h"
#endif
#include "fetchHTMLname_f.h"
#include "html2thot_f.h"
#include "transparse_f.h"

#ifdef _WINDOWS
#include "wininclude.h"
#endif /* _WINDOWS */
/*
 * pattern matching stack associates a node of source structure tree to a 
 * list of pat.
 */
static struct _mStack
  {
     strNode            *Node;
     strListSymb        *Symbols;
  }
matchStack[MAX_STACK];

static int          maxMatchDepth;

static int          topMatchStack;

/*  HTML generation */
/* stack of generated tags */
typedef struct _gStack
  {
     STRING              Tag;
     strAttrDesc        *Attributes;
     int                 Idf;
     int                 Nbc;
  }
strGenStack;

static strGenStack *generationStack[MAX_STACK];

static int          topGenerStack;
static UCHAR_T      TransferMode;
#define ByAttribute 0
#define InBuffer 1
static STRING       bufHTML;		/* HTML text buffer */
static int          szHTML;		/* size of generated HTML code */
static int          lastRulePlace;	/* pointer to the stack*/
static strTransSet *CourTransSet;       /* the transformation set used */
/* identifies the last node generated by the previous rule applied*/
static int          idfCounter;		/* element identifier generator */
static ThotBool     isClosed;

/* correspondance table between the transformations proposed to the user */
/* and the matching descriptors */
static strMatch    *menuTrans[20];
/* did the transformation succed ? */
static ThotBool     resultTrans;

/* pointer to the selected elements */
static int          ffc, flc, lfc, llc;
static int          maxSelDepth;
static Element      myFirstSelect, origFirstSelect;
static Element      myLastSelect, origLastSelect;
static Element      mySelect;

#ifdef __STDC__
void		    TransCallbackDialog(int ref, int typedata, STRING data);
#else
void                TransCallbackDialog (/*ref, typedata, data*/);
#endif
/*----------------------------------------------------------------------
   FreeStructTree: frees the structure tree and matching relations. 
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void         FreeStructTree (StructureTree t)
#else
static void         FreeStructTree (t)
structureTree       t;

#endif
{
   strMatch           *m, *m2;
   strMatchChildren   *mc, *mc2;
   strNode            *c, *n;

   if (t != NULL)
     {
	m = t->Matches;
	while (m != NULL)
	  {
	     mc = m->MatchChildren;
	     while (mc != NULL)
	       {
		  mc2 = mc->Next;
		  TtaFreeMemory ( mc);
		  mc = mc2;
	       }
	     m2 = m->Next;
	     TtaFreeMemory ( m);
	     m = m2;
	  }

	c = t->Child;
	while (c != NULL)
	  {
	     n = c->Next;
	     FreeStructTree (c);
	     c = n;
	  }
	TtaFreeMemory (t->Tag);
	TtaFreeMemory (t->Candidates);
	TtaFreeMemory ( t);
     }
}


/*----------------------------------------------------------------------
  FreeMatchEnv : frees the transformation context . 
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void         FreeMatchEnv ()
#else
static void         FreeMatchEnv ()
#endif
{
   strListElem           *l, *l1;

   /* liberation de la liste des sous-arbres transferes */
   l = strMatchEnv.ListSubTrees;
   strMatchEnv.ListSubTrees = NULL;
   while (l != NULL)
     {
	l1 = l;
	l = l->Next;
	TtaFreeMemory ( l1);
     }
   strMatchEnv.ListSubTrees = NULL;

   /* liberation de l'arbre des tags */
   FreeStructTree (strMatchEnv.SourceTree);
   strMatchEnv.SourceTree = NULL;
}

/*----------------------------------------------------------------------
   InitTransform: initialisation, called during Amaya initialisation
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                InitTransform ()
#else
void                InitTransform ()
#endif
{
   TransBaseDialog = TtaSetCallback (TransCallbackDialog, MAX_TRANS_DLG);
   ppInitAutomaton ();
}

/*----------------------------------------------------------------------
   NewNode: allocation of a Structure tree node.     
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static strNode *NewNode (STRING tag)
#else
static strNode * NewNode (tag)
STRING              tag;

#endif
{
   StructureTree res;

   res = (StructureTree) TtaGetMemory (sizeof (strNode));
   res->Tag = TtaAllocString (NAME_LENGTH);
   ustrcpy (res->Tag, tag);
   res->Matches = NULL;
   res->Candidates = NULL;
   res->Elem = NULL;
   res->MatchSymb = NULL;
   res->Parent = NULL;
   res->Child = NULL;
   res->Next = NULL;
   res->Previous = NULL;
   res->IsTrans = FALSE;
   res->NodeDepth = 0;
   return res;
}


/*----------------------------------------------------------------------
    thot tree -> structure tree
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void         BuildStructureTree (Element elem, Document doc, StructureTree father, int maxdepth, int depth)
#else
static void         BuildStructureTree (elem, doc, father, maxdepth, depth)
Element             elem;
Document            doc;
strNode            *father;
int                 maxdepth;
int                 depth;

#endif
{
   Element             elemCour;
   Attribute           attr;
   ElementType         elemType;
   STRING              tag;
   strNode            *added, *child;
#ifdef AMAYA_DEBUG
   int i;
#endif

   if (depth > maxdepth)
      return;
   tag = TtaAllocString (NAME_LENGTH);
   elemType = TtaGetElementType (elem);
   ustrcpy (tag, GITagNameByType (elemType));
   attr = NULL;
   TtaNextAttribute (elem, &attr);
   if (ustrcmp (tag, TEXT("???")) && ustrcmp (tag, TEXT("none")) && (TtaGetFirstChild (elem) != NULL || attr != NULL || TtaIsLeaf (elemType)))
     {
	added = NewNode (tag);
	added->Elem = elem;
	added->Parent = father;
	added->NodeDepth = depth;
	if (father->Child == NULL)
	  {
	     father->Child = added;
	     added->Previous = NULL;
	  }
	else
	  {
	     child = father->Child;
	     while (child->Next != NULL)
		child = child->Next;
	     child->Next = added;
	     added->Previous = child;
	  }
	depth++;
#ifdef AMAYA_DEBUG
	for (i=0;i<depth;i++)
	  printf("  ");
	printf("%s\n",added->Tag);
#endif
     }
   else
      added = father;
   TtaFreeMemory (tag);
   if ((ustrcmp ( TtaGetSSchemaName (elemType.ElSSchema), TEXT("HTML")) != 0) ||
       (elemType.ElTypeNum != HTML_EL_Comment_ && 
	elemType.ElTypeNum != HTML_EL_Invalid_element))
     {
	elemCour = TtaGetFirstChild (elem);
	while (elemCour != NULL)
	  {
	     BuildStructureTree (elemCour, doc, added, maxdepth, depth);
	     TtaNextSibling (&elemCour);
	  }
     }
}

/*----------------------------------------------------------------------
    pattern matching functions
  ----------------------------------------------------------------------*/

/*----------------------------------------------------------------------
   IntersectMatch: builds the symbol list res, intersection of symbol list LS with the 
    symbols contained in matching relation list MS
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void         IntersectMatch (strListSymb ** res, strListSymb * LS, strMatch * MS)
#else
static void         IntersectMatch (res, LS, MS)
strListSymb          **res;
strListSymb           *LS;
strMatch              *MS;

#endif
{
   strListSymb        *pLS, *pRes;
   strMatch           *pMS;
   ThotBool            found;

   if (*res != NULL)
     {
	FreeList (*res);
	*res = NULL;
     }
   pRes = NULL;
   pLS = LS;
   while (pLS != NULL)
     {
	pMS = MS;
	found = FALSE;
	while (!found && pMS != NULL)
	  {
	     found = (pMS->MatchSymb == pLS->Symbol);
	     pMS = pMS->Next;
	  }
	if (found)
	  {
	     if (pRes == NULL)
	       {
		  *res = (strListSymb *) TtaGetMemory (sizeof (strListSymb));
		  pRes = *res;
	       }
	     else
	       {
		  pRes->Next = (strListSymb *) TtaGetMemory (sizeof (strListSymb));
		  pRes = pRes->Next;
	       }
	     pRes->Next = NULL;
	     pRes->Symbol = pLS->Symbol;
	  }
	pLS = pLS->Next;
     }
}




/*----------------------------------------------------------------------
  initialistion of pattern matching stack
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void InitMatchStack (void)
#else
static void InitMatchStack ()
#endif
{
   int i;

   for (i = 0; i < MAX_STACK; i++)
     {
	matchStack[i].Node = NULL;
	matchStack[i].Symbols = NULL;
     }
    topMatchStack = 0;
}
/*----------------------------------------------------------------------
  push a node of source tree with the symbols it matches in  pattern matching stack
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void PushMatchStack (strNode * t, strListSymb * v)
#else
static void PushMatchStack (t, v)
strNode     *t;
strListSymb *v;
#endif
{
   if (topMatchStack < MAX_STACK)
     {
	matchStack[topMatchStack].Node = t;
	matchStack[topMatchStack++].Symbols = v;
     }
   else
     {
	printf ("Matching stack overflow \n");
     }
}


#ifdef __STDC__
static void PopMatchStack (strNode ** t, strListSymb ** v)
#else
static void PopMatchStack (t, v)
strNode     **t;
strListSymb **v;
#endif
{
   if (topMatchStack > 0)
     {
	*t = matchStack[--topMatchStack].Node;
	*v = matchStack[topMatchStack].Symbols;
	matchStack[topMatchStack].Node = NULL;
	matchStack[topMatchStack].Symbols = NULL;
     }
   else
     {
	*t = NULL;
	*v = NULL;
     }
}

/*----------------------------------------------------------------------
  ConstListMatch: adds the symbol symb to the list of symbols matched with node.
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void ConstListMatch (strNode * node, strSymbDesc * symb)
#else
static void ConstListMatch (node, symb)
strNode            *node;
strSymbDesc        *symb;
#endif
{
   strMatch           *sm;
   strMatchChildren   *smc;
   strNode            *n;
   strListSymb           *dl;

   sm = node->Matches;
   if (sm == NULL)
     {
	node->Matches = (strMatch *) TtaGetMemory (sizeof (strMatch));
	sm = node->Matches;
     }
   else
     {
	while (sm->Next != NULL && sm->MatchSymb != symb)
	   sm = sm->Next;
	if (sm->MatchSymb != symb)
	  {
	     sm->Next = (strMatch *) TtaGetMemory (sizeof (strMatch));
	     sm = sm->Next;
	  }
	else
	   return;
     }
   sm->MatchSymb = symb;
   sm->MatchNode = node;
   sm->Next = NULL;
   sm->MatchChildren = NULL;
   PopMatchStack (&n, &dl);
   while (n != NULL)
     {
	smc = (strMatchChildren *) TtaGetMemory (sizeof (strMatchChildren));
	smc->Next = sm->MatchChildren;
	sm->MatchChildren = smc;
	smc->MatchSymb = dl->Symbol;
	smc->MatchNode = n;
	PopMatchStack (&n, &dl);
     }
}

/*----------------------------------------------------------------------
  ChildrenMatch: return TRUE if the chidren of node n are matched with the possible 
   children of the pattern symbol p.
   ----------------------------------------------------------------------*/

#ifdef __STDC__
static ThotBool ChildrenMatch (strNode * n, strSymbDesc * p)
#else
static ThotBool ChildrenMatch (n, p)
strNode     *n;
strSymbDesc *p;
#endif
{
   strNode            *child;
   strListSymb        *candidate, *ms;
   ThotBool            matchFound, matchFailed;

   matchFound = FALSE;
   matchFailed = FALSE;

   candidate = p->Children;
   if (candidate == NULL)
     /* there is no possible children for p, returns true */
     {
	ConstListMatch (n, p);
	return TRUE;
     }
   if (n->Child == NULL)
     /* n is a leaf */
     {
	if (p->IsOptChild)
	  { /* possible children of p are all optional: returns true */
	     ConstListMatch (n, p);
	     return TRUE;
	  }
	else
	   return FALSE;
     }
   /* looking for a matching of each child of symbol p (candidate) with the children */
   /* of the source node n */ 
   child = n->Child;
   IntersectMatch (&(child->Candidates), p->Children, child->Matches);
   candidate = child->Candidates;
   if (candidate == NULL)
     {
	return FALSE;
     }
   InitMatchStack ();
   while (!matchFailed && !matchFound)
     {
	if (child == NULL && (candidate != NULL && candidate->Symbol == NULL))
	  {
	    /* all the children of n are matched, add a matching for n and p */
	     ConstListMatch (n, p);
	     matchFound = TRUE;
	  }
	else if (child == NULL || candidate == NULL || candidate->Symbol == NULL)
	  {  
	    /* the children are not all matched, pop the matching stack to see if an other */
	    /* symbol can be matched with a node already explored */
	     PopMatchStack (&child, &ms);
	     while (child != NULL &&
		    ms->Next == NULL)
		PopMatchStack (&child, &ms);
	     if (child == NULL)
	       {
		  matchFailed = TRUE;
		  /* no matching of n has been found, exiting */
	       }
	     else
	       {/* a new candidate to matching is aviable */
		  candidate = ms->Next;
	       }
	  }
	else
	  {			/* here:  child!=NULL && candidate != -1 */
	    /* searches if the next siblings of child can be matched with successors */
	    /* of candidate */

	     /* push child and candidate */
	     PushMatchStack (child, candidate);
	     if (child->Next != NULL)
	       {
		  /* build the candidate list for matching of the next sibling */
		  IntersectMatch (&(child->Next->Candidates),
				  candidate->Symbol->Followings,
				  child->Next->Matches);
		  /* try the first candidate */
		  candidate = child->Next->Candidates;
	       }
	     else
	       {		/* child->Next==NULL */
		  ms = candidate->Symbol->Followings;
		  while (ms != NULL && ms->Symbol != NULL)
		     ms = ms->Next;
		  if (ms != NULL)
		     candidate = ms;
	       }
	     child = child->Next;
	  }
     }
   
   if (!matchFound && p->IsOptChild)
     {
	ConstListMatch (n, p);
	matchFound = TRUE;
     }
   return matchFound;
}

/*----------------------------------------------------------------------
  check if elem actually has mandatory attributes of the pattern symbol pSymb
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static ThotBool     MatchAttributes (strSymbDesc * pSymb, Element elem)
#else
static ThotBool     MatchAttributes (pSymb, elem)
strSymbDesc           *pSymb;
Element             elem;

#endif
{
   ThotBool            result;
   strAttrDesc        *pAttr = NULL;
   AttributeType       AttrTyp;
   Attribute           attr = NULL;
   STRING              buf;
   int                 AttrKind, length;

   buf = TtaAllocString (MAX_LENGTH);
   AttrTyp.AttrSSchema = TtaGetElementType (elem).ElSSchema;
   result = TRUE;
   pAttr = pSymb->Attributes;
   while (pAttr != NULL && result)
     {
	AttrTyp.AttrTypeNum = pAttr->ThotAttr;
	if (AttrTyp.AttrTypeNum != 0)
	  {
	     attr = TtaGetAttribute (elem, AttrTyp);
	     result = (attr != (Attribute) NULL);
	  }
	if (result)
	  {
	     TtaGiveAttributeType (attr, &AttrTyp, &AttrKind);
	     if (AttrKind == 2 && !pAttr->IsInt)
	       {
		  length = MAX_LENGTH;
		  TtaGiveTextAttributeValue (attr, buf, &length);
		  result = !ustrcmp (pAttr->TextVal, buf);
	       }
	     else if (AttrKind != 2 && pAttr->IsInt)
	       {
		  result = (TtaGetAttributeValue (attr) == pAttr->IntVal);
	       }
	     else
		result = FALSE;
	  }
	pAttr = pAttr->Next;
     }
   TtaFreeMemory (buf);
   return result;
}


/*----------------------------------------------------------------------
   MatchNode : builds a pointer list to symbols that match with the node n
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static ThotBool     MatchNode (strNode * n)
#else
static ThotBool     MatchNode (n)
strNode            *n;

#endif
{
   strTransDesc          *td;
   strSymbDesc           *sd;

   td = CourTransSet->Transformations;
   while (td != NULL)
     {
	if (td->IsActiveTrans)
	  {
	     sd = td->Symbols;
	     /* for each symbol having the same tag than the node n, seaches if */
	     /* the children of the node can be matched with those of the pattern symbol */
	     while (sd != NULL)
	       {
		  if (sd->IsActiveSymb)
		     if ((n->NodeDepth - sd->Depth <= maxSelDepth)
			 && (n->NodeDepth - sd->Depth >= 0))
			if (!ustrcmp (sd->Tag, TEXT("*")) || !ustrcmp (sd->Tag, n->Tag))
			   if (sd->Attributes == NULL || MatchAttributes (sd, n->Elem))
			      ChildrenMatch (n, sd);
		  sd = sd->Next;
	       }
	     ChildrenMatch (n, td->RootDesc);
	  }
	td = td->Next;
     }
   return TRUE;
}

/*----------------------------------------------------------------------
  PostfixSearch:  Postfix search in source tree t applying a binary function f to every node . 
  the search end if f(n) = False, or if the entiere tree has been traversed.
  returns the node beeing tested if failure, NULL if the entiere tree has been searched.
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static strNode *RecursivePostfixSearch (StructureTree t, ThotBool (*f) (strNode *))
#else
static strNode *RecursivePostfixSearch (t, f)
StructureTree t;
ThotBool      (*f) (strNode *);
#endif
{
   strNode *n, *res;

   n = t->Child;
   while (n != NULL)
     {
	res = RecursivePostfixSearch (n, f);
	if (res == NULL)
	  {
	     if (f (n))
		n = n->Next;
	     else
		return n;
	  }
	else
	   return res;
     }
   return NULL;
}

#ifdef __STDC__
static strNode *PostfixSearch (StructureTree t, ThotBool (*f) (strNode *))
#else
static strNode *PostfixSearch (t, f)
StructureTree t;
ThotBool      (*f) (strNode *);
#endif
{
   strNode *res;

   res = RecursivePostfixSearch (t, f);
   if (res == NULL)
     {
	if (f (t))
	   return NULL;
	else
	   return t;
     }
   else
      return res;
}

/*----------------------------------------------------------------------
 AddListSubTree:  inserts the reference to a structure subtree, with its rank between its sibling,
 and its identifier in the environment's subtrees list
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void AddListSubTree (Element elem, int idf, int rank)
#else
static void AddListSubTree (elem, idf, rank)
Element             elem;
int                 idf;
int                 rank;

#endif
{
   strListElem           *pcour, *pprec;

   pcour = strMatchEnv.ListSubTrees;
   pprec = NULL;
   while (pcour != NULL && (pcour->Id < idf || (pcour->Id == idf && pcour->Rank < rank)))
     {
	pprec = pcour;
	pcour = pcour->Next;
     }
   if (pprec != NULL)
     {
	pprec->Next = (strListElem *) TtaGetMemory (sizeof (strListElem));
	pprec = pprec->Next;
     }
   else
     {
	strMatchEnv.ListSubTrees = (strListElem *) TtaGetMemory (sizeof (strListElem));
	pprec = strMatchEnv.ListSubTrees;
     }
   pprec->Id = idf;
   pprec->Rank = rank;
   pprec->Elem = elem;
   pprec->Next = pcour;
}

/*----------------------------------------------------------------------
  FindListSTreeByLabel
  Searches an element identified with a given label in the environement
  subtrees list.
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static Element      FindListSTreeByLabel (STRING label)
#else
static Element      FindListSTreeByLabel (label)
STRING              label;

#endif
{
   strListElem        *pcour;
   ThotBool            found;

   found = FALSE;
   pcour = strMatchEnv.ListSubTrees;
   while (!found && pcour != NULL)
     {
	found = !ustrcmp (TtaGetElementLabel (pcour->Elem), label);
	if (!found)
	   pcour = pcour->Next;
     }
   if (!found)
      return NULL;
   else
      return pcour->Elem;
}

/*----------------------------------------------------------------------
   search for the next element with the identifier id in the ListSubTrees, 
   searches the first one if *elem =NULL 
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static int          FindListSubTree (int id, Element * elem)
#else
static int          FindListSubTree (id, elem)
int                 id;
Element            *elem;

#endif
{
   strListElem        *pcour;
   ThotBool            found;
   int                 result;

   found = FALSE;
   pcour = strMatchEnv.ListSubTrees;
   while (!found && pcour != NULL)
     {
	found = (pcour->Id == id && (*elem == NULL || *elem == pcour->Elem));
	if (!found)
	   pcour = pcour->Next;
     }
   if (found && *elem == NULL)
     {
	*elem = pcour->Elem;
	result = pcour->Rank;
     }
   else
     {
	*elem = NULL;
	result = 0;
	if (found)
	  {
	     pcour = pcour->Next;
	     found = FALSE;
	     while (!found && pcour != NULL)
	       {
		  found = pcour->Id == id;
		  if (!found)
		     pcour = pcour->Next;
	       }
	     if (found)
	       {
		  *elem = pcour->Elem;
		  result = pcour->Rank;
	       }
	  }
     }
   return result;
}

/*----------------------------------------------------------------------
   ExportSubTree Exports a subtree in generation buffer without any
   change
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static ThotBool ExportSubTree (Element subTree, Document doc)
#else 
static ThotBool ExportSubTree (subTree, doc)
Element subTree;
Document doc;
#endif
{
  CHAR_T	      tmpfilename[25];
  CHAR_T	      charRead;
  FILE		     *inputFile = NULL;
  int                 len;
  ThotBool	      result = FALSE;
  struct stat        *StatBuffer;
  int		      status;

  len = BUFFER_LEN - szHTML;
  ustrcpy (tmpfilename, TtaGetEnvString ("TMPDIR"));
  ustrcat (tmpfilename, DIR_STR);
  ustrcat (tmpfilename, TEXT("amayatrans.tmp"));
  TtaExportTree (subTree, doc, tmpfilename, TEXT("HTMLT"));     
  StatBuffer = (struct stat *) TtaGetMemory (sizeof (struct stat));
  status = ustat (tmpfilename, StatBuffer);
  if (status != -1)
    if (StatBuffer->st_size < len)
      inputFile = TtaReadOpen (tmpfilename);
  charRead = EOS;
  if (inputFile != NULL)
    {
      charRead = getc (inputFile);
      while (charRead != EOF && szHTML < BUFFER_LEN - 1)
	{
	  bufHTML[szHTML++] = charRead;
	  charRead = getc (inputFile);
	}
    }
  TtaReadClose (inputFile);  
  TtaFileUnlink (tmpfilename);
  TtaFreeMemory (StatBuffer);
  if (charRead == EOF)
    result = TRUE;
  bufHTML[szHTML] = EOS;
  return result;
}
      

/*----------------------------------------------------------------------
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void RemoveHTMLTree (Element elem, Document doc)
#else
static void RemoveHTMLTree (elem, doc)
Element elem;
Document doc;
#endif
{
  Element parent;
  SSchema parSch, elSch;
  
  elSch = (TtaGetElementType (elem)).ElSSchema;
  parent = TtaGetParent (elem);
  if (parent !=NULL)
    {
      if (TtaGetParent(parent)!=NULL)
	parSch = (TtaGetElementType (TtaGetParent(parent))).ElSSchema;
      else
	parSch = (TtaGetElementType (parent)).ElSSchema;
      
      while (parent != NULL && 
	     TtaSameSSchemas (parSch, elSch) &&
	     (ustrcmp (GITagName (parent), TEXT("???")) == 0) &&
	     TtaGetFirstChild (parent) == TtaGetLastChild (parent))
	{
	  elem = parent;
	  parent = TtaGetParent (parent);
	  if (parent !=NULL)
	    if (TtaGetParent(parent)!=NULL)
	      parSch = (TtaGetElementType (TtaGetParent(parent))).ElSSchema;
	    else
	      parSch = (TtaGetElementType (parent)).ElSSchema;
	}
    }
  TtaRegisterElementDelete (elem, doc) ;
  TtaRemoveTree (elem, doc);  
}

/*----------------------------------------------------------------------
   StartHtmlParser initializes  parsing environement in order to parse
   the HTML fragment in buffer in the context of a last descendance of the
   element ElFather 
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static ThotBool StartHtmlParser (strMatchChildren * sMatch, Document doc)
#else
static ThotBool StartHtmlParser (sMatch, doc)
strMatch           *sMatch;
Document            doc;
#endif
{
   strMatchChildren   *prevMatch, *DMatch;
   Element             courEl, invEl;
   ElementType         typeEl;
   ThotBool            res;
   SSchema             selSch, courSch;
 
 res = TRUE;
   prevMatch = NULL;
   DMatch = sMatch;
   myFirstSelect = DMatch->MatchNode->Elem;
   /* init an history sequence */
   TtaOpenUndoSequence (doc, myFirstSelect, myLastSelect, 0, 0);
   selSch = (TtaGetElementType (myFirstSelect)).ElSSchema;
   courEl = myFirstSelect;
   /* PreviousHTMLSibling (&myFirstSelect);*/
   TtaPreviousSibling (&myFirstSelect);
   isClosed = TRUE;
   
   while (myFirstSelect == NULL)
     {
       courEl = TtaGetParent (courEl);
       if (TtaGetParent (courEl) != NULL)
	 courSch = (TtaGetElementType (TtaGetParent (courEl))).ElSSchema;
       else
	 courSch = (TtaGetElementType (courEl)).ElSSchema;
       if (ustrcmp (GITagName (courEl), TEXT("???")) == 0 && 
	   TtaSameSSchemas (courSch,selSch))
	 {
	   myFirstSelect = courEl;
	   TtaPreviousSibling (&myFirstSelect);
	   isClosed = TRUE;
	 }
       else
	 {
	   myFirstSelect = courEl;
	   isClosed = FALSE;
	 }
     }
       
   /*   if (myFirstSelect == NULL)
     { 
        myFirstSelect = DMatch->MatchNode->Parent->Elem;
        isClosed = FALSE;
     }
    */
   
   while (DMatch != NULL)
     {
	prevMatch = DMatch;
	DMatch = DMatch->Next;
	if (DMatch != NULL)
	  RemoveHTMLTree (prevMatch->MatchNode->Elem, doc);
     }
   if (prevMatch != NULL)
     {
	myLastSelect = prevMatch->MatchNode->Elem;
	TtaNextSibling (&myLastSelect);
	RemoveHTMLTree (prevMatch->MatchNode->Elem, doc);
     }
   if (ustrcmp (bufHTML, _EMPTYSTR_))
     {
#ifdef AMAYA_DEBUG
       printf("%s\n\n",bufHTML);
#endif
	TtaSetStructureChecking (0, doc);
	ParseSubTree (bufHTML, myFirstSelect, isClosed, doc);
	courEl = myFirstSelect;
	if (isClosed)
	  TtaNextSibling (&courEl);
	else
	  courEl = TtaGetFirstChild (courEl);
	while (courEl != NULL && courEl != myLastSelect)
	  {	    
	    TtaRegisterElementCreate (courEl, doc);
	    TtaNextSibling (&courEl);
	  }
	TtaCloseUndoSequence (doc);
	TtaSetStructureChecking (1, doc);
	typeEl.ElSSchema = TtaGetDocumentSSchema (doc);
	typeEl.ElTypeNum = HTML_EL_Invalid_element;
	invEl = NULL;

	/* checks if invalid elements have been generated */
	/* (invalid transformation) */
	if (isClosed)
	  {
	     courEl = myFirstSelect;
	     TtaNextSibling (&courEl);
	     invEl = TtaSearchTypedElement (typeEl, SearchForward, courEl);
	     if (invEl != NULL)
	       {
		  if ((!myLastSelect && 
		       TtaIsAncestor (invEl, TtaGetParent (myFirstSelect)))
		  || ((myLastSelect && 
		       TtaIsBefore (invEl, myLastSelect))))
		    {		/* if the transformation failed */
		       res = FALSE;
		       TtaUndoNoRedo (doc);
		    }
	       }
	  }
	else
	  {
	     invEl = TtaSearchTypedElement (typeEl, SearchInTree, myFirstSelect);
	     if (invEl != NULL)
	       {
		  if (myLastSelect == NULL || TtaIsBefore (invEl, myLastSelect))
		    {		/* if the transformation failed */
		       res = FALSE;
		       TtaUndoNoRedo (doc);
		    }
	       }
	  }
     }
   return res;
}

/*----------------------------------------------------------------------
  functions for transfert of non-matched descendants of matched nodes for source structure
  tree to the result HTML instance.
  ----------------------------------------------------------------------*/
/*----------------------------------------------------------------------
  FlattenAndInsertElement: tries to fllaten the structure sub-tree rooted elSource in order to 
  insert it as the first child of elDest (if sibling is false) or as its immediate right sibling (if 
  sibling is true) returns the number of higer-level nodes created in nbCreated and the 
  rightmost of them in lastCreated.
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static ThotBool   FlattenAndInsertElement (Element elSource, Element elDest, ThotBool sibling, Document doc, Element * lastCreated, int *nbCreated)
#else
static ThotBool   FlattenAndInsertElement  (elSource, elDest, sibling, doc, lastCreated, nbCreated)
Element             elSource;
Element             elDest;
ThotBool            sibling;
Document            doc;
Element            *lastCreated;
int                *nbCreated;

#endif
{
   Element             elCour, elPreced, elNext;
   int                 nbc;
   ThotBool            res;

   *nbCreated = 0;
   if (TtaIsLeaf (TtaGetElementType (elSource)))
     {			
	/* if the source element is a leaf, it is impossible to flatten it more */
	lastCreated = NULL;
	return FALSE;
     }
   res = TRUE;
   elCour = TtaGetFirstChild (elSource);
   elPreced = NULL;
   if (sibling)
      elPreced = elDest;
   while (elCour != NULL && res)
     {		
       	/* for each child of the source element */
	elNext = elCour;
	TtaNextSibling (&elNext);
	TtaRemoveTree (elCour, doc);
	if (elPreced != NULL)
	   TtaInsertSibling (elCour, elPreced, FALSE, doc);
	else
	   TtaInsertFirstChild (&elCour, elDest, doc);
	res = !(ThotBool) TtaGetErrorCode ();
	if (res)
	  {
	     elPreced = elCour;
	     *lastCreated = elCour;
	     (*nbCreated)++;
	  }
	else
	  {
	     if (elPreced != NULL)
		res =  FlattenAndInsertElement(elCour, elPreced, TRUE, doc, lastCreated, &nbc);
	     else
		res =  FlattenAndInsertElement(elCour, elDest, FALSE, doc, lastCreated, &nbc);
	     elPreced = *lastCreated;
	     (*nbCreated) += nbc;
	  }
	elCour = elNext;
     }
   if (!res && TtaIsConstant (TtaGetElementType (elSource)))
      /* we don't mind about failure when inserting constants */
      return TRUE;
   else
      return res;
}
/*----------------------------------------------------------------------
   RankedInsertion


   Inserts an element as a given child or sibling of an element, creating a descendance 
   between the two elements if necesary. 

   Parameters:
   newEl: the element to be inserted
   parent: the element in which newEl will be inserted
   prevEl : the element after which newEl will be inserted
   rank:  rank of newEl 
   doc: document in which newwEl have to be inserted
  ----------------------------------------------------------------------*/

#ifdef __STDC__
static ThotBool     RankedInsertion (Element newEl, Element parent, Element prevEl, int *rank, Document doc)
#else
static ThotBool     RankedInsertion (newEl, parent, prevEl, *rank, doc)
Element             newEl;
Element             parent;
Element             prevEl;
int                 rank;
Document            doc;

#endif
{
   ElementType         elemTypeNew;
   Element             elFirst, elLast, elPrevious, elCour, elem;
   int                 i;
   ThotBool            result;


   if (prevEl == NULL && parent == NULL)
      return FALSE;
   result = TRUE;
   TtaSetErrorMessages (0);

   elCour = NULL;
   if (*rank > 1 || prevEl != NULL)
     {
	/* if the element have not to be inserted as the first child of parent */
	if (prevEl == NULL)
	  {
	     elCour = TtaGetFirstChild (parent);
	     i = 2;
	  }
	else
	  {
	     elCour = prevEl;
	     i = 1;
	  }
	/* search for the previous sibling of the inserted element */
	for (; i < *rank && elCour != NULL; i++)
	   TtaNextSibling (&elCour);
	/* if there is not enough children of parent, */
	/*insert element as the next sibling of the last child of parent */
	if (elCour == NULL && parent != NULL)
	   elCour = TtaGetLastChild (parent);
	else if (elCour == NULL)
	   elCour = TtaGetLastChild (TtaGetParent (prevEl));
     }
   *rank = 1;
   if (elCour != NULL)
     {
	/* the newEl has to be inserted as the next sibling of elCour */
	TtaInsertSibling (newEl, elCour, FALSE, doc);
     }
   else
      /* the newEl has to be inserted as the first child of parent */
      TtaInsertFirstChild (&newEl, parent, doc);

   if (TtaGetErrorCode ())
     {
	/* the newEl has not been inserted */
	/* create a descendance between the parent and the newEl */
	elemTypeNew = TtaGetElementType (newEl);
	if (parent == NULL)
	   parent = TtaGetParent (elCour);
	elLast = TtaCreateDescent (doc, parent, elemTypeNew);
	if (elLast != NULL && !TtaGetErrorCode ())
	  {
	     /* A descendance has been created */
	     elFirst = TtaGetFirstChild (parent);
	     while (elFirst != NULL && !TtaIsAncestor (elLast, elFirst))
	       {
		  TtaNextSibling (&elFirst);
	       }
	     if (elFirst != NULL)
	       {
		  TtaRemoveTree (elFirst, doc);
		  if (elCour != NULL)
		     TtaInsertSibling (elFirst, elCour, FALSE, doc);
		  else
		     TtaInsertFirstChild (&elFirst, parent, doc);
		  if (TtaGetErrorCode ())
		    {
		       TtaDeleteTree (elFirst, doc);
		       result = FALSE;
		    }
		  else
		    {
		       elPrevious = TtaGetParent (elLast);
		       TtaDeleteTree (elLast, doc);
		       TtaInsertFirstChild (&newEl, elPrevious, doc);
		       if (TtaGetErrorCode ())
			 {
			    TtaDeleteTree (elFirst, doc);
			    result = FALSE;
			 }
		    }
	       }
	  }
	else
	   result = FALSE;
     }
   if (result == FALSE)
      /* impossible to create a descendance to the element */
      /* trying to flatten the element to be transferred */
      /* this is the worst case */
      if (elCour == NULL)
	 result = FlattenAndInsertElement (newEl, parent, FALSE, doc, &elem, rank);
      else
	 result = FlattenAndInsertElement (newEl, elCour, TRUE, doc, &elem, rank);
   return result;
}

/*----------------------------------------------------------------------
 CopySubTreeChildren: copies the descendants of matched nodes from source tree to result 
  instance
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void         CopySubTreeChildren (Element father, Document doc)
#else
static void         CopySubTreeChildren (father, doc)
Element             father;
Document            doc;

#endif
{
   Element             elCour, elOriginal;
   AttributeType       attrType;
   Attribute           attrFound;
   CHAR_T                label[10];
   int                 l, rank, idf, delta;

   attrType.AttrSSchema = TtaGetDocumentSSchema (doc);
   attrType.AttrTypeNum = HTML_ATTR_Ghost_restruct;
   elCour = TtaGetFirstChild (father);
   while (elCour != NULL)
     {
	CopySubTreeChildren (elCour, doc);
	attrFound = TtaGetAttribute (elCour, attrType);
	if (attrFound != NULL)
	  {
	     l = 10;
	     TtaGiveTextAttributeValue (attrFound, label, &l);
	     TtaRemoveAttribute (elCour, attrFound, doc);
	     idf = uctoi (label);
	     elOriginal = NULL;
	     rank = FindListSubTree (idf, &elOriginal);
	     delta = 0;
	     while (elOriginal != NULL)
	       {
		  TtaRemoveTree (elOriginal, doc);
		  RankedInsertion (elOriginal, elCour, NULL, &rank, doc);
		  delta += rank - 1;
		  rank = delta + FindListSubTree (idf, &elOriginal);
	       }
	  }
	TtaNextSibling (&elCour);
     }
}

/*----------------------------------------------------------------------
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void         CopySubTrees (strNode * Root, Document doc)
#else
static void         CopySubTrees (Root, doc)
strNode            *Root;
Document            doc;

#endif
{
   CHAR_T                label[10];
   int                 l, idf, rank, delta;
   AttributeType       attrType;
   Attribute           attrFound;
   Element             elCour, elLast, elOriginal;

   elCour = myFirstSelect;	
   elLast = myLastSelect;	
   if (isClosed)
      TtaNextSibling (&elCour);
   else
      elCour = TtaGetFirstChild (myFirstSelect);
   attrType.AttrSSchema = TtaGetDocumentSSchema (doc);
   attrType.AttrTypeNum = HTML_ATTR_Ghost_restruct;
   while (elCour != NULL && elCour != elLast)
     { 	/* Copies the children of elCour into the result instance */		
	CopySubTreeChildren (elCour, doc);
	/* searches for a ZZGHOST ettribute */
	attrFound = TtaGetAttribute (elCour, attrType);	
	if (attrFound != NULL)
	  {
	     l = 10;
	     TtaGiveTextAttributeValue (attrFound, label, &l);
	     idf = uctoi (label);
	     if (idf != 0)
	       {
		 TtaRemoveAttribute (elCour, attrFound, doc);
		 elOriginal = NULL;
		 /* serches and inserts the children of element elOriginal */
		 rank = FindListSubTree (idf, &elOriginal);
		 delta = 0;
		 while (elOriginal != NULL)
		   {
		     TtaRemoveTree (elOriginal, doc);
		     RankedInsertion (elOriginal, elCour, NULL, &rank, doc);
		     delta += rank - 1;
		     rank = delta + FindListSubTree (idf, &elOriginal);
		   }
	       }
	  }
	TtaNextSibling (&elCour);
     }
   /* insert the subtrees as children of the root */
   elOriginal = NULL;
   elCour = myFirstSelect;
   rank = FindListSubTree (0, &elOriginal);
   delta = 0;
   while (elOriginal != NULL)
     {
	TtaRemoveTree (elOriginal, doc);
	if (isClosed)
	   RankedInsertion (elOriginal, NULL, elCour, &rank, doc);
	else
	   RankedInsertion (elOriginal, elCour, NULL, &rank, doc);
	delta += rank - 1;
	rank = delta + FindListSubTree (0, &elOriginal);
     }
}

/*----------------------------------------------------------------------
   PutBeginTag,PutEndTag,TransfertNode 
   fill the HTML buffer with the generated nodes 
  ----------------------------------------------------------------------*/

#ifdef __STDC__
static ThotBool     PutInHtmlBuffer (STRING s)
#else
static ThotBool     PutInHtmlBuffer (s)
STRING              s;

#endif
{
  int len;
  
  len = ustrlen (s);
  if ((szHTML + len) >= BUFFER_LEN )
    {
      fprintf (stderr, "increase BUFFER_LEN");
      return FALSE;
    }
  else
    {
      szHTML += len;
      ustrcat (bufHTML, s);
      return TRUE;
    }
}

/*----------------------------------------------------------------------
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static ThotBool        PutBeginTag (strNodeDesc * ND, strNode * TN)
#else
static ThotBool        PutBeginTag (ND, TN)
strNodeDesc           *ND;
strNode            *TN;

#endif
{
  ElementType		elType;
  strAttrDesc           *AD;
  strGenStack          *NS;
  STRING              attrValue, tag;
  strNode            *ancestor;
  ThotBool            found;
  AttributeType       attrType;
  Attribute           attr;
  int                 l, attrKind;
  ThotBool	      res = TRUE;
  
  attrType.AttrSSchema = TtaGetDocumentSSchema (TransDoc);
  attrValue = TtaAllocString (NAME_LENGTH);
  tag = TtaAllocString (NAME_LENGTH);
  /* push the new tag on the generation stack */
  generationStack[topGenerStack]->Nbc++;
  NS = (strGenStack *) TtaGetMemory (sizeof (strGenStack));
  NS->Tag = TtaAllocString (NAME_LENGTH);
  ustrcpy (NS->Tag, ND->Tag);
  GIType (NS->Tag, &elType, TransDoc);
 
#ifdef MATHML
  if (elType.ElSSchema == NULL)
    {
      /*specifique a MathML */
      elType.ElSSchema = TtaGetSSchema (TEXT("MathML"), TransDoc);
    }
#endif

  NS->Idf = idfCounter++;
  NS->Nbc = 0;
  if (TransferMode == ByAttribute)
    {
      /* create a ghost attribute with the identifier of the node */     
      NS->Attributes = (strAttrDesc *) TtaGetMemory (sizeof (strAttrDesc));
      NS->Attributes->NameAttr = TtaAllocString (NAME_LENGTH);
      ustrcpy (NS->Attributes->NameAttr, TEXT("ZZGHOST"));
      NS->Attributes->IsInt = TRUE;
      NS->Attributes->IsTransf = FALSE;
      NS->Attributes->IntVal = NS->Idf;
      NS->Attributes->Next = ND->Attributes;
    }
  else
    {
      NS->Attributes = ND->Attributes;
    }
  generationStack[++topGenerStack] = NS;

  /* writing the tag name */
  res = res && PutInHtmlBuffer (TEXT("<"));
  res = res && PutInHtmlBuffer (NS->Tag);

  AD = NS->Attributes;
  /* wrting the attributes */
  while (AD != NULL)
    {
      if (AD->IsTransf)
	{			/* transfer attribute */
	  ancestor = TN;
	  found = FALSE;
	  while (!found && ancestor != NULL)
	    {		/* searching for source element (in current element ancestors) */
	      if (ancestor->MatchSymb != NULL)
		found = (!ustrcmp (ancestor->MatchSymb->SymbolName, AD->AttrTag)
			 || !ustrcmp (ancestor->MatchSymb->Tag, AD->AttrTag));
	      else
		found = (!ustrcmp (ancestor->Tag, AD->AttrTag));
	      if (!found)
		ancestor = ancestor->Parent;
	    }
	  if (found)
	    {		/* searching for an ancestor of the source element which have the wanted attribute  */
	      if (ancestor != NULL)
		{
		  ustrcpy (tag, GITagNameByType (TtaGetElementType (ancestor->Elem)));
		  attrType.AttrTypeNum = MapThotAttr (AD->AttrAttr, tag);
		}
	      attr = NULL;
	      found = FALSE;
	      while (!found && ancestor != NULL)
		{
		  if (attrType.AttrTypeNum != -1)
		    attr = TtaGetAttribute (ancestor->Elem, attrType);
		  found = (attr != NULL);
		  if (!found)
		    {
		      ancestor = ancestor->Parent;
		      if (ancestor != NULL)
			{
			  ustrcpy (tag, GITagNameByType (TtaGetElementType (ancestor->Elem)));
			  attrType.AttrTypeNum = MapThotAttr (AD->AttrAttr, tag);
			}
		    }
		}
	      if (found)
		{		/* the attribute has been found, writing the attribute name */
		  res = res && PutInHtmlBuffer (TEXT(" "));
		  res = res && PutInHtmlBuffer (AD->AttrAttr);
		  res = res && PutInHtmlBuffer (TEXT("="));
		  /* writing the attribute value */
		  TtaGiveAttributeType (attr, &attrType, &attrKind);
		  if (attrKind == 2)
		    {	/* text attribute */
		      l = TtaGetTextAttributeLength (attr);
		      TtaGiveTextAttributeValue (attr, attrValue, &l);
		      res = res && PutInHtmlBuffer (attrValue);
		    }
		  else
		    {	/* int attribute */
		      usprintf (attrValue, TEXT("%d"), TtaGetAttributeValue (attr));
		      res = res && PutInHtmlBuffer (attrValue);
		    }
		}
	    }
	  if (!found)
	    {
	      fprintf (stderr, "can't transfer attribute %s\n", AD->AttrAttr);
	    }
	}
      else
	{			/* creation of an attribute */
	  res = res && PutInHtmlBuffer (TEXT(" "));
	  res = res && PutInHtmlBuffer (AD->NameAttr);
	  res = res && PutInHtmlBuffer (TEXT("="));
	  if (AD->IsInt)
	    {		/* int attribute */
	      usprintf (attrValue, TEXT("%d"), AD->IntVal);
	      res = res && PutInHtmlBuffer (attrValue);
	    }
	  else
	    {		/* text attribute */
	      /* l = ustrlen (bufHTML);
		 bufHTML[l] = '"';
		 bufHTML[l + 1] = EOS;
		 szHTML++;*/
	      res = res && PutInHtmlBuffer (TEXT("\""));
	      res = res && PutInHtmlBuffer (AD->TextVal);
	      res = res && PutInHtmlBuffer (TEXT("\""));
	      /*l = ustrlen (bufHTML);
		bufHTML[l] = '"';
		bufHTML[l + 1] = EOS;
		szHTML++;*/
	    }
	}
      AD = AD->Next;
    }
  /* closing the tag */
  res = res && PutInHtmlBuffer (TEXT(">"));
  if (TransferMode == ByAttribute)
    {
      /*free the ZZGHOST attribute */
      TtaFreeMemory ( NS->Attributes->NameAttr);
      TtaFreeMemory ( NS->Attributes);
    }
  NS->Attributes = NULL;
  TtaFreeMemory (attrValue);
  TtaFreeMemory (tag);
  return res;
}


/*----------------------------------------------------------------------
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static ThotBool PutEndTag (strGenStack * ND)
#else
static ThotBool PutEndTag (ND)
strGenStack *ND;
#endif
{
  ThotBool res = TRUE;

  if (ustrcmp (ND->Tag, TEXT("HR")) && 
      ustrcmp (ND->Tag, TEXT("BR")) &&
      ustrcmp (ND->Tag, TEXT("IMG")))
    {
      res = res && PutInHtmlBuffer (TEXT("</"));
      res = res && PutInHtmlBuffer (ND->Tag);
      res = res && PutInHtmlBuffer (TEXT(">"));
    }
  return res;
}

/*----------------------------------------------------------------------
   TransfertChildren : copies the children of node into the result instance 
  ----------------------------------------------------------------------*/

#ifdef __STDC__
static ThotBool        TransfertChildren (strNode * node)
#else
static ThotBool        TransfertChildren (strNode * node)
strNode            *node;

#endif
{
   strNode            *child;
   ElementType	       elType; 
   ThotBool	       res = TRUE;
 
   child = node->Child;
   while (child != NULL)
     {
	if (TtaGetElementVolume (child->Elem) != 0)
	  {/* if the element is empty: no transfert */
	     generationStack[topGenerStack]->Nbc++;
	     elType = TtaGetElementType (child->Elem);
		 /*ustrcmp (TtaGetSSchemaName (elType.ElSSchema), "HTML") == 0)*/
	     if (TransferMode == ByAttribute)
	       AddListSubTree (child->Elem,
			       generationStack[topGenerStack]->Idf,
			       generationStack[topGenerStack]->Nbc);
	     else
	       res = res && ExportSubTree (child->Elem, TransDoc);
	  }
	child = child->Next;
     }
   return res;
}

/*----------------------------------------------------------------------
   TransfertNode : copies a node and its content into the result instance 
  ----------------------------------------------------------------------*/

#ifdef __STDC__
static ThotBool        TransfertNode (strNode * node, ThotBool inplace)
#else
static ThotBool        TransfertNode (node, inplace)
strNode            *node;
ThotBool            inplace;

#endif
{
  ElementType elType; 
  ThotBool res = TRUE;

  if (TtaGetElementVolume (node->Elem) != 0)
    {	/* if the element is empty: no transfert */
      if (!inplace)
	/* closing previously generated elements */
	while (topGenerStack >= lastRulePlace)
	  {
	    res = res && PutEndTag (generationStack[topGenerStack]);
	    TtaFreeMemory (generationStack[topGenerStack]->Tag);
	    TtaFreeMemory ( generationStack[topGenerStack]);
	    topGenerStack--;
	  }
      elType = TtaGetElementType (node->Elem); 
      generationStack[topGenerStack]->Nbc++;
/* 	ustrcmp (TtaGetSSchemaName (elType.ElSSchema), "HTML") == 0)  */
      if (TransferMode == ByAttribute)
	AddListSubTree (node->Elem,
				     generationStack[topGenerStack]->Idf,
				     generationStack[topGenerStack]->Nbc);
      else
	res = res && ExportSubTree (node->Elem, TransDoc);
    }
  return res;
}


/*----------------------------------------------------------------------
   Fonctions de transformation par regles            
  ----------------------------------------------------------------------*/

#ifdef __STDC__
static ThotBool        TransformNode (strMatchChildren * sm);
#else
static ThotBool        TransformNode (sm);
#endif


#ifdef __STDC__
static ThotBool        ApplyTransChild (strMatchChildren * smc)
#else
static ThotBool        ApplyTransChild (smc)
strMatchChildren   *smc;

#endif
{
   strMatchChildren   *smc2;
   strMatch           *sm;
   ThotBool            found;
   ThotBool            result = TRUE;

   smc2 = smc;
   while (smc2 != NULL)
     {
	smc2->MatchNode->IsTrans = TRUE;
	if (smc2->MatchSymb->Rule == NULL)
	  {
	    /* no transformation rule for the node, searches if a child has been matched */
	     sm = smc2->MatchNode->Matches;
	     found = FALSE;
	     while (!found && sm != NULL)
	       {
		  found = (sm->MatchSymb == smc2->MatchSymb && sm->MatchChildren != NULL);
		  if (!found)
		     sm = sm->Next;
	       }
	     if (found)
	       {
		 /* at least one child has been matched,        */
		 /* applying the transformation to the children */
		  result = result && ApplyTransChild (sm->MatchChildren);
	       }
	     else
	       { /* there is no matching: transferring the node to destination instance */
		  result = result && TransfertNode (smc2->MatchNode, FALSE);
	       }
	  }
	else
	  {  /* there is a transformation rule relative to the matched symbol */
	    /* applying the rule */
	     result = result && TransformNode (smc2);
	  }
	smc2 = smc2->Next;
     }
return result;
}


/*----------------------------------------------------------------------
  TransformNode : transforms a matched node acconding to the rule relative to the 
  matched symbol
  ---------------------------------------------------------------------*/
#ifdef __STDC__
static ThotBool        TransformNode (strMatchChildren * sm)
#else
static ThotBool        TransformNode (sm)
strMatchChildren   *sm;

#endif
{
   int                 courNode, l;
   strMatch           *sm2;
   strNodeDesc        *RNodeCour;
   strRuleDesc	      *currentRule;
   ThotBool            stop, sonsMatch;
   ThotBool	       result = TRUE;
   ThotBool            transChildDone = FALSE;

   sm2 = sm->MatchNode->Matches;
   sonsMatch = FALSE;
   while (sm2 != NULL && !sonsMatch)
     {
	sonsMatch = (sm2->MatchSymb == sm->MatchSymb && sm2->MatchChildren != NULL);
	if (!sonsMatch)
	   sm2 = sm2->Next;
     }
   /* sonsMatch is true if there is at least one matched node in the children of source node */
   sm->MatchNode->MatchSymb = sm->MatchSymb;
   currentRule = sm->MatchSymb->Rule;
   while (currentRule != NULL)
     {
       if (currentRule->DeleteRule)
	 transChildDone = TRUE;
       else
	 {
	   courNode = 1;
	   RNodeCour = currentRule->OptionNodes;
	   stop = (RNodeCour == NULL || courNode > topGenerStack);
	   while (!stop)
	     { /* for each optional tag in the rule */
	       if (!ustrcmp (generationStack[courNode]->Tag, RNodeCour->Tag))
		 {  /* does nothing if the tag is already present in the destination instance */
		   RNodeCour = RNodeCour->Next;
		   courNode++;
		   stop = (RNodeCour == NULL || courNode > topGenerStack);
		 }
	       else
		 {
		   /* a new branch have to be created in the destination */
		   stop = TRUE;
		 }
	     }
	   
	   while (topGenerStack >= courNode)
	     { /* closes the opened tags (on generation stack) */
	       result = result && PutEndTag (generationStack[topGenerStack]);
	       TtaFreeMemory (generationStack[topGenerStack]->Tag);
	       TtaFreeMemory ( generationStack[topGenerStack]);
	       topGenerStack--;
	     }
	   
	   while (RNodeCour != NULL)
	     {/* generates optional nodes not already present */
	       result = result && PutBeginTag (RNodeCour, sm->MatchNode);
	       courNode++;
	       RNodeCour = RNodeCour->Next;
	     }
	   
	   lastRulePlace = courNode;
	   RNodeCour = currentRule->NewNodes;
	   
	   while (RNodeCour != NULL && 
		  RNodeCour->Tag[0] != '"' &&
		  ustrcmp (RNodeCour->Tag, TEXT("*")) != 0 &&
		  ustrcmp (RNodeCour->Tag, TEXT("#")) != 0)
	     { /* generates the new nodes */
	       result = result && PutBeginTag (RNodeCour, sm->MatchNode);
	       courNode++;
	       RNodeCour = RNodeCour->Next;
	     }

	   /* traite le dernier noeud de la regle */
	   if (RNodeCour != NULL && RNodeCour->Tag[0] == '"')
	     {
	       l = ustrlen (RNodeCour->Tag) - 2;
	       ustrncpy (&bufHTML[szHTML], &RNodeCour->Tag[1], l);
	       szHTML += l;
	       bufHTML[szHTML]='\0';
	     }
	   if (RNodeCour != NULL && RNodeCour->Tag[0] == '*')
	     { /* copie du noeud source */
	       result = result && TransfertNode (sm->MatchNode, TRUE);
	       transChildDone = TRUE;
	     }
	   if ((RNodeCour != NULL && RNodeCour->Tag[0] == '#') ||
	       (currentRule->NextRule == NULL))
	     {
	       /* it is the last rule of the node or the explicit children */
	       /* transformation place */
	       /* process the children */
	       if ((currentRule->NextRule != NULL) || transChildDone == FALSE)
		 {
		   if (sonsMatch)
		     {
		       result = result && ApplyTransChild (sm2->MatchChildren);
		     }
		   else
		     {			
		       result = result && TransfertChildren (sm->MatchNode);
		     }
		   transChildDone = TRUE;
		 }
	     }
	 }
       currentRule = currentRule->NextRule;
     }
   return result;
}

/*----------------------------------------------------------------------
  ApplyTransformation: applies the transformation based on sm matching descriptors
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static ThotBool     ApplyTransformation (strMatch * sm, Document doc)
#else
static ThotBool     ApplyTransformation (sm, doc)
strMatch           *sm;
Document            doc;

#endif
{
   strGenStack          *ND;
   strMatchChildren   *DMatch;
   ThotBool            res;

   res = FALSE;
   idfCounter = 1;
   ND = (strGenStack *) TtaGetMemory (sizeof (strGenStack));
   ND->Tag = TtaAllocString (NAME_LENGTH);
   ND->Attributes = NULL;
   ND->Idf = idfCounter++;
   ND->Nbc = 0;

   if (sm->MatchChildren != NULL)
     {
	/* initialize the transformation stack */
	DMatch = sm->MatchChildren;
	topGenerStack = 0;
	ustrcpy (ND->Tag, DMatch->MatchNode->Parent->Tag);
	generationStack[0] = ND;
	lastRulePlace = 1;
	szHTML = 0;
	bufHTML = TtaAllocString (BUFFER_LEN);
	ustrcpy (bufHTML, _EMPTYSTR_);
	
	TransferMode = InBuffer;
	/* applying the transformation */
	res = ApplyTransChild (sm->MatchChildren);
	while (topGenerStack > 0)
	  {
	     res = res && PutEndTag (generationStack[topGenerStack]);
	     TtaFreeMemory (generationStack[topGenerStack]->Tag);
	     TtaFreeMemory ( generationStack[topGenerStack]);
	     topGenerStack--;
	  }
	
	/* parsing the produced structure */
	res = res && StartHtmlParser (sm->MatchChildren, doc);

	TtaFreeMemory (bufHTML);
     }
   if (res)
     {	
       /* if the html parsing was succesful */
       /* transfers the unchanged subtrees */
       if (TransferMode == ByAttribute)
	 CopySubTrees (strMatchEnv.SourceTree, doc);
       /* deletes the source structure elements */
	DMatch = sm->MatchChildren;
	TtaSetErrorMessages (0);
	while (DMatch != NULL)
	  {
	     if (DMatch->MatchNode->Elem != NULL &&
		 FindListSTreeByLabel (TtaGetElementLabel (DMatch->MatchNode->Elem)) == NULL)
	       if (TtaGetParent (DMatch->MatchNode->Elem) != NULL)
		 TtaDeleteTree (TtaGetParent (DMatch->MatchNode->Elem), doc);
	       else
		 TtaDeleteTree (DMatch->MatchNode->Elem, doc);
	     DMatch = DMatch->Next;
	  }
	TtaSetErrorMessages (1);
     }
   TtaFreeMemory (ND);
   return res;
}

/*----------------------------------------------------------------------
   CheckSelection : checks if all the selected element are at the same level. Extends the selction  
   to an element if all its children are selected                                       
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static ThotBool     CheckSelectionLevel (Document doc)
#else
static ThotBool     CheckSelectionLevel (doc)
Document            doc;

#endif
{
   Element             prevFirst, parentFirst, nextLast;
   Element             parentLast = NULL;
   ThotBool            result;

   TtaGiveFirstSelectedElement (doc, &origFirstSelect, &ffc, &flc);
   TtaGiveLastSelectedElement (doc, &origLastSelect, &lfc, &llc);
   myFirstSelect = origFirstSelect;
   myLastSelect = origLastSelect;
   parentFirst = NULL;
   maxSelDepth = 0;
   if (myFirstSelect != myLastSelect)
     {
       /* selection crosses several elements, looks if their parents */
       /* surround the whole selection, if it is the case, */
       /* the parent is considered as a selected element */ 
	if (myFirstSelect != NULL && ffc <= 1)
	  {
	    /* searching for the first selected element */
	     prevFirst = myFirstSelect;
	     TtaPreviousSibling (&prevFirst);
	     parentFirst = TtaGetParent (myFirstSelect);
	     while (parentFirst != NULL && 
		    prevFirst == NULL && 
		    TtaIsBefore (parentFirst, myLastSelect))
	       {
		  myFirstSelect = parentFirst;
		  prevFirst = myFirstSelect;
		  TtaPreviousSibling (&prevFirst);
		  parentFirst = TtaGetParent (myFirstSelect);
	       }
	  }
	if (myLastSelect != NULL && (llc == 0 || (llc > 0 && llc >= TtaGetTextLength (myLastSelect))))
	  {
	    /* searching for the last selected element */
	     nextLast = myLastSelect;
	     TtaNextSibling (&nextLast);
	     parentLast = TtaGetParent (myLastSelect);
	     while (parentLast != NULL && 
		    nextLast == NULL && 
		    TtaIsBefore (myFirstSelect, parentLast))
	       {
		  myLastSelect = parentLast;
		  nextLast = myLastSelect;
		  TtaNextSibling (&nextLast);
		  parentLast = TtaGetParent (myLastSelect);
	       }
	  }
     }
   else if (myFirstSelect != NULL)
     {
       /* only one element is selected, check if its parent surround */
       /* the whole selection */
	prevFirst = TtaGetFirstChild (myFirstSelect);
	nextLast = TtaGetLastChild (myFirstSelect);
	while (prevFirst != NULL && prevFirst == nextLast)
	  {
	     myFirstSelect = prevFirst;
	     prevFirst = TtaGetFirstChild (myFirstSelect);
	     nextLast = TtaGetLastChild (myFirstSelect);
	  }
	if (prevFirst != NULL)
	  {
	     myFirstSelect = prevFirst;
	     myLastSelect = nextLast;
	  }
	else
	  {
	     myLastSelect = myFirstSelect;
	  }
	parentFirst = parentLast = TtaGetParent (myFirstSelect);
     }

   mySelect = NULL;
   result = (myFirstSelect != NULL && parentFirst == parentLast);
   if (result && parentFirst != NULL)
     {		
       /* if all selected elements are at the same level, */
       /* checking if ancestors have any sibling */
       /* if it is not the case, they become the first selected element */
	nextLast = myLastSelect;
	prevFirst = myFirstSelect;
	do
	   TtaNextSibling (&nextLast);
	while (nextLast != NULL && GITagName (nextLast) == NULL);
	do
	   TtaPreviousSibling (&prevFirst);
	while (prevFirst != NULL && GITagName (prevFirst) == NULL);
	while (parentFirst != NULL &&
	       ((ustrcmp (TtaGetSSchemaName (TtaGetElementType (parentFirst).ElSSchema), TEXT("HTML")) != 0) ||
		TtaGetElementType (parentFirst).ElTypeNum != HTML_EL_HTML) &&
	       nextLast == NULL && prevFirst == NULL)
	  {
	     maxSelDepth++;
	     mySelect = parentFirst;
	     parentFirst = TtaGetParent (parentFirst);
	     if (parentFirst != NULL)
	       {
		  nextLast = mySelect;
		  prevFirst = mySelect;
		  do
		     TtaNextSibling (&nextLast);
		  while (nextLast != NULL && GITagName (nextLast) == NULL);
		  do
		     TtaPreviousSibling (&prevFirst);
		  while (prevFirst != NULL && GITagName (prevFirst) == NULL);
	       }
	  }
     }
   return result;
}

/*----------------------------------------------------------------------
   MyNextSelectedElement
   Gives the next selected element, accordingly  to extension given by 
   CheckSelectionLevel 
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void         MyNextSelectedElement (Document doc, Element * elSelect)
#else
static void         MyNextSelectedElement (doc, elSelect)
Document            doc;
Element            *elSelect;

#endif
{
   Element             elFirst;
   int                 fc, lc;

   if (*elSelect == NULL || *elSelect == mySelect || *elSelect == myLastSelect)
     {				/* if the selection is an unique element, or elSelect is the last */
	*elSelect = NULL;
     }
   else
     {
	if (*elSelect == myFirstSelect)
	  {
	     TtaGiveFirstSelectedElement (doc, &elFirst, &fc, &lc);
	     if (elFirst == myFirstSelect)
		TtaGiveNextSelectedElement (doc, &elFirst, &fc, &lc);
	     else
		while (elFirst != NULL && TtaIsAncestor (elFirst, myFirstSelect))
		   TtaGiveNextSelectedElement (doc, &elFirst, &fc, &lc);
	  }
	else
	  {
	     elFirst = *elSelect;
	     TtaGiveNextSelectedElement (doc, &elFirst, &fc, &lc);
	  }
	if (elFirst != NULL && TtaIsAncestor (elFirst, myLastSelect))
	   *elSelect = myLastSelect;
	else
	   *elSelect = elFirst;
     }
}
/*----------------------------------------------------------------------
   IsValidHtmlChild(element, tag)                                       
   returns TRUE if the tag is valid as a direct descendant of an element 
   of type elType 
   WARNING This function works until there are no cycles in S schema
           whithout any HTML element inside....
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static ThotBool     IsValidHtmlChild (ElementType elemType, STRING tag, STRING prevtag)
#else
static ThotBool     IsValidHtmlChild (elemType, tag, prevtag)
Element             element;
STRING              tag;
STRING              prevtag;

#endif
{

  ElementType         elemTypeChild, tagElType, prevElType;
  int                 cardinal, i = 0;
  ElementType 	      *subTypes;
  ThotBool            result, found;
  Construct           constOfType;

  result = FALSE;
  elemTypeChild.ElSSchema = elemType.ElSSchema;
  cardinal = TtaGetCardinalOfType (elemType);
  subTypes = (ElementType *) TtaGetMemory (cardinal * sizeof (ElementType));
  TtaGiveConstructorsOfType (&subTypes, &cardinal, elemType);
  constOfType = TtaGetConstructOfType (elemType);
  GIType (tag, &tagElType, TransDoc);
  if (!TtaSameSSchemas(elemType.ElSSchema, tagElType.ElSSchema) ||
      tagElType.ElTypeNum == 0)
    return FALSE;
  switch (constOfType)
    {
    case ConstructIdentity:
      if (subTypes[0].ElTypeNum == tagElType.ElTypeNum)
	result = TRUE;
      else if (!ustrcmp (GITagNameByType (subTypes[0]), TEXT("???")) ||
	       !ustrcmp (GITagNameByType (subTypes[0]), TEXT("none")))
	/* search if tag can be inserted as a child of the identity */
	result = IsValidHtmlChild (subTypes[0], tag, _EMPTYSTR_);
#ifdef MATHML
      /* any math element can be inserted under <MATH> (only row in MathML.S)*/
      if (!result && !ustrcmp (TtaGetElementTypeName (elemType), TEXT("MathML")) && 
	  ustrcmp (TtaGetSSchemaName (elemType.ElSSchema), TEXT("MathML")) == 0)
	result = IsValidHtmlChild (subTypes[0], tag, _EMPTYSTR_);
#endif
      break;

    case ConstructList:
      if (subTypes[0].ElTypeNum == tagElType.ElTypeNum)
	result = TRUE;
      else if (!ustrcmp (GITagNameByType (subTypes[0]), TEXT("???")) ||
	       !ustrcmp (GITagNameByType (subTypes[0]), TEXT("P*")) ||
	       !ustrcmp (GITagNameByType (subTypes[0]), TEXT("none")))
	result = IsValidHtmlChild (subTypes[0], tag, _EMPTYSTR_);
      break;

    case ConstructChoice:
      for (i = 0; !result && i < cardinal; i++)
	{
	  if (subTypes[i].ElTypeNum == tagElType.ElTypeNum)
	    result = TRUE;
	  else if (!ustrcmp (GITagNameByType (subTypes[i]),TEXT("???")) ||
		   !ustrcmp (GITagNameByType (subTypes[i]), TEXT("P*")) ||
		   !ustrcmp (GITagNameByType (subTypes[i]), TEXT("none")))
	    result = IsValidHtmlChild (subTypes[i], tag, _EMPTYSTR_);
	}
      break;

    case ConstructOrderedAggregate:
      found = (!ustrcmp (prevtag, _EMPTYSTR_));
      GIType (prevtag, &prevElType, (Document)TransDoc);
      found = (prevElType.ElTypeNum == 0);
      /* searches the rule of previous sibling */
      for (i = 0; !found && i < cardinal; i++)
	{
	  if (prevElType.ElTypeNum == subTypes[i].ElTypeNum)
	    found = TRUE;
	  else if (ustrcmp (GITagNameByType (subTypes[i]),TEXT("???")) ||
		   ustrcmp (GITagNameByType (subTypes[i]), TEXT("P*")) ||
		   ustrcmp (GITagNameByType (subTypes[i]), TEXT("none")))
	    i = cardinal;
	}
      if (found)
	{
	  while (!result && i < cardinal)
	    {
	      if (tagElType.ElTypeNum == subTypes[i].ElTypeNum)
		result = TRUE;
	      else if (!ustrcmp (GITagNameByType (subTypes[i]), TEXT("???")) ||
		       !ustrcmp (GITagNameByType (subTypes[i]), TEXT("P*")) ||
		       !ustrcmp (GITagNameByType (subTypes[i]), TEXT("none")))
		result = IsValidHtmlChild (subTypes[i], tag, _EMPTYSTR_);
	      if (!result)
		if (!ustrcmp (GITagNameByType (subTypes[i]), TEXT("???")) ||
		    !ustrcmp (GITagNameByType (subTypes[i]), TEXT("P*")) ||
		    !ustrcmp (GITagNameByType (subTypes[i]), TEXT("none")) ||
		    TtaIsOptionalInAggregate(i,elemType)) 
		  i++;
		else
		  i = cardinal;
	    }
	}
      break;
    case ConstructUnorderedAggregate:
      while (!result && i < cardinal)
	{
	  if (tagElType.ElTypeNum == subTypes[i].ElTypeNum)
	    result = TRUE;
	  else if (!ustrcmp (GITagNameByType (subTypes[i]), TEXT("???")) ||
		   !ustrcmp (GITagNameByType (subTypes[i]), TEXT("P*")) ||
		   !ustrcmp (GITagNameByType (subTypes[i]), TEXT("none")))
	    result = IsValidHtmlChild (subTypes[i], tag, _EMPTYSTR_);
	  if (!result)
	    if (TtaIsOptionalInAggregate(i,elemType)) 
	      i++;
	    else
	      i = cardinal;
	}    
    case ConstructNature:
      {
	if (TtaSameSSchemas (tagElType.ElSSchema, subTypes[0].ElSSchema))
	  {
	    if (subTypes[0].ElTypeNum == 0)
	      TtaGiveTypeFromName (&subTypes[0], TtaGetElementTypeName(elemType));
	    if (tagElType.ElTypeNum == subTypes[0].ElTypeNum)
	      result = TRUE;
	    else if (!ustrcmp (GITagNameByType (subTypes[0]), TEXT("???")) ||
		     !ustrcmp (GITagNameByType (subTypes[0]), TEXT("P*")) ||
		     !ustrcmp (GITagNameByType (subTypes[0]), TEXT("none")))
	    result = IsValidHtmlChild (subTypes[0], tag, _EMPTYSTR_);
	  }
      }
	      
    case ConstructConstant:
    case ConstructReference:
    case ConstructBasicType:
    case ConstructPair:
    case ConstructError:
      break;
    }
  TtaFreeMemory((void*)subTypes);
  return result;
}

/*----------------------------------------------------------------------
   CheckValidTransRoot                                                  
   checks if the higher-level generated elements are possible children of the   
   transformation root element                                          
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static ThotBool     CheckValidTransRoot (strMatch * sm, ElementType elemTypeRoot, STRING prevTag)
#else
static ThotBool     CheckValidTransRoot (sm, elemTypeRoot, prevTag)
strMatch           *sm;
ElementType         elemTypeRoot;
STRING              prevTag;

#endif
{
  strMatchChildren   *smc;
  strMatch           *sm2;
  strNodeDesc           *node;
  ThotBool            result, sonsMatch;
  STRING              curTag;



  curTag = TtaAllocString (NAME_LENGTH);
  result = TRUE;
  smc = sm->MatchChildren;
  while (result && smc != NULL)
    {
      if (smc->MatchSymb->Rule == NULL)
	{			/* there is no rule for the current node */
	  sm2 = smc->MatchNode->Matches;
	  sonsMatch = FALSE;
	  while (sm2 != NULL && !sonsMatch)
	    {		/* checks if the children of the node have been matched */
	      sonsMatch = (sm2->MatchSymb == smc->MatchSymb && sm2->MatchChildren != NULL);
	      if (!sonsMatch)
		sm2 = sm2->Next;
	    }
	  if (!sonsMatch)
	    {		/* if the children of the node have not been matched */
	      /* checks if the node can be transferred in the destination */
	      if (TtaGetElementVolume (smc->MatchNode->Elem) != 0)
		{		/* if the element is empty, it is ignored in transformation */
		  if (ustrcmp (prevTag, smc->MatchNode->Tag))
		    result = IsValidHtmlChild (elemTypeRoot,
					       smc->MatchNode->Tag,
					       prevTag);
		  ustrcpy (prevTag, smc->MatchNode->Tag);
		}
	    }
	  else
	    {		/* if they have been, checks the elements generated by these children */
	      result = CheckValidTransRoot (sm2, elemTypeRoot, prevTag);
	    }
	}
      else
	{			/* there is a rule for the current node */
	  node = smc->MatchSymb->Rule->OptionNodes;
	  if (node != NULL)
	    {		/* if there is at least one place node */
	      if (ustrcmp (prevTag, node->Tag))
		{
		  result = IsValidHtmlChild (elemTypeRoot,
					     node->Tag,
					     prevTag);
		  ustrcpy (prevTag, node->Tag);
		}
	    }
	  else
	    {
	      node = smc->MatchSymb->Rule->NewNodes;
	      if (node != NULL)
		{
		  if (!ustrcmp (node->Tag, TEXT("*")))
		    ustrcpy (curTag, smc->MatchNode->Tag);
		  else
		    ustrcpy (curTag, node->Tag);
		  result = IsValidHtmlChild (elemTypeRoot,
					     curTag,
					     prevTag);
		  ustrcpy (prevTag, curTag);
		}
	      else		/*deleted node */
		if (smc->MatchSymb->Rule->DeleteRule)
		  result = TRUE;
		else
		  result = FALSE;
	    }
	}
      if (result)
	smc = smc->Next;
    }
  TtaFreeMemory (curTag);
  return result;
}

/*----------------------------------------------------------------------
   callback of the transformation selection menu 
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                TransCallbackDialog (int ref, int typedata, STRING data)
#else
void                TransCallbackDialog (ref, typedata, data)
int                 ref;
int                 typedata;
STRING              data;

#endif
{
  int                 val, length;
  DisplayMode         oldDisplayMode;
  SSchema	       sch;
  Element	       elParent, elFound;
  Attribute	       attr;
  AttributeType       attrType;
  ThotBool	       found;
  CHAR_T		       buf [MAX_LENGTH];
  SearchDomain	       domain;

  val = (int) data;
  switch (ref - TransBaseDialog)
    {
    case TransMenu:
      oldDisplayMode = TtaGetDisplayMode (TransDoc);
      /* annule la selection */
      if (oldDisplayMode != DisplayImmediately)
	TtaSetDisplayMode (TransDoc, DisplayImmediately);
      TtaSelectElement (TransDoc, NULL);
      /* passe en mode de display differe */
#ifndef AMAYA_DEBUG
      TtaSetDisplayMode (TransDoc, DeferredDisplay);
#endif		
      resultTrans = ApplyTransformation (menuTrans[val], TransDoc);
      if (!resultTrans)
	{
#ifdef IV
	  TtaSetDisplayMode (TransDoc, DisplayImmediately);
	  /* transformation has failed, restoring the old selection */
	  if (ffc == 0 && flc == 0)
	    TtaSelectElement (TransDoc, origFirstSelect);
	  else
	    TtaSelectString (TransDoc, origFirstSelect, ffc, flc);
	  TtaExtendSelection (TransDoc, origLastSelect, llc);
	  /* display an error message */
	  TtaSetStatus (TransDoc, 1, TtaGetMessage (AMAYA, AM_TRANS_FAILED), NULL);
#endif
	}
      else
	{

	  /* transformation was succesful */ 
	  sch = TtaGetElementType (myFirstSelect).ElSSchema;
	  if (ustrcmp (TtaGetSSchemaName (sch), TEXT("MathML")) == 0)
	    {
	      /* checking the MathML thot tree */
	      if (isClosed)
		elParent = TtaGetParent (myFirstSelect);
	      else
		elParent = myFirstSelect;
	      while (elParent != NULL &&
		     ustrcmp (GITagName (elParent), TEXT("???")) == 0)
		elParent = TtaGetParent (elParent);
	      if (elParent != NULL)
		{
		  TtaSetStructureChecking (0, TransDoc);
		  XMLElementComplete (elParent, TransDoc);
		  TtaSetStructureChecking (1, TransDoc);
		} 
	    }
		   
	  if (myLastSelect == NULL)
	    if (!isClosed)
	      myLastSelect = TtaGetLastChild (myFirstSelect);
	    else
	      myLastSelect = TtaGetLastChild (TtaGetParent (myFirstSelect));
	  else
	    TtaPreviousSibling (&myLastSelect);
	  if (isClosed)
	    TtaNextSibling (&myFirstSelect);
	  else
	    myFirstSelect = TtaGetFirstChild (myFirstSelect);
	  if (ustrcmp (TtaGetSSchemaName (sch), TEXT("HTML")) == 0)
	    {
	      /* displaying the images */
	      attrType.AttrSSchema = sch;
	      attrType.AttrTypeNum = HTML_ATTR_SRC;
	      elFound = NULL;
	      attr = NULL;
	      TtaSearchAttribute (attrType, SearchInTree, myFirstSelect, &elFound, &attr);
	      if (elFound == NULL)
		TtaSearchAttribute (attrType, SearchForward, myFirstSelect, &elFound, &attr);
	      while (elFound != NULL && 
		     (myLastSelect == NULL || 
		      TtaIsBefore (elFound, myLastSelect) ||
		      TtaIsAncestor (elFound, myLastSelect)))
		{
#ifdef DEBUG
		  printf ("Fetching Image...\n");
#endif
		  FetchImage (TransDoc, elFound, NULL, 0, NULL, NULL);
		  TtaSearchAttribute (attrType, SearchForward, elFound, &elFound, &attr);
		}
	    }
	  /* selecting the new elements */
	  /* or setting the selction to the specified node */
	  attrType.AttrSSchema = TtaGetElementType (myFirstSelect).ElSSchema;
	  if (!ustrcmp (TtaGetSSchemaName (attrType.AttrSSchema), TEXT("HTML")))
	    attrType.AttrTypeNum = HTML_ATTR_Ghost_restruct;
#ifdef MATHML
	  else if (!ustrcmp (TtaGetSSchemaName (attrType.AttrSSchema), TEXT("MathML")))
	    attrType.AttrTypeNum = MathML_ATTR_Ghost_restruct;
#endif
	  found = FALSE;
	  elFound = NULL;
	  attr = NULL;
	  domain = SearchInTree;
	  elFound = TtaGetParent(myFirstSelect);
	  while (elFound != NULL && !found)
	    {
	      TtaSearchAttribute (attrType, domain, elFound, &elFound, &attr);
	      domain = SearchForward;
	      if (elFound != NULL)
		{
		  length = MAX_LENGTH;
		  TtaGiveTextAttributeValue (attr, buf, &length);
		  found = !ustrcmp (buf, TEXT("Select"));
		}
	    }
	  if (found)
	    {
	      TtaRemoveAttribute (elFound, attr, TransDoc);
	      myFirstSelect = elFound;
	      myLastSelect = NULL;
	    }
	  TtaSetDocumentModified (TransDoc);
	  TtaSetDisplayMode (TransDoc, DisplayImmediately);
	  TtaSelectElement (TransDoc, myFirstSelect);
	  if (myLastSelect != NULL && TtaIsBefore (myFirstSelect, myLastSelect))
	    TtaExtendSelection (TransDoc, myLastSelect, 0);
	}
      FreeMatchEnv ();
      break;
    }
}

/*----------------------------------------------------------------------
   callback of  the transform entry of special menu 
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                TransformType (Document doc, View view)
#else
void                TransformType (doc, view)
Document            doc;
View                view;
#endif
{
  Element             elemSelect;
  ElementType	       elType;
  int                 i, j, k;
  STRING              menuBuf, tag, nameSet;
  strMatch           *sm;
  StructureTree       node;
  ThotBool	       ok;
  strTransSet*	       transSets [4] = {NULL,NULL,NULL,NULL};
  SSchema	      transSchema;
  
  strMatchEnv.SourceTree = NULL;
  strMatchEnv.ListSubTrees = NULL;
  resultTrans = FALSE;
  TransDoc = doc;
  nameSet = TtaAllocString (NAME_LENGTH);
  ustrcpy (nameSet, _EMPTYSTR_);
  /* context initialisation -- checks the selection */
  ok = CheckSelectionLevel (TransDoc);
  
  if (ok)
    { 
      maxMatchDepth = 0;
      if (mySelect != NULL)
	{ /* parse les transformations correspondant aux schemas */
	  /* des elements entre myFirstSelect et mySelect */
	  i = 0;
	  ok = FALSE;
	  elemSelect = myFirstSelect;
	  while (elemSelect != TtaGetParent(mySelect))
	    {
	      elType = TtaGetElementType (elemSelect);
	      ustrcpy (nameSet, TtaGetSSchemaName (elType.ElSSchema));
	      ok =  ppStartParser (nameSet, elType.ElSSchema, &CourTransSet) || ok;
	      if (i < 4 && ok && (i == 0 || CourTransSet != transSets[i-1]))
		{
		  transSets [i++] = CourTransSet;
		  if (maxMatchDepth < CourTransSet->MaxDepth + maxSelDepth)
		    maxMatchDepth = CourTransSet->MaxDepth + maxSelDepth;
		}
	      elemSelect = TtaGetParent (elemSelect);
	    }
	  
	}
      else
	{ /* parse les transformations correspondant au schema de tous les */
	  /* elements selectionnes */
	  elemSelect = myFirstSelect;
	  elType = TtaGetElementType (elemSelect);
	  ustrcpy (nameSet, TtaGetSSchemaName (elType.ElSSchema));
	  transSchema = elType.ElSSchema;
	  while (elemSelect != NULL)
	    {
	      elType = TtaGetElementType (elemSelect);
	      if (ustrcmp (nameSet, TtaGetSSchemaName (elType.ElSSchema)) == 0)
		{
		  MyNextSelectedElement (TransDoc, &elemSelect);
		}
	      else
		{
		  ustrcpy (nameSet, _EMPTYSTR_);
		  transSchema = NULL;
		  elemSelect = NULL;
		}
	    }
	  ok = ppStartParser (nameSet, transSchema, &CourTransSet);
	  if (ok)
	    {
	      transSets [0] = CourTransSet;
	      maxMatchDepth = CourTransSet->MaxDepth + maxSelDepth;
	    }
	}
    }
  if (ok)
    { /* builds the source structure tree */
      menuBuf = TtaAllocString (MAX_LENGTH);
      strMatchEnv.SourceTree = (StructureTree) NewNode (TEXT("Root"));
      if (mySelect != NULL)
	{
	  (strMatchEnv.SourceTree)->Elem = TtaGetParent (mySelect);
	  BuildStructureTree (mySelect, TransDoc, strMatchEnv.SourceTree, maxMatchDepth, 0);
	}
      else
	{
	  (strMatchEnv.SourceTree)->Elem = TtaGetParent (myFirstSelect);
	  elemSelect = myFirstSelect;
	  while (elemSelect != NULL)
	    {
	      BuildStructureTree (elemSelect, TransDoc, strMatchEnv.SourceTree, maxMatchDepth, 0);
	      MyNextSelectedElement (TransDoc, &elemSelect);
	    }
	}
      i = 0;
      while (i < 4 && transSets [i] != NULL)
	{
	  CourTransSet = transSets [i++];
	  /* pattern matching */
	  PostfixSearch (strMatchEnv.SourceTree, MatchNode);
	}
      /* construct the popup menu with the result of matching*/
      node = strMatchEnv.SourceTree;
      i = 0;
      j = 0;
      tag = TtaAllocString (NAME_LENGTH);
      do
	{			/* for each node above the first selected */
#ifdef AMAYA_DEBUG
	  printf ("Matches for %s element :\n",node->Tag);
#endif
	  sm = node->Matches;
	  while (sm != NULL)
	    {		/* for each matching of the node */
	      if (!ustrcmp (sm->MatchSymb->Tag, TEXT("pattern_root")))
		{ /* if it is matching a pattern root : */
		  /* insert the transformation name in the menu buffer */
	
	  
		  elemSelect = sm->MatchChildren->MatchNode->Elem;
		  TtaPreviousSibling (&elemSelect);
		  if (elemSelect != NULL)
		    ustrcpy (tag, GITagNameByType (TtaGetElementType (elemSelect)));
		  while (elemSelect != NULL && (!ustrcmp (tag, TEXT("???")) || !ustrcmp (tag, TEXT("none"))))
		    {
		      TtaPreviousSibling (&elemSelect);
		      if (elemSelect != NULL)
			ustrcpy (tag, GITagNameByType (TtaGetElementType (elemSelect)));
		    }
		  if (elemSelect == NULL)
		    ustrcpy (tag, _EMPTYSTR_);
		  if (CheckValidTransRoot (sm,
					   TtaGetElementType (sm->MatchNode->Elem),
					   tag))
		    {
#ifdef AMAYA_DEBUG
		      printf ("   %s\n",sm->MatchSymb->SymbolName);
#endif
		      for (k = 0; 
			   k < i && 
			     ustrcmp (menuTrans[k]->MatchSymb->SymbolName, 
				     sm->MatchSymb->SymbolName); 
			   k++);
		      if (k == i)
			{
			   usprintf (&menuBuf[j], TEXT("%s%s"), "B", sm->MatchSymb->SymbolName);
			   j += ustrlen (&menuBuf[j]) + 1;
			   menuTrans[i++] = (strMatch *) sm;
			}
		    }
		}
	      sm = sm->Next;
	    }
	  node = node->Child;
	}
      while (node != NULL &&
	     (!TtaIsAncestor (node->Elem, myFirstSelect)));
      TtaFreeMemory (tag);
      if (i > 0)
	{ /* if some transformations have been matched, shows the menu */
	  TtaNewPopup (TransBaseDialog + TransMenu, 0, 
		       TtaGetMessage (AMAYA, AM_TRANS), i, menuBuf,
		       NULL, 'L');
#     ifndef _WINDOWS
	  TtaSetDialoguePosition ();
#     endif /* !_WINDOWS */
	  TtaShowDialogue (TransBaseDialog + TransMenu, TRUE);
	}
      else
	/* display an status message */
	TtaSetStatus (TransDoc, 1, TtaGetMessage (AMAYA, AM_NO_TRANS), NULL);
      TtaFreeMemory (menuBuf);
    }
  else
    /* display an status message */
    TtaSetStatus (TransDoc, 1, TtaGetMessage (AMAYA, AM_NO_TRANS), NULL);
  TtaFreeMemory (nameSet);
}



/*----------------------------------------------------------------------
   TransformIntoType:  Changes the type of a the selected elements into the given DestType
   Selects the transformations that produces the givem type, matches teir patterns, and
   applies the first matched transformation
  ----------------------------------------------------------------------*/
#ifdef __STDC__
ThotBool            TransformIntoType (ElementType resultType, Document doc)
#else
ThotBool            TransformIntoType (resultType, doc)
ElementType         resultType;
Document            doc;

#endif
{
  CHAR_T                DestTag[20];
  ThotBool            ok, chglev;
  int                 i,last,best;
  Element             elemSelect;
  ElementType         elType;
  strTransDesc       *td;
  strSymbDesc        *sd;
  strMatch           *sm;
  StructureTree       node;
  STRING              tag, nameSet;
  SSchema	      transSchema;

  strMatchEnv.SourceTree = NULL;
  strMatchEnv.ListSubTrees = NULL;
  resultTrans = FALSE;
  TransDoc = doc;
  nameSet = TtaAllocString (NAME_LENGTH);
  ustrcpy (nameSet, _EMPTYSTR_);
  
  /* context initialisation -- checks the selection */
  ok = CheckSelectionLevel (TransDoc);
  if (ok)  
    {
      CourTransSet = NULL;
      ok = FALSE;
      maxMatchDepth = 0;
      if (mySelect != NULL)
	{ /* parse les transformations correspondant aux schemas des elements */
	  /* entre myFirstSelect et mySelect */
	  i = 0;
	  elemSelect = myFirstSelect;
	  while (CourTransSet == NULL && elemSelect != TtaGetParent(mySelect))
	    {
	      elType = TtaGetElementType (elemSelect);
	      if (TtaSameSSchemas (elType.ElSSchema, resultType.ElSSchema))
		{
		  ustrcpy (nameSet, TtaGetSSchemaName (elType.ElSSchema));
		  ok =  ppStartParser (nameSet, elType.ElSSchema, &CourTransSet);
		}
	      if (CourTransSet == NULL)
		elemSelect = TtaGetParent (elemSelect);
	    }
	}
      else
	{
	  elemSelect = myFirstSelect;
	  elType = TtaGetElementType (elemSelect);
	  ustrcpy (nameSet, TtaGetSSchemaName(elType.ElSSchema));
	  transSchema = elType.ElSSchema;
	  while (elemSelect != NULL)
	    {
	      elType = TtaGetElementType (elemSelect);
	      if (ustrcmp (nameSet, TtaGetSSchemaName (elType.ElSSchema)) == 0)
		{
		  MyNextSelectedElement (TransDoc, &elemSelect);
		  if (elemSelect != NULL)
		    elType = TtaGetElementType (elemSelect);
		}
	      else
		{
		  ustrcpy (nameSet, _EMPTYSTR_);
		  transSchema = NULL;
		  elemSelect = NULL;
		}
	    }
	  ok = ppStartParser (nameSet, transSchema, &CourTransSet);
	}
    }
  if (ok)
    {
      ustrcpy (DestTag, GITagNameByType (resultType));
      td = CourTransSet->Transformations;
      CourTransSet->MaxDepth = 0;
      while (td != NULL)
	{
	  if (td->DestinationTag == NULL || ustrcmp (td->DestinationTag, DestTag))
	    {		
	      /* the transformation does not produce the given type, it is desactived */
	      td->IsActiveTrans = FALSE;
	      sd = td->Symbols;
	      while (sd != NULL)
		{
		  sd->IsActiveSymb = FALSE;
		  sd = sd->Next;
		}
	    }
	  else
	    {
	      /* at least 1 transformation produces the given type */		 
	      ok = TRUE;	
	      if (td->PatDepth > CourTransSet->MaxDepth)
		CourTransSet->MaxDepth = td->PatDepth;
	    }
	  td = td->Next;
	}
      maxMatchDepth = CourTransSet->MaxDepth + maxSelDepth;
      
      /* builds the source structure tree */
      strMatchEnv.SourceTree = (StructureTree) NewNode (TEXT("Root"));
      if (mySelect != NULL)
	{
	  (strMatchEnv.SourceTree)->Elem = TtaGetParent (mySelect);
	  BuildStructureTree (mySelect, TransDoc, strMatchEnv.SourceTree, maxMatchDepth, 0);
	}
      else
	{
	  (strMatchEnv.SourceTree)->Elem = TtaGetParent (myFirstSelect);
	  elemSelect = myFirstSelect;
	  while (elemSelect != NULL)
	    {
	      BuildStructureTree (elemSelect, TransDoc, strMatchEnv.SourceTree, maxMatchDepth, 0);
	      MyNextSelectedElement (TransDoc, &elemSelect);
	    }
	}
      
      /* pattern matching */
      PostfixSearch (strMatchEnv.SourceTree, MatchNode);
	
      node = strMatchEnv.SourceTree;
      while (node->Child != NULL && 
	     TtaIsAncestor (myFirstSelect, node->Elem))
	node = node->Child;
      i = 0;
      last = -1;
      best = -1;
      
      tag = TtaAllocString (NAME_LENGTH);
      do
	{ /* for each node above the first selected */
	  chglev = TRUE;
	  node = node->Parent;
	  sm = node->Matches;
	  while (sm != NULL)
	    {		/* for each matching of the node */
	      if (!ustrcmp (sm->MatchSymb->Tag, TEXT("pattern_root")))
		{	/* if it is matching a pattern root : insert the transformation */
		  /* in the matched transformations list */
		  
		  ustrcpy (tag, _EMPTYSTR_);
		  if (CheckValidTransRoot (sm,
					   TtaGetElementType (sm->MatchNode->Elem),
					   tag))
		    {
		      if (chglev)
			{
			  chglev = FALSE;
			  if (last == -1 || 
			      TtaIsAncestor (sm->MatchNode->Child->Elem, origFirstSelect))
			    last = i;
			  if (origFirstSelect == sm->MatchNode->Child->Elem)
			    best = i;
			}
		      menuTrans[i++] = (strMatch *) sm;
		    }
		}
	      sm = sm->Next;
	    }
	}
      while (node != strMatchEnv.SourceTree);
      TtaFreeMemory (tag);
      if (i > 0)
	/* if at least one transformation have been matched, apply the best  one */
	{
	  if (best == -1)
	    /* no transformation for the actual selection : take another one */
	    best = last;
	  TransCallbackDialog (TransBaseDialog + TransMenu, 0, (STRING) best);
	}
    }
  TtaFreeMemory (nameSet);  
  return resultTrans;
}
