/* # skkinput (Simple Kana-Kanji Input)
 * skkconfig.c --- Read dot.skkinput.
 * This file is part of skkinput.
 * Copyright (C) 1997
 * Takashi SAKAMOTO (sakamoto@yajima.kuis.kyoto-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include "skkmain.h"
#include "SkkInput.h"
#include "config.h"
#define skkconfig_c
#include "skkkey.h"

/*
 * ¤Τġ
 */
enum {
  PARAM_BOOLEAN, PARAM_INTEGER, PARAM_STRING, PARAM_PATH,
  PARAM_INPUT_VECTOR, PARAM_ROM_KANA_RULE_LIST, 
  PARAM_DEFINE_KEY_SKKMAP, PARAM_DEFINE_KEY_ABBREVMAP,
  PARAM_END,
} ;

typedef struct {
  char *string ;
  int action ;
  void *argument ;
} SkkInpDef ;

struct skk_rom_kana_rule_list {
  struct skk_rom_kana_rule element ;
  struct skk_rom_kana_rule_list *next ;
} ;

/*
 * ץȥ
 */
/* skkldic.c */
extern char *expand_file_name( char *path ) ;
extern int euc_fgets
( char *buffer, int len, FILE *fp, int echara,
  int coding_system, int ecflag ) ;
extern int check_skkinput_jisyo_code( char *path ) ;

/*
 * skkinput ưꤹѿ
 */
extern int  skkserv_portnum ;
extern char *skkserv_host ;
extern char *skkinput_local_jisyo_name ;
extern char *skkinput_backup_jisyo_name ;
extern char *skk_local_jisyo_name ;
extern char *skkinput_record_name ;
extern int  skk_egg_like_newline ;
extern int  skkinput_chatadaptermode ;
extern int  skkinput_search_skk_jisyo ;
extern int  skkinput_keep_record ;
extern int  skkinput_date_ad ;
extern int  skkinput_number_style ;
extern int  skkinput_delete_implies_kakutei ;
extern int  skkinput_use_numeric_conversion ;
extern int  skkinput_jisyo_dirty ;
static int  skkinput_tab_width ;

/*
 * ϥ٥ȥ롣
 */
static unsigned char *skkinput_input_vector[ 128 ] = {
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  "",  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,
  NULL,  NULL,  NULL,  NULL,  "",  "",  "",  NULL,
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  "",  "",  NULL,  NULL,  NULL,  "",
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  NULL,  "",  NULL,  "",  NULL,  NULL,
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,
} ;

/*
 * ϥ٥ȥ롣
 */
static unsigned char *skkinput_zenkaku_vector[ 128 ] = {
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  "",  "",  "",  "",  "",  "",  "",  "",
  "",  "",  "",  "",  "",  "",  "",  "",
  "",  "",  "",  "",  "",  "",  "",  "", 
  "",  "",  "",  "",  "",  "",  "",  "",
  "",  "",  "",  "",  "",  "",  "",  "",
  "",  "",  "",  "",  "",  "",  "",  "", 
  "",  "",  "",  "",  "",  "",  "",  "",  
  "",  "",  "",  "",  "",  "",  "",  "",  
  "",  "",  "",  "",  "",  "",  "",  "",  
  "",  "",  "",  "",  "",  "",  "",  "",  
  "",  "",  "",  "",  "",  "",  "",  "",  
  "",  "",  "",  "",  "",  "",  "",  NULL,
} ;

static struct skk_rom_kana_rule
skkinput_default_rom_kana_rule_list[] = {
  { "nn", NULL, "", "" },
  { "n'", NULL, "", "" },
  { NULL, NULL, NULL, NULL },
} ;

static struct skk_rom_kana_rule
*skkinput_rom_kana_rule_list ;

static struct skk_rom_kana_rule_list *skkinput_rom_kana_rule_list_top ;

/*
 * .skkinput ˵ҤǤΰ
 */
SkkInpDef skkinput_initdef[] = {
  { "skk-port-num:",		
      PARAM_INTEGER,		 &skkserv_portnum },
  { "skk-server-host:",	
      PARAM_STRING,		 &skkserv_host },
  { "skkinput-jisyo:",
      PARAM_PATH,		 &skkinput_local_jisyo_name },
  { "skkinput-backup-jisyo:",
      PARAM_PATH,		 &skkinput_backup_jisyo_name },
  { "skk-jisyo:",
      PARAM_PATH,		 &skk_local_jisyo_name },
  { "skkinput-record:",
      PARAM_PATH,		 &skkinput_record_name },
  { "skkinput-egg-like-newline:",
      PARAM_BOOLEAN,		 &skk_egg_like_newline },
  { "skkinput-chat-adapter-mode:",
      PARAM_BOOLEAN,		 &skkinput_chatadaptermode },
  { "skkinput-keep-record:",
      PARAM_BOOLEAN,		 &skkinput_keep_record },
  { "skkinput-search-skk-jisyo:",
      PARAM_BOOLEAN,		 &skkinput_search_skk_jisyo },
  { "skkinput-date-ad:",
      PARAM_BOOLEAN,		 &skkinput_date_ad },
  { "skkinput-number-style:",
      PARAM_INTEGER,		 &skkinput_number_style },
  { "skkinput-delete-implies-kakutei:",
      PARAM_BOOLEAN,		 &skkinput_delete_implies_kakutei },
  { "skkinput-use-numeric-conversion:",
      PARAM_BOOLEAN,		 &skkinput_use_numeric_conversion },
  { "skkinput-zenkaku-vector:",
      PARAM_INPUT_VECTOR,	 skkinput_zenkaku_vector },
  { "skkinput-input-vector:",
      PARAM_INPUT_VECTOR,	 skkinput_input_vector },
  { "skkinput-rom-kana-rule-list:",
      PARAM_ROM_KANA_RULE_LIST,  &skkinput_rom_kana_rule_list_top },
  { "define-key-skkmap:",
      PARAM_DEFINE_KEY_SKKMAP,	 NULL },
  { "define-key-abbrevmap:",
      PARAM_DEFINE_KEY_ABBREVMAP,NULL },
  { "tab-width:",
      PARAM_INTEGER,		 &skkinput_tab_width },
  { NULL,
    PARAM_END,		NULL },
} ;

/*
 * ꤵ줿ʸ󤬶ʸʤΤɤ֤ؿ
 *----
 * Ȱʲ϶ʸǤȤư
 */
static int is_empty_line( unsigned char *ptr )
{
  while( *ptr != '\0' && *ptr != ';' ){
    if( *ptr != 0x20 && *ptr != '\t' && *ptr != '\n' && *ptr != '\r' )
      return False ;
    ptr ++ ;
  }
  return True ;
}

/*
 * ʸɤФؿ
 */
static unsigned char *skip_empty_symbols( unsigned char *ptr )
{
  while( *ptr != '\0' ){
    if( *ptr != 0x20 && *ptr != '\t' && *ptr != '\n' && *ptr != '\r' )
      break ;
    ptr ++ ;
  }
  return ptr ;
}

/*
 * Boolean Ĥޤ "t"  "nil" Τɤ餬ꤵƤ뤫ؿ
 */
static int skkinput_analize_boolean
( unsigned char *oneline, void *argument )
{
  oneline = skip_empty_symbols( oneline ) ;

  /* Ȥ٤ꤵƤ뤫ɤȽǤ롣*/
  if( *oneline == 't' && is_empty_line( oneline + 1 ) ){
    *( int *)argument = True ;
    return True ;
  } 
  /* Ȥ٤ꤵƤ뤫ɤȽǤ롣*/
  if( !strncmp( oneline, "nil", 3 ) ){
    if( is_empty_line( oneline + 3 ) ){
      *( int *)argument = False ;
      return True ;
    }
  }
  return False ;
}

static unsigned char *skip_number_digit( unsigned char *ptr )
{
  /* ³¤ꥹåפ롣*/
  while( *ptr != '\0' ){
    if( *ptr < '0' || *ptr > '9' )
      break ;
    ptr ++ ;
  }
  return ptr ;
}

static unsigned char *analize_integer_sub
( unsigned char *ptr, int *result, int *value )
{
  unsigned char *start ;
  int ch ;
  /* ޤʸ򥹥åפơȡ*/
  start = skip_empty_symbols( ptr ) ;
  /* 줫顢ʸȳФȴФޤ*/
  ptr = skip_number_digit( start ) ;
  /* ʸʤä顢ȡ*/
  if( start == ptr ){
    *result = False ;
    return ptr ;
  }
  /* äƤ顢ȡ*/
  ch     = *ptr ;
  *ptr   = '\0' ;
  *value = atoi( start ) ;
  *ptr   = ch ;
  *result = True ;
  return ptr ;
}

/*
 * ʤꤵƤ뤫ؿ
 */
static int skkinput_analize_integer
( unsigned char *oneline, void *argument )
{
  int result, value ;
  oneline = analize_integer_sub( oneline, &result, &value ) ;
  if( !result )
    return False ;
  /* ͤμǤϤʤˤϥ顼*/
  if( !is_empty_line( oneline ) ){
    /* ͤϤʤ*/
    return False ;
  }
  *( int *)argument = value ;
  return True ;
}

/*
 * β򤭤()ؿ
 */
static int my_strcpy( unsigned char *dest, unsigned char *src )
{
  int i, code_num ;
  while( *src != '\0' ){
    /* Хåξβġ*/
    if( *src == '\\' ){
      src ++ ;
      if( *src >= '0' && *src <= '8' ){
	/* äˤϣʸޤǸơΥɤͭˤ롣*/
	code_num = 0 ;
	for( i = 0 ; i < 3 ; i ++ ){
	  if( *src < '0' || *src > '8' )
	    break ;
	  code_num = code_num * 8 + ( *src - '0' ) ;
	  src ++ ;
	}
	/* 128 ĶʤѤ͡͡͡ */
	if( code_num <= 128 )
	  *dest ++ = code_num ;
      }
    }
    /* ǥåΤϡbackslash αƶθᡣ*/
    if( *src == '\0' )
      break ;
    *dest ++ = *src ++ ;
  }
  *dest = '\0' ;
  return True ;
}

static unsigned char *analize_string_sub
( unsigned char *ptr, int *result, unsigned char **value )
{
  unsigned char *start ;
  /* ޤ̤ϲФƤʤꤷƤ*/
  *value  = NULL ;
  *result = False ;
  ptr = skip_empty_symbols( ptr ) ;
  if( *ptr != 0x22 )
    return ptr ;
  ptr ++ ;
  start = ptr ;
  /* ʸȤĴ٤롣*/
  while( *ptr != '\0' && *ptr != 0x22 ){
    /* ⤷ double quote 줿Ȥ׵᤬äˡġ*/
    if( *ptr == '\\' ){
      ptr ++ ;
      /* ʸ󤬽äƤޤä顢ѡ*/
      if( *ptr == '\0' )
	return ptr ;
    }
    ptr ++ ;
  }
  /* ơɤޤǹԤäʡ */
  if( *ptr != 0x22 ){
    return ptr ;
  }
  /* ʸġ*/
  if( ptr == start ){
    /* ʸäƤʤȥ֥륯Ȥǽü *
     * Ƥʤ*/
    *value  = NULL ;
    ptr ++ ;
  } else {
    *ptr = '\0' ;
    /* ʸĹmallocޤ*/
    if( ( *value = malloc( strlen( start ) + 1 ) ) == NULL ){
      fprintf( stderr, "Fatal: memory exhausted.\n" ) ;
      exit( 1 ) ;
    }
    /* νġ*/
    my_strcpy( *value, start ) ;
    *ptr ++ = 0x22 ;
  }
  *result = True ;
  return ptr ;
}

/*
 * ʤʸꤵƤ뤫ؿ
 */
static int skkinput_analize_string
( unsigned char *oneline, void *argument, int flag )
{
  unsigned char *result ;
  int ret ;
  oneline = analize_string_sub( oneline, &ret, &result ) ;
  /* ̵ֻ⤷ϡ顼*/
  if( !ret || result == NULL )
    return False ;
  if( !is_empty_line( oneline ) ){
    /* ˥ߤդƤ뤼줸ܤ٥ӡ */
    free( result ) ;
    return False ;
  }
  /* ʸ򥳥ԡ */
  if( flag ){
    *( unsigned char **)argument = expand_file_name( result ) ;
    free( result ) ;
  } else {
    *( unsigned char **)argument = result ;
  }
  return True ;
}

static int skkinput_analize_input_vector
( unsigned char *oneline, void *argument )
{
  unsigned char *ptr, *rstr = NULL ;
  int result, value, ret ;
  /* ǽζʸ򥹥åפޤ*/
  oneline = skip_empty_symbols( oneline ) ;
  /* ( 123  "" ) */ 
  if( *oneline != '(' )
    goto err_ret ;
  oneline ++ ;
  /* ͤȴФޤ*/
  ptr = analize_integer_sub( oneline, &result, &value ) ;
  /* ̤֤äƤʤä顢ܤǤ硣*/
  if( !result )
    goto err_ret ;
  /* input vector ʤͤä顢ΤƤޤ礦*/
  if( value > 128 || value < 0 )
    goto err_ret ;
  /* Τޤ޺ǸޤǹԤäƤޤä顢ܤǤ礦*/
  ptr = skip_empty_symbols( ptr ) ;
  if( *ptr != '.' )
    goto err_ret ;
  /* ʸϤޤ*/
  ptr = analize_string_sub( ptr + 1, &ret, &rstr ) ;
  /* ȤäƤʤä顢ԤǤ*/
  if( !ret )
    goto err_ret ;
  /* ³ʸФޤ*/
  ptr = skip_empty_symbols( ptr ) ;
  /* Ǹ ")" ĤƤʤСܤǤ*/
  if( *ptr != ')' )
    goto err_ret ;
  ptr ++ ;
  /* θ˥ߤäƤϤޤ*/
  if( !is_empty_line( ptr ) )
    goto err_ret ;
  /* input vector 򹹿ޤ*/
  ((unsigned char **)argument)[ value ] = rstr ;
  return True ;

err_ret:
  if( rstr != NULL )
    free( rstr ) ;
  return False ;
}

static void add_skkinput_rule_list
( struct skk_rom_kana_rule_list **top,
  struct skk_rom_kana_rule element )
{
  struct skk_rom_kana_rule_list *node ;
  if( *top == NULL ){
    *top = node = malloc( sizeof( struct skk_rom_kana_rule_list ) ) ;
  } else {
    node = *top ;
    while( node->next != NULL )
      node = node->next ;
    node->next = malloc( sizeof( struct skk_rom_kana_rule_list ) ) ;
    node = node->next ;
  }
  if( node == NULL ){
    fprintf( stderr, "Fatal: Memory Exhausted.\n" ) ;
    exit( 1 ) ;
  }
  node->next = NULL ;
  /* ¤Τ򥳥ԡ롣*/
  node->element = element ;
  return ;
}

static int skkinput_analize_rule_list
( unsigned char *oneline, void *argument )
{
  unsigned char *ptr, *rstr[ 4 ] ;
  struct skk_rom_kana_rule node ;
  int i, result ;

  /* ǽζʸ򥹥åפޤ*/
  oneline = skip_empty_symbols( oneline ) ;
  /* ( "nn" "" "" "" ) */ 
  if( *oneline != '(' )
    return False ;
  oneline ++ ;
  ptr = oneline ;
  for( i = 0 ; i < 4 ; i ++ ){
    /* ʸϤޤ*/
    ptr = analize_string_sub( ptr, &result, &rstr[ i ] ) ;
    /* ȤäƤʤä顢ԤǤ*/
    if( !result ){
      while( i < 4 )
	rstr[ i ++ ] = NULL ;
      goto err_analize ;
    }
  }
  /* ʸ󤬽ä顢")" ĤƤȦ*/
  ptr = skip_empty_symbols( ptr ) ;
  if( *ptr != ')' )
    goto err_analize ;
  /* ǡ")" ³϶ʸȦ(⤷ϥȤɤ) */
  ptr ++ ;
  if( !is_empty_line( ptr ) )
    goto err_analize ;
  /* ǡȴФޤơȡ*/
  node.state   = rstr[ 0 ] ;
  node.next    = rstr[ 1 ] ;
  node.houtput = rstr[ 2 ] ;
  node.koutput = rstr[ 3 ] ;
  /* ߤξ̵֤ä顢㥨顼äϤåޤ*/
  if( node.state == NULL )
    goto err_analize ;
  add_skkinput_rule_list( argument, node ) ;
  return True ;
err_analize:
  for( i = 0 ; i < 4 ; i ++ ){
    if( rstr[ i ] != NULL )
      free( rstr[ i ] ) ;
  }
  return False ;
}

/* ( "@" . "j-self-insert" ) */
static int skkinput_analize_define_key
( unsigned char *oneline, unsigned char **rkeystring,
                          unsigned char **rkeyfunction )
{
  unsigned char *keystring, *keyfunction , *ptr ;
  int result ;

  keystring = keyfunction = NULL ;

  /* ǽζʸ򥹥åפޤ*/
  oneline = skip_empty_symbols( oneline ) ;
  /* ( "nn" "" "" "" ) */ 
  if( *oneline != '(' )
    goto err_ret ;
  oneline ++ ;
  /* ʸϤޤ*/
  ptr = analize_string_sub( oneline, &result, &keystring ) ;
  /* Ԥġ*/
  if( !result || keystring == NULL )
    goto err_ret ;
  /* Τޤ޺ǸޤǹԤäƤޤä顢ܤǤ礦*/
  ptr = skip_empty_symbols( ptr ) ;
  if( *ptr != '.' )
    goto err_ret ;
  /* ʸϤޤ*/
  ptr = analize_string_sub( ptr + 1, &result, &keyfunction ) ;
  /* Ԥġ*/
  if( !result || keyfunction == NULL )
    goto err_ret ;
  /* ³ʸФޤ*/
  ptr = skip_empty_symbols( ptr ) ;
  /* Ǹ ")" ĤƤʤСܤǤ*/
  if( *ptr != ')' )
    goto err_ret ;
  ptr ++ ;
  /* θ˥ߤäƤϤޤ*/
  if( !is_empty_line( ptr ) )
    goto err_ret ;
  /* ̤ޤ*/
  *rkeystring   = keystring ;
  *rkeyfunction = keyfunction ;
  return True ;
err_ret:
  if( keystring != NULL )
    free( keystring ) ;
  if( keyfunction != NULL )
    free( keyfunction ) ;
  return False ;
}

/*
 * ؿ̾ؿֹѴԤؿ
 */
static int skkinput_string2function_number( unsigned char *keyfunction )
{
  int func_no ;
  static unsigned char *function_string_table[] = {
    "self-insert-command",
    "j-self-insert",
    "j-self-zenkaku-insert",
    "j-display-code-for-char-at-point",
    "j-set-henkan-point",
    "j-set-henkan-point-subr",
    "j-insert-a",
    "j-insert-e",
    "j-insert-i",
    "j-insert-o",
    "j-insert-u",
    "j-kana-input",
    "j-start-henkan",
    "j-insert-comma",
    "j-insert-period",
    "j-purge-from-jisyo",
    "j-input-by-code-or-menu",
    "j-mode-off",
    "j-toggle-kana",
    "j-previous-candidate",
    "j-kakutei",
    "j-abbrev-input",
    "j-abbrev-period",
    "j-abbrev-comma",
    "j-zenkaku-eiji",
    "j-zenkaku-henkan",
    "j-today",
    "save-skkinput-local-jisyo",

    "j-kanainput-mode-on",
    "newline",
    "set-mark-command",
    "forward-char",
    "backward-char",
    "delete-char",
    "delete-backward-char",
    "j-try-comletion",
    "end-of-line",
    "beginning-of-line",
    "kill-line",
    "yank",
    "kill-region",
    "kill-ring-save",
    "exchange-point-and-mark",
    "previous-line",
    "next-line",
    "transpose-chars",
    "redraw",
    "prefix-char",
    "sendback-key",
    "keyboard-quit",
    "close-skkinput",
    "vc-toggle-chatmode",
    NULL,
  } ;
  /* ɤʸ˰פΤĴ٤롣*/
  for( func_no = 0 ; function_string_table[ func_no ] != NULL ; func_no ++ )
    if( !strcmp( function_string_table[ func_no ], keyfunction ) )
      return func_no ;
  return ERR ;
}

/*
 * define-key-skkmap ᤹ؿ
 */
static int skkinput_analize_define_key_skkmap
( unsigned char *oneline, int flag )
{
  unsigned char *keystring, *keyfunction ;
  int func_no, ret ;
  ret = False ;
  /* ޤʸΥڥ᤹롣*/
  if( !skkinput_analize_define_key( oneline, &keystring, &keyfunction ) )
    return False ;
  /* ɤδؿʤΤֹ򸫤Ĵ٤롣*/
  if( ( func_no = skkinput_string2function_number( keyfunction ) ) < 0 )
    goto err_ret ;
  /* ʸʾϤ٤󤷤ơġ(;_;) */
  if( keystring[ 1 ] != '\0' || keystring[ 0 ] >= 0x80 )
    goto err_ret ;
  /* ޥåפ롣*/
  if( flag ){
    skkinput_skkmap[ keystring[ 0 ] ] = func_no ;
  } else {
    skkinput_skkabbrevmap[ keystring[ 0 ] ] = func_no ;
  }
  ret = True ;
err_ret:
  free( keystring ) ;
  free( keyfunction ) ;
  return ret ;
}

/*
 * Ԥ¹Ԥؿ
 */
static int skkinput_analize_line( unsigned char *oneline )
{
  int i ;
  unsigned char *ptr1, *ptr2 ;
  for( i = 0 ; skkinput_initdef[ i ].string != NULL ; i ++ ){
    /* ʸ󤬰פ뤫ɤȽǤ롣*/
    ptr1 = oneline ;
    ptr2 = skkinput_initdef[ i ].string ;
    while( *ptr2 != '\0' ){
      if( *ptr1 != *ptr2 || *ptr1 == '\0' )
	goto not_equal ;
      ptr1 ++ ;
      ptr2 ++ ;
    }
    /* פʸб륢¹Ԥ롣*/
    switch( skkinput_initdef[ i ].action ){
    case PARAM_BOOLEAN :
      return skkinput_analize_boolean
	( ptr1, skkinput_initdef[ i ].argument ) ;
    case PARAM_INTEGER :
      return skkinput_analize_integer
	( ptr1, skkinput_initdef[ i ].argument ) ;
    case PARAM_STRING :
      return skkinput_analize_string
	( ptr1, skkinput_initdef[ i ].argument, False ) ;
    case PARAM_PATH :
      return skkinput_analize_string
	( ptr1, skkinput_initdef[ i ].argument, True ) ;
    case PARAM_INPUT_VECTOR :
      return skkinput_analize_input_vector
	( ptr1, skkinput_initdef[ i ].argument ) ;
    case PARAM_ROM_KANA_RULE_LIST :
      return skkinput_analize_rule_list
	( ptr1, skkinput_initdef[ i ].argument ) ;
    case PARAM_DEFINE_KEY_SKKMAP :
      return skkinput_analize_define_key_skkmap( ptr1, True ) ;
    case PARAM_DEFINE_KEY_ABBREVMAP :
      return skkinput_analize_define_key_skkmap( ptr1, False ) ;
    default :
      return False ;
      break ;
    }
  not_equal:
    continue ;
  }
  return False ;
}

static struct skk_rom_kana_rule *makeSkkRomKanaRuleList
( struct skk_rom_kana_rule_list *top )
{
  int i ;
  struct skk_rom_kana_rule_list *node = top, *nNode ;
  struct skk_rom_kana_rule *rule_list = NULL ;

  if( node == NULL ){
    return skkinput_default_rom_kana_rule_list ;
  } else {
    for( i = 0 ; node != NULL ; i ++ )
      node = node->next ;
    /* 3 ĤϥǥեȤǤäƤΤ +3 */
    rule_list = malloc( sizeof( struct skk_rom_kana_rule ) * ( i + 3 ) ) ;
    if( rule_list == NULL )
      exit( 1 ) ;
    node = top ;
    for( i = 0 ; node != NULL ; i ++ ){
      rule_list[ i ] = node->element ;
#ifdef DEBUG
      fprintf( stderr, "(\"%s\", \"%s\", \"%s\", \"%s\")\n", 
	       rule_list[ i ].state, rule_list[ i ].next,
	       rule_list[ i ].houtput, rule_list[ i ].koutput ) ;
#endif
      nNode = node->next ;
      free( node ) ;
      node = nNode ;
    }
    rule_list[ i++ ] = skkinput_default_rom_kana_rule_list[ 0 ] ;
    rule_list[ i++ ] = skkinput_default_rom_kana_rule_list[ 1 ] ;
    rule_list[ i++ ] = skkinput_default_rom_kana_rule_list[ 2 ] ;
  }
  return rule_list ;
}

/*
 * եɤ߹ؿ
 *------
 * ֶϤʤΤϡ-option ǤȻפ顢ޤ DEFAULT 
 * ꡢˤեɤߡǸ option ǷꤷĤɡġ
 * ȡ ե뤬ˤΤ option ǻǤʤʤ
 *  ܰդʤġ
 */
int skkinput_readConfigFile( char *config_file )
{
  int chck = False, coding_system ;
  unsigned char buffer[ TEXTBUFSIZE ] ;
  char *config_path ;
  /* ƣ patch ġֹֹ椬ľפȤΤФ patch*/
  unsigned long line = 1 ;
  int c, count ;
  FILE *fp ;

  skkinput_rom_kana_rule_list_top = NULL ;
  /* ǥեȤΤΤѤ褦ؼ줿ν*/
  config_path = expand_file_name( config_file ) ;
  chck = True ;
  coding_system = check_skkinput_jisyo_code( config_path ) ;
  /* ե򳫤Ƥߤ롣*/
  if( ( fp = fopen( config_path, "rb" ) ) == NULL ){
#if 0
    fprintf( stderr, "Warning: configuration file does not exist.\n" ) ;
    fflush( stderr ) ;
#endif    
    if( chck )
      free( config_path ) ;
    return False ;
  }
  /* ե뤬³¤ɤ߽Ф*/
  while( !feof( fp ) ){
    /* ɤ߽Ф*/
    count = euc_fgets
      ( buffer, TEXTMAXLEN, fp, '\n', coding_system, False ) ;
    /* ʸȤƽü롣*/
    buffer[ TEXTMAXLEN ] = '\0' ;
    /* ХåեդƤޤä? */
    if( count <= 0 ){
      /* դƤޤäʬϼΤƤ롣*/
      while( ( c = fgetc( fp ) ) != EOF )
	if( c == '\n' )
	  break ;
    }
    /* ȹԤäˤ̵뤹롣*/
    if( !is_empty_line( buffer ) ){
      if( !skkinput_analize_line( buffer ) ){
	fprintf( stderr, "Warning: I ignore line %ld.\n", line ) ;
      }
    }
    line ++ ;
  }
  fclose( fp ) ;
  /* ƽλ롣*/
  if( chck )
    free( config_path ) ;

  skkinput_rom_kana_rule_list = 
    makeSkkRomKanaRuleList( skkinput_rom_kana_rule_list_top ) ;
  return True ;
}

/*
 * skkinput ưҤ򤹤ѿؿ
 *------
 * ǥեȤͤƤޤؿǤ⤢롣
 */
void initSkkinputDousaketteiVariables( void )
{
  /* ɽ꼭νߤΤʤȤƤ*/
  skkinput_local_jisyo_name  = DEFAULT_SKKLJISYO ;
  skkinput_backup_jisyo_name = DEFAULT_SKKBJISYO ;
  skkinput_record_name       = DEFAULT_SKKRECORD ;
  skk_local_jisyo_name	     = DEFAULT_SKKLOCALJISYO ;
  skkserv_portnum            = atoi( DEFAULT_SKKPORT ) ;
  /* Ķѿ SKKSERV ꤵƤС DEFAULT  skkserv *
   * host Ȥ롣*/
  if( ( skkserv_host = getenv( "SKKSERVER" ) ) == NULL )
    skkserv_host        = DEFAULT_SKKSERVER ;

  /* egg ߴ newline 򥵥ݡȤ뤫ݤ*/
  skk_egg_like_newline            = False ;
  /* chat-adapter-mode Ĥޤ newline ƬǤʤ뤫*/
  skkinput_chatadaptermode        = False ;
  /* skk-local-jisyo 򸡺뤫ɤ*/
  skkinput_search_skk_jisyo       = True ;
  /* skkinput-record 뤫ɤ*/
  skkinput_keep_record            = True ;
  /* դɽ*/
  skkinput_date_ad                = False ;
  skkinput_number_style           = False ;
  /* ϳεǽޤǤΤݤ*/
  skkinput_delete_implies_kakutei = True ;
  /* ѴǽѤ롣*/
  skkinput_use_numeric_conversion = True ;
  skkinput_rom_kana_rule_list_top = NULL ;
  skkinput_rom_kana_rule_list     = skkinput_default_rom_kana_rule_list ;
  /* */
  skkinput_tab_width              = 8 ;
  return ;
}

/*
 * Ѵץȥʤ󤫤˴طʤ SkkInputWidget 򵯤ˤäɬ
 * פ򤹤ؿ
 */
int skkinput_setSkkInputValues( Widget gw, int flag, int cmode, int eggnl )
{
  Arg arg[ 20 ] ;
  Cardinal i ;
  i = 0 ;
  if( !flag ){
    XtSetArg( arg[ i ], XtNchatAdapter,    skkinput_chatadaptermode ) ;
    i ++ ;  
    XtSetArg( arg[ i ], XtNeggLikeNewline, skk_egg_like_newline ) ;
  } else {
    XtSetArg( arg[ i ], XtNchatAdapter,    cmode ) ;
    i ++ ;  
    XtSetArg( arg[ i ], XtNeggLikeNewline, eggnl ) ;
  }
  i ++ ;  
  XtSetArg( arg[ i ], XtNdateAd,         skkinput_date_ad ) ;
  i ++ ;  
  XtSetArg
    ( arg[ i ], XtNnumberStyle,          skkinput_number_style ) ;
  i ++ ;  
  XtSetArg
    ( arg[ i ], XtNdeleteImpliesKakutei, skkinput_delete_implies_kakutei ) ;
  i ++ ;  
  XtSetArg
    ( arg[ i ], XtNuseNumericConversion, skkinput_use_numeric_conversion ) ;
  i ++ ;  
  XtSetArg( arg[ i ], XtNjisyoDirty,     skkinput_jisyo_dirty ) ;
  i ++ ;  
  XtSetArg( arg[ i ], XtNkeySkkMap,      skkinput_skkmap ) ;
  i ++ ;  
  XtSetArg( arg[ i ], XtNkeyAbbrevMap,   skkinput_skkabbrevmap ) ;
  i ++ ;  
  XtSetArg( arg[ i ], XtNkeyTwoMap,      skkinput_twokeymap ) ;
  i ++ ;  
  XtSetArg( arg[ i ], XtNinputVector, 	 skkinput_input_vector ) ;
  i ++ ;  
  XtSetArg( arg[ i ], XtNzenkakuVector,  skkinput_zenkaku_vector ) ;
  i ++ ;  
  XtSetArg( arg[ i ], XtNromKanaRuleList,skkinput_rom_kana_rule_list ) ;
  i ++ ;  
  XtSetArg( arg[ i ], XtNtabWidth,       skkinput_tab_width ) ;
  i ++ ;  
  XtSetValues( gw, arg, i ) ;
  return 0 ;
}
