/* safecat.c -- write stdin to a directory using the maildir algorithm.
 * 
 * Safecat implements the maildir algorithm of Professor DJ Bernstein,
 * which is also used by his mail agent Qmail.  This program can be
 * used to deliver data to a qmail maildir from a shell script,
 * safely, or for other related purposes (such as spooling data to a
 * directory).
 */

#include "config.h"
#include "sig.h"
#include "tempfile.h"
#include "stat_dir.h"
#include "writefile.h"
#include "strcat_alloc.h"
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>

/* Function prototypes. */


/* ****************************************************************** */
int main(int argc, char *argv[]) {
  char *tempdir  = NULL;
  char *destdir  = NULL;
  int   outfd    = 0;
  char *outfile  = NULL;
  char *tmppath  = NULL;
  char *dstpath  = NULL;
  struct stat filestat;
  unsigned int count = 0;

  /* Check that we were called with the correct number of arguments. */
  if(argc != 3) {
    fprintf(stderr, "usage: %s <tempdir> <destdir>\n", argv[0]);
    exit(-1);
  }

  /* Scan the command line arguments, to get the temp directory
     and destination directory names. */
  tempdir = argv[1];
  destdir = argv[2];

  /* Declare a handler for SIGALRM so we can time out. */
  set_handler(14, alarm_handler);

  /* Step 1:  Check that the supplied directories are OK. */
  stat_dir(tempdir);
  stat_dir(destdir);

  /* Step 2:  Stat the temporary file.  Wait for ENOENT as a response. */
  for(count=0;;count++) {
    /* Get the temporary filename to use now for dumping data. */
    outfile = mk_tempfile();
    tmppath = strcat_alloc(tempdir,outfile);
    if(stat(tmppath,&filestat) == -1 && errno == ENOENT) break;

    /* Try up to 5 times, every 2 seconds. */
    if(count == 5) {
      fprintf(stderr, "Could not stat temporary file.\n");
      exit(1);
    }

    /* Wait 2 seconds, and try again. */
    free(tmppath);
    free(outfile);
    sleep(2);
  }
  dstpath = strcat_alloc(destdir,outfile);

  /* Step 4:  Create the file tempdir/time.pid.host */
  alarm(86400);
  outfd = open(tmppath,O_WRONLY | O_EXCL | O_CREAT,0644);
  if(outfd == -1) {
    fprintf(stderr, "Could not create temp file.\n");
    exit(-1);
  }

  /* Step 5:  Copy stdin to the temp file. */
  writefile(outfd);

  /* Close the file, checking the return value. */
  if(fsync(outfd) == -1 || close(outfd) == -1) {
    fprintf(stderr, "Problem syncing file\n");
    unlink(tmppath);
    exit(-1);
  }

  /* Step 6:  Link the temp file to its final destination. */
  if(link(tmppath,dstpath) == -1) {
    perror("Could not create destnation file");
    exit(-1);
  }
  /* We've succeeded!  Now, no matter what, we return "success" */

  /* Okay, delete the temporary file. */
  unlink(tmppath);

  /* Print the name of the file we've created, as a curtesy. */
  write(STDOUT_FILENO, outfile, strlen(outfile));
  write(STDOUT_FILENO, "\n", 1);
  exit(0);
}
/* ****************************************************************** */



