/* $Id: fnode.c,v 1.4 1999/05/30 11:35:05 fraserm Exp $
   $Log: fnode.c,v $
   Revision 1.4  1999/05/30 11:35:05  fraserm
   added support for multiple sort chunks

   Revision 1.3  1999/05/26 23:51:37  fraserm
   added custom command stuff

   Revision 1.2  1999/05/16 19:48:20  fraserm
   changed n N and L to n +n N and +N

   Revision 1.1  1999/04/20 23:43:00  fraserm
   Initial revision

*/

/* file node utility functions */

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <sys/stat.h>
#include <unistd.h>
#include <pwd.h>
#include <grp.h>
#include <limits.h>
#include "limo.h"

extern struct custcmd cmds[]; /* custom commands */
extern int numcustcmds; /* number of custom commands defined */


char *sort_chunks;

/* allocate a new file node, and populate the stat structure */
struct fnode *newfnode(int argc, char *argv[], char *name, int options)
{
  struct fnode *tmp;
  int len;
  int ll;
  int i;
  static char ltbuf[PATH_MAX + 1];

  /* allocate the base structure */
  if ((tmp = (struct fnode *) malloc(sizeof(struct fnode))) == NULL) {
    fprintf(stderr, "%s: newfnode: fnode: %s\n", argv[0], strerror(errno));
    return NULL;
  }
  tmp->s = NULL;
  tmp->username = tmp->groupname = NULL; /* make sure the names are empty */
  /* allocate space to store the name */
  if ((tmp->name = (char *) malloc((len = strlen(name) + 1))) == NULL) {
    fprintf(stderr, "%s: newfnode: name: %s\n", argv[0], strerror(errno));
    return NULL;
  }
  strcpy(tmp->name, name); /* copy the name there */
  tmp->namelen = len; /* note the file length as an optimization for later */
  /* allocate space to store the stat structure */
  if ((tmp->s = (struct stat *) malloc(sizeof(struct stat))) == NULL) {
    fprintf(stderr, "%s: newfnode: s: %s\n", argv[0], strerror(errno));
    return NULL;
  }
  tmp->linklen = 0;
  /* perform the stat */
  if (lstat(name, tmp->s) != 0) {
    fprintf(stderr, "%s: %s: %s\n", argv[0], name,
	    strerror(errno)); /* just whine, don't die */
    free(tmp->s);
    tmp->s = NULL; /* null it out so it's known to be bad */
  }
  else { /* stat worked, let's check if we need to get the user or group */
    if (options & GET_OWNER) {
      struct passwd *pwd;

      errno = 0;
      if ((pwd = getpwuid(tmp->s->st_uid)) == NULL) {
	if (errno == ENOMEM) {
	  fprintf(stderr, "%s: getpwuid: %s\n", argv[0], strerror(errno));
	  tmp->username = "getpwerr";
	}
	else {
	  if ((tmp->username = (char *) malloc(10)) == NULL) { /* ugly */
	    fprintf(stderr, "%s: malloc: %s\n", argv[0], strerror(errno));
	    tmp->username = "allocerr";
	  }
	  else {
	    sprintf(tmp->username, "%d", tmp->s->st_uid); /* use the numeric
							     in emergencies */
	  }
	}
      }
      else {
	if ((tmp->username = (char *) malloc(strlen(pwd->pw_name) + 1))
	    == NULL) {
	  fprintf(stderr, "%s: malloc: %s\n", argv[0], strerror(errno));
	  tmp->username = "allocerr";
	}
	else {
	  strcpy(tmp->username, pwd->pw_name);
	}
      }
    }
    if (options & GET_GROUP) {
      struct group *grp;

      errno = 0;
      if ((grp = getgrgid(tmp->s->st_gid)) == NULL) {
	if (errno == ENOMEM) {
	  fprintf(stderr, "%s: getgrgid: %s\n", argv[0], strerror(errno));
	  tmp->groupname = "getgrerr";
	}
	else {
	  if ((tmp->groupname = (char *) malloc(10)) == NULL) { /* ugly */
	    fprintf(stderr, "%s: malloc: %s\n", argv[0], strerror(errno));
	    tmp->groupname = "allocerr";
	  }
	  else {
	    sprintf(tmp->groupname, "%d", tmp->s->st_gid); /* use the numeric
							      in emergencies */
	  }
	}
      }
      else {
	if ((tmp->groupname = (char *) malloc(strlen(grp->gr_name) + 1))
	    == NULL) {
	  fprintf(stderr, "%s: malloc: %s\n", argv[0], strerror(errno));
	  tmp->groupname = "allocerr";
	}
	else {
	  strcpy(tmp->groupname, grp->gr_name);
	}
      }
    }
    if ((options & GET_LINK_TARGET) && S_ISLNK(tmp->s->st_mode)) {
      if ((ll = readlink(tmp->name, ltbuf, PATH_MAX)) == -1) {
	strcpy(ltbuf, strerror(errno));
	fprintf(stderr, "%s: readlink of %s: %s\n",
		argv[0], tmp->name, strerror(errno));
      }
      ltbuf[ll] = '\0'; /* terminate it */
      if ((tmp->linktarget = (char *) malloc(ll+1)) == NULL) {
	fprintf(stderr, "%s: malloc: %s\n", argv[0], strerror(errno));
	tmp->linktarget = "!Error allocating memory!";
      }
      else {
	strcpy(tmp->linktarget, ltbuf);
      }
      tmp->linklen = ll; /* note link length as optimization for later */
    }
  }
  /* get custom commands */
  if (numcustcmds > 0) {
    if ((tmp->custvals = (char **) malloc((sizeof (char **)) * numcustcmds))
	== NULL
	|| (tmp->custlens = (int *) malloc((sizeof (int *)) * numcustcmds))
	== NULL) {
      fprintf(stderr, "%s: malloc: %s\n", argv[0], strerror(errno));
      return NULL;
    }
    for (i = 0; i < numcustcmds; ++i) {
      tmp->custvals[i] = get_custom(argc, argv, &cmds[i], tmp->name);
      tmp->custlens[i] = strlen(tmp->custvals[i]);
    }
  }
  /*printf("st_rdev %x\n", tmp->s->st_rdev);*/
  /* printf("name %s namelen %d linklen %d\n", tmp->name, tmp->namelen,
     tmp->linklen); */
  return tmp;
}

void fnode_free(struct fnode *fn)
{
  if (fn->s != NULL) {
    free(fn->s);
  }
  free(fn->name);
  if (fn->username != NULL) {
    free(fn->username);
  }
  if (fn->groupname != NULL) {
    free(fn->groupname);
  }
  free(fn);
}

/* generic compare routine called by qsort() */
int fnode_compare(struct fnode **first, struct fnode **second)
{
  int mult = 1;
  char *i;
  struct fnode *a = *first, *b = *second;
  int rv = 0; /* return value */

  for (i = sort_chunks; rv == 0 && *i != '\0'; ++i) {
    /* decide what to return based on the current sort_chunk */
    switch (*i) {
    case SORT_DIR_ASC:
      mult = 1;
      break;
    case SORT_DIR_DESC:
      mult = -1;
      break;
    case NAME:
      rv = mult * strcmp(a->name, b->name);
      break;
    case SIZEBLOCKS:
      rv = mult * ((long)a->s->st_blocks - (long)b->s->st_blocks);
      break;
    case BYTESSHORT:
      rv = mult * ((long)a->s->st_size - (long)b->s->st_size);
      break;
    case DESCSHORT:
      rv = mult * ((int)(a->s->st_mode & S_IFMT)
		   - (int)(b->s->st_mode & S_IFMT));
      break;
    case MODESHORT:
      rv = mult * ((int)a->s->st_mode - (int)b->s->st_mode);
      break;
    case OWNERNUMBER:
      rv = mult * ((int)a->s->st_uid - (int)b->s->st_uid);
      break;
    case OWNERNAME:
      rv = mult * strcmp(a->username, b->username);
      break;
    case GROUPNUMBER:
      rv = mult * ((int)a->s->st_gid - (int)b->s->st_gid);
      break;
    case GROUPNAME:
      rv = mult * strcmp(a->groupname, b->groupname);
      break;
    case INODE:
      rv = mult * ((long)a->s->st_ino - (long)b->s->st_ino);
      break;
    case ACCESSSHORT:
      rv = mult * ((long)a->s->st_atime - (long)b->s->st_atime);
      break;
    case MODIFYSHORT:
      rv = mult * ((long)a->s->st_mtime - (long)b->s->st_mtime);
      break;
    case CHANGESHORT:
      rv = mult * ((long)a->s->st_ctime - (long)b->s->st_ctime);
      break;
    default:
      rv = 0;
    }
  }
  return rv;
}
