/* $Id: qic02conf.c,v 1.6 1994/05/03 02:04:48 root Exp hennus $
 * Runtime configuration for QIC02 tape driver.
 * Written by Hennus Bergman.
 *
 * PRE-RELEASE version.
 *
 * This could be a dangerous program
 * if you don't know how to use it properly!
 */


#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <getopt.h>
#include <string.h>
#include <fcntl.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mtio.h>
#include <sys/file.h>
#include <linux/tpqic02.h>


char *progname = NULL;
char *tapedev = NULL;
int fd = -1;
int f_set = 0;

struct mtconfiginfo conf;

struct option longopts[] =
{
  {"file", 1, NULL, 'f'},

  {"help", 0, NULL, 'V'},
  {"version", 0, NULL, 'V'},

  {"drive", 1, NULL, 'r'},
  {"card", 1, NULL, 'c'},
  {"irq", 1, NULL, 'i'},
  {"dma", 1, NULL, 'd'},
  {"port", 1, NULL, 'p'},	/* IO base address */

  {"debug", 1, NULL, 'x'},
	
  {"have-dens", 1, NULL, 20},
  {"have-bsf", 1, NULL, 21},
  {"have-fsr", 1, NULL, 22},
  {"have-bsr", 1, NULL, 23},
  {"have-eod", 1, NULL, 24},
  {"have-seek", 1, NULL, 25},
  {"have-tell", 1, NULL, 26},
  {"have-ras1", 1, NULL, 27},
  {"have-ras2", 1, NULL, 28},
  {"have-ras3", 1, NULL, 29},
  {"have-qfa", 1, NULL, 30},

  /********* add debugging stuff here !!!! ***********/

  {NULL, 0, NULL, 0}
};



/* Note that the `id' values below *MUST* match the values used
 * by the kernel driver internally! (As listed in tpqic02.h)
 */
static struct ifc_names_struct {
  char *name;
  int id;
  char *comment;
} ifc_names[] = {
  { "wangtek", 1, "Plain Wangtek interface card" },
  { "archive", 3, "Archive SC400, SC402, SC499R cards" },
  { "everex", 2, "Everex 811V, 831V cards" },
  { "teac", 2, "Teac DC-1 interface" },
  { "wangtek3", 2, "Wangtek with different DMA3 enabler" },
  { "mountain", 5, "Mountain interface card" }
};
#define MAX_IFCNAMES (sizeof(ifc_names)/sizeof(struct ifc_names_struct))


static char * ifc_id2name(int n)
{
  int i;

  for (i=0; i<MAX_IFCNAMES; i++)
    if (ifc_names[i].id==n)
      return ifc_names[i].name;
  return "unknown";
}



void usage()
{
  int i;

  fprintf(stderr, "%s: Change configuration settings for QIC-02 tape driver.\n" \
	"$Id: qic02conf.c,v 1.6 1994/05/03 02:04:48 root Exp hennus $\n" \
	"Written by Hennus Bergman.\n\n" \
	"Use ``%s'' to show the current settings and\n" \
	"``%s ...options...'' to change them. Be careful!\n",
	  progname, progname, progname);
  fprintf(stderr, "\nKnown interface card types:\n");
  for (i=0; i<MAX_IFCNAMES; i++)
    fprintf(stderr, "\t%-15s\t%s\n", ifc_names[i].name, ifc_names[i].comment);

  exit(1);
}



volatile void die(char *s, char *t)
{
  fprintf(stderr, "%s: %s: %s\n", progname, s, t);
  exit(1);
}


void check_type(char *dev, int desc)
{
  struct stat stats;

  printf("Checking %s\n", dev);
  if (fstat (desc, &stats) == -1)
    die("cannot fstat()", dev);
  if ((stats.st_mode & S_IFMT) != S_IFCHR)
    die("not a character special file", dev);
}



void getconf(void)
{
  if (ioctl(fd, MTIOCGETCONFIG, &conf)) {
    perror(tapedev);
    die("something is wrong", "get config failed");
  }
}



void check_open(void)
{
  if (fd<0) {
    if (tapedev==NULL)
      tapedev = getenv("TAPE");

#ifdef DEFTAPE                  /* From sys/mtio.h. */
    if (tapedev==NULL)
      tapedev = DEFTAPE;
#endif

    if (tapedev==NULL)
      die("no tape device specified", "use --file=name");

    if ((fd = open(tapedev, O_RDWR))<0) {
      perror(tapedev);
      die("Could not open tape", tapedev);
    }
    check_type(tapedev, fd);
    getconf();
  }
}    



int atonum(char *s) /* stolen from setserial-2.02 */
{
  int n;

  while (*s == ' ')
    s++;
  if (strncmp(s, "0x", 2) == 0 || strncmp(s, "0X", 2) == 0)
    sscanf(s + 2, "%x", &n);
  else if (s[0] == '0' && s[1])
    sscanf(s + 1, "%o", &n);
  else
    sscanf(s, "%d", &n);
  return n;
}


void setconf(void)
{
  check_open();
  if (ioctl(fd, MTIOCSETCONFIG, &conf)) {
    perror(tapedev);
    die("something is wrong", "set config failed");
  }
}


void printconf(void)
{
  check_open();

  printf("%s: Current configuration settings are:\n" \
	 "--drive=%ld \\\n" \
	 "--card=%s \\\n" \
	 "--irq=%d \\\n" \
	 "--dma=%d \\\n" \
	 "--port=0x%x \\\n" \

	 "--debug=0x%lx \\\n" \

	 "--have-dens=%u \\\n" \
	 "--have-bsf=%u \\\n" \
	 "--have-fsr=%u \\\n" \
	 "--have-bsr=%u \\\n" \
	 "--have-eod=%u \\\n" \
	 "--have-seek=%u \\\n" \
	 "--have-tell=%u \\\n" \
	 "--have-ras1=%u \\\n" \
	 "--have-ras2=%u \\\n" \
	 "--have-ras3=%u \\\n" \
	 "--have-qfa=%u \n",
	 progname,

	 conf.mt_type,
	 ifc_id2name(conf.ifc_type),
	 conf.irqnr,
	 conf.dmanr,
	 conf.port,

	 conf.debug,

	 conf.have_dens,
	 conf.have_bsf,
	 conf.have_fsr,
	 conf.have_bsr,
	 conf.have_eod,
	 conf.have_seek,
	 conf.have_tell,
	 conf.have_ras1,
	 conf.have_ras2,
	 conf.have_ras3,
	 conf.have_qfa);

}



long doint(void)
{
  check_open();
  f_set = 1;
  return atonum(optarg);
}

#define douint() (unsigned) doint()

#define dobool() douint()
/* later */


int doifc(void)
{
  int i;

  check_open();
  f_set = 1;

  for (i=0; i<MAX_IFCNAMES; i++)
    if (!strcasecmp((char *) optarg, ifc_names[i].name))
      return (ifc_names[i].id);
  die("Unrecognized interface card name", (char *) optarg);
  /*NOTREACHED*/
  return -1; /*GCCFODDER*/
}


int main(int argc, char *argv[], char *envp[])
{
  int i;

  progname = argv[0];

  while ((i = getopt_long (argc, argv, "f:Vr:c:i:d:p:x:", longopts, (int *) 0)) != -1)
  {
      switch (i)
      {
	case 'f':
	  if (fd>=0)
	    die("Already opened", tapedev);
	  tapedev = optarg;
	  check_open();
	  break;

	case 'V':
	  usage();
	  break;

	case 'r':
	  conf.mt_type = doint();
	  break;

	case 'c':
	  conf.ifc_type = doifc();
	  break;

	case 'i':
	  conf.irqnr = doint();
	  if (conf.irqnr==2)
	    conf.irqnr=9;	/* bloody PC architecture... */
	  break;

	case 'd':
	  conf.dmanr = doint();
	  break;

	case 'p':
	  conf.port = doint() & 0xffe; /* force the last bit to be 0 */
	  break;

	case 'x':
	  conf.debug = douint();

	case 20:
	  conf.have_dens = dobool();
	  break;
	case 21:
	  conf.have_bsf = dobool();
	  break;
	case 22:
	  conf.have_fsr = dobool();
	  break;
	case 23:
	  conf.have_bsr = dobool();
	  break;
	case 24:
	  conf.have_eod = dobool();
	  break;
	case 25:
	  conf.have_seek = dobool();
	  break;
	case 26:
	  conf.have_tell = dobool();
	  break;
	case 27:
	  conf.have_ras1 = dobool();
	  break;
	case 28:
	  conf.have_ras2 = dobool();
	  break;
	case 29:
	  conf.have_ras3 = dobool();
	  break;
	case 30:
	  conf.have_qfa = dobool();
	  break;

	default:
	  printf("Unrecognized option (0x%x) \n", i);
	  usage ();
      }
  }

  if (optind != argc)
    usage ();

  if (f_set)
    setconf();
  else
    printconf();

  exit(0);
}
