/* This program is under the GNU copyright 
 * Author Damiano Bolla (Italy)
 */
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <time.h>

#include "defines.h"

time_t dtime,btime,user_time=0;

char *device;					  /* Name of the floppy device to use    */

void wait_for_disk(int num)
{
	char dummy[10];
	fprintf(stderr, "\nInsert disk %d in device %s, then press ENTER\n",num+1,device);
	read(2, dummy, 2);
}


int main(int argc, char *argv[])
{
	struct archive_header arch;				/* Struct that holds all info */
	char archname[128] = "";				/* The name of archive (read from 1st disk)  */

	char *buff;					  /* IO buffer for the data        */
	int count;
	int TmpLeng;
	int totalread=0;
	int inf;
	int More=MORE,Leng;

	int DiskNum = 0;					  /* The current disk number */


	/* Device name */
	if(argc>1)
	  device=strdup(argv[1]);
	else
	  device=DEFAULT_DEVICE;
	
	
	buff = (char *) malloc(BUFFSIZE);
	if (buff == NULL) {
		fprintf(stderr, "Sorry can't allocate buffer\n");
		 exit(2);
	 }
	
	dtime=btime=time(NULL);
	
	/* The sequence of operations is :
	 * Get the informations from the disk read data, write data
	 */
	
	while (More == MORE) {
		
		wait_for_disk(DiskNum);
		
		inf = open(device, O_RDONLY);
		if (inf < 0) {
			fprintf(stderr, "Sorry can't open device %s\n", device);
			continue;
		}

		
		/* Read archive info struct from disk.*/
		if(read(inf,&arch,sizeof(struct archive_header))<0){
			fprintf(stderr,"\nfloppyrestore: Error reading archive header!\n");
			exit(1);								/* fix */
		}

		/* Check whether this floppy is a floppybackup disk at all.*/
		if(strncmp(arch.magicbytes,MAGICBYTES,strlen(MAGICBYTES))){
			fprintf(stderr,"Not a floppybackup disk.\n");
			close(inf);
			continue;
		}
		
		/* Get the archive name from the floppy and check if OK     */
		if(DiskNum==0)
		  strcpy(archname, arch.name);
		else {
			if (strcmp(archname, arch.name) != 0) {
				fprintf(stderr,
						  "Wrong backup set.\n"
						  "This floppy belongs to set %s.\n"
						  "I want disk %d of set %s\n", arch.name,DiskNum+1,archname);
				close(inf);
				continue;
			}
		}
		
		/* Then I would like to know ehat floppy it is in the archive   */
		if (DiskNum != atoi(arch.fd_number)) {
			fprintf(stderr,"Wrong floppy - got disk %d, want disk %d\n", atoi(arch.fd_number)+1,DiskNum+1);
			close(inf);
			continue;
		}
		
		user_time+=(time(NULL)-dtime);
		dtime=time(NULL);
		
		More = arch.fd_more;
		Leng = atoi(arch.fd_length);
		
		fprintf(stderr,
				  "Archive name  : %s\n"
				  "Current disk  : %d\n"
				  "More disks    : %s\n",
				  archname,DiskNum+1,(More) ? ("Yes") : ("No"));
		/*fprintf(stderr, "Bytes on disk : %d\n", Leng);*/
		
		TmpLeng = 0;

		while ((count = read(inf, buff, BUFFSIZE)) > 0) {			
			/* The floppy will read MORE than needed, then TRUNCATE */
			if ((TmpLeng + count) > Leng) {
				write(1, buff, (Leng - TmpLeng));
				fprintf(stderr,
						  "\n(Dumped %d bytes. This is not an error.)\n",
						  (TmpLeng+count)-Leng);
				TmpLeng += count;
			} else {
				write(1, buff, count);
				TmpLeng += count;
			}
			
			if (TmpLeng >= Leng) {
				totalread+=TmpLeng;
				close(inf);
				sync();
				DiskNum++;		  /* Want the next disk         */
				break;			  /* Exit from this while       */
			}
			
			fprintf(stderr, "\rReading... %d/%d",TmpLeng,Leng);
			if(time(NULL)>dtime)
			  fprintf(stderr," (%d bytes/sec) ",TmpLeng/(int)(time(NULL)-dtime));
			fflush(stderr);
		}
		if(count<0){
			fprintf(stderr,"\nfloppyrestore: Error reading data!\n");
			exit(1);
		}
		fprintf(stderr, "\n");
		/* This while exit when more == 0 ( No more disks in archive   */
		dtime=time(NULL);						/* Needed to calc user_time in the beginning of loop */
	}
	
	close(1);						  /* Possibly you loose chars if you exit without close */
	sync();
	
	fprintf(stderr,"\nSummary: Read %d bytes ",totalread);
	dtime=time(NULL)-btime;
	if(dtime>0){
		fprintf(stderr,
				  "in %d seconds (%d bytes/sec).\n"
				  "%d seconds of the time (%d%%) was spent waiting for a new disk\n",
				  (int)(time(NULL)-btime),
				  totalread/(int)(time(NULL)-btime),
				  (int)(user_time),
				  (int)(((int)(user_time)*100)/(int)(dtime)));
	}
	fprintf(stderr,"\n");
	
	exit(0);
}
