/*
 *  apcserial.c -- Serial line functions
 *
 *  apcupsd.c   -- Simple Daemon to catch power failure signals from a
 *                 BackUPS, BackUPS Pro, or SmartUPS (from APCC).
 *              -- Now SmartMode support for SmartUPS and BackUPS Pro.
 *
 *  Copyright (C) 1996-99 Andre M. Hedrick
 *                        <hedrick@astro.dyer.vanderbilt.edu>
 *  All rights reserved.
 *
 */

/*
 *                     GNU GENERAL PUBLIC LICENSE
 *                        Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *                           675 Mass Ave, Cambridge, MA 02139, USA
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *  IN NO EVENT SHALL ANY AND ALL PERSONS INVOLVED IN THE DEVELOPMENT OF THIS
 *  PACKAGE, NOW REFERRED TO AS "APCUPSD-Team" BE LIABLE TO ANY PARTY FOR
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 *  OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ANY OR ALL
 *  OF THE "APCUPSD-Team" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  THE "APCUPSD-Team" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 *  ON AN "AS IS" BASIS, AND THE "APCUPSD-Team" HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  THE "APCUPSD-Team" HAS ABSOLUTELY NO CONNECTION WITH THE COMPANY
 *  AMERICAN POWER CONVERSION, "APCC".  THE "APCUPSD-Team" DID NOT AND
 *  HAS NOT SIGNED ANY NON-DISCLOSURE AGREEMENTS WITH "APCC".  ANY AND ALL
 *  OF THE LOOK-A-LIKE ( UPSlink(tm) Language ) WAS DERIVED FROM THE
 *  SOURCES LISTED BELOW.
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <stdlib.h>
#include <stdarg.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <termios.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <apc_config.h>
#include <apc_i18n.h>
#include <apc_version.h>
#include <apc_defines.h>
#include <apc_struct.h>
#include <apc_extern.h>

/*********************************************************************/
int setup_serial (UPSINFO *ups, int killpwr)
{
	static int has_done_setup = 0;
	
	if (!has_done_setup)
		/*
		 * Marko Sakari Asplund <Marko.Asplund@cern.ch>
		 * 
		 * prevents double init of serial port 9/25/98
		 */
		{
		has_done_setup++;
	} else {
		fprintf(stderr, "%s: serial port already initialized.\n",
		        ups->argvalue);
		exit(1);
	}
	
	/* Open the the device */
	if ((ups->fd = open(ups->device ,O_RDWR | O_NOCTTY | O_APPEND)) < 0) {
		fprintf(stderr, "%s: Cannot open UPS tty %s: %s\n",
		        ups->argvalue, ups->device, strerror(errno));
		return(1);
	}

	/*
	 * next 5 lines by BSC
	 *
	 * If create_lockfile fails there's no need to delete_lockfile.
	 * -RF
	 */
	if (create_lockfile(ups, killpwr) == LCKERROR) {
		close(ups->fd);
		ups->fd = -1;
		return(1);
	}

	tcgetattr(ups->fd, &oldtio); /* Save old settings */

	signal(SIGHUP, terminate);
	signal(SIGINT, terminate);
	signal(SIGTERM, terminate);
	signal(SIGKILL, terminate); /* But I think this is not effective -RF */

	newtio.c_cflag = DEFAULT_SPEED | CS8 | CLOCAL | CREAD;
	newtio.c_iflag = IGNPAR;             /* Ignore errors, raw input */
	newtio.c_oflag = 0;                  /* Raw output */
	newtio.c_lflag = 0;                  /* No local echo */

#ifdef __freebsd__
	newtio.c_ispeed = DEFAULT_SPEED;     /* Set input speed */
	newtio.c_ospeed = DEFAULT_SPEED;     /* Set output speed */
#endif /* __freebsd__ */

	/* w.p. This makes a non.blocking read() with 5 sec. timeout */
	newtio.c_cc[VMIN] = 0;
	newtio.c_cc[VTIME] = 50;

	tcflush(ups->fd, TCIFLUSH);
	tcsetattr(ups->fd, TCSANOW, &newtio);

	/* Have to clear RTS line to access the serial cable mode PnP */
	switch(ups->cable.type) {
		case APC_940_0095A:
		case APC_940_0095C:
			(void) ioctl(ups->fd, TIOCMBIC, &rts_bit);
		default:
			break;
	}

	if ((ups->cable.type >= CUSTOM_SMART) && (ups->mode.type >= BKPRO)) {
		char response[32];	/* w.p. */
		char a = 'Y';
		int attempts = 0;

		write(ups->fd, &a, 1);        /* This one might not work, if UPS is */
		sleep(1);                     /* in an unstable communication state */
		tcflush(ups->fd, TCIOFLUSH);  /* Discard UPS's response, if any */
smart_again:
		response[0]='\0';
		write(ups->fd, &a, 1);
		getline(response, ups); 

		if (strcmp("SM", response)) { /* w.p. */
			if (!attempts) {
				sleep(1);
				write(ups->fd, &a, 1);
				sleep(1);
				write(ups->fd, &a, 1);
				sleep(1);
				attempts++;
				goto smart_again;
			}

			fprintf(stderr,
				"%s: PANIC! Cannot talk to UPS\n\a",
				ups->argvalue);
			sleep(2);
			/*
			 * No need to do anything.
			 * Just delete the lockfile now and terminate.
			 * Killed that "if (...type >= BKPRO && open(...))" condition,
			 * seemed crappy to me: if we get here the type _is_ >= BKPRO
			 * and checking for existence with an open() is ... *argh* ...
			 * it means that someone out there don't thrust the logic of _all_
			 * the apcupsd locking code. We _have_ to trust the logic so
			 * please don't insert bogus checks.
			 * -RF
			 */
			delete_lockfile(ups);
			terminate(0);
		}
	}
	return(0);
}

/*********************************************************************/
void kill_serial (UPSINFO *ups, int killpwr)
{
	char response[32];                                         /* w.p. */
	char a;
	FILE *pwdf;
	int errflag = 0;
	response[0] = '\0';

	if (killpwr) {
		if (ups->mode.type <= SHAREBASIC) {
			for (killcount=0; killcount<3; killcount++) {
				ioctl(ups->fd, TIOCMGET, &flags);
				sleep(1);
				switch(ups->cable.type) {
					case CUSTOM_SIMPLE:
						ups->LineUp = (flags & cd_bit);
						break;
					case APC_940_0020B:
						ups->LineUp = (flags & cts_bit);
						break;
					case APC_940_0023A:
						ups->LineUp = (flags & cd_bit);
						break;
					case APC_940_0095A:
					case APC_940_0095C:
						ups->LineUp = (flags & rng_bit);
						break;
					case CUSTOM_SMART:
					case APC_940_0024B:
					case APC_940_0024C:
					case APC_940_1524C:
					case APC_940_0024G:
					case APC_NET:
						break;
					default:
						ups->LineUp = 1;
						break;
				}
			}
		}
		if ((((pwdf = fopen(PWRFAIL, "r" )) == NULL) &&
		    (ups->mode.type != BK)) ||
		    (((pwdf = fopen(PWRFAIL, "r" )) == NULL) &&
		    (ups->LineUp != 0) && (ups->mode.type == BK))) {
			fprintf(stderr,
			        "%s: Attempting to kill the power!\n" \
			        "%s: Nice TRY but not TODAY!\n" \
			        "%s: Shutdown not called first.\n" \
			        "%s: Bug found by Tom Kunicki\n",
			        ups->argvalue, ups->argvalue,
			        ups->argvalue, ups->argvalue);
			terminate(0);
		} else {
			errflag=0;                             /* w.p. */
			if ((ups->class.type == SHAREMASTER) ||
			    (ups->class.type == SHARENETMASTER)) {
				fprintf(stderr,
				        "%s: Waiting 30 seconds for slave(s) to shutdown.\n" \
				        "%s: BETA blind code writing. Here!!\n" \
				        "%s: Someone tell me if this works...\n",
				        ups->argvalue, ups->argvalue, ups->argvalue);
				sleep(30);
			}
			if (ups->class.type == NETMASTER) {
				fprintf(stderr,
					"%s: Waiting 15 seconds for slave(s) to shutdown.\n",
					ups->argvalue);
				sleep(15);
			}
			fclose(pwdf);

			fprintf(stderr,
				"%s: Attempting to kill the power!\n",
				ups->argvalue);

			if (ups->mode.type == BK) {
				switch(ups->cable.type) {
					case CUSTOM_SIMPLE:           /* killpwr_bit */
					case APC_940_0095A:
					case APC_940_0095C:          /* killpwr_bit */
						(void) ioctl(ups->fd, TIOCMBIS, &rts_bit);
						(void) ioctl(ups->fd, TIOCMBIS, &rts_bit);
						(void) ioctl(ups->fd, TIOCMBIS, &rts_bit);
						break;
					case APC_940_0020B:           /* killpwr_bit */
						(void) ioctl(ups->fd, TIOCMBIS, &dtr_bit);
						(void) ioctl(ups->fd, TIOCMBIS, &dtr_bit);
						(void) ioctl(ups->fd, TIOCMBIS, &dtr_bit);
						break;
					case APC_940_0023A:          /* killpwr_bit */
						break;
					case CUSTOM_SMART:
					case APC_940_0024B:
					case APC_940_0024C:
					case APC_940_1524C:
					case APC_940_0024G:
					case APC_NET:
					default:
						break;
				}
				sleep(10);
				close(ups->fd);
				fprintf(stderr,
					"%s: Drat!! Failed to kill the power!\n" \
					"%s: Great Mains Returned, Trying a REBOOT!\n",
					ups->argvalue, ups->argvalue);
				system(ups->reboot_cmd);
				fprintf(stderr,
					"%s: Perform CPU-reset or power-off\n",
					ups->argvalue);
				terminate(0);
			} else if (ups->mode.type == SHAREBASIC) {
				sleep(10);
				close(ups->fd);
				fprintf(stderr,
					"%s: Waiting For ShareUPS Master to shutdown.\n",
					ups->argvalue);
				sleep(60);
				fprintf(stderr,
					"%s: Drat!! Failed to have power killed by Master!\n" \
					"%s: Great Mains Returned, Trying a REBOOT!\n",
					ups->argvalue, ups->argvalue);
				system(ups->reboot_cmd);
				fprintf(stderr,
					"%s: Perform CPU-reset or power-off\n",
					ups->argvalue);
				terminate(0);
			} else {
				a = 'S';
				write(ups->fd, &a, 1);
				/*  Check wether the UPS has acknowledged the power-off command.
				 *  This might not happen in rare cases, when mains-power returns
				 *  just after LINUX starts with the shutdown sequence. 
				 *   If the UPS doesnt acknowledge the command, it will not interrupt
				 *  the ouput-power. So LINUX will not come up without operator
				 *  intervention.  w.p.
				 */
				sleep(10);
				getline(response, ups);
				if (strcmp(response, "OK") == 0)
					fprintf(stderr, "%s: UPS will power off ...\n",
						ups->argvalue);
				if (strcmp(response,"NA") == 0) {
					fprintf(stderr,
						"%s: Mains-power returned during shutdown!\n" \
						"%s: Trying alternate method for killing power.\n",
						ups->argvalue, ups->argvalue);
					/*
					 *  w.p. experiments shows that UPS needs
					 *  w.p. delays between chars to accept
					 *  w.p. this command
					 */
					a = '@';
					write(ups->fd, &a, 1);
					sleep(1);
					a = '0';
					write(ups->fd, &a, 1); 
					sleep(1);
					a = '0';
					write(ups->fd, &a, 1);
					sleep(1); 
					a = '0';
					write(ups->fd, &a, 1);
					sleep(2);
					getline(response, ups);
					if (strcmp(response, "OK") == 0)
						fprintf(stderr,
							"%s: UPS power-cycle in progress ...\n",
							ups->argvalue);
					else
						errflag++;
				}
				if (strcmp(response, "OK") && strcmp(response, "NA")) {
					fprintf(stderr,
						"%s: Mains-power returned during shutdown!\n" \
						"%s: Trying alternate method for killing power.\n" \
						"%s: Using non SU SmartUPS alternate method.\n",
						ups->argvalue, ups->argvalue, ups->argvalue);
					a = '@';
					write(ups->fd, &a, 1);
					sleep(1);
					a = '0';
					write(ups->fd, &a, 1);
					sleep(1);
					a = '0';
					write(ups->fd, &a, 1);
					sleep(1);
					if ((ups->mode.type == BKPRO) || (ups->mode.type == VS)) {
						a = '1';
						fprintf(stderr,
							"%s: BackUPS Pro and SmartUPS v/s sleep for 6 min.\n",
							ups->argvalue);
					} else {
						a = '0';
					}
					write(ups->fd, &a, 1);
					sleep(1);
					a = 'K';
					write(ups->fd, &a, 1);
					sleep(2);
					getline(response, ups);
					if ((strcmp(response,"*") == 0) ||
					    (ups->mode.type >= BKPRO))
						fprintf(stderr,
							"%s: UPS power-cycle in progress ...\n",
							ups->argvalue);
					else
						errflag++;
				}
				if (errflag) {
					fprintf(stderr,
						"%s: Unexpected error!\n" \
						"%s: Perform CPU-reset or power-off\n",
						ups->argvalue, ups->argvalue);
					system(ups->reboot_cmd);
				}
			}
			/* w.p. */
			sleep(10);
			terminate(0);
		}
	} else {
		if (ups->mode.type <= SHAREBASIC) {
			switch(ups->cable.type) {
				case CUSTOM_SIMPLE:
					(void) ioctl(ups->fd, TIOCMBIC, &dtr_bit);  /* power_bit */
					(void) ioctl(ups->fd, TIOCMBIC, &rts_bit);  /* killpwr_bit */
					break;
				case APC_940_0020B:
					(void) ioctl(ups->fd, TIOCMBIC, &dtr_bit);  /* killpwr_bit */
					break;
				case APC_940_0023A:
					break;
				case APC_940_0095A:
				case APC_940_0095C:
					(void) ioctl(ups->fd, TIOCMBIC, &rts_bit);  /* killpwr_bit */
					(void) ioctl(ups->fd, TIOCMBIC, &cd_bit);   /* lowbatt_bit */
					(void) ioctl(ups->fd, TIOCMBIC, &rts_bit);  /* killpwr_bit */
					break;
				case CUSTOM_SMART:
				case APC_940_0024B:
				case APC_940_0024C:
				case APC_940_1524C:
				case APC_940_0024G:
				case APC_NET:
					default:
					fprintf(stderr,
						"%s: Simple Cable Signal Lost!!!\n\a",
						ups->argvalue);
					terminate(0);
					break;
			}
		}
	}
}

/*********************************************************************/
int check_serial (UPSINFO *ups)
{
	if (ups->mode.type <= SHAREBASIC) {
		ioctl(ups->fd, TIOCMGET, &flags);
		switch(ups->cable.type) {
			case CUSTOM_SIMPLE:
				ups->LineUp = (flags & cd_bit);
				if (flags & cts_bit)
					ups->BattUp = 0;
				else
					ups->BattUp = 1;
				ups->LineDown = (flags & sr_bit);
				break;
			case APC_940_0020B:
				ups->LineUp = (flags & cts_bit);
				ups->BattUp = (flags & cd_bit);
				ups->LineDown = 0;
				break;
			case APC_940_0023A:
				ups->LineUp = (flags & cd_bit);

/* BOGUS STUFF MUST FIX IF POSSIBLE */

				ups->BattUp = (flags & sr_bit);

/* BOGUS STUFF MUST FIX IF POSSIBLE */

				ups->LineDown = 0;
				break;
			case APC_940_0095A:
			case APC_940_0095C:
				ups->LineUp = (flags & rng_bit);
				ups->BattUp = (flags & cd_bit);
				ups->LineDown = 0;
				break;
			case CUSTOM_SMART:
			case APC_940_0024B:
			case APC_940_0024C:
			case APC_940_1524C:
			case APC_940_0024G:
			case APC_NET:
			default:
				return(1);
		}
	} else
		fillUPS(ups);
	sleep(1);
	return(0);
}

/*********************************************************************/
void prep_serial (UPSINFO *ups)
{
	if (ups->mode.type >= BKPRO)
#if 0
		setup_extended(ups);
#else
		read_extended(ups);
#endif
	if (ups->slave_count >= 1)
		ups->slave_bool = TRUE;
	if (logfile != NULL)
		time(&last_time_logging);
	if (procfile != NULL)
		time(&last_time_procfs);

	if (ups->slave_bool) {
		int i = 0;
		for (i=0; i<ups->slave_count; i++)
			logprintf("found slave: %s\n", ups->slaves[i].name);
	}

	if (ups->nettime != 0)
		time(&last_time_net);
}

#ifndef NEW_THREADS

/*********************************************************************/
void do_serial (UPSINFO *ups)
{
	int count = 0;

	while(1) {

		if ((!ups->LineUp) && (ups->maxtime != 0))
			time(&last_time_on_line);
		if (!ups->LineUp)
			time(&last_time_annoy);
		if (!ups->LineUp)
			time(&last_time_delay);
		if (!ups->LineUp)
			time(&last_time_nologon);
		/*
		 *  Now sample the status_bit line.
		 */
		if (check_serial(ups))
			logprintf("%s: Simple Cable Signal Lost!!!\n", ups->argvalue);

		if (ups->enable_access.type == TRUE) {
			(void) pipe_requests(ups);
			/*
			 *  Must SHUTDOWN Remote System Calls
			 */
			if (ups->remotedown) {
				ups->BatteryUp = 0;
				ups->ShutDown = 1;
				if (ups->slave_bool)
					send_to_slaves(ups);
				if (ups->remote_cmd[0])
					system(ups->remote_cmd);
				delete_lockfile(ups);
				if (ups->nologin_file != TRUE)
					make_file(NOLOGIN);
				ups->nologin_file = TRUE;
				logonfail(0);
				system(ups->remote_shutdown);
				make_file(PWRFAIL);
				powerfail(3);
			}
		}
		/*
		 *  Did status_bit go high? Then the power is failing.
		 */
		if (ups->OldLineUp == 0 && ups->LineUp != 0) {
			count++;
			if (count > 3) {
				system(ups->powerout);
				if (ups->slave_bool)
					send_to_slaves(ups);
				if (logfile != NULL)
					updating_logs(logfile, ups);
				if (procfile != NULL)
					updating_procfs(procfile, ups);
				if (ups->pwr_cmd[0])
					system(ups->pwr_cmd);
				/*
				 *  Now enable the DTR. (power_bit) for _CUSTOM
				 */
				if (ups->cable.type == CUSTOM_SIMPLE)
					(void) ioctl(ups->fd, TIOCMBIS, &dtr_bit);
				ups->OldLineUp = ups->LineUp;
				ups->OldBattUp = ups->BattUp;
				ups->OldLineDown = ups->LineDown;
				sleep(10);

				/*
				 *  Now re-sample the status_bit line.
				 *
				 *  This is done to prevent over reaction of the daemon,
				 *  if the local power is fuzzy.
				 *  Usually stormy nights/
				 */
				(void) check_serial(ups);
				sleep(1);
				/*
				 *  Did the second test verify the power is failing?
				 */
				if ((ups->LineUp) && (ups->OldLineUp)) {
					if (ups->slave_bool)
						send_to_slaves(ups);
					if (logfile != NULL)
						updating_logs(logfile, ups);
					if (procfile != NULL)
						updating_procfs(procfile, ups);
					ups->BatteryUp = 1;
#if 0
					if (ups->nologin.type == ALWAYS) {
						make_file(NOLOGIN);
						ups->nologin_file = TRUE;
						logonfail(0);
					}
#endif
					system(ups->onbatt);
				}
			} else
				continue;
		}
		/*
		 *  Did lowbatt_bit go high? Then the battery power is failing.
		 *  Normal Power down during Power Failure
		 */
		if ((ups->OldBattUp == 0 && ups->BattUp > 0) &&
		    (ups->BatteryUp == 1 && ups->ShutDown == 0)) {
			count++;
			if (count > 3) {
				ups->BatteryUp = 0;
				ups->ShutDown = 1;
				if (ups->slave_bool)
					send_to_slaves(ups);
				delete_lockfile(ups);
				system(ups->failing);
				if (ups->batt_cmd[0]) system(ups->batt_cmd);
					make_file(PWRFAIL);
				powerfail(2);
				if (ups->nologin_file != TRUE)
					make_file(NOLOGIN);
				logonfail(0);
				system(ups->doshutdown);
			} else
				continue;
			}
		/*
		 *  Did MaxTimeOnBattery Expire?
		 *  Normal Power down during Power Failure
		 */
		if ((ups->maxtime != 0) && ups->LineUp &&
		    ((time(NULL) - last_time_on_line) > ups->maxtime)) {
			count++;
			if (count > 3) {
				ups->ShutDown = 1;
				ups->timedout = TRUE;
				if (ups->slave_bool)
					send_to_slaves(ups);
				if (logfile != NULL)
					updating_logs(logfile, ups);
				if (procfile != NULL)
					updating_procfs(procfile, ups);
				if (ups->time_cmd[0])
					system(ups->time_cmd);
				delete_lockfile(ups);
				system(ups->timeout);
				make_file(PWRFAIL);
				powerfail(2);
				if (ups->nologin_file != TRUE)
					make_file(NOLOGIN);
				logonfail(0);
				system(ups->doshutdown);
			} else
				continue;
		}
		/*
		 *  Did Battery Charge go below percent cutoff?
		 *  Normal Power down during Power Failure
		 */
		if ((ups->mode.type >= NBKPRO) && ups->LineUp &&
		    (ups->BatLoad <= ups->percent)) {
			count++;
			if (count > 3) {
				ups->ShutDown = 1;
				ups->load = TRUE;
				if (ups->slave_bool)
					send_to_slaves(ups);
				if (logfile != NULL)
					updating_logs(logfile, ups);
				if (procfile != NULL)
					updating_procfs(procfile, ups);
				if (ups->load_cmd[0])
					system(ups->load_cmd);
				delete_lockfile(ups);
				system(ups->loadlimit);
				make_file(PWRFAIL);
				powerfail(2);
				if (ups->nologin_file != TRUE)
					make_file(NOLOGIN);
				logonfail(0);
				system(ups->doshutdown);
			} else
				continue;
		}
		/*
		 *  Did Battery Runtime go below percent cutoff?
		 *  Normal Power down during Power Failure
		 */
		if ((ups->mode.type >= NBKPRO) && ups->LineUp &&
		    (ups->TimeLeft <= ups->runtime)) {
			count++;
			if (count > 3) {
				ups->ShutDown = 1;
				ups->timelout = TRUE;
				if (ups->slave_bool)
					send_to_slaves(ups);
				if (logfile != NULL)
					updating_logs(logfile, ups);
				if (procfile != NULL)
					updating_procfs(procfile, ups);
				if (ups->limit_cmd[0])
					system(ups->limit_cmd);
				delete_lockfile(ups);
				system(ups->runlimit);
				make_file(PWRFAIL);
				powerfail(2);
				if (ups->nologin_file != TRUE)
					make_file(NOLOGIN);
				logonfail(0);
				system(ups->doshutdown);
			} else
				continue;
		}
		/*
		 *  Did status_bit go down again? Then the power is back.
		 *  No shutdown to cancel.
		 */
		if ((ups->OldLineUp > 0 && ups->LineUp == 0) && ups->ShutDown == 0) {
			count++;
			if (count > 3) {
				ups->BatteryUp = 0;
				system(ups->mainsback);
				if (ups->slave_bool)
					send_to_slaves(ups);
				if (logfile != NULL)
					updating_logs(logfile, ups);
				if (procfile != NULL)
					updating_procfs(procfile, ups);
				if (ups->ret_cmd[0])
					system(ups->ret_cmd);
				logonfail(1);
				remove_file(NOLOGIN);
				ups->nologin_file = FALSE;
				if (ups->cable.type == CUSTOM_SIMPLE) {
					(void) ioctl(ups->fd, TIOCMBIC, &dtr_bit);
					/*
					 *  Clearing DTR and TxD (ST).
					 */
					(void) ioctl(ups->fd, TIOCMBIC, &st_bit);
				} 
			} else
				continue;
		}
		/*
		 *  Did status_bit go down again? Then the power is back.
		 *  Shutdown to cancel.
		 */
		if ((ups->OldLineUp > 0 && ups->LineUp == 0) && ups->ShutDown == 1) {
			count++;
			if (count > 3) {
				ups->BatteryUp = 0;
				ups->ShutDown = 0;
				system(ups->waitasec);
				if (ups->slave_bool)
					send_to_slaves(ups);
				if (logfile != NULL)
					updating_logs(logfile, ups);
				if (procfile != NULL)
					updating_procfs(procfile, ups);
				if (ups->ret_cmd[0])
					system(ups->ret_cmd);
				powerfail(1);
				remove_file(PWRFAIL);
				logonfail(1);
				remove_file(NOLOGIN);
				ups->nologin_file = FALSE;

				if (ups->cable.type == CUSTOM_SIMPLE) {
					(void) ioctl(ups->fd, TIOCMBIC, &dtr_bit);
					/*
					 *  Clearing DTR and TxD (ST).
					 */
					(void) ioctl(ups->fd, TIOCMBIC, &st_bit);
				}
			} else
				continue;
		}
		/*
		 *  Announce to LogOff, with initial delay ....
		 */
		if ((ups->BatteryUp == 1 && ups->ShutDown == 0 && ups->LineUp > 0) &&
		    ((time(NULL) - last_time_delay) > ups->delay) &&
		    ((time(NULL) - last_time_annoy) > ups->annoy) &&
		    (ups->nologin_file == TRUE)) {
			system(ups->annoyme);
			if (ups->slave_bool)
				send_to_slaves(ups);
			if (ups->pwr_cmd[0])
				system(ups->pwr_cmd);
			time(&last_time_annoy);
		}

		/*
		 *  Delay NoLogons....
		 */
		if ((ups->BatteryUp == 1) && (ups->ShutDown == 0) &&
		    (ups->LineUp > 0) && (ups->nologin_file != TRUE)) {
			count++;
			if (count > 3) {
				switch(ups->nologin.type) {
					case NEVER:
						break;
					case TIMEOUT:
						if ((time(NULL) - last_time_nologon) > ups->nologin_time) {
							make_file(NOLOGIN);
							ups->nologin_file = TRUE;
							logonfail(0);
						}
						break;
					case PERCENT:
						if (ups->nologin_time >= ups->BatLoad) {
							make_file(NOLOGIN);
							ups->nologin_file = TRUE;
							logonfail(0);
						}
						break;
					case MINUTES:
						if (ups->nologin_time >= ups->TimeLeft) {
							make_file(NOLOGIN);
							ups->nologin_file = TRUE;
							logonfail(0);
						}
						break;
					case ALWAYS:
					default:
						make_file(NOLOGIN);
						ups->nologin_file = TRUE;
						logonfail(0);
						break;
				}
			} else
				continue;
		}
		/*
		 *  Did lowbatt_bit never go low then high?
		 *  Then the battery power has failed!!!
		 *  Must SHUTDOWN NOW
		 */
		if ((ups->LineUp != 0 && ups->BattUp == 1 && ups->LineDown == 0) &&
		    (ups->BatteryUp == 1 && ups->ShutDown == 0)) {
			count++;
			if (count > 3) {
				ups->BatteryUp = 0;
				ups->ShutDown = 1;
				ups->emergencydown = TRUE;
				if (ups->slave_bool)
					send_to_slaves(ups);
				delete_lockfile(ups);
				system(ups->emergency);
				if (ups->batt_cmd[0])
					system(ups->batt_cmd);
				if (ups->nologin_file != TRUE)
					make_file(NOLOGIN);
				ups->nologin_file = TRUE;
				logonfail(0);
				make_file(PWRFAIL);
				powerfail(2);
			} else
				continue;
		}

		if (ups->slave_bool) {
			int i = 0;
			int j = 0;
			for (i=0; i<ups->slave_count; i++) {
				if (ups->slaves[i].present == 3) {
					for (j=0; j<5; j++) {
						if (!send_to_each(ups, i))
							goto next_slave;
					}
				}
next_slave:
			}
		}

		if ((logfile != NULL) &&
		    ((time(NULL) - last_time_logging) > ups->logtime)) {
			updating_logs(logfile, ups);
		}             /* Do Logging */

		if ((procfile != NULL) &&
		    ((time(NULL) - last_time_procfs) > ups->proctime)) {
			updating_procfs(procfile, ups);
		}          /* Do Procfs */

		if ((ups->slave_bool) &&
		    ((time(NULL) - last_time_net) > ups->nettime)) {
			send_to_slaves(ups);
		}        /* Do slaves */

		/*
		 *  Reset count, remember status and sleep 2 seconds.
		 */
		count = 0;
		ups->OldLineUp = ups->LineUp;
		ups->OldBattUp = ups->BattUp;
		ups->OldLineDown = ups->LineDown;
		sleep(2);
	}
}

#else /* NEW_THREADS */
/*********************************************************************/
void timer_serial (int sig)
{
	UPSINFO *ups = &myUPS;

	read_andlock_shmarea(ups);

	if ((!ups->LineUp) && (ups->maxtime != 0))
		time(&last_time_on_line);
	if (!ups->LineUp)
		time(&last_time_annoy);
	if (!ups->LineUp)
		time(&last_time_delay);
	if (!ups->LineUp)
		time(&last_time_nologon);

	/*
	 *  Now sample the status_bit line.
	 */

	if (check_serial(ups))
		logprintf("%s: Simple Cable Signal Lost!!!\n", ups->argvalue);

	write_andunlock_shmarea(ups);
	signal(SIGALRM, timer_serial);	/* reinstall signal */
	alarm(TIMER_SERIAL);
}

/*********************************************************************/
void do_serial (void) {
	init_thread_signals();
	init_timer(TIMER_SERIAL, timer_serial);
	sleep_forever();
}
#endif /* NEW_THREADS */
