/* $Id: gladesig.c,v 2.1 1998/09/19 03:31:24 sgifford Exp $ */

/* Main program and signal handlers.  Partially generated by Glade. */

/*
    Copyright (C) 1998  Scott Gifford

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    If you have any questions, comments, or compliments, feel free to contact
    me at <sgifford@tir.com>.
*/

#include <stdio.h>
#include <ctype.h>
#include <gtk/gtk.h>

#include "getopt.h"
#include "utils.h"
#include "gladesrc.h"
#include "gladesig.h"
#include "dict.h"

int numwindows=0;

#define OPT_DEFINE    0
#define OPT_SEARCH    1
#define OPT_CLIPBOARD 2
#define OPT_DICT      3
#define OPT_HELP      4
#define OPT_VERSION   5

struct option opts[]={
  {
    "define",
    0,
    NULL,
    0
  },
  {
    "search",
    0,
    NULL,
    0
  },
  {
    "clipboard",
    0,
    NULL,
    0
  },
  {
    "dict",
    1,
    NULL,
    0
  },
  {
    "help",
    0,
    NULL,
    0
  },
  {
    "version",
    0,
    NULL,
    0
  },
};
  
void usage(char *progname)
{
  printf("Usage:\n     %s [--help|--version] [--search|--define] [--clipboard] [--dict=/path/to/dict]\n",progname);
  printf("     --help: Display all options available for wordinspect.
     --version: Display wordinspect's current version and copyright.
     --search: Search for the word provided on the command line 
               [Default].
     --define: Define the word provided on the command line.
     --clipboard: Use the word currently in X's primary selection (the
                  clipboard) as if it were typed on the command line.
                  Useful for launching from a menu or hot-key.
     --dict=/path/to/dict: Use the copy of dict in /path/to/dict.
");

}

#define DO_NORMAL 0
#define DO_DEFINE 1
#define DO_SEARCH 2

#define SO_NONE      0
#define SO_CL        1
#define SO_CLIPBOARD 2
int
main (int argc, char *argv[])
{
  int c;
  int digit_optind=0;
  int dowhat=DO_NORMAL;
  int source=SO_NONE;
  char *word;
  
  gtk_set_locale ();
  gtk_init (&argc, &argv);

  while ((c=getopt_long(argc, argv, "",opts,&digit_optind))!=-1)
  {
    switch(c)
    {
      case 0:
        switch(digit_optind)
        {
          case OPT_DEFINE:
            dowhat=DO_DEFINE;
            break;
          case OPT_SEARCH:
            dowhat=DO_SEARCH;
            break;
          case OPT_DICT:
            dictprog=safe_strdup(optarg);
            break;
          case OPT_CLIPBOARD:
            source=SO_CLIPBOARD;
            break;
          case OPT_HELP:
            usage(argv[0]);
            exit(0);
          case OPT_VERSION:
            printf("Word Inspector version %s\nCopyright (C) 1998 Scott Gifford <sgifford@tir.com>\nLicensed under the GNU Public License.\n",VERSION);
            exit(0);
          default:
            usage(argv[0]);
            exit(-1);
        }
        break;
      default:
        usage(argv[0]);
        exit(-1);
    }
  }

  if (source==SO_CLIPBOARD)
  {
    /* Well, MAYBE... */
    word="*CLIPBOARD*";
  }
  else if (argc >= optind)
    word=argv[optind];
  else
    word="";

  if (dowhat==DO_DEFINE)
  {
    if (!word || !word[0])
    {
      fprintf(stderr,"Define what?\n");
      usage(argv[0]);
      exit(-1);
    }
    do_define(NULL,word);
  }
  else
  {
    if (!word) word="";
    do_find(NULL,word);
  }
/*   winWordFinder = create_winWordFinder (); */
/*   gtk_widget_show (winWordFinder); */

  gtk_main ();
  return 0;
}

void
do_quit                                (GtkButton       *button,
                                        gpointer         user_data)
{
  gtk_widget_destroy(GTK_WIDGET(button));
}

int do_delwindow(GtkWidget *win, gpointer user_data)
{
  if (!--numwindows)
    gtk_main_quit();
    
  return TRUE;
}

/* Our state machine constants */
#define DEFST_TEXT      0
#define DEFST_SEEALSO   1
#define DEFST_LINESTART 2
#define DEFST_SOURCE    3

int on_buttonSearch_clicked(GtkWidget *w, gpointer user_data)
{
  define_word(gtk_entry_get_text(GTK_ENTRY(user_data)));
  return TRUE;
}

int isdictchar(char c)
{
  return ( (isalnum(c)) || (c=='\'') );
}

int define_word(char *word)
{
  GtkWidget *winWordInspector;
  
  winWordInspector = create_winWordInspector ();
  gtk_widget_show (winWordInspector);
  ++numwindows;

  if (strcmp(word,"*CLIPBOARD*")==0)
  {
    GdkAtom target_atom;

    gtk_signal_connect(GTK_OBJECT(winWordInspector),"selection_received",GTK_SIGNAL_FUNC(selection_received_define),NULL);
    target_atom = gdk_atom_intern("STRING",FALSE);
    gtk_selection_convert(winWordInspector,GDK_SELECTION_PRIMARY,target_atom,GDK_CURRENT_TIME);
  }
  else
  {
    set_def(winWordInspector,word);
  }
  return TRUE;
}

void set_def(GtkWidget *winWordInspector, char *word)
{
  GtkWidget *labelWord, *labelDef;
  DEFINITIONLIST *deflist;
  GdkColormap *cmap;
  GdkColor     seealso_color;
  GdkColor    *color;
  GdkFont     *font;
  GdkFont     *enum_font;
  GdkFont     *source_font;
  int          state;
  char        *curpos;
  int          indentnum;
  char        *title;

  if (!(deflist=dict_lookup((WORD *)word)))
  {
    /* The error should have already been handled. */
    return;
  }
  labelWord=gtk_object_get_data(GTK_OBJECT(winWordInspector),"labelWord");
  gtk_label_set(GTK_LABEL(labelWord),word);

  title=safe_malloc(18+strlen(word));
  sprintf(title,"Word Inspector - %s",word);
  gtk_window_set_title(GTK_WINDOW(winWordInspector),title);
  free(title);

  labelDef=gtk_object_get_data(GTK_OBJECT(winWordInspector),"labelDef");

  /* Should we free these? */
  if (!(enum_font=gdk_font_load("-*-*-bold-*-*-*-*-*-*-*-*-*-*-*")))
    g_error("couldn't load bold font");
  if (!(source_font=gdk_font_load("-*-*-bold-i-*-*-*-120-*-*-*-*-*-*")))
    g_error("couldn't load big, bold, italic font");
  font=NULL;

  cmap = gdk_colormap_get_system();
  seealso_color.red = 0;
  seealso_color.green = 0;
  seealso_color.blue = 0xffff;
  if (!gdk_color_alloc(cmap, &seealso_color)) {
    g_error("couldn't allocate color");
  }
  color=NULL;
  
  gtk_text_freeze(GTK_TEXT(labelDef));
  state=DEFST_TEXT;
  indentnum=0;
  for(curpos=(char *)deflist->defs[0]->text;*curpos;curpos++)
  {
    switch(state)
    {
      case DEFST_TEXT:
        switch(*curpos)
        {
          case '{':
            state=DEFST_SEEALSO;
            color=&seealso_color;
            continue;
          case '\n':
            state=DEFST_LINESTART;
            indentnum=0;
            break;
        }
        break;
      case DEFST_SEEALSO:
        switch(*curpos)
        {
          case '}':
            state=DEFST_TEXT;
            color=NULL;
            continue;
        }
        break;
      case DEFST_LINESTART:
        if (isspace(*curpos))
        {
          if (*curpos!='\n')
          {
            indentnum++;
            font=NULL;
            state=DEFST_TEXT;
          }
          else
          {
          }
        }
        else
        {
          /* Sometimes we can tell what part we're on by how far it's
           * indented.
           */
          switch(indentnum)
          {
            case 0:
              state=DEFST_SOURCE;
              font=source_font;
              break;
          }
        }
        break;
      case DEFST_SOURCE:
        if (*curpos=='\n')
          state=DEFST_LINESTART;
        break;
    }
    gtk_text_insert(GTK_TEXT(labelDef),font,color,NULL,curpos,1);
  }
  gtk_text_thaw(GTK_TEXT(labelDef));
}  

void
selection_received_define (GtkWidget *w, GtkSelectionData *selection_data, 
                           gpointer data)
{
  gchar *fixed;
  
  /* **** IMPORTANT **** Check to see if retrieval succeeded  */
  if (selection_data->length < 0)
  {
    g_print ("Selection retrieval failed\n");
    return;
  }

  /* Make sure we got the data in the expected form */
  if (selection_data->type != GDK_SELECTION_TYPE_STRING)
  {
    g_print ("Selection \"STRING\" was not returned as string!\n");
    return;
  }
        
  fixed=cleanup((char *)selection_data->data);
  set_def(w,fixed);
  free(fixed);
}


int on_labelDef_button_release_event_before(GtkWidget *w, gpointer user_data)
{
  GdkEventButton *bevent=(GdkEventButton *)user_data;
  gchar *sel;

  if (bevent->type != GDK_BUTTON_PRESS)
    return FALSE;

  if (bevent->button != 3)
    return FALSE;
  
  if ((GTK_EDITABLE(w)->has_selection) &&
      (GTK_EDITABLE(w)->selection_start_pos != GTK_EDITABLE(w)->selection_end_pos))
  {
    gchar *fixedsel;
    
    sel=gtk_editable_get_chars(GTK_EDITABLE(w),
                               GTK_EDITABLE(w)->selection_start_pos,
                               GTK_EDITABLE(w)->selection_end_pos);

    fixedsel=cleanup(sel);
    free(sel);

    /* Jot this down... */
    gtk_object_set_data (GTK_OBJECT (w), "PrevSel", fixedsel);
    return TRUE;
  }
  /* Clear this out, just in case. */
  if ( (sel=gtk_object_get_data(GTK_OBJECT(w),"PrevSel")) != NULL)
  {
    free(sel);
    gtk_object_set_data (GTK_OBJECT (w), "PrevSel", NULL);
  }
  return FALSE;
}

int on_labelDef_button_release_event(GtkWidget *w, gpointer user_data)
{
  int left, right,max;
  char *chunk;
  guint pos;
  GdkEventButton *bevent=(GdkEventButton *)user_data;
  
  if (bevent->type != GDK_BUTTON_PRESS)
    return FALSE;

  if (bevent->button != 3)
    return FALSE;
  
  if ( (chunk=gtk_object_get_data(GTK_OBJECT(w),"PrevSel")) != NULL )
  {
    chunk=safe_strdup(chunk);
  }
  else
  {
    /* Find our current word */
    pos=GTK_TEXT(w)->cursor_mark.index;
    
    max=gtk_text_get_length(GTK_TEXT(w));
    
    for(;(pos>0)&&(!isdictchar(GTK_TEXT_INDEX(GTK_TEXT(w),pos)));pos--)
      ;
    for(left=pos;(left>0)&&(isdictchar(GTK_TEXT_INDEX(GTK_TEXT(w),left)));left--)
      ;
    for(right=pos;((right<max)&&(isdictchar(GTK_TEXT_INDEX(GTK_TEXT(w),right))));right++)
      ;
    
    if (left==0)
      left--;
  
    chunk=gtk_editable_get_chars(GTK_EDITABLE(w),left+1,right);
  }

  do_menu(w,bevent,chunk);

  free(chunk);
  
  /* The press has been processed by the TextBox, but not the release. */
  /* Cheat, and use the internals of gtk_text to work around this. */
  gtk_grab_remove(w);
  GTK_TEXT(w)->button=0;
  
  return TRUE;
}

gint do_menu(GtkWidget *w, GdkEventButton *bevent, const char *word)
{
  GtkWidget *root_menu,*menu_item;
  char *menu_word;

  menu_word=safe_malloc(strlen(word)+14);
  sprintf(menu_word,"Define '%s'",word);

  root_menu=gtk_menu_new();

  /* How can we avoid the memory leak in the signal_connects below? */
  menu_item=gtk_menu_item_new_with_label(menu_word);
  gtk_menu_append(GTK_MENU(root_menu),menu_item);
  gtk_signal_connect_object(GTK_OBJECT(menu_item),"activate",
                            GTK_SIGNAL_FUNC(do_define),(gpointer) safe_strdup(word));
  gtk_widget_show(menu_item);
  
  sprintf(menu_word,"Search for '%s'",word);
  menu_item=gtk_menu_item_new_with_label(menu_word);
  gtk_menu_append(GTK_MENU(root_menu),menu_item);
  gtk_signal_connect_object(GTK_OBJECT(menu_item),"activate",
                            GTK_SIGNAL_FUNC(do_find),(gpointer) safe_strdup(word));
  gtk_widget_show(menu_item);
  
  free(menu_word);
  
  gtk_menu_popup(GTK_MENU(root_menu),NULL,NULL,NULL,NULL,
                 bevent->button,bevent->time);

  return TRUE;
}

gint do_define(GtkWidget *w, gpointer user_data)
{
  define_word((gchar *)user_data);
  return TRUE;
}

gint do_find(GtkWidget *w, gpointer user_data)
{
  GtkWidget *winWordFinder;
  
  winWordFinder = create_winWordFinder();
  gtk_widget_show(winWordFinder);
  ++numwindows;

  if (strcmp(user_data,"*CLIPBOARD*")==0)
  {
    GdkAtom target_atom;

    gtk_signal_connect(GTK_OBJECT(winWordFinder),"selection_received",GTK_SIGNAL_FUNC(selection_received_search),NULL);
    target_atom = gdk_atom_intern("STRING",FALSE);
    gtk_selection_convert(winWordFinder,GDK_SELECTION_PRIMARY,target_atom,GDK_CURRENT_TIME);
  }
  else
  {
    set_find(winWordFinder,(char *)user_data);
  }
  return TRUE;
}


void set_find(GtkWidget *winWordFinder, const char *word)
{
  GtkWidget *entryWord;

  entryWord=gtk_object_get_data(GTK_OBJECT(winWordFinder),"entryWord");
  gtk_entry_set_text(GTK_ENTRY(entryWord),(char *)word);
}

/* Signal handler called when the selections owner returns the data */
void
selection_received_search (GtkWidget *w, GtkSelectionData *selection_data, 
                    gpointer data)
{
  gchar *fixed;
  
  /* **** IMPORTANT **** Check to see if retrieval succeeded  */
  if (selection_data->length < 0)
  {
    g_print ("Selection retrieval failed\n");
    return;
  }

  /* Make sure we got the data in the expected form */
  if (selection_data->type != GDK_SELECTION_TYPE_STRING)
  {
    g_print ("Selection \"STRING\" was not returned as string!\n");
    return;
  }
        
  fixed=cleanup((char *)selection_data->data);
  set_find(w,fixed);
  free(fixed);
}

/* Return value is allocated freshly.  You need to free it. */
char *cleanup(const char *sel)
{
  const gchar *pos1;
  gchar *pos2;
  int    in_blank;
  gchar *fixedsel;
  
  /* Remove all characters that would confuse dict. */
  fixedsel=safe_malloc(strlen(sel)+1);
  in_blank=1;
  for(pos1=sel,pos2=fixedsel;*pos1;pos1++)
  {
    if (isdictchar(*pos1))
    {
      in_blank=0;
      *pos2=*pos1;
      pos2++;
    }
    else
    {
      if (!in_blank)
      {
        in_blank=1;
        *pos2=' ';
        pos2++;
      }
    }
  }
  if (in_blank)
    pos2--;
  *pos2='\0';

  return fixedsel;
}
