/*
 * Copyright (c) 1997, 1998, 1999  Motoyuki Kasahara
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <syslog.h>
#include <ctype.h>
#include <signal.h>
#include <sys/types.h>

#if defined(STDC_HEADERS) || defined(HAVE_STRING_H)
#include <string.h>
#if !defined(STDC_HEADERS) && defined(HAVE_MEMORY_H)
#include <memory.h>
#endif /* not STDC_HEADERS and HAVE_MEMORY_H */
#else /* not STDC_HEADERS and not HAVE_STRING_H */
#include <strings.h>
#endif /* not STDC_HEADERS and not HAVE_STRING_H */

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <eb/appendix.h>
#include <eb/error.h>
#include <eb/eb.h>
#include <eb/font.h>
#include <eb/text.h>

#include "ndtpd.h"
#include "confutil.h"
#include "ioall.h"
#include "linebuf.h"
#include "wildcard.h"

#ifndef HAVE_STRCHR
#define strchr index
#define strrchr rindex
#endif /* HAVE_STRCHR */

#ifndef HAVE_MEMCPY
#define memcpy(d, s, n) bcopy((s), (d), (n))
#ifdef __STDC__
void *memchr(const void *, int, size_t);
int memcmp(const void *, const void *, size_t);
void *memmove(void *, const void *, size_t);
void *memset(void *, int, size_t);
#else /* not __STDC__ */
char *memchr();
int memcmp();
char *memmove();
char *memset();
#endif /* not __STDC__ */
#endif

#ifndef HAVE_STRTOL
#ifdef __STDC__
long strtol(const char *, char **, int);
#else /* not __STDC__ */
long strtol();
#endif /* not __STDC__ */
#endif /* not HAVE_STRTOL */

/*
 * Unexported functions.
 */
static int command_L NDTPD_P((const char *, const char *));
static int command_Q NDTPD_P((const char *, const char *));
static int command_A NDTPD_P((const char *, const char *));
static int command_u NDTPD_P((const char *, const char *));
static int command_P NDTPD_P((const char *, const char *));
static int command_F NDTPD_P((const char *, const char *));
static int command_S NDTPD_P((const char *, const char *));
static int command_I NDTPD_P((const char *, const char *));
static int command_v NDTPD_P((const char *, const char *));
static int command_T NDTPD_P((const char *, const char *));
static int command_t NDTPD_P((const char *, const char *));
static int command_X NDTPD_P((const char *, const char *));
static int command_XL NDTPD_P((const char *, const char *));
static int command_XI NDTPD_P((const char *, const char *));
static int command_XB NDTPD_P((const char *, const char *));
static int command_Xb NDTPD_P((const char *, const char *));
static int book_number_to_name NDTPD_P((char *));
static void reverse_word NDTPD_P((char *));
static void iso8859_1_to_ascii_str NDTPD_P((char *));
static void iso8859_1_to_ascii_mem NDTPD_P((char *, size_t));

/*
 * NDTP command table.
 */
typedef struct {
    char name;
    int (*function) NDTPD_P((const char *, const char *));
} NDTP_Command;

static const NDTP_Command ndtp_commands[] = {
    {'L', command_L},
    {'Q', command_Q},
    {'A', command_A},
    {'u', command_u},
    {'P', command_P},
    {'F', command_F},
    {'S', command_S},
    {'I', command_I},
    {'v', command_v},
    {'T', command_T},
    {'t', command_t},
    {'X', command_X},
    {'\0', NULL}
};

#define MAXLEN_SHORTBUF 4

/*
 * Communicate with a client by NDTP.
 */
int
ndtp_main()
{
    char buf[MAXLEN_NDTP_LINE + 1];
    const NDTP_Command *cmd;
    char cmdname[2];
    ssize_t len;

    reset_line_buffer(&line_buffer);

    for (;;) {
	/*
	 * Set an alarm.
	 */
	alarm(idle_timeout);
	
	/*
	 * Read a line received from the client.
	 */
	len = read_line_buffer(&line_buffer, buf, accepted_file);
	if (len < 0)
	    goto die;
	else if (len == 0)
	    continue;

	/*
	 * Check for the line length.
	 */
	if (MAXLEN_NDTP_LINE < len) {
	    if (skip_line_buffer(&line_buffer, accepted_file) < 0)
		goto die;
	    *(buf + MAXLEN_SHORTBUF) = '\0';
	    syslog(LOG_INFO, "the line too long: line=%s..., user=%s, \
host=%s(%s)",
		buf, client_user, client_hostname, client_address);
	    if (write_all(accepted_file, "$?\n", 3) < 0)
		return -1;
	    continue;
	}
	syslog(LOG_DEBUG, "debug: received line: %s", buf);

	/*
	 * Scan the command table.
	 */
	for (cmd = ndtp_commands; cmd->name != '\0'; cmd++) {
	    if (cmd->name == *buf)
		break;
	}
	if (cmd->name == '\0') {
	    *(buf + 1) = '\0';
	    syslog(LOG_INFO, "unknown command: command=%s, user=%s, \
host=%s(%s)",
		buf, client_user, client_hostname, client_address);
	    if (write_all(accepted_file, "$?\n", 3) < 0)
		goto die;

	    continue;
	}

	/*
	 * Set an alarm, again.
	 */
	alarm(idle_timeout);

	/*
	 * Dispatch.
	 */
	cmdname[0] = *buf;
	cmdname[1] = '\0';
	if ((cmd->function)(cmdname, buf + 1) < 0)
	    goto die;

	/*
	 * Quit when the received command is `Q'. (Quit even when
	 * ndtpd receives too many arguments for `Q'.)
	 */
	if (*buf == 'Q')
	    break;
    }

    alarm(0);
    return 0;

  die:
    alarm(0);
    return -1;
}


/*
 * Command `L'.  Change the current book/subbook.
 * It takes an argument which represents a book/subbook name.
 */
static int
command_L(command, arg)
    const char *command;
    const char *arg;
{
    char expandarg[MAXLEN_BOOK_NAME + 1 + EB_MAXLEN_BASENAME + 1];
    char bookname[MAXLEN_BOOK_NAME + 1];
    char subname[EB_MAXLEN_BASENAME + 1];
    EB_Subbook_Code subbook, appsub;
    char *slash;

    /*
     * Unset the current book.
     */
    if (current_book != NULL) {
	eb_suspend(&current_book->book);
	eb_suspend_appendix(&current_book->appendix);
	current_book->subname = NULL;
	current_book->height = 0;
	current_book = NULL;
    }

    /*
     * If `arg' is `nodict', unset the current book/subbook.
     */
    if (strcmp(arg, "nodict") == 0) {
	if (write_all(accepted_file, "$*\n", 3) < 0)
	    return -1;
	syslog(LOG_INFO, "succeeded: command=%s, book=nodict, user=%s, \
host=%s(%s)",
	    command, client_user, client_hostname, client_address);
	goto failed;
    }

    /*
     * For dserver 2.0 compatibility, check for aliases `eiwa', `waei',
     * and `kojien'.
     */
    if (strcmp(arg, "eiwa") == 0 && alias_eiwa[0] != '\0')
	strcpy(expandarg, alias_eiwa);
    else if (strcmp(arg, "waei") == 0 && alias_waei[0] != '\0')
	strcpy(expandarg, alias_waei);
    else if (strcmp(arg, "kojien") == 0 && alias_kojien[0] != '\0')
	strcpy(expandarg, alias_kojien);
    else
	strcpy(expandarg, arg);

    /*
     * If the argument doesn't contain a slash (`/'), it must be a book
     * number.  We convert it to the corresponding book name.
     */
    slash = strchr(expandarg, '/');
    if (slash == NULL) {
	if (book_number_to_name(expandarg) < 0) {
	    syslog(LOG_INFO, "unknown book: command=%s, book=%s, user=%s, \
host=%s(%s)",
		command, expandarg, client_user, client_hostname,
		client_address);
	    if (write_all(accepted_file, "$&\n", 3) < 0)
		return -1;
	    goto failed;
	}

	/*
	 * The number `0' is converted to `nodict'.
	 * Unset the current book/subbook.
	 */
	if (strcmp(expandarg, "nodict") == 0) {
	    if (write_all(accepted_file, "$*\n", 3) < 0)
		return -1;
	    syslog(LOG_INFO, "succeeded: command=%s, book=nodict, user=%s, \
host=%s(%s)",
		command, client_user, client_hostname, client_address);
	    goto failed;
	}

	/*
	 * Find a slash (`/') again.
	 */
	slash = strchr(expandarg, '/');
    }

    /*
     * Get book and subbook names separated by a slash (`/').
     */
    strncpy(bookname, expandarg, (int)(slash - expandarg));
    *(bookname + (int)(slash - expandarg)) = '\0';
    strcpy(subname, slash + 1);

    /*
     * Set the current book.
     */
    current_book = find_book(bookname);
    if (current_book == NULL) {
	syslog(LOG_INFO, "unknown book: command=%s, book=%s, user=%s, \
host=%s(%s)",
	    command, expandarg, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$&\n", 3) < 0)
	    return -1;
	goto failed;
    }
    current_book->subname = NULL;
    current_book->height = 0;

    /*
     * Check for access permission to the book.
     */
    if (!current_book->permflag) {
	syslog(LOG_ERR, "denied: command=%s, book=%s, user=%s, host=%s(%s)",
	    command, expandarg, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$&\n", 3) < 0)
	    return -1;
	goto failed;
    }

    /*
     * Get a ticket to access the book.
     * (Don't release the ticket when the current book is unset.)
     */
    if (current_book->max_clients != 0
	&& get_ticket(&current_book->ticket_stock) < 0) {
	syslog(LOG_INFO, "full of clients: command=%s, book=%s, user=%s, \
host=%s(%s)",
	    command, expandarg, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$&\n", 3) < 0)
	    return -1;
	goto failed;
    }

    /*
     * Find the subbook in the current book.
     */
    subbook = find_subbook(subname);
    if (subbook < 0) {
	syslog(LOG_INFO, "unknown book: command=%s, book=%s, user=%s, \
host=%s(%s)",
	    command, expandarg, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$&\n", 3) < 0)
	    return -1;
	goto failed;
    }

    /*
     * Set the current subbook.
     */
    if (eb_set_subbook(&current_book->book, subbook) < 0) {
	syslog(LOG_INFO, "%s: command=%s, book=%s, user=%s, host=%s(%s)",
	    eb_error_message(), command, expandarg, client_user,
	    client_hostname, client_address);
	if (write_all(accepted_file, "$&\n", 3) < 0)
	    return -1;
	goto failed;
    }

    /*
     * Find the subbook in the current appendix.
     */
    if (eb_is_appendix_bound(&current_book->appendix)) {
	appsub = find_appendix_subbook(subname);
	if (appsub < 0) {
	    syslog(LOG_INFO, "unknown appendix: command=%s, book=%s, \
user=%s, host=%s(%s)",
		command, expandarg, client_user, client_hostname,
		client_address);
	    if (write_all(accepted_file, "$&\n", 3) < 0)
		return -1;
	    goto failed;
	}

	/*
	 * Set the current subbook.
	 */
	if (eb_set_appendix_subbook(&current_book->appendix, appsub) < 0) {
	    syslog(LOG_INFO, "%s: command=%s, book=%s, user=%s, host=%s(%s)",
		eb_error_message(), command, expandarg, client_user,
		client_hostname, client_address);
	    if (write_all(accepted_file, "$&\n", 3) < 0)
		return -1;
	    goto failed;
	}
    }

    /*
     * Get the current subbook name.
     */
    current_book->subname = eb_subbook_directory(&current_book->book);
    if (current_book->subname == NULL) {
	syslog(LOG_INFO, "%s: command=%s, book=%s, user=%s, host=%s(%s)",
	    eb_error_message(), command, expandarg, client_user,
	    client_hostname, client_address);
	if (write_all(accepted_file, "$&\n", 3) < 0)
	    return -1;
	goto failed;
    }

    /*
     * Send a response.
     */
    if (write_all(accepted_file, "$*\n", 3) < 0)
	return -1;

    syslog(LOG_INFO, "succeeded: command=%s, book=%s, user=%s, host=%s(%s)",
	command, expandarg, client_user, client_hostname, client_address);
    return 0;

    /*
     * An error occurs...
     */
  failed:
    if (current_book != NULL) {
	eb_suspend(&current_book->book);
	eb_suspend_appendix(&current_book->appendix);
	current_book = NULL;
    }
    return 0;
}


/*
 * Command `Q'.  Quit.
 */
static int
command_Q(command, arg)
    const char *command;
    const char *arg;
{
    /*
     * The argument should be empty.
     */
    if (*arg != '\0') {
	syslog(LOG_INFO, "too many arguments: command=%s, user=%s, \
host=%s(%s)",
	    command, client_user, client_hostname, client_address);
	return 0;
    }

    syslog(LOG_INFO, "succeeded: command=%s, user=%s, host=%s(%s)",
	command, client_user, client_hostname, client_address);
    return 0;
}


/*
 * Command `A'.  Get access permission.
 * The command is not supported.  It always sends the `succeeded' response.
 */
static int
command_A(command, arg)
    const char *command;
    const char *arg;
{
    /*
     * Send a success code.
     */
    if (write_all(accepted_file, "$A\n", 3) < 0)
	return -1;

    syslog(LOG_INFO, "succeeded: command=%s, user=%s, host=%s(%s)",
	command, client_user, client_hostname, client_address);
    return 0;
}


/*
 * Command `u'.  List current users.
 * The command is not supported.  It always sends an empty list.
 */
static int
command_u(command, arg)
    const char *command;
    const char *arg;
{
    /*
     * The argument must be empty.
     */
    if (*arg != '\0') {
	syslog(LOG_INFO, "too many arguments: command=%s, user=%s, \
host=%s(%s)",
	    command, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$?\n", 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Send an empty list.
     */
    if (write_all(accepted_file, "$U\n$$\n", 6) < 0)
	return -1;

    syslog(LOG_INFO, "succeeded: command=%s, user=%s, host=%s(%s)",
	command, client_user, client_hostname, client_address);
    return 0;
}


/* Index size of a hit list. */
#define NUM_HIT_LIST	50

/*
 * Command `P'.  Search a word.
 * It takes two arguments without a separater.
 * The first argument is a character which represents an index type.
 * The second argument represents a word to search.
 */
static int
command_P(command, arg)
    const char *command;
    const char *arg;
{
    int (*search) NDTPD_P((EB_Book *, const char *));
    EB_Character_Code charcode;
    EB_Hit hitlist[NUM_HIT_LIST];
    EB_Hit *hit;
    int hitcount;
    char pos[MAXLEN_STRING + 1];
    char headbuf1[MAXLEN_STRING + 1];
    char headbuf2[MAXLEN_STRING + 1];
    char *head;
    char *prevhead;
    ssize_t headlen;
    char word[EB_MAXLEN_WORD + 1];
    size_t wordlen;
    char method;
    int prevpage;
    int prevoffset;
    int n;
    int i;

    /*
     * Check for the current book.
     */
    if (current_book == NULL) {
	syslog(LOG_INFO, "no current book: command=%s, user=%s, host=%s(%s)",
	    command, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$<\n", 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Set a word to search and a search method.
     */
    wordlen = strlen(arg) - 1;
    if (EB_MAXLEN_WORD < wordlen) {
	syslog(LOG_INFO, "%s: command=%s, user=%s, host=%s(%s)", 
	    eb_error_message2(EB_ERR_TOO_LONG_WORD), command, client_user,
	    client_hostname, client_address);
	if (write_all(accepted_file, "$?\n", 3) < 0)
	    return -1;
	return 0;
    }
    if (*arg == '\0') {
	syslog(LOG_INFO, "empty method name: command=%s, user=%s, \
host=%s(%s)",
	    command, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$?\n", 3) < 0)
	    return -1;
	return 0;
    }
    strcpy(word, arg + 1);
    method = *arg;

    /*
     * Check for the search method.
     */
    if (method != 'a' && method != 'k' && method != 'A' && method != 'K') {
	syslog(LOG_INFO, "bad search method: command=%s, method=%c, \
user=%s, host=%s(%s)",
	    command, method, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$?\n", 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Remove `*' in the tail of the word and choose a search function.
     * If the word is written in ISO 8859-1 and it ends with `\xa1\xf6'
     * (upside-down `!' and umlaut accent `o'), it is recognized as
     * asterisk (`*') of JIS X 0208 by mistake.
     */
    if (1 <= wordlen && *(word + wordlen - 1) == '*') {
	*(word + wordlen - 1) = '\0';
	if (method == 'a' || method == 'k')
	    search = eb_search_word;
	else
	    search = eb_search_endword;
    } else if (2 <= wordlen && (unsigned char)*(word + wordlen - 2) == 0xa1
	&& (unsigned char)*(word + wordlen - 1) == 0xf6) {
	*(word + wordlen - 2) = '\0';
	if (method == 'a' || method == 'k')
	    search = eb_search_word;
	else
	    search = eb_search_endword;
    } else {
	search = eb_search_exactword;
    }

    /*
     * Reverse the word when endword search.
     */
    if (method == 'A' || method == 'K')
	reverse_word(word);

    /*
     * Search the word.
     */
    if (search(&current_book->book, word) < 0) {
	syslog(LOG_INFO, "%s: command=%s, book=%s/%s, method=%c, word=%s, \
user=%s, host=%s(%s)",
	    eb_error_message(), command, current_book->name,
	    current_book->subname, method, arg + 1, client_user,
	    client_hostname, client_address);
	if (write_all(accepted_file, "$0\n$$\n", 6) < 0)
	    return -1;
	return 0;
    }
	
    if (write_all(accepted_file, "$0\n", 3) < 0)
	return -1;

    hitcount = 0;
    head = headbuf1;
    prevhead = headbuf2;
    *prevhead = '\0';
    prevpage = 0;
    prevoffset = 0;
    charcode = eb_character_code(&current_book->book);
    while (hitcount < max_hits || max_hits == 0) {
	/*
	 * Get hit entries.
	 */
	n = eb_hit_list(&current_book->book, hitlist, NUM_HIT_LIST);
	if (n <= 0)
	    break;

	for (i = 0, hit = hitlist; i < n; i++, hitcount++, hit++) {
	    /*
	     * Seek the current subbook.
	     */
	    if (eb_seek(&current_book->book, &hit->heading) < 0) {
		syslog(LOG_INFO, "%s: command=%s, book=%s/%s, method=%c, \
word=%s, user=%s, host=%s(%s)",
		    eb_error_message(), command, current_book->name,
		    current_book->subname, method, arg + 1, client_user,
		    client_hostname, client_address);
		continue;
	    }

	    /*
	     * Get heading.
	     */
	    headlen = eb_heading(&current_book->book, &current_book->appendix,
		&text_hookset, head, MAXLEN_STRING);
	    if (headlen < 0) {
		syslog(LOG_INFO, "%s: command=%s, book=%s/%s, method=%c, \
word=%s, user=%s, host=%s(%s)",
		    eb_error_message(), command, current_book->name,
		    current_book->subname, method, arg + 1, client_user,
		    client_hostname, client_address);
		break;
	    }
	    if (headlen == 0)
		continue;
	    if (charcode == EB_CHARCODE_ISO8859_1)
		iso8859_1_to_ascii_mem(head, headlen);

	    /*
	     * Ignore a hit entry if its heading and text location are
	     * equal to those of the previous hit entry.
	     */
	    if (prevpage == hit->text.page && prevoffset == hit->text.offset 
		&& strcmp(head, prevhead) == 0)
		continue;

	    /*
	     * Output the heading and text position.
	     */
	    if (write_all(accepted_file, head, strlen(head)) < 0)
		return -1;
	    sprintf(pos, "\n%x:%x\n", hit->text.page, hit->text.offset);
	    if (write_all(accepted_file, pos, strlen(pos)) < 0)
		return -1;

	    /*
	     * Keep the last message, page, and offset, to remove
	     * duplicated entries.
	     */
	    if (head == headbuf1) {
		head = headbuf2;
		prevhead = headbuf1;
	    } else {
		head = headbuf1;
		prevhead = headbuf2;
	    }
	    prevpage = hit->text.page;
	    prevoffset = hit->text.offset;
	}
    }

    if (n < 0) {
	syslog(LOG_INFO, "%s: command=%s, book=%s/%s, method=%c, word=%s, \
user=%s, host=%s(%s)",
	    eb_error_message(), command, current_book->name,
	    current_book->subname, method, arg + 1, client_user,
	    client_hostname, client_address);
    }
    if (write_all(accepted_file, "$$\n", 3) < 0)
	return -1;

    syslog(LOG_INFO, "succeeded: command=%s, book=%s/%s, method=%c, \
word=%s, matches=%d, user=%s, host=%s(%s)",
	command, current_book->name, current_book->subname, method, arg + 1,
	hitcount, client_user, client_hostname, client_address);
    return 0;
}


/*
 * Command `F'.  Send a page.
 * It takes an argument which represents a page to send.
 * The page number must be a hexadecimal integer.
 */
static int
command_F(command, arg)
    const char *command;
    const char *arg;
{
    EB_Position pos;
    char message[EB_SIZE_PAGE + 3];  /* "$F" + page + "\n" */
    char *endp;

    /*
     * Set header and footer.
     */
    message[0] = '$';
    message[1] = 'F';
    message[EB_SIZE_PAGE + 2] = '\n';

    /*
     * Check for the current book.
     */
    if (current_book == NULL) {
	syslog(LOG_INFO, "no current book: command=%s, user=%s, host=%s(%s)",
	    command, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$<\n", 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Parse an page number.
     */
    pos.offset = 0;
    pos.page = strtol(arg, &endp, 16);
    if (!isxdigit(*arg) || *endp != '\0') {
	syslog(LOG_INFO, "bad page number: command=%s, page=%s, user=%s, \
host=%s(%s)",
	    command, arg, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$?\n", 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Read the page.
     */
    if (eb_seek(&current_book->book, &pos) < 0) {
	syslog(LOG_INFO, "%s: command=%s, book=%s/%s, page=%d, user=%s, \
host=%s(%s)",
	    eb_error_message(), command, current_book->name,
	    current_book->subname, pos.page, client_hostname,
	    client_address);
	memset(message + 2, '\0', EB_SIZE_PAGE);
	if (write_all(accepted_file, message, EB_SIZE_PAGE + 3) < 0)
	    return -1;
	return 0;
    }
    if (eb_rawtext(&current_book->book, message + 2, EB_SIZE_PAGE)
	!= EB_SIZE_PAGE) {
	syslog(LOG_INFO, "%s: command=%s, book=%s/%s, page=%d, user=%s, \
host=%s(%s)",
	    eb_error_message(), command, current_book->name,
	    current_book->subname, pos.page, client_user,
	    client_hostname, client_address);
	memset(message + 2, '\0', EB_SIZE_PAGE);
	if (write_all(accepted_file, message, EB_SIZE_PAGE + 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Send the page to the client ("$F" + page + "\n").
     */
    if (write_all(accepted_file, message, EB_SIZE_PAGE + 3) < 0)
	return -1;

    syslog(LOG_INFO, "succeeded: command=%s, book=%s/%s, page=%d, \
user=%s, host=%s(%s)",
	command, current_book->name, current_book->subname, pos.page,
	client_user, client_hostname, client_address);
    return 0;
}


/*
 * Command `S'.  Send text.
 * It takes two arguments; page number and offset separeted by a colon (`:').
 * Both the page number and offset must be hexadecimal integers.
 */
static int
command_S(command, arg)
    const char *command;
    const char *arg;
{
    EB_Position pos;
    EB_Character_Code charcode;
    char message[EB_SIZE_PAGE];
    char *endp;
    char *offsetp;
    size_t rest_size;
    size_t request_size;
    ssize_t len;

    /*
     * Check for the current book.
     */
    if (current_book == NULL) {
	syslog(LOG_INFO, "no current book: command=%s, user=%s, host=%s(%s)",
	    command, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$<\n", 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Parse a page number.
     */
    pos.page = strtol(arg, &endp, 16);
    if (!isxdigit(*arg) || *endp != ':') {
	syslog(LOG_INFO, "bad position: command=%s, position=%s, user=%s, \
host=%s(%s)",
	    command, arg, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$?\n", 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Parse an offset number.
     */
    offsetp = endp + 1;
    pos.offset = strtol(offsetp, &endp, 16);
    if (!isxdigit(*offsetp) || *endp != '\0') {
	syslog(LOG_INFO, "bad position: command=%s, position=%s, user=%s, \
host=%s(%s)",
	    command, arg, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$?\n", 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Seek the current subbook.
     */
    if (eb_seek(&current_book->book, &pos) < 0) {
	syslog(LOG_INFO, "%s: command=%s, book=%s/%s, position=%s, user=%s, \
host=%s(%s)",
	    eb_error_message(), command, current_book->name,
	    current_book->subname, arg, client_user, client_hostname,
	    client_address);
	if (write_all(accepted_file, "$1\n$$\n", 6) < 0)
	    return -1;
	return 0;
    }

    /*
     * Read text and send them to the client.
     */
    if (write_all(accepted_file, "$1\n", 3) < 0)
	return -1;

    charcode = eb_character_code(&current_book->book);
    rest_size = max_text_size;
    for (;;) {
	if (EB_SIZE_PAGE - 1 < rest_size || max_text_size == 0)
	    request_size = EB_SIZE_PAGE - 1;
	else
	    request_size = rest_size;
	if (request_size <= 0)
	    break;

	len = eb_text(&current_book->book, &current_book->appendix, 
	    &text_hookset, message, request_size);
	if (len <= 0)
	    break;
	rest_size -= len;
	if (charcode == EB_CHARCODE_ISO8859_1)
	    iso8859_1_to_ascii_mem(message, len);
	if (write_all(accepted_file, message, len) < 0)
	    return -1;
    }

    if (write_all(accepted_file, "\n$$\n", 4) < 0)
	return -1;

    syslog(LOG_INFO, "succeeded: command=%s, book=%s/%s, position=%s, \
user=%s, host=%s(%s)",
	command, current_book->name, current_book->subname, arg, client_user,
	client_hostname, client_address);
    return 0;
}


/*
 * Command `I'.  Send an index list.
 * The page numbers in the table are dummy except for menu search.
 */
static int
command_I(command, arg)
    const char *command;
    const char *arg;
{
    char message[MAXLEN_STRING + 1];
    EB_Position pos;

    /*
     * The argument must be empty.
     */
    if (*arg != '\0') {
	syslog(LOG_INFO, "too many arguments: command=%s, user=%s, \
host=%s(%s)",
	    command, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$?\n", 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Check for the current book.
     */
    if (current_book == NULL) {
	syslog(LOG_INFO, "no current book: command=%s, user=%s, host=%s(%s)",
	    command, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$<\n", 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Send response data.
     */
    if (write_all(accepted_file, "$I\n", 3) < 0)
	return -1;
    if (eb_have_word_search(&current_book->book)) {
	strcpy(message, "IA 1\nIK 1\n");
	if (write_all(accepted_file, message, strlen(message)) < 0)
	    return -1;
    }
    if (eb_have_endword_search(&current_book->book)) {
	strcpy(message, "BA 1\nBK 1\n");
	if (write_all(accepted_file, message, strlen(message)) < 0)
	    return -1;
    }
    if (0 <= eb_menu(&current_book->book, &pos)) {
	sprintf(message, "HA %x\n", pos.page);
	if (write_all(accepted_file, message, strlen(message)) < 0)
	    return -1;
    }
    if (0 <= eb_copyright(&current_book->book, &pos)) {
	sprintf(message, "OK %x\n", pos.page);
	if (write_all(accepted_file, message, strlen(message)) < 0)
	    return -1;
    }

    if (write_all(accepted_file, "$$\n", 3) < 0)
	return -1;

    syslog(LOG_INFO, "succeeded: command=%s, book=%s/%s, user=%s, \
host=%s(%s)",
	command, current_book->name, current_book->subname, client_user,
	client_hostname, client_address);
    return 0;
}


/*
 * Command `v'.  Output version number of the server.
 */
static int
command_v(command, arg)
    const char *command;
    const char *arg;
{
    char message[MAXLEN_STRING + 1];

    /*
     * The argument must be empty.
     */
    if (*arg != '\0') {
	syslog(LOG_INFO, "too many arguments: command=%s, user=%s, \
host=%s(%s)",
	    command, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$?\n", 3) < 0)
	    return -1;
	return 0;
    }

    sprintf(message, "$v%s version %s on %s\n", program_name, program_version,
	server_name);
    if (write_all(accepted_file, message, strlen(message)) < 0)
	return -1;

    syslog(LOG_INFO, "succeeded: command=%s, user=%s, host=%s(%s)",
	command, client_user, client_hostname, client_address);
    return 0;
}


/*
 * Command `T'.  Output a book list.
 */
static int
command_T(command, arg)
    const char *command;
    const char *arg;
{
    EB_Subbook_Code sublist[EB_MAX_SUBBOOKS];
    EB_Character_Code charcode;
    Book *book;
    char message[MAXLEN_STRING + 1];
    const char *title;
    int subcount;
    int i, j;

    /*
     * The argument must be empty.
     */
    if (*arg != '\0') {
	syslog(LOG_INFO, "too many arguments: command=%s, user=%s, \
host=%s(%s)",
	    command, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$?\n", 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Send a book list.
     */
    for (book = book_registry, i = 1;
	 book != NULL; book = (Book *)book->next) {
	if (!eb_is_bound(&book->book))
	    continue;
	subcount = eb_subbook_list(&book->book, sublist);
	if (subcount < 0 || !book->permflag)
	    continue;
	charcode = eb_character_code(&book->book);
	for (j = 0; j < subcount; j++, i++) {
	    title = eb_subbook_title2(&book->book, sublist[j]);
	    if (title == NULL)
		title = "(not available)";
	    sprintf(message, "%d\t%s\n", i, title);
	    if (charcode == EB_CHARCODE_ISO8859_1)
		iso8859_1_to_ascii_str(message);
	    if (write_all(accepted_file, message, strlen(message)) < 0)
		return -1;
	}
    }
    if (write_all(accepted_file, "$*\n", 3) < 0)
	return -1;

    syslog(LOG_INFO, "succeeded: command=%s, user=%s, host=%s(%s)",
	command, client_user, client_hostname, client_address);
    return 0;
}


/*
 * Command `t'.  List current users and books.
 * The command is not supported.  It always sends an empty list.
 */
static int
command_t(command, arg)
    const char *command;
    const char *arg;
{
    EB_Subbook_Code sublist[EB_MAX_SUBBOOKS];
    EB_Character_Code charcode;
    Book *book;
    char message[MAXLEN_STRING + 1];
    const char *title, *dir;
    int subcount;
    int i, j;

    /*
     * The argument must be empty.
     */
    if (*arg != '\0') {
	syslog(LOG_INFO, "too many arguments: command=%s, user=%s, \
host=%s(%s)",
	    command, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$?\n", 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Send a book list.
     */
    for (book = book_registry, i = 1;
	 book != NULL; book = (Book *)book->next) {
	if (!eb_is_bound(&book->book))
	    continue;
	subcount = eb_subbook_list(&book->book, sublist);
	if (subcount < 0 || !book->permflag)
	    continue;
	charcode = eb_character_code(&book->book);
	for (j = 0; j < subcount; j++, i++) {
	    dir = eb_subbook_directory2(&book->book, sublist[j]);
	    if (dir == NULL)
		dir = "*";
	    title = eb_subbook_title2(&book->book, sublist[j]);
	    if (title == NULL)
		title = "(not available)";
	    sprintf(message, "%d\t%s\t%s/%s\t%d\t%d\t%d\n", i, title,
		book->name, dir, 0, book->max_clients, idle_timeout);
	    if (charcode == EB_CHARCODE_ISO8859_1)
		iso8859_1_to_ascii_str(message);
	    if (write_all(accepted_file, message, strlen(message)) < 0)
		return -1;
	}
    }
    if (write_all(accepted_file, "$*\n", 3) < 0)
	return -1;

    syslog(LOG_INFO, "succeeded: command=%s, user=%s, host=%s(%s)",
	command, client_user, client_hostname, client_address);
    return 0;
}


/*
 * `X' prefixed Commands.
 */
static int
command_X(command, arg)
    const char *command;
    const char *arg;
{
    switch (*arg) {
    case 'I':
	return command_XI("XI", arg + 1);
    case 'L':
	return command_XL("XL", arg + 1);
    case 'B':
	return command_XB("XB", arg + 1);
    case 'b':
	return command_Xb("Xb", arg + 1);
    }

    syslog(LOG_INFO, "unknown command: command=%s, user=%s, host=%s(%s)",
	command, client_user, client_hostname, client_address);
    if (write_all(accepted_file, "$?\n", 3) < 0)
	return -1;

    return 0;
}


/*
 * Command `XI'.  Send a bitmap size list.
 */
static int
command_XI(command, arg)
    const char *command;
    const char *arg;
{
    EB_Font_Code fontlist[EB_MAX_FONTS];
    int fontcount;
    char message[MAXLEN_STRING + 1];
    int i;

    /*
     * The argument must be empty.
     */
    if (*arg != '\0') {
	syslog(LOG_INFO, "too many arguments: command=%s, user=%s, \
host=%s(%s)",
	    command, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$?\n", 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Check for the current book.
     */
    if (current_book == NULL) {
	syslog(LOG_INFO, "no current book: command=%s, user=%s, host=%s(%s)",
	    command, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$<\n", 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Send response data.
     */
    if (write_all(accepted_file, "$I\n", 3) < 0)
	return -1;

    fontcount = eb_font_list(&current_book->book, fontlist);
    for (i = 0; i < fontcount; i++) {
	sprintf(message, "%d\n", fontlist[i]);
	if (write_all(accepted_file, message, strlen(message)) < 0)
	    return -1;
    }
    if (write_all(accepted_file, "$$\n", 3) < 0)
	return -1;

    syslog(LOG_INFO, "succeeded: command=%s, book=%s/%s, user=%s, \
host=%s(%s)",
	command, current_book->name, current_book->subname, client_user,
	client_hostname, client_address);
    return 0;
}


/*
 * Command `XL'.  Set the current bitmap size.
 */
static int
command_XL(command, arg)
    const char *command;
    const char *arg;
{
    int height;
    char *endp;

    /*
     * Check for the current book.
     */
    if (current_book == NULL) {
	syslog(LOG_INFO, "no current book: command=%s, user=%s, host=%s(%s)",
	    command, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$<\n", 3) < 0)
	    return -1;
	goto failed;
    }

    /*
     * Unset the current bitmap size.
     */
    eb_unset_font(&current_book->book);
    current_book->height = 0;

    /*
     * Parse an argument.
     */
    height = strtol(arg, &endp, 10);
    if (!isdigit(arg[0]) || *endp != '\0') {
	syslog(LOG_INFO, "bad bitmap height: command=%s, bitmap-size=%s, \
user=%s, host=%s(%s)",
	    command, arg, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$?\n", 3) < 0)
	    return -1;
	goto failed;
    }

    /*
     * If height is `0', unset the bitmap size.
     */
    if (height == 0) {
	if (write_all(accepted_file, "$*\n", 3) < 0)
	    return -1;
	syslog(LOG_INFO, "succeeded: command=%s, book=%s/%s, bitmap-size=0, \
user=%s, host=%s(%s)",
	    command, current_book->name, current_book->subname, client_user,
	    client_hostname, client_address);
	goto failed;
    }

    /*
     * Set the current bitmap size.
     */
    if (eb_set_font(&current_book->book, height) < 0) {
	syslog(LOG_INFO, "%s: command=%s, book=%s/%s, bitmap-size=%d, \
user=%s, host=%s(%s)",
	    eb_error_message(), command, current_book->name,
	    current_book->subname, height, client_user, client_hostname,
	    client_address);
	if (write_all(accepted_file, "$&\n", 3) < 0)
	    return -1;
	goto failed;
    }
    current_book->height = height;

    if (write_all(accepted_file, "$*\n", 3) < 0)
	return -1;
    syslog(LOG_INFO, "succeeded: command=%s, book=%s/%s, bitmap-size=%d, \
user=%s, host=%s(%s)",
	command, current_book->name, current_book->subname, height,
	client_user, client_hostname, client_address);
    return 0;

    /*
     * An error occurs...
     */
  failed:
    if (current_book != NULL) {
	eb_unset_font(&current_book->book);
	current_book->height = 0;
    }
    return 0;
}


/*
 * Command `XB'.  Send bitmap data of all local characters defined in
 * the current subbook.
 */
static int
command_XB(command, arg)
    const char *command;
    const char *arg;
{
    char bitmap[EB_SIZE_WIDE_FONT_48];
    char xbm[EB_SIZE_NARROW_FONT_48_XBM];
    char header[MAXLEN_STRING + 1];
    int width;
    int start, end;
    int ch;
    size_t xbmsize;

    /*
     * The argument must be empty.
     */
    if (*arg != '\0') {
	syslog(LOG_INFO, "too many arguments: command=%s, user=%s, \
host=%s(%s)",
	    command, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$?\n", 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Check for the current book and bitmap size.
     */
    if (current_book == NULL) {
	syslog(LOG_INFO, "no current book: command=%s, user=%s, host=%s(%s)",
	    command, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$<\n", 3) < 0)
	    return -1;
	return 0;
    }
    if (current_book->height == 0) {
	syslog(LOG_INFO, "no current bitmap size: command=%s, book=%s/%s, \
user=%s, host=%s(%s)",
	    command, current_book->name, current_book->subname, client_user,
	    client_hostname, client_address);
	if (write_all(accepted_file, "$<\n", 3) < 0)
	    return -1;
	return 0;
    }

    if (write_all(accepted_file, "$I\n", 3) < 0)
	return -1;

    /*
     * Send bitmap data of narrow characters.
     */
    do {
	if (!eb_have_narrow_font(&current_book->book))
	    break;
	width = eb_narrow_font_width(&current_book->book);
	if (width < 0) {
	    syslog(LOG_INFO, "%s: command=%s, book=%s/%s, bitmap-size=%d, \
user=%s, host=%s(%s)",
		eb_error_message(), command, current_book->name,
		current_book->subname, current_book->height, client_user,
		client_hostname, client_address);
	    break;
	}

	start = eb_narrow_font_start(&current_book->book);
	if (start < 0) {
	    syslog(LOG_INFO, "%s: command=%s, book=%s/%s, bitmap-size=%d, \
user=%s, host=%s(%s)",
		eb_error_message(), command, current_book->name,
		current_book->subname, current_book->height, client_user,
		client_hostname, client_address);
	}

	end = eb_narrow_font_end(&current_book->book);
	if (end < 0) {
	    syslog(LOG_INFO, "%s: command=%s, book=%s/%s, bitmap-size=%d, \
user=%s, host=%s(%s)",
		eb_error_message(), command, current_book->name,
		current_book->subname, current_book->height, client_user,
		client_hostname, client_address);
	    break;
	}

	ch = start;
	while (0 <= ch) {
	    if (eb_narrow_font_character_bitmap(&current_book->book, ch,
		bitmap) < 0)
		continue;
	    xbmsize = eb_bitmap_to_xbm(xbm, bitmap, width,
		current_book->height);
	    if (xbmsize < 0)
		continue;
	    sprintf(header, "$=h%04x\n", ch);
	    if (write_all(accepted_file, header, strlen(header)) < 0)
		return -1;
	    if (write_all(accepted_file, xbm, xbmsize) < 0)
		return -1;
	    ch = eb_forward_narrow_font_character(&current_book->book, ch, 1);
	}
    } while (0);

    /*
     * Send bitmap data of wide characters.
     */
    do {
	if (!eb_have_wide_font(&current_book->book))
	    break;
	width = eb_wide_font_width(&current_book->book);
	if (width < 0) {
	    syslog(LOG_INFO, "%s: command=%s, book=%s/%s, bitmap-size=%d, \
user=%s, host=%s(%s)",
		eb_error_message(), command, current_book->name,
		current_book->subname, current_book->height, client_user,
		client_hostname, client_address);
	    break;
	}

	start = eb_wide_font_start(&current_book->book);
	if (start < 0) {
	    syslog(LOG_INFO, "%s: command=%s, book=%s/%s, bitmap-size=%d, \
user=%s, host=%s(%s)",
		eb_error_message(), command, current_book->name,
		current_book->subname, current_book->height, client_user,
		client_hostname, client_address);
	    break;
	}

	end = eb_wide_font_end(&current_book->book);
	if (end < 0) {
	    syslog(LOG_INFO, "%s: command=%s, book=%s/%s, bitmap-size=%d, \
user=%s, host=%s(%s)",
		eb_error_message(), command, current_book->name,
		current_book->subname, current_book->height, client_user,
		client_hostname, client_address);
	    break;
	}

	ch = start;
	while (0 <= ch) {
	    if (eb_wide_font_character_bitmap(&current_book->book, ch,
		bitmap) < 0)
		continue;
	    xbmsize = eb_bitmap_to_xbm(xbm, bitmap, width,
		current_book->height);
	    if (xbmsize < 0)
		continue;
	    sprintf(header, "$=z%04x\n", ch);
	    if (write_all(accepted_file, header, strlen(header)) < 0)
		return -1;
	    if (write_all(accepted_file, xbm, xbmsize) < 0)
		return -1;
	    ch = eb_forward_wide_font_character(&current_book->book, ch, 1);
	}
    } while (0);

    /*
     * End of the response data.
     */
    if (write_all(accepted_file, "$$\n", 3) < 0)
	return -1;

    syslog(LOG_INFO, "succeeded: command=%s, book=%s/%s, bitmap-size=%d, \
user=%s, host=%s(%s)",
	command, current_book->name, current_book->subname,
	current_book->height, client_user, client_hostname, client_address);
    return 0;
}


/*
 * Command `Xb'.  Send bitmap data of a local character.
 */
static int
command_Xb(command, arg)
    const char *command;
    const char *arg;
{
    char bitmap[EB_SIZE_WIDE_FONT_48];
    char xbm[EB_SIZE_NARROW_FONT_48_XBM];
    char *endp;
    int chno, chtype;
    int width;
    int flag;
    size_t xbmsize;

    /*
     * Check for the current book and bitmap size.
     */
    if (current_book == NULL) {
	syslog(LOG_INFO, "no current book: command=%s, user=%s, host=%s(%s)",
	    command, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$<\n", 3) < 0)
	    return -1;
	return 0;
    }
    if (current_book->height == 0) {
	syslog(LOG_INFO, "no current bitmap size: command=%s, book=%s/%s, \
user=%s, host=%s(%s)",
	    command, current_book->name, current_book->subname, client_user,
	    client_hostname, client_address);
	if (write_all(accepted_file, "$<\n", 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Parse a charcter number;
     */
    chtype = *arg;
    chno = strtol(arg + 1, &endp, 16);
    if (!isxdigit(*(arg + 1)) || *endp != '\0'
	|| (chtype != 'h' && chtype != 'z')) {
	syslog(LOG_INFO, "bad character number: command=%s, character=%s, \
user=%s, host=%s(%s)",
	    command, arg, client_user, client_hostname, client_address);
	if (write_all(accepted_file, "$?\n", 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Get width of the character.
     */
    if (chtype == 'h')
	width = eb_narrow_font_width(&current_book->book);
    else
	width = eb_wide_font_width(&current_book->book);
    if (width < 0) {
	syslog(LOG_INFO, "%s: command=%s, book=%s/%s, bitmap-size=%d, \
character=%s, user=%s, host=%s(%s)",
	    eb_error_message(), command, current_book->name,
	    current_book->subname, current_book->height, arg, client_user,
	    client_hostname, client_address);
	if (write_all(accepted_file, "$&\n", 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Get bitmap data of the character.
     */
    if (chtype == 'h') {
	flag = eb_narrow_font_character_bitmap(&current_book->book, chno,
	    bitmap);
    } else {
	flag = eb_wide_font_character_bitmap(&current_book->book, chno,
	    bitmap);
    }
    if (flag < 0) {
	syslog(LOG_INFO, "%s: command=%s, book=%s/%s, bitmap-size=%d, \
character=%s, user=%s, host=%s(%s)",
	    eb_error_message(), command, current_book->name,
	    current_book->subname, current_book->height, arg, client_user,
	    client_hostname, client_address);
	if (write_all(accepted_file, "$&\n", 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Get XBM data of the character.
     */
    xbmsize = eb_bitmap_to_xbm(xbm, bitmap, width, current_book->height);
    if (xbmsize < 0) {
	syslog(LOG_INFO, "%s: command=%s, book=%s/%s, bitmap-size=%d, \
character=%s, user=%s, host=%s(%s)",
	    eb_error_message(), command, current_book->name,
	    current_book->subname, current_book->height, arg, client_user,
	    client_hostname, client_address);
	if (write_all(accepted_file, "$&\n", 3) < 0)
	    return -1;
	return 0;
    }

    /*
     * Send bitmap data of the character.
     */
    if (write_all(accepted_file, "$I\n", 3) < 0)
	return -1;
    if (write_all(accepted_file, xbm, xbmsize) < 0)
	return -1;
    if (write_all(accepted_file, "$$\n", 3) < 0)
	return -1;

    syslog(LOG_INFO, "succeeded: command=%s, book=%s/%s, bitmap-size=%d, \
character=%s, user=%s, host=%s(%s)",
	command, current_book->name, current_book->subname,
	current_book->height, arg, client_user, client_hostname,
	client_address);

    return 0;
}


/*
 * Convert a book number `arg' to the corresponding book name.
 *
 * If succeeds, 0 is returned, and `arg' is overwritten.  Upon return,
 * `arg' represents the book name.  Otherwise, -1 is returned and 
 * `arg' is unchanged.
 */
static int
book_number_to_name(arg)
    char *arg;
{
    int number;
    Book *book;
    int subcount;
    EB_Subbook_Code sublist[EB_MAX_SUBBOOKS];
    const char *dir;
    char *endp;
    int i;

    /*
     * Get a book number.
     */
    number = strtol(arg, &endp, 10);
    if (!isdigit(arg[0]) || *endp != '\0')
	return -1;

    /*
     * If `arg' is `0', it is equivalent to `nodict'.
     * Unset the current book/subbook.
     */
    if (number == 0) {
	strcpy(arg, "nodict");
	return 0;
    }

    /*
     * Find `book/subbook' corresponding with `number'.
     */
    for (book = book_registry, i = 1;
	 book != NULL && i <= number; book = (Book *)book->next) {
	if (!eb_is_bound(&book->book))
	    continue;
	subcount = eb_subbook_list(&book->book, sublist);
	if (subcount < 0 || !book->permflag)
	    continue;
	if (number < i + subcount) {
	    dir = eb_subbook_directory2(&book->book, sublist[number - i]);
	    if (dir == NULL)
		return -1;
	    sprintf(arg, "%s/%s", book->name, dir);
	    return 0;
	}
	i += subcount;
    }
    
    return -1;
}


/*
 * Reverse a word for ENDWORD SEARCH.
 * 
 * `word' is a word to reverse.  It must be an alphabetic word.
 * The reversed word is also put into `word'.
 */
static void
reverse_word(word)
    char *word;
{
    char buf[MAXLEN_NDTP_LINE + 1];
    unsigned char *p1, *p2;

    strcpy(buf, word);
    p1 = (unsigned char *)word;
    p2 = (unsigned char *)buf + strlen(word);

    *p2-- = '\0';
    while (*p1 != '\0') {
	if (*p1 == 0x8f) {
	    /*
	     * `*p1' is SS3.
	     * (It may be rejected by `eb_search_word', though.)
	     */
	    if (*(p1 + 1) == '\0' || *(p1 + 2) == '\0')
		break;
	    *p2 = *(p1 + 2);
	    *(p2 - 1) = *(p1 + 1);
	    *(p2 - 2) = *p1;
	    p1 += 3;
	    p2 -= 3;
	} else if ((0xa1 <= *p1 && *p1 <= 0xfe) || *p1 == 0x8e) {
	    /*
	     * `*p1' is SS2 or a character in JIS X 0208.
	     */
	    if (*(p1 + 1) == '\0')
		break;
	    *p2 = *(p1 + 1);
	    *(p2 - 1) = *p1;
	    p1 += 2;
	    p2 -= 2;
	} else {
	    /*
	     * Otherwise.
	     */
	    *p2 = *p1;
	    p1 += 1;
	    p2 -= 1;
	}
    }

    strcpy(word, p2 + 1);
}


/*
 * ISO 8859 1 to EUC-JP conversion table.
 */
static const char iso8859_1_to_euc_table[] = {
    ' ',   ' ',   ' ',   ' ',   ' ',   ' ',   ' ',   ' ',  /* 80 - 87 */
    ' ',   ' ',   ' ',   ' ',   ' ',   ' ',   ' ',   ' ',  /* 88 - 8f */
    ' ',   ' ',   ' ',   ' ',   ' ',   ' ',   ' ',   ' ',  /* 90 - 97 */
    ' ',   ' ',   ' ',   ' ',   ' ',   ' ',   ' ',   ' ',  /* 98 - 9f */
    ' ',   ' ',   ' ',   ' ',   ' ',   ' ',   '|',   ' ',  /* a0 - a7 */
    '\'',  'C',   ' ',   '<',   ' ',   '-',   'R',   '~',  /* a8 - af */
    ' ',   ' ',   '2',   '3',   '\'',  'u',   '*',   '*',  /* b0 - b7 */
    ',' ,  '1',   ' ',   '>',   ' ',   ' ',   ' ',   ' ',  /* b8 - bf */
    'A',   'A',   'A',   'A',   'A',   'A',   ' ',   'C',  /* c0 - c7 */
    'E',   'E',   'E',   'E',   'I',   'I',   'I',   'I',  /* c8 - cf */
    'D',   'N',   'O',   'O',   'O',   'O',   'O',   'x',  /* d0 - d7 */
    'O',   'U',   'U',   'U',   'U',   'Y',   'P',   'B',  /* d8 - df */
    'a',   'a',   'a',   'a',   'a',   'a',   ' ',   'c',  /* e0 - e7 */
    'e',   'e',   'e',   'e',   'i',   'i',   'i',   'i',  /* e8 - ef */
    'o',   'n',   'o',   'o',   'o',   'o',   'o',   '/',  /* f0 - f7 */
    'o',   'u',   'u',   'u',   'u',   'y',   'p',   'y',  /* f8 - ff */
};


/*
 * Convert an ISO 8859-1 string to ASCII.
 */
static void
iso8859_1_to_ascii_str(string)
    char *string;
{
    unsigned char *strp = (unsigned char *)string;

    while (*strp != '\0') {
	if (0x80 <= *strp)
	    *(char *)strp = iso8859_1_to_euc_table[*strp - 0x80];
	strp++;
    }
}


/*
 * Convert an ISO 8859-1 stream to ASCII.
 */
static void
iso8859_1_to_ascii_mem(stream, len)
    char *stream;
    size_t len;
{
    unsigned char *strp = (unsigned char *)stream;
    size_t i;

    for (i = len; 0 < i; i--) {
	if (0x80 <= *strp)
	    *(char *)strp = iso8859_1_to_euc_table[*strp - 0x80];
	strp++;
    }
}


