/*
 * Copyright (c) 1997, 1998, 1999  Motoyuki Kasahara
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>

#include <eb/eb.h>
#include <eb/appendix.h>
#include <eb/text.h>

#include "ndtpd.h"

/*
 * Unexported functions.
 */
static int hook_initialize NDTPD_P((EB_Book *, EB_Appendix *,
    char *, EB_Hook_Code, int, const int *));
static int hook_narrow_font_text NDTPD_P((EB_Book *, EB_Appendix *,
    char *, EB_Hook_Code, int, const int *));
static int hook_narrow_font_bitmap NDTPD_P((EB_Book *, EB_Appendix *,
    char *, EB_Hook_Code, int, const int *));
static int hook_wide_font_text NDTPD_P((EB_Book *, EB_Appendix *,
    char *, EB_Hook_Code, int, const int *));
static int hook_wide_font_bitmap NDTPD_P((EB_Book *, EB_Appendix *,
    char *, EB_Hook_Code, int, const int *));
static int hook_newline NDTPD_P((EB_Book *, EB_Appendix *, char *,
    EB_Hook_Code, int, const int *));
static int hook_reference NDTPD_P((EB_Book *, EB_Appendix *, char *,
    EB_Hook_Code, int, const int *));
static int hook_gb2312 NDTPD_P((EB_Book *, EB_Appendix *, char *,
    EB_Hook_Code, int, const int *));

/*
 * Hooks.
 */
static const EB_Hook generic_hooks[] = {
    {EB_HOOK_INITIALIZE, hook_initialize},
    {EB_HOOK_NARROW_FONT, hook_narrow_font_text},
    {EB_HOOK_WIDE_FONT, hook_wide_font_text},
    {EB_HOOK_NEWLINE, hook_newline},
    {EB_HOOK_STOPCODE, eb_hook_stopcode},
    {EB_HOOK_END_REFERENCE, hook_reference},
    {EB_HOOK_END_MENU, hook_reference},
    {EB_HOOK_NARROW_JISX0208, eb_hook_euc_to_ascii},
    {EB_HOOK_GB2312, hook_gb2312},
    {EB_HOOK_NULL, NULL},
};

static const EB_Hook text_hooks[] = {
    {EB_HOOK_NARROW_FONT, hook_narrow_font_text},
    {EB_HOOK_WIDE_FONT, hook_wide_font_text},
    {EB_HOOK_NULL, NULL},
};

static const EB_Hook bitmap_hooks[] = {
    {EB_HOOK_NARROW_FONT, hook_narrow_font_bitmap},
    {EB_HOOK_WIDE_FONT, hook_wide_font_bitmap},
    {EB_HOOK_NULL, NULL},
};


/*
 * Initialize text hook set.
 */
void
initialize_hookset()
{
    eb_initialize_hookset(&text_hookset);
    eb_set_hooks(&text_hookset, (EB_Hook *)generic_hooks);
}


/*
 * Hook for intialization.
 */
static int
hook_initialize(book, appendix, workbuf, code, argc, argv)
    EB_Book *book;
    EB_Appendix *appendix;
    char *workbuf;
    EB_Hook_Code code;
    int argc;
    const int *argv;
{
    if (current_book->height != 0)
	eb_set_hooks(&text_hookset, (EB_Hook *)bitmap_hooks);
    else
	eb_set_hooks(&text_hookset, (EB_Hook *)text_hooks);

    return 0;
}


/*
 * Hook for narrow font characters.
 */
static int
hook_narrow_font_text(book, appendix, workbuf, code, argc, argv)
    EB_Book *book;
    EB_Appendix *appendix;
    char *workbuf;
    EB_Hook_Code code;
    int argc;
    const int *argv;
{
    if (eb_narrow_alt_character_text(&current_book->appendix, argv[0], workbuf)
	< 0)
	sprintf(workbuf, "#%04x", argv[0]);

    return 0;
}


/*
 * Hook for narrow font characters.
 */
static int
hook_narrow_font_bitmap(book, appendix, workbuf, code, argc, argv)
    EB_Book *book;
    EB_Appendix *appendix;
    char *workbuf;
    EB_Hook_Code code;
    int argc;
    const int *argv;
{
    sprintf(workbuf, "<gaiji:h%04x>", argv[0]);

    return 0;
}


/*
 * Hook for wide font characters.
 */
static int
hook_wide_font_text(book, appendix, workbuf, code, argc, argv)
    EB_Book *book;
    EB_Appendix *appendix;
    char *workbuf;
    EB_Hook_Code code;
    int argc;
    const int *argv;
{
    if (eb_wide_alt_character_text(&current_book->appendix, argv[0], workbuf)
	< 0)
	sprintf(workbuf, "#%04x", argv[0]);

    return 0;
}


/*
 * Hook for wide font characters.
 */
static int
hook_wide_font_bitmap(book, appendix, workbuf, code, argc, argv)
    EB_Book *book;
    EB_Appendix *appendix;
    char *workbuf;
    EB_Hook_Code code;
    int argc;
    const int *argv;
{
    sprintf(workbuf, "<gaiji:z%04x>", argv[0]);

    return 0;
}


/*
 * Hook for a newline character.
 */
static int
hook_newline(book, appendix, workbuf, code, argc, argv)
    EB_Book *book;
    EB_Appendix *appendix;
    char *workbuf;
    EB_Hook_Code code;
    int argc;
    const int *argv;
{
    char *p = workbuf;

    *p++ = '\n';
    *p = '\0';

    return 0;
}


/*
 * Hook for a reference to text.
 */
static int
hook_reference(book, appendix, workbuf, code, argc, argv)
    EB_Book *book;
    EB_Appendix *appendix;
    char *workbuf;
    EB_Hook_Code code;
    int argc;
    const int *argv;
{
    sprintf(workbuf, "<%x:%x>", argv[1], argv[2]);
    return 0;
}


/*
 * Hook for a GB 2312 character.
 */
static int
hook_gb2312(book, appendix, workbuf, code, argc, argv)
    EB_Book *book;
    EB_Appendix *appendix;
    char *workbuf;
    EB_Hook_Code code;
    int argc;
    const int *argv;
{
    sprintf(workbuf, "<&g0:%02d%02d>",
	*((unsigned char *)workbuf) - 0xa0,
	*((unsigned char *)workbuf + 1) - 0xa0);
    return 0;
}


