/*
 * Copyright (c) 1997, 1998, 1999  Motoyuki Kasahara
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

/*
 * This program requires the following Autoconf macros:
 *   AC_C_CONST
 *   AC_TYPE_SIZE_T
 *   AC_CHECK_TYPE(ssize_t, int)
 *   AC_HEADER_STDC
 *   AC_CHECK_HEADERS(string.h, memory.h, stdlib.h, unistd.h)
 *   AC_CHECK_FUNCS(memcpy)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <sys/types.h>
#include <syslog.h>
#include <errno.h>

#if defined(STDC_HEADERS) || defined(HAVE_STRING_H)
#include <string.h>
#if !defined(STDC_HEADERS) && defined(HAVE_MEMORY_H)
#include <memory.h>
#endif /* not STDC_HEADERS and HAVE_MEMORY_H */
#else /* not STDC_HEADERS and not HAVE_STRING_H */
#include <strings.h>
#endif /* not STDC_HEADERS and not HAVE_STRING_H */

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "linebuf.h"

#ifdef USE_FAKELOG
#include "fakelog.h"
#endif

#ifndef HAVE_MEMCPY
#define memcpy(d, s, n) bcopy((s), (d), (n))
#ifdef __STDC__
void *memchr(const void *, int, size_t);
int memcmp(const void *, const void *, size_t);
void *memmove(void *, const void *, size_t);
void *memset(void *, int, size_t);
#else /* not __STDC__ */
char *memchr();
int memcmp();
char *memmove();
char *memset();
#endif /* not __STDC__ */
#endif /* not HAVE_MEMCPY */


/*
 * Initialize `linebuffer'.
 */
int
initialize_line_buffer(linebuffer, maxlen)
    Line_Buffer *linebuffer;
    size_t maxlen;
{
    /*
     * Allocate memories to `buffer'.
     */
    linebuffer->buffer = (char *)malloc(maxlen + 1);
    if (linebuffer->buffer == NULL) {
	syslog(LOG_ERR, "memory exhausted");
	return -1;
    }

    /*
     * Initialize members.
     */ 
    linebuffer->bufsize = maxlen + 1;
    linebuffer->current = linebuffer->buffer;
    linebuffer->restsize = 0;
    return 0;
}


/*
 * Clear `linebuffer'.
 */
void
clear_line_buffer(linebuffer)
    Line_Buffer *linebuffer;
{
    free(linebuffer->buffer);
    linebuffer->buffer = NULL;
    linebuffer->current = NULL;
    linebuffer->bufsize = 0;
    linebuffer->restsize = 0;
}


/*
 * Examine whether `linebuffer' has unprocessed data.
 */
int
have_rest_in_line_buffer(linebuffer)
    Line_Buffer *linebuffer;
{
    return (linebuffer->restsize != 0);
}


/*
 * Clear unprocessed data in `linebuffer'.
 */
void
reset_line_buffer(linebuffer)
    Line_Buffer *linebuffer;
{
    linebuffer->current = linebuffer->buffer;
    linebuffer->restsize = 0;
}


/*
 * Get buffer size of the `linebuffer'.
 */
size_t
line_buffer_size(linebuffer)
    Line_Buffer *linebuffer;
{
    return linebuffer->bufsize;
}


/*
 * Read a line from `file'.
 *
 * This function is useful when `file' is a socket, but also works
 * when `file' is a regular file.
 *
 * The line read from `file' is copied into `line'.
 *
 * The function recognizes `\n' and `\r\n' as the end of the line.
 * `\n' and `\r\n' are not copied to `buffer', and `\0' is added to
 * the end of the line, instead.
 *
 * If succeeded to read a line, the number of characters in the line
 * is returned.  It doesn't count `\n' and `\r\n' in the tail of the
 * line, so that 0 is returned for an empty line.
 * If EOF is received or an error occurs, -1 is returned.
 */
ssize_t
read_line_buffer(linebuffer, line, file)
    Line_Buffer *linebuffer;
    char *line;
    int file;
{
    char *newline = NULL;
    ssize_t len;
    ssize_t n;

    if (linebuffer->restsize != 0) {
	/*
	 * Unprocessed data are left in `linebuffer->buffer'.
	 * Slide them to the beginning of `linebuffer->buffer'.
	 */
	memmove(linebuffer->buffer, linebuffer->current, linebuffer->restsize);
	newline = (char *)memchr(linebuffer->buffer, '\n',
	    linebuffer->restsize);
    }
    linebuffer->current = linebuffer->buffer + linebuffer->restsize;
    
    while (newline == NULL) {
	/*
	 * Check for the length of the current line.
	 */
	if (linebuffer->restsize == linebuffer->bufsize) {
    	    memcpy(line, linebuffer->buffer, linebuffer->bufsize);
	    linebuffer->restsize = 0;
	    return linebuffer->bufsize;
	}

	/*
	 * Read from `file'.
	 */
	n = read(file, linebuffer->current,
	    linebuffer->bufsize - linebuffer->restsize);
	if (n < 0) {
	    if (errno == EINTR)
		continue;
	    syslog(LOG_INFO, "read() failed, %m");
	    return -1;
	} else if (n == 0) {
	    syslog(LOG_INFO, "read() received EOF");
	    return -1;
	}

	newline = (char *)memchr(linebuffer->current, '\n', n);
	linebuffer->restsize += n;
	linebuffer->current += n;
    }

    *newline = '\0';
    len = newline - linebuffer->buffer;
    linebuffer->current = newline + 1;
    linebuffer->restsize -= len + 1;

    /*
     * If the line is end with `\r\n', remove not only `\n' but `\r'.
     */
    if (0 < len && *(newline - 1) == '\r') {
	*(newline - 1) = '\0';
	len--;
    }

    /*
     * Copy the line to `line'.
     */
    memcpy(line, linebuffer->buffer, len);
    *(line + len) = '\0';

    return len;
}


/*
 * Skip too long line read by read_line_buffer().
 *
 * If a line is read by read_line_buffer() doesn't contain a
 * newline character, the line is too long to store whole of the
 * line into the buffer.
 * The function reads and discards the rest of the line.
 * 
 * If EOF is received or an error occurs, -1 is returned.
 * Otherwise, 0 is returned.
 */
int
skip_line_buffer(linebuffer, file)
    Line_Buffer *linebuffer;
    int file;
{
    ssize_t len;

    /*
     * Read data until the end of the line is found.
     */
    for (;;) {
	len = read_line_buffer(linebuffer, linebuffer->buffer, file);
	if (len < 0)
	    return -1;
	if (len < linebuffer->bufsize)
	    break;
    }

    return 0;
}


