/*
 * Module for file view(display diff result) in two-panes mode
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#if defined(HAVE_ERRNO_H)
#include <errno.h>
#else defined(HAVE_SYS_ERRNO_H)
#include <sys/errno.h>
#endif
#include <gnome.h>
#include "gdiff.h"
#include "misc.h"
#include "gui.h"
#include "guimisc.h"
#include "fileview.h"
#include "guiwin.h"
#include "rmenu.h"
#include "style.h"


/* Private function declarations */
static void create_textw(GDiffFileViews *gfileviews);
static void draw_text(GDiffFileViews *gfileviews, int whichfile);
static void show_hide_numbers(GDiffFileViews *gfileviews, int whichfile, gboolean b_show);
static void create_overview(GDiffFileViews *gfileviews);
static void draw_overview(GDiffFileViews *gfileviews, int whichfile);
static void draw_overview_lines(GtkWidget *widget, GdkEventExpose *event, gpointer data);
static void create_vscrollbar(GDiffFileViews *gfileviews);
static void vscrollbar_vchanged(GtkAdjustment *vadj, gpointer data);
static void guts_move_diff(GDiffFileViews *gfileviews, MoveDiff mv_diff);
static void adjust_vscrollbar(GDiffFileViews *gfileviews);



/**
 * twopane_create_widgets:
 * Called from gdiff_fileviews_new().
 * Create widgets and show them.
 * The contents(diff results) are not shown here, which is up to twopane_display().
 * Input:
 * GDiffFileViews *gfileviews;
 * Output:
 * GDiffFileViews *gfileviews; updated.
 **/
void
twopane_create_widgets(GDiffFileViews *gfileviews)
{
	GDiffWindow *gdwin = gfileviews->gdwin;
	GtkWidget *hbox;
	GtkWidget *label;
	gint page;
	
	/* hbox */
	hbox = gtk_hbox_new(FALSE, 0);
	gfileviews->pane.two.hbox = GTK_HBOX(hbox);
	gfileviews->base = GTK_WIDGET(gfileviews->pane.two.hbox);/* Virtual base widget. */

	/* Add to the notebook */
	label = make_label_for_notebook(gfileviews->filename1, gfileviews->filename2, TWO_PANE_VIEW);
	gfileviews->pane.two.label = GTK_LABEL(label);
	gtk_notebook_append_page(gdwin->notebook, hbox, label);

	/* text area */
	create_textw(gfileviews);
	/* overview area */
	create_overview(gfileviews);
	
	set_style_fileviews(gfileviews, TWO_PANE);
	gtk_widget_show(hbox);
	/* Is this a proper way ? */
	page = gtk_notebook_page_num(gdwin->notebook, hbox);
	gtk_notebook_set_page(gdwin->notebook, page);

	rmenu_create(gfileviews);
}


/**
 * twopane_display:
 * Show the diff result in two-panes mode.
 * Input:
 * GDiffFileViews *gfileviews;
 * Output:
 * GDiffFileViews *gfileviews; updated.
 **/
void
twopane_display(GDiffFileViews *gfileviews)
{
	DiffFiles *files = gfileviews->dfiles;
	int n;

	if (is_files_different(files, TRUE) == FALSE) {
		gtk_widget_hide(GTK_WIDGET(gfileviews->base));
		return;
	}
	
	for (n = 0; n < NUM_COMPARE_FILES; n++) {
		draw_text(gfileviews, n);
		draw_overview(gfileviews, n);
		if (gfileviews->pref.show_line_num == TRUE) {
			show_hide_numbers(gfileviews, n, TRUE);
		}
	}
	/* to control both part */
	create_vscrollbar(gfileviews);

	/* Set data for right-click menu (especially, edit file) */
	for (n = 0; n < NUM_COMPARE_FILES; n++) {
		GtkText *text;
		text = gfileviews->pane.two.text[n];
		gtk_object_set_user_data(GTK_OBJECT(text), (gpointer)n);
	}
}

/**
 * twopane_show_linenum:
 * Show(Hide) the line numbers on text widget.
 * Input:
 * GDiffFileViews *gfileviews;
 * gboolean to_show; If TRUE, show. if FALSE, hide.
 * Output:
 * GDiffFileViews *gfileviews; updated.
 **/
void
twopane_show_linenum(GDiffFileViews *gfileviews, gboolean to_show)
{
	int n;

	if (to_show == TRUE && gfileviews->pref.show_line_num == FALSE) {
		for (n = 0; n < NUM_COMPARE_FILES; n++) {
			show_hide_numbers(gfileviews, n, TRUE);
		}
		gfileviews->pref.show_line_num = TRUE;
	} else if (to_show == FALSE && gfileviews->pref.show_line_num == TRUE) {
		for (n = 0; n < NUM_COMPARE_FILES; n++) {
			show_hide_numbers(gfileviews, n, FALSE);
		}
		gfileviews->pref.show_line_num = FALSE;
	}	
}

/**
 * twopane_move_diff:
 * Move to a difference, such as next, previous, first or last.
 **/ 
void
twopane_move_diff(GDiffFileViews *gfileviews, MoveDiff mv_diff)
{
	char *msg = NULL;

	guts_move_diff(gfileviews, mv_diff);
	adjust_vscrollbar(gfileviews);

	/* Update status-bar */
	if (mv_diff == MOVED_REL_NEXT || mv_diff == MOVED_REL_PREV) {
		msg = make_current_info_msg(gfileviews->filename1, gfileviews->filename2, -1, -1, -1, -1);
	} else {
		const DiffLines *dlines;

		dlines = gfileviews->cur_dlines_list ? gfileviews->cur_dlines_list->data : NULL;
		if (dlines) {
			msg = make_current_info_msg(gfileviews->filename1, gfileviews->filename2,
										dlines->between[FIRST_FILE].begin,
										dlines->between[FIRST_FILE].end,
										dlines->between[SECOND_FILE].begin,
										dlines->between[SECOND_FILE].end);
		}
	}
	if (msg) {
		statusbar_update(gfileviews->gdwin, msg);
		g_free(msg);
	}
}


/* ---The followings are private functions--- */
/**
 * create_textw:
 * Create GtkText widget.
 **/
static void
create_textw(GDiffFileViews *gfileviews)
{
	GtkHBox *hbox = gfileviews->pane.two.hbox;
	GtkWidget *hpaned;
	int total_width;
	int n;

	hpaned = gtk_hpaned_new();
	gtk_box_pack_start(GTK_BOX(hbox), hpaned, TRUE, TRUE, 0);
	total_width = GTK_WIDGET(gfileviews->gdwin->notebook)->allocation.width;
	/* Try to put the handle at the center */
	total_width -= (WIDTH_OVERVIEW * 3);
	if (total_width > 0)
		gtk_paned_set_position(GTK_PANED(hpaned), total_width / 2);
	gtk_widget_show(hpaned);

	for (n = 0; n < NUM_COMPARE_FILES; n++) {
		GtkWidget *text;
		GtkWidget *scrollwin;
		
		scrollwin = gtk_scrolled_window_new(NULL, NULL);
		/* XXX: I can't use horizontal scrollbar,
		   because of the current text widget limitation. */
		gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin),
									   GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
		if (n == FIRST_FILE)
			gtk_paned_add1(GTK_PANED(hpaned), scrollwin);
		else if (n == SECOND_FILE)
			gtk_paned_add2(GTK_PANED(hpaned), scrollwin);
#ifdef DEBUG
		else
			g_error("impossible\n");
#endif
		gtk_widget_show(scrollwin);
	
		text = gtk_text_new(NULL, NULL);
		gtk_text_set_editable(GTK_TEXT(text), FALSE);
		gtk_text_set_word_wrap(GTK_TEXT(text), FALSE);
		gtk_text_set_line_wrap(GTK_TEXT(text), gfileviews->pref.line_wrap);
		gtk_container_add(GTK_CONTAINER(scrollwin), text);
		/*XXX gtk_widget_grab_focus(text);*/
		gtk_widget_show(text);

		gfileviews->pane.two.text[n] = GTK_TEXT(text);
	}
}

/**
 * draw_text:
 * Draw text with coloring different parts.
 **/
static void
draw_text(GDiffFileViews *gfileviews, int whichfile)
{
	DiffFiles *files = gfileviews->dfiles;
	const FileInfo *fi = dfiles_get_fileinfo(files, whichfile, TRUE);
	GtkText *text = gfileviews->pane.two.text[whichfile];
	GdkFont *font = GTK_WIDGET(text)->style->font;
	const char *pt;
	const char *text_last;/* for calculation of the length left */
	int line;
	GList *node;/* node of DiffLines' list */
	FontProp fprop;
	GdkColor *fg_color = &gfileviews->pref.diff_fg[whichfile];
	GdkColor *bg_color = &gfileviews->pref.diff_bg[whichfile];
	BufInfo binfo;

	if (fi->text == NULL) /* Maybe, file has been deleted. */
		return;

	fprop.font = font;

	line = 1;
	pt = fi->text;
	text_last = fi->text + fi->lenb;
	gtk_text_freeze(text);
	for (node = files->dlines_list; node; node = node->next) {
		DiffLines *dlines;
		int begin, end;
		int lenb;
		
		dlines = node->data;
		/* Use local variables for ease to read */
		begin = dlines->between[whichfile].begin;
		end = dlines->between[whichfile].end;

		if (end == 0)
			continue;
		fprop.fg = NULL;
		fprop.bg = NULL;
		binfo.buf = pt;
		binfo.lenb = text_last - pt;
		lenb = draw_text_lines(text, &fprop, &binfo, begin - line);
		pt += lenb;

		if ((dlines->difftype == CHANGE)
			|| (dlines->difftype == F1ONLY && whichfile == FIRST_FILE)
			|| (dlines->difftype == F2ONLY && whichfile == SECOND_FILE)) {
			fprop.fg = fg_color;
			fprop.bg = bg_color;
		} else if ((dlines->difftype == F2ONLY && whichfile == FIRST_FILE)
				   || (dlines->difftype == F1ONLY && whichfile == SECOND_FILE)) {
			fprop.fg = NULL;
			fprop.bg = NULL;
		}
		binfo.buf = pt;
		binfo.lenb = text_last - pt;
		lenb = draw_text_lines(text, &fprop, &binfo, end - begin + 1);
		pt += lenb;
		line = end + 1;
	}
	/* Draw the remained part */
#ifdef DEBUG	
	if (line >= fi->nline) {
		g_print("Problem?\n");
	}
#endif	
	fprop.fg = NULL;
	fprop.bg = NULL;
	binfo.buf = pt;
	binfo.lenb = text_last - pt;
	draw_text_lines(text, &fprop, &binfo, fi->nline - line + 1);
	gtk_text_thaw(text);
}


/* Routines for line numbers */
/**
 * show_hide_numbers:
 * Show(or hide) line numbers on the head of each lines.
 **/
static void
show_hide_numbers(GDiffFileViews *gfileviews, int whichfile, gboolean b_show)
{
	DiffFiles *files = gfileviews->dfiles;
	const FileInfo *fi = dfiles_get_fileinfo(files, whichfile, TRUE);
	GtkText *text = gfileviews->pane.two.text[whichfile];
	GdkFont *font = GTK_WIDGET(text)->style->font;
	const char *pt;
	const char *text_last;/* for calculation of the length left */
	int line;
	int pos = 0;/* position in text widget */
	GList *node;/* node of DiffLines' list */
	FontProp fprop;
	GdkColor *fg_color = &gfileviews->pref.diff_fg[whichfile];
	GdkColor *bg_color = &gfileviews->pref.diff_bg[whichfile];
	char format_common[32];
	char format[32];
	BufInfo binfo;
	LineFormat lformat;

	if (fi->text == NULL) /* Maybe, file has been deleted. */
		return;

	fprop.font = font;

	line = 1;
	pt = fi->text;
	text_last = fi->text + fi->lenb;
	/* the number of columns for representing line numbers */
	lformat.n_col = calc_number_places(fi->nline);
	sprintf(format_common, "%%%dd%s", lformat.n_col, MARK_COMMON);/* e.g. "%4d  " */
	if (whichfile == FIRST_FILE)
		sprintf(format, "%%%dd%s", lformat.n_col, MARK_FILE1);/* e.g. "%4d< " */
	else
		sprintf(format, "%%%dd%s", lformat.n_col, MARK_FILE2);/* e.g. "%4d> " */

	gtk_text_freeze(text);
	for (node = files->dlines_list; node; node = node->next) {
		DiffLines *dlines;
		int begin, end;
		int lenb;
		
		dlines = node->data;
		/* Use local variables for ease to read */
		begin = dlines->between[whichfile].begin;
		end = dlines->between[whichfile].end;
		if (end == 0)
			continue;

		fprop.fg = NULL;
		fprop.bg = NULL;
		binfo.buf = pt;
		binfo.lenb = text_last - pt;
		lformat.format = format_common;
		lenb = insert_remove_line_numbers(text, b_show, &pos, &fprop, &binfo,
										  line, begin - line, &lformat);
		pt += lenb;
		if ((dlines->difftype == CHANGE)
			|| (dlines->difftype == F1ONLY && whichfile == FIRST_FILE)
			|| (dlines->difftype == F2ONLY && whichfile == SECOND_FILE)) {
			fprop.fg = fg_color;
			fprop.bg = bg_color;
			lformat.format = format;
		} else if ((dlines->difftype == F2ONLY && whichfile == FIRST_FILE)
				   || (dlines->difftype == F1ONLY && whichfile == SECOND_FILE)) {
			fprop.fg = NULL;
			fprop.bg = NULL;
			lformat.format = format_common;
		}
		binfo.buf = pt;
		binfo.lenb = text_last - pt;
		lenb = insert_remove_line_numbers(text, b_show, &pos, &fprop, &binfo,
										  begin, end - begin + 1, &lformat);
		pt += lenb;
		line = end + 1;
	}
	/* Draw the remained part */
	fprop.fg = NULL;
	fprop.bg = NULL;
	binfo.buf = pt;
	binfo.lenb = text_last - pt;
	lformat.format = format_common;
	insert_remove_line_numbers(text, b_show, &pos, &fprop, &binfo,
							   line, fi->nline - line + 1, &lformat);
	gtk_text_thaw(text);
}


/* Routines for drawing overview */
/**
 * create_overview:
 * Create two GdiffOverview widgets, and drawing-area widget between them to draw lines.
 * See "gdiffoverview.[ch]" about GdiffOverview widget.
 **/
static void
create_overview(GDiffFileViews *gfileviews)
{
	GtkHBox *hbox = gfileviews->pane.two.hbox;
	GtkAdjustment *vadj;
	GtkWidget *ov1;
	GtkWidget *ov2;
	GtkWidget *darea;/* drawing area */

	
	darea = gtk_drawing_area_new();
	gtk_widget_set_name(GTK_WIDGET(darea), "drawing overview");
	vadj = GTK_TEXT(gfileviews->pane.two.text[FIRST_FILE])->vadj;
	ov1 = gdiff_overview_new(vadj);
	gtk_widget_set_name(GTK_WIDGET(ov1), "file1 overview");
	gdiff_overview_set_foreground(GDIFF_OVERVIEW(ov1), &gfileviews->pref.diff_bg[FIRST_FILE]);
	vadj = GTK_TEXT(gfileviews->pane.two.text[SECOND_FILE])->vadj;
	ov2 = gdiff_overview_new(vadj);
	gtk_widget_set_name(GTK_WIDGET(ov2), "file2 overview");
	gdiff_overview_set_foreground(GDIFF_OVERVIEW(ov2), &gfileviews->pref.diff_bg[SECOND_FILE]);

	gtk_box_pack_start(GTK_BOX(hbox), ov1, FALSE, FALSE, 0);
	gtk_widget_show(ov1);
	gtk_box_pack_start(GTK_BOX(hbox), darea, FALSE, FALSE, 0);
	gtk_widget_show(darea);
	gtk_box_pack_start(GTK_BOX(hbox), ov2, FALSE, FALSE, 0);
	gtk_widget_show(ov2);
	
	/* adjust drawing-area */
	gtk_drawing_area_size(GTK_DRAWING_AREA(darea), WIDTH_OVERVIEW, 0);/*XXX*/
	gdiff_overview_size(GDIFF_OVERVIEW(ov1), WIDTH_OVERVIEW, 0);/*XXX*/
	gdiff_overview_size(GDIFF_OVERVIEW(ov2), WIDTH_OVERVIEW, 0);/*XXX*/
	
	gfileviews->pane.two.darea = GTK_DRAWING_AREA(darea);
	gfileviews->pane.two.overview[FIRST_FILE] = GDIFF_OVERVIEW(ov1);
	gfileviews->pane.two.overview[SECOND_FILE] = GDIFF_OVERVIEW(ov2);
}

/**
 * draw_overview:
 * Draw diff parts on overview widget.
 **/
static void
draw_overview(GDiffFileViews *gfileviews, int whichfile)
{
	DiffFiles *files = gfileviews->dfiles;
	const FileInfo *fi = dfiles_get_fileinfo(files, whichfile, TRUE);
	GdiffOverview *ov = gfileviews->pane.two.overview[whichfile];
	GtkDrawingArea *darea = gfileviews->pane.two.darea;
	GList *node;/* node of DiffLines' list */
	int all_nline;

	all_nline = fi->nline;
	if (all_nline == 0)
		return;

	for (node = files->dlines_list; node; node = node->next) {
		DiffLines *dlines;
		int begin, end;
		
		dlines = node->data;
		/* Use local variables for ease to read */
		begin = dlines->between[whichfile].begin;
		end = dlines->between[whichfile].end;

		if ((dlines->difftype == CHANGE)
			|| (dlines->difftype == F1ONLY && whichfile == FIRST_FILE)
			|| (dlines->difftype == F2ONLY && whichfile == SECOND_FILE)) {
			gdiff_overview_insert_paintrange(ov, (gdouble)begin/all_nline, (gdouble)end/all_nline);
		} else if ((dlines->difftype == F2ONLY && whichfile == FIRST_FILE)
				   || (dlines->difftype == F1ONLY && whichfile == SECOND_FILE)) {
			/* This condition means that no different part in one of files.
			   However I will draw it for your information.
			   In this case, begin==end.
			   This will be taken care of by GdiffOverview widget,
			   then it draws a line. */
			gdiff_overview_insert_paintrange(ov, (gdouble)begin/all_nline, (gdouble)end/all_nline);
		}
	}
	/* Overview drawing area */
	gtk_signal_connect(GTK_OBJECT(darea), "expose_event",
					   GTK_SIGNAL_FUNC(draw_overview_lines), gfileviews);
	
	return;
}

/**
 * draw_overview_lines:
 * Draw lines which connects two diff parts of two files.
 **/
static void
draw_overview_lines(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
	GDiffFileViews *gfileviews = data;
	DiffFiles *files = gfileviews->dfiles;
	const FileInfo *fi1 = dfiles_get_fileinfo(files, FIRST_FILE, TRUE);
	const FileInfo *fi2 = dfiles_get_fileinfo(files, SECOND_FILE, TRUE);
	int all_nline1 = fi1->nline;
	int all_nline2 = fi2->nline;
	GList *node;/* node of DiffLines' list */
	GdkWindow *w = widget->window;
	GdkGC *gc = widget->style->fg_gc[GTK_STATE_NORMAL];
	int darea_width;
	int darea_height;

	gdk_window_get_size(w, &darea_width, &darea_height);
	for (node = files->dlines_list; node; node = node->next) {
		int y1, y2;
		DiffLines *dlines;
		int begin1, end1;
		int begin2, end2;

		dlines = node->data;
		/* Use local variables for ease to read */
		begin1 = dlines->between[FIRST_FILE].begin;
		end1 = dlines->between[FIRST_FILE].end;
		begin2 = dlines->between[SECOND_FILE].begin;
		end2 = dlines->between[SECOND_FILE].end;

		if (all_nline1 != 0)
			y1 = darea_height * ((double)(begin1+end1)/(2*all_nline1));
		else
			y1 = 0;
		if (all_nline2 != 0)
			y2 = darea_height * ((double)(begin2+end2)/(2*all_nline2));
		else
			y2 = 0;
		gdk_draw_line(w, gc, 0, y1, darea_width, y2);
	}
}


/* Routines for scrollbar to control both files */
/**
 * create_vscrollbar:
 **/
static void
create_vscrollbar(GDiffFileViews *gfileviews)
{
#define PAGE_RATIO		10.0
	GtkHBox *hbox = gfileviews->pane.two.hbox;
	GtkWidget *vscrollbar;
	GtkAdjustment *adj;
	int max_nlines;

	max_nlines = dfiles_get_max_nlines(gfileviews->dfiles);
	adj = GTK_ADJUSTMENT(gtk_adjustment_new(0, 0, max_nlines, 1, max_nlines/PAGE_RATIO, 1));/*XXX*/
	vscrollbar = gtk_vscrollbar_new(adj);
	gtk_box_pack_start(GTK_BOX(hbox), vscrollbar, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
					   GTK_SIGNAL_FUNC(vscrollbar_vchanged), gfileviews);
	gtk_widget_show(vscrollbar);
	gfileviews->pane.two.vscrollboth = GTK_VSCROLLBAR(vscrollbar);
}

/**
 * vscrollbar_vchanged:
 * Called when a user changes the value of scrollbar to control both files.
 **/
static void
vscrollbar_vchanged(GtkAdjustment *vadj, gpointer data)
{
	GDiffFileViews *gfileviews = data;
	GtkAdjustment *adj1 = gfileviews->pane.two.text[FIRST_FILE]->vadj;
	GtkAdjustment *adj2 = gfileviews->pane.two.text[SECOND_FILE]->vadj;
	gdouble f1_max = adj1->upper - adj1->lower - adj1->page_size;
	gdouble f2_max = adj2->upper - adj2->lower - adj2->page_size;
	gdouble max_nlines = vadj->upper;
	gdouble value = vadj->value;

	g_return_if_fail(max_nlines != 0);

	gtk_adjustment_set_value(adj1, value / max_nlines * f1_max);
	gtk_adjustment_set_value(adj2, value / max_nlines * f2_max);
}


/* Routines for move to a difference */
/**
 * guts_move_diff:
 * This is driven by the first file's difference.
 * It might be O.K., if it displayed the first line of the difference
 * at the top on the text widget.
 * But I adjust the position to near the center (minus page_size/2).
 **/
static void
guts_move_diff(GDiffFileViews *gfileviews, MoveDiff mv_diff)
{
	GtkAdjustment *vadj1 = gfileviews->pane.two.text[FIRST_FILE]->vadj;
	GtkAdjustment *vadj2 = gfileviews->pane.two.text[SECOND_FILE]->vadj;
	DiffFiles *files = gfileviews->dfiles;
	const FileInfo *fi1 = dfiles_get_fileinfo(files, FIRST_FILE, TRUE);
	const FileInfo *fi2 = dfiles_get_fileinfo(files, SECOND_FILE, TRUE);
	int cur_line1;
	const DiffLines *dlines = NULL;

	switch (mv_diff) {
	case MOVED_NEXT:
		gfileviews->cur_dlines_list = dfiles_find_nextl(files, gfileviews->cur_dlines_list);
		dlines = gfileviews->cur_dlines_list ? gfileviews->cur_dlines_list->data : NULL;
		break;
	case MOVED_PREV:
		gfileviews->cur_dlines_list = dfiles_find_prevl(files, gfileviews->cur_dlines_list);
		dlines = gfileviews->cur_dlines_list ? gfileviews->cur_dlines_list->data : NULL;
		break;
	case MOVED_FIRST:
		gfileviews->cur_dlines_list = dfiles_get_firstl(files, gfileviews->cur_dlines_list);
		dlines = gfileviews->cur_dlines_list ? gfileviews->cur_dlines_list->data : NULL;
		break;
	case MOVED_LAST:
		gfileviews->cur_dlines_list = dfiles_get_lastl(files, gfileviews->cur_dlines_list);
		dlines = gfileviews->cur_dlines_list ? gfileviews->cur_dlines_list->data : NULL;
		break;
	case MOVED_REL_NEXT:
		cur_line1 = guess_visible_bottom_line(vadj1, fi1->nline);
		dlines = dfiles_find_rel_nextl(files, FIRST_FILE, cur_line1);
		break;
	case MOVED_REL_PREV:
		cur_line1 = guess_visible_top_line(vadj1, fi1->nline);		
		dlines = dfiles_find_rel_prevl(files, FIRST_FILE, cur_line1);
		break;
	}

	if (dlines) {
		gdouble value;
		gdouble max;

		if (fi1->nline) {
			max = vadj1->upper - vadj1->lower;
			g_return_if_fail(max != 0);
			value = ((double)dlines->between[FIRST_FILE].begin / fi1->nline * max) - (vadj1->page_size / 2);
			gtk_adjustment_set_value(vadj1, value);
		}
		if (fi2->nline) {
			max = vadj2->upper - vadj2->lower;
			g_return_if_fail(max != 0);
			value = ((double)dlines->between[SECOND_FILE].begin / fi2->nline * max) - (vadj2->page_size / 2);
			gtk_adjustment_set_value(vadj2, value);
		}
	}
}

/**
 * adjust_vscrollbar:
 * Called when moving next or previous difference.
 * Calculate the value of vscrollbar which controls both files.
 * Theoretically, adj1 or adj2 don't matter for the calculation.
 * I'm using adj1.
 * The calculation is reverse of vscrollbar_vchanged()'s one.
 **/
static void
adjust_vscrollbar(GDiffFileViews *gfileviews)
{
	GtkAdjustment *adj1 = gfileviews->pane.two.text[FIRST_FILE]->vadj;
	GtkAdjustment *vadj;
	gdouble max_nlines;
	gdouble f1_max = adj1->upper - adj1->lower - adj1->page_size;

	if (f1_max == 0)
		return;
	
	vadj = gtk_range_get_adjustment(GTK_RANGE(gfileviews->pane.two.vscrollboth));
	max_nlines = vadj->upper;

	gtk_signal_handler_block_by_func(
		GTK_OBJECT(vadj), GTK_SIGNAL_FUNC(vscrollbar_vchanged), gfileviews);
	gtk_adjustment_set_value(vadj, adj1->value / f1_max * max_nlines);
	gtk_signal_handler_unblock_by_func(
		GTK_OBJECT(vadj), GTK_SIGNAL_FUNC(vscrollbar_vchanged), gfileviews);
}
