/*
 * Misc module
 * This module is for generic functions.
 * Independent from specific data structure.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#if defined(HAVE_STRING_H)
#include <string.h>
#elif defined(HAVE_STRINGS_H)
#include <strings.h>
#endif
#include <sys/stat.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <glib.h>
#include <gdk/gdk.h>/* for gdk_mbstowcs */
#include "misc.h"


/**
 * skip_n_lines:
 * Skip "nl" lines in the buffer, and return the head of the line.
 * Input:
 * const char *ptr; Buffer. Not null-terminated.
 * int nl; Number of lines to skip
 * int lenb; buffer length(byte)
 * Output:
 * Return value; Pointer to the head of the line. If exceeds, NULL.
 **/
const char*
skip_n_lines(const char *ptr, int nl, int lenb)
{
	int ml = 0;
	const char *pt2;

	if (nl <= 0)
		return ptr;
	while ((pt2 = memchr(ptr, '\n', lenb))) {
		ml++;
		if (ml == nl)
			break;
		lenb -= (pt2 + 1 - ptr);
		ptr = pt2 + 1;
	}
	return pt2 ? (pt2 + 1) : NULL;
}


/**
 * check_filetype:
 * Simple and generic file type check.
 * Input:
 * const char *fname; file name.
 * Output:
 * Return value: FileType;
 **/
FileType
check_filetype(const char *fname)
{
	struct stat sb;

	if (fname == NULL || fname[0] == '\0')
		return OTHERFILE;
	if (stat(fname, &sb) == -1) {
		g_warning("%s: stat in check_filetype()", fname);
		return OTHERFILE;
	}
	if (sb.st_mode & S_IFREG)
		return REGULARFILE;
	else if (sb.st_mode & S_IFDIR)
		return DIRECTORY;
	else
		return OTHERFILE;
}


/**
 * get_num_chars:
 * Retrun the number of characters in the buffer.
 * In SBCS(Single Byte Character Set) environment, not called for efficiency.
 * In that case, you can assume the number of characters is the same as one of bytes.
 * Input:
 * const char *src; buffer, normally MBCS.
 * int lenb; length in bytes.
 * Output:
 * Return value; the number of characters.
 **/
int
get_num_chars(const char *src, int lenb)
{
	int num_char;
	GdkWChar *pwcs;
	GdkWChar wcs[BUFSIZ];
	char *pmbs;
	char mbs[BUFSIZ];/* buffer to keep nul-terminated string */
	gboolean buf_allocated = FALSE;

	if (lenb < BUFSIZ) {
		pwcs = wcs;
		pmbs = mbs;
	} else {
		pmbs = (char*)g_malloc(lenb+1);
		pwcs = (GdkWChar*)g_malloc((lenb+1)*sizeof(GdkWChar));
		buf_allocated = TRUE;
	}
	memcpy(pmbs, src, lenb);
	pmbs[lenb] = '\0';
	/*XXX:this would be better.
	  num_char = gdk_mbstowcs(NULL, pbuf, 0);*/
	num_char = gdk_mbstowcs(pwcs, pmbs, lenb+1);
	
	if (buf_allocated == TRUE) {
		g_free(pmbs);
		g_free(pwcs);
	}	
	return num_char;
}

/**
 * calc_number_places:
 * Calculate the number of places on base 10.
 * e.g. 999 => 3, 9999 => 4, 99999 => 5 ...
 * Input:
 * int n;
 * Output:
 * Return value;
 **/
int
calc_number_places(int n)
{
	int ret  = 0;

	while (n) {
		n /= 10;
		ret++;
	}
	return ret;
}

/**
 * get_file_name:
 * Get file name from file path.
 **/
void
get_file_name(char *fname, const char *fpath)
{
	char *ptr;

	if (fpath == NULL || fpath[0] == '\0') {
		fname[0] = '\0';
		return;
	}
	ptr = strrchr(fpath, '/');
	if (ptr) {
		strcpy(fname, ptr + 1);
	} else {
		strcpy(fname, fpath);
	}
}


/**
 * close_allfd_except_std:
 * close all file descriptors except standard fds(stdin, stdout, stderr).
 * Usually, called by a forked child process.
 * Its main purpose is to close fd for the socket connected to X server.
 * Input:
 * None;
 * Output:
 * None;
 **/
void
close_allfd_except_std(void)
{
	int fd_max;
	int fd;

	fd_max = sysconf(_SC_OPEN_MAX);
	for (fd = STDERR_FILENO+1; fd < fd_max; fd++) {
		close(fd);
	}
}

/**
 * def_signal_handler:
 * default (Unix)signal handler.
 * Now, it takes care of child process death.
 **/
void
def_signal_handler(int sig)
{
	int sta;

	if (sig == SIGCHLD)
		wait(&sta);
}
