/*
 * GUI file view module
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <gnome.h>
#include "gdiff.h"
#include "gui.h"
#include "fileview.h"
#include "guiwin.h"
#include "dirview.h"


/**
 * gdiff_fileviews_new:
 * Allocate GDiffFileViews, intialize it and return its pointer.
 * And it also creates widgets for displaying the result.
 * Input:
 * GDiffWindow *gdwin; kept in GDiffFileViews for reference.
 * GDiffDirViews *gdirviews; kept in GDiffFileViews for reference.
 * const DiffFiles *dfiles; (Back-end data)kept in GDiffFileViews for reference.
 * PaneMode pmode; ONE_PANE or TWO_PANE.
 * Ouput:
 * Return value; Allocated GDiffFileViews.
 **/
GDiffFileViews*
gdiff_fileviews_new(GDiffWindow *gdwin, GDiffDirViews *gdirviews, DiffFiles *dfiles, PaneMode pmode)
{
	GDiffFileViews *gfileviews;
	const FileInfo *fi1;
	const FileInfo *fi2;

	gfileviews = g_new(GDiffFileViews, 1);

	/* Initialize the preference */
	gfileviews->pref = g_pref.fvpref;
	gfileviews->pref.pane_mode = pmode;
	
	gfileviews->dfiles = dfiles;
	gfileviews->cur_dlines_list = NULL;
	
	/* Can be referred to from each other. */
	gdirviews->gdfile_list = g_slist_prepend(gdirviews->gdfile_list, gfileviews);
	gfileviews->gdirviews = gdirviews;

	/* Can be referred to from GDiffWindow. */
	gdwin->gfilev_list = g_list_prepend(gdwin->gfilev_list, gfileviews);
	gfileviews->gdwin = gdwin;

	fi1 = dfiles_get_fileinfo(dfiles, FIRST_FILE, FALSE);
	fi2 = dfiles_get_fileinfo(dfiles, SECOND_FILE, FALSE);
	gfileviews->filename1 = g_strdup(fi1->fname);
	gfileviews->filename2 = g_strdup(fi2->fname);

	/* Create widgets for display diff files */
	if (gfileviews->pref.pane_mode == ONE_PANE)
		onepane_create_widgets(gfileviews);
	else if (gfileviews->pref.pane_mode == TWO_PANE)
		twopane_create_widgets(gfileviews);

	gfileviews->merged_nline = 0;
			
	return gfileviews;
}


/**
 * gdiff_fileviews_delete:
 * Finalize GDiffFileViews structure, and free its memory.
 * It also destroys the related widgets.
 * Input:
 * GDiffFileViews *gfileviews;
 * Output:
 * None;
 **/
void
gdiff_fileviews_delete(GDiffFileViews *gfileviews)
{
	GDiffWindow *gdwin;

	g_return_if_fail(gfileviews != NULL);

	gdwin = gfileviews->gdwin;

	g_free(gfileviews->filename1);
	g_free(gfileviews->filename2);

	/* XXX: to destory is enough ? */
	/* Remove its own page from the notebook widget */
	if (gfileviews->base) {
		gint page;
		page = gtk_notebook_page_num(gfileviews->gdwin->notebook, GTK_WIDGET(gfileviews->base));
		gtk_notebook_remove_page(gfileviews->gdwin->notebook, page);
		/*gtk_widget_destroy(GTK_WIDGET(gfileviews->base));*/
	}

	/* Remove own node from the list managed by GDiffWindow */
	gdwin->gfilev_list = g_list_remove(gdwin->gfilev_list, gfileviews);

	/* Free its own memory */
	g_free(gfileviews);
}
