/*
 * Back-end main module
 * See "gdiff.h" for the details of data structure.
 * This module should be independent from GUI frontend.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#if defined(HAVE_STRING_H)
#include <string.h>
#elif defined(HAVE_STRINGS_H)
#include <strings.h>
#endif
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#if defined(HAVE_ERRNO_H)
#include <errno.h>
#else defined(HAVE_SYS_ERRNO_H)
#include <sys/errno.h>
#endif
#include <glib.h>
#include "gdiff.h"


/* Private function declarations */
static DiffFiles* dfiles_new(const char *fname1, const char *fname2);
static void dfiles_delete(DiffFiles *dfiles);
static void init_fileinfo(FileInfo *fi, char const *fname);
static void term_fileinfo(FileInfo *fi);
static int get_nlines(const char *ptr, int lenb);

/**
 * diff_dir_new:
 * Allocate DiffDir structure, and return its pointer.
 * Input:
 * const char *fname1; Regular file or directory.
 * const char *fname2; Regular file or directory.
 * const char *args; Argument string to diff(1).
 * Output:
 * Return value: DiffDir*;
 **/
DiffDir*
diff_dir_new(const char *fname1, const char *fname2, const char *args)
{
	DiffDir *diffdir;

	diffdir = g_new(DiffDir, 1);
	diffdir->dfiles_list = NULL;
	diff_dir_add_files(diffdir, fname1, fname2);
	run_diff(diffdir, fname1, fname2, args, NULL);
	
	return diffdir;
}

/**
 * diff_dir_delete:
 * Finalize DiffDir structure, and free its memory.
 * Input:
 * DiffDir *diffdir;
 * Output:
 * None;
 **/
void
diff_dir_delete(DiffDir *diffdir)
{
	GSList *list;
	
	for (list = diffdir->dfiles_list; list; list = list->next) {
		dfiles_delete(list->data);
	}
	g_slist_free(diffdir->dfiles_list);
	g_free(diffdir);
}

/**
 * diff_dir_add_files:
 * Allocate DiffFiles(call dfiles_new()), 
 * and add it to GSList *dfiles_list of DiffDir structure.
 * Never move the first node, which is a special node.
 * On the other hand, the other nodes' order doesn't matter.
 * Input:
 * DiffDir *diffdir;
 * const char *fname1;
 * const char *fname2;
 * Output:
 * DiffDir *diffdir; GSList *dfiles_list is updated.
 * Return value: DiffFiles*; Added DiffFiles' pointer.
 **/
DiffFiles*
diff_dir_add_files(DiffDir *diffdir, const char *fname1, const char *fname2)
{
	DiffFiles *dfiles;

	dfiles = dfiles_new(fname1, fname2);
	/* append is not so efficient, but don't care */
	diffdir->dfiles_list = g_slist_append(diffdir->dfiles_list, dfiles);

	return dfiles;
}

/**
 * diff_dir_remove_files:
 * Remove specified DiffFiles from singly linked list in DiffDir.
 * Input:
 * DiffDir *diffdir;
 * DiffFiles *dfiles;
 * Output:
 * DiffDir *diffdir; GSList *dfiles_list is updated.
 * DiffFiles *dfiles; Freed.
 **/
void
diff_dir_remove_files(DiffDir *diffdir, DiffFiles *dfiles)
{
	dfiles_delete(dfiles);
	diffdir->dfiles_list = g_slist_remove(diffdir->dfiles_list, dfiles);
}

/**
 * dfiles_get_fileinfo:
 * Return the pointer to FileInfo(member variable of DiffFiles).
 * If the caller needs an actual text content or its number of lines,
 * it should set TRUE for f_mmap.
 * In general, mmap() is delayed, so if "f_mmap" is TRUE, mmap() is executed here.
 * If it's already mmap'ed(fi->text != NULL), do nothing.
 * Input:
 * DiffFiles *dfiles;
 * WhichFile n;
 * gboolean f_mmap; if TRUE do mmap its text, unless it has done.
 * Output:
 * DiffFiles *dfiles; "text" and "nlines" can be updated by mmap().
 * Return value; pointer to FileInfo.
 **/
const FileInfo*
dfiles_get_fileinfo(DiffFiles *dfiles, WhichFile n, gboolean f_mmap)
{
	FileInfo *fi = &dfiles->fileinfo[n];

	if (f_mmap == TRUE
		&& fi->f_dir == FALSE && fi->text == NULL && fi->lenb > 0) {
		int fd;

		if ((fd = open(fi->fname, O_RDONLY)) == -1) {
			g_warning("dfiles_get_fileinfo:open %s %d", fi->fname, errno);
			return fi;
		}
		/*XXX: MAP_NOEXTEND is dependent on platform */
		/*fi->text = mmap(0, fi->lenb, PROT_READ, MAP_PRIVATE|MAP_NOEXTEND, fd, 0);*/
		fi->text = mmap(0, fi->lenb, PROT_READ, MAP_PRIVATE, fd, 0);
		if (fi->text == (caddr_t)-1) {
			g_warning("dfiles_get_fileinfo:mmap %s %d", fi->fname, errno);
			fi->text = NULL;
			return fi;
		}
		fi->nline = get_nlines(fi->text, fi->lenb);
		close(fd);
	}
	return fi;
}

/**
 * dfiles_add_dlines:
 * Allocate DiffLines, and add it to GList *dlines_list of DiffFiles structure. 
 * Input:
 * DiffFiles *dfiles;
 * DiffType dtype;
 * int f1n1, f1n2, f2n1, f2n2; Each is line number.
 * Output:
 * DiffFiles* dfiles; GList *dlines_list is updated.
 **/
void
dfiles_add_dlines(DiffFiles* dfiles, DiffType dtype, int f1n1, int f1n2, int f2n1, int f2n2)
{
	DiffLines *dlines;
	
	dlines = g_new(DiffLines, 1);
	dlines->difftype = dtype;
	dlines->between[FIRST_FILE].begin = f1n1;
	dlines->between[FIRST_FILE].end = f1n2;
	dlines->between[SECOND_FILE].begin = f2n1;
	dlines->between[SECOND_FILE].end = f2n2;

	/* Not so efficient, but don't care */
	dfiles->dlines_list = g_list_append(dfiles->dlines_list, dlines);
}



/* ---The followings are private functions--- */
/**
 * dfiles_new:
 * Allocate DiffFiles structure, initialize and return its pointer.
 * Input:
 * const char *fname1; First file name. Directory is acceptable.
 * const char *fname2; Second file name. Directory is acceptable.
 * Output:
 * Return value: DiffFile*; pointer to the allocated DiffFiles structure;
 **/
static DiffFiles*
dfiles_new(const char *fname1, const char *fname2)
{
	DiffFiles *dfiles;

	dfiles = g_new(DiffFiles, 1);

	/* In the case only one file exists, don't open the file. */
	if (fname1[0] == '\0' || fname2[0] == '\0') {
		init_fileinfo(&dfiles->fileinfo[FIRST_FILE], NULL);
		init_fileinfo(&dfiles->fileinfo[SECOND_FILE], NULL);
		if (fname1[0])
			dfiles->fileinfo[FIRST_FILE].fname = g_strdup(fname1);
		if (fname2[0])
			dfiles->fileinfo[SECOND_FILE].fname = g_strdup(fname2);
	} else {
		init_fileinfo(&dfiles->fileinfo[FIRST_FILE], fname1);		
		init_fileinfo(&dfiles->fileinfo[SECOND_FILE], fname2);
	}
	dfiles->dlines_list = NULL;
	dfiles->binary = FALSE;

	return dfiles;
}


/**
 * dfiles_delete:
 * Finalize DiffFiles structure, and free its memory.
 * Input:
 * DiffFiles *dfiles;
 * Output:
 * None;
 **/
static void
dfiles_delete(DiffFiles *dfiles)
{
	GList *list;
	
	term_fileinfo(&dfiles->fileinfo[FIRST_FILE]);
	term_fileinfo(&dfiles->fileinfo[SECOND_FILE]);

	for (list = dfiles->dlines_list; list; list = list->next) {
		g_free(list->data);
	}
	g_list_free(dfiles->dlines_list);

	g_free(dfiles);
}


/**
 * init_fileinfo:
 * Initialize FileInfo structure.
 * But it doesn't do mmap(), and make it delayed for dfiles_get_fileinfo().
 * Input:
 * const char *fname;
 * Output:
 * FileInfo *fi;
 **/
static void
init_fileinfo(FileInfo *fi, const char *fname)
{
	int fd;
	struct stat sb;

	/* default values */
	fi->nline = 0;
	fi->text = NULL;
	fi->lenb = 0;
	fi->f_dir = FALSE;

	if (fname) {
		fi->fname = g_strdup(fname);
		if ((fd = open(fname, O_RDONLY)) == -1) {
			g_warning("init_fileinfo:open %s %d", fname, errno);
			return;
		}
		if (fstat(fd, &sb) == -1) {
			g_warning("init_fileinfo:fstat %s %d", fname, errno);
			close(fd);
			return;
		}
		if (sb.st_mode & S_IFREG) {
			fi->lenb = sb.st_size;
		}
		else if (sb.st_mode & S_IFDIR) {
			fi->f_dir = TRUE;
		}
		else {
			/* not regular-file, nor directory. */
			g_warning("init_fileinfo:special file");
		}
		fi->mtime = sb.st_mtime;
		fi->ctime = sb.st_ctime;
		close(fd);
	} else {
		fi->fname = NULL;
	}
}

/**
 * term_fileinfo:
 * Terminate FileInfo structure. (not free its own memory.)
 * Input:
 * FileInfo *fi;
 * Output:
 * None;
 **/
static void
term_fileinfo(FileInfo *fi)
{
	if (fi->fname) {
		g_free(fi->fname);
	}
	if (fi->text) {
		if (munmap(fi->text, fi->lenb) == -1)
			g_warning("munmap in term_fileinfo %d", errno);
	}
}


/**
 * get_nlines:
 * Return the number of lines of the buffer.
 * Counts the last line, although it isn't ended with '\n' character.
 * Input:
 * const char *ptr; Buffer. Not null-terminated.
 * int lenb; Buffer length(Bytes).
 * Output:
 * Return value; The number of lines of the buffer.
 **/
static int
get_nlines(const char *ptr, int lenb)
{
	int nl = 0;
	const char *pt2;

	while ((pt2 = memchr(ptr, '\n', lenb))) {
		nl++;
		lenb -= (pt2 + 1 - ptr);
		ptr = pt2 + 1;
	}
	return nl;
}
