=head1 NAME

docbook2texi-spec - convert DocBook Books to a Texinfo document

=head1 DESCRIPTION

This is a sgmlspl spec file that produces a Texinfo file
from DocBook markup.

=head1 LIMITATIONS

Trying docbook2info on non-DocBook or non-conformant SGML results in
undefined behavior. :-)

This program is a slow, dodgy Perl script.  

=head1 COPYRIGHT

Copyright (C) 1998-1999 Steve Cheng <steve@ggi-project.org>

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the Free
Software Foundation; either version 2, or (at your option) any later
version.

You should have received a copy of the GNU General Public License along with
this program; see the file COPYING.  If not, please write to the Free
Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

=cut

# $Id: docbook2texi-spec.pl,v 1.7 1999/07/30 21:57:12 steve Exp $

use SGMLS;			# Use the SGMLS package.
use SGMLS::Output;		# Use stack-based output.
use SGMLS::Refs;

########################################################################
# SGMLSPL script produced automatically by the script sgmlspl.pl
#
# Document Type: BOOK
# Edited by: me :)
########################################################################


$nocollapse_whitespace = 0;	# Current whitespace collapse counter.
$newline_last = 1;		# At beginning of line?

$skip_inline = 0;
$id_counter = 1;
$raw_cdata = 0;

$separator = '';		# See docbook2man about this ugly KLUDGE

$basename = shift || "db2texi";

sgml('start', sub {
	$Refs = new SGMLS::Refs("$basename.refs");
});
sgml('end', sub {
	$Refs->warn();
});




########################################################################
#
# Output helpers 
#
########################################################################

sub save_cdata
{
	$raw_cdata++;
	push_output('string');
}
		
# Copied from docbook2man.
# Texinfo's newline rules aren't so stringent, so
# perhaps we can do away with at least some of these cases for speed.

sub texi_sgml
{
	if(ref($_[1]) eq 'CODE') {
		return &sgml;
	}
	
	my $s = $_[1];

	$s =~ s/\\/\\\\/g;
	$s =~ s/'/\\'/g;

	sgml($_[0], eval("sub { texi_output '$s' }"));
}

sub texi_output
{
	if($separator eq 'full') {
		output "\n" unless $newline_last++;
		output "\@noindent\n";
		$separator = '';
	}
	
	$_ = shift;
	if(s/^\n//) {
		output "\n" unless $newline_last++;
	}
	return if $_ eq '';
	
	output $_;

	if(@_) {
		output @_;
		$newline_last = (pop(@_) =~ /\n$/);
	} else {
		$newline_last = ($_ =~ /\n$/)
	}
}

# Fold lines into one, quote some characters
sub fold_string
{
	$_ = shift;
	
	for($_[0]) {
		tr/\t/\n  /;
		s/\@/\@\@/g;
		s/\{/\@\{/g;
		s/\}/\@\}/g;
	}
	
	# Trim whitespace from beginning and end.
	s/^ +//;
	s/ +$//;

	return $_;
}

# Another version of sgml(), for 'inline' @-commands,
# and prevent nesting them.
sub _inline
{
	my ($gi, $tcmd) = @_;
	
	$tcmd =~ s/\\/\\\\/g;
	$tcmd =~ s/'/\\'/g;

	sgml($gi, eval("sub { texi_output '${tcmd}\{' unless \$raw_cdata or \$skip_inline++ }"));
	
	$gi =~ s/^</<\//;
	sgml($gi, eval("sub { texi_output '}' unless \$raw_cdata or --\$skip_inline }"));
}


sub generate_id {
	return "ID" . $id_counter++;
}
    


########################################################################
#
# Metadata
#
########################################################################

sub author_start {
	if($_[0]->within('BOOKINFO')) {
		save_cdata();
	}
}
sub author_end {
	if($_[0]->within('BOOKINFO')) {
		texi_output('@author ', fold_string(pop_output()), "\n");
		$raw_cdata--;
	}
}

sgml('<AUTHOR>', \&author_start);
sgml('</AUTHOR>', \&author_end);
sgml('<EDITOR>', \&author_start);
sgml('</EDITOR>', \&author_end);
sgml('<COLLAB>', \&author_start);
sgml('</COLLAB>', \&author_end);
sgml('<CORPAUTHOR>', \&author_start);
sgml('</CORPAUTHOR>', \&author_end);
sgml('<OTHERCREDIT>', \&author_start);
sgml('</OTHERCREDIT>', \&author_end);


# Ignore content.
sgml('<CONTRIB>', sub { push_output('nul') });
sgml('</CONTRIB>', sub { pop_output });
sgml('<AUTHORBLURB>', sub { push_output('nul') });
sgml('</AUTHORBLURB>', sub { pop_output });


sgml('<TITLEABBREV>', sub { push_output('nul') });
sgml('</TITLEABBREV>', sub { pop_output() });

sgml('<LEGALNOTICE>', sub {
	if($_[0]->in('BOOKINFO')) {
		texi_output "\n\@page\n\@vskip 0pt plus 1fill\n";
	} else {
		push_output('nul');
	}
});
sgml('</LEGALNOTICE>', sub {
	if(!$_[0]->in('BOOKINFO')) {
		pop_output();
	}
});



# Don't support them yet!
sgml('<DOCINFO>', sub { push_output('nul'); });
sgml('</DOCINFO>', sub { pop_output(); });
sgml('<REFSECT1INFO>', sub { push_output('nul'); });
sgml('</REFSECT1INFO>', sub { pop_output(); });
sgml('<REFSECT2INFO>', sub { push_output('nul'); });
sgml('</REFSECT2INFO>', sub { pop_output(); });
sgml('<REFSECT3INFO>', sub { push_output('nul'); });
sgml('</REFSECT3INFO>', sub { pop_output(); });



sgml('<BOOKINFO>', sub {
	texi_output("\n\@titlepage\n");
	texi_output('@title{', $doc_title, "\}\n");
});
#texi_sgml('</BOOKINFO>', "\n\@end titlepage\n");
texi_sgml('</BOOKINFO>', sub {
	texi_output "\n\@end titlepage\n\n";
	texi_output "\@node Top\n\@top\n";

	# Generate top menu for BOOK.
	$_[0]->parent->ext->{'output_toc'} = 1;
		
	# This is so that elements below can find me.
	$_[0]->parent->ext->{'id'} = $_[0]->parent->attribute('ID')->value || generate_id();
});



sgml('<TITLE>', \&save_cdata);
sgml('</TITLE>', sub { 
	my $title = fold_string(pop_output());
	$raw_cdata--;
	
	if($_[0]->in('BOOKINFO')) {
		texi_output("\n\@subtitle{", $title, "\}\n");
	}
	elsif(exists $_[0]->parent->ext->{'title'}) {
		# By far the easiest case.  Just fold the string as
		# above, and then set the parent element's variable.
		$_[0]->parent->ext->{'title'} = $title;
	}
	elsif(exists $_[0]->parent->ext->{'nodesection'}) {
		# Start new node, since we now know its title.

		# This node.
		$me = $_[0]->parent;
		
		my $nodename = fold_string($me->attribute('XREFLABEL')->value) || $title;
		$nodename =~ s/,/ /;
		
		texi_output("\n\n\@node $nodename\n");

		# The heading such as @chapter $title.
		texi_output($me->ext->{'nodesection'}, " $title\n");

		# This is so that elements below can find me.
		my $id = $me->attribute('ID')->value || generate_id();
		$me->ext->{'id'} = $id;

		# Don't overwrite info from previous parse.
		if($Refs->get($id) eq '') {
			$Refs->put($id, $nodename);
		
			# Add myself to parent node's list of nodes.
			my $pid = $me->parent->ext->{'id'};
			$Refs->put($pid, $Refs->get($pid) . ",$id");
		}
	}
	else {
		if($_[0]->in('BOOK') or $_[0]->in('ARTICLE')) {
			$doc_title = $title;
		}
		output $title, "\n";
		$newline_last++;
	}
});


# FIXME!!!
texi_sgml('</ARTHEADER>', sub {
	texi_output "\n\n\@node Top\n\@top\n";

	# Generate top menu for BOOK.
	$_[0]->parent->ext->{'output_toc'} = 1;
		
	# This is so that elements below can find me.
	$_[0]->parent->ext->{'id'} = $_[0]->parent->attribute('ID')->value || generate_id();
});


########################################################################
#
# (major) Document elements
#
########################################################################

sub start_document {
	if($_[0]->parent eq '') {
		output "\\input texinfo\n\@settitle ";
	} else {
		start_node $_[0], '@chapter';
	}
}

sub end_document {
	if($_[0]->parent eq '') {
		texi_output "\n\n\@bye\n";
	}
}

# SET
# I recommend you process each Book separately.

sgml('<BOOK>', \&start_document);
sgml('</BOOK>', \&end_document);

sgml('<ARTICLE>', \&start_document);
sgml('</ARTICLE>', \&end_document);

sgml('<REFERENCE>', \&start_document);
sgml('</REFERENCE>', \&end_document);



########################################################################
#
# Major sectioning elements
#
########################################################################


# Start a new section with new node.  
# Most of the handling is at </TITLE>.
sub start_node
{
	$separator = '';
	
	$_[0]->ext->{'nodesection'} = $_[1];

	# Flag that the first child node ...
	$_[0]->ext->{'output_toc'} = 1;
	
	# ... need to output a list of nodes.
	if($_[0]->parent->ext->{'output_toc'}) {
		$_[0]->parent->ext->{'output_toc'} = 0;
		
		my @nodes = split(/,/, $Refs->get($_[0]->parent->ext->{'id'}));
		shift @nodes;

		texi_output("\n\n\@menu\n");

		foreach(@nodes) {
			my ($nodename) = split(/,/, $Refs->get($_));
			output "* ", $nodename, "::\n";
		}

		texi_output("\n\n\@end menu\n");
	}
}

sgml('<PREFACE>', sub { start_node $_[0], '@chapter'; });
sgml('<CHAPTER>', sub { start_node $_[0], '@chapter'; });

sgml('<SECT1>', sub { start_node $_[0], '@section'; });
sgml('<SECT2>', sub { start_node $_[0], '@subsection'; });
sgml('<SECT3>', sub { start_node $_[0], '@subsubsection'; });
sgml('<SECT4>', sub { start_node $_[0], '@subsubsection'; });
sgml('<SECT5>', sub { start_node $_[0], '@subsubsection'; });

sgml('<SIMPLESECT>', sub {
	if($_[0]->in('CHAPTER')) {
		start_node $_[0], '@section';
	} elsif($_[0]->in('SECT1')) {
		start_node $_[0], '@subsection';
	} else {
		# From Sect2 and after
		start_node $_[0], '@subsubsection';
	}
});

sgml('<APPENDIX>', sub { start_node $_[0], '@appendix'; });

# PART: this requires hacking into node handling.


########################################################################
#
# Reference pages
#
########################################################################

sgml('<REFENTRY>', sub {
	# Determine what it is under ...
	# FIXME! Add more of these parents!
	if($_[0]->in('CHAPTER') or $_[0]->in('REFERENCE')) {
		start_node $_[0], '@section';
	} elsif($_[0]->in('SECT1')) {
		start_node $_[0], '@subsection';
	} else {
		# From Sect2 and after
		start_node $_[0], '@subsubsection';
	}
});

sgml('<REFENTRYTITLE>', sub { 
	if($_[0]->in('REFMETA')) { 
		save_cdata();
	}
});
sgml('</REFENTRYTITLE>', sub { 
	if($_[0]->in('REFMETA')) { 
		my $title = fold_string(pop_output());
		$raw_cdata--;

		# This node.
		my $me = $_[0]->parent->parent;
				
		my $nodename = fold_string($me->attribute('XREFLABEL')->value) || $title;
		$nodename =~ s/,/ /;
		
		texi_output("\n\n\@node $nodename\n");

		# The heading such as @chapter $title.
		texi_output($me->ext->{'nodesection'}, " $title\n");

		# This is so that elements below can find me.
		my $id = $me->attribute('ID')->value || generate_id();
		$me->ext->{'id'} = $id;

		# Don't overwrite info from previous parse.
		if($Refs->get($id) eq '') {
			$Refs->put($id, $nodename);
		
			# Add myself to parent node's list of nodes.
			my $pid = $me->parent->ext->{'id'};
			$Refs->put($pid, $Refs->get($pid) . ",$id");
		}
	}
});

sgml('<MANVOLNUM>', sub { 
	if($_[0]->in('REFMETA')) { push_output('nul') }
	else			 { texi_output '(' }
});
sgml('</MANVOLNUM>', sub { 
	if($_[0]->in('REFMETA')) { pop_output() }
	else			 { texi_output ')' }
});

sgml('<REFMISCINFO>', sub { push_output('nul') });
sgml('</REFMISCINFO>', sub { pop_output() });

sgml('<REFNAMEDIV>', sub { 
	$_[0]->ext->{'first_refname'} = 1;

	# FIXME! Add more of these parents!
	if($_[0]->parent->in('CHAPTER')) {
		texi_output "\n\n\@unnumberedsubsec Name\n";
	} else {
		texi_output "\n\n\@unnumberedsubsubsec Name\n";
	}
});

sgml('<REFNAME>', sub {
	if($_[0]->parent->ext->{'first_refname'}) {
		$_[0]->parent->ext->{'first_refname'} = 0;
		return;
	}
	output ", ";
});

sgml('<REFPURPOSE>', " --- ");

# RefDescriptor

sgml('<REFSYNOPSISDIV>', sub { 
	# FIXME! Add more of these parents!
	if($_[0]->parent->in('CHAPTER')) {
		texi_output "\n\@unnumberedsubsec Synopsis\n";
	} else {
		texi_output "\n\@unnumberedsubsubsec Synopsis\n";
	}
});

sgml('<REFSECT1>', sub {
	$separator = '';
	
	if($_[0]->parent->in('CHAPTER')) {
		texi_output "\n\n\@unnumberedsubsec ";
	} else {
		texi_output "\n\n\@unnumberedsubsubsec ";
	}
});

sgml('<REFSECT2>', sub { $separator = ''; texi_output "\n\n\@unnumberedsubsubsec " });
sgml('<REFSECT3>', sub { $separator = ''; texi_output "\n\n\@unnumberedsubsubsec " });

texi_sgml('<BEGINPAGE>', "\n\@page\n");

########################################################################
#
# Indices
#
########################################################################

# hAXed

sgml('<INDEXTERM>', \&save_cdata);
sgml('</INDEXTERM>', sub {
	texi_output "\n\@cindex ", fold_string(pop_output()), "\n";
});

sgml('<SECONDARY>', ': ');
sgml('<TERTIARY>', ': ');
sgml('<SEE>', '; see ');
sgml('<SEEALSO>', '; see also ');









########################################################################
#
# Synopses
#
########################################################################

sgml('<FUNCSYNOPSIS>', sub { texi_output "\n\n"; $skip_inline++ });
sgml('</FUNCSYNOPSIS>', sub { texi_output "\n\n"; $skip_inline-- });
sgml('<CMDSYNOPSIS>', sub { texi_output "\n\n"; $skip_inline++ });
sgml('</CMDSYNOPSIS>', sub { texi_output "\n\n"; $skip_inline-- });

texi_sgml('<FUNCPROTOTYPE>', "\n\n");

# Arguments to functions.  This is C convention.
texi_sgml('<PARAMDEF>', '(');
texi_sgml('</PARAMDEF>', ');');
texi_sgml('<VOID>', '(void);');

sub arg_start
{
	# The content model for CmdSynopsis doesn't include #PCDATA,
	# so we won't see any of the whitespace in the source file,
	# so we have to add it after each component.
	texi_output ' ';

	if($_[0]->attribute('CHOICE')->value =~ /opt/i) {
		texi_output '[ ';
	}
}
sub arg_end
{
	if($_[0]->attribute('REP')->value =~ /^Repeat/i) {
		texi_output '...';
	}
	if($_[0]->attribute('CHOICE')->value =~ /opt/i) {
		texi_output ' ] ';
	}
}

sgml('<ARG>', \&arg_start);
sgml('</ARG>', \&arg_end);
sgml('<GROUP>', \&arg_start);
sgml('</GROUP>', \&arg_end);





########################################################################
#
# 'Useful highlighting' 
#
########################################################################


_inline('<CITETITLE>', '@cite');
_inline('<EMAIL>', '@email');
_inline('<FIRSTTERM>', '@dfn');
_inline('<FILENAME>', '@file');

_inline('<ACCEL>', '@key');
_inline('<KEYCAP>', '@key');
_inline('<KEYCOMBO>', '@key');
_inline('<KEYSYM>', '@key');
_inline('<USERINPUT>', '@kbd');

_inline('<LITERAL>', '@samp');
_inline('<MARKUP>', '@samp');
_inline('<SGMLTAG>', '@samp');
_inline('<TOKEN>', '@samp');

_inline('<CLASSNAME>', '@code');
_inline('<ENVAR>', '@code');
_inline('<FUNCTION>', '@code');
_inline('<PARAMETER>', '@code'); # not always
_inline('<RETURNVALUE>', '@code');
_inline('<STRUCTFIELD>', '@code');
_inline('<STRUCTNAME>', '@code');
_inline('<SYMBOL>', '@code');
_inline('<TYPE>', '@code');

_inline('<LITERAL>', '@code');
_inline('<MARKUP>', '@code');
_inline('<SGMLTAG>', '@code');

_inline('<ACRONYM>', '@acronym');
_inline('<EMPHASIS>', '@emph');

# Only in certain contexts
#sgml('<REPLACEABLE>'

sgml('<OPTIONAL>', "[");
sgml('</OPTIONAL>', "]");

sgml('<COMMENT>', "[Comment: ");
sgml('</COMMENT>', "]");

# ABBREV
# CITATION
# FOREIGNPHRASE
# PHRASE
# QUOTE
# WORDASWORD

# COMPUTEROUTPUT
# PROMPT
# TOKEN

# DATABASE
# HARDWARE
# INTERFACE
# MEDIALABEL
# SYSTEMITEM

# NO formatting
sgml('<CITEREFENTRY>', sub { $skip_inline++ });
sgml('</CITEREFENTRY>', sub { $skip_inline-- });

# These create spaces between content in special elements
# without PCDATA content.
texi_sgml('</HONORIFIC>', " ");
texi_sgml('</FIRSTNAME>', " ");
texi_sgml('</SURNAME>', " ");
texi_sgml('</LINEAGE>', " ");
texi_sgml('</OTHERNAME>', " ");

texi_sgml('<AFFILIATION>', "(");
texi_sgml('</AFFILIATION>', ") ");
texi_sgml('<CONTRIB>', "(");
texi_sgml('</CONTRIB>', ") ");

texi_sgml('</STREET>', " ");
texi_sgml('</POB>', " ");
texi_sgml('</POSTCODE>', " ");
texi_sgml('</CITY>', " ");
texi_sgml('</STATE>', " ");
texi_sgml('</COUNTRY>', " ");
texi_sgml('</PHONE>', " ");
texi_sgml('</FAX>', " ");
texi_sgml('</OTHERADDRESS>', " ");

texi_sgml('</ALT>', ": ");
texi_sgml('<GRAPHIC>', " [GRAPHIC] ");





########################################################################
#
# 'Block' elements 
#
########################################################################

sub para_start {
	if($separator eq '' or $separator eq 'full') {
		texi_output "\n\n";
		$separator = '';
	} elsif($separator eq 'none' ) {
		$_[0]->parent->ext->{'separator'} = 'blank';
		$separator = '';
	}
}

# Actually applies to a few other block elements as well
sub para_end {
	$separator = $_[0]->parent->ext->{'separator'};
	texi_output "\n";
}

sgml('<PARA>', \&para_start);
sgml('<SIMPARA>', \&para_end);

sgml('<EXAMPLE>', sub { $separator = ''; texi_output "\n\@unnumberedsubsubsec "});
sgml('</EXAMPLE>', \&para_end);




########################################################################
#
# Elements implemented using "Block enclosing commands"
#
########################################################################

sgml('<ATTRIBUTION>', sub {
	if($_[0]->in('BLOCKQUOTE')) {
		push_output('string');
	}
});
sgml('</ATTRIBUTION>', sub {
	if($_[0]->in('BLOCKQUTE')) {
		$_[0]->parent->ext->{'attribution'} = pop_output();
	} else {
		# For an Epigraph.
		texi_output "\n\n";
	}
});

# This element is almost like an admonition (below),
# only the default title is blank :)

sgml('<BLOCKQUOTE>', sub {
	$_[0]->ext->{'title'} = '';
	&para_start;
	push_output('string');
});
sgml('</BLOCKQUOTE>', sub {
	my $content = pop_output();

	if($_[0]->ext->{'title'}) {
		texi_output $_[0]->ext->{'title'}, ":\"\n\@sp 1\n";
	}

	texi_output "\@quotation\n";

	output $content;

	texi_output "\@end quotation\n";

	if($_[0]->ext->{'attribution'}) {
		output "\n" unless $newline_last++;
		# One place where roff's space-sensitivity makes sense
		# :)
		texi_output " --- ", $_[0]->ext->{'attribution'}, "\n";
	}
});

sgml('<ADDRESS>', "\n\@display\n");
sgml('</ADDRESS>', "\n\@end display\n");



########################################################################
#
# Verbatim displays. 
#
########################################################################

sub verbatim_on {
	texi_output("\n\n");
	texi_output("\@example\n") unless $nocollapse_whitespace++;
}

sub verbatim_off {
	texi_output("\n\n");
	texi_output("\@end example\n") unless --$nocollapse_whitespace;
}

texi_sgml('<PROGRAMLISTING>', \&verbatim_on); 
texi_sgml('</PROGRAMLISTING>', \&verbatim_off);
texi_sgml('<SCREEN>', \&verbatim_on); 
texi_sgml('</SCREEN>', \&verbatim_off);
texi_sgml('<LITERALLAYOUT>', \&verbatim_on); 
texi_sgml('</LITERALLAYOUT>', \&verbatim_off);
texi_sgml('<SYNOPSIS>', \&verbatim_on); 
texi_sgml('</SYNOPSIS>', \&verbatim_off);





########################################################################
#
# Admonitions and other 'outside flow'
#
########################################################################

sub admonition_end {
	my $content = pop_output();

	# When the admonition is only one paragraph,
	# it looks nicer if the title was inline.
	my $num_para;
	while ($content =~ /^\n\n/) { $num_para++ }
	if($num_para==1) {
		$content =~ s/\n\n//;
	}

	output $_[0]->ext->{'title'}, ":\n\@sp 1\n";
	output $content;

	output "\n\@end cartouche\n";
	$newline_last++;
}

sgml('<NOTE>', sub {
	# We can't see right now whether or not there is a TITLE
	# element, so we have to save the output now and add it back
	# at the end of this admonition.
	$_[0]->ext->{'title'} = 'Note';

	texi_output "\n\n\@cartouche\n";

	push_output('string');
});
sgml('</NOTE>', \&admonition_end);

sgml('<WARNING>', sub {
	$_[0]->ext->{'title'} = 'Warning';
	texi_output "\n\n\@cartouche\n";
	push_output('string');
});
sgml('</WARNING>', \&admonition_end);
sgml('<TIP>', sub {
	$_[0]->ext->{'title'} = 'Tip';
	texi_output "\n\n\@cartouche\n";
	push_output('string');
});
sgml('</TIP>', \&admonition_end);
sgml('<CAUTION>', sub {
	$_[0]->ext->{'title'} = 'Caution';
	texi_output "\n\n\@cartouche\n";
	push_output('string');
});
sgml('</CAUTION>', \&admonition_end);
sgml('<IMPORTANT>', sub {
	$_[0]->ext->{'title'} = 'Important';
	texi_output "\n\n\@cartouche\n";
	push_output('string');
});
sgml('</IMPORTANT>', \&admonition_end);




########################################################################
#
# Lists
#
########################################################################

texi_sgml('<VARIABLELIST>', "\n\@table \@asis\n");
texi_sgml('</VARIABLELIST>', sub {
	texi_output "\n\@end table\n";
	$_[0]->parent->ext->{'separator'} = 'full';
	$separator = 'full';
});
texi_sgml('<ORDEREDLIST>', "\n\@enumerate\n");
texi_sgml('</ORDEREDLIST>', sub {
	texi_output "\n\@end enumerate\n";
	$_[0]->parent->ext->{'separator'} = 'full';
	$separator = 'full';
});
texi_sgml('<ITEMIZEDLIST>', "\n\@itemize \@bullet\n");
texi_sgml('</ITEMIZEDLIST>', sub {
	texi_output "\n\@end itemize\n";
	$_[0]->parent->ext->{'separator'} = 'full';
	$separator = 'full';
});

sgml('<VARLISTENTRY>', sub { 
	$_[0]->ext->{'first_term'} = 1;
});

sgml('<TERM>', sub { 
	if($_[0]->parent->ext->{'first_term'}) {
		$_[0]->parent->ext->{'first_term'} = 0;
		texi_output "\n\n\@item ";
	} else {
		texi_output "\n\n\@itemx ";
	}
	push_output('string');
});
sgml('</TERM>', sub { 
	my $term = pop_output();
	$term =~ tr/\n/ /;
	output "$term\n";
	$newline_last = 1;
});

sgml('<LISTITEM>', sub {
	texi_output "\n\n\@item\n" unless $_[0]->in('VARLISTENTRY');
	$_[0]->ext->{'separator'} = 'none';
	$separator = 'none';
});

sgml('<SIMPLELIST>', sub { 
	$_[0]->ext->{'first_member'} = 1;
});

sgml('<MEMBER>', sub {
	my $listtype = $_[0]->parent->attribute(TYPE)->value;
	
	if($listtype =~ /Inline/i) {
		if($_[0]->parent->ext->{'first_member'}) {
			# If this is the first member don't put any commas
			$_[0]->parent->ext->{'first_member'} = 0;
		} else {
			output ", ";
		}
	} elsif($listtype =~ /Vert/i) {
		texi_output("\n\n");
	}
});







########################################################################
#
# Linkage, cross references
#
########################################################################

sgml('<ULINK>', sub {
	if($skip_inline++) { return; }	# hopefully doesn't happen
	texi_output '@uref{', $_[0]->attribute('URL')->value, ', '
});
sgml('</ULINK>', sub {
	texi_output '}' unless --$skip_inline;
});

sgml('<XREF>', sub {
	my $id = $_[0]->attribute('LINKEND')->value;
	my ($nodename) = split(/,/, $Refs->get($id));

        if($nodename) {
		texi_output("\@xref\{$nodename\}");
	} else {
		$blank_xrefs++;
		texi_output "[XRef to $id]";
        }
});

# Anchor




########################################################################
#
# SDATA 
#
########################################################################

texi_sgml('|[lt    ]|', '<');
texi_sgml('|[gt    ]|', '>');
texi_sgml('|[amp   ]|', '&');
texi_sgml('|[ndash ]|', '--');
texi_sgml('|[mdash ]|', '---');

sgml('sdata',sub { texi_output "|[", $_[0], "]|"; });

#
# Default handlers (uncomment these if needed).  Right now, these are set
# up to gag on any unrecognised elements, sdata, processing-instructions,
# or entities.
#
# sgml('start_element',sub { die "Unknown element: " . $_[0]->name; });
# sgml('end_element','');

sgml('cdata', sub
{ 
	if($raw_cdata) { output $_[0]; return; }
	
	if($separator eq 'full') {
		output "\n" unless $newline_last++;
		output "\@noindent\n";
		$separator = '';
	}

	for($_[0]) {
		s/\@/\@\@/g;
		s/\{/\@\{/g;
		s/\}/\@\}/g;
	
               	# TeX is smart enough to collapse spaces, but not info.
		#  ... At least they document this braindamage ...
        	if(!$nocollapse_whitespace) {
			$_[0] =~ tr/\t / /s;
	                $_[0] =~ s/^ // if $newline_last;
		}
	}

	$newline_last = 0;
	output $_[0];
});

sgml('re', sub
{
	if($nocollapse_whitespace || !$newline_last) {
		output "\n";
	}

	$newline_last = 1;
});

sgml('pi',sub{});
sgml('entity',sub { die "Unknown external entity: " . $_[0]->name; });
sgml('start_subdoc',sub { die "Unknown subdoc entity: " . $_[0]->name; });
sgml('end_subdoc',sub{});
# sgml('conforming',sub{});
1;

