#ifndef __aspell_data_hh_
#define __aspell_data_hh_

#include <string>

#include "emulation.hh"
#include "copy_ptr.hh"

namespace aspell {

  using namespace autil;

  class Manager;
  class Language;
  class Config;

  class DataSet {
    friend Manager;
  private:
    CopyPtr<const Language> lang_;
    int                     attach_count_;
    string                  name_;
  private:
    void attach(const Language &) ;
    void detach();
  public:
    //this is here because dynamic_cast in gcc 2.95.1 took too dam long
    enum BasicType {no_type, basic_word_set, basic_replacement_set};
    BasicType basic_type;

    DataSet();
    virtual ~DataSet();
    const char * name() const {return name_.c_str();}
    void check_lang(const string& lang);
    void set_check_lang(const string & lang, const Config *);
    const Language * lang() const {return lang_.ptr;};
    const char * lang_name() const;
    bool is_attached () const ;
  };

  class WritableDataSet {
  public:
    virtual void merge(const string &) = 0;
    virtual void synchronize() = 0;
    virtual void save_noupdate() = 0;
    virtual void save_as(const string &) = 0;
    virtual void clear() = 0;
  };

  struct SoundslikeWord {
    const char * soundslike;
    const void * word_list_pointer;

    operator bool () const {return soundslike;}
  
    SoundslikeWord() : soundslike(0) {}
    SoundslikeWord(const char * w, const void * p) 
      : soundslike(w), word_list_pointer(p) {}
  };

  class BasicWordSet : public DataSet
  {
  public:
    BasicWordSet() {
      basic_type =  basic_word_set;
    }
    
    typedef Emulation<const char *>       Emul;
    typedef VirEmulation<const char *>    VirEmul;
    typedef const char *                  Value;
    typedef unsigned int                  Size;
    typedef SoundslikeWord                SoundslikeValue;
    typedef Emulation<SoundslikeWord>     SoundslikeEmul;
    typedef VirEmulation<SoundslikeWord>  VirSoundslikeEmul;

    virtual void load(const string &, const Config *) = 0;
    virtual const char * file_name() const = 0;

    virtual VirEmul * elements() const = 0;
    virtual Size   size()     const = 0;
    virtual bool   empty()    const {return !size();}
  
    virtual bool operator [] (const char * word) const = 0;
    virtual bool operator [] (const string & word) const = 0;

    virtual VirEmul * words_w_soundslike(const char * sondslike) const = 0;
    virtual VirEmul * words_w_soundslike(SoundslikeWord soundslike) const = 0;
    
    virtual VirSoundslikeEmul * soundslike_elements() const = 0;
  };

  class WritableWordSet : public BasicWordSet,
			  public WritableDataSet
  {
  public:
    virtual void add(const string &w) = 0;
    virtual void add(const string &w, const string &s) = 0;
  };

  struct ReplacementList {
    typedef Emulation<const char *>    Emul;
    typedef const char *               Value;
    typedef VirEmulation<const char *> VirEmul;

    const char *  misspelled_word;
    VirEmul    *  elements; // you are responable for freeing this with delete
    bool empty() const {return elements == 0;}

    ReplacementList()
      : elements(0) {}
    ReplacementList(const char * w, VirEmul * els)
      : misspelled_word(w), elements(els) {}
  };

  class BasicReplacementSet : public DataSet
  {
  public:
    BasicReplacementSet() {
      basic_type = basic_replacement_set;
    }
    
    typedef Emulation<ReplacementList>    Emul;
    typedef VirEmulation<ReplacementList> VirEmul;
    typedef const char *                  Value;
    typedef unsigned int                  Size;
    typedef SoundslikeWord                SoundslikeValue;
    typedef Emulation<SoundslikeWord>     SoundslikeEmul;
    typedef VirEmulation<SoundslikeWord>  VirSoundslikeEmul;

    virtual void load(const string &, const Config *) = 0;
    virtual const char * file_name() const = 0;

    virtual VirEmul * elements() const = 0;
    virtual Size   size()     const = 0;
    virtual bool   empty()    const {return !size();}

    virtual VirEmul * repls_w_soundslike(const char * soundslike) const = 0;
    virtual VirEmul * repls_w_soundslike(SoundslikeWord soundslike) const = 0;
    
    virtual VirSoundslikeEmul * soundslike_elements() const = 0;
  };

  class WritableReplacementSet : public BasicReplacementSet,
				 public WritableDataSet
  {
  public:
    virtual void add(const string &mis, const string &cor) = 0;
    virtual void add(const string &mis, const string &cor, const string &s) = 0;
  };
  
  // implemented in readonly_ws.cc
  BasicWordSet * new_default_readonly_word_set();
  void create_default_readonly_word_set(VirEmulation<const char *> * els,
					const Config & config);

  // implemented in writable_ws.cc
  WritableWordSet * new_default_writable_word_set();

  // implemented in writable_repl.cc
  WritableReplacementSet * new_default_writable_replacement_set();


}

#endif

