/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-1999  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "xmms.h"
#include <gdk/gdkprivate.h>
#include <X11/Xlib.h>
#include <sys/ipc.h>
#include <ctype.h>

int find_file(char *dest, const char *dirname, const char *file)
{
	DIR *dir;
	struct dirent *dirent;

	if ((dir = opendir(dirname)) != NULL)
	{
		while ((dirent = readdir(dir)) != NULL)
		{
			if (!strcasecmp(dirent->d_name, file))
			{
				sprintf(dest, "%s/%s", dirname, dirent->d_name);
				closedir(dir);
				return 1;
			}
		}
		closedir(dir);
	}
	return 0;
}

/* known bug in libc6:
   Field d_type is not implemented as of libc6 2.0.4 and will
   always return 0 (unknown).
   this means: find out file-type using low-level-functions */
/* find_file_recursively() by Jrg Schuler Wed, 17 Feb 1999 23:50:52 +0100
   Placed under GPL version 2 or (at your option) any later version */
/* find <file> in directory <dirname> or subdirectories.
   return pointer to complete filename which has to be freed by
   calling "free()" after use. Returns NULL if file could not be found. */
gchar *find_file_recursively(const char *dirname, const char *file)
{
	DIR *dir;
	struct dirent *dirent;
	gchar *result;
	gchar *found;
	struct stat buf;

	if ((dir = opendir(dirname)) != NULL)
	{
		while ((dirent = readdir(dir)) != NULL)
		{
			/* Don't scan "." and ".." */
			if (strcmp(dirent->d_name, ".") == 0 || strcmp(dirent->d_name, "..") == 0)
				continue;
			/* We need this in order to find out if file is directory */
			found = g_strdup_printf("%s/%s", dirname, dirent->d_name);
			if (stat(found, &buf) != 0)
			{	/* Error occured */
				g_free(found);
				closedir(dir);
				return NULL;
			}
			if (S_ISDIR(buf.st_mode))
			{	/* Found directory -- descend into it */
				result = find_file_recursively(found, file);
				if (result != NULL)
				{	/* Found file there -- exit */
					g_free(found);
					closedir(dir);
					return result;
				}
			}
			else
			{	/* Normal file -- maybe just what we are looking for? */
				if (!strcasecmp(dirent->d_name, file))
				{	/* Yes! */
					if (strlen(found) > FILENAME_MAX)
					{	/* No good! File + path too long */
						g_free(found);
						closedir(dir);
						return NULL;
					}
					else
					{
						closedir(dir);
						return found;
					}
				}
			}
			g_free(found);
		}
		closedir(dir);
	}
	return NULL;
}

void del_directory(const char *dirname)
{
	DIR *dir;
	struct dirent *dirent;
	char *file;

	if ((dir = opendir(dirname)) != NULL)
	{
		while ((dirent = readdir(dir)) != NULL)
		{
			if (strcmp(dirent->d_name, ".") && strcmp(dirent->d_name, ".."))
			{
				file = (char *) g_malloc(strlen(dirname) + strlen(dirent->d_name) + 2);
				sprintf(file, "%s/%s", dirname, dirent->d_name);
				if (unlink(file) == -1)
					if (errno == EISDIR)
						del_directory(file);
				g_free(file);
			}
		}
		closedir(dir);
	}
	rmdir(dirname);
}

GdkImage *create_dblsize_image(GdkImage * img)
{
	GdkImage *dblimg;
	register guint x, y;

	/*
	 * This needs to be optimized
	 */

	dblimg = gdk_image_new(GDK_IMAGE_NORMAL, gdk_visual_get_best(), img->width << 1, img->height << 1);
	if (dblimg->bpp == 1)
	{
		register guint8 *srcptr, *ptr, *ptr2, pix;

		srcptr = ((GdkImagePrivate *) img)->ximage->data;
		ptr = ((GdkImagePrivate *) dblimg)->ximage->data;
		ptr2 = ptr + dblimg->bpl;

		for (y = 0; y < img->height; y++)
		{
			for (x = 0; x < img->width; x++)
			{
				pix = *srcptr++;
				*ptr++ = pix;
				*ptr++ = pix;
				*ptr2++ = pix;
				*ptr2++ = pix;
			}
			srcptr += img->bpl - img->width;
			ptr += (dblimg->bpl << 1) - dblimg->width;
			ptr2 += (dblimg->bpl << 1) - dblimg->width;
		}
	}
	if (dblimg->bpp == 2)
	{
		guint16 *srcptr, *ptr, *ptr2, pix;

		srcptr = (guint16 *) ((GdkImagePrivate *) img)->ximage->data;
		ptr = (guint16 *) ((GdkImagePrivate *) dblimg)->ximage->data;
		ptr2 = ptr + (dblimg->bpl >> 1);

		for (y = 0; y < img->height; y++)
		{
			for (x = 0; x < img->width; x++)
			{
				pix = *srcptr++;
				*ptr++ = pix;
				*ptr++ = pix;
				*ptr2++ = pix;
				*ptr2++ = pix;
			}
			srcptr += (img->bpl >> 1) - img->width;
			ptr += (dblimg->bpl) - dblimg->width;
			ptr2 += (dblimg->bpl) - dblimg->width;
		}
	}
	if (dblimg->bpp == 3)
	{
		register guint8 *srcptr, *ptr, *ptr2, pix1, pix2, pix3;

		srcptr = ((GdkImagePrivate *) img)->ximage->data;
		ptr = ((GdkImagePrivate *) dblimg)->ximage->data;
		ptr2 = ptr + dblimg->bpl;

		for (y = 0; y < img->height; y++)
		{
			for (x = 0; x < img->width; x++)
			{
				pix1 = *srcptr++;
				pix2 = *srcptr++;
				pix3 = *srcptr++;
				*ptr++ = pix1;
				*ptr++ = pix2;
				*ptr++ = pix3;
				*ptr++ = pix1;
				*ptr++ = pix2;
				*ptr++ = pix3;
				*ptr2++ = pix1;
				*ptr2++ = pix2;
				*ptr2++ = pix3;
				*ptr2++ = pix1;
				*ptr2++ = pix2;
				*ptr2++ = pix3;

			}
			srcptr += img->bpl - (img->width * 3);
			ptr += (dblimg->bpl << 1) - (dblimg->width * 3);
			ptr2 += (dblimg->bpl << 1) - (dblimg->width * 3);
		}
	}
	if (dblimg->bpp == 4)
	{
		register guint32 *srcptr, *ptr, *ptr2, pix;

		srcptr = (guint32 *) ((GdkImagePrivate *) img)->ximage->data;
		ptr = (guint32 *) ((GdkImagePrivate *) dblimg)->ximage->data;
		ptr2 = ptr + (dblimg->bpl >> 2);

		for (y = 0; y < img->height; y++)
		{
			for (x = 0; x < img->width; x++)
			{
				pix = *srcptr++;
				*ptr++ = pix;
				*ptr++ = pix;
				*ptr2++ = pix;
				*ptr2++ = pix;
			}
			srcptr += (img->bpl >> 2) - img->width;
			ptr += (dblimg->bpl >> 1) - dblimg->width;
			ptr2 += (dblimg->bpl >> 1) - dblimg->width;
		}
	}
	return dblimg;
}

char *read_ini_string(const char *filename, const char *section, const char *key)
{
	FILE *file;
	char *buffer, *ret_buffer = NULL;
	int found_section = 0, found_key = 0, off = 0, len = 0;
	struct stat statbuf;

	if (!filename)
		return NULL;

	if ((file = fopen(filename, "r")) != NULL)
	{
		stat(filename, &statbuf);
		buffer = (char *) g_malloc(statbuf.st_size);
		fread(buffer, 1, statbuf.st_size, file);
		while (!found_key && off < statbuf.st_size)
		{
			while ((buffer[off] == '\r' || buffer[off] == '\n' || buffer[off] == ' ' || buffer[off] == '\t') && off < statbuf.st_size)
				off++;
			if (off >= statbuf.st_size)
				break;
			if (buffer[off] == '[')
			{
				off++;
				if (off >= statbuf.st_size)
					break;
				if (off < statbuf.st_size - strlen(section))
				{
					if (!strncasecmp(section, &buffer[off], strlen(section)))
					{
						off += strlen(section);
						if (off >= statbuf.st_size)
							break;
						if (buffer[off] == ']')
							found_section = 1;
						else
							found_section = 0;
						off++;
						if (off >= statbuf.st_size)
							break;
					}
					else
						found_section = 0;
				}
				else
					found_section = 0;

			}
			else if (found_section)
			{
				if (off < statbuf.st_size - strlen(key))
				{
					if (!strncasecmp(key, &buffer[off], strlen(key)))
					{
						off += strlen(key);
						while ((buffer[off] == ' ' || buffer[off] == '\t') && off < statbuf.st_size)
							off++;
						if (off >= statbuf.st_size)
							break;
						if (buffer[off] == '=')
						{
							off++;
							while ((buffer[off] == ' ' || buffer[off] == '\t') && off < statbuf.st_size)
								off++;
							if (off >= statbuf.st_size)
								break;
							len = 0;
							while (buffer[off + len] != '\r' && buffer[off + len] != '\n' && buffer[off + len] != ';' && off + len < statbuf.st_size)
								len++;
							ret_buffer = (char *) g_malloc(len + 1);
							strncpy(ret_buffer, &buffer[off], len);
							ret_buffer[len] = '\0';
							off += len;
							found_key = 1;
						}
					}
				}
			}
			while (buffer[off] != '\r' && buffer[off] != '\n' && off < statbuf.st_size)
				off++;
		}
		g_free(buffer);
		fclose(file);
	}
	return ret_buffer;
}

GArray *string_to_garray(const gchar * str)
/* Converts a string of integers to a GArray   -havardk */
{
	GArray *array;
	gint temp;
	const gchar *ptr = str;
	gchar *endptr;

	array = g_array_new(FALSE, TRUE, sizeof (gint));
	for (;;)
	{
		temp = strtol(ptr, &endptr, 10);
		if (ptr == endptr)
			break;
		g_array_append_val(array, temp);
		ptr = endptr;
		while (!isdigit(*ptr) && (*ptr) != '\0')
			ptr++;
		if (*ptr == '\0')
			break;
	}
	return (array);
}

GArray *read_ini_array(const gchar * filename, const gchar * section, const gchar * key)
{
	gchar *temp;
	GArray *a;

	if ((temp = read_ini_string(filename, section, key)) == NULL)
		return NULL;
	a = string_to_garray(temp);
	g_free(temp);
	return a;
}

void glist_movedown(GList * list)
{
	gpointer temp;

	if (g_list_next(list))
	{
		temp = list->data;
		list->data = g_list_next(list)->data;
		g_list_next(list)->data = temp;
	}
}

void glist_moveup(GList * list)
{
	gpointer temp;

	if (g_list_previous(list))
	{
		temp = list->data;
		list->data = g_list_previous(list)->data;
		g_list_previous(list)->data = temp;

	}
}

/*
 * util_item_factory_popup() is a dropin replacement for gtk_item_factory_popup().
 * The difference is that the menu is always poped up whithin the screen.
 * Note that this means it does not neccesarily pop up at (x,y).
 */

void util_item_factory_popup(GtkItemFactory * ifactory, guint x, guint y, guint mouse_button, guint32 time)
{
	static GQuark quark_user_menu_pos = 0;
	struct Pos
	{
		gint x;
		gint y;
	}
	   *pos;

	if (!quark_user_menu_pos)
		quark_user_menu_pos = g_quark_from_static_string("user_menu_pos");

	pos = gtk_object_get_data_by_id(GTK_OBJECT(ifactory), quark_user_menu_pos);
	if (!pos)
	{
		pos = g_malloc0(sizeof (struct Pos));

		gtk_object_set_data_by_id_full(GTK_OBJECT(ifactory->widget), quark_user_menu_pos, pos, g_free);
	}
	pos->x = x;
	pos->y = y;

	gtk_menu_popup(GTK_MENU(ifactory->widget), NULL, NULL, NULL, pos, mouse_button, time);
}

GtkWidget* util_create_add_url_window(gchar *caption, GtkSignalFunc func)
{
	GtkWidget *win, *vbox, *bbox, *ok, *cancel, *entry;

	win = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_window_set_title(GTK_WINDOW(win), caption);
	gtk_window_set_position(GTK_WINDOW(win), GTK_WIN_POS_MOUSE);
	gtk_container_set_border_width(GTK_CONTAINER(win), 10);
	
	vbox = gtk_vbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(win), vbox);
	
	entry = gtk_entry_new();
	gtk_signal_connect(GTK_OBJECT(entry), "activate", func, entry);
	gtk_box_pack_start(GTK_BOX(vbox), entry, FALSE, FALSE, 0);
	gtk_window_set_focus(GTK_WINDOW(win), entry);
	gtk_widget_show(entry);
	
	bbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(bbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(bbox), 5);
	
	ok = gtk_button_new_with_label("Ok");
	gtk_signal_connect(GTK_OBJECT(ok), "clicked", func, entry);
	GTK_WIDGET_SET_FLAGS(ok, GTK_CAN_DEFAULT);
	gtk_window_set_default(GTK_WINDOW(win), ok);
	gtk_box_pack_start(GTK_BOX(bbox), ok, FALSE, FALSE, 0);
	gtk_widget_show(ok);
	
	cancel = gtk_button_new_with_label("Cancel");
	gtk_signal_connect_object(GTK_OBJECT(cancel), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(win));
	GTK_WIDGET_SET_FLAGS(cancel, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(bbox), cancel, FALSE, FALSE, 0);
	gtk_widget_show(cancel);
	
	gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 0);
	gtk_widget_show(bbox);
	gtk_widget_show(vbox);
	return win;
}
