/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-1999  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "xmms.h"

gint playlist_list_auto_drag_down_func(gpointer data)
{
	PlayList_List *pl = data;

	if (pl->pl_auto_drag_down)
	{
		playlist_list_move_down();
		pl->pl_first++;
		playlistwin_update_list();
		return TRUE;

	}
	return FALSE;
}

gint playlist_list_auto_drag_up_func(gpointer data)
{
	PlayList_List *pl = data;

	if (pl->pl_auto_drag_up)
	{
		playlist_list_move_up();
		pl->pl_first--;
		playlistwin_update_list();
		return TRUE;

	}
	return FALSE;
}

void playlist_list_move_up(void)
{
	GList *list;

	list = get_playlist();
	if (((PlaylistEntry *) list->data)->selected)
		return;
	while (list)
	{
		if (((PlaylistEntry *) list->data)->selected)
			glist_moveup(list);
		list = g_list_next(list);
	}
}

void playlist_list_move_down(void)
{
	GList *list;

	list = g_list_last(get_playlist());
	if (((PlaylistEntry *) list->data)->selected)
		return;
	while (list)
	{
		if (((PlaylistEntry *) list->data)->selected)
			glist_movedown(list);
		list = g_list_previous(list);
	}
}

void playlist_list_select_range(gint min, gint max, gboolean sel)
{
	GList *list;
	PlaylistEntry *entry;
	int i;

	if (min > max)
	{
		i = min;
		min = max;
		max = i;
	}

	list = g_list_nth(get_playlist(), min);
	for (i = min; i <= max && list; i++)
	{
		entry = (PlaylistEntry *) list->data;
		entry->selected = sel;
		list = list->next;
	}
}

void playlist_list_button_press_cb(GtkWidget * widget, GdkEventButton * event, PlayList_List * pl)
{
	GList *list;
	PlaylistEntry *entry;
	gint nr, y;

	if (get_playlist() && inside_widget(event->x, event->y, &pl->pl_widget))
	{
		if (event->button == 1)
		{
			y = event->y - pl->pl_widget.y;
			nr = (y / pl->pl_fheight) + pl->pl_first;
			if (nr >= get_playlist_length())
				nr = get_playlist_length() - 1;
			if (!(event->state & GDK_CONTROL_MASK))
			{
				list = get_playlist();
				while (list)
				{
					entry = (PlaylistEntry *) list->data;
					entry->selected = FALSE;
					list = list->next;
				}
			}

			if (event->state & GDK_SHIFT_MASK && pl->pl_prev_selected != -1)
			{
				if (pl->pl_prev_min != -1)
					playlist_list_select_range(pl->pl_prev_min, pl->pl_prev_max, FALSE);
				playlist_list_select_range(pl->pl_prev_selected, nr, TRUE);
				pl->pl_prev_min = pl->pl_prev_selected;
				pl->pl_prev_max = nr;
				pl->pl_drag_pos = nr - pl->pl_first;
			}
			else
			{
				if ((list = g_list_nth(get_playlist(), nr)) != NULL)
				{
					entry = list->data;
					entry->selected = !entry->selected;
					pl->pl_prev_selected = nr;
					pl->pl_drag_pos = nr - pl->pl_first;
					pl->pl_prev_min = -1;
				}
			}
			if (event->type == GDK_2BUTTON_PRESS)
			{
				playlist_set_position(nr);
				if (!get_input_playing())
					playlist_play();
			}
			pl->pl_dragging = TRUE;
			playlistwin_update_list();
		}
	}
}

void playlist_list_motion_cb(GtkWidget * widget, GdkEventMotion * event, PlayList_List * pl)
{
	gint nr, y, off, i;

	if (pl->pl_dragging)
	{
		y = event->y - pl->pl_widget.y;
		nr = (y / pl->pl_fheight);
		if (nr < 0)
		{
			nr = 0;
			if (!pl->pl_auto_drag_up)
			{
				pl->pl_auto_drag_up = TRUE;
				pl->pl_auto_drag_up_tag = gtk_timeout_add(100, playlist_list_auto_drag_up_func, pl);
			}
		}
		else if (pl->pl_auto_drag_up)
			pl->pl_auto_drag_up = FALSE;

		if (nr >= pl->pl_num_visible)
		{
			nr = pl->pl_num_visible;
			if (!pl->pl_auto_drag_down)
			{
				pl->pl_auto_drag_down = TRUE;
				pl->pl_auto_drag_down_tag = gtk_timeout_add(100, playlist_list_auto_drag_down_func, pl);
				pl->pl_first++;
			}
		}
		else if (pl->pl_auto_drag_down)
			pl->pl_auto_drag_down = FALSE;

		off = nr - pl->pl_drag_pos;
		if (off)
		{
			for (i = 0; i < abs(off); i++)
			{
				if (off < 0)
					playlist_list_move_up();
				else
					playlist_list_move_down();

			}
			playlistwin_update_list();
		}
		pl->pl_drag_pos = nr;
	}
}

void playlist_list_button_release_cb(GtkWidget * widget, GdkEventButton * event, PlayList_List * pl)
{
	pl->pl_dragging = FALSE;
	pl->pl_auto_drag_down = FALSE;
	pl->pl_auto_drag_up = FALSE;
}

void playlist_list_draw(Widget * w)
{
	PlayList_List *pl = (PlayList_List *) w;
	GdkGC *gc;
	GdkPixmap *obj;
	GdkFont *font = NULL;
	gint width, height;
	GList *list;
	PlaylistEntry *entry;
	gchar *text, *title, length[20], *tmp, *tmp2;
	gint i, len, tw, max_first;

	gc = pl->pl_widget.gc;
	width = pl->pl_widget.width;
	height = pl->pl_widget.height;

	obj = pl->pl_widget.parent;

	gdk_gc_set_foreground(gc, get_skin_color(SKIN_PLEDIT_NORMALBG));
	gdk_draw_rectangle(obj, gc, TRUE, pl->pl_widget.x, pl->pl_widget.y, width, height);

	if ((font = gdk_font_load(cfg.playlist_font)) != NULL)
	{
		list = get_playlist();
		pl->pl_fheight = font->ascent + font->descent + 1;
		pl->pl_num_visible = height / pl->pl_fheight;

		max_first = (g_list_length(list) - pl->pl_num_visible);
		if (max_first < 0)
			max_first = 0;
		if (pl->pl_first >= max_first)
			pl->pl_first = max_first;
		if (pl->pl_first < 0)
			pl->pl_first = 0;
		for (i = 0; i < pl->pl_first; i++)
			list = g_list_next(list);
		i = pl->pl_first;

		while (list && i < pl->pl_first + pl->pl_num_visible)
		{
			entry = (PlaylistEntry *) list->data;
			if (entry->selected)
			{
				gdk_gc_set_foreground(gc, get_skin_color(SKIN_PLEDIT_SELECTEDBG));
				gdk_draw_rectangle(obj, gc, TRUE, pl->pl_widget.x, pl->pl_widget.y + ((i - pl->pl_first) * pl->pl_fheight), width, pl->pl_fheight);
			}
			if (i == get_playlist_position())
				gdk_gc_set_foreground(gc, get_skin_color(SKIN_PLEDIT_CURRENT));
			else
				gdk_gc_set_foreground(gc, get_skin_color(SKIN_PLEDIT_NORMAL));

			title = entry->title;
			if (!title)
			{
				title = strrchr(entry->filename, '/');
				if (!title)
					title = entry->filename;
				else
					title++;
			}

			if (entry->length != -1)
			{
				sprintf(length, "%d:%-2.2d", entry->length / 60000, (entry->length / 1000) % 60);
				gdk_draw_text(obj, font, gc, pl->pl_widget.x + width - gdk_text_width(font, length, strlen(length)) - 2, pl->pl_widget.y + (i - pl->pl_first) * pl->pl_fheight + font->ascent, length, strlen(length));
				tw = width - gdk_text_width(font, length, strlen(length)) - 5;
			}
			else
			{
				strcpy(length, "");
				tw = width;
			}
			text = g_malloc(strlen(title) + 20);
			if (cfg.show_numbers_in_pl)
				sprintf(text, "%d. %s", i + 1, title);
			else
				sprintf(text, "%s", title);
			if (cfg.convert_underscore)
				while ((tmp = strchr(text, '_')) != NULL)
					*tmp = ' ';
			if (cfg.convert_twenty)
				while ((tmp = strstr(text, "%20")) != NULL)
				{
					tmp2 = tmp + 3;
					*(tmp++) = ' ';
					while (*tmp2)
						*(tmp++) = *(tmp2++);
					*tmp = '\0';
				}
			len = strlen(text);
			while (gdk_text_width(font, text, len) > tw && len > 4)
			{
				len--;
				text[len - 3] = '.';
				text[len - 2] = '.';
				text[len - 1] = '.';
				text[len] = '\0';
			}

			gdk_draw_text(obj, font, gc, pl->pl_widget.x, pl->pl_widget.y + (i - pl->pl_first) * pl->pl_fheight + font->ascent, text, len);
			g_free(text);
			list = list->next;
			i++;
		}
		gdk_font_unref(font);
	}
	else
	{
		fprintf(stderr, "CRITICAL: Couldn't open playlist font\n");
	}
}

PlayList_List *create_playlist_list(GList ** wlist, GdkPixmap * parent, GdkGC * gc, gint x, gint y, gint w, gint h)
{
	PlayList_List *pl;

	pl = (PlayList_List *) g_malloc0(sizeof (PlayList_List));
	pl->pl_widget.parent = parent;
	pl->pl_widget.gc = gc;
	pl->pl_widget.x = x;
	pl->pl_widget.y = y;
	pl->pl_widget.width = w;
	pl->pl_widget.height = h;
	pl->pl_widget.visible = TRUE;
	pl->pl_widget.button_press_cb = GTK_SIGNAL_FUNC(playlist_list_button_press_cb);
	pl->pl_widget.button_release_cb = GTK_SIGNAL_FUNC(playlist_list_button_release_cb);
	pl->pl_widget.motion_cb = GTK_SIGNAL_FUNC(playlist_list_motion_cb);
	pl->pl_widget.draw = playlist_list_draw;
	pl->pl_prev_selected = -1;
	pl->pl_prev_min = -1;
	pl->pl_prev_max = -1;
	add_widget(wlist, pl);
	return pl;
}
