
/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-1999  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <netdb.h>
#include <glib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <stdio.h>

#include <pthread.h>

#include "mpg123.h"
#include "config.h"
#include "libxmms/util.h"

#define min(x,y) ((x)<(y)?(x):(y))
#define min3(x,y,z) (min(x,y)<(z)?min(x,y):(z))
#define min4(x,y,z,w) (min3(x,y,z)<(w)?min3(x,y,z):(w))

static gchar *icy_name = NULL;
static gint icy_metaint = 0;

extern gint mpg123_bitrate, mpg123_frequency, mpg123_stereo;
extern gboolean mpg123_stereo;

static gboolean prebuffering, going, eof = FALSE;
static gint sock, rd_index, wr_index, buffer_length, prebuffer_length;
static guint64 buffer_read = 0;
static gchar *buffer;
static pthread_t thread;
static GtkWidget *error_dialog = NULL;

static FILE *output_file = NULL;

#define BASE64_LENGTH(len) (4 * (((len) + 2) / 3))

/* Encode the string S of length LENGTH to base64 format and place it
   to STORE.  STORE will be 0-terminated, and must point to a writable
   buffer of at least 1+BASE64_LENGTH(length) bytes.  */
static void base64_encode (const gchar *s, gchar *store, gint length)
{
	/* Conversion table.  */
	static gchar tbl[64] = {
		'A','B','C','D','E','F','G','H',
		'I','J','K','L','M','N','O','P',
		'Q','R','S','T','U','V','W','X',
		'Y','Z','a','b','c','d','e','f',
		'g','h','i','j','k','l','m','n',
		'o','p','q','r','s','t','u','v',
		'w','x','y','z','0','1','2','3',
		'4','5','6','7','8','9','+','/'
	};
	gint i;
	guchar *p = (guchar *)store;

	/* Transform the 3x8 bits to 4x6 bits, as required by base64.  */
	for (i = 0; i < length; i += 3)
	{
		*p++ = tbl[s[0] >> 2];
		*p++ = tbl[((s[0] & 3) << 4) + (s[1] >> 4)];
		*p++ = tbl[((s[1] & 0xf) << 2) + (s[2] >> 6)];
		*p++ = tbl[s[2] & 0x3f];
		s += 3;
	}
	/* Pad the result if necessary...  */
	if (i == length + 1)
		*(p - 1) = '=';
	else if (i == length + 2)
		*(p - 1) = *(p - 2) = '=';
	/* ...and zero-terminate it.  */
	*p = '\0';
}

/* Create the authentication header contents for the `Basic' scheme.
   This is done by encoding the string `USER:PASS' in base64 and
   prepending `HEADER: Basic ' to it.  */
static gchar *basic_authentication_encode (const gchar *user, const gchar *passwd, const gchar *header)
{
	gchar *t1, *t2, *res;
	gint len1 = strlen (user) + 1 + strlen (passwd);
	gint len2 = BASE64_LENGTH (len1);

	t1 = g_strdup_printf("%s:%s", user, passwd);
	t2 = g_malloc0(len2 + 1);
	base64_encode (t1, t2, len1);
	res = g_strdup_printf("%s: Basic %s\r\n", header, t2);
	g_free(t2);
	g_free(t1);
	
	return res;
}

static void parse_url(gchar * url, gchar ** user, gchar ** pass, gchar ** host, int *port, gchar ** filename)
{
	gchar *h, *p, *pt, *f;

	if (!strncasecmp("http://", url, 7))
		url += 7;
	if ((h = strchr(url, '@')))
	{
		*h = '\0';
		if ((p = strchr(url, ':')))
		{
			*p = '\0';
			p++;
			*pass = g_strdup(p);
		}
		else
			*pass = NULL;
		*user = g_strdup(url);
		h++;
		url = h;
	}
	else
		h = url;
	if ((pt = strchr(url, ':')))
	{
		*pt = '\0';
		pt++;
		if ((f = strchr(pt, '/')))
		{
			*f = '\0';
			f++;
		}
		else
			f = NULL;
		*port = atoi(pt);
	}
	else
	{
		*port = 80;
		if ((f = strchr(h, '/')))
		{
			*f = '\0';
			f++;
		}
		else
			f = NULL;
	}
	*host = g_strdup(h);
	if (f)
		*filename = g_strdup(f);
	else
		*filename = NULL;
}

void mpg123_http_close(void)
{
	going = FALSE;

	pthread_join(thread, NULL);
	if (icy_name)
		g_free(icy_name);
	icy_name = NULL;
}


static gint http_used(void)
{
	if (wr_index >= rd_index)
		return wr_index - rd_index;
	return buffer_length - (rd_index - wr_index);
}

static gint http_free(void)
{
	if (rd_index > wr_index)
		return (rd_index - wr_index) - 1;
	return (buffer_length - (wr_index - rd_index)) - 1;
}

static void http_wait_for_data(gint bytes)
{
	while ((prebuffering || http_used() < bytes) && !eof && going && mpg123_info->going)
		xmms_usleep(10000);
}

static void show_error_message(gchar *error)
{
	if(!error_dialog)
	{
		GDK_THREADS_ENTER();
		error_dialog = xmms_show_message("Error", error, "Ok", FALSE,
						 NULL, NULL);
		gtk_signal_connect(GTK_OBJECT(error_dialog),
				   "destroy",
				   GTK_SIGNAL_FUNC(gtk_widget_destroyed),
				   &error_dialog);
		GDK_THREADS_LEAVE();
	}
}

int mpg123_http_read(gpointer data, gint length)
{
	gint len, cnt, off = 0, meta_len, meta_off = 0, i;
	gchar *meta_data, **tags, *temp, *title;

	http_wait_for_data(length);

	if (!going && !mpg123_info->going)
		return 0;
	len = min(http_used(), length);

	while (len && http_used())
	{
		if ((mpg123_cfg.cast_title_streaming) && (icy_metaint > 0) && (buffer_read % icy_metaint) == 0 && (buffer_read > 0))
		{
			meta_len = *((guchar *) buffer + rd_index) * 16;
			rd_index = (rd_index + 1) % buffer_length;
			if (meta_len > 0)
			{
				http_wait_for_data(meta_len);
				meta_data = g_malloc0(meta_len);
				if (http_used() >= meta_len)
				{
					while (meta_len)
					{
						cnt = min(meta_len, buffer_length - rd_index);
						memcpy(meta_data + meta_off, buffer + rd_index, cnt);
						rd_index = (rd_index + cnt) % buffer_length;
						meta_len -= cnt;
						meta_off += cnt;
					}
					tags = g_strsplit(meta_data, "';", 0);

					for (i = 0; tags[i]; i++)
					{
						if (!strncasecmp(tags[i], "StreamTitle=", 12))
						{
							temp = g_strdup(tags[i] + 13);
							title = g_strdup_printf("%s (%s)", temp, icy_name);
							mpg123_ip.set_info(title, -1, mpg123_bitrate * 1000, mpg123_frequency, mpg123_stereo);
							g_free(title);
						}

					}
					g_strfreev(tags);

				}
				g_free(meta_data);
			}
			cnt = min3(len, buffer_length - rd_index, http_used());
		}
		else if ((icy_metaint > 0) && (mpg123_cfg.cast_title_streaming))
			cnt = min4(len, buffer_length - rd_index, http_used(), icy_metaint - (buffer_read % icy_metaint));
		else
			cnt = min3(len, buffer_length - rd_index, http_used());
		if (output_file)
			fwrite(buffer + rd_index, 1, cnt, output_file);

		memcpy(data + off, buffer + rd_index, cnt);
		rd_index = (rd_index + cnt) % buffer_length;
		buffer_read += cnt;
		len -= cnt;
		off += cnt;
	}
	return off;
}

static gboolean http_check_for_data(void)
{

	fd_set set;
	struct timeval tv;
	gint ret;

	tv.tv_sec = 0;
	tv.tv_usec = 20000;
	FD_ZERO(&set);
	FD_SET(sock, &set);
	ret = select(sock + 1, &set, NULL, NULL, &tv);
	if (ret > 0)
		return TRUE;
	return FALSE;
}

gint mpg123_http_read_line(gchar * buf, gint size)
{
	gint i = 0;

	while (going && i < size - 1)
	{
		if (http_check_for_data())
		{
			if (read(sock, buf + i, 1) <= 0)
				return -1;
			if (buf[i] == '\n')
				break;
			if (buf[i] != '\r')
				i++;
		}
	}
	if (!going)
		return -1;
	buf[i] = '\0';
	return i;
}

void *http_buffer_loop(void *arg)
{
	gchar line[1024], *user = NULL, *pass = NULL, *host = NULL, *filename = NULL,
	     *status, *url = arg, *temp, *file, *output_name;
	gchar *chost;
	gint cnt, written, error, err_len, port, cport;
	fd_set set;
	struct hostent *hp;
	struct sockaddr_in address;
	struct timeval tv;

	url = g_strstrip(url);

	temp = g_strdup(url);
	parse_url(url, &user, &pass, &host, &port, &filename);
	g_free(url);

	if ((!filename || !*filename) && temp[strlen(temp) - 1] != '/')
		url = g_strconcat(temp, "/", NULL);
	else
		url = g_strdup(temp);
	g_free(temp);

	chost = mpg123_cfg.use_proxy ? mpg123_cfg.proxy_host : host;
	cport = mpg123_cfg.use_proxy ? mpg123_cfg.proxy_port : port;

	sock = socket(AF_INET, SOCK_STREAM, 0);
	fcntl(sock, F_SETFL, O_NONBLOCK);
	address.sin_family = AF_INET;

	status = g_strdup_printf("LOOKING UP %s", chost);
	mpg123_ip.set_info_text(status);
	g_free(status);

	if (!(hp = gethostbyname(chost)))
	{
		status = g_strdup_printf("Couldn't look up host %s", chost);
		show_error_message(status);
		g_free(status);

		mpg123_ip.set_info_text(NULL);
		eof = TRUE;
	}

	if (!eof)
	{
		memcpy(&address.sin_addr.s_addr, *(hp->h_addr_list), sizeof (address.sin_addr.s_addr));
		address.sin_port = g_htons(cport);

		status = g_strdup_printf("CONNECTING TO %s:%d", chost, cport);
		mpg123_ip.set_info_text(status);
		g_free(status);
		if (connect(sock, (struct sockaddr *) &address, sizeof (struct sockaddr_in)) == -1)
		{
			if (errno != EINPROGRESS)
			{
				status = g_strdup_printf("Couldn't connect to host %s", chost);
				show_error_message(status);
				g_free(status);

				mpg123_ip.set_info_text(NULL);
				eof = TRUE;
			}
		}
		while (going)
		{
			tv.tv_sec = 0;
			tv.tv_usec = 10000;
			FD_ZERO(&set);
			FD_SET(sock, &set);
			if (select(sock + 1, NULL, &set, NULL, &tv) > 0)
			{
				err_len = sizeof (error);
				getsockopt(sock, SOL_SOCKET, SO_ERROR, &error, &err_len);
				if (error)
				{
					status = g_strdup_printf("Couldn't connect to host %s",
								 chost);
					show_error_message(status);
					g_free(status);
					
					mpg123_ip.set_info_text(NULL);
					eof = TRUE;
						
				}
				break;
			}
		}
		if (!eof)
		{
			gchar *auth = NULL, *proxy_auth = NULL;

			if(user && pass)
				auth = basic_authentication_encode(user, pass, "Authorization");

			if (mpg123_cfg.use_proxy)
			{
				file = g_strdup(url);
				if(mpg123_cfg.proxy_use_auth && mpg123_cfg.proxy_user && mpg123_cfg.proxy_pass)
				{
					printf("%s %s\n", mpg123_cfg.proxy_user, mpg123_cfg.proxy_pass);
					proxy_auth = basic_authentication_encode(mpg123_cfg.proxy_user,
										 mpg123_cfg.proxy_pass,
										 "Proxy-Authorization");
				}
			}
			else
				file = g_strconcat("/", filename, NULL);
			temp = g_strdup_printf("GET %s HTTP/1.0\r\n\
Host: %s\r\n\
User-Agent: %s/%s\r\n\
%s%s%s\r\n",
					       file, host, PACKAGE, VERSION, 
					       proxy_auth ? proxy_auth : "", auth ? auth : "",
					       mpg123_cfg.cast_title_streaming ? "Icy-MetaData:1\r\n" : "");

			g_free(file);
			if(proxy_auth)
				g_free(proxy_auth);
			if(auth)
				g_free(auth);
			write(sock, temp, strlen(temp));
			g_free(temp);
			mpg123_ip.set_info_text("CONNECTED: WAITING FOR REPLY");
			while (going && !eof)
			{
				if (http_check_for_data())
				{
					if (mpg123_http_read_line(line, 1024))
					{
						status = strchr(line, ' ');
						if (status)
						{
							if (status[1] == '2')
								break;
							else
							{
								status = g_strdup_printf("Couldn't connect to host %s\nServer reported: %s", chost, status);
								show_error_message(status);
								g_free(status);
								break;
							}
						}
					}
					else
					{
						eof = TRUE;
						mpg123_ip.set_info_text(NULL);
					}
				}
			}

			while (going)
			{
				if (http_check_for_data())
				{
					if ((cnt = mpg123_http_read_line(line, 1024)) != -1)
					{
						if (!cnt)
							break;
						if (!strncmp(line, "icy-name:", 9))
							icy_name = g_strdup(line + 9);
						else if (!strncmp(line, "x-audiocast-name:", 17))
							icy_name = g_strdup(line + 17);
						if (!strncmp(line, "icy-metaint:", 12))
							icy_metaint = atoi(line + 12);
					}
					else
					{
						eof = TRUE;
						mpg123_ip.set_info_text(NULL);
						break;
					}
				}
			}
		}
	}
	if (mpg123_cfg.save_http_stream)
	{
		output_name = g_strdup(mpg123_http_get_title(url));
		file = output_name;
		if (!strncasecmp(output_name, "http://", 7))
			output_name += 7;
		temp = strrchr(output_name, '.');
		if (temp && !strcasecmp(temp, ".mp3"))
			*temp = '\0';

		while ((temp = strchr(output_name, '/')))
			*temp = '_';
		output_name = g_strdup_printf("%s/%s.mp3", mpg123_cfg.save_http_path, output_name);

		g_free(file);

		output_file = fopen(output_name, "wb");
		g_free(output_name);
	}

	while (going)
	{
		if (!http_used() && !mpg123_ip.output->buffer_playing())
			prebuffering = TRUE;
		if (http_free() > 0 && !eof)
		{
			if (http_check_for_data())
			{
				cnt = min(http_free(), buffer_length - wr_index);
				if (cnt > 1024)
					cnt = 1024;
				written = read(sock, buffer + wr_index, cnt);
				if (written <= 0)
				{
					eof = TRUE;
					if (prebuffering)
					{
						prebuffering = FALSE;

						mpg123_ip.set_info_text(NULL);
					}

				}
				else
					wr_index = (wr_index + written) % buffer_length;
			}

			if (prebuffering)
			{
				if (http_used() > prebuffer_length)
				{
					prebuffering = FALSE;
					mpg123_ip.set_info_text(NULL);
				}
				else
				{
					status = g_strdup_printf("PRE-BUFFERING: %dKB/%dKB", http_used() / 1024, prebuffer_length / 1024);
					mpg123_ip.set_info_text(status);
					g_free(status);
				}

			}
		}
		else
			xmms_usleep(10000);
	}
	if (output_file)
		fclose(output_file);
	close(sock);
	if (user)
		g_free(user);
	if (pass)
		g_free(pass);
	if (host)
		g_free(host);
	if (filename)
		g_free(filename);
	g_free(buffer);
	g_free(url);
	pthread_exit(NULL);
}

int mpg123_http_open(gchar * _url)
{
	gchar *url;

	url = g_strdup(_url);

	rd_index = 0;
	wr_index = 0;
	buffer_length = mpg123_cfg.http_buffer_size * 1024;
	prebuffer_length = (buffer_length * mpg123_cfg.http_prebuffer) / 100;
	buffer_read = 0;
	icy_metaint = 0;
	prebuffering = TRUE;
	going = TRUE;
	eof = FALSE;
	buffer = g_malloc(buffer_length);

	pthread_create(&thread, NULL, http_buffer_loop, url);

	return 0;
}

gchar *mpg123_http_get_title(gchar * url)
{
	if (icy_name)
		return g_strdup(icy_name);
	return g_strdup(url);
}
