#include "snd.h"

/* envelope editor and viewer */

static Widget enved_dialog = NULL;
static Widget mainform,applyB,apply2B,cancelB,drawer,colB,colF,showB,saveB,revertB,undoB,redoB,printB,brkptL,graphB,fltB,ampB,srcB,rbrow,clipB;
static Widget nameL,textL,screnvlst,screnvname,dBB,orderL;
static Widget expB,linB,lerow,baseScale,baseLabel,baseValue,baseSep,selectionB;
static GC gc,rgc,ggc;

static char *env_names[3] = {STR_amp_env_p,STR_flt_env_p,STR_src_env_p};

static int showing_all_envs = 0; /* edit one env (0), or view all currently defined envs (1) */
static int apply_to_selection = 0;

static int env_list_size = 0;    /* current size of env edits list */
static int env_list_top = 0;     /* one past current active position in list */
static env **env_list = NULL;    /* env edits list (for local undo/redo/revert) */

static env **all_envs = NULL;    /* all envs, either loaded or created in editor */
static char **all_names = NULL;  /* parallel names */
static int all_envs_size = 0;    /* size of this array */
static int all_envs_top = 0;     /* one past pointer to last entry in this array */

static int env_window_width = 0;
static int env_window_height = 0;

static int *current_xs = NULL;
static int *current_ys = NULL;
static int current_size = 0;

static chan_info *active_channel = NULL;

static env* selected_env = NULL; /* if during view, one env is clicked, it is "selected" and can be pasted elsewhere */
static env* active_env = NULL;   /* env currently being edited */

static void set_current_point(int pos, int x, int y)
{
  if (pos == current_size)
    {
      if (current_size == 0)
	{
	  current_size = 32;
	  current_xs = (int *)CALLOC(current_size,sizeof(int));
	  current_ys = (int *)CALLOC(current_size,sizeof(int));
	}
      else
	{
	  current_size += 32;
	  current_xs = (int *)REALLOC(current_xs,current_size * sizeof(int));
	  current_ys = (int *)REALLOC(current_ys,current_size * sizeof(int));
	}
    }
  current_xs[pos] = x;
  current_ys[pos] = y;
}

chan_info *new_env_axis(snd_state *ss, Widget grf)
{
  chan_info *acp;
  axis_info *ap;
  axis_context *ax;
  acp = (chan_info *)CALLOC(1,sizeof(chan_info));
  acp->s_type = CHAN_INFO;
  acp->printing = 0;
  acp->drawing = 1;
  acp->state = ss;
  ap = (axis_info *)CALLOC(1,sizeof(axis_info));
  ap->s_type = AXIS_INFO;
  acp->axis = ap;
  ax = (axis_context *)CALLOC(1,sizeof(axis_context));
  ap->ax = ax;
  ap->ss = ss;
  ap->cp = acp;
  ax->label_font = axis_label_FONTSTRUCT(ap);
  ax->numbers_font = axis_numbers_FONTSTRUCT(ap);
  ax->dp = XtDisplay(grf);
  ax->wn = XtWindow(grf);
  return(acp);
}

axis_info *new_wave_axis(snd_state *ss, Widget grf, axis_info *ap, GC ggc)
{
  axis_info *gap;
  axis_context *gray_ax;
  gap = (axis_info *)CALLOC(1,sizeof(axis_info));
  gap->s_type = AXIS_INFO;
  gray_ax = (axis_context *)CALLOC(1,sizeof(axis_context));
  gap->ax = gray_ax;
  gap->ss = ss;
  gray_ax->label_font = axis_label_FONTSTRUCT(ap);
  gray_ax->numbers_font = axis_numbers_FONTSTRUCT(ap);
  gray_ax->dp = XtDisplay(grf);
  gray_ax->wn = XtWindow(grf);
  gray_ax->gc = ggc;
  return(gap);
}

void init_env_axes(chan_info *acp, char *name, GC cur_gc, int ex0, int ey0, int width, int height, float xmin, float xmax, float ymin, float ymax)
{
  axis_info *ap;
  axis_context *ax;
  ap = acp->axis;
  ax = ap->ax;
  if (ap->xlabel) FREE(ap->xlabel);
  ax->gc = cur_gc;
  ap->xmin = xmin;
  ap->xmax = xmax;
  ap->ymin = ymin;
  ap->ymax = ymax;
  ap->xlabel = copy_string(name);
  ap->ylabel = NULL;
  ap->x0 = xmin;
  ap->x1 = xmax;
  ap->y0 = ymin;
  ap->y1 = ymax;
  ap->width = width;
  ap->window_width = width;
  ap->y_offset = ey0;
  ap->height = height;
  ap->graph_x0 = ex0;
  make_axes_1(acp, ap, X_IN_SECONDS, 1);
  /* if this is too small for an axis, it still sets up the fields needed for grf_x|y, so tiny envelope graphs will work */
}



static chan_info *axis_cp = NULL;
static axis_info *gray_ap = NULL;

static void make_axis_cp(snd_state *ss, char *name, GC cur_gc, int ex0, int ey0, int width, int height, float xmin, float xmax, float ymin, float ymax)
{
  /* conjure up minimal context for axis drawer in snd-axis.c */
  if (!axis_cp) axis_cp = new_env_axis(ss,drawer);
  if (!gray_ap) gray_ap = new_wave_axis(ss,drawer,axis_cp->axis,ggc);
  init_env_axes(axis_cp,name,cur_gc,ex0,ey0,width,height,xmin,xmax,ymin,ymax);
}

#define EXP_SEGLEN 4

static short grf_y_dB(snd_state *ss, float val, axis_info *ap)
{
  if (enved_dBing(ss))
    return(grf_y(dB(ss,val),ap));
  else return(grf_y(val,ap));
}

static double ungrf_y_dB(snd_state *ss, axis_info *ap, int y)
{
  if (enved_dBing(ss))
    return(un_dB(ss,ungrf_y(ap,y)));
  else return(ungrf_y(ap,y));
}

static void display_env(snd_state *ss, env *e, char *name, GC cur_gc, int x0, int y0, int width, int height, int dots)
{
  int i,j,k;
  float ex0,ey0,ex1,ey1,val,offset;
  int ix0,ix1,iy0,iy1,size,half,lx0,lx1,ly0,ly1;
  Display *dp;
  Drawable wn;
  float *efm,*ef;
  float scl,env_val,env_power,env_incr,logbase,curx,xincr;
  env *newe;
  int dur,env_ctr,pass;
  dp = XtDisplay(drawer);
  wn = XtWindow(drawer);
  if (e)
    {
      ex0 = e->data[0];
      ey0 = e->data[1];
      ex1 = e->data[(e->pts*2) - 2];
      ey1 = ey0;
      for (i=3;i<e->pts*2;i+=2)
	{
	  val = e->data[i];
	  if (ey0 > val) ey0 = val;
	  if (ey1 < val) ey1 = val;
	}
      if (ey0 > 0.0) ey0 = 0.0;
      if ((ey0 == ey1) && (ey1 == 0.0)) ey1 = 1.0; /* fixup degenerate case */
      if ((dots) && (ey1 < 1.0)) ey1 = 1.0;
    }
  else
    {
      ex0 = 0.0;
      ex1 = 1.0;
      ey0 = 0.0;
      ey1 = 1.0;
    }

  if (enved_dBing(ss)) {ey0 = ss->min_dB; ey1 = 0.0;}

  make_axis_cp(ss, name, cur_gc, x0, y0, width, height, ex0, ex1, ey0, ey1); 
  /* grf_x and grf_y (x|y, ap) can be used directly with XDrawLine */

  if (e)
    {
      ix1 = grf_x(e->data[0],axis_cp->axis);
      iy1 = grf_y_dB(ss,e->data[1],axis_cp->axis);
      if (dots)
	{
	  if (e->pts < 100)
	    size = ss->enved_point_size;
	  else size = (int)(ss->enved_point_size * 0.4);
	  half = (int)(size * 0.5);
	  set_current_point(0,ix1,iy1);
	  XFillArc(dp,wn,cur_gc,ix1-half,iy1-half,size,size,0,360*64);
	}
      if (e->base == 1.0)
	{
	  if (enved_dBing(ss))
	    {
	      for (j=1,i=2;i<e->pts*2;i+=2,j++)
		{
		  ix0 = ix1;
		  iy0 = iy1;
		  ix1 = grf_x(e->data[i],axis_cp->axis);
		  iy1 = grf_y_dB(ss,e->data[i+1],axis_cp->axis);
		  if (dots)
		    {
		      set_current_point(j,ix1,iy1);
		      XFillArc(dp,wn,cur_gc,ix1-half,iy1-half,size,size,0,360*64);
		    }
		  /* now try to fill in from the last point to this one */
		  if ((ix1-ix0) < (2*EXP_SEGLEN))
		    {
		      /* points are too close to be worth interpolating */
		      XDrawLine(dp,wn,cur_gc,ix0,iy0,ix1,iy1);
		    }
		  else
		    {
		      /* interpolate so the display looks closer to dB (we should use the env base...) */
		      dur = (ix1-ix0) / EXP_SEGLEN;
		      xincr = (e->data[i] - e->data[i-2]) / (float)dur;
		      curx = e->data[i-2] + xincr;
		      lx1 = ix0;
		      ly1 = iy0;
		      for (k=1;k<dur;k++,curx+=xincr)
			{
			  lx0 = lx1;
			  ly0 = ly1;
			  lx1 = grf_x(curx,axis_cp->axis);
			  val = list_interp(curx,e->data,e->pts);
			  ly1 = grf_y(dB(ss,val),axis_cp->axis);
			  XDrawLine(dp,wn,cur_gc,lx0,ly0,lx1,ly1);
			}
		      XDrawLine(dp,wn,cur_gc,lx1,ly1,ix1,iy1);
		    }
		}
	    }
	  else
	    {
	      for (j=1,i=2;i<e->pts*2;i+=2,j++)
		{
		  ix0 = ix1;
		  iy0 = iy1;
		  ix1 = grf_x(e->data[i],axis_cp->axis);
		  iy1 = grf_y(e->data[i+1],axis_cp->axis);
		  if (dots)
		    {
		      set_current_point(j,ix1,iy1);
		      XFillArc(dp,wn,cur_gc,ix1-half,iy1-half,size,size,0,360*64);
		    }
		  XDrawLine(dp,wn,cur_gc,ix0,iy0,ix1,iy1);
		  if (axis_cp->printing) ps_draw_line(axis_cp,ix0,iy0,ix1,iy1);
		}
	    }
	}
      else
	{
	  if (e->base <= 0.0)
	    {
	      for (j=1,i=2;i<e->pts*2;i+=2,j++)
		{
		  ix0 = ix1;
		  iy0 = iy1;
		  ix1 = grf_x(e->data[i],axis_cp->axis);
		  iy1 = grf_y_dB(ss,e->data[i+1],axis_cp->axis);
		  if (dots)
		    {
		      set_current_point(j,ix1,iy1);
		      XFillArc(dp,wn,cur_gc,ix1-half,iy1-half,size,size,0,360*64);
		    }
		  XDrawLine(dp,wn,cur_gc,ix0,iy0,ix1,iy0);
		  XDrawLine(dp,wn,cur_gc,ix1,iy0,ix1,iy1);
		  if (axis_cp->printing) 
		    {
		      ps_draw_line(axis_cp,ix0,iy0,ix1,iy0);
		      ps_draw_line(axis_cp,ix1,iy0,ix1,iy1);
		    }
		}
	    }
	  else
	    {
	      /* exponential case */
	      dur = width / EXP_SEGLEN;
	      if (dur < e->pts) dur = e->pts;
	      scl = 1.0;
	      offset = 0.0;
	      logbase = log(e->base);
	      ef = fixup_exp_env(e,&offset,&scl,e->base);
	      if (ef == NULL) {snd_error("%s[%d] %s: fixup exp env failed",__FILE__,__LINE__,__FUNCTION__); return;}
	      newe = make_envelope(ef,e->pts*2);
	      newe->base = 1.0; /* ? */
	      efm = magify_env(newe,dur,1.0);
	      env_power = newe->data[1];
	      env_val = offset + scl * (exp(logbase * env_power) - 1.0);
	      env_ctr = 0;
	      env_incr = efm[1];
	      pass = (int)(efm[0]);
	      ix1 = grf_x(0.0,axis_cp->axis);
	      iy1 = grf_y_dB(ss,env_val,axis_cp->axis);
	      xincr = (ex1-ex0) / (float)dur;
	      j=1;
	      for (i=1,curx=ex0;i<dur;i++,curx+=xincr)
		{
		  iy0 = iy1;
		  ix0 = ix1;
		  env_power += env_incr;
		  env_val = offset + scl * (exp(logbase * env_power) - 1.0);
		  ix1 = grf_x(curx,axis_cp->axis);
		  iy1 = grf_y_dB(ss,env_val,axis_cp->axis);
		  XDrawLine(dp,wn,cur_gc,ix0,iy0,ix1,iy1);
		  if (axis_cp->printing) ps_draw_line(axis_cp,ix0,iy0,ix1,iy1);
		  pass--;
		  if (pass <= 0)
		    {
		      if (dots)
			{
			  set_current_point(j++,ix1,iy1);
			  XFillArc(dp,wn,cur_gc,ix1-half,iy1-half,size,size,0,360*64);
			}
		      env_ctr += 2;
		      pass = (int)(efm[env_ctr]);
		      env_incr = efm[env_ctr+1];
		    }
		}
	      if (curx < ex1)
		{
		  iy0 = iy1;
		  ix0 = ix1;
		  ix1 = grf_x(ex1,axis_cp->axis);
		  iy1 = grf_y_dB(ss,e->data[e->pts*2 -1],axis_cp->axis);
		  XDrawLine(dp,wn,cur_gc,ix0,iy0,ix1,iy1);
		  if (axis_cp->printing) ps_draw_line(axis_cp,ix0,iy0,ix1,iy1);
		}
	      if (dots)
		{
		  set_current_point(j++,ix1,iy1);
		  XFillArc(dp,wn,cur_gc,ix1-half,iy1-half,size,size,0,360*64);
		}
	      if (ef) FREE(ef);
	      if (efm) FREE(efm);
	      if (newe) free_env(newe);
	    }
	}
    }
}

static void view_envs(snd_state *ss)
{
  /* divide space available into a grid (if needed) that shows all currently defined envelopes */
  /* I suppose if there were several hundred envelopes, we'd need a scrollable viewer... */
  int cols,rows,i,j,width,height,x,y,k;
  if (all_envs_top > 1)
    {
      cols = round(sqrt((float)(all_envs_top * env_window_width) / (float)env_window_height));
      rows = round((float)all_envs_top/(float)cols);
      if ((rows*cols) < all_envs_top) rows++;
    }
  else
    {
      cols = 1;
      rows = 1;
    }
  width = (int)((float)env_window_width/(float)cols);
  height = (int)((float)env_window_height/(float)rows);
  k=0;
  for (i=0,x=0;i<cols;i++,x+=width)
    {
      for (j=0,y=0;j<rows;j++,y+=height)
	{
	  display_env(ss,all_envs[k],all_names[k],(selected_env == all_envs[k]) ? rgc : gc,x,y,width,height,0);
	  k++;
	  if (k == all_envs_top) return;
	}
    }
}

static int hit_env(int xe, int ye)
{
  int cols,rows,i,j,width,height,x,y,k;
  if (all_envs_top == 0)
    return(-1);
  else
    {
      if (all_envs_top == 1)
	return(0);
      else
	{
	  cols = round(sqrt((float)(all_envs_top * env_window_width) / (float)env_window_height));
	  rows = round((float)all_envs_top/(float)cols);
	  if ((rows*cols) < all_envs_top) rows++;
	  width = (int)((float)env_window_width/(float)cols);
	  height = (int)((float)env_window_height/(float)rows);
	  k=0;
	  for (i=0,x=width;i<cols;i++,x+=width)
	    {
	      if (x > xe)
		{
		  for (j=0,y=height;j<rows;j++,y+=height)
		    {
		      if (y > ye) return(k);
		      k++;
		    }
		}
	      else k+=rows;
	    }
	}
    }
  return(0);
}

static void do_env_edit(env *new_env, int loading)
{
  int i;
  if (env_list_top == env_list_size)
    {
      env_list_size += 16;
      if (env_list)
	{
	  env_list = (env **)REALLOC(env_list,env_list_size * sizeof(env *));
	  for (i=env_list_top;i<env_list_size;i++) env_list[i] = NULL;
	}
      else env_list = (env **)CALLOC(env_list_size,sizeof(env *));
    }
  /* clear out current edit list above this edit */
  for (i=env_list_top;i<env_list_size;i++)
    {
      if (env_list[i]) env_list[i] = free_env(env_list[i]);
    }
  env_list[env_list_top] = copy_env(new_env);
  env_list_top++;
  if (!loading)
    {
      XtSetSensitive(undoB,TRUE);
      XtSetSensitive(redoB,FALSE);
      XtSetSensitive(saveB,TRUE);
      XtSetSensitive(revertB,TRUE);
    }
}

static void redo_env_edit(void)
{
  if (env_list)
    {
      if ((env_list_top < env_list_size) && (env_list[env_list_top])) 
	{
	  env_list_top++;
	  XtSetSensitive(undoB,TRUE);
	  XtSetSensitive(revertB,TRUE);
	}
      if ((env_list_top == env_list_size) || (env_list[env_list_top] == NULL)) 
	XtSetSensitive(redoB,FALSE);
      XtSetSensitive(saveB,TRUE);
    }
}

static void undo_env_edit(void)
{
  if (env_list)
    {
      if (env_list_top > 0)
	{
	  env_list_top--;
	  XtSetSensitive(redoB,TRUE);
	}
      if (env_list_top == 0)
	{
	  XtSetSensitive(undoB,FALSE);
	  XtSetSensitive(revertB,FALSE);
	}
      XtSetSensitive(saveB,TRUE);
    }
}

static void revert_env_edit(void)
{
  if (env_list)
    {
      if (env_list_top > 1) 
	env_list_top = 1; 
      else 
	{
	  env_list_top = 0;
	  XtSetSensitive(undoB,FALSE);
	  XtSetSensitive(revertB,FALSE);
	  XtSetSensitive(saveB,FALSE);
	}
    }
}


static int find_env(char *name)
{ /* -1 upon failure */
  int i;
  if (all_envs)
    {
      for (i=0;i<all_envs_top;i++)
	{
	  if (strcmp(name,all_names[i]) == 0) return(i);
	}
    }
  return(-1);
}

static void make_scrolled_env_list (snd_state *ss)
{
  XmString *strs;
  int n;
  if (!(ss->using_schemes)) XtVaSetValues(screnvlst,XmNbackground,(ss->sgx)->highlight_color,NULL); 
  strs = (XmString *)CALLOC(all_envs_top,sizeof(XmString)); 
  for (n=0;n<all_envs_top;n++) strs[n] = XmStringCreate(all_names[n],"button_font");
  XtVaSetValues(screnvlst,XmNitems,strs,XmNitemCount,all_envs_top,NULL);
  for (n=0;n<all_envs_top;n++) XmStringFree(strs[n]);
  FREE(strs);
}

static void add_envelope(snd_state *ss, char *name, env *val)
{
  if (all_envs_top == all_envs_size)
    {
      all_envs_size += 16;
      if (all_envs)
	{
	  all_envs = (env **)REALLOC(all_envs,all_envs_size * sizeof(env *));
	  all_names = (char **)REALLOC(all_names,all_envs_size * sizeof(char *));
	}
      else
	{
	  all_envs = (env **)CALLOC(all_envs_size,sizeof(env *));
	  all_names = (char **)CALLOC(all_envs_size,sizeof(char *));
	}
    }
  all_envs[all_envs_top] = val;
  all_names[all_envs_top] = name;
  all_envs_top++;
  if (enved_dialog)
    {
      XtSetSensitive(showB,TRUE);
      make_scrolled_env_list(ss);
    }
}

void alert_envelope_editor(snd_state *ss, char *name, env *val)
{
  /* whenever an envelope is defined or setf'd, we get notification through this function */
  int i;
  i = find_env(name);
  if (i != -1)
    all_envs[i] = val;
  else add_envelope(ss,name,val);
}

void alert_enved_amp_env(snd_info *sp)
{
  snd_state *ss;
  ss = sp->state;
  if ((enved_dialog) && (active_channel) && (enved_waving(ss)) && (XtIsManaged(enved_dialog)))
    {
      if (active_channel->sound == sp) env_redisplay(sp->state);
    }
}

void new_active_channel_alert(snd_state *ss)
{
  if (enved_dialog)
    {
      /* if showing current active channel in gray, update */
      active_channel = current_channel(ss);
      env_redisplay(ss);
    }
}

static void Dismiss_Enved_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  if (ss->checking_explicitly)
    ss->stopped_explicitly = 1;
  else XtUnmanageChild(enved_dialog);
}

static void Help_Enved_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
#if HAVE_XmHTML
  snd_help((snd_state *)clientData,"Envelope Editor","#editenvelope");
#else
  snd_help((snd_state *)clientData,"Envelope Editor",get_envelope_editor_help());
#endif
}

static void apply_enved(snd_state *ss)
{
  XmString s1;
  if (active_env)
    {
      active_channel = current_channel((void *)ss);
      if (active_channel)
	{
	  XtSetSensitive(applyB,FALSE);
	  XtSetSensitive(apply2B,FALSE);
	  s1 = XmStringCreate(STR_Stop,XmFONTLIST_DEFAULT_TAG);
	  XtVaSetValues(cancelB,XmNlabelString,s1,NULL);
	  XmStringFree(s1);
	  check_for_event(ss);
	  switch (enved_target(ss))
	    {
	    case AMPLITUDE_ENV:
	      apply_env(active_channel,active_env,0,current_ed_samples(active_channel),1.0,apply_to_selection,TRUE,"Enved: amp"); 
	      /* calls update_graph, I think, but in short files that doesn't update the amp-env */
	      if (enved_waving(ss)) env_redisplay(ss);
	      break;
	    case SPECTRUM_ENV: 
	      apply_filter(active_channel,filter_env_order(ss),active_env,TRUE,"Enved: flt",apply_to_selection);
	      break;
	    case SRATE_ENV:
	      src_env_or_num(ss,active_env,0.0,FALSE,TRUE,"Enved: src",apply_to_selection);
	      if (enved_waving(ss)) env_redisplay(ss);
	      break;
	    }
	  XtSetSensitive(applyB,TRUE);
	  XtSetSensitive(apply2B,TRUE);
	  s1 = XmStringCreate(STR_Dismiss,XmFONTLIST_DEFAULT_TAG);
	  XtVaSetValues(cancelB,XmNlabelString,s1,NULL);
	  XmStringFree(s1);
	}
    }
}

void display_frequency_response(snd_state *ss, env *e, axis_info *ap, axis_context *gax, int order, int dBing)
{
  float *coeffs = NULL;
  int height,width,i,pts,x0,y0,x1,y1;
  float samps_per_pixel,invpts,resp,frq,pix;
  coeffs = get_filter_coeffs(order,e);
  if (!coeffs) return;
  height = (ap->y_axis_y1 - ap->y_axis_y0);
  width = (ap->x_axis_x1 - ap->x_axis_x0);
  pts = order*4;
  if (width<50) pts = width; else if (pts < 50) pts = 50;
  if (pts <= 0) pts = 1;
  invpts = 1.0/(float)pts;
  samps_per_pixel = (float)(ap->x_axis_x1 - ap->x_axis_x0) * invpts;
  x1 = ap->x_axis_x0;
  resp = frequency_response(coeffs,order,0.0);
  if (dBing)
    y1 = (int)(ap->y_axis_y0 + (ss->min_dB - dB(ss,resp)) * height / ss->min_dB);
  else y1 = (int)(ap->y_axis_y0 + resp * height);
  for (i=1,pix=x1,frq=invpts;i<pts;i++,pix+=samps_per_pixel,frq+=invpts)
    {
      x0 = x1;
      x1 = (int)(pix);
      y0 = y1;
      resp = frequency_response(coeffs,order,frq);
      if (dBing)
	y1 = (int)(ap->y_axis_y0 + (ss->min_dB - dB(ss,resp)) * height / ss->min_dB);
      else y1 = (int)(ap->y_axis_y0 + resp * height);
      XDrawLine(gax->dp,gax->wn,gax->gc,x0,y0,x1,y1);
    }
  FREE(coeffs);
}

void env_redisplay(snd_state *ss)
{
  char *name = NULL;
  int samps,srate,pts;
  float samps_per_pixel;
  axis_info *ap,*active_ap;
  XClearWindow(XtDisplay(drawer),XtWindow(drawer));
  if (showing_all_envs) 
    view_envs(ss);
  else 
    {
      name = XmTextGetString(textL);
      if (!name) name = copy_string("noname");
      display_env(ss,active_env,name,gc,0,0,env_window_width,env_window_height,1);
      if (name) FREE(name);
      if (enved_waving(ss))
	{
	  if ((enved_target(ss) == SPECTRUM_ENV) && (active_env))
	    {
	      display_frequency_response(ss,active_env,axis_cp->axis,gray_ap->ax,filter_env_order(ss),enved_dBing(ss));
	    }
	  else
	    {
	      active_channel = current_channel(ss);
	      if (active_channel)
		{
		  if (active_channel->amp_env)
		    {
		      /* show current channel overall view in gray scale */
		      ap = axis_cp->axis;
		      samps = current_ed_samples(active_channel);
		      srate = snd_SRATE(active_channel);
		      active_ap = active_channel->axis;
		      samps_per_pixel = (float)samps / (float)(ap->x_axis_x1 - ap->x_axis_x0);
		      gray_ap->x0 = 0.0;
		      gray_ap->x1 = (float)samps / (float)srate;
		      gray_ap->losamp = 0;
		      gray_ap->hisamp = samps-1;
		      gray_ap->y0 = active_ap->y0;
		      gray_ap->y1 = active_ap->y1;
		      gray_ap->x_axis_x0 = ap->x_axis_x0;
		      gray_ap->x_axis_x1 = ap->x_axis_x1;
		      gray_ap->y_axis_y0 = ap->y_axis_y0;
		      gray_ap->y_axis_y1 = ap->y_axis_y1;
		      gray_ap->x_scale = ((double)(gray_ap->x_axis_x1 - gray_ap->x_axis_x0))/((double)(gray_ap->x1 - gray_ap->x0));
		      gray_ap->y_scale = (gray_ap->y_axis_y1 - gray_ap->y_axis_y0)/(gray_ap->y1 - gray_ap->y0);
		      pts = amp_env_graph(active_channel,gray_ap,samps_per_pixel,srate);
		      draw_both_grfs(gray_ap->ax,pts);
		    }
		  else
		    {
		      start_amp_env(active_channel->sound,1); /* a no-op if one already in progress, will callback to us upon completion */
		    }
		}
	    }
	}
    }
}

static void order_field_activated(snd_state *ss)
{
  /* return in order text field */
  char *str=NULL;
  int order;
  str = XmTextGetString(orderL);
  if ((str) && (*str))
    {
      sscanf(str,"%d",&order);
      if (order&1) order++;
      if ((order>0) && (order<2000)) set_filter_env_order(ss,order);
    }
  if (str) FREE(str);
}

static void text_field_activated(snd_state *ss)
{ /* might be breakpoints to load or an envelope name (<cr> in enved text field) */
  char *name = NULL,*str;
  env *e = NULL;
  int err = 0;
  int newenv = 0;
  name = XmTextGetString(textL);
  if ((name) && (*name))
    {
      str = name;
      while (isspace((int)(*str))) str++;
      e = name_to_env(str);
      if (!e)
	{
	  e = scan_envelope_and_report_error(NULL,str,&err); /* null sp turns off error report */
	  if (e) newenv = 1;
	}
      if (e) 
	{
	  if (active_env) active_env = free_env(active_env);
	  active_env = copy_env(e);
	  env_list_top = 0;
	  do_env_edit(active_env,TRUE);
	  XtSetSensitive(saveB,newenv);
	  env_redisplay(ss);
	}
      /* else if str is alphabetic, rename active_env as str and save a copy in the list */
      if (isalpha((int)(str[0])))
	{
	  alert_envelope_editor(ss,str,copy_env(active_env));
	  add_or_edit_symbol(str,active_env);
	  XtSetSensitive(saveB,FALSE);
	  env_redisplay(ss); /* updates label */
	}
    }
  if (name) FREE(name);
}

static void save_button_pressed(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  char *name = NULL;
  name = XmTextGetString(textL);
  if ((!name) || (!(*name))) 
    name = copy_string("unnamed");
  alert_envelope_editor(ss,name,copy_env(active_env));
  add_or_edit_symbol(name,active_env);
  XtSetSensitive(saveB,FALSE);
  env_redisplay(ss);
  if (name) FREE(name);
}

static void save_button_help_callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Save Current Envelope",
"The envelope in the editor is not saved until you\n\
press the 'save' button.  Once saved, it can be\n\
used in other portions of Snd (for example, as the\n\
argument to C-x C-a), or saved to a file via the\n\
Option menu Save State option.  Similarly, a file\n\
of CLM (lisp) envelope definitions can be loaded\n\
using load.\n\
");
}

static void Apply_Enved_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  /* apply current envs to currently sync'd channels */
  snd_state *ss = (snd_state *)clientData;
  state_context *sgx;
  XmAnyCallbackStruct *cb = (XmAnyCallbackStruct *)callData;
  sgx = ss->sgx;
  /* since this is the OK button from Motif's point of view, text activation (<cr> in the text field)
   * causes a bogus activation of this callback; we trap that by looking for text_activate_event.
   * cr in text => save under that name
   */
  if (cb->event != sgx->text_activate_event)
    apply_enved((snd_state *)clientData);
  else 
    {
      if (sgx->text_widget == textL)
	text_field_activated(ss);
      else order_field_activated(ss);
    }
}

static void Undo_and_Apply_Enved_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  /* undo upto previous amp env, then apply */
  /* this blindly undoes the previous edit (assumed to be an envelope) -- if the user made some other change in the meantime, too bad */
  snd_state *ss = (snd_state *)clientData;
  if ((active_channel) && (active_channel == current_channel((void *)ss)))
    {
      undo_EDIT((void *)ss,1);
    }
  apply_enved(ss);
}

static void Undo_and_Apply_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Undo and Apply",
"This button first tries to undo a previous apply\n\
then calls apply, the point obviously being to make\n\
it easy to retry an envelope after some minor change.\n\
");
}

static void reflect_segment_state (snd_state *ss)
{
  if (enved_dialog)
    {
      if (!(ss->using_schemes))
	{
	  XmChangeColor(expB,(enved_exping(ss)) ? ((Pixel)(ss->sgx)->yellow) : ((Pixel)(ss->sgx)->highlight_color));
	  XmChangeColor(linB,(enved_exping(ss)) ? ((Pixel)(ss->sgx)->highlight_color) : ((Pixel)(ss->sgx)->yellow));
	}
      if ((active_env) && (!(showing_all_envs))) env_redisplay(ss);
    }
}

static void select_or_edit_env(snd_state *ss, int pos)
{
  XmString s1;
  if (selected_env == all_envs[pos]) 
    {
      if (showing_all_envs)
	{
	  showing_all_envs = 0;
	  s1=XmStringCreate(STR_view_envs,"button_font");
	  XtVaSetValues(showB,XmNlabelString,s1,NULL);
	  XmStringFree(s1);
	}
      if (active_env) active_env = free_env(active_env);
      active_env = copy_env(all_envs[pos]);
      XmTextSetString(textL,all_names[pos]);
      env_list_top = 0;
      do_env_edit(active_env,TRUE);
      XtSetSensitive(undoB,FALSE);
      XtSetSensitive(revertB,FALSE);
      XtSetSensitive(saveB,FALSE);
      set_enved_exping(ss,(active_env->base != 1.0));
      set_enved_base(ss,active_env->base);
      env_redisplay(ss);
    }
  else
    {
      selected_env = all_envs[pos];
      if (showing_all_envs) view_envs(ss);
    }
}

static void clear_point_label(void)
{
  XtVaSetValues(brkptL,XmNlabelType,XmSTRING,XmNlabelString,NULL,NULL);
}

static char brkpt_buf[32];

static void display_point_label(snd_state *ss, float x, float y)
{
  XmString s1;
  if ((enved_dBing(ss)) && (ss->min_dB < -60))
    sprintf(brkpt_buf,"%.3f : %.5f",x,y);
  else sprintf(brkpt_buf,"%.3f : %.3f",x,y);
  s1=XmStringCreate(brkpt_buf,"button_font");
  XtVaSetValues(brkptL,XmNlabelType,XmSTRING,XmNlabelString,s1,NULL);
  XmStringFree(s1);
}

void display_enved_progress(snd_state *ss, char *str, Pixmap pix)
{
  XmString s1;
  if (pix == 0)
    {
      s1=XmStringCreate(str,"button_font");
      XtVaSetValues(brkptL,XmNlabelType,XmSTRING,XmNlabelString,s1,NULL);
      XmStringFree(s1);
    }
  else
    {
      XtVaSetValues(brkptL,XmNlabelType,XmPIXMAP,XmNlabelPixmap,pix,NULL);
    }
  check_for_event(ss);
}

static Time down_time;
static int env_dragged = 0;
static int env_pos = 0;
static int click_to_delete = 0;

static void drawer_button_motion(Widget w,XtPointer clientData,XEvent *event,Boolean *cont) 
{
  snd_state *ss = (snd_state *)clientData;
  XMotionEvent *ev = (XMotionEvent *)event;
  Time motion_time;
  axis_info *ap;
  float x0,x1,x,y;
  if (!showing_all_envs)
    {
      motion_time = ev->time;
      if ((motion_time - down_time) < (0.5 * XtGetMultiClickTime(XtDisplay(w)))) return;
      env_dragged = 1;
      click_to_delete = 0;
      ap = axis_cp->axis;
      x = ungrf_x(ap,ev->x);
      if (env_pos > 0) x0 = active_env->data[env_pos*2-2]; else x0 = 0.0;
      if (env_pos < active_env->pts) x1 = active_env->data[env_pos*2+2]; else x1 = 1.0;
      if (x<x0) x=x0;
      if (x>x1) x=x1;
      if (env_pos == 0) x = active_env->data[0];
      if (env_pos == (active_env->pts-1)) x = active_env->data[(active_env->pts-1)*2];
      y = ungrf_y(ap,ev->y);
      if ((enved_clipping(ss)) || (enved_dBing(ss)))
	{
	  if (y<ap->y0) y=ap->y0;
	  if (y>ap->y1) y=ap->y1;
	}
      if (enved_dBing(ss)) y=un_dB(ss,y);
      move_point(active_env,env_pos,x,y);
      display_point_label(ss,x,y);
      env_redisplay(ss);
    }
}

static void drawer_button_press(Widget w,XtPointer clientData,XEvent *event,Boolean *cont) 
{
  XButtonEvent *ev = (XButtonEvent *)event;
  snd_state *ss = (snd_state *)clientData;
  int pos;
  float x,y;
  axis_info *ap;
  down_time = ev->time;
  env_dragged = 0;
  if (showing_all_envs)
    {
      pos = hit_env(ev->x,ev->y);
      XmListSelectPos(screnvlst,pos+1,FALSE);
      if ((pos >= 0) && (pos < all_envs_top)) select_or_edit_env((snd_state *)clientData,pos);
    }
  else
    {
      if (!active_env)
	{
	  active_env = default_env(0.0);
	  active_env->base = 1.0;
	  env_redisplay(ss); /* needed to get current_xs set up correctly */
	}
      ap = axis_cp->axis;
      pos = hit_point(ss,current_xs,current_ys,active_env->pts,ev->x,ev->y);
      x = ungrf_x(ap,ev->x);
      y = ungrf_y_dB(ss,ap,ev->y);
      if (enved_clipping(ss))
	{
	  if (y<ap->y0) y=ap->y0;
	  if (y>ap->y1) y=ap->y1;
	}
      if (pos == -1)
	{
	  if (x < ap->x0)
	    {
	      pos = 0;
	      x = ap->x0;
	    }
	  else 
	    if (x > ap->x1) 
	      {
		pos = active_env->pts-1;
		x = ap->x1;
	      }
	}
      env_pos = pos;
      /* if not -1, then user clicked existing point -- wait for drag/release to decide what to do */
      if (pos == -1) 
	{
	  pos = place_point(current_xs,active_env->pts,ev->x);
	  /* place returns left point index of current segment or pts if off left end */
	  /* in this case, user clicked in middle of segment, so add point there */
	  add_point(active_env,pos+1,x,y);
	  env_pos = pos+1;
	  click_to_delete = 0;
	  env_redisplay(ss);
	}
      else click_to_delete = 1;
      display_point_label(ss,x,y);
    }
}

static void drawer_button_release(Widget w,XtPointer clientData,XEvent *event,Boolean *cont) 
{
  if (!showing_all_envs)
    {
      if ((click_to_delete) && (!env_dragged) && (env_pos != 0)) /* might want to protect last point also */
	{
	  delete_point(active_env,env_pos);
	}
      do_env_edit(active_env,FALSE);
      env_pos = 0;
      env_dragged = 0;
      click_to_delete = 0;
      env_redisplay((snd_state *)clientData);
      clear_point_label();
    }
}

static void drawer_resize(Widget w,XtPointer clientData,XtPointer callData) 
{
  /* update display, can be either view of all envs or sequence of current envs */
  env_window_width = get_window_width(w);
  env_window_height = get_window_height(w);
  env_redisplay((snd_state *)clientData);
}

static void Drawer_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Envelope Editor Display",
"This portion of the envelope editor displays either\n\
the envelope currently being edited, or all currently\n\
loaded envelopes.  In the latter case, click an envelope\n\
to select it, click the selected envelope to load it into\n\
the editor portion.  In the former case, click anywhere in\n\
the graph to place a new breakpoint at that point; click\n\
an existing point to delete it; drag an existing point to\n\
move it.  Put some envelope name in the text field and click\n\
the 'save' button to save the current envelope under the\n\
chosen name.  Unlimited undo/redo are available through the\n\
'undo' and 'redo' buttons.  Set the 'w' button to see the\n\
currently active sound under the envelope; set the 'f' button\n\
to apply the envelope to the sound's frequency, rather than\n\
amplitude; set the 'c' button to clip mouse motion to the current\n\
y axis bounds.\n\
");
}

static void Text_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Envelope Name",
"To load an exisiting envelope into the editor,\n\
type its name in this field; to give a name to\n\
the envelope as it is currently defined in the\n\
graph viewer, type its name in this field, then\n\
either push return or the 'save' button; to define\n\
a new envelope by its break point values, give the\n\
values in this field as though it were a defvar\n\
call in M-X -- that is, '(0 0 1 1)<cr> in this\n\
field fires up a ramp as a new envelope.\n\
");
}

static void Scrolled_List_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Loaded Envelope Names",
"This is the list of all currently loaded envelopes\n\
by name.  When a new envelope is saved, its name is\n\
added to the end of the list.  Click a name to select\n\
that envelope; click the selected envelope to load it\n\
into the editor.  The selected envelope is displayed in\n\
red.\n\
");
}

static void Brkpt_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Breakpoint Data",
"This portion ofthe envelope editor displays the\n\
current breakpoint values while the mouse is dragging\n\
a point\n\
");
}

static void show_button_pressed(Widget w,XtPointer clientData,XtPointer callData) 
{
  /* if show all (as opposed to show current), loop through loaded LV_LISTs */
  XmString s1;
  showing_all_envs = (!showing_all_envs);
  s1=XmStringCreate((showing_all_envs) ? STR_edit_env : STR_view_envs,"button_font");
  XtVaSetValues(showB,XmNlabelString,s1,NULL);
  XmStringFree(s1);
  env_redisplay((snd_state *)clientData);
}

static void show_button_help_callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "View Envs or Edit Env",
"There are two sides to the Envelope Editor, the\n\
editor itself, and an envelope viewer.  This button\n\
chooses which is active.  In the envelope viewer,\n\
all currently loaded envelopes are displayed, and\n\
you can select one by clicking it; click the selected\n\
envelope to load it into the editor portion.  The\n\
other choice is the editor which displays the current\n\
state of the envelope being edited.\n\
");
}

static void selection_button_pressed(Widget s, XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  apply_to_selection = (!apply_to_selection);
  XmChangeColor(selectionB,(apply_to_selection) ? ((Pixel)(ss->sgx)->yellow) : ((Pixel)(ss->sgx)->highlight_color));
}

static void selection_button_help_callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Apply to Selection or Sound",
"The 'Apply' button can affect either the current selection\n\
or the entire current sound.\n\
");
}

static void revert_button_pressed(Widget w,XtPointer clientData,XtPointer callData) 
{
  revert_env_edit();
  if (active_env) active_env = free_env(active_env);
  if (env_list_top > 0) active_env = copy_env(env_list[env_list_top-1]); else active_env = NULL;
  env_redisplay((snd_state *)clientData);
}

static void revert_button_help_callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Revert to Original Envelope",
"The revert button undoes all the edits back to\n\
the original state, and if clicked again at that\n\
point, clears the editor to a clean state (that\n\
is, the envelope '(0 0 1 0)).\n\
");
}

static void undo_button_pressed(Widget w,XtPointer clientData,XtPointer callData) 
{
  undo_env_edit();
  if (active_env) active_env = free_env(active_env);
  if (env_list_top > 0) active_env = copy_env(env_list[env_list_top-1]); else active_env = NULL;
  env_redisplay((snd_state *)clientData);
}

static void undo_button_help_callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Undo Edit",
"The undo button undoes the previous edit.  The\n\
list of edits is not cleared until you changed to\n\
a new envelope, so you can undo and redo every edit\n\
without limitation.\n\
");
}

static void redo_button_pressed(Widget w,XtPointer clientData,XtPointer callData) 
{
  redo_env_edit();
  if (active_env) active_env = free_env(active_env);
  if (env_list_top > 0) active_env = copy_env(env_list[env_list_top-1]); else active_env = NULL;
  env_redisplay((snd_state *)clientData);
}

static void redo_button_help_callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Redo Edit",
"The redo button redoes an edit that was previously\n\
undone.  There is no limit on the number of edits\n\
that can be undone and redone.\n\
");
}

static void reflect_apply_state (snd_state *ss)
{
  XmString s1;
  s1=XmStringCreate(env_names[enved_target(ss)],XmFONTLIST_DEFAULT_TAG);
  XtVaSetValues(nameL,XmNlabelString,s1,NULL);
  XmStringFree(s1);
  XmChangeColor(ampB,(enved_target(ss) == AMPLITUDE_ENV) ? ((Pixel)(ss->sgx)->green) : ((Pixel)(ss->sgx)->highlight_color));
  XmChangeColor(fltB,(enved_target(ss) == SPECTRUM_ENV) ? ((Pixel)(ss->sgx)->green) : ((Pixel)(ss->sgx)->highlight_color));
  XmChangeColor(srcB,(enved_target(ss) == SRATE_ENV) ? ((Pixel)(ss->sgx)->green) : ((Pixel)(ss->sgx)->highlight_color));
  if ((!showing_all_envs) && (enved_waving(ss))) env_redisplay(ss);
}

static void amp_button_pressed(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  in_set_enved_target(ss,enved_target(ss)+1);
  if (enved_target(ss) > SRATE_ENV) in_set_enved_target(ss,AMPLITUDE_ENV);
  reflect_apply_state(ss);
}

static void amp_button_help_callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Env Label",
"If you click the envelope name label, it\n\
is equivalent to pushing the next button\n\
in the sequence amp, flt, src; that is,\n\
it chooses the next in the list of possible\n\
envelope applications.\n\
");
}

static void Freq_Button_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  in_set_enved_target(ss,SPECTRUM_ENV);
  reflect_apply_state(ss);
}

static void Freq_Button_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Apply Envelope to Spectrum",
"When the 'Apply' button is pressed, the envelope\n\
in the editor window is applied to the current sound\n\
and any sounds sync'd to it. If this button is set,\n\
the envelope is interpreted as a frequency response\n\
curve, and used to design an FIR filter that is then\n\
applied to the current sound. The order of the filter\n\
is determined by the variable " S_filter_env_order "\n\
which defaults to 40.\n\
");
}

static void Amp_Button_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  in_set_enved_target(ss,AMPLITUDE_ENV);
  reflect_apply_state(ss);
}

static void Amp_Button_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Apply Envelope to Amplitude",
"When the 'Apply' button is pressed, the envelope\n\
in the editor window is applied to the current sound\n\
and any sounds sync'd to it. If this button is set,\n\
the envelope is interpreted as a amplitude envelope.\n\
");
}

static void Src_Button_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  in_set_enved_target(ss,SRATE_ENV);
  reflect_apply_state(ss);
}

static void Src_Button_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Apply Envelope to Srate",
"When the 'Apply' button is pressed, the envelope\n\
in the editor window is applied to the current sound\n\
and any sounds sync'd to it. If this button is set,\n\
the envelope is applied to the sampling rate.\n\
");
}

void enved_print(snd_state *ss, char *name)
{
  print_enved(name,axis_cp,env_window_height);
}

static void print_button_pressed(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  ss->print_choice = PRINT_ENV;
  File_Print_Callback(w,clientData,callData);
}

static void print_button_help_callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Print Envelopes",
"The print button causes the current envelope editor\n\
display (whether the envelope being edited, or all the\n\
envelopes currently loaded) to be saved in a postscript\n\
file named aaa.eps.  You can send this to a printer with\n\
lpr.\n\
");
}

static void env_browse_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  XmListCallbackStruct *cb = (XmListCallbackStruct *)callData;
  select_or_edit_env((snd_state *)clientData,cb->item_position - 1);
}

static void Graph_Button_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  XmToggleButtonCallbackStruct *cb = (XmToggleButtonCallbackStruct *)callData;
  snd_state *ss = (snd_state *)clientData; 
  in_set_enved_waving(ss,cb->set);
  env_redisplay(ss);
}

static void Graph_Button_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Show Current Active Channel",
"This button, if set, causes a light-colored version\n\
of the current channel's overall data to be displayed\n\
in the envelope editor window, making it easier to\n\
design an envelope to fit the exact contours of the\n\
sound.  Since the envelope is always applied to the\n\
entire sound, the x-axis bounds in the envelope editor\n\
are arbitrary.\n\
");
}

static void dB_Button_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  XmToggleButtonCallbackStruct *cb = (XmToggleButtonCallbackStruct *)callData;
  snd_state *ss = (snd_state *)clientData; 
  in_set_enved_dBing(ss,cb->set);
  env_redisplay(ss);
}

static void dB_Button_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Linear or Log Y Axis",
"This button, if set, causes the y axis to be in dB\n\
");
}

static void Clip_Button_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData; 
  XmToggleButtonCallbackStruct *cb = (XmToggleButtonCallbackStruct *)callData; 
  in_set_enved_clipping(ss,cb->set);
}

static void Clip_Button_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Y Axis Clipped",
"This button, if set, causes break point motion\n\
in the y direction to be clipped at the current\n\
axis bounds.  If unset, the bounds will try to\n\
change to accomodate the current mouse position.\n\
");
}

static void Exp_Button_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData; 
  XmToggleButtonCallbackStruct *cb = (XmToggleButtonCallbackStruct *)callData; 
  in_set_enved_exping(ss,cb->set);
  if ((active_env) && (!(showing_all_envs)))
    {
      if (enved_exping(ss))
	active_env->base = enved_base(ss);
      else active_env->base = 1.0;
      XtSetSensitive(saveB,TRUE);
    }
  reflect_segment_state(ss);
}

static void Exp_Button_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Linear or Exponential Segments",
"This button, if set, causes the segments to be\n\
drawn with an exponential curve\n\
");
}

static void Lin_Button_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData; 
  XmToggleButtonCallbackStruct *cb = (XmToggleButtonCallbackStruct *)callData; 
  in_set_enved_exping(ss,(!(cb->set)));
  if ((active_env) && (!(showing_all_envs)))
    {
      if (enved_exping(ss))
	active_env->base = enved_base(ss);
      else active_env->base = 1.0;
      XtSetSensitive(saveB,TRUE);
    }
  reflect_segment_state(ss);
}

static void Lin_Button_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Linear or Exponential Segments",
"This button, if set, causes the segments to be\n\
drawn with an straight line.\n\
");
}

static void Base_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Exponential Envelope Base",
"This slider sets the base of the exponential used to\n\
connect the envelope breakpoints.\n\
");
}

static void Order_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Filter Order",
"This text field sets the order of the FIR filter used by\n\
the 'flt' envelope.\n\
");
}

#define BASE_MAX 400
#define BASE_MID 200
/* these two just set the granularity of the scale widget, not the user-visible bounds */

static void make_base_label(snd_state *ss, float bval)
{
  char *sfs,*buf;
  int i,len,scale_len;
  len = (int)(enved_power(ss) * 4);
  if (len < 32) len = 32;
  sfs = (char *)CALLOC(len,sizeof(char));
  sprintf(sfs,"%f",bval);
  scale_len = (int)(enved_power(ss) + 3);
  buf = (char *)CALLOC(scale_len,sizeof(char));
  for (i=0;i<scale_len-1;i++) buf[i] = sfs[i];
  make_name_label(baseValue,buf);
  FREE(sfs);
  FREE(buf);
  in_set_enved_base(ss,bval);
  if ((active_env) && (!(showing_all_envs))) 
    {
      active_env->base = enved_base(ss);
      if (enved_exping(ss)) env_redisplay(ss);
    }
}

static void base_changed(snd_state *ss, int val)
{
  float bval;
  if (val == 0) 
    bval = 0.0;
  else 
    {
      if (val == BASE_MID)
	bval = 1.0;
      else
	{
	  if (val>BASE_MID)
	    bval = pow(1.0 + (10.0 * ((float)(val - BASE_MID)/(float)BASE_MID)),enved_power(ss));  
	  else 
	    bval = pow(((float)val / (float)BASE_MID),enved_power(ss) - 1.0);
	}
    }
  make_base_label(ss,bval);
  if ((active_env) && (enved_exping(ss))) XtSetSensitive(saveB,TRUE); /* what about undo/redo here? */
}

static void reflect_changed_base(snd_state *ss, float val)
{
  int ival;
  if (val <= 0.0) 
    ival = 0;
  else
    {
      if (val == 1.0)
	ival = BASE_MID;
      else
	{
	  if (val <= 1.0)
	    ival = (int)(pow(val,1.0/(enved_power(ss)-1.0))*BASE_MID);
	  else ival = (int)(BASE_MID + ((BASE_MID * (pow(val,(1.0/(enved_power(ss)))) - 1)) / 10.0));
	}
    }
  XtVaSetValues(baseScale,XmNvalue,ival,NULL);
  make_base_label(ss,val);
}

static void Base_Drag_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  XmScrollBarCallbackStruct *sb = (XmScrollBarCallbackStruct *)callData;
  base_changed(ss,sb->value);
}

static int base_last_value = BASE_MID;

static void Base_ValueChanged_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  XmScrollBarCallbackStruct *sb = (XmScrollBarCallbackStruct *)callData;
  base_last_value = sb->value;
  base_changed(ss,sb->value);
}

static void Base_Click_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  XmPushButtonCallbackStruct *cb = (XmPushButtonCallbackStruct *)callData;
  snd_state *ss = (snd_state *)clientData;
  XButtonEvent *ev;
  int val;
  ev = (XButtonEvent *)(cb->event);
  if (ev->state & (snd_ControlMask | snd_MetaMask)) val = base_last_value; else val = BASE_MID;
  base_changed(ss,val);
  XtVaSetValues(baseScale,XmNvalue,val,NULL);
}

void create_envelope_editor (snd_state *ss)
{
  int n;
  Arg args[32];
  Widget spacer,spacer1,aform;
  XmString xhelp,xdismiss,xapply,titlestr,s1;
  XGCValues gv;
  char str[8];

  if (!enved_dialog)
    {

      /* -------- DIALOG -------- */
      xdismiss = XmStringCreate(STR_Dismiss,XmFONTLIST_DEFAULT_TAG);
      xhelp = XmStringCreate(STR_Help,XmFONTLIST_DEFAULT_TAG);
      titlestr = XmStringCreate(STR_Edit_Envelope,XmFONTLIST_DEFAULT_TAG);
      xapply = XmStringCreate(STR_Apply,XmFONTLIST_DEFAULT_TAG);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNautoUnmanage,FALSE); n++;
      XtSetArg(args[n],XmNcancelLabelString,xdismiss); n++;
      XtSetArg(args[n],XmNhelpLabelString,xhelp); n++;
      XtSetArg(args[n],XmNokLabelString,xapply); n++;
      XtSetArg(args[n],XmNdialogTitle,titlestr); n++;
      XtSetArg(args[n],XmNresizePolicy,XmRESIZE_GROW); n++;

#if RESIZE_DIALOG
      XtSetArg(args[n],XmNnoResize,FALSE); n++;
#endif
      XtSetArg(args[n],XmNtransient,FALSE); n++;
      enved_dialog = XmCreateTemplateDialog(main_SHELL(ss),STR_envelope_editor,args,n);
      add_dialog(ss,enved_dialog);
#if OVERRIDE_TOGGLE
      override_form_translation(enved_dialog);
#endif
  
      XtAddCallback(enved_dialog,XmNcancelCallback,Dismiss_Enved_Callback,ss);
      XtAddCallback(enved_dialog,XmNhelpCallback,Help_Enved_Callback,ss);
      XtAddCallback(enved_dialog,XmNokCallback,Apply_Enved_Callback,ss);

      XmStringFree(xhelp);
      XmStringFree(xdismiss);
      XmStringFree(titlestr);
      XmStringFree(xapply);

      if (!(ss->using_schemes))
	{
	  XtVaSetValues(XmMessageBoxGetChild(enved_dialog,XmDIALOG_CANCEL_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	  XtVaSetValues(XmMessageBoxGetChild(enved_dialog,XmDIALOG_HELP_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	  XtVaSetValues(XmMessageBoxGetChild(enved_dialog,XmDIALOG_OK_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	}

      n=0;
      if (!(ss->using_schemes)) 
	{
	  n = background_basic_color(args,n,ss);
	  XtSetArg(args[n],XmNarmColor,(ss->sgx)->pushed_button_color); n++;
	}
      apply2B = XtCreateManagedWidget(STR_Undo_and_Apply,xmPushButtonWidgetClass,enved_dialog,args,n);
      XtAddCallback(apply2B,XmNactivateCallback,Undo_and_Apply_Enved_Callback,ss);
      XtAddCallback(apply2B,XmNhelpCallback,Undo_and_Apply_Help_Callback,ss);

      /* -------- MAIN WIDGET -------- */
      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNbottomWidget,XmMessageBoxGetChild(enved_dialog,XmDIALOG_SEPARATOR)); n++;
      mainform = sndCreateFormWidget("formd",enved_dialog,args,n);

      /* the order in which widgets are defined matters a lot here:
       * we need to build from the bottom up so that the graph portion expands
       * when the window is resized (if top-down, the slider at the bottom expands!)
       */

      /* -------- EXP SLIDER AT BOTTOM -------- */
      n=0;      
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNrecomputeSize,FALSE); n++;
      XtSetArg(args[n],XmNshadowThickness,0); n++;
      XtSetArg(args[n],XmNhighlightThickness,0); n++;
      XtSetArg(args[n],XmNfillOnArm,FALSE); n++;
      baseLabel = sndCreatePushButtonWidget (STR_exp_base,mainform,args,n);
      XtAddCallback(baseLabel,XmNhelpCallback,Base_Help_Callback,ss);
      XtAddCallback(baseLabel,XmNactivateCallback,Base_Click_Callback,ss);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      s1 = XmStringCreate("1.000",XmFONTLIST_DEFAULT_TAG);
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,baseLabel); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,baseLabel); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      /*      XtSetArg(args[n],XmNrecomputeSize,FALSE); n++; */
      XtSetArg(args[n],XmNlabelString,s1); n++;
      baseValue = XtCreateManagedWidget ("base-label",xmLabelWidgetClass,mainform,args,n);
      XtAddCallback(baseValue,XmNhelpCallback,Base_Help_Callback,ss);
      XmStringFree(s1);

      n=0;      
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNcolumns,3); n++;
      XtSetArg(args[n],XmNrecomputeSize,FALSE); n++;
#ifndef SGI
      XtSetArg(args[n],XmNheight,24); n++;
#endif
      XtSetArg(args[n],XmNresizeWidth,FALSE); n++;
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNmarginHeight,0); n++;
      XtSetArg(args[n],XmNmarginBottom,0); n++;
      sprintf(str,"%d",filter_env_order(ss));
      XtSetArg(args[n],XmNvalue,str); n++;
      orderL = sndCreateTextFieldWidget(ss,"orderL",mainform,args,n,ACTIVATABLE,NO_COMPLETER);
      XtAddCallback(orderL,XmNhelpCallback,Order_Help_Callback,ss);

      n=0;      
      if (!(ss->using_schemes)) n = background_position_color(args,n,ss);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,baseLabel); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,baseValue); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNrightWidget,orderL); n++;
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNmaximum,BASE_MAX); n++;
      XtSetArg(args[n],XmNvalue,BASE_MID); n++;
      XtSetArg(args[n],XmNincrement,1); n++;
      XtSetArg(args[n],XmNpageIncrement,1); n++;
      XtSetArg(args[n],XmNdragCallback,make_callback_list(Base_Drag_Callback,(XtPointer)ss)); n++;
      XtSetArg(args[n],XmNvalueChangedCallback,make_callback_list(Base_ValueChanged_Callback,(XtPointer)ss)); n++;
      baseScale = XtCreateManagedWidget("expscl",xmScrollBarWidgetClass,mainform,args,n);
      XtAddCallback(baseScale,XmNhelpCallback,Base_Help_Callback,ss);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNbottomWidget,baseScale); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNmargin,LINE_MARGIN); n++;
      XtSetArg(args[n],XmNheight,LINE_MARGIN); n++;
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNseparatorType,XmNO_LINE); n++;
      XtSetArg(args[n],XmNheight,5); n++;
      baseSep = XtCreateManagedWidget("snd-rec-sep",xmSeparatorWidgetClass,mainform,args,n);

      /* -------- AMP ENV NAME -------- */
      n=0;
      s1=XmStringCreate(STR_amp_env_p,XmFONTLIST_DEFAULT_TAG);
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNbottomWidget,baseSep); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      XtSetArg(args[n],XmNshadowThickness,0); n++;
      XtSetArg(args[n],XmNhighlightThickness,0); n++;
      XtSetArg(args[n],XmNfillOnArm,FALSE); n++;
      nameL = sndCreatePushButtonWidget("nameL",mainform,args,n);
      XtAddCallback(nameL,XmNactivateCallback,amp_button_pressed,ss);
      XtAddCallback(nameL,XmNhelpCallback,amp_button_help_callback,ss);
      XmStringFree(s1);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNbottomWidget,baseSep); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,nameL); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      textL = sndCreateTextFieldWidget(ss,"textL",mainform,args,n,ACTIVATABLE,NO_COMPLETER);
      XtAddCallback(textL,XmNhelpCallback,Text_Help_Callback,ss);

      /* -------- dB, GRAPH ('wave') AND CLIP BUTTONS -------- */
      n=0;
      if (!(ss->using_schemes)) 
	{
	  n = background_basic_color(args,n,ss);
	  XtSetArg(args[n],XmNselectColor,(ss->sgx)->pushed_button_color); n++;
	}
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNbottomWidget,baseSep); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      if (ss->toggle_size > 0) {XtSetArg(args[n],XmNindicatorSize,ss->toggle_size); n++;}
      dBB = sndCreateToggleButtonWidget(STR_dB,mainform,args,n);
      XtAddCallback(dBB,XmNvalueChangedCallback,dB_Button_Callback,ss);
      XtAddCallback(dBB,XmNhelpCallback,dB_Button_Help_Callback,ss);

      n=0;
      if (!(ss->using_schemes)) 
	{
	  n = background_basic_color(args,n,ss);
	  XtSetArg(args[n],XmNselectColor,(ss->sgx)->pushed_button_color); n++;
	}
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNbottomWidget,baseSep); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNrightWidget,dBB); n++;
      if (ss->toggle_size > 0) {XtSetArg(args[n],XmNindicatorSize,ss->toggle_size); n++;}
      graphB = sndCreateToggleButtonWidget(STR_wave,mainform,args,n);
      XtAddCallback(graphB,XmNvalueChangedCallback,Graph_Button_Callback,ss);
      XtAddCallback(graphB,XmNhelpCallback,Graph_Button_Help_Callback,ss);

      n=0;
      if (!(ss->using_schemes)) 
	{
	  n = background_basic_color(args,n,ss);
	  XtSetArg(args[n],XmNselectColor,(ss->sgx)->pushed_button_color); n++;
	}
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNbottomWidget,baseSep); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNrightWidget,graphB); n++;
      if (ss->toggle_size > 0) {XtSetArg(args[n],XmNindicatorSize,ss->toggle_size); n++;}
      clipB = sndCreateToggleButtonWidget(STR_clip,mainform,args,n);
      XtAddCallback(clipB,XmNvalueChangedCallback,Clip_Button_Callback,ss);
      XtAddCallback(clipB,XmNhelpCallback,Clip_Button_Help_Callback,ss);

      /* -------- BREAKPOINT DATA DISPLAY LABEL -------- */
      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNbottomWidget,baseSep); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,textL); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNrightWidget,clipB); n++;
      XtSetArg(args[n],XmNrecomputeSize,FALSE); n++;
      XtSetArg(args[n],XmNfontList,button_FONT(ss)); n++;
      XtSetArg(args[n],XmNlabelType,XmSTRING); n++;
      brkptL = XtCreateManagedWidget("         ",xmLabelWidgetClass,mainform,args,n);
      XtAddCallback(brkptL,XmNhelpCallback,Brkpt_Help_Callback,ss);

      /* -------- SPACERS TO DIVIDE WINDOW IN TWO -------- */
      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNbottomWidget,textL); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNheight,4); n++;
      XtSetArg(args[n],XmNseparatorType,XmNO_LINE); n++;
      spacer = XtCreateManagedWidget("spacer",xmSeparatorWidgetClass,mainform,args,n);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNbottomWidget,spacer); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNseparatorType,XmSHADOW_ETCHED_IN); n++;
      spacer1 = XtCreateManagedWidget("spacer1",xmSeparatorWidgetClass,mainform,args,n);
      /* second separator needed because marginTop seems to be broken or non-existent for these widgets */

      /* -------- WINDOW LEFT WIDGET HOLDER -------- */
      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNbottomWidget,spacer1); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      aform = sndCreateFormWidget("aform",mainform,args,n);

      /* -------- BUTTON BOX AT TOP LEFT -------- */
      n=0;
      if (!(ss->using_schemes)) n = background_zoom_color(args,n,ss);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNshadowThickness,4); n++;
      colF = sndCreateFrameWidget("env-button-frame",aform,args,n);

      n=0;
      if (!(ss->using_schemes)) n = background_highlight_color(args,n,ss);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNorientation,XmVERTICAL); n++;
      XtSetArg(args[n],XmNspacing,0); n++;
      colB = sndCreateRowColumnWidget("env-button-holder",colF,args,n);

      n=0;
      if (!(ss->using_schemes)) 
	{
	  n = background_highlight_color(args,n,ss);
	  XtSetArg(args[n],XmNfontList,button_FONT(ss)); n++;
	  XtSetArg(args[n],XmNarmColor,(ss->sgx)->pushed_button_color); n++;
	}

      showB = XtCreateManagedWidget(STR_view_envs,xmPushButtonWidgetClass,colB,args,n);
      selectionB = XtCreateManagedWidget(STR_selection,xmPushButtonWidgetClass,colB,args,n);
      saveB = XtCreateManagedWidget(STR_save,xmPushButtonWidgetClass,colB,args,n);
      revertB = XtCreateManagedWidget(STR_revert,xmPushButtonWidgetClass,colB,args,n);
      undoB = XtCreateManagedWidget(STR_undo,xmPushButtonWidgetClass,colB,args,n);
      redoB = XtCreateManagedWidget(STR_redo,xmPushButtonWidgetClass,colB,args,n);
      printB = XtCreateManagedWidget(STR_print,xmPushButtonWidgetClass,colB,args,n);

      XtAddCallback(showB,XmNactivateCallback,show_button_pressed,ss);
      XtAddCallback(selectionB,XmNactivateCallback,selection_button_pressed,ss);
      XtAddCallback(saveB,XmNactivateCallback,save_button_pressed,ss);
      XtAddCallback(revertB,XmNactivateCallback,revert_button_pressed,ss);
      XtAddCallback(undoB,XmNactivateCallback,undo_button_pressed,ss);
      XtAddCallback(redoB,XmNactivateCallback,redo_button_pressed,ss);
      XtAddCallback(printB,XmNactivateCallback,print_button_pressed,ss);

      XtAddCallback(showB,XmNhelpCallback,show_button_help_callback,ss);
      XtAddCallback(selectionB,XmNhelpCallback,selection_button_help_callback,ss);
      XtAddCallback(saveB,XmNhelpCallback,save_button_help_callback,ss);
      XtAddCallback(revertB,XmNhelpCallback,revert_button_help_callback,ss);
      XtAddCallback(undoB,XmNhelpCallback,undo_button_help_callback,ss);
      XtAddCallback(redoB,XmNhelpCallback,redo_button_help_callback,ss);
      XtAddCallback(printB,XmNhelpCallback,print_button_help_callback,ss);

      /* enved_target choice (a row of three push buttons that acts like a "radio box") */
      n=0;
      if (!(ss->using_schemes)) n = background_highlight_color(args,n,ss);
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNspacing,0); n++;
      XtSetArg(args[n],XmNmarginWidth,0); n++;
      rbrow = sndCreateRowColumnWidget("asf-holder",colB,args,n);  

      n=0;
      if (!(ss->using_schemes)) 
	{
	  n = background_highlight_color(args,n,ss);
	  XtSetArg(args[n],XmNfontList,button_FONT(ss)); n++;
	  XtSetArg(args[n],XmNarmColor,(ss->sgx)->pushed_button_color); n++;
	}
      /* XtSetArg(args[n],XmNmarginWidth,0); n++; */
      XtSetArg(args[n],XmNshadowThickness,1); n++;
      ampB = XtCreateManagedWidget(STR_amp,xmPushButtonWidgetClass,rbrow,args,n);
      fltB = XtCreateManagedWidget(STR_flt,xmPushButtonWidgetClass,rbrow,args,n);
      srcB = XtCreateManagedWidget(STR_src,xmPushButtonWidgetClass,rbrow,args,n);

      XtAddCallback(fltB,XmNactivateCallback,Freq_Button_Callback,ss);
      XtAddCallback(fltB,XmNhelpCallback,Freq_Button_Help_Callback,ss);
      XtAddCallback(ampB,XmNactivateCallback,Amp_Button_Callback,ss);
      XtAddCallback(ampB,XmNhelpCallback,Amp_Button_Help_Callback,ss);
      XtAddCallback(srcB,XmNactivateCallback,Src_Button_Callback,ss);
      XtAddCallback(srcB,XmNhelpCallback,Src_Button_Help_Callback,ss);

      /* similar secondary box for linear/exp buttons */
      n=0;
      if (!(ss->using_schemes)) n = background_highlight_color(args,n,ss);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      lerow = sndCreateFormWidget("le-holder",colB,args,n);  

      n=0;
      if (!(ss->using_schemes)) 
	{
	  n = background_highlight_color(args,n,ss);
	  XtSetArg(args[n],XmNfontList,button_FONT(ss)); n++;
	  XtSetArg(args[n],XmNarmColor,(ss->sgx)->yellow); n++;
	}
      /* XtSetArg(args[n],XmNmarginWidth,0); n++; */
      XtSetArg(args[n],XmNshadowThickness,1); n++;
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_CENTER); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_POSITION); n++;
      XtSetArg(args[n],XmNrightPosition,50); n++;
      linB = XtCreateManagedWidget(STR_linear,xmPushButtonWidgetClass,lerow,args,n);
      n-=3;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,linB); n++;
      expB = XtCreateManagedWidget(STR_exp,xmPushButtonWidgetClass,lerow,args,n);

      XtAddCallback(linB,XmNactivateCallback,Lin_Button_Callback,ss);
      XtAddCallback(linB,XmNhelpCallback,Lin_Button_Help_Callback,ss);
      XtAddCallback(expB,XmNactivateCallback,Exp_Button_Callback,ss);
      XtAddCallback(expB,XmNhelpCallback,Exp_Button_Help_Callback,ss);


      /* -------- ENV LIST AT LEFT UNDER BUTTONS -------- */
      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,colF); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      screnvname = XtCreateManagedWidget(STR_envs_p,xmLabelWidgetClass,aform,args,n);
      XtAddCallback(screnvname,XmNhelpCallback,Scrolled_List_Help_Callback,ss);

      n=0;
#ifdef LESSTIF_VERSION
      if (!(ss->using_schemes)) n = background_white_color(args,n,ss);
#else
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
#endif
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,screnvname); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNfontList,button_FONT(ss)); n++;
      screnvlst = XmCreateScrolledList(aform,"scrolled-env-list",args,n);
      XtManageChild(screnvlst); 
      XtAddCallback(screnvlst,XmNbrowseSelectionCallback,env_browse_Callback,ss);
      map_over_children(screnvlst,set_main_color_of_widget,(void *)ss);
      if (all_envs) make_scrolled_env_list(ss);
      XtAddCallback(screnvlst,XmNhelpCallback,Scrolled_List_Help_Callback,ss);

      /* -------- MAIN GRAPH -------- */

      n=0;
      if (!(ss->using_schemes)) 
	{
	  XtSetArg(args[n],XmNbackground,(ss->sgx)->graph_color); n++;
	}
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNbottomWidget,spacer1 /* textL */); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,aform); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNheight,350); n++;
      XtSetArg(args[n],XmNallowResize,TRUE); n++;
      drawer = sndCreateDrawingAreaWidget("drawer",mainform,args,n);
      XtAddCallback(drawer,XmNhelpCallback,Drawer_Help_Callback,ss);

      gv.function = GXcopy;
      XtVaGetValues(drawer,XmNbackground, &gv.background,XmNforeground, &gv.foreground,NULL);
      gc = XtGetGC(drawer, GCForeground | GCFunction, &gv);
      gv.foreground = (ss->sgx)->red;
      rgc = XtGetGC(drawer, GCBackground | GCForeground | GCFunction, &gv);
      gv.foreground = (ss->sgx)->enved_waveform_color;
      ggc = XtGetGC(drawer, GCBackground | GCForeground | GCFunction, &gv);

      XtManageChild(enved_dialog); /* needed so that window is valid when resize callback is invoked */
      applyB = XmMessageBoxGetChild(enved_dialog,XmDIALOG_OK_BUTTON);
      cancelB = XmMessageBoxGetChild(enved_dialog,XmDIALOG_CANCEL_BUTTON);

      XtAddCallback(drawer,XmNresizeCallback,drawer_resize,ss);
      XtAddCallback(drawer,XmNexposeCallback,drawer_resize,ss);

      XtAddEventHandler(drawer,ButtonPressMask,FALSE,drawer_button_press,ss);
      XtAddEventHandler(drawer,ButtonMotionMask,FALSE,drawer_button_motion,ss);
      XtAddEventHandler(drawer,ButtonReleaseMask,FALSE,drawer_button_release,ss);

      if (!all_envs)
	{
	  XtSetSensitive(showB,FALSE);
	}
      XtSetSensitive(revertB,FALSE);
      XtSetSensitive(undoB,FALSE);
      XtSetSensitive(redoB,FALSE);
      XtSetSensitive(saveB,FALSE);
      if (!(selection_is_current())) XtSetSensitive(selectionB,FALSE);

      XmToggleButtonSetState(clipB,enved_clipping(ss),FALSE);
      XmToggleButtonSetState(graphB,enved_waving(ss),FALSE);
      XmToggleButtonSetState(dBB,enved_dBing(ss),FALSE);

      reflect_apply_state(ss);
      reflect_segment_state(ss);
    }
  else raise_dialog(enved_dialog);
  if (!XtIsManaged(enved_dialog)) XtManageChild(enved_dialog);
  active_channel = current_channel(ss);
}

void save_envelope_editor_state(snd_state *ss, FILE *fd)
{
  char *estr;
  int i;
  for (i=0;i<all_envs_top;i++)
    {
      estr = env_to_string(all_envs[i]);
      if (estr)
	{
	  fprintf(fd,"(defvar %s %s)",all_names[i],estr);
	  if (all_envs[i]->base != 1.0)
	    fprintf(fd," (%s \"%s\" %.4f)",S_set_env_base,all_names[i],all_envs[i]->base);
	  fprintf(fd,"\n");
	  FREE(estr);
	}
    }
}

void set_enved_clipping(snd_state *ss, int val) {in_set_enved_clipping(ss,val); if (enved_dialog) XmToggleButtonSetState(clipB,val,FALSE);}
void set_enved_exping(snd_state *ss, int val) {in_set_enved_exping(ss,val); if (enved_dialog) XmToggleButtonSetState(expB,val,FALSE); reflect_segment_state(ss);}
void set_enved_target(snd_state *ss, int val) {in_set_enved_target(ss,val); if (enved_dialog) reflect_apply_state(ss);}
void set_enved_waving(snd_state *ss, int val) {in_set_enved_waving(ss,val); if (enved_dialog) XmToggleButtonSetState(graphB,val,FALSE);}
void set_enved_dBing(snd_state *ss, int val) {in_set_enved_dBing(ss,val); if (enved_dialog) XmToggleButtonSetState(dBB,val,FALSE);}
void set_enved_base(snd_state *ss, float val) {in_set_enved_base(ss,val); if (enved_dialog) reflect_changed_base(ss,val);}

int set_env_base(char *name, float val)
{
  int i;
  i = find_env(name);
  if (i != -1)
    {
      all_envs[i]->base = val;
      return(1);
    }
  return(0);
}

int enved_dialog_is_active(void)
{
  return((enved_dialog) && (XtIsManaged(enved_dialog)));
}

char *env_name_completer(char *text)
{
  int i,j,len,curlen,matches = 0;
  char *current_match = NULL;
  if ((all_envs) && (text) && (*text))
    {
      len = strlen(text);
      for (i=0;i<all_envs_top;i++)
	{
	  if (strncmp(text,all_names[i],len) == 0)
	    {
	      matches++;
	      add_possible_completion(all_names[i]);
	      if (current_match == NULL)
		current_match = copy_string(all_names[i]);
	      else 
		{
		  curlen = strlen(current_match);
		  for (j=0;j<curlen;j++)
		    if (current_match[j] != all_names[i][j])
		      {
			current_match[j] = '\0';
			break;
		      }
		}
	    }
	}
    }
  set_completion_matches(matches);
  if ((current_match) && (*current_match))
    return(current_match);
  return(copy_string(text));
}

void set_filter_env_order(snd_state *ss, int order)
{
  char str[8];
  if ((order>0) && (order<2000))
    {
      if (order&1) 
	in_set_filter_env_order(ss,order+1);
      else in_set_filter_env_order(ss,order);
      if (enved_dialog_is_active())
	{
	  sprintf(str,"%d",filter_env_order(ss));
	  XmTextFieldSetString(orderL,str);
	  if ((enved_target(ss) == SPECTRUM_ENV) && (enved_waving(ss)) && (!showing_all_envs))
	    env_redisplay(ss);
	}
    }
}

void enved_reflect_selection(int on)
{
  snd_state *ss;
  if (enved_dialog)
    {
      XtSetSensitive(selectionB,on);
      if ((apply_to_selection) && (!on))
	{
	  ss = get_global_state();
	  apply_to_selection = 0;
	  XmChangeColor(selectionB,(Pixel)(ss->sgx)->highlight_color);
	}
    }
}

void color_enved_waveform(Pixel pix)
{
  snd_state *ss;
  ss = get_global_state();
  (ss->sgx)->enved_waveform_color = pix;
  if (enved_dialog)
    {
      XSetForeground(main_DISPLAY(ss),ggc,pix);
      if ((enved_waving(ss)) && (enved_dialog_is_active())) env_redisplay(ss);
    }
}
