;;; Auto-test sequence for Snd as driven from CLM
;;;
;;; to run snd under dbx while driving it from clm, 
;;;   start clm, :pa clm, (snd-open-fifos)
;;;   dbx -r snd
;;;   (receive-snd), (setf snd-is-connected t), :pa USER, :ld snd-test
;;;
;;; in gdb, gdb snd, (receive-snd) run

(in-package :clm)
(trace send-snd receive-snd)


(defvar libefence-loaded nil)    ;if -lefence, don't fire up all the dialogs (I'm short on memory)
(defvar test-loops 1)         ;number of times to repeat this entire test sequence
(defvar randomized t)        ;if t, test sequences are looped and run more or less at random


(defvar error-file nil)
(defvar rloops 1)
(defvar gh-code-loaded nil)
(defvar tst 0)
(defvar i-loop 0)
(defvar menu-added nil)

(defun rs () (or (not randomized) (> (random 1.0) .5)))

(defun report-error (frm &rest args)
  (if (not error-file)
      (setf error-file (open "test.errors" :direction :output :if-exists :supersede :if-does-not-exist :create)))
  (format error-file "[~D:~D] " i-loop tst)
  (apply #'format error-file frm args)
  (format error-file "~%~%")
  (print (apply #'format nil frm args)))

(defun srss (n &optional no-wait)
  (if (not no-wait) (sleep 1))
  (incf tst) (print (format nil "[~D:~D]" i-loop tst))
  (let ((val (send-and-receive-snd n)))
    (if (eq val nil) 0
      (if (eq val t) 1
	val))))

(defun con-test (var-name true-val)
  (let ((val (srss (format nil "(let ((val ~A)) val)" var-name) t)))
    (if (/= val true-val) (report-error "~S is ~A (not ~A)" var-name val true-val))))

(defun sndlib-test (var-name true-val &optional other-val)
  (let ((val (srss (format nil "(~A \"oboe.snd\")" var-name) t)))
    (if (and (/= val true-val) (or (not other-val) (/= val other-val))) (report-error "~S is ~A (not ~A)" var-name val true-val))))

(defun var-test (var-name init-val new-val)
  (let ((val (srss (concatenate 'string "(" var-name ")") t)))
    (if (/= val init-val) (report-error "~S is ~A (not ~A)" var-name val init-val))
    (srss (concatenate 'string "(set-" var-name (format nil " ~S" new-val) ")") t)
    (let ((curval (srss (concatenate 'string "(" var-name ")") t)))
      (if (/= curval new-val) (report-error "~S set to ~A (not ~A)" var-name curval new-val))
      (srss (concatenate 'string "(set-" var-name (format nil " ~S" init-val) ")") t))))

(defun bool-eq (a b)
  (or (eq a b)
      (and a (numberp b) (/= b 0))
      (and b (numberp a) (/= a 0))
      (and (not a) (numberp b) (= b 0))
      (and (not b) (numberp a) (= a 0))))

(defun bvar-test (var-name init-val new-val)
  (let ((val (srss (concatenate 'string "(" var-name ")") t)))
    (if (not (bool-eq val init-val)) (report-error "~S is ~A (not ~A)" var-name val init-val))
    (srss (concatenate 'string "(set-" var-name (if new-val " #t" " #f") ")") t)
    (let ((curval (srss (concatenate 'string "(" var-name ")") t)))
      (if (not (bool-eq curval new-val)) (report-error "~S set to ~A (not ~A)" var-name curval new-val))
      (srss (concatenate 'string "(set-" var-name (if init-val " #t" " #f") ")") t))))

(defun in-var-test (var-name new-val &optional arg1 arg2)
  (srss (concatenate 'string "(set-" var-name " " (or arg1 "") " " (or arg2 "") (format nil " ~A)" new-val)) t)
  (let ((val (srss (concatenate 'string "(" var-name " " (or arg1 "") " " (or arg2 "") ")") t)))
    (if (/= val new-val) (report-error "~A is ~A (not ~A)" var-name val new-val))))


(defun decode-state (n) (if (= n 0) "MD_CS" (if (= n 1) "MD_TITLE" "MD_M")))
(defvar z-snd-index 0)
(defvar obtest-index 0)
(defvar new-index -1)

(defvar hdr-dir-accessible t)		;does ~/sf have all the CLM test sound files
(setf nap-time 4)
(setf i 0)

(run-in-shell "cp" "oboe.snd obtest.snd") ; used in save/save-as/save-mark tests
(when (probe-file "/tmp/snd_snd_fifo") (delete-file "/tmp/snd_snd_fifo"))

;;; (setf clm::*snd-process* "./snd -separate")
(start-snd)
(sleep 2)
(print (format nil "Snd version: ~A" (srss "(version)"))) (force-output)

(defvar ob-snd "ob.snd")
(defvar ob-snd-index -1)
(if (not (probe-file ob-snd)) (setf ob-snd "oboe.snd"))
(defvar now-snd "now.snd")
(defvar now-snd-index -1)
(if (not (probe-file now-snd)) (setf now-snd "oboe.snd"))
(defvar four-snd "4.snd")
(defvar four-snd-index -1)
(if (not (probe-file four-snd))
    (with-sound (:channels 4 :output "4.snd" :play nil :wait t) (user::fm-violin 0 1 440 .1)))
(defvar two-snd "2.snd")
(defvar two-snd-index -1)
(if (not (probe-file two-snd))
    (with-sound (:channels 2 :output "2.snd" :play nil :wait t) (user::fm-violin 0 1 440 .1)))
(defvar four-aiff "4.aiff")
(defvar four-aiff-index -1)
(if (not (probe-file four-aiff)) (setf four-aiff four-snd))
(defvar two-aiff "2.aiff")
(defvar two-aiff-index -1)
(if (not (probe-file two-aiff)) (setf two-aiff two-snd))
(if (not (probe-file "z.snd")) (with-sound (:channels 1 :output "z.snd" :play nil :wait t) ))
(defvar s8-index -1)
(defvar s8-snd "s8.snd")
(defvar sound-files nil)
(when randomized
  (let ((files-list (sound-files-in-directory ".")))
    (setf sound-files (make-array (length files-list) :initial-contents files-list))))

(defun get-random-file-1 (&optional chans-needed)
  (let* ((len (length sound-files))
	 (ind (round (random len)))
	 (file (aref sound-files ind))
	 (chans (sound-chans file)))
    (print (format nil "check (aref sound-files ~D (of ~D)) => ~S: ~D chans (~D needed)" ind len file chans chans-needed))
    (if (or (not chans-needed) (= chans chans-needed))
	file
      nil)))

(defun get-random-file (chans-needed trial trials)
  (if (< trial trials)
      (let ((val (get-random-file-1 chans-needed)))
	(or val
	    (get-random-file chans-needed (1+ trial) trials)))))

(defvar have-guile nil)
(defvar have-xmhtml nil)
(defvar have-clm nil)

(let ((guile (srss "(guile?)")))
  (setf have-guile (and guile (numberp guile) (= guile 1))))

(let ((xmhtml (srss "(XmHTML?)")))
  (setf have-xmhtml (and xmhtml (numberp xmhtml) (= xmhtml 1))))

(let ((clm (srss "(clm?)")))
  (setf have-clm (and clm (numberp clm) (= clm 1))))

(setf tst 0)
(setf i-loop 0)

(loop for i-loops from 0 below test-loops do

  (when (and randomized sound-files)
    (let* ((trials (1+ (round (random 10))))
	   (obsnd (get-random-file 1 0 trials))
	   (nowsnd (get-random-file 1 0 trials))
	   (foursnd (get-random-file 4 0 trials))
	   (fouraiff (get-random-file 4 0 trials))
	   (twosnd (get-random-file 2 0 trials))
	   (twoaiff (get-random-file 2 0 trials)))
      (if obsnd (setf ob-snd obsnd))
      (if nowsnd (setf now-snd nowsnd))
      (if twosnd (setf two-snd twosnd))
      (if foursnd (setf four-snd foursnd))
      (if twoaiff (setf two-aiff twoaiff))
      (if fouraiff (setf four-aiff fouraiff))
      (print (format nil "using ~S ~S ~S ~S ~S ~S" ob-snd now-snd two-snd four-snd two-aiff four-aiff))))

  (setf ob-snd-index (or (srss (format nil "(open-sound ~S)" ob-snd)) -1))
  (let ((n (srss "(file-name)"))
	(sn (srss "(short-file-name)"))
	(r (srss "(read-only)")))
    (print (format nil "opened ~A (~A -- ~A)" sn n (if r "read-only" "writable"))))
  (sleep 2)
  (let ((val (srss "(count-matches \"y>.147\")" t))) (if (and val (/= val 2)) (print (format nil "count-matches y>.147 in ~S is ~D" ob-snd val))))
  (let ((val (srss "(count-matches \"y==0.0\")" t))) (if (and val (/= val 80)) (print (format nil "count-matches y==0.0 in ~S is ~D" ob-snd val))))
  (let ((val (srss "(find \"y>.147\" 0)" t))) (if (and val (/= val 24823)) (print (format nil "find y>.147 in ~S is ~D" ob-snd val))))
  (srss "(play)")
  (if have-guile (srss "(set-graph-style graph-dots)") (srss "(set-graph-style 1)"))
  (srss "(set-show-fft-peaks #f)")
  (srss "(set-cursor 1200)")
  (setf i (srss "(cursor)"))
  (if (/= i 1200) (report-error "cursor at ~D?" i))
  (srss "(set-ffting #t)")
  (srss "(add-sound-file-extension \"avr\")")
  (print (format nil "temp-dir: ~S, eps-file: ~S, recorder: ~S, save-state: ~S"
		 (srss "(temp-dir)") (srss "(eps-file)") (srss "(recorder-file)") (srss "(save-state-file)")))
  (if have-xmhtml (print (format nil "html-dir: ~S" (srss "(html-dir)"))))
  (when have-guile
    (srss "(set-eps-file \"hiho.eps\")")
    (srss "(set-recorder-file \"hiho.snd\")")
    (srss "(set-save-state-file \"hiho.scm\")")
    (print (format nil "save-state: ~S, eps-file: ~S, recorder: ~S"
		   (srss "(save-state-file)") (srss "(eps-file)") (srss "(recorder-file)"))))
  
  (when (and have-guile (zerop i-loops))
    ;; ---------------- Constants 
    (con-test "next-sound-file" NeXT-sound-file)
    (con-test "aiff-sound-file" AIFF-sound-file)
    (con-test "riff-sound-file" RIFF-sound-file)
    (con-test "nist-sound-file" NIST-sound-file)
    (con-test "raw-sound-file" raw-sound-file)
    (con-test "ircam-sound-file" IRCAM-sound-file)
    (con-test "snd-16-linear" snd-16-linear)
    (let ((val (srss "(sound-bytes-per-sample snd-16-linear)")))
      (if (/= val 2) (report-error "(sound-bytes-per-sample snd-16-linear) returns ~D?" val)))
    (con-test "snd-8-mulaw" snd-8-mulaw)
    (con-test "snd-8-linear" snd-8-linear)
    (con-test "snd-32-float" snd-32-float)
    (con-test "snd-32-linear" snd-32-linear)
    (con-test "snd-8-alaw" snd-8-alaw)
    (con-test "snd-8-unsigned" snd-8-unsigned)
    (con-test "snd-24-linear" snd-24-linear)
    (con-test "snd-16-unsigned" snd-16-unsigned)
    (con-test "snd-64-double" snd-64-double)
    (con-test "snd-16-linear-little-endian" snd-16-linear-little-endian)
    (con-test "snd-32-linear-little-endian" snd-32-linear-little-endian)
    (con-test "snd-24-linear-little-endian" snd-24-linear-little-endian)
    (con-test "snd-16-unsigned-little-endian" snd-16-unsigned-little-endian)
    (con-test "snd-32-float-little-endian" snd-32-float-little-endian)
    (con-test "snd-64-double-little-endian" snd-64-double-little-endian)
    (con-test "amplitude-env" 0)
    (con-test "spectrum-env" 1)
    (con-test "srate-env" 2)
    (con-test "graph-lines" 0)
    (con-test "graph-dots" 1)
    (con-test "graph-filled" 2)
    (con-test "graph-dots-and-lines" 3)
    (con-test "graph-lollipops" 4)
    (con-test "normal-fft" 0)
    (con-test "sonogram" 1)
    (con-test "spectrogram" 2)
    (con-test "focus-left" 0)
    (con-test "focus-right" 1)
    (con-test "focus-active" 2)
    (con-test "focus-middle" 3)
    (con-test "x-in-seconds" 0)
    (con-test "x-in-samples" 1)
    (con-test "x-to-one" 2)
    (con-test "speed-as-float" 0)
    (con-test "speed-as-ratio" 1)
    (con-test "speed-as-semitone" 2)
    (con-test "channels-separate" 0)
    (con-test "channels-combined" 1)
    (con-test "channels-superimposed" 2)
    (con-test "fourier-transform" 0)
    (con-test "wavelet-transform" 1)
    (con-test "hankel-transform" 2)
    (con-test "walsh-transform" 3)
    (con-test "autocorrelation" 4)
    (con-test "chebyshev-transform" 5)
    (con-test "legendre-transform" 6)
    (con-test "hadamard-transform" 7)
    (con-test "rectangular-window" 0)
    (con-test "hanning-window" 1)
    (con-test "welch-window" 2)
    (con-test "parzen-window" 3)
    (con-test "bartlett-window" 4)
    (con-test "hamming-window" 5)
    (con-test "blackman2-window" 6)
    (con-test "blackman3-window" 7)
    (con-test "blackman4-window" 8)
    (con-test "exponential-window" 9)
    (con-test "riemann-window" 10)
    (con-test "kaiser-window" 11)
    (con-test "cauchy-window" 12)
    (con-test "poisson-window" 13)
    (con-test "gaussian-window" 14)
    (con-test "tukey-window" 15)
    (con-test "cursor-in-view" 0)
    (con-test "cursor-on-left" 1)
    (con-test "cursor-on-right" 2)
    (con-test "cursor-in-middle" 3)
    (con-test "cursor-update-display" 4)
    (con-test "cursor-no-action" 5)
    (con-test "cursor-claim-selection" 6)
    (con-test "keyboard-no-action" 7)
    )
  
;;; ---------------- Sndlib (has oboe.snd as built-in arg)
  (sndlib-test "sound-samples" 50828)
  (sndlib-test "sound-datum-size" 2)
  (sndlib-test "sound-data-location" 28)
  (sndlib-test "sound-chans" 1)
  (sndlib-test "sound-srate" 22050)
  (sndlib-test "sound-header-type" 0)
  (sndlib-test "sound-data-format" 1)
  (sndlib-test "sound-length" 101684)
  (sndlib-test "sound-type-specifier" 1684960046 779316836)
  (print (format nil "oboe.snd: ~A ~A comment: ~A" (srss "(sound-type-name 0)") (srss "(sound-format-name 1)") (srss "(sound-comment \"oboe.snd\")")))
  
  
;;; ---------------- Variables
  (var-test "fft-size" 256 512)
  (var-test "fft-style" 0 1)
  (var-test "sinc-width" 10 20)
  (var-test "speed-tones" 12 20)
  (var-test "auto-update" 0 1)
  (var-test "dac-size" 256 1024)
  (let ((val (srss "(default-contrasting)" t))) (if (and val (/= val 0)) (report-error "default-contrasting is ~D" val)))
  (let ((val (srss "(default-expanding)" t))) (if (and val (/= val 0)) (report-error "default-expanding is ~D" val)))
  (let ((val (srss "(default-reverbing)" t))) (if (and val (/= val 0)) (report-error "default-reverbing is ~D" val)))
  (let ((val (srss "(default-filtering)" t))) (if (and val (/= val 0)) (report-error "default-filtering is ~D" val)))
  (let ((val (srss "(default-filter-dBing)" t))) (if (and val (/= val 0)) (report-error "default-filter-dBing is ~D" val)))
  (let ((val (srss "(default-output-type)" t))) (if (and val (/= val next-sound-file)) (report-error "default-output-type is ~A" (snd-type-name val))))
  (let ((val (srss "(fft-window)" t))) (if (and val (/= val 6)) (report-error "fft-window is ~D" val)))
  (var-test "default-filter-order" 20 40)
  (let ((val (srss "(normalize-on-open)" t))) (if (or (not val) (and (numberp val) (/= val 1))) (report-error "normalize-on-open is ~D" val)))
  (let ((val (srss "(auto-resize)" t))) (if (or (not val) (and (numberp val) (/= val 1))) (report-error "auto-resize is ~D" val)))
  (let ((val (srss "(color-inverted)" t))) (if (or (not val) (and (numberp val) (/= val 1))) (report-error "color-inverted is ~D" val)))
  (var-test "zero-pad" 0 1)
  (var-test "line-size" 128 256)
  (let ((val (srss "(ask-before-overwrite)" t))) (if (and val (/= val 0)) (report-error "ask-before-overwrite is ~D" val)))
  (var-test "x-axis-style" 0 1)
  (let ((val (srss "(wavo)" t))) (if (and val (/= val 0)) (report-error "wavo is ~D" val)))
  (var-test "wavo-hop" 3 20)
  (var-test "wavo-trace" 64 20)
  (var-test "spectro-hop" 4 20)
  (var-test "colormap" -1 0)
  (var-test "wavelet-type" 0 1)
  (var-test "transform-type" 0 1)
  (var-test "dot-size" 1 20)
  (var-test "speed-style" 0 1)
  (let ((val (srss "(movies)" t))) (if (or (not val) (and (numberp val) (/= val 1))) (report-error "movies is ~D" val)))
  (let ((val (srss "(normalize-fft)" t))) (if (or (not val) (and (numberp val) (/= val 1))) (report-error "normalize-fft is ~D" val)))
  (let ((val (srss "(show-mix-waveforms)" t))) (if (and val (/= val 1)) (report-error "show-mix-waveforms is ~D" val)))
  (let ((val (srss "(mix-waveform-height)" t))) (if (and val (/= val 20)) (report-error "mix-waveform-height is ~D" val)))
  (var-test "fit-data-on-open" 1 0)
  (var-test "zoom-focus-style" 2 0)
  (let ((val (srss "(save-state-on-exit)" t))) (if (and val (/= val 0)) (report-error "save-state-on-exit is ~D" val)))
  (var-test "graph-style" 1 0)
  (var-test "filter-env-order" 40 20)
  (var-test "mixer-groups" 6 8)
  (var-test "mixer-group-max-out-chans" 4 8)
  (let ((val (srss "(verbose-cursor)" t))) (if (and val (/= val 0)) (report-error "verbose-cursor is ~D" val)))
  (let ((val (srss "(show-fft-peaks)" t))) (if (and val (/= val 0)) (report-error "show-fft-peaks is ~D" val)))
  (let ((val (srss "(show-y-zero)" t))) (if (and val (/= val 0)) (report-error "show-y-zero is ~D" val)))
  (let ((val (srss "(show-marks)" t))) (if (or (not val) (and (numberp val) (/= val 1))) (report-error "show-marks is ~D" val)))
  (let ((val (srss "(show-axes)" t))) (if (or (not val) (and (numberp val) (/= val 1))) (report-error "show-axes is ~D" val)))
  (let ((val (srss "(fft-log-magnitude)" t))) (if (and val (/= val 0)) (report-error "fft-log-magnitude is ~D" val)))
  (let ((val (srss "(fft-log-frequency)" t))) (if (and val (/= val 0)) (report-error "fft-log-frequency is ~D" val)))
  (var-test "channel-style" 0 1)
  (var-test "raw-srate" 44100 22050)
  (var-test "raw-chans" 1 2)
  (var-test "raw-format" 1 2)
  (var-test "raw-type" 0 1)
  (var-test "min-dB" -60.0 -90.0)
  (var-test "corruption-time" 60.0 90.0)
  (let ((val (srss "(use-raw-defaults)" t))) (if (and val (/= val 0)) (report-error "use-raw-defaults is ~D" val)))
  (var-test "print-length" 12 20)
  (var-test "show-mix-consoles" 1 0)
  (let ((val (srss "(recorder-autoload)" t))) (if (and val (/= val 0)) (report-error "recorder-autoload is ~D" val)))
  (var-test "recorder-buffer-size" 4096 1024)
  (var-test "recorder-out-chans" 2 1)
  (var-test "recorder-srate" 22050 44100)
  (var-test "recorder-trigger" 0.0 0.1)
  (var-test "recorder-max-duration" 1000000.0 1.0)
  (let ((val (srss "(enved-waving)" t))) (if (and val (/= val 0)) (report-error "enved-waving is ~D" val)))
  (let ((val (srss "(enved-clipping)" t))) (if (and val (/= val 0)) (report-error "enved-clipping is ~D" val)))
  (let ((val (srss "(enved-exping)"))) (if (and val (/= val 0)) (report-error "enved-exping is ~D" val)))
  (let ((val (srss "(enved-dBing)"))) (if (and val (/= val 0)) (report-error "enved-dBing is ~D" val)))
  (var-test "enved-target" 0 1)
  (if have-guile (let ((val (srss "(string-length \"enved-dBing\")"))) (if (and val (/= val 11)) (report-error "(string-length \"enved-dBing\") is ~D" val))))
  (var-test "default-amp" 1.0 2.0)
  (var-test "default-contrast" 0.0 1.0)
  (var-test "default-contrast-amp" 1.0 0.0)
  (var-test "default-expand-length" .15 .10)
  (var-test "default-expand-ramp" .4 .20)
  (var-test "default-expand-hop" .05 .025)
  (var-test "default-reverb-feedback" 1.09 1.0)
  (var-test "default-reverb-lowpass" 0.7 0.8)
  (var-test "default-expand" 1.0 2.0)
  (var-test "default-speed" 1.0 2.0)
  (var-test "default-reverb-scale" 0.0 1.0)
  (var-test "default-reverb-length" 1.0 2.0)
  (var-test "reverb-decay" 1.0 3.0)
  (when (rs)
    (srss "(save-control-panel)")
    (srss "(restore-control-panel)"))
  (var-test "amp" 1.0 2.0)
  (var-test "contrast" 0.0 1.0)
  (bvar-test "contrasting" nil t)
  (var-test "contrast-amp" 1.0 1.0)
  (var-test "expand" 1.0 2.0)
  (bvar-test "expanding" nil t)
  (var-test "expand-length" .15 .10)
  (var-test "expand-ramp" .4 .20)
  (var-test "expand-hop" .05 .025)
  (var-test "reverb-feedback" 1.09 1.0)
  (var-test "reverb-lowpass" 0.7 0.8)
  (bvar-test "reverbing" nil t)
  (var-test "speed" 1.0 2.0)
  (var-test "reverb-scale" 0.0 1.0)
  (var-test "reverb-length" 1.0 2.0)
  (bvar-test "filtering" nil t)
  (var-test "filter-dBing" 0 1)
  (var-test "filter-order" 20 30)
  (bvar-test "cursor-follows-play" nil t)
  (var-test "prefix-arg" 0 1)
  (var-test "previous-files-sort" 0 1)
  (if (= test-loops 1) (var-test "show-usage-stats" 0 1))
  (var-test "show-selection-transform" 0 1)
  (let ((x (srss "(window-x)"))
	(y (srss "(window-y)")))
    (print (format nil "window at ~A ~A" x y))
    (srss "(set-window-x 0)")
    (srss "(set-window-y 0)")
    (print "window should be at 0 0")
    (sleep 1)
    (srss (format nil "(set-window-x ~D)" (or x 100)))
    (srss (format nil "(set-window-y ~D)" (or y 10))))
  (var-test "max-regions" 16 8)
  (var-test "max-fft-peaks" 100 32)
  (var-test "max-fft-size" 0 8192)
  
  (var-test "initial-x0" 0.0 .10)
  (var-test "initial-x1" 0.1 .20)
  (var-test "initial-y0" -1.0 -2.0)
  (var-test "initial-y1" 1.0 2.0)
  (var-test "xmax" 0.0 0.0)
  (var-test "xmin" 0.0 0.0)
  (var-test "ymax" 1.0 1.0)
  (var-test "ymin" -1.0 -1.0)
  (var-test "spectro-x-scale" 1.0 2.0)
  (var-test "spectro-y-scale" 1.0 2.0)
  (var-test "spectro-z-scale" 0.1 2.0)
  (var-test "spectro-z-angle" -2.0 2.0)
  (var-test "spectro-x-angle" 90.0 2.0)
  (var-test "spectro-y-angle" 0.0 2.0)
  (var-test "fft-beta" 0.0 1.0)
  (var-test "color-cutoff" .003 .10)
  (var-test "color-scale" 1.0 .20)
  (var-test "spectro-cutoff" 1.0 .50)
  (var-test "vu-size" 1.0 2.0)
  (var-test "vu-font-size" 1.0 2.0)
  (var-test "enved-base" 1.0 2.0)

  (srss "(set-max-regions 5)") ;we don't need a bunch of these, and they're glomming up memory

  (srss "(set-corruption-time -1.0)")
  
;;; ---------------- Functions
  (if randomized (setf rloops (1+ (random 10))) (setf rloops 1))
  (loop for rl from 0 below rloops do
    (setf now-snd-index (or (srss (format nil "(view-sound ~S)" now-snd)) -1))
    (if (/= ob-snd-index -1) (srss (format nil "(close-sound ~D)" ob-snd-index))) (setf ob-snd-index -1)
    (setf z-snd-index (or (srss "(open-sound \"z.snd\")") -1))
    (srss "(set-fft-log-magnitude 0)")
    (srss "(set-waving 0)")
    (srss (format nil "(close-sound ~D)" z-snd-index)) (setf z-snd-index -1)
    (if (rs) (srss "(set-show-y-zero)"))
    (when (rs)
      (srss "(set-window-width 600)")
      (srss "(set-window-height 400)")
      (let ((wid (srss "(window-width)"))
	    (hgt (srss "(window-height)")))
	(if (/= wid 600) (report-error "(window-width) returns ~A rather than 600?" wid))
	(if (/= hgt 400) (report-error "(window-height) returns ~A rather than 400?" wid))))
    (if (rs) (if have-guile (srss "(set-graph-style graph-lines)") (srss "(set-graph-style 0)")))
    (if (rs) (srss "(set-show-fft-peaks)"))
    (if (rs) (srss "(set-fft-log-magnitude)"))
    (when (rs)
      (srss "(set-ffting)")
      (let ((val (srss "(ffting)"))) (if (or (not val) (and (numberp val) (/= val 1))) (report-error "(ffting) returns ~A" val))))
    (if (rs) (srss "(set-show-marks)"))
    (if (rs) (srss "(graph->ps)"))
    (if (rs) (srss "(set-show-axes #f)"))
    (if (rs) (srss "(set-cursor 12000)"))
    (when (rs)
      (let* ((mi (srss "(add-mark 12000)"))
	     (ms (srss (format nil "(mark-sample ~D)" mi))))
	(if (/= ms 12000) (report-error "(mark-sample ~D) is ~D rather than 12000?" mi ms))
	(srss (format nil "(set-mark-sample ~D 9000)" mi))
	(setf ms (srss (format nil "(mark-sample ~D)" mi)))
	(if (/= ms 9000) (report-error "(mark-sample ~D) is ~D rather than 9000?" mi ms))))
    (if (rs) (srss "(set-cursor 6000)"))
    (when (rs)
      (let ((mi (srss "(add-mark 6000)")))
	(srss (format nil "(set-mark-name ~D \"hiho\")" mi))
	(let ((name (srss (format nil "(mark-name ~D)" mi))))
	  (if (not (string-equal "hiho" name))
	      (report-error "(mark-name ~D) returns ~S rather than \"hiho\"?)" mi name)))))
    (if (rs) (srss "(delete-mark 1)"))
    (if (rs) (srss "(set-cursor 18000)"))
    (if (rs) (srss "(set-show-y-zero 0)"))
    (if (rs) (srss "(set-fft-log-magnitude 0)"))
    (if (rs) (srss "(find-mark 12000)"))
    (if (rs) (srss "(set-ffting 0)"))
    (if (rs) (srss "(set-x-bounds 0.0 1.0)"))
    (if (rs) (srss "(set-x-bounds 0.1 0.15)"))
    (if (rs) (srss "(set-waving 0)"))
    (srss "(set-show-axes #t)")
    (if (rs) (if have-guile (srss "(set-fft-style sonogram)") (srss "(set-fft-style 1)")))
    (if (rs) (srss "(set-ffting 1)"))
    (if (rs) (srss "(set-spectro-cutoff .5)"))
    (if (rs) (srss "(set-showing-controls)"))
    (when (rs)
      (srss "(set-waving)")
      (let ((val (srss "(waving)"))) (if (or (not val) (and (numberp val) (/= val 1))) (report-error "(waving) returns ~A" val))))
    (if (rs) (srss "(set-y-bounds -3.0 3.0)"))
    (if (rs) (srss "(set-x-bounds 0.0 2.0)"))
    (if (rs) (srss "(set-show-marks 0)"))
    (if (rs) (if have-guile (srss "(set-transform-type fourier-transform)") (srss "(set-transform-type 0)")))
    (if (rs) (srss "(set-show-fft-peaks)"))
    (if (rs) (srss "(set-fft-log-frequency)"))
    (if (rs) (srss "(set-fft-log-magnitude 0)"))
    (if (rs) (srss "(set-showing-controls 0)"))
    (if (rs) (srss "(set-x-bounds 0.0 0.1)"))
    (if (rs) (if have-guile (srss "(set-fft-style spectrogram)") (srss "(set-fft-style 2)")))
    (if (rs) (srss "(set-fft-log-frequency 0)"))
    (setf ob-snd-index (or (srss (format nil "(open-sound ~S)" ob-snd)) -1))
    (let ((val (srss (format nil "(find-sound ~S)" ob-snd))))
      (if (or (not val) (/= val ob-snd-index)) (report-error "(find-sound ~S) returns ~A but it should be ~A" ob-snd val ob-snd-index)))
    (srss (format nil "(close-sound ~D)" now-snd-index)) (setf now-snd-index -1)

    (if (rs) (srss "(delete-samples 0 1200)"))
    (if (rs) (srss "(undo 1)"))
    (if (rs) (srss "(redo 1)"))
    (if (rs) (srss "(undo 1)"))
    (if (and have-guile (rs)) (srss "(env-sound '(0 0 1 1 2 0) 0 22050)"))
    (if (and have-guile (rs)) (srss "(insert-samples 120 10 #(.1 .2 .3 .4 .5 .4 .3 .2 .1 0))"))
    (if (rs) (srss "(undo 1)"))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\f) 4)))
    (when (rs)
      (srss "(set-syncing)")
      (let ((val (srss "(syncing)")))
	(if (or (not val) (and (numberp val) (/= val 1))) (report-error "(syncing) returns ~A?" val))))
    (setf now-snd-index (or (srss (format nil "(open-sound ~S)" now-snd)) -1))
    (if (rs) (srss "(set-syncing)"))
    (if (rs) (srss "(normalize-view)"))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\e) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\f) 4)))
    (if (rs) (srss "(set-syncing 0)"))
    ;; try various keyboard commands  
    (if (rs) (srss "(set-just-sounds)"))
    (when (rs)
      (srss (format nil "(select-sound ~D)" ob-snd-index))
      (let ((val (srss "(selected-sound)")))
	(if (or (not val) (not (numberp val)) (/= val ob-snd-index)) (report-error "(selected-sound) returns ~A rather than ~A" val ob-snd-index))))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\>) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\<) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\>) 0)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\<) 0)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\a) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\e) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\b) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\p) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\n) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\l) 4)))
    (when (rs) (srss (format nil "(key ~D ~D)" (char-code #\x) 4))
	  (srss (format nil "(key ~D ~D)" (char-code #\b) 0)))
    (if (rs) (srss "(set-cursor 1200)"))
    (if (rs) (srss "(make-region 1000 2000)"))
    (if (rs) (srss "(cut)"))
    (if (rs) (srss "(set-cursor 0)"))
    (if (rs) (srss "(insert-region (cursor))"))
    (srss (format nil "(revert-sound ~D)" ob-snd-index))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\m) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\v) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\d) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\z) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\o) 4)))
    (when (rs) (srss (format nil "(key ~D ~D)" (char-code #\x) 4))
	  (srss (format nil "(key ~D ~D)" (char-code #\u) 0)))
    (if (rs) (srss "(undo 2)"))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\<) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (when (and have-guile (rs))
      (srss "(save-edit-history \"hiho.edits\")")
      (when (probe-file "hiho.edits")
	(with-open-file (f "hiho.edits")
	  (let ((happy t))
	    (loop while happy do
	      (let ((c (read-char f nil nil)))
		(if c (princ c) (setf happy nil))))))
	(delete-file "hiho.edits")
	(terpri)))
    (let ((data (make-array 32 :element-type 'integer :initial-element 8192))
	  (indata nil))
      (loop for i from 0 to 31 do (setf (aref data i) (* (- 32 i) 256)))
      (srss "(clm-snd-set-samples 632)")
      (clm-data 32 data)
      (srss "(set-cursor 0)")
      (srss "(undo 1)")
      (srss "(undo 1)")
      (srss "(redo 1)")
      #-clisp (if have-guile (let ((val (srss "(edits)"))) (report-error "edits: ~A" val)))
      (srss "(redo 1)")
      (srss "(undo 2)")
      (srss "(make-region 1000 1010)")
      #-clisp (setf indata (srss "(region-samples 0)"))
      (print indata))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\w) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\y) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\q) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\t) 4)))
    (if (rs) (srss "(set-cursor 8000)"))
    (when (rs) (srss (format nil "(key ~D ~D)" (char-code #\x) 4))
	  (srss (format nil "(key ~D ~D)" (char-code #\f) 0))
	  (srss (format nil "(key ~D ~D)" (char-code #\g) 4)))
    (when (rs) (srss (format nil "(key ~D ~D)" (char-code #\x) 4))
	  (srss (format nil "(key ~D ~D)" (char-code #\i) 0))
	  (srss (format nil "(key ~D ~D)" (char-code #\g) 4)))
    (when (rs) (srss (format nil "(key ~D ~D)" (char-code #\x) 4))
	  (srss (format nil "(key ~D ~D)" (char-code #\l) 0))
	  (srss (format nil "(key ~D ~D)" (char-code #\g) 4)))
    (when (rs) (srss (format nil "(key ~D ~D)" (char-code #\x) 4))
	  (srss (format nil "(key ~D ~D)" (char-code #\u) 0)))
    (when (rs) (srss (format nil "(key ~D ~D)" (char-code #\x) 4))
	  (srss (format nil "(key ~D ~D)" (char-code #\r) 0)))
    (when (rs) (srss (format nil "(key ~D ~D)" (char-code #\x) 4))
	  (srss (format nil "(key ~D ~D)" (char-code #\v) 0)))
    (when (rs) (srss (format nil "(key ~D ~D)" (char-code #\x) 4))
	  (srss (format nil "(key ~D ~D)" (char-code #\o) 4)))
    (when (rs) (srss (format nil "(key ~D ~D)" (char-code #\x) 4))
	  (srss (format nil "(key ~D ~D)" (char-code #\u) 4)))
    (if (rs) (srss (format nil "(revert-sound ~D)" ob-snd-index)))
    ;; now pound on file open/close for awhile
    (when (rs) (srss (format nil "(key ~D ~D)" (char-code #\x) 4))
	  (srss (format nil "(key ~D ~D)" (char-code #\k) 0)))
    (if (/= now-snd-index -1) (srss (format nil "(close-sound ~D)" now-snd-index))) (setf now-snd-index -1)
    (if (/= ob-snd-index -1) (srss (format nil "(close-sound ~D)" ob-snd-index))) (setf ob-snd-index -1)
    ;; should be just menu-bar here
    ;; most commands work, pound on file open/close (widgets get a work out and sound list gets reallocated)
    )
  
  (when (and (not libefence-loaded) (or (not randomized) (zerop i-loops)) hdr-dir-accessible)
    (let ((o2i (srss "(open-sound \"~/sf/o2.bicsf\") "))
	  (r8ui (srss "(open-sound \"~/sf/riff-8-u.snd\") ")))
      (srss (format nil "(close-sound ~D)" r8ui))
      (let ((wi (srss "(open-sound \"~/sf/wood.riff\") ")))
	(srss (format nil "(close-sound ~D)" o2i))
	(let ((wmi (srss "(open-sound \"~/sf/wood.maud\") "))
	      (wsfi (srss "(open-sound \"~/sf/wood.sf\") ")))
	  (srss (format nil "(close-sound ~D)" wmi))
	  (srss (format nil "(close-sound ~D)" wi))
	  (let ((w12i (srss "(open-sound \"~/sf/wood12.aiff\")"))
		(w24i (srss "(open-sound \"~/sf/wood24.aiff\")")))
	    (srss (format nil "(set-read-only #t ~D)" wsfi))
	    (srss (format nil "(close-sound ~D)" wsfi))
	    (let ((n24i (srss "(open-sound \"~/sf/next24.snd\") "))
		  (e16i (srss "(open-sound \"~/sf/esps-16.snd\")")))
	      (srss (format nil "(close-sound ~D)" n24i))
	      (srss (format nil "(close-sound ~D)" e16i)))
	    (srss (format nil "(close-sound ~D)" w12i))
	    (srss (format nil "(close-sound ~D)" w24i)))
	  (let ((sd16i (srss "(open-sound \"~/sf/sd1-16.snd\") "))
		(in16i (srss "(open-sound \"~/sf/inrs-16.snd\")"))
		(s16i (srss "(open-sound \"~/sf/sun-16-afsp.snd\")"))
		(n16i (srss "(open-sound \"~/sf/next-16.snd\") "))
		(smi (srss "(open-sound \"~/sf/sun-mulaw.snd\") "))
		(nmi (srss "(open-sound \"~/sf/next-mulaw.snd\")")))
	    (srss "(normalize-view)")
	    (let ((n8i (srss "(open-sound \"~/sf/next-8.snd\") ")))
	      (srss (format nil "(close-sound ~D)" in16i))
	      (srss (format nil "(close-sound ~D)" s16i))
	      (srss (format nil "(close-sound ~D)" n16i))
	      (let ((nist16i (srss "(open-sound \"~/sf/nist-16.snd\") ")))
		(srss (format nil "(close-sound ~D)" sd16i))
		(srss (format nil "(close-sound ~D)" smi))
		(let ((v8i (srss "(open-sound \"~/sf/voc-8-u.snd\") "))
		      (r16i (srss "(open-sound \"~/sf/riff-16.snd\") "))
		      (a8i (srss "(open-sound \"~/sf/aiff-8.snd\") "))
		      (a16i (srss "(open-sound \"~/sf/aiff-16.snd\")")))
		  (srss "(normalize-view) ")
		  (srss (format nil "(close-sound ~D)" nist16i))
		  (srss (format nil "(close-sound ~D)" v8i))
		  (srss (format nil "(close-sound ~D)" r16i))
		  (srss (format nil "(close-sound ~D)" a8i))
		  (let ((wbi (srss "(open-sound \"~/sf/woodblock.aiff\")")))
		    (srss (format nil "(close-sound ~D)" nmi))
		    (srss (format nil "(close-sound ~D)" n8i))
		    (srss (format nil "(close-sound ~D)" a16i))
		    (let ((uli (srss "(open-sound \"~/sf/ulaw.aifc\") "))
			  (ali (srss "(open-sound \"~/sf/alaw.aifc\") ")))
		      (srss (format nil "(close-sound ~D)" wbi))
		      (srss (format nil "(close-sound ~D)" uli))
		      (srss (format nil "(close-sound ~D)" ali)))))))))))
    (let ((ki (srss "(open-sound \"~/sf/kirk.wve\")"))
	  (ai (srss "(open-sound \"~/sf/addf8.d\") ")))
      (srss (format nil "(close-sound ~D)" ki))
      (srss (format nil "(close-sound ~D)" ai))))
  
  (if (/= ob-snd-index -1) (srss (format nil "(close-sound ~D)" ob-snd-index)))  (setf ob-snd-index -1)
  (if (rs) (srss (format nil "(set-graph-style ~D)" (floor (random 5)))))
    
  (when (and randomized (not libefence-loaded))
    (let ((o2i (srss (format nil "(open-sound ~S)" (get-random-file-1))))
	  (r8ui (srss (format nil "(open-sound ~S)" (get-random-file-1)))))
      (srss (format nil "(close-sound ~D)" r8ui))
      (let ((wi (srss (format nil "(open-sound ~S)" (get-random-file-1)))))
	(srss (format nil "(close-sound ~D)" o2i))
	(let ((wmi (srss (format nil "(open-sound ~S)" (get-random-file-1))))
	      (wsfi (srss (format nil "(open-sound ~S)" (get-random-file-1)))))
	  (srss (format nil "(close-sound ~D)" wmi))
	  (srss (format nil "(close-sound ~D)" wi))
	  (let ((w12i (srss (format nil "(open-sound ~S)" (get-random-file-1))))
		(w24i (srss (format nil "(open-sound ~S)" (get-random-file-1)))))
	    (srss (format nil "(close-sound ~D)" wsfi))
	    (let ((n24i (srss (format nil "(open-sound ~S)" (get-random-file-1))))
		  (e16i (srss (format nil "(open-sound ~S)" (get-random-file-1)))))
	      (srss (format nil "(close-sound ~D)" n24i))
	      (srss (format nil "(close-sound ~D)" e16i)))
	    (srss (format nil "(close-sound ~D)" w12i))
	    (srss (format nil "(close-sound ~D)" w24i)))
	  (let ((sd16i (srss (format nil "(open-sound ~S)" (get-random-file-1))))
		(in16i (srss (format nil "(open-sound ~S)" (get-random-file-1))))
		(s16i (srss (format nil "(open-sound ~S)" (get-random-file-1))))
		(n16i (srss (format nil "(open-sound ~S)" (get-random-file-1))))
		(smi (srss (format nil "(open-sound ~S)" (get-random-file-1))))
		(nmi (srss (format nil "(open-sound ~S)" (get-random-file-1)))))
	    (srss "(normalize-view)")
	    (let ((n8i (srss (format nil "(open-sound ~S)" (get-random-file-1)))))
	      (srss (format nil "(close-sound ~D)" in16i))
	      (srss (format nil "(close-sound ~D)" s16i))
	      (srss (format nil "(close-sound ~D)" n16i))
	      (let ((nist16i (srss (format nil "(open-sound ~S)" (get-random-file-1)))))
		(srss (format nil "(close-sound ~D)" sd16i))
		(srss (format nil "(close-sound ~D)" smi))
		(let ((v8i (srss (format nil "(open-sound ~S)" (get-random-file-1))))
		      (r16i (srss (format nil "(open-sound ~S)" (get-random-file-1))))
		      (a8i (srss (format nil "(open-sound ~S)" (get-random-file-1))))
		      (a16i (srss (format nil "(open-sound ~S)" (get-random-file-1)))))
		  (srss "(normalize-view) ")
		  (srss (format nil "(close-sound ~D)" nist16i))
		  (srss (format nil "(close-sound ~D)" v8i))
		  (srss (format nil "(close-sound ~D)" r16i))
		  (srss (format nil "(close-sound ~D)" a8i))
		  (let ((wbi (srss (format nil "(open-sound ~S)" (get-random-file-1)))))
		    (srss (format nil "(close-sound ~D)" nmi))
		    (srss (format nil "(close-sound ~D)" n8i))
		    (srss (format nil "(close-sound ~D)" a16i))
		    (let ((uli (srss (format nil "(open-sound ~S)" (get-random-file-1))))
			  (ali (srss (format nil "(open-sound ~S)" (get-random-file-1)))))
		      (srss (format nil "(close-sound ~D)" wbi))
		      (srss (format nil "(close-sound ~D)" uli))
		      (srss (format nil "(close-sound ~D)" ali)))))))))))
    (let ((ki (srss (format nil "(open-sound ~S)" (get-random-file-1))))
	  (ai (srss (format nil "(open-sound ~S)" (get-random-file-1)))))
      (srss (format nil "(close-sound ~D)" ki))
      (srss (format nil "(close-sound ~D)" ai)))
    )


  (if randomized (setf rloops (1+ (random 10))) (setf rloops 1))
  (loop for rl from 0 below rloops do

    ;; basic stuff is ok -- now pound on regions and the sync button (multi-channel ops)
    (if (not (probe-file s8-snd)) (setf s8-snd "8chan.snd"))
    (if (not (probe-file s8-snd))
	(setf four-snd-index (or (srss (format nil "(open-sound ~S)" four-snd)) -1))
      (setf s8-index (or (srss (format nil "(open-sound ~S)" s8-snd)) -1)))
    (if (rs) (srss "(set-window-width 600)"))
    (if (rs) (srss "(set-window-height 800)"))
    (if (rs) (srss "(normalize-view) "))
    (when (rs) (srss (format nil "(key ~D ~D)" (char-code #\x) 4))
	  (srss (format nil "(key ~D ~D)" (char-code #\o) 0)))
    (when (rs) (srss (format nil "(key ~D ~D)" (char-code #\x) 4))
	  (srss (format nil "(key ~D ~D)" (char-code #\o) 0)))
  
    (if (rs)
	(let ((data (make-array 32 :element-type 'integer :initial-element 8192))
	      (indata nil))
	  (loop for k from 0 to 3 do
	    (loop for i from 0 to 31 do (setf (aref data i) (* (- 32 i) 256)))
	    (srss "(clm-snd-set-samples 632)")
	    (clm-data 32 data)
	    (loop for j from 0 to 6 do
	      (srss (format nil "(key ~D ~D)" (char-code #\m) 4))
	      (srss (format nil "(set-cursor ~D)" (* j 1000))))
	    (srss "(set-cursor 0)")
	    (srss "(undo 1)")
	    (srss "(reverse-sound)")
	    (srss "(undo 2)")
	    (srss "(redo 2)")
	    (srss "(undo 2)")
	    (srss "(make-region 1000 1010)")
	    (srss (format nil "(key ~D ~D)" (char-code #\w) 4))
	    (srss (format nil "(key ~D ~D)" (char-code #\y) 4))
	    (srss (format nil "(key ~D ~D)" (char-code #\x) 4))
	    (srss (format nil "(key ~D ~D)" (char-code #\o) 0)))))

    (setf two-snd-index (or (srss (format nil "(open-sound ~S)" two-snd)) -1))
    (if (rs) (srss "(set-syncing)"))
    (if (rs)
	(let ((data (make-array 32 :element-type 'integer :initial-element 8192))
	      (indata nil))
	  (loop for k from 0 to 1 do
	    (loop for j from 0 to 6 do
	      (srss (format nil "(key ~D ~D)" (char-code #\m) 4))
	      (srss (format nil "(set-cursor ~D)" (* j 1000))))
	    (loop for i from 0 to 31 do (setf (aref data i) (* (- 32 i) 256)))
	    (srss "(clm-snd-set-samples 632)")
	    (clm-data 32 data)
	    (srss "(set-cursor 0)")
	    (srss "(undo 1)")
	    (srss "(undo 1)")
	    (srss "(redo 2)")
	    (srss "(undo 2)")
	    (srss "(make-region 1000 1010)")
	    (srss (format nil "(key ~D ~D)" (char-code #\w) 4))
	    (srss (format nil "(key ~D ~D)" (char-code #\y) 4))
	    (srss (format nil "(key ~D ~D)" (char-code #\x) 4))
	    (srss (format nil "(key ~D ~D)" (char-code #\o) 0)))))
    (setf ob-snd-index (or (srss (format nil "(open-sound ~S)" ob-snd)) -1))

    (if (= s8-index -1)
	(progn
	  (srss (format nil "(revert-sound ~D)" four-snd-index))
	  (srss (format nil "(close-sound ~D)" four-snd-index))
	  (setf four-snd-index -1))
      (progn
	(srss (format nil "(revert-sound ~D)" s8-index))
	(srss (format nil "(close-sound ~D)" s8-index))
	(setf s8-index -1)))
	
    (srss (format nil "(revert-sound ~D)" two-snd-index))
    (srss (format nil "(close-sound ~D)" two-snd-index)) (setf two-snd-index -1)
    
    (loop for i from 0 to 30 do
      (when (rs)
	(srss (format nil "(make-region ~D ~D)" (* i 100) (* i 110)))
	(if (and (> i 2) (rs)) (srss (format nil "(delete-region ~D)" (floor (random (1- i)))))))
      (when (> (random 1.0) .5) 
	(srss (format nil "(key ~D ~D)" (char-code #\x) 4))
	(srss (format nil "(key ~D ~D)" (char-code #\p) 0))))
    (srss (format nil "(close-sound ~D)" ob-snd-index)) (setf ob-snd-index -1)
    (setf two-snd-index (or (srss (format nil "(open-sound ~S)" two-snd)) -1))
    (setf ob-snd-index (or (srss (format nil "(open-sound ~S)" ob-snd)) -1))
    (setf four-snd-index (or (srss (format nil "(open-sound ~S)" four-snd)) -1))
    
    (if (rs) (srss "(normalize-view)"))
    (srss (format nil "(close-sound ~D)" four-snd-index)) (setf four-snd-index -1)
    (srss (format nil "(close-sound ~D)" two-snd-index)) (setf two-snd-index -1)
    (srss (format nil "(close-sound ~D)" ob-snd-index)) (setf ob-snd-index -1)
    (setf ob-snd-index (or (srss (format nil "(open-sound ~S)" ob-snd)) -1))
    
    (if (rs) (srss "(set-syncing)"))
    (setf four-snd-index (or (srss (format nil "(open-sound ~S)" four-snd)) -1))
    (srss (format nil "(set-syncing 1 ~D)" four-snd-index))
    (setf two-snd-index (or (srss (format nil "(open-sound ~S)" two-snd)) -1))
    
    (if (rs) (srss (format nil "(set-syncing 1 ~D)" two-snd-index)))
    ;; now multi-channel ops
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\>) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\<) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\>) 0)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\<) 0)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\a) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\e) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\b) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\p) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\n) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\l) 4)))
    (when (rs) (srss (format nil "(key ~D ~D)" (char-code #\x) 4))
	  (srss (format nil "(key ~D ~D)" (char-code #\b) 0)))
    (if (rs) (srss "(set-cursor 1200)"))
    (if (rs) (srss "(make-region 1000 2000)"))
    (if (rs) (srss "(cut)"))
    (if (rs) (srss "(set-cursor 0)"))
    (if (rs) (srss "(insert-region 0 0)"))
    (if (rs) (srss "(reverse-sound)"))
    (if (rs) (srss "(undo 3)"))
    (if (rs) (srss "(redo)"))
    (if (rs) (srss "(undo)"))
    (if (rs) (srss "(redo)"))
    (if (rs) (srss "(undo)"))
    (if (rs) (srss "(set-syncing 0)"))
    (srss (format nil "(close-sound ~D)" two-snd-index))
    (srss (format nil "(close-sound ~D)" four-snd-index))
    (srss (format nil "(close-sound ~D)" ob-snd-index))
    (setf four-aiff-index (or (srss (format nil "(open-sound ~S)" four-aiff)) -1))
    (if (rs) (srss "(set-syncing)"))
    (setf two-aiff-index (or (srss (format nil "(open-sound ~S)" two-aiff)) -1))
    (if (rs) (srss "(set-syncing)"))
    ;; now multi-channel ops
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\>) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\<) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\>) 0)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\<) 0)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\a) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\e) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\b) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\p) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\n) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\i) 4)))
    (if (rs) (srss (format nil "(key ~D ~D)" (char-code #\l) 4)))
    (when (rs) (srss (format nil "(key ~D ~D)" (char-code #\x) 4))
	  (srss (format nil "(key ~D ~D)" (char-code #\b) 0)))
    (if (rs) (srss "(set-cursor 1200)"))
    (if (rs) (srss "(make-region 1000 2000)"))
    (if (rs) (srss "(cut)"))
    (if (rs) (srss "(set-cursor 0)"))
    (if (rs) (srss "(insert-region 0 0)"))
    (if (rs) (srss "(undo 2)"))
    (if (rs) (srss "(redo)"))
    (if (rs) (srss "(undo)"))
    (if (rs) (srss "(redo)"))
    (if (rs) (srss "(undo)"))
    (if (rs) (srss "(set-syncing 0)"))
    (srss (format nil "(close-sound ~D)" four-aiff-index))
    (srss (format nil "(close-sound ~D)" two-aiff-index))
    )

  (if have-guile
      (setf new-index (or (srss "(new-sound \"hiho.wave\" next-sound-file snd-16-linear 22050 1)") -1))
    (setf new-index (or (srss "(new-sound \"hiho.wave\" 0 1 22050 1)") -1)))
  (if (rs) (srss "(set-syncing)"))
  (srss "(mix \"pistol.snd\" 100)")
  (if (rs) (let ((val (srss "(mix-position 0)"))) (if (/= val 100) (report-error "mix is at ~D, not 100?" val))))
  (if (rs) (let ((val (srss "(mix-length 0)"))) (if (/= val 41623) (report-error "mix len: ~D, not 41623?" val))))
  (if (rs) (let ((val (srss "(mix-anchor 0)"))) (if (/= val 0) (report-error "mix anchor is at ~D, not 0?" val))))
  (if (rs) (let ((val (srss "(mix-speed 0)"))) (if (/= val 1.0) (report-error "mix speed is ~F, not 1.0?" val))))
  (if (rs) (let ((val (srss "(mix-amp 0 0)"))) (if (/= val 1.0) (report-error "mix amp is ~F, not 1.0?" val))))
  (if (rs) (let ((val (srss "(mix-groups 0)"))) (if (/= val 0) (report-error "mix groups is ~D, not 0?" val))))
  (if (rs) (srss "(set-mix-position 0 200)"))
  (if (rs) (let ((val (srss "(mix-position 0)"))) (if (/= val 200) (report-error "mix is at ~D, not 200?" val))))
  (if (rs) (srss "(set-mix-speed 0 2.0)"))
  (if (rs) (let ((val (srss "(mix-speed 0)"))) (if (/= val 2.0) (report-error "mix speed is ~F, not 2.0?" val))))
  (if (rs) (srss "(set-mix-amp 0 0 .5)"))
  (if (rs) (let ((val (srss "(mix-amp 0 0)"))) (if (/= val 0.5) (report-error "mix amp is ~F, not 0.5?" val))))
  (setf two-aiff-index (or (srss (format nil "(open-sound ~S)" two-aiff)) -1))
  (srss (format nil "(close-sound ~D)" new-index))
  
  (if (and (zerop i-loops) (not libefence-loaded))
      (progn
	(srss "(peaks)")
	(srss "(describe-audio)")
	(srss "(add-mark 100)")
	(srss "(enved-dialog)")
	(srss "(color-dialog)")
	(srss "(orientation-dialog)")
	(srss "(transform-dialog)")
	(srss "(file-dialog)")
	(srss "(region-dialog)")
	(srss "(edit-header-dialog)")
	(srss "(recorder-dialog)")
	(srss "(group-dialog)")
	(srss "(help-dialog \"Test\" \"snd-test here\")"))
    (srss "(dismiss-all-dialogs)"))
  
  (sleep 2)

  (if (rs) (in-var-test "recorder-gain" .5 "0"))
  (if (rs) (in-var-test "recorder-in-amp" .25 "0" "0"))
  (if (rs) (in-var-test "recorder-out-amp" .8 "1"))
  
  (if (rs) (in-var-test "mix-groups" 1 "0"))
  (if (rs) (in-var-test "group-tempo" .5 "0"))
  (if (rs) (in-var-test "group-speed" 1.5 "0"))
  (if (rs) (in-var-test "group-beg" 0 "0"))
  (if (rs) (in-var-test "group-end" .1 "0"))
  (if (rs) (in-var-test "group-amp" .5 "0" "0"))
  
  (if (rs) (in-var-test "sample" .5 "0"))
  (if (rs) (in-var-test "cursor" 2130))
  (if (rs) (in-var-test "left-sample" 1000))
  (if (rs) (in-var-test "right-sample" 3000))


  (if randomized (setf rloops (1+ (random 10))) (setf rloops 1))
  (loop for rl from 0 below rloops do
    (if (rs) (srss "(forward-sample)"))
    (if (rs) (srss "(backward-sample)"))
    (if (rs) (srss "(forward-graph)"))
    (if (rs) (srss "(backward-graph)"))
    (if (rs) (srss "(forward-mark)"))
    (if (rs) (srss "(backward-mark)"))
    (if (rs) (srss "(forward-mix)"))
    (if (rs) (srss "(backward-mix)"))
    (if (rs) (srss "(regions)"))
    (if (rs) (srss "(region-length)"))
    (if (rs) (srss "(region-srate)"))
    (if (rs) (srss "(region-chans)"))
    (if (rs) (srss "(region-maxamp)"))
    (if (rs) (srss "(select-region 2)"))
    (if (rs) (srss "(protect-region 2 1)"))
    (if (rs) (srss "(play-region 2)"))
    (if (rs) (srss "(mix-region)"))
    (if (rs) (srss "(region-sample)"))
    (if (rs) (srss "(update-graph)"))
    (if (rs) (srss "(update-fft)"))
    (when (rs) 
      (srss "(play)")
      (sleep 3)
      (srss "(stop-playing)"))
    (if (rs) (srss "(groups)"))
    (if (rs) (srss "(group-ok? 0)"))
    (if (rs) (srss "(scale-to .1)"))
    (if (rs) (srss "(scale-by 2.0)"))
    (if (rs) (srss "(normalize-view)"))
    (if (rs) (srss "(transform-sample)"))
    )
  (when (and (not menu-added) (zerop i-loops))
    (setf menu-added t)
    (srss "(add-to-main-menu \"Test\")")
    (srss "(add-to-menu 5 \"Denoise\" \"(report-in-minibuffer \\\"denoise\\\")\")"))

  (srss "(report-in-minibuffer \"hi\")")
  (srss "(append-to-minibuffer \"ho\")")
  
  (srss (format nil "(close-sound ~D)" two-aiff-index))
  
  (when have-guile
    (if (rs)
	(progn
	  (send-snd "(load \"gh-snd-test.scm\")")
	  (receive-snd) 
	  (setf ob-snd-index (or (srss (format nil "(open-sound ~S)" ob-snd)) -1))
	  (srss "(set-ffting #t)")  
	  (srss "(set-x-bounds .1 .2)")
	  (setf gh-code-loaded t)
	  (srss "(add-hook! graph-hook display-energy)")
	  (srss "(shell \"df\")")
	  (srss "(copyfile)")
	  (srss "(reset-hook! graph-hook)")
	  (srss "(add-hook! graph-hook correlate)")
	  (srss "(set-x-bounds .1 .12)")
	  (srss (format nil "(scale-by 1.5 ~D 0)" ob-snd-index))
	  (srss "(set-x-bounds .1 .2)")
	  (srss "(remove-hook! graph-hook correlate)")
	  (srss "(set-graphing #f)")
	  (srss (format nil "(close-sound ~D)" ob-snd-index))
	  (setf ob-snd-index (srss "(open-sound \"pistol.snd\")"))
	  (srss "(map-chan (map-silence .01 #f))")
	  (srss "(every-sample? (lambda (y) (< y .5)))")
	  (srss "(scan-across-all-chans (data-max))")
	  (if have-clm (srss "(map-chan (echo .5 .75) 0 60000)"))
	  (srss (format nil "(save-sound-as \"pistol.1\" ~D)" ob-snd-index))
	  (srss (format nil "(revert-sound ~D)" ob-snd-index))
	  (srss (format nil "(close-sound ~D)" ob-snd-index))
	  )
      (progn
	(srss "(reset-hook! fft-hook)"))))
  
  (when (and (zerop i-loops) gh-code-loaded)
    #+linux (if (string-equal (clm::clm-getenv "HOSTNAME") "hummer.hiho") (setf header-dir "/space/home/bil/sf/") (setf header-dir "/home/bil/sf/"))
    #+linux (srss "(clear-audio-inputs)")
    #+sgi (setf header-dir "/usr/people/bil/sf/")
    #+sun (setf header-dir "/space/home/bil/sf/")
    
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "8svx-8.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "Fnonull.aif")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "Pmiscck.aif")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "Pmiscck.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "Pnossnd.aif")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "Poffset.aif")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "Porder.aif")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "Ptjunk.aif")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "Ptjunk.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "SINE24-S.WAV")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "a1.asf")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "a2.asf")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "addf8.afsp")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "addf8.d")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "addf8.dwd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "addf8.nh")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "addf8.sd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "addf8.sf_mipseb")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "addf8.sf_sun")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "addf8.sf_vax_b")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "addf8.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "aebass.krz")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "aiff-16.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "aiff-8.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "alaw.aifc")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "alaw.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "astor_basia.mp2")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "c.asf")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "ce-c3.w02")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "ce-c4.w03")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "ce-d2.w01")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "clbonef.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "cranker.krz")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "d40130.aif")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "d40130.au")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "d40130.dsf")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "d40130.fsm")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "d40130.iff")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "d40130.pat")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "d40130.sds")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "d40130.sdx")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "d40130.sf")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "d40130.smp")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "d40130.sou")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "d40130.st3")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "d40130.uwf")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "d40130.voc")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "d40130.w00")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "d40130.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "d43.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "digit0v0.aiff")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "esps-16.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "g721.au")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "g722.aifc")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "gong.wve")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "gsm610.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "hcom-16.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "ieee-text-16.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "inrs-16.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "kirk.wve")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "m.asf")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "markers.asf")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "mary-sun4.sig")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "midi60.mid")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "mocksong.ra")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "mocksong.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "mono24.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "msadpcm.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "mus10.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "n8.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "nasahal.aif")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "nasahal.avi")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "nasahal.dig")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "nasahal.ivc")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "nasahal.pat")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "nasahal.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "nasahal.svx")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "nasahal.v8")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "nasahal.voc")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "nasahal.vox")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "nasahal8.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "nasahalad.smp")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "next-16.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "next-8.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "next-dbl.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "next-flt.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "next-mulaw.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "next24.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "nist-01.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "nist-10.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "nist-16.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "nist-shortpack.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "none.aifc")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "nylon2.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2.avr")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2.bicsf")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2.mpeg1")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2.sd2")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2.sf2")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2.smp")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2.voc")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2.wave")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2_12bit.aiff")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2_18bit.aiff")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2_711u.wave")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2_722.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2_726.aiff")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2_726.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2_728.aiff")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2_8.iff")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2_8.voc")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2_dvi.wave")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2_float.bicsf")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2_gsm.aiff")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2_u8.avr")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "o2_u8.wave")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "oboe.g721")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "oboe.g723_24")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "oboe.g723_40")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "oboe.sf2")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "oki.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "orv-dvi-adpcm.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "riff-16.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "riff-8-u.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "rooster.wve")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "s.asf")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "sd1-16.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "segfault.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "sf-16.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "si654.adc")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "smp-16.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "sound.pat")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "sound.sap")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "sound.sds")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "sound.sfr")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "sound.v8")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "sound.vox")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "step.omf")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "step.qt")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "sun-16-afsp.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "sun-mulaw.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "sw1038t_short.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "swirl.pat")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "sy85.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "sy99.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "telephone.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "truspech.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "ulaw.aifc")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "voc-8-u.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "voxware.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wd.w00")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wd1.smp")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wd1.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wheel.mat")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood.dsf")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood.dvi")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood.dwd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood.fsm")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood.mad")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood.maud")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood.pat")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood.ra")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood.riff")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood.rifx")))
;    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood.sbk")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood.sds")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood.sdx")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood.sf")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood.sndr")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood.sndt")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood.st3")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood.uwf")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood.w00")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood12.aiff")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood16.dwd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood16.wav")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "wood24.aiff")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "woodblock.aiff")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "woodflt.snd")))
    (if (rs) (srss (format nil "(info \"~A~A\")" header-dir "zulu_a4.w11")))
    )
  
  (if randomized (setf rloops (1+ (random 10))) (setf rloops 1))
  (if (= i-loops (1- test-loops)) (srss "(show-listener)"))
  (loop for rl from 0 below rloops do
    
;;; now test some of the selection functions
    (setf ob-snd-index (or (srss (format nil "(open-sound ~S)" ob-snd)) -1))
    (when (rs)
      (srss "(make-region 1000 2000)")
      (srss "(src-selection .5)")
      (srss "(undo 1)"))
    (when (rs)
      (srss "(make-region 1000 2000)")
      (srss "(src-selection -1.5)")
      (srss "(undo 1)"))
    (when (rs)
      (srss "(make-region 1000 2000)")
      (srss "(scale-selection-by .5)")
      (srss "(undo 1)"))
    (when (rs)
      (srss "(make-region 1000 2000)")
      (if have-guile (srss "(env-selection '(0 0 1 1 2 0))"))
      (srss "(undo 1)"))
    (when (rs)
      (srss "(scale-selection-to .5)")
      (srss "(reverse-selection)")
      (srss "(undo 2)"))
    (if (and have-guile (rs)) (srss "(src-selection '(.5 0 1 1))"))

    (srss (format nil "(revert-sound ~D)" ob-snd-index))
    (srss (format nil "(close-sound ~D)" ob-snd-index))
    
    (setf ob-snd-index (or (srss (format nil "(open-sound ~S)" ob-snd)) -1))
    (srss (format nil "(set-syncing 1 ~D)" ob-snd-index))
    (setf two-snd-index (or (srss (format nil "(open-sound ~S)" two-snd)) -1))
    (srss (format nil "(set-syncing 1 ~D)" two-snd-index))
    (when (rs)
      (srss "(make-region 1000 2000)")
      (srss "(src-selection .5)")
      (srss "(undo 1)"))
    (when (rs)
      (srss "(make-region 1000 2000)")
      (srss "(src-selection -1.5)")
      (srss "(undo 1)"))
    (when (rs)
      (srss "(make-region 1000 2000)")
      (srss "(scale-selection-by .5)")
      (srss "(undo 1)"))
    (when (rs)
      (srss "(scale-selection-to .5)")
      (srss "(reverse-selection)")
      (srss "(undo 2)"))
    (if have-guile (srss "(src-selection '(.5 0 1 1))"))

    (srss "(undo)")
    (srss (format nil "(revert-sound ~D)" ob-snd-index))
    (srss (format nil "(close-sound ~D)" ob-snd-index))
    (srss (format nil "(revert-sound ~D)" two-snd-index))
    (srss (format nil "(close-sound ~D)" two-snd-index))
    
    (when (and have-guile (probe-file "fyow.snd"))
      (setf ob-snd-index (or (srss (format nil "(open-sound ~S)" ob-snd)) -1))
      (srss "(make-region 0 (frames))")
      (srss "(convolve-selection-with \"fyow.snd\" .5)")
      (srss "(play)") (sleep 3)
      ;(srss "(mark-segments)")
      ;(srss "(save-segments 24 .005 \"seg.temp\")")
      (srss (format nil "(revert-sound ~D)" ob-snd-index))
      (srss "(select-all)")
      (srss "(convolve-with \"fyow.snd\" .25)")
      (srss "(play)") (sleep 3)
      (srss "(insert-sound \"oboe.snd\")")
      (srss (format nil "(revert-sound ~D)" ob-snd-index))
      (srss (format nil "(close-sound ~D)" ob-snd-index)))
    )
  (srss "(reset-hook! graph-hook)")
  (srss "(reset-hook! fft-hook)")

  ;; try to get a segfault
  (setf z-snd-index (or (srss "(open-sound \"z.snd\")") -1))
  (srss "(mix \"4.aiff\")")
  (srss "(set-mix-amp-scaler .5)")
  (let ((val (srss "(mix-amp-scaler)"))) (print (format nil "mix amp scl: ~A" val)))
  (srss "(set-mix-speed-scaler .5)")
  (let ((val (srss "(mix-speed-scaler)"))) (print (format nil "mix speed scl: ~A" val)))
  (srss "(set-mix-tempo-scaler .5)")
  (let ((val (srss "(mix-tempo-scaler)"))) (print (format nil "mix tempo scl: ~A" val)))
  (srss "(set-mix-anchor 0 50)")
  (let ((val (srss "(mix-anchor 0)"))) (print (format nil "mix anchor: ~A" val)))
  (srss "(add-mark 0)")
  (srss "(add-mark 1200)")
  (srss "(delete-marks)")
  (srss "(revert-sound)")
  (srss "(mix \"s8.snd\" 24000)")
  (let ((val (srss "(frames)"))) (if (< val 1000) (report-error "mix is confused: ~D samples" val)))
  (srss "(revert-sound)")
  (srss "(select-all)")
  (if have-guile (srss "(filter-selection '(0 0 .2 1 .5 0 1 0) 40)"))
  (srss "(cut)")
  (srss "(mix-region)")
  (srss "(close-sound)")
  (setf s8-index (or (srss (format nil "(open-sound ~S)" s8-snd)) -1))
  (srss "(select-channel 5)")
  (let ((val (srss "(selected-channel)"))) (if (or (not (numberp val)) (/= val 5)) (report-error "selected-channel returns ~A rather than 5" val)))
  (srss "(select-all)")
  (srss "(cut)")
  (srss "(mix \"4.aiff\")")
  (srss "(set-syncing 1)")
  (srss "(mix \"oboe.snd\" 60000)")
  (srss "(scale-by .1)")
  (srss "(set-syncing 1)")
  (srss "(select-channel 3)")
  (srss "(insert-region 80000)")
  (srss "(revert-sound)")

  (if (= s8-index -1) (setf s8-index (or (srss "(open-sound \"oboe.snd\")") -1  )))
  (if (rs) (srss "(src-sound 2.5)"))
  (if (rs) (srss "(src-sound -2.5)"))
  (if (rs) (srss "(src-sound .5)"))
  (if (rs) (srss "(revert-sound)"))
  (if (rs) (srss "(src-sound -.5)"))
  (if (and have-guile (rs)) (srss "(src-sound '(0 .5 1 1.5))"))
  (if (rs) (srss "(revert-sound)"))
  (if (and have-guile (rs)) (srss "(filter-sound '(0 1 .2 0 .5 1 1 0) 20)"))
  (if (and have-guile (rs)) (srss "(env-sound '(0 0 .5 1 1 0))"))
  (if (rs) (srss "(insert-sample 1200 .1)"))
  (if (rs) (srss "(revert-sound)"))
  (let ((val (srss "(uniting)"))) (if (and (numberp val) (/= val 0)) (report-error "(uniting) returns ~A" val)))
  (let ((val (srss "(graphing)"))) (print (format nil "(graphing) returns ~A" val)))
  (if have-guile (let ((val (srss "(audio-error-name (audio-error))"))) (print (format nil "(audio-error) returns ~A" val))))
  (when have-guile
    (let ((val (srss "(sound-format-name (recorder-in-format))")))
      (print (format nil "(recorder-in-format) returns ~A" val)))
    (let ((val (srss "(sound-format-name (recorder-out-format))")))
      (print (format nil "(recorder-out-format) returns ~A" val))))
  (srss "(close-sound)")
  (srss "(hide-listener)")
  (let ((val (srss "(active-sounds)")))
    (if (or (not val) (/= val 0))
	(report-error "(active-sounds) returns ~A?" val)))

  (setf ob-snd-index (srss "(open-sound \"oboe.snd\")"))
  (setf obtest-index (srss (format nil "(open-alternate-sound \"obtest.snd\")")))
  (let ((val (srss "(find-sound \"oboe.snd\")")))
    (if (and val (numberp val) (/= val ob-snd-index)) (report-error "(find-sound \"oboe.snd\") returns ~A!" val)))
  (if have-guile
      (setf new-index (or (srss "(new-sound \"hiho.au\" next-sound-file snd-16-linear 22050 1)") -1))
    (setf new-index (or (srss "(new-sound \"hiho.au\" 0 1 22050 1)") -1)))
  (srss (format nil "(delete-samples 10 20 ~D)" obtest-index))
  (srss (format nil "(save-sound ~D)" obtest-index))
  (srss (format nil "(save-sound-as \"hiho.au\" ~D)" obtest-index))
  (srss (format nil "(update-sound ~D)" new-index))
  (srss (format nil "(revert-sound ~D)" obtest-index))
  (srss (format nil "(close-sound ~D)" new-index))
  (srss (format nil "(close-sound ~D)" obtest-index))

  (setf obtest-index (srss (format nil "(open-sound \"pistol.snd\")")))
  ;; now run apply a few times

    (srss "(set-amp .5)")
    (srss "(set-speed 2.0)")
    (srss "(call-apply)")
    (srss "(play)")
    (srss "(set-reverbing #t)")
    (srss "(set-reverb-scale .2)")
    (srss "(call-apply)")
    (srss "(play)")
    (srss "(set-contrasting #t)")
    (srss "(set-contrast .5)")
    (srss "(call-apply)")
    (srss "(play)")
    (srss "(set-expanding #t)")
    (srss "(set-expand 2.5)")
    (srss "(call-apply)")
    (srss "(play)")
    (when have-guile
      (srss "(set-filtering #t)")
      (srss "(set-filter-order 40)")
      ;; apparently there's a hiccup somewhere in the snd-list-to-clm path causing a premature return
      (send-snd "(set-filter-env '(0 0 .1 1 .2 0 1 0))")
      (sleep 1)
      (receive-snd)
      (send-snd "(filter-env)")
      (sleep 1)
      (let ((val (receive-snd))) (print (format nil "filter env is ~A" val)))
      (srss "(call-apply)")
      (srss "(play)"))
    (srss "(set-amp 1.5)")
    (srss "(call-apply)")
    (srss "(play)")
    (srss (format nil "(revert-sound ~D)" obtest-index))
    (srss (format nil "(close-sound ~D)" obtest-index))
  
  (incf i-loop)
  )

(when have-guile (srss "(gc)"))

(print "finished normally!!")
(when error-file 
  (close error-file)
  (setf error-file nil)
  (print "there were errors (see test.errors)..."))


