#include "snd.h"

#if defined(NEXT) || defined(HAVE_SYS_DIR_H)
  #include <sys/dir.h>
  #include <sys/dirent.h>
  #define dirent direct
#else
  #if defined(WINDOZE) && (!(defined(__CYGWIN__)))
    #include <direct.h>
  #else
    #include <dirent.h>
  #endif
#endif
      
#if defined(HAVE_SYS_STATFS_H)
  #include <sys/statfs.h>
#endif
#if defined(HAVE_SYS_VFS_H)
  #include <sys/vfs.h>
#endif
#if defined(HAVE_SYS_MOUNT_H)
  #include <sys/mount.h>
#endif

#if defined(WINDOZE) || (!(defined(FSTATFS_ARGS))) || (FSTATFS_ARGS == 0)
  int disk_kspace (int fd) {return(1234567);}
  time_t file_write_date(char *filename) {return(1234567);}
  int is_link(char *filename) {return(0);}
  int is_directory(char *filename) {return(0);}
#else

int disk_kspace (int fd)
{
  struct statfs buf;
  int err;
#if (FSTATFS_ARGS == 4)
  err = fstatfs(fd,&buf,sizeof(buf),0);
#else
  err = fstatfs(fd,&buf);
#endif
  /* in 32 bit land, the number of bytes can easily go over 2^32, so we'll look at kbytes here */
  if (err == 0) 
    {
#ifndef NEXT
      if (buf.f_bsize == 1024) return(buf.f_bfree);
      else if (buf.f_bsize == 512) return(buf.f_bfree >> 1);
      else return((int)(buf.f_bsize * ((float)(buf.f_bfree)/1024.0)));
#else
      return(buf.f_bavail);
#endif
    }
  return(err);
}

time_t file_write_date(char *filename)
{
  struct stat statbuf;
  int err;
  err = stat(filename,&statbuf);
  if (err < 0) return(err);
  return((time_t)(statbuf.st_mtime));
}

int is_link(char *filename)
{
  struct stat statbuf;
  if (lstat(filename,&statbuf) >= 0) return(S_ISLNK(statbuf.st_mode));
  return(0);
}

int is_directory(char *filename)
{
  struct stat statbuf;
  if (lstat(filename,&statbuf) >= 0) return(S_ISDIR(statbuf.st_mode));
  return(0);
}

#if 0
static int is_regular_file(char *filename)
{
  struct stat statbuf;
  if ((lstat(fullpath,&statbuf) >= 0) && (S_ISDIR(statbuf.st_mode) == 0))
    return((statbuf.st_mode & S_IFMT) == S_IFREG);
  return(0);
}
#endif
#endif

file_info *make_file_info_1(char *fullname, snd_state *ss)
{
  int len,fd;
  file_info *hdr;
  fd = open(fullname,O_RDONLY,0);
  if (fd == -1)
    {
      snd_error("%s: %s",fullname,strerror(errno));
      return(NULL);
    }
  else
    close(fd);  
  hdr = (file_info *)CALLOC(1,sizeof(file_info));
  hdr->s_type = FILE_INFO;
  len=strlen(fullname);
  hdr->name = (char *)CALLOC(len+1,sizeof(char));
  strcpy(hdr->name,fullname);
  hdr->type = sound_header_type(fullname);
  if ((hdr->type == raw_sound_file) && (use_raw_defaults(ss)))
    {
      hdr->srate = raw_srate(ss);
      hdr->format = raw_format(ss);
      hdr->chans = raw_chans(ss);
      mus_set_raw_header_defaults(raw_srate(ss),raw_chans(ss),raw_format(ss));
    }
  else
    {
      hdr->srate = sound_srate(fullname);
      if ((hdr->srate) == 0) hdr->srate = 1;
      hdr->chans = sound_chans(fullname);
      if ((hdr->chans) == 0) hdr->chans = 1;
      hdr->format = sound_data_format(fullname);
    }
  hdr->samples = sound_samples(fullname); /* total samples, not per channel */
  hdr->data_location = sound_data_location(fullname);
  hdr->comment = NULL;
  return(hdr);
}

file_info *copy_header(char *fullname, file_info *ohdr)
{
  int len;
  file_info *hdr;
  hdr = (file_info *)CALLOC(1,sizeof(file_info));
  hdr->s_type = FILE_INFO;
  len=strlen(fullname);
  hdr->name = (char *)CALLOC(len+1,sizeof(char));
  strcpy(hdr->name,fullname);
  hdr->comment = NULL;
  hdr->samples = ohdr->samples;
  if (ohdr)
    {
      hdr->data_location = ohdr->data_location;
      hdr->srate = ohdr->srate;
      hdr->chans = ohdr->chans;
      hdr->format = ohdr->format;
      hdr->type = ohdr->type;
    }
  return(hdr);
}

static file_info *translate_file(char *filename, snd_state *ss)
{
  char *newname,*tempname;
  file_info *hdr = NULL;
  int err,len,fd;
  len = strlen(filename);
  newname = (char *)CALLOC(len+5,sizeof(char));
  strcpy(newname,filename);
  /* too many special cases to do anything smart here -- I'll just tack on '.snd' */
  newname[len]='.';
  newname[len+1]='s';
  newname[len+2]='n';
  newname[len+3]='d';
  newname[len+4]='\0';
  /* before calling the translator we need to check for write-access to the current directory,
   * and if none, try to find a temp directory we can write to
   */
  fd = creat(newname,0666);
  if (fd == -1)
    {
      tempname = snd_tempnam(ss);
      fd = creat(tempname,0666);
      if (fd == -1)
	{
	  snd_error(STR_cant_write_translator_file,newname,tempname);
	  return(NULL);
	}
      FREE(newname);
      newname = tempname;
    }
  close(fd);
  err = snd_translate(ss,filename,newname);
  if (err == 0)
    {
      err = mus_read_header(newname);
      if (err == 0)
	{
	  hdr = make_file_info_1(newname,ss);
	  if (hdr) ss->pending_change = newname;
	}
    }
  return(hdr);
}

static file_info *make_file_info_2(char *fullname, snd_state *ss)
{
  int type,format,fd;
  fd = open(fullname,O_RDONLY,0);
  if (fd != -1)
    {
      close(fd);  
      type = sound_header_type(fullname);
      format = sound_data_format(fullname);
      if ((sound_srate(fullname) <= 0) || (sound_srate(fullname) > 100000000) ||
	  (sound_chans(fullname) >= 256) || (sound_chans(fullname) <= 0))
	{
	  if ((type != MIDI_sample_dump) && (type != IEEE_sound_file) &&
	      (type != MUS10_sound_file) && (type != HCOM_sound_file))
	    return(get_reasonable_file_info(fullname,ss,make_file_info_1(fullname,ss)));
	}
      if (format == SNDLIB_UNSUPPORTED) return(translate_file(fullname,ss));
      if (type == raw_sound_file)
	{
	  /* need type format chans srate (latter 2 need to over-ride current for hdr build below) */
	  return(get_file_info(fullname,ss));
	  /* this can become a pending request -- when user clicks "ok" we go on */
	}
      return(make_file_info_1(fullname,ss));
    }
  set_snd_IO_error(SND_CANNOT_FIND_FILE);
  return(NULL);
}

file_info *make_file_info(char *fullname, snd_state *ss)
{
  file_info *hdr;
  hdr = make_file_info_2(fullname,ss);
  if (!hdr) 
    {
      if (snd_IO_error() != SND_PENDING_OPEN)
	{
	  if (snd_IO_error() != SND_NO_ERROR)
	    {
	      if (!(selected_sound(ss)))
		{
		  if (errno != 0)
		    snd_error("%s:\n  %s\n  (%s)",fullname,strerror(errno),snd_error_name(snd_IO_error()));
		  else snd_error("%s:\n  %s",fullname,snd_error_name(snd_IO_error()));
		}
	      else
		{
		  if (errno != 0)
		    snd_error("%s: %s (%s)",fullname,strerror(errno),snd_error_name(snd_IO_error()));
		  else snd_error("%s: %s",fullname,snd_error_name(snd_IO_error()));
		}
	    }
	}
    }
  return(hdr);
}

file_info *make_temp_header(char *fullname, file_info *old_hdr, int samples)
{
  /* make a header for Sun/NeXT, 16-bit linear, no comment, copy other old_hdr fields */
  file_info *hdr;
  hdr = (file_info *)CALLOC(1,sizeof(file_info));
  hdr->s_type = FILE_INFO;
  hdr->name = (char *)CALLOC(strlen(fullname)+1,sizeof(char));
  strcpy(hdr->name,fullname);
  hdr->samples = samples;
  hdr->data_location = 28;
  hdr->srate = old_hdr->srate;
  hdr->chans = old_hdr->chans;
  hdr->format = SNDLIB_16_LINEAR;
  hdr->type = NeXT_sound_file;
  hdr->comment = NULL;
  return(hdr);
}

file_info *free_file_info(file_info *hdr)
{
  if (hdr)
    {
      if (hdr->name) FREE(hdr->name);
      if (hdr->comment) FREE(hdr->comment);
      FREE(hdr);
    }
  return(NULL);
}

int *make_file_state(int fd, file_info *hdr, int direction, int chan, int suggested_bufsize, snd_state *ss)
{
  int *datai;
  int i,bufsize,chansize;
  bufsize = suggested_bufsize;
  if (direction == SND_IO_IN_FILE)
    {
      chansize = (hdr->samples/hdr->chans);
      if (MAX_BUFFER_SIZE > chansize) bufsize = chansize+1;
    }
  else chansize = 0;
  datai = (int *)CALLOC(SND_IO_DATS + SND_AREF_HEADER_SIZE + hdr->chans + 2,sizeof(int)); /* why the +2?? */
  datai[SND_IO_DIR] = direction;
  datai[SND_IO_FD] = fd;
  datai[SND_IO_CHANS] = hdr->chans;
  datai[SND_IO_SIZE] = chansize;
  datai[SND_IO_BEG] = 0;
  datai[SND_IO_END] = bufsize-1;
  datai[SND_IO_BUFSIZ] = bufsize;
  datai[SND_IO_DATA_START] = 0; 
  datai[SND_IO_DATA_END] = 0;
  datai[SND_IO_HDR_END] = hdr->data_location; 
  datai[SND_IO_DATS+SND_AREF_BLOCK]=SND_IO_DATS + SND_AREF_HEADER_SIZE;
  datai[SND_IO_DATS+SND_AREF_SIZE]=hdr->chans;
  if (direction == SND_IO_IN_FILE)
    {
#if LONG_INT_P
      datai[SND_IO_DATS + SND_AREF_HEADER_SIZE + chan] = list_ptr((int *)CALLOC(bufsize,sizeof(int)));
#else
      datai[SND_IO_DATS + SND_AREF_HEADER_SIZE + chan] = (int)CALLOC(bufsize,sizeof(int));
#endif
    }
  else
    {
      for (i=0;i<hdr->chans;i++) 
#if LONG_INT_P
	datai[SND_IO_DATS + SND_AREF_HEADER_SIZE+i] = list_ptr((int *)CALLOC(bufsize,sizeof(int)));
#else
	datai[SND_IO_DATS + SND_AREF_HEADER_SIZE+i] = (int)CALLOC(bufsize,sizeof(int));
#endif
    }
  if (direction == SND_IO_IN_FILE) mus_file_reset(0,datai,datai); /* get ready to read -- we're assuming mus_read_chans here */
  return(datai);
}

int *free_file_state(int *datai)
{
  /* gotta free the IO buffers as well as the descriptor buffer */
  int i,chans;
  if (datai)
    {
      chans = datai[SND_IO_CHANS];
      for (i=0;i<chans;i++)
	{
#if LONG_INT_P
	  if (datai[SND_IO_DATS + SND_AREF_HEADER_SIZE+i]) freearray(datai[SND_IO_DATS + SND_AREF_HEADER_SIZE+i]);
#else
	  if (datai[SND_IO_DATS + SND_AREF_HEADER_SIZE+i]) FREE((int *)(datai[SND_IO_DATS + SND_AREF_HEADER_SIZE+i]));
#endif
	}
      FREE(datai);
    }
  return(NULL);
}

/* mus_read_header here (or stripped-down equivalent) was very slow, and is just as easy to
 * fool as an extension check (file might start with the word ".snd" or whatever).  Also
 * the scanning functions in unix.c descend directories, and set up complete header records,
 * neither of which is needed at this point, so ...
 */

static dir *make_dir (char *name)
{
  dir *dp;
  dp = (dir *)CALLOC(1,sizeof(dir));
  dp->files = (char **)CALLOC(32,sizeof(char *));
  dp->name = (char *)CALLOC(strlen(name)+1,sizeof(char));
  strcpy(dp->name,name);
  dp->len = 0;
  dp->size = 32;
  return(dp);
}

dir *free_dir (dir *dp)
{
  int i;
  if (dp->name) FREE(dp->name);
  if (dp->files)
    {
      for (i=0;i<dp->len;i++) {if (dp->files[i]) FREE(dp->files[i]);}
    }
  FREE(dp);
  return(NULL);
}
  
static void add_snd_file_to_dir_list(dir *dp, char *name)
{
  int i;
  dp->files[dp->len] = copy_string(name);
  dp->len++;
  if (dp->len == dp->size) 
    {
      dp->size += 32;
      dp->files = (char **)REALLOC(dp->files,dp->size*sizeof(char *));
      for (i=dp->size-32;i<dp->size;i++) dp->files[i] = NULL;
    }
}

static char **sound_file_extensions = NULL;
static int sound_file_extensions_size = 0;
static int sound_file_extensions_end = 0;

void add_sound_file_extension(char *ext)
{
  if (sound_file_extensions_end == sound_file_extensions_size)
    {
      sound_file_extensions_size += 8;
      if (sound_file_extensions == NULL)
	sound_file_extensions = (char **)CALLOC(sound_file_extensions_size,sizeof(char *));
      else sound_file_extensions = (char **)REALLOC(sound_file_extensions,sound_file_extensions_size * sizeof(char *));
    }
  sound_file_extensions[sound_file_extensions_end] = copy_string(ext);
  sound_file_extensions_end++;
}

void init_sound_file_extensions(void)
{
  add_sound_file_extension("snd");
  add_sound_file_extension("aiff");
  add_sound_file_extension("aif");
  add_sound_file_extension("wav");
  add_sound_file_extension("au");
  add_sound_file_extension("aifc");
  add_sound_file_extension("voc");
  add_sound_file_extension("wve");
}

dir *find_sound_files_in_dir (char *name)
{
#if defined(_MSC_VER)
  return(NULL);
#else
  struct dirent *dirp;
  DIR *dpos;
  char *dot,*sp;
  dir *dp = NULL;
  int i;
  if ((dpos=opendir(name)) != NULL)
    {
      dp = make_dir(name);
      while ((dirp=readdir(dpos)) != NULL)
	{
	  if (dirp->d_name[0] != '.')
	    {
	      dot = NULL;
	      for (sp=dirp->d_name;(*sp) != '\0';sp++) if ((*sp) == '.') dot=(++sp);
	      if (dot)
		{
		  for (i=0;i<sound_file_extensions_end;i++)
		    {
		      if (strcmp(dot,sound_file_extensions[i]) == 0)
			{
			  add_snd_file_to_dir_list(dp,dirp->d_name);
			  break;
			}
		    }
		}
	    }
	}
#if defined(CLOSEDIR_VOID)
      closedir(dpos);
#else
      if (closedir(dpos) != 0) snd_error("%s[%d] %s: closedir failed!",__FILE__,__LINE__,__FUNCTION__);
#endif
    }
  return(dp);
#endif
}

static int names_match(char *filename, char *pattern)
{
  /* just "*" for wildcards here */
  char *sn,*sp;
  sn = filename;
  sp = pattern;
  if ((!sn) || (!sp)) {if ((sn) || (sp)) return(0); else return(1);}
  while ((*sn) && (*sp))
    {
      if ((*sp) == '*') 
	{
	  sp++; 
	  while ((*sp) == '*') {sp++;} 
	  if (!(*sp)) return(1);
	  while ((*sn) && ((*sn) != (*sp))) {sn++;}
	  if (!(*sn)) return(0);
	}
      else 
	{
	  if ((*sn) != (*sp)) return(0);
	  sn++; sp++;
	}
    }
  return(1);
}

dir *filter_sound_files(dir *dp, char *pattern)
{
  int i;
  dir *ndp;
  ndp = make_dir("");
  for (i=0;i<dp->len;i++)
    {
      if (names_match(dp->files[i],pattern)) {add_snd_file_to_dir_list(ndp,dp->files[i]);}
    }
  return(ndp);
}

static int incf_active_sounds (snd_state *ss) {ss->active_sounds++; return(ss->active_sounds);}
static int decf_active_sounds (snd_state *ss) {ss->active_sounds--; return(ss->active_sounds);}

typedef struct {
  int active_sounds;
  char **names;
  int *sounds;
} active_sound_list;

static int add_sound_to_active_list (snd_info *sp, void *sptr1)
{
  active_sound_list *sptr = (active_sound_list *)sptr1;
  sptr->names[sptr->active_sounds] = sp->fullname;
  sptr->sounds[sptr->active_sounds] = sp->index;
  (sptr->active_sounds)++;
  return(0); /*assume no problem -- nothing can go wrong! */
}

static char title_buffer[4*(SNDLIB_MAX_FILE_NAME)];

static void reflect_file_change_in_title(snd_state *ss)
{
  active_sound_list *alist;
  int i,j,k;
  char *s1 = NULL,*s2;
  alist = (active_sound_list *)CALLOC(1,sizeof(active_sound_list));
  alist->sounds = (int *)CALLOC(ss->max_sounds,sizeof(int));
  alist->names = (char **)CALLOC(ss->max_sounds,sizeof(char *));
  map_over_sounds(ss,add_sound_to_active_list,alist);
  sprintf(title_buffer,"%s%s",ss->startup_title,((alist->active_sounds > 0) ? ": " : ""));
  k = strlen(title_buffer);
  if (alist->active_sounds > 0)
    {
      if (alist->active_sounds < 4) j=alist->active_sounds; else j=4;
      s1 = (char *)(title_buffer+k);
      for (i=0;i<j;i++)
	{
	  for (s2 = filename_without_home_directory(alist->names[i]);(*s1 = *s2) != '\0';++s1,++s2);
	  if (i<j-1) {*s1 = ','; s1++;  *s1 = ' '; s1++;}
	}
      if (alist->active_sounds>4) {for (i=0;i<3;i++) {*s1='.'; s1++;}}
    }
  set_title(ss,title_buffer);
  FREE(alist->sounds);
  FREE(alist->names);
  FREE(alist);
}

static char *memo_file_name(snd_info *sp)
{
  char *newname;
  int len;
  len = strlen(sp->fullname);
  newname = (char *)CALLOC(len+5,sizeof(char));
  strcpy(newname,sp->fullname);
  newname[len]='.'; newname[len+1]='s'; newname[len+2]='c'; newname[len+3]='m'; newname[len+4]='\0'; 
  return(newname);
}

static void read_memo_file(snd_info *sp)
{
  /* sp->fullname + ".scm" = possible memo file */
  char *newname;
  newname = memo_file_name(sp);
  if (file_write_date(newname) >= sp->write_date)
    snd_load_file(sp->state,newname);
  FREE(newname);
}


static snd_info *snd_open_file_1 (char *filename, snd_state *ss, int select)
{
  snd_info *sp;
  int files,val;
#if HAVE_GUILE
  if (dont_open(ss,filename)) return(NULL);
#endif
  sp = add_sound_window(mus_complete_filename(filename),ss); /* snd-xsnd.c -> make_file_info */
  if (sp)
    {
#if HAVE_GUILE
      after_open(sp);
#endif
      sp->write_date = file_write_date(sp->fullname);
      sp->need_update = 0;
      if (ss->viewing) sp->read_only = 1;
      files = incf_active_sounds(ss);
      if (files == 1) reflect_file_open_in_menu();
      set_normalize_option(active_channels(ss,0) > 1);
      reflect_file_change_in_title(ss);
      unlock_ctrls(sp);
      greet_me(ss,sp->shortname);
      check_group_max_out_chans(ss,sp->nchans);
    }
  map_over_separate_chans(ss,channel_open_pane,NULL);
  map_over_separate_chans(ss,channel_unlock_pane,NULL);
  ss->viewing = 0;
  if (sp) 
    {
      if (select) select_channel(sp,0);
      read_memo_file(sp);
      if ((sp->combining != CHANNELS_SEPARATE) && (sp->nchans > 1)) 
	{
	  val = sp->combining;
	  sp->combining = CHANNELS_SEPARATE; 
	  if (val == CHANNELS_COMBINED)
	    combine_sound(sp);
	  else superimpose_sound(sp);
	}
    }
  return(sp);
}

snd_info *snd_open_file (char *filename, snd_state *ss) {return(snd_open_file_1(filename,ss,TRUE));}
snd_info *snd_open_file_unselected (char *filename, snd_state *ss) {return(snd_open_file_1(filename,ss,FALSE));}


void snd_close_file(snd_info *sp, snd_state *ss)
{
  int files;
#if HAVE_GUILE
  if (dont_close(ss,sp)) return;
#endif
  sp->inuse = 0;
  remember_me(ss,sp->shortname,sp->fullname);
  if (sp->playing) stop_playing_sound(sp);
  clear_minibuffer(sp);
  if ((region_ok(0)) && (selection_member(sp))) 
    {
      cancel_keyboard_selection();
      deactivate_selection();
    }
  if (sp == selected_sound(ss)) ss->selected_sound = NO_SELECTION;
  free_snd_info(sp);
  files = decf_active_sounds(ss);
  if (files == 0) reflect_file_lack_in_menu();
  reflect_file_change_in_title(ss);
  set_normalize_option(active_channels(ss,0) > 1);
}


int copy_file(snd_state *ss, char *oldname, char *newname)
{
  /* make newname a copy of oldname */
  int ifd,ofd;
  long bytes,wb,total;
  char *buf = NULL;
  total = 0;
  ifd = open(oldname,O_RDONLY,0);
  if (ifd == -1) return(SND_CANNOT_FIND_FILE);
  ofd = open(newname,O_RDWR,0);
  if (ofd == -1)
    {
      ofd = creat(newname,0666);
      if (ofd == -1) {close(ifd); return(SND_CANNOT_WRITE_DATA);}
    }
  buf = (char *)CALLOC(8192,sizeof(char));
  while ((bytes = read(ifd,buf,8192)))
    {
      total += bytes;
      wb = write(ofd,buf,bytes);
      if (wb != bytes) {close(ofd); close(ifd); FREE(buf); return(SND_CANNOT_WRITE_DATA);}
    }
  close(ifd);
  total = total >> 10;
  wb = disk_kspace(ofd);
  if (wb < 0) 
    snd_error(strerror(errno));
  else
    if (total > wb) snd_error(STR_were_getting_short_on_disk_space,(int)total,(int)wb);
  FREE(buf);
  close(ofd);
  return(SND_NO_ERROR);
}

snd_info *make_sound_readable(snd_state *ss, char *filename, int post_close)
{
  /* conjure up just enough Snd structure to make this sound readable by the edit-tree readers */
  snd_info *sp;
  chan_info *cp;
  file_info *hdr;
  snd_data *sd;
  int *datai;
  int i,fd,len;
  hdr = make_file_info_1(filename,ss);
  if (!hdr) return(NULL);
  sp = (snd_info *)CALLOC(1,sizeof(snd_info));
  sp->s_type = SND_INFO;
  sp->nchans = sound_chans(filename);
  sp->allocated_chans = sp->nchans;
  sp->chans = (chan_info **)CALLOC(sp->nchans,sizeof(chan_info *));
  sp->hdr = hdr;
  sp->inuse = 1;
  sp->state = ss;
  sp->actions = NULL;
  sp->expand = 1.0;
  sp->expanding = 0;
  sp->amp = 1.0;
  sp->srate = 1.0;
  sp->play_direction = 1;
  sp->contrasting = 0;
  sp->contrast = 0.0;
  sp->reverbing = 0;
  sp->revscl = 0.0;
  sp->filtering = 0;
  len = (hdr->samples)/(hdr->chans);
  for (i=0;i<sp->nchans;i++)
    {
      cp = make_chan_info(NULL,i,sp,ss);
      FREE((cp->cgx)->ax);
      FREE(cp->cgx);
      cp->cgx = NULL;
      sp->chans[i] = cp;
      add_channel_data_1(cp,sp,ss,0);
      cp->edits[0] = initial_ed_list(0,len-1);
      cp->edit_size = 1;
      cp->sound_size = 1;
      fd = snd_open_read(ss,filename);
      mus_open_file_descriptors(fd,hdr->format,sound_datum_size(filename),hdr->data_location);
      datai = make_file_state(fd,hdr,SND_IO_IN_FILE,i,(post_close) ? MAX_BUFFER_SIZE : MIX_FILE_BUFFER_SIZE,ss);
#if LONG_INT_P
      cp->sounds[0] = make_snd_data_file(filename,datai,
					 delist_ptr(datai[SND_IO_DATS + SND_AREF_HEADER_SIZE+i]),
					 copy_header(hdr->name,hdr),
					 0,cp->edit_ctr,i);
#else
      cp->sounds[0] = make_snd_data_file(filename,datai,
					 (int *)(datai[SND_IO_DATS + SND_AREF_HEADER_SIZE+i]),
					 copy_header(hdr->name,hdr),
					 0,cp->edit_ctr,i);
#endif
      if (post_close) {snd_close(fd); sd = cp->sounds[0]; sd->open = FD_CLOSED; datai[SND_IO_FD] = -1;}
      /* this is not as crazy as it looks -- we've read in the first 64K (or whatever) samples,
       * and may need this file channel for other opens, so this file can be closed until mus_file_reset
       */
    }
  return(sp);
}

static snd_info *snd_update_1(snd_state *ss, snd_info *sp, char *ur_filename)
{
  /* we can't be real smart here because the channel number may have changed and so on */
  float *axis_data;
  int *ffts,*waves;
  int i,j,old_chans,old_sync,old_combine,need_update;
  int raw_def;
  float duration;
  chan_info *cp;
  axis_info *ap;
  snd_info *nsp;
  char *filename;
  need_update = 0;
  filename = copy_string(ur_filename);
  old_chans = sp->nchans;
  old_sync = sp->syncing;
  old_combine = sp->combining;
  axis_data = (float *)CALLOC(4*old_chans,sizeof(float));
  ffts = (int *)CALLOC(old_chans,sizeof(int));
  waves = (int *)CALLOC(old_chans,sizeof(int));
  for (i=0;i<old_chans;i++)
    {
      cp = sp->chans[i];
      ap = cp->axis;
      axis_data[(i*4)+0]=ap->x0;
      axis_data[(i*4)+1]=ap->x1;
      axis_data[(i*4)+2]=ap->y0;
      axis_data[(i*4)+3]=ap->y1;
      waves[i] = cp->waving;
      ffts[i] = cp->ffting;
    }
  snd_close_file(sp,ss);
  /* this normalizes the fft/lisp/wave state so we need to reset it after reopen */
  alert_new_file();
  /* if it's a raw sound file being updated, we don't want to re-confirm the sound format and whatnot
   * so we set the use-raw-defaults flag in a sort of ugly wrapper around the snd_open_file
   */
  raw_def = use_raw_defaults(ss);
  set_use_raw_defaults(ss,1);
  nsp = snd_open_file(filename,ss);
  set_use_raw_defaults(ss,raw_def);
  /* end wrapper */
  duration = (float)sound_samples(filename) / (float)(sound_chans(filename) * sound_srate(filename));
  for (i=0,j=0;i<nsp->nchans;i++)
    {
      cp = nsp->chans[i];
      if (duration < axis_data[(j*4)+0]) axis_data[(j*4)+0]=duration-.1;
      if (duration < axis_data[(j*4)+1]) axis_data[(j*4)+1]=duration;
      set_axes(cp,axis_data[(j*4)+0],axis_data[(j*4)+1],axis_data[(j*4)+2],axis_data[(j*4)+3]);
      update_graph(cp,NULL); /* get normalized state before messing with it */
      if (ffts[j]) {fftb(cp,TRUE); need_update = 1;}
      if (!(waves[j])) {waveb(cp,FALSE); need_update = 1;}
      if (j<(old_chans-1)) j++;
    }
  if (nsp->combining != old_combine) combineb(nsp,old_combine);
  if (nsp->syncing != old_sync) syncb(nsp,old_sync);
  if (need_update) {for (i=0;i<nsp->nchans;i++) update_graph(nsp->chans[i],NULL);}
  FREE(axis_data);
  FREE(waves);
  FREE(ffts);
  FREE(filename);
  return(nsp);
}

void snd_update(snd_state *ss, snd_info *sp)
{
  char *buf;
  if (sp->edited_region) return;
  if ((snd_probe_file(ss,sp->fullname)) == FILE_DOES_NOT_EXIST)
    {
      /* user deleted file while editing it? */
      buf = (char *)CALLOC(256,sizeof(char));
      sprintf(buf,STR_no_longer_exists,sp->shortname);
      report_in_minibuffer(sp,buf);
      FREE(buf);
      return;
    }
  save_window_size(ss);
  sp = snd_update_1(ss,sp,sp->fullname);
  buf = (char *)CALLOC(64,sizeof(char));
  sprintf(buf,STR_updated,sp->shortname);
  report_in_minibuffer(sp,buf);
  FREE(buf);
  restore_window_size(ss);
}

char *filename_completer(char *text)
{
#if (!(defined(_MSC_VER)))
  /* assume text is a partial filename */
  /* get directory name, opendir, read files checking for match */
  /* return name of same form as original (i.e. don't change user's directory indication) */
  /* if directory, add "/" -- is_directory(name) static in snd-xfile.c */
  char *full_name = NULL,*dir_name = NULL,*file_name = NULL,*current_match = NULL;
  int i,j,k,len,curlen,matches = 0;
  struct dirent *dirp;
  DIR *dpos;

  full_name = mus_complete_filename(text); /* don't free! */
  len = snd_strlen(full_name);
  for (i=len-1;i>0;i--)
    if (full_name[i] == '/')
      break;

  dir_name = (char *)CALLOC(i+1,sizeof(char));
  strncpy(dir_name,full_name,i);
  file_name = (char *)CALLOC(len-i+2,sizeof(char));
  for (j=0,k=i+1;k<len;j++,k++) file_name[j] = full_name[k];
  len = snd_strlen(file_name);
  if ((dpos=opendir(dir_name)) != NULL)
    {
      while ((dirp=readdir(dpos)) != NULL)
	{
	  if (dirp->d_name[0] != '.')
	    {
	      /* match dirp->d_name against rest of text */
	      if (strncmp(dirp->d_name,file_name,len) == 0)
		{
		  matches++;
		  add_possible_completion(dirp->d_name);
		  if (current_match == NULL)
		    current_match = copy_string(dirp->d_name);
		  else 
		    {
		      curlen = strlen(current_match);
		      for (j=0;j<curlen;j++)
			if (current_match[j] != dirp->d_name[j])
			  {
			    current_match[j] = '\0';
			    break;
			  }
		      }
		}
	    }
	}
#if defined(CLOSEDIR_VOID)
      closedir(dpos);
#else
      if (closedir(dpos) != 0) snd_error("%s[%d] %s: closedir failed!",__FILE__,__LINE__,__FUNCTION__);
#endif
    }
  if (dir_name) FREE(dir_name);
  if (file_name) FREE(file_name);
  set_completion_matches(matches);
  if ((current_match) && (*current_match))
    {
      /* attach matched portion to user's indication of dir */
      len = snd_strlen(text);
      for (i=len-1;i>=0;i--)
	if (text[i] == '/')
	  break;
      if (i < 0) return(current_match);
      curlen = strlen(current_match) + len + 3;
      file_name = (char *)CALLOC(curlen,sizeof(char));
      strncpy(file_name,text,i+1);
      strcat(file_name,current_match);
      if (is_directory(file_name)) strcat(file_name,"/");
      FREE(current_match);
      return(file_name);
    }
#endif
  return(copy_string(text));
}
