/*================================================================
 * General Pipe and Tcl/Tk interface mode
 *
 * Copyright (C) 1996-1998 Takashi Iwai
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *================================================================*/

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include "config.h"
#ifdef INCLUDE_TK_MODE
#include "trace.h"
#endif
#include "util.h"
#include "sighandle.h"
#include "channel.h"
#include "midievent.h"
#include "controls.h"
#include "seq.h"
#include "ext_c.h"
#include "options.h"

static void ctl_update(MidiInfo *mp);
static void ctl_total_time(int tt);
static void ctl_file_name(char *name);
static void ctl_title(char *name);
static void ctl_current_time(int ct);
static void ctl_note(int ch, int note, int vel);
static void ctl_program(int ch, int val);
static void ctl_control_change(int channel, int type, int val);
static void ctl_pitch_bend(int channel, int val);
static void ctl_bank(int ch, int val);
static void ctl_system_change(int type, int mode);
static void ctl_effects_change(int type, int ch, int val);
static void ctl_master_vol(int atten);

static void ctl_reset(MidiInfo *mp);
static void ctl_pitch_sense(int ch, int val);
static int ctl_read(MidiInfo *mp, int *valp);
static int cmsg(int type, int verbosity_level, char *fmt, ...);

static int ctl_blocking_read(MidiInfo *mp, int *valp);
static void pipe_printf(char *fmt, ...);
static void pipe_puts(char *str);
static int pipe_gets(char *str, int maxlen);
static int pipe_read_ready();
static void start_midi(void);

#ifdef INCLUDE_TK_MODE
static int tk_open(int using_stdin, int using_stdout);
static void tk_close(void);
static void tk_pass_playing_list(int number_of_files, char *list_of_files[]);
#endif

#ifdef INCLUDE_PIPE_MODE
static int pipe_open(int using_stdin, int using_stdout);
static void pipe_close(void);
static void pipe_pass_playing_list(int number_of_files, char *list_of_files[]);
#endif

/*----------------------------------------------------------------
 * shared parameters
 *----------------------------------------------------------------*/

PanelInfo *extpanel;

int gen_shmid = -1;
int pipe_in = -1, pipe_out = -1;


/*----------------------------------------------------------------
 * control interface
 *----------------------------------------------------------------*/

#ifdef INCLUDE_PIPE_MODE
ControlMode pipe_control_mode =
{
	"general pipe interface", 'p',
	FALSE, /* need_args */
	TRUE, /* need_sync */
	TRUE, /* fancy_prev */
	0, 0, 0, 0, /* verbose, trace, open, repeat */
	pipe_open, pipe_pass_playing_list, pipe_close,
	ctl_read, ctl_blocking_read, cmsg,
	ctl_update, ctl_reset, ctl_file_name, ctl_title, ctl_total_time,
	ctl_current_time, ctl_note, ctl_program, ctl_bank, ctl_control_change,
	ctl_pitch_bend, ctl_pitch_sense, ctl_system_change,
	ctl_effects_change, ctl_master_vol,
};
#endif

#ifdef INCLUDE_TK_MODE
ControlMode tk_control_mode =
{
	"Tcl/Tk interface", 'k',
	FALSE, /* need_args */
	TRUE, /* need_sync */
	TRUE, /* fancy_prev */
	0, 0, 0, 0, /* verbose, trace, open, repeat */
	tk_open, tk_pass_playing_list, tk_close,
	ctl_read, ctl_blocking_read, cmsg,
	ctl_update, ctl_reset, ctl_file_name, ctl_title, ctl_total_time,
	ctl_current_time, ctl_note, ctl_program, ctl_bank, ctl_control_change,
	ctl_pitch_bend, ctl_pitch_sense, ctl_system_change,
	ctl_effects_change, ctl_master_vol,
};
#endif


/*----------------------------------------------------------------
 * controls via pipe
 *----------------------------------------------------------------*/

static int cmsg(int type, int verbosity_level, char *fmt, ...)
{
	char local[1000];
#define TOO_LONG	980

	va_list ap;
	if ((type==CMSG_TEXT || type==CMSG_INFO || type==CMSG_WARNING) &&
	    ctl->verbosity<verbosity_level)
		return 0;

	va_start(ap, fmt);
	if (! ctl->opened || pipe_out < 0) {
		if (type == CMSG_ERROR) {
			vfprintf(stderr, fmt, ap);
			putc('\n', stderr);
		}
	} else if (type == CMSG_ERROR) {
		if (strlen(fmt) > TOO_LONG)
			fmt[TOO_LONG] = 0;
		vsprintf(local, fmt, ap);
		pipe_printf("CERR %d", type);
		pipe_puts(local);
	} else {
		if (type == CMSG_TEXT) {
			char *p;
			while ((p = strtok(fmt, "\n")) != NULL) {
				pipe_printf("CMSG %d %d", type, strlen(p));
				pipe_puts(p);
				fmt = NULL;
			}
		} else {
			if (strlen(fmt) > TOO_LONG)
				fmt[TOO_LONG] = 0;
			vsprintf(local, fmt, ap);
			pipe_printf("CMSG %d %d", type, strlen(local));
			pipe_puts(local);
		}
	}
	va_end(ap);
	return 0;
}

static void ctl_update(MidiInfo *mp) {}

static void ctl_total_time(int tt)
{
	pipe_printf("TIME %d", tt);
}

static void ctl_file_name(char *name)
{
	char tmp[32+1], *p;
	sprintf(tmp, "%-32.32s", name);
	if ((p = strchr(tmp, '\n')) != NULL) *p = 0;
	pipe_printf("FILE %s", tmp);
}

static void ctl_title(char *name)
{
	char tmp[32+1], *p;
	sprintf(tmp, "%-32.32s", name);
	if ((p = strchr(tmp, '\n')) != NULL) *p = 0;
	pipe_printf("TITL %s", tmp);
}

static void ctl_system_change(int type, int mode)
{
	switch (type) {
	case SY_CHORUS_MODE:
		pipe_printf("CHRS %d", mode);
		break;
	case SY_REVERB_MODE:
		pipe_printf("EVRB %d", mode);
		break;
	}
}

static void ctl_master_vol(int vol)
{
	pipe_printf("VOLM %d", vol);
}

static void ctl_effects_change(int ch, int type, int value)
{
}


/*----------------------------------------------------------------
 * controls vis shared memory
 *----------------------------------------------------------------*/

static void ctl_pitch_sense(int ch, int val)
{
	if (!(ctl->playing_mode & PLAY_TRACE))
		return;
	extpanel->channels[ch].pitchsense = val;
}

static void ctl_pitch_bend(int ch, int val)
{
	if (!(ctl->playing_mode & PLAY_TRACE))
		return;
	extpanel->channels[ch].pitchbend = val;
}

static void ctl_current_time(int ct)
{
	if (extpanel)
		extpanel->curcs = ct;
}

static int cnote[MAX_MIDI_CHANNELS];
static int cvel[MAX_MIDI_CHANNELS];

static void ctl_note(int ch, int note, int vel)
{
	if (!(ctl->playing_mode & PLAY_TRACE))
		return;

	if (vel == 0) {
		if (note == cnote[ch]) {
			cvel[ch] = 0;
			extpanel->vel[ch] = 0;
		}
	} else if (vel >= cvel[ch]) {
		cvel[ch] = vel;
		cnote[ch] = note;
		extpanel->vel[ch] = vel * extpanel->channels[ch].controls[CTL_MAIN_VOLUME] *
			extpanel->channels[ch].controls[CTL_EXPRESSION] / (127*127);
		if (extpanel->vel[ch] > extpanel->maxvel[ch])
			extpanel->maxvel[ch] = extpanel->vel[ch];
	}
}

static void ctl_program(int ch, int val)
{
	if (!(ctl->playing_mode & PLAY_TRACE))
		return;
	extpanel->channels[ch].preset = val;
}

static void ctl_bank(int ch, int val)
{
	if (!(ctl->playing_mode & PLAY_TRACE))
		return;
	extpanel->channels[ch].bank = val;
}

static void ctl_control_change(int ch, int type, int val)
{
	if (!(ctl->playing_mode & PLAY_TRACE))
		return;
	extpanel->channels[ch].controls[type] = val;
	if (type == CTL_MAIN_VOLUME || type == CTL_EXPRESSION)
		ctl_note(ch, cnote[ch], cvel[ch]);
}


/*----------------------------------------------------------------
 * reset controls
 *----------------------------------------------------------------*/

static void ctl_reset(MidiInfo *mp)
{
	int i;
	if (!(ctl->playing_mode & PLAY_TRACE))
		return;
	if (extpanel->multi_part != mp->multi_part) {
		extpanel->multi_part = mp->multi_part;
		extpanel->reset_panel = 1;
	}
	for (i = 0; i < MAX_MIDI_CHANNELS; i++) {
		ChannelStat *cp = &channels[i];
		ctl_program(i, cp->preset);
		ctl_bank(i, cp->bank);
		ctl_pitch_sense(i, cp->pitchsense);
		ctl_pitch_bend(i, cp->pitchbend);
		ctl_control_change(i, CTL_SUSTAIN, cp->controls[CTL_SUSTAIN]);
		ctl_control_change(i, CTL_PAN, cp->controls[CTL_PAN]);
		ctl_control_change(i, CTL_MAIN_VOLUME, cp->controls[CTL_MAIN_VOLUME]);
		ctl_control_change(i, CTL_EXPRESSION, cp->controls[CTL_EXPRESSION]);
		cvel[i] = 0;
		extpanel->vel[i] = extpanel->maxvel[i] = 0;
	}
}


#ifdef INCLUDE_PIPE_MODE
/*================================================================
 * open/close general pipe interface
 *================================================================*/

static int pipe_open(int using_stdin, int using_stdout)
{
	if (ctl->playing_mode & PLAY_TRACE) {
		if (gen_shmid < 0)
			ctl->playing_mode &= ~PLAY_TRACE;
		else {
			extpanel = (PanelInfo *)shmat(gen_shmid, 0, 0);
			extpanel->reset_panel = 0;
			extpanel->multi_part = 0;
		}
	}
	ctl->opened = 1;
	return 0;
}

static void pipe_close(void)
{
	if (ctl->opened) {
		ctl->opened = 0;
	}
}
#endif


#ifdef INCLUDE_TK_MODE
/*================================================================
 * open/close Tcl/Tk interface
 *================================================================*/

char *tk_display = NULL;
char *tk_geometry = NULL;
char *tk_effect = NULL;

static int child_pid = -1;

static void shm_alloc(void);
static void shm_free(void);
static void child_killed(void);
static void start_panel(void);
static int AppInit(Tcl_Interp *interp);
static int StartPlayer(ClientData clientData, Tcl_Interp *interp,
		       int argc, char *argv[]);
static int ReallyQuit(ClientData clientData, Tcl_Interp *interp,
		      int argc, char *argv[]);
static int InitCmdLine(ClientData clientData, Tcl_Interp *interp,
		       int argc, char *argv[]);
static int ExitAll(ClientData clientData, Tcl_Interp *interp,
		       int argc, char *argv[]);

static void child_killed(void)
{
	int st;
	waitpid(child_pid, &st, WNOHANG);
	if (! WIFEXITED(st)) {
		reset_signal(SIGCHLD);
		return;
	}
	child_pid = -1;
	shm_free();
	exit(0);
}

static void shm_alloc(void)
{
	gen_shmid = shmget(IPC_PRIVATE, sizeof(PanelInfo),
			   IPC_CREAT|0600);
	if (gen_shmid < 0) {
		fprintf(stderr, "can't allocate shared memory\n");
		exit(1);
	}
	extpanel = (PanelInfo *)shmat(gen_shmid, 0, 0);
	extpanel->reset_panel = 0;
	extpanel->multi_part = 0;
}

static void shm_free(void)
{ 
	int st;
	if (child_pid != -1) {
		int st;
		signal(SIGCHLD, SIG_DFL);
		kill(child_pid, SIGTERM);
		waitpid(child_pid, &st, 0);
	}
	if (gen_shmid >= 0) {
		shmctl(gen_shmid, IPC_RMID,NULL);
		shmdt((char *)extpanel);
		gen_shmid = -1;
	}
}

static int tk_open(int using_stdin, int using_stdout)
{
	int pipeAppli[2], pipePanel[2];
	int rc;

	if (pipe(pipeAppli) || pipe(pipePanel)) {
		fprintf(stderr, "can't open pipe\n");
		return 1;
	}
    
	shm_alloc();

	if ((child_pid = fork()) == 0) {
		/*child*/
		close(pipePanel[1]); 
		close(pipeAppli[0]);
	    
		/* redirect to stdin/out */
		dup2(pipePanel[0], fileno(stdin));
		close(pipePanel[0]);
		dup2(pipeAppli[1], fileno(stdout));
		close(pipeAppli[1]);
		start_panel();
	} else {
		close(pipePanel[0]);
		close(pipeAppli[1]);
    
		pipe_in = pipeAppli[0];
		pipe_out = pipePanel[1];
	}

	add_signal(SIGCHLD, child_killed, 0);
	add_signal(SIGTERM, shm_free, 1);
	add_signal(SIGINT, shm_free, 1);

	ctl->opened = 1;
	return 0;
}

static void tk_close(void)
{
	if (ctl->opened) {
		ctl->opened = 0;
		shm_free();
	}
}

static void start_panel(void)
{
	char *argv[128];
	int argc;
    
	argc = 0;
	argv[argc++] = "-f";
	argv[argc++] = TKPROGPATH;

	if (tk_display) {
		argv[argc++] = "-display";
		argv[argc++] = tk_display;
	}
	if (tk_geometry) {
		argv[argc++] = "-geometry";
		argv[argc++] = tk_geometry;
	}
	
	/* call Tk main routine */
	Tk_Main(argc, argv, AppInit);

	exit(0);
}

Tcl_Interp *my_interp;

static int AppInit(Tcl_Interp *interp)
{
	my_interp = interp;

	if (Tcl_Init(interp) == TCL_ERROR) {
		return TCL_ERROR;
	}
	if (Tk_Init(interp) == TCL_ERROR) {
		return TCL_ERROR;
	}

	Tcl_CreateCommand(interp, "ReallyQuit", ReallyQuit,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "StartPlayer", StartPlayer,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "InitCmdLine", InitCmdLine,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "TraceCreate", TraceCreate,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "TraceUpdate", TraceUpdate,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "TraceReset", TraceReset,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "ExitAll", ExitAll,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	return TCL_OK;
}

static int ReallyQuit(ClientData clientData, Tcl_Interp *interp,
		      int argc, char *argv[])
{
	v_eval("WriteMsg \"ZAPP\"");
	return TCL_OK;
}

static int StartPlayer(ClientData clientData, Tcl_Interp *interp,
		       int argc, char *argv[])
{
	return TCL_OK;
}

static int ExitAll(ClientData clientData, Tcl_Interp *interp,
		   int argc, char *argv[])
{
	/* window is killed; kill the parent process, too */
	kill(getppid(), SIGTERM);
	for (;;)
		sleep(1000);
}

static int InitCmdLine(ClientData clientData, Tcl_Interp *interp,
		       int argc, char *argv[])
{
	if (tk_effect)
		v_eval("SetUserEffects %s", tk_effect);

	if (ctl->playing_mode & PLAY_SHUFFLE)
		v_eval("set Config(ShufflePlay) 1");
	else
		v_eval("set Config(ShufflePlay) 0");

	if (ctl->playing_mode & PLAY_TRACE) {
		v_eval("set Stat(Tracing) 1");
		v_eval("set Config(Disp:trace) 1");
	} else {
		v_eval("set Stat(Tracing) 0");
		v_eval("set Config(Disp:trace) 0");
	}

	if (ctl->playing_mode & PLAY_AUTOSTART)
		v_eval("set Config(AutoStart) 1");
	if (ctl->playing_mode & PLAY_AUTOEXIT)
		v_eval("set Config(AutoExit) 1");
	if (ctl->playing_mode & PLAY_REPEAT)
		v_eval("set Config(RepeatPlay) 1");

	return TCL_OK;
}

#endif

/*----------------------------------------------------------------
 * Read information coming from the window in a BLOCKING way
 *----------------------------------------------------------------*/

/* commands are: PREV, NEXT, QUIT, STOP, LOAD, JUMP, VOLM */

static int ctl_blocking_read(MidiInfo *mp, int *valp)
{
	char buf[256], *arg;
	int rc;

	rc = pipe_gets(buf, sizeof(buf)-1);
	arg = buf + 5;

	switch (*buf) {
	case 'J': /* jump */
		*valp = atoi(arg);
		return RC_JUMP;

	case 'M': /* move */
		*valp = atoi(arg);
		return RC_MOVE;
		
	case 'Z': /* kill */
		return RC_KILL;

	case 'Q': /* stop */
		return RC_QUIT;
		
	case 'L': /* load file; the file name appears in next line */
		return RC_LOAD_FILE;		  
	  
	case 'N': /* next */
		return RC_NEXT;
	  
	case 'P': /* previous */
		return RC_PREVIOUS;
	  
	case 'R': /* restart */
		return RC_RESTART;
	  
	case 'F': /* forward */
		*valp = 200;
		return RC_MOVE;
	  
	case 'B': /* backward */
		*valp = -200;
		return RC_MOVE;

	case 'C': /* chorus mode */
		*valp = atoi(arg);
		return RC_CHANGE_CHORUS;

	case 'E': /* reverb mode */
		*valp = atoi(arg);
		return RC_CHANGE_REVERB;

	case 'V': /* volume */
		*valp = atoi(arg);
		switch (buf[1]) {
		case 'O':
			return RC_CHANGE_VOLUME;
		case 'B':
			return RC_CHANGE_BASS;
		case 'T':
			return RC_CHANGE_TREBLE;
		}
		return RC_NONE;

	case 'S': /* pause/continue */
		return *valp ? RC_CONTINUE : RC_PAUSE;

	case 'O': /* key offset */
		*valp = atoi(arg);
		return RC_BASE_CHANGE;

	case 'A': /* auto repeat mode */
		ctl->repeated = atoi(arg);
		return RC_NONE;

	case 'D': /* drum channels */
		channel_drums = atoi(arg);
		return RC_NONE;
	case 'X':
		if (mp)
			parse_index_line(mp, arg, TRUE);
		break;
	}
	  
	return RC_NONE;
}

/* 
 * Read information coming from the window in a non blocking way
 */
static int ctl_read(MidiInfo *mp, int *valp)
{
	if (pipe_in >= 0 && pipe_read_ready() > 0)
		return ctl_blocking_read(mp, valp);
	else
		return RC_NONE;
}


/*----------------------------------------------------------------
 * pass file lists and start playing
 *----------------------------------------------------------------*/

#ifdef INCLUDE_PIPE_MODE
static void pipe_pass_playing_list(int number_of_files, char *list_of_files[])
{
	int i;

	/* Pass the list to the interface */
	if (ctl->playing_mode & PLAY_SHUFFLE) {
		srandom(getpid());
		for (i = 0; i < number_of_files; i++) {
			int j = random() % (number_of_files - i);
			if (j) {
				char *tmp = list_of_files[i];
				list_of_files[i] = list_of_files[i + j];
				list_of_files[i + j] = tmp;
			}
		}
	}

	if (pipe_in < 0) {
		MidiInfo minfo;
		int command;
		for (i = 0; i < number_of_files; i++) {
			memcpy(&minfo, &glinfo, sizeof(minfo));
			minfo.filename = list_of_files[i];
			if ((command = play_midi_file(&minfo)) == RC_KILL)
				break;
		}
		return;
	}
	start_midi();
}
#endif

#ifdef INCLUDE_TK_MODE
static void tk_pass_playing_list(int number_of_files, char *list_of_files[])
{
	int i;

	/* Pass the list to the interface */
	if (number_of_files > 0 && list_of_files != NULL) {
		pipe_printf("LIST %d", number_of_files);
		for (i = 0; i <number_of_files; i++)
			pipe_puts(list_of_files[i]);
	}

	start_midi();
}
#endif

/* start playing */
static void start_midi(void)
{
	int command;
	int val;
	char local[1000];
	MidiInfo minfo;

	/* read the first command */
	memcpy(&minfo, &glinfo, sizeof(minfo));
	command = ctl_blocking_read(&minfo, &val);

	/* Main Loop */
	while (command != RC_KILL) {
		if (command == RC_LOAD_FILE) {
			/* Read a LoadFile command */
			pipe_gets(local, sizeof(local)-1);
			memcpy(&minfo, &glinfo, sizeof(minfo));
			minfo.filename = local;
			command = play_midi_file(&minfo);
		} else {
			switch (command) {
			case RC_QUIT:
				midi_close(&minfo);
				break;
			case RC_NEXT:
			case RC_TUNE_END:
				pipe_puts("NEXT");
				break;
			case RC_REALLY_PREVIOUS:
				pipe_puts("PREV");
				break;
			}
			command = ctl_blocking_read(&minfo, &val);
		}
	}
}

/*================================================================
 * pipe comunication
 *================================================================*/

static int pipe_read_ready()
{
	int num;
	ioctl(pipe_in, FIONREAD, &num); /* see how many chars in buffer. */
	return num;
}

static void pipe_printf(char *fmt, ...)
{
	char buf[1024];
	va_list ap;
	va_start(ap, fmt);
	if (pipe_out < 0) return;
	vsprintf(buf, fmt, ap);
	pipe_puts(buf);
}

static void pipe_puts(char *str)
{
	int len;
	char lf = '\n';
	if (pipe_out < 0) return;
	len = strlen(str);
	write(pipe_out, str, len);
	write(pipe_out, &lf, 1);
}


int pipe_gets(char *str, int maxlen)
{
	/* blocking reading */
	char *p;
	int len;

	/* at least 5 letters (4+\n) command */
	len = 0;
	for (p = str; ; p++) {
		read(pipe_in, p, 1);
		if ((*p == '\n') || (len >= maxlen))
			break;
		len++;
	}

	*p = 0;
	return len;
}
