/*================================================================
 * awe_effect.h
 *	convert NRPN parameters to Emu8000 raw effects
 *
 * Copyright (C) 1996-1998 Takashi Iwai
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *================================================================*/

#include <stdio.h>
#include <math.h>
#include "config.h"
#include "util.h"
#include "midievent.h"
#include "awe_effect.h"
#ifdef linux
#include <linux/awe_voice.h>
#else
#include <awe_voice.h>
#endif

/*----------------------------------------------------------------
 * AWE32 NRPN effects
 *----------------------------------------------------------------*/

static unsigned short fx_delay(int val);
static unsigned short fx_attack(int val);
static unsigned short fx_hold(int val);
static unsigned short fx_decay(int val);
static unsigned short fx_the_value(int val);
static unsigned short fx_twice_value(int val);
static unsigned short fx_conv_pitch(int val);
static unsigned short fx_conv_Q(int val);

/* function for each NRPN */		/* [range]  units */
#define fx_env1_delay	fx_delay	/* [0,5900] 4msec */
#define fx_env1_attack	fx_attack	/* [0,5940] 1msec */
#define fx_env1_hold	fx_hold		/* [0,8191] 1msec */
#define fx_env1_decay	fx_decay	/* [0,5940] 4msec */
#define fx_env1_release	fx_decay	/* [0,5940] 4msec */
#define fx_env1_sustain	fx_the_value	/* [0,127] 0.75dB */
#define fx_env1_pitch	fx_the_value	/* [-127,127] 9.375cents */
#define fx_env1_cutoff	fx_the_value	/* [-127,127] 56.25cents */

#define fx_env2_delay	fx_delay	/* [0,5900] 4msec */
#define fx_env2_attack	fx_attack	/* [0,5940] 1msec */
#define fx_env2_hold	fx_hold		/* [0,8191] 1msec */
#define fx_env2_decay	fx_decay	/* [0,5940] 4msec */
#define fx_env2_release	fx_decay	/* [0,5940] 4msec */
#define fx_env2_sustain	fx_the_value	/* [0,127] 0.75dB */

#define fx_lfo1_delay	fx_delay	/* [0,5900] 4msec */
#define fx_lfo1_freq	fx_twice_value	/* [0,127] 84mHz */
#define fx_lfo1_volume	fx_twice_value	/* [0,127] 0.1875dB */
#define fx_lfo1_pitch	fx_the_value	/* [-127,127] 9.375cents */
#define fx_lfo1_cutoff	fx_twice_value	/* [-64,63] 56.25cents */

#define fx_lfo2_delay	fx_delay	/* [0,5900] 4msec */
#define fx_lfo2_freq	fx_twice_value	/* [0,127] 84mHz */
#define fx_lfo2_pitch	fx_the_value	/* [-127,127] 9.375cents */

#define fx_init_pitch	fx_conv_pitch	/* [-8192,8192] cents */
#define fx_chorus	fx_the_value	/* [0,255] -- */
#define fx_reverb	fx_the_value	/* [0,255] -- */
#define fx_cutoff	fx_twice_value	/* [0,127] 62Hz */
#define fx_filterQ	fx_conv_Q	/* [0,127] -- */

static unsigned short fx_delay(int val)
{
	return (0x8000 - val * 4 * 1000 / 725);
}

static short attack_time_tbl[128] = {
32767, 32767, 5989, 4235, 2994, 2518, 2117, 1780, 1497, 1373, 1259, 1154, 1058, 970, 890, 816,
707, 691, 662, 634, 607, 581, 557, 533, 510, 489, 468, 448, 429, 411, 393, 377,
361, 345, 331, 317, 303, 290, 278, 266, 255, 244, 234, 224, 214, 205, 196, 188,
180, 172, 165, 158, 151, 145, 139, 133, 127, 122, 117, 112, 107, 102, 98, 94,
90, 86, 82, 79, 75, 72, 69, 66, 63, 61, 58, 56, 53, 51, 49, 47,
45, 43, 41, 39, 37, 36, 34, 33, 31, 30, 29, 28, 26, 25, 24, 23,
22, 21, 20, 19, 19, 18, 17, 16, 16, 15, 15, 14, 13, 13, 12, 12,
11, 11, 10, 10, 10, 9, 9, 8, 8, 8, 8, 7, 7, 7, 6, 0,
};

/* search an index for specified time from given time table */
static int
calc_parm_search(int msec, short *table)
{
	int left = 1, right = 127, mid;
	while (left < right) {
		mid = (left + right) / 2;
		if (msec < (int)table[mid])
			left = mid + 1;
		else
			right = mid;
	}
	return left;
}

static unsigned short fx_attack(int val)
{
	return calc_parm_search(val, attack_time_tbl);
}

static unsigned short fx_hold(int val)
{
	return (unsigned char)((0x7f * 92 - val) / 92);
}

static short decay_time_tbl[128] = {
32767, 32767, 22614, 15990, 11307, 9508, 7995, 6723, 5653, 5184, 4754, 4359, 3997, 3665, 3361, 3082,
2828, 2765, 2648, 2535, 2428, 2325, 2226, 2132, 2042, 1955, 1872, 1793, 1717, 1644, 1574, 1507,
1443, 1382, 1324, 1267, 1214, 1162, 1113, 1066, 978, 936, 897, 859, 822, 787, 754, 722,
691, 662, 634, 607, 581, 557, 533, 510, 489, 468, 448, 429, 411, 393, 377, 361,
345, 331, 317, 303, 290, 278, 266, 255, 244, 234, 224, 214, 205, 196, 188, 180,
172, 165, 158, 151, 145, 139, 133, 127, 122, 117, 112, 107, 102, 98, 94, 90,
86, 82, 79, 75, 72, 69, 66, 63, 61, 58, 56, 53, 51, 49, 47, 45,
43, 41, 39, 37, 36, 34, 33, 31, 30, 29, 28, 26, 25, 24, 23, 22,
};

static unsigned short fx_decay(int val)
{
	return calc_parm_search(val, decay_time_tbl);
}

static unsigned short fx_the_value(int val)
{
	return (unsigned short)(val & 0xff);
}

static unsigned short fx_twice_value(int val)
{
	return (unsigned short)((val * 2) & 0xff);
}

static unsigned short fx_conv_pitch(int val)
{
	return (short)(val * 4096 / 1200);
}

static unsigned short fx_conv_Q(int val)
{
	return (unsigned short)((val / 8) & 0xff);
}


struct awe_effect_rec awe_effects[] =
{
	{AWE_FX_LFO1_DELAY,	fx_lfo1_delay,	"L1_delay"},
	{AWE_FX_LFO1_FREQ,	fx_lfo1_freq,	"L1_freq"},
	{AWE_FX_LFO2_DELAY,	fx_lfo2_delay,	"L2_delay"},
	{AWE_FX_LFO2_FREQ,	fx_lfo2_freq,	"L2_freq"},

	{AWE_FX_ENV1_DELAY,	fx_env1_delay,	"e1_delay"},
	{AWE_FX_ENV1_ATTACK,	fx_env1_attack,	"e1_attack"},
	{AWE_FX_ENV1_HOLD,	fx_env1_hold,	"e1_hold",},
	{AWE_FX_ENV1_DECAY,	fx_env1_decay,	"e1_decay"},
	{AWE_FX_ENV1_SUSTAIN,	fx_env1_sustain,"e1_sustain"},
	{AWE_FX_ENV1_RELEASE,	fx_env1_release,"e1_release"},

	{AWE_FX_ENV2_DELAY,	fx_env2_delay,	"e2_delay"},
	{AWE_FX_ENV2_ATTACK,	fx_env2_attack,	"e2_attack"},
	{AWE_FX_ENV2_HOLD,	fx_env2_hold,	"e2_hold"},
	{AWE_FX_ENV2_DECAY,	fx_env2_decay,	"e2_decay"},
	{AWE_FX_ENV2_SUSTAIN,	fx_env2_sustain,"e2_sustain"},
	{AWE_FX_ENV2_RELEASE,	fx_env2_release,"e2_release"},

	{AWE_FX_INIT_PITCH,	fx_init_pitch,	"fx_init_pitch"},
	{AWE_FX_LFO1_PITCH,	fx_lfo1_pitch,	"L1_pitch"},
	{AWE_FX_LFO2_PITCH,	fx_lfo2_pitch,	"L2_pitch"},
	{AWE_FX_ENV1_PITCH,	fx_env1_pitch,	"e1_pitch"},
	{AWE_FX_LFO1_VOLUME,	fx_lfo1_volume,	"L1_volume"},
	{AWE_FX_CUTOFF,		fx_cutoff,	"fx_cutoff"},
	{AWE_FX_FILTERQ,	fx_filterQ,	"fx_filterQ"},
	{AWE_FX_LFO1_CUTOFF,	fx_lfo1_cutoff,	"L1_cutoff"},
	{AWE_FX_ENV1_CUTOFF,	fx_env1_cutoff,	"e1_cutoff"},
	{AWE_FX_CHORUS,		fx_chorus,	"fx_chorus"},
	{AWE_FX_REVERB,		fx_reverb,	"fx_reverb"},
};

int num_awe_effects = numberof(awe_effects);

/*----------------------------------------------------------------
 * GS(SC88) NRPN effects; still experimental
 *----------------------------------------------------------------*/

/* cutoff: quarter semitone step, max=255 */
static unsigned short gs_cutoff(MidiInfo *mp, int val)
{
	return (val - 64) * mp->fx_sense[FX_CUTOFF] / 50;
}

/* resonance: 0 to 15(max) */
static unsigned short gs_filterQ(MidiInfo *mp, int val)
{
	return (val - 64) * mp->fx_sense[FX_RESONANCE] / 50;
}

/* attack: */
static unsigned short gs_attack(MidiInfo *mp, int val)
{
	return -(val - 64) * mp->fx_sense[FX_ATTACK] / 50;
}

/* decay: */
static unsigned short gs_decay(MidiInfo *mp, int val)
{
	return -(val - 64) * mp->fx_sense[FX_RELEASE] / 50;
}

/* release: */
static unsigned short gs_release(MidiInfo *mp, int val)
{
	return -(val - 64) * mp->fx_sense[FX_RELEASE] / 50;
}

/* vibrato freq: 0.042Hz step, max=255 */
static unsigned short gs_vib_rate(MidiInfo *mp, int val)
{
	return (val - 64) * mp->fx_sense[FX_VIBRATE] / 50;
}

/* vibrato depth: max=127, 1 octave */
static unsigned short gs_vib_depth(MidiInfo *mp, int val)
{
	return (val - 64) * mp->fx_sense[FX_VIBDEPTH] / 50;
}

/* vibrato delay: -0.725msec step */
static unsigned short gs_vib_delay(MidiInfo *mp, int val)
{
	return -(val - 64) * mp->fx_sense[FX_VIBDELAY] / 50;
}

struct gs_effect_rec gs_effects[] =
{
	{32, AWE_FX_CUTOFF,	gs_cutoff, "gs_cutoff"},
	{33, AWE_FX_FILTERQ,	gs_filterQ, "gs_filterQ"},
	{99, AWE_FX_ENV2_ATTACK, gs_attack, "gs_attack"},
	{100, AWE_FX_ENV2_DECAY, gs_decay, "gs_decay"},
	{102, AWE_FX_ENV2_RELEASE, gs_release, "gs_release"},
	{8, AWE_FX_LFO1_FREQ, gs_vib_rate, "gs_vib_rate"},
	{9, AWE_FX_LFO1_VOLUME, gs_vib_depth, "gs_vib_depth"},
	{10, AWE_FX_LFO1_DELAY, gs_vib_delay, "gs_vib_delay"},
};

int num_gs_effects = numberof(gs_effects);


/*----------------------------------------------------------------
 * XG control effects; still experimental
 *----------------------------------------------------------------*/

/* cutoff: quarter semitone step, max=255 */
static unsigned short xg_cutoff(MidiInfo *mp, int val)
{
	return (val - 64) * mp->fx_sense[FX_CUTOFF] / 64;
}

/* resonance: 0(open) to 15(most nasal) */
static unsigned short xg_filterQ(MidiInfo *mp, int val)
{
	return (val - 64) * mp->fx_sense[FX_RESONANCE] / 64;
}

/* attack: */
static unsigned short xg_attack(MidiInfo *mp, int val)
{
	return -(val - 64) * mp->fx_sense[FX_ATTACK] / 64;
}

/* release: */
static unsigned short xg_release(MidiInfo *mp, int val)
{
	return -(val - 64) * mp->fx_sense[FX_RELEASE] / 64;
}

struct xg_effect_rec xg_effects[] =
{
	{71, AWE_FX_CUTOFF,	xg_cutoff, "xg_cutoff"},
	{74, AWE_FX_FILTERQ,	xg_filterQ, "xg_filterQ"},
	{72, AWE_FX_ENV2_RELEASE, xg_release, "gs_release"},
	{73, AWE_FX_ENV2_ATTACK, xg_attack, "gs_attack"},
};

int num_xg_effects = numberof(xg_effects);

