/*
 *  The driver for the ForteMedia FM801 based soundcards
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "../include/driver.h"
#include "../include/fm801.h"
#include "../include/mpu401.h"
#include "../include/opl3.h"
#include "../include/initval.h"

int snd_index[SND_CARDS] = SND_DEFAULT_IDX;	/* Index 1-MAX */
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;	/* ID for this card */
int snd_dac_frame_size[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 128};
int snd_adc_frame_size[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 128};
MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for the FM801 soundcard.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for the FM801 soundcard.");
MODULE_PARM(snd_dac_frame_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dac_frame_size, "DAC frame size in kB for the FM801 soundcard.");
MODULE_PARM(snd_adc_frame_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_adc_frame_size, "ADC frame size in kB for the FM801 soundcard.");

struct snd_fm801 {
	struct pci_dev *pci;
	snd_irq_t *irqptr;
	snd_dma_t *dma1ptr;	/* DAC frame */
	snd_dma_t *dma2ptr;	/* ADC frame */
	snd_card_t *card;
	fm801_t *codec;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
	snd_rawmidi_t *midi_uart;
	snd_hwdep_t *synth;
};

static struct snd_fm801 *snd_fm801_cards[SND_CARDS] = SND_DEFAULT_PTR;

static void snd_card_fm801_use_inc(snd_card_t * card)
{
	MOD_INC_USE_COUNT;
}

static void snd_card_fm801_use_dec(snd_card_t * card)
{
	MOD_DEC_USE_COUNT;
}

static int snd_card_fm801_detect(snd_card_t * card, struct snd_fm801 *scard)
{
	if ((scard->pci = pci_find_device(PCI_VENDOR_ID_FORTEMEDIA,
					  PCI_DEVICE_ID_FORTEMEDIA_FM801,
					  scard->pci)) == NULL)
		return -ENODEV;
#ifdef NEW_PCI
	if (snd_register_ioport(card, scard->pci->resource[0].start, 0x80, "FM801", NULL) < 0)
#else
	if (snd_register_ioport(card, scard->pci->base_address[0] & ~PCI_BASE_ADDRESS_SPACE, 0x80, "FM801", NULL) < 0)
#endif
		return -ENODEV;
	return 0;
}

static void snd_card_fm801_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct snd_fm801 *scard = (struct snd_fm801 *) dev_id;
	fm801_t *codec;
	unsigned int status;

	if (!scard || !(codec = scard->codec))
		return;
	status = inw(FM801_REG(codec, IRQ_STATUS));
	if (status & (FM801_IRQ_PLAYBACK|FM801_IRQ_CAPTURE|FM801_IRQ_VOLUME)) {
		if (status & (FM801_IRQ_PLAYBACK|FM801_IRQ_CAPTURE))
			snd_fm801_interrupt(codec, status);
		if (status & FM801_IRQ_MPU) {
			if (scard->midi_uart)
				snd_mpu401_uart_interrupt(scard->midi_uart);
			outw(FM801_IRQ_MPU, FM801_REG(codec, IRQ_STATUS));
		}
		if (status & FM801_IRQ_VOLUME) {
			/* TODO */
			outw(FM801_IRQ_VOLUME, FM801_REG(codec, IRQ_STATUS));
		}
	}
}

static int snd_card_fm801_resources(snd_card_t * card,
                                  struct snd_fm801 *scard, int dev)
{
	int err;

	if ((err = snd_register_interrupt(card,
			"FM801", scard->pci->irq,
			SND_IRQ_TYPE_PCI, snd_card_fm801_interrupt,
			scard, NULL, &scard->irqptr)) < 0)
		return err;
	if ((err = snd_register_dma_channel(card,
			"FM801 - DAC frame", 0,
			SND_DMA_TYPE_PCI, snd_dac_frame_size[dev],
			NULL, &scard->dma1ptr)) < 0)
		return err;
	if ((err = snd_register_dma_channel(card,
			"FM801 - ADC frame", 2,
			SND_DMA_TYPE_PCI, snd_adc_frame_size[dev],
			NULL, &scard->dma2ptr)) < 0)
		return err;
	return 0;
}

static int snd_card_fm801_probe(int dev, struct snd_fm801 *scard)
{
	snd_card_t *card;
	snd_pcm_t *pcm = NULL;
	snd_kmixer_t *mixer = NULL;
	snd_rawmidi_t *midi_uart = NULL;
	snd_hwdep_t *synth = NULL;

	card = snd_card_new(snd_index[dev], snd_id[dev],
			    snd_card_fm801_use_inc, snd_card_fm801_use_dec);
	if (!card)
		return -ENOMEM;
	card->static_data = scard;
	card->type = SND_CARD_TYPE_CS461X;
	scard->pci = NULL;
	do {
		if (!snd_card_fm801_detect(card, scard))
			break;
	} while (scard->pci);
	if (scard->pci == NULL) {
		snd_card_free(card);
		return -ENODEV;
	}
	if (snd_card_fm801_resources(card, scard, dev) < 0) {
		snd_card_free(card);
		return -ENODEV;
	}
	scard->codec = snd_fm801_create(card, scard->pci,
				         scard->dma1ptr,
				         scard->dma2ptr,
				         scard->irqptr);
	if (!scard->codec)
		goto __nodev;
	mixer = snd_fm801_mixer(scard->codec, 0);
	if (!mixer)
		goto __nodev;
	pcm = snd_fm801_pcm(scard->codec);
	if (!pcm)
		goto __nodev;
	midi_uart = snd_mpu401_uart_new_device(card, MPU401_HW_FM801,
						 FM801_REG(scard->codec, MPU401_DATA),
						 scard->irqptr->irq);
	if (!midi_uart)
		goto __nodev;
	synth = snd_opl3_new_device(card, FM801_REG(scard->codec, OPL3_BANK0),
					  FM801_REG(scard->codec, OPL3_BANK1),
					  OPL3_HW_OPL3_FM801, -1);
	if (!synth)
		goto __nodev;
 
	if (snd_mixer_register(mixer, 0) < 0)
		goto __nodev;
	if (snd_pcm_register(pcm, 0) < 0) {
		snd_mixer_unregister(mixer);
		mixer = NULL;
		goto __nodev;
	}
	if (snd_rawmidi_register(midi_uart, 0) < 0) {
		snd_pcm_unregister(pcm);
		pcm = NULL;
		snd_mixer_unregister(mixer);
		mixer = NULL;
		goto __nodev;
	}
	if (snd_hwdep_register(synth, 0) < 0) {
		snd_pcm_unregister(pcm);
		pcm = NULL;
		snd_mixer_unregister(mixer);
		mixer = NULL;
		snd_rawmidi_unregister(midi_uart);
		midi_uart = NULL;
		goto __nodev;
	}
	strcpy(card->abbreviation, "FM801");
	strcpy(card->shortname, "ForteMedia FM801");
	sprintf(card->longname, "%s at 0x%lx, irq %i",
		card->shortname,
#ifdef NEW_PCI
		scard->pci->resource[0].start,
#else
		scard->pci->base_address[0] & ~PCI_BASE_ADDRESS_SPACE,
#endif
		scard->irqptr->irq);

	if (!snd_card_register(card)) {
		scard->card = card;
		scard->mixer = mixer;
		scard->pcm = pcm;
		scard->midi_uart = midi_uart;
		scard->synth = synth;
		return 0;
	}
	goto __nodev;

      __nodev:
	if (synth)
		snd_hwdep_free(synth);
	if (midi_uart)
		snd_rawmidi_free(midi_uart);
	if (mixer)
		snd_mixer_free(mixer);
	if (pcm)
		snd_pcm_free(pcm);
	if (scard->codec) {
		snd_fm801_free(scard->codec);
		scard->codec = NULL;
	}
	snd_card_free(card);
	return -ENXIO;
}

static int snd_card_fm801_free(int dev)
{
	struct snd_fm801 *scard;
	snd_pcm_t *pcm;

	scard = snd_fm801_cards[dev];
	snd_fm801_cards[dev] = NULL;
	if (scard) {
		snd_card_unregister(scard->card);
		if (scard->synth)
			snd_hwdep_unregister(scard->synth);
		if (scard->midi_uart)
			snd_rawmidi_unregister(scard->midi_uart);
		if (scard->mixer)
			snd_mixer_unregister(scard->mixer);
		if (scard->pcm) {
			pcm = scard->pcm;
			scard->pcm = NULL;	/* turn off interrupts */
			snd_pcm_unregister(pcm);
		}
		if (scard->codec) {
			snd_fm801_free(scard->codec);
			scard->codec = NULL;
		}
		snd_card_free(scard->card);
		snd_kfree(scard);
	}
	return 0;
}

int init_module(void)
{
	int dev, cards;
	struct snd_fm801 *scard;

	for (dev = cards = 0; dev < SND_CARDS; dev++) {
		scard = (struct snd_fm801 *)
				snd_kcalloc(sizeof(struct snd_fm801), GFP_KERNEL);
		if (!scard)
			continue;
		if (snd_card_fm801_probe(dev, scard) < 0) {
			snd_kfree(scard);
			break;
		}
		snd_fm801_cards[dev] = scard;
		cards++;
	}
	if (!cards) {
		snd_printk("ForteMedia FM801 soundcard #%i not found or device busy\n", dev + 1);
		return -ENODEV;
	}
	return 0;
}

void cleanup_module(void)
{
	int dev;

	for (dev = 0; dev < SND_CARDS; dev++)
		snd_card_fm801_free(dev);
}
