/* Some helpful utility functions for gpasm
   Copyright (C) 1998 James Bowman

This file is part of gpasm.

gpasm is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

gpasm is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with gpasm; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"

#include "gpasm.h"
#include "gperror.h"
#include "symbol.h"


/*
 * Parse a numeric constant
 */
int gpasm_number(char *s)
{
  char *endptr;
  int r;

  switch (tolower(s[0])) {
  case 'd':
    r = strtol(s + 2, &endptr, 10);
    break;
  case 'h':
    r = strtol(s + 2, &endptr, 16);
    break;
  case 'o':
    r = strtol(s + 2, &endptr, 8);
    break;
  case 'b':
    r = strtol(s + 2, &endptr, 2);
    break;
  default:
    assert(0); /* This should have been caught in the lexical stage */
  }

  /* Check that the number was OK */
  if (*endptr != '\'') {
    char complaint[80];

    sprintf(complaint,
	    isprint(*endptr) ?
	    "Illegal character '%c' in numeric constant" :
	    "Illegal character %#x in numeric constant",
	    *endptr);
    gperror(113, complaint);
  }

  return r;
}

void set_global(char *name, gpasmVal value)
{
  struct symbol *sym;
  struct variable *var;

  /* Search the entire stack (i.e. include macro's local symbol
     tables) for the symbol.  If not found, then add it to the global
     symbol table.  */

  sym = get_symbol(state.stTop, name);
  if (sym == NULL)
    sym = add_symbol(state.stGlobal, name);
  var = get_symbol_annotation(sym);
  if (var == NULL) {
    var = malloc(sizeof(*var));
    annotate_symbol(sym, var);
    var->value = value;
  } else {
    if ((state.pass == 2) &&
	(var->value != value)) {
      char message[BUFSIZ];

      sprintf(message,
	      "Value of symbol \"%s\" differs on second pass",
	      name);
      gperror(114, message);
    }
  }
}

void select_radix(char *radix_name)
{
  if (strcasecmp(radix_name, "hex") == 0)
    state.radix = 16;
  else if (strcasecmp(radix_name, "dec") == 0)
    state.radix = 10;
  else if (strcasecmp(radix_name, "oct") == 0)
    state.radix = 8;
  else {
    gperror(123, "Unrecognized radix");
  }
}
