/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999-2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <netdb.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <signal.h>
#include <sys/signal.h>
#include <sys/types.h>
#include <fcntl.h>
#include <pthread.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include "debug.h"
#include "sockets.h"

/*********************
**********************  Defines / Enumerated types
*********************/

#define BUFSIZE 8192

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

extern int h_errno;

/***********
************  Public
***********/

/***********
************  Private
***********/

#if 0
static int reads_til_fail = 500;
#endif

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PUBLIC ROUTINES
************/

/*****
******
*****/

PanSocket*
pan_socket_new (const gchar *address, gint port)
{
	PanSocket *sock = g_new0 (PanSocket, 1);	
	debug (DEBUG_PAN_OBJECT, "pan_socket_new: %p", sock);
	pan_socket_constructor (sock, pan_socket_destructor, address, port);
	return sock;
}

void
pan_socket_constructor (PanSocket *sock,
	PanObjectDestructor destructor,
	const gchar *address,
	gint port)
{
	struct sigaction act, oact;
	struct sockaddr_in serv_addr;
	struct in_addr inaddr;
	struct hostent host_info;
	int on;
	struct hostent *hp;
	int sockfd;
#ifdef TRY_AGAIN
	unsigned int max_retries = 3;
#endif

	pan_object_constructor (PAN_OBJECT(sock), destructor);
	
	/* sanity checks */
	g_return_if_fail (address!=NULL);
	g_return_if_fail (port>=0);

	/* set up the serv_addr struct */
	memset ( &serv_addr, 0, sizeof(serv_addr) );
	serv_addr.sin_family = AF_INET;
	serv_addr.sin_port = htons(port);

	/* Try to convert host name as dotted decimal */
	if ( inet_aton(address, &inaddr) != 0 )
	{
		memcpy(&serv_addr.sin_addr,&inaddr,sizeof(inaddr));
		host_info.h_name = NULL;
	}
	else /* If that failed, then look up the host name */
	{
		while (NULL==(hp = gethostbyname(address)))
		{
#ifdef TRY_AGAIN
			max_retries--;
			if (max_retries && (h_errno == TRY_AGAIN)) {
				sleep(1);
				continue;
			}
#endif
			g_warning (
				"Can't resolve %s -- PanSocket not created",
			       	address);
			return;
		}
		host_info = *hp;
		memcpy(&serv_addr.sin_addr, hp->h_addr, hp->h_length);
	}

	sockfd = socket(AF_INET, SOCK_STREAM, 0);
	if (sockfd < 0) {
		g_warning ("Can't create socket: %s", strerror(errno));
		return;
	}

	if (connect(sockfd, (struct sockaddr*)&serv_addr, sizeof(serv_addr))<0)
	{
		close(sockfd);
		g_warning ("socket connect failed: %s", strerror(errno));
		return;
	}

	on = 1;
	if (setsockopt (sockfd, SOL_SOCKET, SO_KEEPALIVE,
	                (char*)&on, sizeof(on)) < 0)
	{
		close(sockfd);
		g_warning ("socket keepalive option failed: %s",
			strerror(errno));
		return;
	}

	sock->sockfd = sockfd;
	sock->inbuf = g_malloc0 (BUFSIZE);
	sock->endbuf = sock->inbuf;
	sock->inptr = sock->inbuf;
	sock->line_buffer_size = 512;
	sock->line_buffer = g_malloc0 ( sock->line_buffer_size );
	sock->error = FALSE;

	/* If we write() to a dead socket, then let write return EPIPE rather
	   than crashing the program. */
	act.sa_handler = SIG_IGN;
	sigemptyset (&act.sa_mask);
	act.sa_flags = 0;
	sigaction (SIGPIPE, &act, &oact);

	debug(DEBUG_SOCKET_OUTPUT|DEBUG_SOCKET_INPUT,
		"socket created: %p for %s, port %d", sock, address, port );
        debug (DEBUG_PAN_OBJECT, "pan_socket_constructor: %p", sock);
}


void
pan_socket_destructor (PanObject *sock)
{
	g_return_if_fail (sock != NULL);

	debug (DEBUG_SOCKET_OUTPUT|DEBUG_SOCKET_INPUT,
		"socket closed: %p", sock);
        debug (DEBUG_PAN_OBJECT, "pan_socket_destructor: %p", sock);

	/* free up our pieces */
	close (PAN_SOCKET(sock)->sockfd);
	g_free (PAN_SOCKET(sock)->line_buffer);
	g_free (PAN_SOCKET(sock)->inbuf);

	/* pass along to superclass */
	pan_object_destructor (sock);
}

/*****
******
*****/

/* simple read buffering to hopefully speed things up. */
static int
read_char (PanSocket* ps, gchar* c)
{
	if ( ps->inptr == ps->endbuf )
	{
		int read_count = 0;
		int retval = 0;
		struct timeval tv;
		fd_set rfds;
		FD_ZERO (&rfds);
		FD_SET (ps->sockfd, &rfds);

#if 0
		if (!--reads_til_fail) {
			g_message ("Testing socket read failure");
			ps->error = TRUE;
			return -1;
		}
#endif

		/* We will timeout after 2 minutes. */
		tv.tv_sec = 120;
		tv.tv_usec = 0;
		retval = select (ps->sockfd+1, &rfds, NULL, NULL, &tv);
		if (retval <= 0) {
			g_warning (_("Timed out waiting for select()"));
			ps->error = TRUE;
			return -1;
		}

		read_count = read (ps->sockfd, ps->inbuf, BUFSIZE);
		ps->endbuf = ps->inbuf + read_count;
		ps->inptr = ps->inbuf;
		if (read_count <= 0)
		   return read_count;
	}

	*c = *ps->inptr++;

	return 1;
}


/*
 * get_server - read from a socket until \n
 */
const gint
pan_socket_getline(PanSocket *sock, const gchar **setme)
{
	gchar c;
	int len = 0;

	g_return_val_if_fail (sock->line_buffer_size>0, -1);

	sock->last_action_time=time(0);

	for ( ;; )
	{
		/* get the next char */
		if (sock->inptr!=sock->endbuf)
			c = *sock->inptr++;
		else if (read_char(sock, &c) != 1) {
			g_warning ("socket [%p] failed its read", sock);
			return -1;
		}

		/* grow the line buffer if we need to */
		if (len >= sock->line_buffer_size - 1)
		{
			sock->line_buffer_size *= 2;
			sock->line_buffer = g_realloc (
				sock->line_buffer,
				sock->line_buffer_size );
		}

		/* add the char to our line buffer */
		sock->line_buffer[len++] = c;

		if ( c == '\n' )
			break;
	}

	sock->line_buffer[len] = '\0';
	if ( !len || sock->line_buffer[len-1]!='\n' ) {
		debug(DEBUG_SOCKET_INPUT, "socket [%p] failed its read", sock);
		return -1;
	}

	debug(DEBUG_SOCKET_INPUT,
		"socket [%p] received [%s]", sock, sock->line_buffer);

	//pan_stats.bytes_received += len;

	*setme = sock->line_buffer;
	return 0;
}


gint 
pan_socket_putline (PanSocket     *sock,
                    const gchar   *line)
{
	const gchar* pch = line;
	gint nbytes = strlen (line);
	gint nleft = nbytes;

	sock->last_action_time=time(0);

	debug(DEBUG_SOCKET_OUTPUT,
		"[%ld][socket %p putline][%s]", time(0), (void*)sock, line );

	while (nleft>0)
	{
		/* write */
		const gint nwritten = write(sock->sockfd, pch, nleft);
		if (nwritten == -1) {
			sock->error = TRUE;
			return -1;
		}
		nleft -= nwritten;
		pch += nwritten;
	}

	return nleft;
}


gboolean 
pan_socket_putline_va (PanSocket    *sock,
                       const gchar  *format,
                       ...)
{
	va_list args;
	char* line;
	gint retval;

	g_return_val_if_fail (format!=NULL, -1);

	va_start(args, format);
	line = g_strdup_vprintf(format, args);
	va_end(args);

	retval = pan_socket_putline (sock, line);

	g_free (line);
	return retval;
}

/*****
******
*****/

