/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <string.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-stock.h>
//#include <libgnomeui/libgnomeui.h>

#include "articlelist.h"
#include "globals.h"
#include "group.h"
#include "grouplist.h"
#include "gui.h"
#include "queue.h"
#include "server.h"
#include "queue-item-headers.h"
#include "queue-item-grouplist.h"
#include "util.h"

/***
*** Global Variables
***/

/***
*** Public Routines
***/

group_data*
group_new (void)
{
	group_data *gdata;

	gdata = g_new0 (group_data, 1);
	gdata->state_filter = ~STATE_FILTER_KILLFILE;

	return gdata;
}

group_data*
group_copy (const group_data *src)
{
	group_data *dest = NULL;

	g_return_val_if_fail (src!=NULL, NULL);

	dest = group_new();
	dest->name = g_strdup (src->name);
	dest->description = g_strdup (src->description);
	dest->flags = src->flags;

	return dest;
}


void
group_set_attrib_string (const server_data *sdata, const group_data *gdata, const char *attrib, const char *value)
{
	if ( gdata->name )
	{
		gchar* key = g_strdup_printf ( "%s.%s", gdata->name, attrib );
		pan_db_put_value_str ( sdata->db, key, value );
		pan_db_sync ( sdata->db );
		g_free ( key );
	}
}


gchar*
group_get_attrib_string (const server_data *sdata, const group_data *gdata, const char *attrib)
{
	if ( !gdata->name )
		return NULL;
	else {
		gchar* key = g_strdup_printf ( "%s.%s", gdata->name, attrib );
		gchar* retval = pan_db_get_value_str ( sdata->db, key );
		g_free ( key );
		return retval;
	}
}


void
group_set_attrib_i (const server_data *sdata, const group_data *gdata, const char *attrib, int value)
{
	if ( gdata && gdata->name && attrib )
	{
		gchar* key = g_strdup_printf ( "%s.%s", gdata->name, attrib );
		pan_db_put_value_i ( sdata->db, key, value );
		pan_db_sync ( sdata->db );
		g_free ( key );

		/* FIXME this isn't broken, but it's kind of a kludge */
		if ( !strcmp(attrib,"Total") || !strcmp(attrib,"Read") )
			grouplist_update_row ( gdata );
	}
}


int
group_get_attrib_i (const server_data *sdata, const group_data *gdata, const char *attrib)
{

	if ( sdata && gdata && gdata->name && attrib )
	{
		gchar* key = g_strdup_printf ( "%s.%s", gdata->name, attrib );
		int retval = pan_db_get_value_i ( sdata->db, key );
		g_free ( key );
		return retval;
	}
	else return 0;
}


void
group_add_attrib_i (const server_data *sdata, const group_data *gdata, const char *attrib, int i)
{
	if ( sdata && gdata && gdata->name && attrib && i )
	{
		gchar* key = g_strdup_printf ( "%s.%s", gdata->name, attrib );
		pan_db_add_value_i ( sdata->db, key, i );
		pan_db_sync ( sdata->db );
		g_free ( key );

		/* Make the change visible */
		grouplist_update_row (gdata);
	}
}


int
compare_group_data_std (const void* va, const void* vb)
{
	return compare_group_data (*(group_data**)va, *(group_data**)vb);

}
gint
compare_group_data (group_data *a, group_data *b)
{
	return strcmp(a->name, b->name);
}


void
check_new_group_list_for_newness ( GSList* list )
{
	/* assumes list and Pan.grouplist are sorted by name.. */
	GSList* a = list;
	server_data *sdata = grouplist_get_current_server ();
	GSList* b = (sdata ? sdata->grouplist : NULL);
	while ( a!=NULL && b!=NULL )
	{
		group_data* group_a = (group_data*) a->data;
		const group_data* group_b = (group_data*) b->data;
		const int rel = strcmp ( group_a->name, group_b->name );

		if ( rel>0 ) { /* old group not in new list */
		   b = b->next;
		   continue;
		}

		/* group_a is new if it's actually new or just
		   matches an old group still flagged as new... */
		if ( rel!=0 || (group_b->flags&GROUP_NEW) )
			group_a->flags |= GROUP_NEW;
		else
			group_a->flags &= ~GROUP_NEW;

		a = a->next;

		if ( !rel )
			b = b->next;
	}
	while ( a!=NULL ) /* exhausted Pan.grouplist, so all these are new... */
	{
	   ((group_data*)a->data)->flags |= GROUP_NEW;
	   a = a->next;
	}
}

void
group_free (group_data *gdata)
{
	g_free (gdata->name);
	g_free (gdata->description);
	g_free (gdata);
}

static void
choose_download_dialog_clicked_cb (
	GnomeDialog *dialog,
	int button_number,
	gpointer gdata)
{
	server_data* sdata =
		(server_data*)gtk_object_get_data(GTK_OBJECT(dialog),"server");
	g_assert (sdata!=NULL);

	switch (button_number)
	{
		case 0: /* All Headers */
		{
			queue_add(QUEUE_ITEM(
				queue_item_headers_new(
					sdata, TRUE, gdata, HEADERS_ALL)));
			break;
		}
		case 1: /* New Headers */
		{
			queue_add(QUEUE_ITEM(
				queue_item_headers_new(
					sdata, TRUE, gdata, HEADERS_SAMPLE)));
			break;
		}
		default: /* Cancel */
		{
			break;
		}
	};

	gtk_widget_destroy (GTK_WIDGET(dialog));
}

void
choose_download_dialog (const server_data *sdata, group_data *gdata)
{
	GtkWidget *dialog = NULL;
	GtkWidget *label = NULL;
	char *temp = NULL;

	pan_lock();

	dialog = gnome_dialog_new (_("Select Header Download Option"),
				   _("Download All Headers"),
				   _("Sample Some Headers"),
				   GNOME_STOCK_BUTTON_CANCEL, NULL);

	gtk_signal_connect (GTK_OBJECT(dialog), "clicked",
			    GTK_SIGNAL_FUNC (choose_download_dialog_clicked_cb),
			    gdata);
	gtk_object_set_data(GTK_OBJECT(dialog), "server", (gpointer)sdata);

	temp = g_strdup_printf (_("Select an option for group: %s"), gdata->name);
	label = gtk_label_new (temp);
	g_free (temp);

	gtk_container_add (GTK_CONTAINER(GNOME_DIALOG(dialog)->vbox), label);
	pan_unlock();
	gui_popup_draw (dialog, Pan.window);
}


/*---[ group_empty ]--------------------------------------------------
 * actual function that empties the group and deletes the groups
 * database file and associated gnome_config sections
 *--------------------------------------------------------------------*/
void
group_empty (server_data *sdata, group_data *gdata)
{
	if ( gdata )
	{
		/* unload group (if it's loaded) */
		if (articlelist_get_current_group() == gdata) {
			articlelist_clear ();
		}

		/* clear out the article headers */
		ahdb_delete_db_file ( sdata, gdata );

		/* wipe out all records */
		group_set_attrib_i (sdata, gdata, "Total", 0);
		group_set_attrib_i (sdata, gdata, "Read", 0);

		/* flush the flags */
		if (gdata->flags & GROUP_SUBSCRIBED)
			gdata->flags = GROUP_SUBSCRIBED;
		else
			gdata->flags = 0;

		/* update display */
		grouplist_update_row ( gdata );
	}
}


/*--------------------------------------------------------------------
 * callback for group_empty, called when a button in group_empty_dialog
 * is clicked
 *--------------------------------------------------------------------*/
static void
group_empty_cb (
	GnomeDialog *dialog,
	int button_number,
	gpointer gdata)
{
	switch (button_number)
	{
		case 0: /* 'yes' */
			group_empty (grouplist_get_current_server (), gdata);
			break;
		case 1: /* 'no' */
			break;
		default: /* cancel */
			break;
	};


	gtk_widget_destroy (GTK_WIDGET(dialog));
}


/*--------------------------------------------------------------------
 * dialog popup asking user if they'd like to remove all the messages
 * for a given group.  called when the user tries to unsubscribe
 * or they choose the menu option to empty the group manually
 *--------------------------------------------------------------------*/

void
group_empty_dialog (group_data *gdata)
{
	GtkWidget *dialog = NULL;
	GtkWidget *label = NULL;
	gchar *pch = NULL;

	/* sanity check */
	g_return_if_fail (gdata!=NULL);

	pan_lock();

	/* create dialog */
	pch = g_strdup_printf (_("Clean group ``%s''"), gdata->name);
	dialog = gnome_dialog_new (
		pch, GNOME_STOCK_BUTTON_YES, GNOME_STOCK_BUTTON_NO, NULL);
	g_free (pch);

	/* body */
	pch = g_strdup_printf (
		_("Remove all messages from group: \n\n%s ?"), gdata->name);
	label = gtk_label_new (pch);
	g_free (pch);

	/* listen for clicks */
	gtk_signal_connect (
		GTK_OBJECT(dialog), "clicked",
		GTK_SIGNAL_FUNC (group_empty_cb), gdata);

	gtk_container_add (GTK_CONTAINER(GNOME_DIALOG(dialog)->vbox), label);
	pan_unlock();

	gui_popup_draw (dialog, Pan.window);
}

//============================================================================

//        if ( 0 < group_db_store_article_header_gslist ( gdata, list, 0 ) )
//		gdata->flags |= GROUP_ONDISK;
//        gui_set_progress ( 1 );


